#include "RootZoneEvapoTranspirationCalc.h"

//Root zone evapotranspiration only removes water between field capacity and wilting point, referred to as capillarity held water
//Note: Consider refactor to allow soil areas lower soil moisture limit than wilting point, such as theta residual

//Note: Root zone refers to soil below all pervious areas and any impervious area with tree cover, not only the soil land cover type.
//Note: Soil below any impervious area with tree cover is considered recharged via the lateral redistribution of groundwater. 

//Note: The model StatisticalHydro uses the input data of potential ET to determine atmospheric demand
//Note:	The model SpatialTemperatureHydro uses latent energy to determine atmospheric demand

//Note: EvapoTranspiration_SoilEvapZone_m (m) is reduced from potential ET by ratio of available soil moisture based on theory, including ...
//Note: ... ET_act = ET_pot * (soil storage)/(soil storage max) based on Saxton (1986) Eq in Fisher et al. (2005)
//Note: ... Fisher et al (2005) show potential ET based on Penmam-Monteith is too high relative to observed actual ET. 
//Note: ... The Eqs of Fisher et al. (2005) are not numbered, the same LHS variable is used for multiple Eqs of potential ET
//Note: ... Fisher et al (2005) give the adjustment in text near citation of Saxton (1986), as f(phi); improvement illustrated in Fig 2 vs Fig 1. 
//References:
//Fisher, J. B., DeBiase, T. A., Qi, Y., Xu, M., & Goldstein, A. H. (2005). Evapotranspiration models compared on a Sierra Nevada forest ecosystem. Environmental Modelling & Software, 20(6), 783-796. doi:https://doi.org/10.1016/j.envsoft.2004.04.009

//RootZoneEvapoTranspirationCalc::Calculate function handles StatisticalHydro and SpatialTemperatureHydro models
void RootZoneEvapoTranspirationCalc::Calculate(Inputs* input, CompactRagged* beC, DataFolder* folder, int TI_bin, int DataDrawer_ID, int DataFolder_ID, int timeStep)
{
	//ia = TI_bin as current TI bin to get appropriate topographic index properties
	int ia = TI_bin;

	//Initialize LAI_BAI_Tree_m2_p_m2 and LAI_BAI_SVeg_m2_p_m2 
	double LAI_BAI_Tree_m2_p_m2 = input->LAI_BAI_Tree_m2_p_m2[timeStep];
	double LAI_BAI_SVeg_m2_p_m2 = input->LAI_BAI_SVeg_m2_p_m2[timeStep];
	//Note: If LAI_BAI_Tree_m2_p_m2 < 1, then set to 1 so that its position as denominator does not enlarge TreeLEE_W_p_m2 when / LAI_BAI_Tree_m2_p_m2
	if (LAI_BAI_Tree_m2_p_m2 < 1) { LAI_BAI_Tree_m2_p_m2 = 1; }
	//Note: If LAI_BAI_SVeg_m2_p_m2  < 1, then set to 1 so that its position as denominator does not enlarge SVegLEE_W_p_m2 when / LAI_BAI_SVeg_m2_p_m2 
	if (LAI_BAI_SVeg_m2_p_m2 < 1) { LAI_BAI_SVeg_m2_p_m2 = 1; }

	//EvapoTranspiration_SoilEvapZone_m (m) reset to zero at start of each TI value
	double EvapoTranspiration_SoilEvapZone_m = 0;
	double Storage_EvapoTransZone_m = 0;
	//StorageDeficit_FieldCapacity_SoilEvapZone_Max_m (m) is the maximum soil deficit for capillarity held water for ET; = Evapotranspiration_Depth_m * (Soil_FieldCapacity_m3pm3 - Soil_WiltingPoint_m3pm3)
	double StorageDeficit_FieldCapacity_SoilEvapZone_Max_m = folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"];
	
	//If Model is StatisticalHydro or Model is SpatialTemperatureHydro And timeStep equals 1, then call statistical calculation based on weather input data, given the spatial calculation has not yet been called
	if (input->SimulationStringParams["Model_Selection"] == "StatisticalHydro" || ((input->SimulationStringParams["Model_Selection"] == "SpatialTemperatureHydro" || input->SimulationStringParams["Model_Selection"] == "SpatialBufferwGI") && timeStep == 0)) {
		//EvapoTranspiration_SoilEvapZone_m (m) = potential ET read in from Evaporation.csv input file
		//Note: PotentialEvapotranspiration_TreeCover_m (m) computed for tree, yet used for tree and pervious areas 
		//Note: ... This is a model simplification for the 1st timestep, while in the next time steps cover types are represented
		//Note: Depths of evapotranspiration are relative to the area of analysis, and do not get scaled by fraction of cover type
		EvapoTranspiration_SoilEvapZone_m = input->PotentialEvapotranspiration_TreeCover_m[timeStep];
	}
	//Else If input->SimulationStringParams["Model_Selection"] is "SpatialTemperatureHydro" or "SpatialBufferwGI" and timeStep > 0
	else {
		//Temperature_Air_C (C) is converted from folder Tair_K (K) 
		double Temperature_Air_C = folder->VarDict["Tair_K"] - 273.15;

		//Calculate HeatMetrics with call to HeatMetrics_Calc functions
		//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions with sending input
		HeatMetrics_Calc HeatMetricsCalc(input);

		//DensityWater_kgpm3 = HeatMetricsCalc.Density_Water_kgpm3_Calc(folder->VarDict["Tair_K"])
		double DensityWater_kgpm3 = HeatMetricsCalc.Density_Water_kgpm3_Calc(folder->VarDict["Tair_K"]);

		//LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(Tair_K)
		double LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(folder->VarDict["Tair_K"]);

		//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) initialized
		double EvapoTranspiration_SoilEvapZone_m_p_s = 0;

		//If Storage_PerviousPondedWater_m or Storage_ImperviousPondedWater_m > Constant_1E_negative6 then presume TreeLEE_W_p_m2 needs to be reduced due to some used to evaporate depression storage
		if (folder->VarDict["Storage_PerviousPondedWater_m"] > Constant_1E_negative6 || 
			folder->VarDict["Storage_ImperviousPondedWater_m"] > Constant_1E_negative6) {
			//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) derived as latent energy (W/m2) divided by (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3)
			//Note: If depression storage received TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2, then TreeLET_W_p_m2 - (TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) remains
			//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
			//Note: Latent energy is (SoilLE_W_p_m2 * SoilCover_noTreeCanopy_frac + (SVegLET_W_p_m2 - SVegLET_W_p_m2 / LAI_BAI_SVeg_m2_p_m2) * ShortVegCover_noTreeCanopy_frac + (TreeLET_W_p_m2 - TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) * TreeCover_frac), and ...
			//Note: OpenWaterEvaporationCalc::calculateImpDepEvapTemperature computes Evaporation_PerviousPondedWater_m_p_s using ...
			//Note: ... TreeLE_W_p_m2 / LAI_BAI_Tree_m2_p_m2 as energy allowed to evaporate ponded water under the tree canopy
			//Note: ... The TreeLET_W_p_m2 - (TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) remains available for evapotranspiration 
			//Note: ... The TreeLEE_W_p_m2 - (TreeLEE_W_p_m2 / LAI_BAI_Tree_m2_p_m2) remains available for canopy evaporation
			//Note: ... SVegLE_W_p_m2 / LAI_BAI_SVeg_m2_p_m2 as energy allowed to evaporate ponded water under the short veg canopy
			//Note: ... The SVegLET_W_p_m2 - (SVegLET_W_p_m2 / LAI_BAI_SVeg_m2_p_m2) remains available for evapotranspiration
			//Note: ... The SVegLEE_W_p_m2 - (SVegLEE_W_p_m2 / LAI_BAI_SVeg_m2_p_m2) remains available for canopy evaporation
			//Note: TreeCover_frac vs TreeCanopyCover_overPervious_frac is used given tree cover over pervious & impervious areas have roots within soil layer 
			//Note: Soils covered by impervious may receive recharge from groundwater via lateral redistribution.
			//Note: RHS numerator is W/m2 = J/s/m2, and RHS denominator is (J/kg) * (kg/m3) = J/m3, therefore RHS quotient is J/s/m2 / J/m3 = m/s.
			EvapoTranspiration_SoilEvapZone_m_p_s = (folder->VarDict["SoilLE_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "SoilCover_noTreeCanopy_frac") + folder->VarDict["ImpLE_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "PermeablePavementCover_noTreeCanopy_frac") + (folder->VarDict["SVegLET_W_p_m2"] - input->SafeDivide(folder->VarDict["SVegLET_W_p_m2"], LAI_BAI_SVeg_m2_p_m2)) * beC->by_key(DataDrawer_ID, DataFolder_ID, "ShortVegCover_noTreeCanopy_frac") + (folder->VarDict["TreeLET_W_p_m2"] - input->SafeDivide(folder->VarDict["TreeLET_W_p_m2"], LAI_BAI_Tree_m2_p_m2)) * beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCover_frac")) / (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3);
		}
		//Else Storage_PerviousPondedWater_m <= Constant_1E_negative6 and use full amount of TreeLEE_W_p_m2 
		else {
			//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) derived as latent energy (W/m2) divided by (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3)
			//Note: If depression storage received TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2, then TreeLET_W_p_m2 - (TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) remains
			//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
			//Note: Latent energy is (SoilLE_W_p_m2 * SoilCover_noTreeCanopy_frac + SVegLET_W_p_m2 * ShortVegCover_noTreeCanopy_frac + ...
			//Note: ... TreeLET_W_p_m2 * TreeCover_frac), and ...
			//Note: TreeCover_frac is used given tree cover over pervious and impervious areas have roots within the soil layer 
			//Note: Soils covered by impervious may receive recharge from groundwater via lateral redistribution.
			//Note: RHS numerator is W/m2 = J/s/m2, and RHS denominator is (J/kg) * (kg/m3) = J/m3, therefore RHS quotient is J/s/m2 / J/m3 = m/s.
			EvapoTranspiration_SoilEvapZone_m_p_s = (folder->VarDict["SoilLE_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "SoilCover_noTreeCanopy_frac") + folder->VarDict["ImpLE_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "PermeablePavementCover_noTreeCanopy_frac") + folder->VarDict["SVegLET_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "ShortVegCover_noTreeCanopy_frac") + folder->VarDict["TreeLET_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCover_frac")) / (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3);
		}

		//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) derived as latent energy (W/m2) divided by (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3)
		//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
		//Note: Latent energy is (SoilLE_W_p_m2 * SoilCover_noTreeCanopy_frac + SVegLET_W_p_m2 * ShortVegCover_noTreeCanopy_frac + ...
		//Note: ... TreeLET_W_p_m2 + TreeCover_frac), and ...
		//Note: TreeCover_frac is used given tree cover over pervious and impervious areas have roots within the soil layer 
		//Note: Soils covered by impervious may receive recharge from groundwater via lateral redistribution.
		//Note: RHS numerator is W/m2 = J/s/m2, and RHS denominator is (J/kg) * (kg/m3) = J/m3, therefore RHS quotient is J/s/m2 / J/m3 = m/s.
		EvapoTranspiration_SoilEvapZone_m_p_s = (folder->VarDict["SoilLE_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "SoilCover_noTreeCanopy_frac") + folder->VarDict["ImpLE_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "PermeablePavementCover_noTreeCanopy_frac") + folder->VarDict["SVegLET_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "ShortVegCover_noTreeCanopy_frac") + folder->VarDict["TreeLET_W_p_m2"] * beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCover_frac")) / (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3);

		//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) next divided by pervious area + tree over impervious so it is scaled to cover area
		EvapoTranspiration_SoilEvapZone_m_p_s = input->SafeDivide(EvapoTranspiration_SoilEvapZone_m_p_s, (beC->by_key(DataDrawer_ID, DataFolder_ID, "SoilCover_noTreeCanopy_frac") + beC->by_key(DataDrawer_ID, DataFolder_ID, "PermeablePavementCover_noTreeCanopy_frac") + beC->by_key(DataDrawer_ID, DataFolder_ID, "ShortVegCover_noTreeCanopy_frac") + beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCover_frac")));

		//Control for negative values
		EvapoTranspiration_SoilEvapZone_m_p_s = Inputs::forceZeroIfLessThanOrAlmostEqualZero(EvapoTranspiration_SoilEvapZone_m_p_s, Epsilon_Tolerance_1E_negative15);

		//EvapoTranspiration_SoilEvapZone_m (m) is product of EvapoTranspiration_SoilEvapZone_m_p_s (m/s) and SimulationTimeStep_Duration_sec[timeStep]
		EvapoTranspiration_SoilEvapZone_m = EvapoTranspiration_SoilEvapZone_m_p_s * input->SimulationTimeStep_Duration_sec[timeStep];
	}

	//If Type equals GI do not proceed with updates to topographic index (TI) terms
	//Note: GI or non-BulkArea folder does not use TI_bin value that is passed in function
	//Note: Consider Refactor to implement soil water availability constraints of modified water retention curve, as for BulkArea
	if (folder->ParamStringDict["Type"] != "BulkArea") {
		//Storage_EvapoTransZone_m (m) = GI water depth available for ET, defined in Inflow_StormwaterDevice.cpp as folder->VarDict["Storage_EvapoTransZone_m"]
		Storage_EvapoTransZone_m = folder->VarDict["Storage_EvapoTransZone_m"];
		//EvapoTranspiration_SoilEvapZone_m (m) equals EvapoTranspiration_SoilEvapZone_m, potential ET 
		folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] = EvapoTranspiration_SoilEvapZone_m;

		//If the EvapoTranspiration_SoilEvapZone_m (m) is greater than Storage_EvapoTransZone_m (m), then reset to available soil evapotranspiration zone water
		if (folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] > Storage_EvapoTransZone_m) {
			//EvapoTranspiration_SoilEvapZone_TI_m (m) is limited to Storage_EvapoTransZone_m (m), available soil evapotranspiration zone water
			folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] = Storage_EvapoTransZone_m;
		}
	}

	//if Type equals BulkArea then compute Storage_EvapoTransZone_m as function of StorageDeficit_FieldCapacity_SoilEvapZone_TI_m
	if (folder->ParamStringDict["Type"] == "BulkArea") {

		//Initialize vectors at start of simulation 
		if (timeStep == 0) {
			//If EvapoTranspiration_SoilEvapZone_TI_m size() is less than or equal to ia then populate vector with zero
			if (folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"].size() <= ia) {
				//EvapoTranspiration_SoilEvapZone_TI_m (m) is soil evapotranspiration, initialized to 0
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"].push_back(0.0);
			}
		}
		//EvapoTranspiration_SoilEvapZone_TI_m (m) reset to zero at start of each TI_Value vector
		folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = 0.0;

		//if EvapoTranspiration_SoilEvapZone_m > 0 then remove water from SoilEvapZone
		if (EvapoTranspiration_SoilEvapZone_m > 0.0) {
			//If StorageDeficit_FieldCapacity_SoilEvapZone_Max_m greater than zero, then enter for division by StorageDeficit_FieldCapacity_SoilEvapZone_Max_m
			if (folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"] > 0) {
				//EvapoTranspiration_SoilEvapZone_TI_m (m) is EvapoTranspiration_SoilEvapZone_m * (1 - (StorageDeficit_FieldCapacity_SoilEvapZone_TI_m / StorageDeficit_FieldCapacity_SoilEvapZone_Max_m))
				//Note: Equal to EvapoTranspiration_SoilEvapZone_m when StorageDeficit_FieldCapacity_SoilEvapZone_TI_m = 0, no capacity remains
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = EvapoTranspiration_SoilEvapZone_m * (1 - (folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] / folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"]));
				//Storage_EvapoTransZone_m (m) = StorageDeficit_FieldCapacity_SoilEvapZone_Max_m - StorageDeficit_FieldCapacity_SoilEvapZone_TI_m; 
				//	Example: StorageDeficit_FieldCapacity_SoilEvapZone_Max_m = 0.1 m, StorageDeficit_FieldCapacity_SoilEvapZone_TI_m = 0.08 m, then Storage_EvapoTransZone_m = 0.1 - 0.08 = 0.02 m
				Storage_EvapoTransZone_m = folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"] - folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia];
			}
			else {
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = 0;
				//Storage_EvapoTransZone_m (m) equals zero
				Storage_EvapoTransZone_m = 0;
			}

			//cte 2025 Paused refactor to represent soil water availability constraints via modified water retention curve
			//... constrain Saturation_SoilStorage_frac based modified water retention curve, WRC
			//Call RootZoneEvapoTranspirationCalc::SoilWaterRetentionCurve(input, folder, TI_bin, timeStep)
			//Note: Theta_WRC_ETZ_scalingFactor_frac represents effect of WRC on decreasing soil water availability in profile
			//cte 2015 RootZoneEvapoTranspirationCalc::SoilWaterRetentionCurve(input, folder, TI_bin, timeStep);

			//EvapoTranspiration_SoilEvapZone_TI_m = Theta_WRC_ETZ_scalingFactor_frac * EvapoTranspiration_SoilEvapZone_TI_m
			//Note: Theta_WRC_ETZ_scalingFactor_frac represents how water retention curve decreases soil water in profile
			//cte 2025 folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = folder->ParamDict["Theta_WRC_ETZ_scalingFactor_frac"] * folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia];

			//If the EvapoTranspiration_SoilEvapZone_TI_m (m) > Storage_EvapoTransZone_m (m), then reset to available soil evapotranspiration zone water
			if (Inputs::isGreaterThan(folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia], Storage_EvapoTransZone_m)) {
				//EvapoTranspiration_SoilEvapZone_TI_m (m) is limited to Storage_EvapoTransZone_m (m), available soil evapotranspiration zone water
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = Storage_EvapoTransZone_m;
			}
		}

		//StorageDeficit_FieldCapacity_SoilEvapZone_TI_m (m) is increased by removal of EvapoTranspiration_SoilEvapZone_TI_m
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_TI_m decreases due to Infiltration_TI_m in DrainageToEvapotranspirationZone.cpp
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_TI_m is a deficit, aka capacity; all TI bins have same thickness
		folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] = folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] + folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia];

		//EvapoTranspiration_SoilEvapZone_m (m) = sum EvapoTranspiration_SoilEvapZone_TI_m for each TI bin for time step, weighted by watershed fraction
		//	HydroPlus weights discrete TI parameter values, i.e, SoilParameter[ia], with TI_binArea_frac, catchment area bounding discrete value
		folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] = folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] + folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] * folder->VarDict["TI_binArea_frac"];

		//StorageDeficit_FieldCapacity_SoilEvapZone_m (m) = sum StorageDeficit_FieldCapacity_SoilEvapZone_TI_m 
		//Note: Summing for each TI bin for time step, weighted by watershed fraction, TI_binArea_frac
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_m ranges frm 0=saturated, 1=dry; available storage up to field capacity
		//HydroPlus weights discrete TI parameter values, i.e, SoilParameter[ia], with TI_binArea_frac, catchment area bounding discrete value
		folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_m"] = folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_m"] + folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] * folder->VarDict["TI_binArea_frac"];
	}
	//Ending of BulkArea calculations
}

//cte 2025: Exploring potential refactor with this function; it has never been active for any HydroPlus test cases
//... It may make sense to only turn this on in periods after rain, and/or when Theta_Sat_frac < 0.15

//RootZoneEvapoTranspirationCalc::SoilWaterRetentionCurve function handles StatisticalHydro and SpatialTemperatureHydro models
//Theory: Soil water retention curves represent the relationship between the amount of water in the soil and the soil's matric potential
//Theory: Multi-parameter water retention curves were created by Brooks and Corey (1964), van Genuchten (1980), and Saxton and Rawls (2006)
//Note: HydroPlus uses soil water retention curve theory to represent how soil columns dry from top to bottom under evapotranspiration
//Note: Given HydroPlus does not simulate matric potential and uses a uniform vadose zone above the saturated zone, theoretical curves do not fit
//Note: Instead HydroPlus uses a modified water retention curve that is linear to approximate the capillary water above the wilting point
//Note: Consider refactoring to use pedotransfer functions and theoretical curves when resources become available for HydroPlus
//References:
//Brooks, R. H., & Corey, A. T. (1964). Hydraulic Properties of Porous Media Affecting Fluid Flow. Journal of Irrigation and Drainage Engineering, ASCE, 92(IR2), 61-68. 
//Saxton, K. E., & Rawls, W. J. (2006). Soil Water Characteristic Estimates by Texture and Organic Matter for Hydrologic Solutions. Soil Science Society of America Journal, 70(5), 1569-1578. 
//van Genuchten, M. T. (1980). A Closed-form Equation for Predicting the Hydraulic Conductivity of Unsaturated Soils. Soil Science Society of America Journal, 44(5), 892-898. 
void RootZoneEvapoTranspirationCalc::SoilWaterRetentionCurve(Inputs* input, CompactRagged* beC, DataFolder* folder, int TI_bin, int MapPixel_ID, int DataFolder_ID, int timeStep)
{
	//ia = TI_bin as current TI bin to get appropriate topographic index properties
	int ia = TI_bin;

	//Theta_Sat_frac = 1 - (StorageDeficit_FieldCapacity_SoilEvapZone_TI_m[ia] / StorageDeficit_FieldCapacity_SoilEvapZone_Max_m)
	//Note: Represents soil water from dry (0) to saturated (1)
	double Theta_Sat_frac = 1 - Inputs::SafeDivide(folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia], folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"]);

	//Depth_ETZ_m = beC->by_key(MapPixel_ID, DataFolder_ID, "Evapotranspiration_Depth_m"); coordinate and simplify variable naming for function
	double Depth_ETZ_m = beC->by_key(MapPixel_ID, DataFolder_ID, "Evapotranspiration_Depth_m");

	//Depth_ETZ_max_m = max(input->InputXml["Evapotranspiration_Depth_max_m"], 1.0); where 1.0 m is considered a typical depth for WRC analysis
	//Note: Evapotranspiration_Depth_max_m defined as max of EvapotranspirationDepth_TreeCover_m, EvapotranspirationDepth_SVegCover_m, EvapotranspirationDepth_SoilCover_m in BuildDataOrganizer
	double Depth_ETZ_max_m = max(input->InputXml["Evapotranspiration_Depth_max_m"], 1.0);

	//Depth_ETZ_max_m = max(Depth_ETZ_m, Depth_ETZ_max_m)
	//Note: Ensures Depth_ETZ_max_m is no less than Evapotranspiration_Depth_m based on HydroPlusConfig.xml inputs
	Depth_ETZ_max_m = max(Depth_ETZ_m, Depth_ETZ_max_m);

	//WRC_Point_E_x, WRC_Point_E_z, WRC_Point_G_x intialized
	double WRC_Point_E_x, WRC_Point_E_z, WRC_Point_G_x;
	//if (Inputs::isGreaterThan(Theta_Sat_frac, 0.5)) then compute Case 1 of WRC_Point_E_x and WRC_Point_E_z
	if (Inputs::isGreaterThan(Theta_Sat_frac, 0.5)) {
		//WRC_Point_E_x = 1.0 - 2.0 * (1.0 - Theta_Sat_frac); fits diagram at left below
		 WRC_Point_E_x = 1.0 - 2.0 * (1.0 - Theta_Sat_frac);
		//WRC_Point_E_z = 0; fits diagram at left below 
		WRC_Point_E_z = 0; 
	} 
	//else Theta_Sat_frac <= 0.5, then compute Case 2 of WRC_Point_E_x and WRC_Point_E_z
	else {
		//WRC_Point_E_x = 2.0 * Theta_Sat_frac; fits diagram at right below
		WRC_Point_E_x = 2.0 * Theta_Sat_frac;
		//WRC_Point_E_z = Depth_ETZ_max_m; fits diagram at right below 
		WRC_Point_E_z = Depth_ETZ_max_m;
	}

	//WRC_Point_A_x = 0; fits diagrams below
	double WRC_Point_A_x = 0;
	//WRC_Point_A_z = 0; fits diagrams below
	double WRC_Point_A_z = 0;
	//WRC_Point_B_x = 1; fits diagrams below
	double WRC_Point_B_x = 1;
	//WRC_Point_C_x = WRC_Point_B_x; fits diagrams below
	double WRC_Point_C_x = WRC_Point_B_x;
	//WRC_Point_C_z = Depth_ETZ_max_m; fits diagrams below
	double WRC_Point_C_z = Depth_ETZ_max_m;
	//WRC_Point_D_z = WRC_Point_C_z ; fits diagrams below
	double WRC_Point_D_z = WRC_Point_C_z;
	//WRC_Point_G_z = Depth_ETZ_m; fits diagrams below
	double WRC_Point_G_z = Depth_ETZ_m;
	//WRC_Point_F_z = Depth_ETZ_m; fits diagrams below
	double WRC_Point_F_z = Depth_ETZ_m;

	//if (Inputs::isGreaterThan(Theta_Sat_frac, 0.5)) then compute Case 1 of WRC_Point_G_x
	if (Inputs::isGreaterThan(Theta_Sat_frac, 0.5)) {
		//WRC_Point_G_x = WRC_Point_C_x - WRC_Point_G_z * ((WRC_Point_C_x - WRC_Point_E_x) / (WRC_Point_C_z - WRC_Point_E_z)); interpolation of WRC
		WRC_Point_G_x = WRC_Point_C_x - WRC_Point_G_z * Inputs::SafeDivide((WRC_Point_C_x - WRC_Point_E_x), (WRC_Point_C_z - WRC_Point_E_z));
	}
	//else Theta_Sat_frac <= 0.5, then compute Case 2 of WRC_Point_G_x
	else {
		//WRC_Point_G_x = WRC_Point_A_x + WRC_Point_G_z * ((WRC_Point_A_x - WRC_Point_E_x) / (WRC_Point_A_z - WRC_Point_E_z)); interpolation of WRC
		WRC_Point_G_x = WRC_Point_A_x + WRC_Point_G_z * Inputs::SafeDivide((WRC_Point_A_x - WRC_Point_E_x), (WRC_Point_A_z - WRC_Point_E_z));
	}

	//WRC_ETZ_m initialized
	double WRC_ETZ_m;

	//if (Inputs::isGreaterThan(Theta_Sat_frac, 0.5)) then compute Case 1 of trapezoid AEGF  
	if (Inputs::isGreaterThan(Theta_Sat_frac, 0.5)) {
		//WRC_ETZ_m = 0.5 * (WRC_Point_E_x + WRC_Point_G_x) * WRC_Point_F_z
		//Note: Area of trapezoid is A_trapezoid = 0.5 * (sum of parallel sides) * height
		WRC_ETZ_m = 0.5 * (WRC_Point_E_x + WRC_Point_G_x) * WRC_Point_F_z;
	}
	//else Theta_Sat_frac <= 0.5, then compute Case 2 of triangle AGF or WRC_ETZ_m = 0.5*G(x)*F(z), where F(z) = Depth_ETZ_m
	else {
		//WRC_ETZ_m = 0.5 * WRC_Point_G_x * WRC_Point_F_z
		//Note: Area of trapezoid is A_triangle = 0.5 * base * height
		WRC_ETZ_m = 0.5 * (WRC_Point_E_x + WRC_Point_G_x) * WRC_Point_F_z;
	}

	//WRC_max_m = WRC_Point_B_x * WRC_Point_D_z; compute area of WRC rectangle
	//Note: Area of rectangle is A_rectangle = base * height
	double WRC_max_m = WRC_Point_B_x * WRC_Point_D_z;

	//Theta_WRC_ETZ_scalingFactor_frac = WRC_ETZ_m / WRC_max_m; scaling factor to approximate WRC effect on soil water availability
	double Theta_WRC_ETZ_scalingFactor_frac = Inputs::SafeDivide(WRC_ETZ_m, WRC_max_m);

	double Theta_Depth_ETZ_scalingFactor_frac = Inputs::SafeDivide(Depth_ETZ_m, Depth_ETZ_max_m);

	Theta_WRC_ETZ_scalingFactor_frac = clamp(Theta_WRC_ETZ_scalingFactor_frac + Theta_Sat_frac, 0.0, 1.0);

	Theta_Depth_ETZ_scalingFactor_frac = clamp(Theta_Depth_ETZ_scalingFactor_frac + Theta_Sat_frac, 0.0, 1.0);

	//folder->ParamDict["Theta_WRC_ETZ_scalingFactor_frac"] = Theta_Depth_ETZ_scalingFactor_frac;

	if (input->isReferenceStationPixel(MapPixel_ID)) {
		//cte 2025 folder->ParamDict["Theta_WRC_ETZ_scalingFactor_frac"] = 1;
	}

	//Theta_WRC_ETZ_scalingFactor_frac = Depth_ETZ_m / Depth_ETZ_max_m; scaling factor to approximate WRC effect on soil water availability
	//cte 2025 folder->ParamDict["Theta_WRC_ETZ_scalingFactor_frac"] = Theta_Depth_ETZ_scalingFactor_frac;


	//cte 2025
	/*
	cout << "TimeStep " << timeStep << " TI_bin " << TI_bin << endl;
	cout << "RZ WRC: Theta_Sat_frac= " << Theta_Sat_frac << endl;
	cout << "RZ WRC: Theta_WRC_ETZ_scalingFactor_frac= " << Theta_WRC_ETZ_scalingFactor_frac << endl;
	cout << "RZ WRC: Theta_Depth_ETZ_scalingFactor_frac= " << Theta_Depth_ETZ_scalingFactor_frac << endl;
	cout << "RZ WRC: WRC_Point_E_x= " << WRC_Point_E_x << endl;
	cout << "RZ WRC: WRC_Point_F_z= " << WRC_Point_F_z << endl;
	cout << "RZ WRC: WRC_Point_G_x= " << WRC_Point_G_x << endl;
	cout << "RZ WRC: Depth_ETZ_m= " << Depth_ETZ_m << endl;
	cout << "RZ WRC: Depth_ETZ_max_m= " << Depth_ETZ_max_m << endl;
	cout << "RZ WRC: WRC_ETZ_m= " << WRC_ETZ_m << endl;
	cout << "RZ WRC: WRC_max_m= " << WRC_max_m << endl;
	cout << "RZ WRC: Theta_WRC_ETZ_scalingFactor_frac= " << folder->ParamDict["Theta_WRC_ETZ_scalingFactor_frac"] << endl;
	cout << endl;
	*/

	/*
	* Illustration of water retention curve (WRC) under two moisture conditions, theta saturation fraction (Theta_Sat_frac) > 0.5 and <= 0.5
	* WRC is anchored to bottom right but moves along top for Theta_sat_frac > 0.5, within a unit range of 1
	* WRC is anchored to top left but moves along bottom for Theta_sat_frac <= 0.5, within a unit range of 1
	* Vertices A (top left), B (top right), C (bottom right), and D (bottom left) bound the WRC within a rectangle in x-z coordinate space
	* The x-axis is relative soil saturation level, Theta_Sat_frac and goes from 0 (left) to 1 
	* The z-axis is effective depth of evapotranspiration zone, Depth_ETZ_m, and goes from 0 (top) to Depth_ETZ_max_m
	* Point E represents the WRC tip as it passes through the x-axis for the TI bin or pixel, at top (Theta_Sat_frac > 0.5) or bottom (Theta_Sat_frac <= 0.5) 
	* Point F represents the depth of the actual Depth_ETZ_m for the TI bin or pixel
	* Point G represents the WRC segment as it passes through the Depth_ETZ_m for the TI bin or pixel
	* 
	* Define Terms:
	* 
	* Theta_Sat_frac = 1 - (folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"] / folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia]
	* 
	* Depth_to_Groundwater_Table_TI_m = folder->VarVecDict["StorageDeficit_VadoseZone_TI_m"][ia] / input->InputXml["Porosity_Drainable_m3pm3"];
	* 
	* Depth_ETZ_max_m = maximum(EvapotranspirationDepth_TreeCover_m, EvapotranspirationDepth_SVegCover_m, EvapotranspirationDepth_SoilCover_m, 1 m)
	* Depth_ETZ_max_m = minimum(Depth_to_Groundwater_Table_TI_m, Depth_ETZ_max_m); places Line DE above saturation
	* Depth_ETZ_max_m = maximum(beC->by_key(MapPixel_ID, DataFolder_ID, "Evapotranspiration_Depth_m"), Depth_ETZ_max_m); ensures Point F < Point D
	* 
	* Note: Last Depth_ETZ_max_m adjustment avoids resetting user inputs used to compute Evapotranspiration_Depth_m, and ...
	* Note: 1) Evapotranspiration_Depth_m & StorageDeficit_FieldCapacity_SoilEvapZone_TI_m can extend beneath the groundwater table, ...
	* Note: ... as it represents capillary pore storage, while StorageDeficit_VadoseZone_TI_m represents gravitational pore storage
	* Note: 2) If Evapotranspiration_Depth_m & StorageDeficit_FieldCapacity_SoilEvapZone_TI_m extend beneath the groundwater table, ...
	*
	* Find Point E(x,z) via scaling to length 0 to 1 from 0.5 to 1 (Theta_Sat_frac > 0.5) or from 0 to 0.5 (Theta_Sat_frac <= 0.5): 
	* If Theta_Sat_frac > 0.5, then E(x) = [1-2*(1-Theta_Sat_frac)], E(z) = 0
	* If Theta_Sat_frac <= 0.5, then E(x) = 2*Theta_Sat_frac, E(z) = Depth_ETZ_max_m
	* 
	* Find Point G(x,z) via interpolation between E and C (Theta_Sat_frac > 0.5) or E can A (Theta_Sat_frac <= 0.5):
	* If Theta_Sat_frac > 0.5, then G(x) = C(x) - G(z)*[(C(x) - E(x)) / (C(z) - E(z))], G(z) = Depth_ETZ_m
	* If Theta_Sat_frac <= 0.5, then G(x) = A(x) + G(z)*[(A(x) - E(x)) / (A(z) - E(z))], G(z) = Depth_ETZ_m
	* 
	* Find unit Area of evapotranspiration zone available water for Theta_Sat_frac > 0.5 or Theta_Sat_frac <= 0.5
	* If Theta_Sat_frac > 0.5, then trapezoid AEGF or WRC_ETZ_m = 0.5*[E(x) + G(x)]*F(z), where F(z) = Depth_ETZ_m
	* If Theta_Sat_frac <= 0.5, then triangle AGF or WRC_ETZ_m = 0.5*G(x)*F(z), where F(z) = Depth_ETZ_m
	* 
	* Scale the unit Area of evapotranspiration zone available water, WRC_ETZ_m, based on ...
	* ... unit Area of WRC maximum capillary water capacity within rectangle ABCD, as WRC_max_m = B(x)*C(z)
	* Creating WRC scaling factor for evapotranspiration zone water, Theta_WRC_ETZ_scalingFactor_frac = WRC_ETZ_m / WRC_max_m
	* 
	* Constrain Storage_EvapoTransZone_m water using the WRC scaling factor:
	* folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = Theta_WRC_ETZ_scalingFactor_frac * folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia]
	*
	* 
	* Theta_Sat_frac > 0.5			  Theta_Sat_frac <= 0.5
	A----------\E-----------B		A-----------------------B	Depth = 0
	|		 <--\-->		|		| \						|
	|			 \			|		|  \					|
	|			  \			|		|	\					|
	F==============\G=======|		F====\G=================|	Depth = Depth_ETZ_m
	|				\		|		|	  \					|
	|				 \		|		|	   \				|
	|				  \		|		|		\				|
	|				   \	|		|		 \				|
	|					\	|		|		  \				|
	|					 \	|		|		   \			|
	|					  \	|		|		 <--\-->		|
	D-----------------------C		D------------\E---------C	Depth = Depth_ETZ_max_m
		
	*/

}