#include "DynamicFunctions.h"
#include "../Outputs/WaterQuality_Output.h"
#include <algorithm>
#include <math.h>

//Author(s) : Li Zhang 
//lzhan126@syr.edu Feb 2021

//#include <boost/math/distributions/normal.hpp>


double DynamicFunctions::ecdf(double eventdata, Inputs* input)
{
    //Number of elements < event data
    double num=0;
    double ecdf = 0.00;
    std::vector<double> HydrologicData_Record = input->DischargeRecord;

    for (auto it = HydrologicData_Record.begin(); it != HydrologicData_Record.end(); ++it) {
        if (*it <= eventdata) { ++num; }
    }
    //ecdf range from 0-1
    ecdf = num / HydrologicData_Record.size();
    return ecdf;
}


double DynamicFunctions::quantile(const std::vector<double>& datavector, double Probability_Cummulative_frac, Inputs* input)
{
    //https://www.rdocumentation.org/packages/stats/versions/3.6.2/topics/quantile
    int HydrologicData_size = 0;
    int HydrologicVariable_rank_floor = 0;
    double Probability_Adjustment_frac = 0.0;
    double HydrologicVariable_rank_decimal = 0.0;
    double HydrologicVariable_QuantileValue_SourceUnits = 0.0;

    //HydrologicData_size is the sample size
    HydrologicData_size = int(datavector.size());
    //For types 4 through 9, m given below
    //type4: linear interpolation of the empirical cdf
    if (input->BufferTemporalLoadingNumericalParams["QuantileType"] == 4) {
        Probability_Adjustment_frac = 0;
    }
    //type5: piecewise linear function where the knots are the values midway through the steps of the empirical cdf
    if (input->BufferTemporalLoadingNumericalParams["QuantileType"] == 5) {
        Probability_Adjustment_frac = 0.5;
    }
    //type6: used by Minitaband by SPSS.
    if (input->BufferTemporalLoadingNumericalParams["QuantileType"] == 6) {
        Probability_Adjustment_frac = Probability_Cummulative_frac;
    }
    //type7: This is used by S.
    if (input->BufferTemporalLoadingNumericalParams["QuantileType"] == 7) {
        Probability_Adjustment_frac = 1 - Probability_Cummulative_frac;
    }
    //type8: The resulting quantile estimates are approximately median-unbiased regardless of the distribution of x
    if (input->BufferTemporalLoadingNumericalParams["QuantileType"] == 8) {
        Probability_Adjustment_frac = (Probability_Cummulative_frac + 1)/3;
    }
    //type9: The resulting quantile estimates are approximately unbiased for the expected order statistics if x is normally distributed
    if (input->BufferTemporalLoadingNumericalParams["QuantileType"] == 9) {
        Probability_Adjustment_frac = Probability_Cummulative_frac/4 + 3/8;
    }
    
    HydrologicVariable_rank_floor = int(floor(HydrologicData_size * Probability_Cummulative_frac + Probability_Adjustment_frac));
    HydrologicVariable_rank_decimal = HydrologicData_size * Probability_Cummulative_frac + Probability_Adjustment_frac - HydrologicVariable_rank_floor;
    
    //HydrologicVariable_rank_floor = order number; 
    //insert one more data (0) at the beginning of the vector, so we can use the same order as R code: HydrologicVariable_rank_floor & HydrologicVariable_rank_floor+1
    //if we remove this line, since the vector begin from HydrologicVariable_rank_floor=0, HydrologicVariable_rank_floor & HydrologicVariable_rank_floor+1 in the equation-> HydrologicVariable_rank_floor-1 & HydrologicVariable_rank_floor
    //VectorSorted.insert(VectorSorted.begin(), 0);
    //HydrologicVariable_QuantileValue_SourceUnits = (1 - HydrologicVariable_rank_decimal) * VectorSorted[HydrologicVariable_rank_floor] + HydrologicVariable_rank_decimal * VectorSorted[HydrologicVariable_rank_floor+1];
    if (HydrologicVariable_rank_floor == int(datavector.size())) {
        HydrologicVariable_QuantileValue_SourceUnits = (1 - HydrologicVariable_rank_decimal) * datavector[HydrologicVariable_rank_floor - 1];
    }
    else {
        HydrologicVariable_QuantileValue_SourceUnits = (1 - HydrologicVariable_rank_decimal) * datavector[HydrologicVariable_rank_floor - 1] +
            HydrologicVariable_rank_decimal * datavector[HydrologicVariable_rank_floor];
    }

     return HydrologicVariable_QuantileValue_SourceUnits;
}

vector< vector<double> > DynamicFunctions::pollutant_landcover_list;
void DynamicFunctions::Interpolation(Inputs* input) {
    double Q1st, Q2nd, Qlast, Qlast2nd,slope_Q_begin, slope_Q_end,EC_begin, EC_end;
    std::vector<double> HydrologicData_Record = input->DischargeRecord;
    std::vector<double> DataRecordSorted(HydrologicData_Record.size());
    //sort the discharge record
    partial_sort_copy(HydrologicData_Record.begin(), HydrologicData_Record.end() , DataRecordSorted.begin(), DataRecordSorted.end());
    //use the quantile function to get the HydrologicVariable_QuantileValue_SourceUnits at the tails
    Q1st= quantile(DataRecordSorted, 0, input);//0th
    Qlast = quantile(DataRecordSorted, 1, input);//100th
    Q2nd = quantile(DataRecordSorted, 0.05, input);//5th
    Qlast2nd = quantile(DataRecordSorted, 0.95, input);//95th
    //find the slope at the tails
    if (Q2nd == 0) {
        slope_Q_begin = 0;
    }
    else slope_Q_begin = Q1st / Q2nd;

    if (Qlast2nd == 0) {
        slope_Q_end = 0;
    }
    else slope_Q_end = Qlast / Qlast2nd;
    //creating list of vectors
    //Total Phosphorus
    //Urban
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TP"]);//21
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TP"]);//22
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TP"]);//23
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TP"]);//24
    //Forest
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TP"]);//41
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TP"]);//42
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TP"]);//43
    //Rangeland
    pollutant_landcover_list.push_back(input->EC_vec_map["Rangeland_TP"]);//71
    //Pasture_hay
    pollutant_landcover_list.push_back(input->EC_vec_map["Pasture_hay_TP"]);//81
    //Cropland
    pollutant_landcover_list.push_back(input->EC_vec_map["Cropland_TP"]);//82

    //Total Nitrogen 
    //Urban
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TN"]);//21
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TN"]);//22
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TN"]);//23
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TN"]);//24
    //Forest
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TN"]);//41
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TN"]);//42
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TN"]);//43
    //Rangeland
    pollutant_landcover_list.push_back(input->EC_vec_map["Rangeland_TN"]);//71
    //Pasture_hay
    pollutant_landcover_list.push_back(input->EC_vec_map["Pasture_hay_TN"]);//81
    //Cropland
    pollutant_landcover_list.push_back(input->EC_vec_map["Cropland_TN"]);//82

    //total suspended solids
    //Urban
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TSS"]);//21
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TSS"]);//22
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TSS"]);//23
    pollutant_landcover_list.push_back(input->EC_vec_map["Urban_TSS"]);//24
    //Forest
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TSS"]);//41
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TSS"]);//42
    pollutant_landcover_list.push_back(input->EC_vec_map["Forest_TSS"]);//43
    //Rangeland
    pollutant_landcover_list.push_back(input->EC_vec_map["Rangeland_TSS"]);//71
    //Pasture_hay
    pollutant_landcover_list.push_back(input->EC_vec_map["Pasture_hay_TSS"]);//81
    //Cropland
    pollutant_landcover_list.push_back(input->EC_vec_map["Cropland_TSS"]);//82


    for (auto list_it = pollutant_landcover_list.begin(); list_it != pollutant_landcover_list.end(); ++list_it) {
        //vector<double> &V = *list_it;
        int num_insert = 0;
        EC_begin = (*list_it)[0] * slope_Q_begin;
        EC_end = (*list_it).back() * slope_Q_end;
        (*list_it).insert((*list_it).begin(), EC_begin);
        (*list_it).insert((*list_it).end(), EC_end);
        //now we have 0th, 5th, 10th, 25th, 50th, 75th, 90th, 95th, 100th (9 in total)
        //insert 12 new values using linear interpolation, to have 5th interval (21 in total)
        int interval = 5;
        //filling the 1st gap
        double x_increment1 = 15;//25-10
        double y_increment1 = (*list_it)[3] - (*list_it)[2];
        double slope1 = y_increment1 / x_increment1;
        for (auto i = 0; i < x_increment1 / interval-1; ++i) {//insert 2 times
            //inserts EC at (3+i)-th location, for this loop, we add 2 values, add at begin() + 3 and begin() + 4
            auto it = (*list_it).insert((*list_it).begin() + 3+i, (*list_it)[2]+slope1* interval*(i+1));
        }
        num_insert += int(x_increment1 / interval - 1);
        
        //filling the 2nd gap
        double x_increment2 = 25;//50-25
        double y_increment2 = (*list_it)[3 + num_insert+1] - (*list_it)[3+num_insert ];
        double slope2 = y_increment2 / x_increment2;
        for (auto i = 0; i < x_increment2 / interval - 1; ++i) {//insert 4 times
            auto it = (*list_it).insert((*list_it).begin() + 4+ num_insert + i, (*list_it)[3+ num_insert]+ slope2 * interval * (i + 1));
        }
        num_insert += int(x_increment2 / interval - 1);

        //filling the 3rd gap
        double x_increment3 = 25;//75-50
        double y_increment3 = (*list_it)[4+ num_insert+1] - (*list_it)[4 + num_insert];
        double slope3 = y_increment3 / x_increment3;
        for (auto i = 0; i < x_increment3 / interval - 1; ++i) {//insert 4 times
            auto it = (*list_it).insert((*list_it).begin() + 5 + num_insert + i, (*list_it)[4+ num_insert] + slope3 * interval * (i + 1));
        }
        num_insert += int(x_increment3 / interval - 1);

        //filling the 4rd gap
        double x_increment4 = 15;//90-75
        double y_increment4 = (*list_it)[5 + num_insert + 1] - (*list_it)[5 + num_insert];
        double slope4 = y_increment4 / x_increment4;
        for (auto i = 0; i < x_increment4 / interval - 1; ++i) {//insert 2 times
            auto it = (*list_it).insert((*list_it).begin() + 6 + num_insert + i, (*list_it)[5 + num_insert] + slope4 * interval * (i + 1));
        }
        num_insert += int(x_increment4 / interval - 1);
        
        /*cout << Qlast << "," << Qlast2nd << endl;
        vector<double> temp = *list_it;
        for (auto w = temp.begin(); w !=temp.end(); w++) {
            
            cout << *w << ", ";
        }
        cout << endl;*/
    }
}

double DynamicFunctions::LCEventload(const std::vector<double>& Landuse_EC_vectors, double Eventdata, double cell_num, Inputs* input) {
    double LCEC = 0.0;
    double EventECDF = 0.0;
    double EventTotal = 0.0;
    double LC_area_fraction = 0.0;
    double cell_count = input->cell_count_Nowater;
    std::vector<double> HydrologicData_Record = input->DischargeRecord;
    //get the ecdf for this event, function returns cumulative distribution probability between 0 and 1, it is (1- exceedance probability)
    //DataRecord is a vector that reading the daily discharge, Eventdata is a scaler of that day's discharge value, cfs
    EventECDF = ecdf(Eventdata, input);
    //use the quantile function to find the EC of this event in this landcover
    LCEC = quantile(Landuse_EC_vectors, EventECDF, input);
    //EC_L,t * (AL/A), the contribution of pollutant from each landcover type
    LC_area_fraction = cell_num / cell_count;
    EventTotal = LCEC * LC_area_fraction;
    return EventTotal; 
}



void DynamicFunctions::YearAverage(Inputs* input) {

    std::vector<double> HydrologicData_Record = input->DischargeRecord;
    std::vector<int> YearRecord = input->year;
    std::vector<int> MonthRecord = input->month;
    std::vector<int> DayRecord = input->day;

    double Eventdata = 0.000;
    std::vector<double> uniqueyear;
    std::vector<double>AverageYearLoad_TP;
    std::vector<double>AverageYearLoad_TN;
    std::vector<double>AverageYearLoad_TSS;

    double area = 0.0;
    double cell_size = 0.0;
    double cell_count = 0.0;
    
    //vectors saving data for each time step, to write extened output
    std::vector<double> TS_Load_TP_entered, TS_Load_TN_entered, TS_Load_TSS_entered,
        TS_Load_TP_delivered, TS_Load_TN_delivered, TS_Load_TSS_delivered,
        TS_Storage_TP_enter, TS_Storage_TN_enter, TS_Storage_TSS_enter,
        TS_Storage_TP_NA, TS_Storage_TN_NA, TS_Storage_TSS_NA,
        TS_Storage_TP_release, TS_Storage_TN_release, TS_Storage_TSS_release,
        TS_Storage_TP_kg, TS_Storage_TN_kg, TS_Storage_TSS_kg,
        TS_daily_TP_load, TS_daily_TN_load, TS_daily_TSS_load,
        TS_Delivery_ratio, TS_ActiveRiverLength_m, TS_ECDF,
        TS_date_year, TS_date_month, TS_date_day, TS_datarecord;


    double river_length_m = input->BufferTemporalLoadingNumericalParams["River_length_km"] * 1000;
    double ActiveRiverLength_Coefficient_C = input->BufferTemporalLoadingNumericalParams["ActiveRiverLength_Coefficient_C"];
    double Threshold = input->BufferTemporalLoadingNumericalParams["Threshold_frac"];
    double Base_delivery_ratio = input->BufferTemporalLoadingNumericalParams["Base_delivery_ratio_frac"];//sediment delivery ratio 
    double Lambda = input->BufferTemporalLoadingNumericalParams["Lambda"];
    double Active_Storge_days = input->BufferTemporalLoadingNumericalParams["Active_Storge_days"];

    double ActiveStorage_initial_TP = input->BufferTemporalLoadingNumericalParams["ActiveStorage_initial_TP_kg_P_d"];
    double ActiveStorage_initial_TN = input->BufferTemporalLoadingNumericalParams["ActiveStorage_initial_TN_kg_P_d"];
    double ActiveStorage_initial_TSS = input->BufferTemporalLoadingNumericalParams["ActiveStorage_initial_TSS_kg_P_d"];
    //initialize
    double Storage_TP_kg = ActiveStorage_initial_TP * Active_Storge_days;
    double Storage_TN_kg = ActiveStorage_initial_TN * Active_Storge_days;
    double Storage_TSS_kg = ActiveStorage_initial_TSS * Active_Storge_days;
    //Storage at Active_Storge_days
    double StorageP_atT, StorageN_atT, StorageSS_atT;
    
    //Linear Interpolation
    Interpolation(input);

    //The cell size of the blockgroup map is 370m
    cell_size = input -> cellsize;
    cell_count = input->cell_count_Nowater;
    //watershed area is cell number(m)*cell_size (m)^2/10000 ha
    area = cell_count * (cell_size * cell_size) / 10000;
    

    //use the sort and unique function within algorithm library to find out the unique years we are simulating
    /*uniqueyear = YearRecord;
    sort(uniqueyear.begin(), uniqueyear.end());
    uniqueyear.erase(unique(uniqueyear.begin(), uniqueyear.end()), uniqueyear.end());*/

    // counter, for writing the value into vector
    //int i = 0;
    //for each year
    for (int year = input->SimulationNumericalParams["StartDate_YYYYMMDD"]/10000; year < input->SimulationNumericalParams["StopDate_YYYYMMDD"]/10000 +1; ++year) {
        std::vector<double>datarecord_year;//a temp value
        //set an iterator at the beginning of the discharge data vector, record_it function will iterate with year_it      
        auto year_it = YearRecord.begin();
        auto month_it = MonthRecord.begin();
        auto day_it = DayRecord.begin();
        for (auto record_it = HydrologicData_Record.begin(); record_it != HydrologicData_Record.end(); ++record_it) {
            if (*year_it == year) {
                datarecord_year.push_back(*record_it);
                //saving values to vectors, to write out the extened output
                TS_date_year.push_back(*year_it);
                TS_date_month.push_back(*month_it);
                TS_date_day.push_back(*day_it);
                TS_datarecord.push_back(*record_it);
            }
            ++year_it;
            ++month_it;
            ++day_it;
        }
        
        //sum of the load for each year
        double yearly_TP_load_kg = 0.0, yearly_TN_load_kg = 0.0, yearly_TSS_load_kg = 0.0;
        //for each event in this year
        //if use all data record to build the probability distribution, rather than using event > th, change DataRecord_th to DataRecord for the follwing line within the loop 
        for (auto record_day = datarecord_year.begin(); record_day != datarecord_year.end(); ++record_day) {
            double daily_TP_load_kg_d, daily_TN_load_kg_d, daily_TSS_load_kg_d;
            Eventdata = *record_day;
            //all of these parameters are temp, will be initialized for each day 
            double ECTP21, ECTP22, ECTP23, ECTP24, ECTP41, ECTP42, ECTP43, ECTP71, ECTP81, ECTP82;
            double ECTN21, ECTN22, ECTN23 , ECTN24, ECTN41, ECTN42, ECTN43, ECTN71, ECTN81, ECTN82;
            double ECTSS21, ECTSS22 , ECTSS23, ECTSS24, ECTSS41, ECTSS42, ECTSS43, ECTSS71, ECTSS81, ECTSS82;
            double ActiveRiverLength_m, Delivery_ratio, Length_ratio;
            double Load_TP_delivered_kg_d, Load_TN_delivered_kg_d, Load_TSS_delivered_kg_d;
            double Load_TP_entered_kg_d, Load_TN_entered_kg_d, Load_TSS_entered_kg_d;
            double Storage_TP_enter_kg_d, Storage_TN_enter_kg_d, Storage_TSS_enter_kg_d;
            double Storage_TP_NA_kg_d, Storage_TN_NA_kg_d, Storage_TSS_NA_kg_d;
            double Storage_TP_release_kg_d, Storage_TN_release_kg_d, Storage_TSS_release_kg_d;
            
            ECTP21 = LCEventload(pollutant_landcover_list[0], Eventdata, input->Cell_number[0], input);
            ECTP22 = LCEventload(pollutant_landcover_list[1], Eventdata, input->Cell_number[1], input);
            ECTP23 = LCEventload(pollutant_landcover_list[2], Eventdata, input->Cell_number[2], input);
            ECTP24 = LCEventload(pollutant_landcover_list[3], Eventdata, input->Cell_number[3], input);
            ECTP41 = LCEventload(pollutant_landcover_list[4], Eventdata, input->Cell_number[4], input);
            ECTP42 = LCEventload(pollutant_landcover_list[5], Eventdata, input->Cell_number[5], input);
            ECTP43 = LCEventload(pollutant_landcover_list[6], Eventdata, input->Cell_number[6], input);
            ECTP71 = LCEventload(pollutant_landcover_list[7], Eventdata, input->Cell_number[7], input);
            ECTP81 = LCEventload(pollutant_landcover_list[8], Eventdata, input->Cell_number[8], input);
            ECTP82 = LCEventload(pollutant_landcover_list[9], Eventdata, input->Cell_number[9], input);

            ECTN21 = LCEventload(pollutant_landcover_list[10], Eventdata, input->Cell_number[0], input);
            ECTN22 = LCEventload(pollutant_landcover_list[11], Eventdata, input->Cell_number[1], input);
            ECTN23 = LCEventload(pollutant_landcover_list[12], Eventdata, input->Cell_number[2], input);
            ECTN24 = LCEventload(pollutant_landcover_list[13], Eventdata, input->Cell_number[3], input);
            ECTN41 = LCEventload(pollutant_landcover_list[14], Eventdata, input->Cell_number[4], input);
            ECTN42 = LCEventload(pollutant_landcover_list[15], Eventdata, input->Cell_number[5], input);
            ECTN43 = LCEventload(pollutant_landcover_list[16], Eventdata, input->Cell_number[6], input);
            ECTN71 = LCEventload(pollutant_landcover_list[17], Eventdata, input->Cell_number[7], input);
            ECTN81 = LCEventload(pollutant_landcover_list[18], Eventdata, input->Cell_number[8], input);
            ECTN82 = LCEventload(pollutant_landcover_list[19], Eventdata, input->Cell_number[9], input);

            ECTSS21 = LCEventload(pollutant_landcover_list[20], Eventdata, input->Cell_number[0], input);
            ECTSS22 = LCEventload(pollutant_landcover_list[21], Eventdata, input->Cell_number[1], input);
            ECTSS23 = LCEventload(pollutant_landcover_list[22], Eventdata, input->Cell_number[2], input);
            ECTSS24 = LCEventload(pollutant_landcover_list[23], Eventdata, input->Cell_number[3], input);
            ECTSS41 = LCEventload(pollutant_landcover_list[14], Eventdata, input->Cell_number[4], input);
            ECTSS42 = LCEventload(pollutant_landcover_list[25], Eventdata, input->Cell_number[5], input);
            ECTSS43 = LCEventload(pollutant_landcover_list[26], Eventdata, input->Cell_number[6], input);
            ECTSS71 = LCEventload(pollutant_landcover_list[27], Eventdata, input->Cell_number[7], input);
            ECTSS81 = LCEventload(pollutant_landcover_list[28], Eventdata, input->Cell_number[8], input);
            ECTSS82 = LCEventload(pollutant_landcover_list[29], Eventdata, input->Cell_number[9], input);
           
            //Load entering the drainage network, convert from kg/yr/ha to kg/day
            Load_TP_entered_kg_d = (ECTP21 + ECTP22 + ECTP23 + ECTP24 + ECTP41 + ECTP42 + ECTP43 + ECTP71 + ECTP81 + ECTP82) / datarecord_year.size() * area;
            Load_TN_entered_kg_d = (ECTN21 + ECTN22 + ECTP23 + ECTN24 + ECTN41 + ECTN42 + ECTN43 + ECTN71 + ECTN81 + ECTN82) / datarecord_year.size() * area;
            Load_TSS_entered_kg_d = (ECTSS21 + ECTSS22 + ECTSS23 + ECTSS24 + ECTSS41 + ECTSS42 + ECTSS43 + ECTSS71 + ECTSS81 + ECTSS82) / datarecord_year.size() * area;

            //Dynamic ActiveRiverLength_m Eq
            if (Threshold == 1){
                ActiveRiverLength_m = 0;
            }
            else {
                ActiveRiverLength_m = ActiveRiverLength_Coefficient_C * river_length_m / (1 - Threshold) * exp(-Lambda * (1 + (1 - ecdf(Eventdata, input)) / (1 - Threshold)));
            }
            
            //Dynamic Delivery_ratio Eq
            if (ActiveRiverLength_m > river_length_m) {
                Length_ratio = 1;
                Delivery_ratio = Length_ratio;
            }else {
                Length_ratio = ActiveRiverLength_m / river_length_m;
                if (Threshold == 1) {
                    Delivery_ratio = Base_delivery_ratio;
                }else{
                    Delivery_ratio = Base_delivery_ratio + Lambda * exp(-Lambda * (1 + (1 - ecdf(Eventdata, input)) / (1 - Threshold)) + 1);
                }
                
            }
            //Delivery_ratio must be <= 1
            if (Delivery_ratio > 1) {
                Delivery_ratio = 1;
            }

            //Load released to the basin outlet
            Load_TP_delivered_kg_d = Load_TP_entered_kg_d * Delivery_ratio;
            Load_TN_delivered_kg_d = Load_TN_entered_kg_d * Delivery_ratio;
            Load_TSS_delivered_kg_d = Load_TSS_entered_kg_d * Delivery_ratio;
             
            //if run storage
            if (input->BufferTemporalLoadingNumericalParams["Flag_RunStorage"] ==1) {
                //pollutant load entering storage within the drainage network 
                Storage_TP_enter_kg_d = Load_TP_entered_kg_d - Load_TP_delivered_kg_d;
                Storage_TN_enter_kg_d = Load_TN_entered_kg_d - Load_TN_delivered_kg_d;
                Storage_TSS_enter_kg_d = Load_TSS_entered_kg_d - Load_TSS_delivered_kg_d;
            
                //The value of the unavailable storage for each time step 
                Storage_TP_NA_kg_d = Storage_TP_kg / Active_Storge_days;
                Storage_TN_NA_kg_d = Storage_TN_kg / Active_Storge_days;
                Storage_TSS_NA_kg_d = Storage_TSS_kg / Active_Storge_days;

                //1st time update the storage
                Storage_TP_kg = Storage_TP_kg + Storage_TP_enter_kg_d - Storage_TP_NA_kg_d;
                Storage_TN_kg = Storage_TN_kg + Storage_TN_enter_kg_d - Storage_TN_NA_kg_d;
                Storage_TSS_kg = Storage_TSS_kg + Storage_TSS_enter_kg_d - Storage_TSS_NA_kg_d;
                
                // if p>threshold, release some pollutant from the storage
                if (ecdf(Eventdata, input) > Threshold) {
                    Storage_TP_release_kg_d = Storage_TP_kg * Delivery_ratio * Length_ratio;
                    Storage_TN_release_kg_d = Storage_TN_kg * Delivery_ratio * Length_ratio;
                    Storage_TSS_release_kg_d = Storage_TSS_kg * Delivery_ratio * Length_ratio;
                
                    if (Storage_TP_release_kg_d > Storage_TP_kg) {
                        Storage_TP_release_kg_d = Storage_TP_kg;
                    }
                    if (Storage_TN_release_kg_d > Storage_TN_kg) {
                        Storage_TN_release_kg_d = Storage_TN_kg;
                    }
                    if (Storage_TSS_release_kg_d > Storage_TSS_kg) {
                        Storage_TSS_release_kg_d = Storage_TSS_kg;
                    }
                    //2nd time update the storage
                    Storage_TP_kg = Storage_TP_kg - Storage_TP_release_kg_d;
                    Storage_TN_kg = Storage_TN_kg - Storage_TN_release_kg_d;
                    Storage_TSS_kg = Storage_TSS_kg - Storage_TSS_release_kg_d;
                }
                else {
                    Storage_TP_release_kg_d = 0;
                    Storage_TN_release_kg_d = 0;
                    Storage_TSS_release_kg_d = 0;
                }
                //load for each day
                daily_TP_load_kg_d = Load_TP_delivered_kg_d + Storage_TP_release_kg_d;
                daily_TN_load_kg_d = Load_TN_delivered_kg_d + Storage_TN_release_kg_d;
                daily_TSS_load_kg_d = Load_TSS_delivered_kg_d + Storage_TSS_release_kg_d;
            }
            else {// if not run storage, daily load will be load delivered
                daily_TP_load_kg_d = Load_TP_delivered_kg_d;
                daily_TN_load_kg_d = Load_TN_delivered_kg_d;
                daily_TSS_load_kg_d = Load_TSS_delivered_kg_d;
            }

            
            //accumulating load for a year
            yearly_TP_load_kg += daily_TP_load_kg_d;
            yearly_TN_load_kg += daily_TN_load_kg_d;
            yearly_TSS_load_kg += daily_TSS_load_kg_d;
            
            //saving discharge event data
            TS_ECDF.push_back(ecdf(Eventdata, input));
            TS_Delivery_ratio.push_back(Delivery_ratio);
            TS_ActiveRiverLength_m.push_back(ActiveRiverLength_m);

            //saving data into vectors for each time step
            TS_Load_TP_entered.push_back(Load_TP_entered_kg_d);
            TS_Load_TN_entered.push_back(Load_TN_entered_kg_d);
            TS_Load_TSS_entered.push_back(Load_TSS_entered_kg_d);
            
            TS_Load_TP_delivered.push_back(Load_TP_delivered_kg_d);
            TS_Load_TN_delivered.push_back(Load_TN_delivered_kg_d);
            TS_Load_TSS_delivered.push_back(Load_TSS_delivered_kg_d);

            //if run storage
            if (input->BufferTemporalLoadingNumericalParams["Flag_RunStorage"] == 1) {
                TS_Storage_TP_enter.push_back(Storage_TP_enter_kg_d);
                TS_Storage_TN_enter.push_back(Storage_TN_enter_kg_d);
                TS_Storage_TSS_enter.push_back(Storage_TSS_enter_kg_d);

                TS_Storage_TP_NA.push_back(Storage_TP_NA_kg_d);
                TS_Storage_TN_NA.push_back(Storage_TN_NA_kg_d);
                TS_Storage_TSS_NA.push_back(Storage_TSS_NA_kg_d);

                TS_Storage_TP_release.push_back(Storage_TP_release_kg_d);
                TS_Storage_TN_release.push_back(Storage_TN_release_kg_d);
                TS_Storage_TSS_release.push_back(Storage_TSS_release_kg_d);

                TS_Storage_TP_kg.push_back(Storage_TP_kg);
                TS_Storage_TN_kg.push_back(Storage_TN_kg);
                TS_Storage_TSS_kg.push_back(Storage_TSS_kg);
            }
            

            TS_daily_TP_load.push_back(daily_TP_load_kg_d);
            TS_daily_TN_load.push_back(daily_TN_load_kg_d);
            TS_daily_TSS_load.push_back(daily_TSS_load_kg_d);
        }
        //saving yearly load to vector, kg/yr
        AverageYearLoad_TP.push_back(yearly_TP_load_kg);
        AverageYearLoad_TN.push_back(yearly_TN_load_kg);
        AverageYearLoad_TSS.push_back(yearly_TSS_load_kg);

    }


    vector <pair<string, vector<double>>> result_P_map, result_N_map, result_TSS_map;
    //create a map of vectors to write out the extened output
    if (input->BufferTemporalLoadingNumericalParams["Flag_RunStorage"] == 1) {//if running storage
        //for user manully spin up to improve the initial guess of active storage
        double gap_P_frac, gap_N_frac, gap_SS_frac;
        StorageP_atT = TS_Storage_TP_kg[Active_Storge_days]/ Active_Storge_days;
        StorageN_atT = TS_Storage_TN_kg[Active_Storge_days]/ Active_Storge_days;
        StorageSS_atT = TS_Storage_TSS_kg[Active_Storge_days]/ Active_Storge_days;

        gap_P_frac = (ActiveStorage_initial_TP - StorageP_atT)/ StorageP_atT;
        gap_N_frac = (ActiveStorage_initial_TN - StorageN_atT) / StorageN_atT;
        gap_SS_frac = (ActiveStorage_initial_TSS - StorageSS_atT) / StorageSS_atT;
        cout << "Averaged active TP storage at time step " << Active_Storge_days << " days is " << StorageP_atT <<"kg, initial storage was set to "<< ActiveStorage_initial_TP << " kg/day, the difference is " << gap_P_frac *100 << " %. Adjust it if it's needed." << endl;
        cout << "Averaged active TN storage at time step " << Active_Storge_days << " days is " << StorageN_atT << "kg, initial storage was set to " << ActiveStorage_initial_TN << " kg/day, the difference is " << gap_N_frac * 100 << " %. Adjust it if it's needed." << endl;
        cout << "Averaged active TSS storage at time step " << Active_Storge_days << " days is " << StorageSS_atT << "kg, initial storage was set to " << ActiveStorage_initial_TSS << " kg/day, the difference is " << gap_SS_frac * 100 << " %. Adjust it if it's needed." << endl;
       
        result_P_map = {
            {"Year", TS_date_year},
            {"Month", TS_date_month},
            {"Day", TS_date_day},
            {"Discharge", TS_datarecord},
            {"p_cumulative", TS_ECDF},
            {"Delivery_ratio", TS_Delivery_ratio},
            {"ActiveRiverLength_m", TS_ActiveRiverLength_m},
            {"Load_TP_entered_kg_d", TS_Load_TP_entered},
            {"Load_TP_delivered_kg_d", TS_Load_TP_delivered},
            {"Storage_TP_enter_kg_d", TS_Storage_TP_enter}, 
            {"Storage_TP_NA_kg_d", TS_Storage_TP_NA},
            {"Storage_TP_release_kg_d", TS_Storage_TP_release},
            {"Storage_TP_kg", TS_Storage_TP_kg},
            {"Daily_TP_load_kg_d", TS_daily_TP_load}
        };
       result_N_map = {
            {"Year", TS_date_year},
            {"Month", TS_date_month},
            {"Day", TS_date_day},
            {"Discharge", TS_datarecord},
            {"p_cumulative", TS_ECDF},
            {"Delivery_ratio", TS_Delivery_ratio},
            {"ActiveRiverLength_m", TS_ActiveRiverLength_m},
            {"Load_TN_entered_kg_d", TS_Load_TN_entered},
            {"Load_TN_delivered_kg_d", TS_Load_TN_delivered},
            {"Storage_TN_enter_kg_d", TS_Storage_TN_enter},
            {"Storage_TN_NA_kg_d", TS_Storage_TN_NA},
            {"Storage_TN_release_kg_d", TS_Storage_TN_release},
            {"Storage_TN_kg", TS_Storage_TN_kg},
            {"Daily_TN_load_kg_d", TS_daily_TN_load},
            
       };
       result_TSS_map = {
            {"Year", TS_date_year},
            {"Month", TS_date_month},
            {"Day", TS_date_day},
            {"Discharge", TS_datarecord},
            {"p_cumulative", TS_ECDF},
            {"Delivery_ratio", TS_Delivery_ratio},
            {"ActiveRiverLength_m", TS_ActiveRiverLength_m},
            {"Load_TSS_entered_kg_d", TS_Load_TSS_entered},
            {"Load_TSS_delivered_kg_d", TS_Load_TSS_delivered},
            {"Storage_TSS_enter_kg_d", TS_Storage_TSS_enter},
            {"Storage_TSS_NA_kg_d", TS_Storage_TSS_NA},
            {"Storage_TSS_release_kg_d", TS_Storage_TSS_release},
            {"Storage_TSS_kg", TS_Storage_TSS_kg},
            {"Daily_TSS_load_kg_d", TS_daily_TSS_load}
       };
    }
    else {//if not running storage
        result_P_map = {
            {"Year", TS_date_year},
            {"Month", TS_date_month},
            {"Day", TS_date_day},
            {"Discharge", TS_datarecord},
            {"p_cumulative", TS_ECDF},
            {"Delivery_ratio", TS_Delivery_ratio},
            {"ActiveRiverLength_m", TS_ActiveRiverLength_m},

            {"Load_TP_entered_kg_d", TS_Load_TP_entered},
            {"Load_TP_delivered_kg_d", TS_Load_TP_delivered},

            {"Daily_TP_load_kg_d", TS_daily_TP_load}
        };
        result_N_map = {
            {"Year", TS_date_year},
            {"Month", TS_date_month},
            {"Day", TS_date_day},
            {"Discharge", TS_datarecord},
            {"p_cumulative", TS_ECDF},
            {"Delivery_ratio", TS_Delivery_ratio},
            {"ActiveRiverLength_m", TS_ActiveRiverLength_m},

            {"Load_TN_entered_kg_d", TS_Load_TN_entered},
            {"Load_TN_delivered_kg_d", TS_Load_TN_delivered},

            {"Daily_TN_load_kg_d", TS_daily_TN_load},

        };
        result_TSS_map = {
            {"Year", TS_date_year},
            {"Month", TS_date_month},
            {"Day", TS_date_day},
            {"Discharge", TS_datarecord},
            {"p_cumulative", TS_ECDF},
            {"Delivery_ratio", TS_Delivery_ratio},
            {"ActiveRiverLength_m", TS_ActiveRiverLength_m},

            {"Load_TSS_entered_kg_d", TS_Load_TSS_entered},
            {"Load_TSS_delivered_kg_d", TS_Load_TSS_delivered},

            {"Daily_TSS_load_kg_d", TS_daily_TSS_load}
        };
    }
    
    uniqueyear = TS_date_year;
    sort(uniqueyear.begin(), uniqueyear.end());
    uniqueyear.erase(unique(uniqueyear.begin(), uniqueyear.end()), uniqueyear.end());

    //WaterQuality_Output::writeLoad_BufferDynamic function will process total phosphorus TP values
    WaterQuality_Output::writeLoad_BufferDynamic(AverageYearLoad_TP, uniqueyear, input, "TP");
    //WaterQuality_Output::writeLoad_BufferDynamic function will process total nitrogen TN values
    WaterQuality_Output::writeLoad_BufferDynamic(AverageYearLoad_TN, uniqueyear, input, "TN");
    //WaterQuality_Output::writeLoad_BufferDynamic function will process total suspended solids TSS values
    WaterQuality_Output::writeLoad_BufferDynamic(AverageYearLoad_TSS, uniqueyear, input, "TSS");

    if (input->SimulationStringParams["Flag_ExtendedOutputs"] == "1") {
        //writing extened output
        WaterQuality_Output::writeLoad_BufferDynamic_extended(result_P_map, input, "TP");
        WaterQuality_Output::writeLoad_BufferDynamic_extended(result_N_map, input, "TN");
        WaterQuality_Output::writeLoad_BufferDynamic_extended(result_TSS_map, input, "TSS");
    }
    std::cout << "Writing output to: " << input->SimulationStringParams["OutputFolder_Path"] << std::endl;
    std::cout << "===" << std::endl;
}