#include "BlockGroupHandler.h"
#include <algorithm>

void BlockGroupHandler::AddToBlockGroup(int DEMLocationIndex, Inputs *input, DataFolder *DEMLocationFolder)
{
	bool folderIsInABlockGroupRange = PrintBlockGroupCheck(DEMLocationIndex, input);
	if (input->BlockGroup_ID[DEMLocationIndex] == Inputs::NODATA) {
		return;
	}
	if (input->TemperatureCalculationParams["PrintAllBlockGroups"] == 1 || folderIsInABlockGroupRange) {
		AddDEMLocationFolderValuesToBlockGroupFolder(DEMLocationFolder, input->BlockGroup_ID[DEMLocationIndex]);
	}
}

//Note: For model = SpatialTemperatureHydro, DataDrawer_ID has maximum = maxRows * maxCols, and DataFolder_ID typically has maximum = 1
bool BlockGroupHandler::PrintBlockGroupCheck(int DataDrawer_ID, Inputs *input)
{
	for (int iter2 = 0; iter2 <input->StartingBlockIdsToPrintTempOutput.size(); ++iter2) {
		if (input->BlockGroup_ID[DataDrawer_ID] >= input->StartingBlockIdsToPrintTempOutput[iter2] 
			&& input->BlockGroup_ID[DataDrawer_ID] <= input->EndingBlockIdsToPrintTempOutput[iter2]) {

			return true;
		}
	}
	return false;
}

void BlockGroupHandler::TimeStepProcessing(Inputs *input, int timeStep)
{
	int Time_HH_int;
	//Flag_TS_LastForDay initiated to false
	bool Flag_TS_LastForDay = false;
	//Year_YYYY = input->SimulationDate_Output_GD[timeStep] / 10000; determine year for Month_Year_to_Days_in_Month
	int Year_YYYY = input->SimulationDate_Output_GD[timeStep] / 10000;
	///Flag_LeapYear = ((Year_YYYY % 4 == 0 && Year_YYYY % 100 != 0) || (Year_YYYY % 400 == 0))
	//Note: Leap years are divisible by 4 and not divisible by 100, or divisible by 400
	bool Flag_LeapYear = ((Year_YYYY % 4 == 0 && Year_YYYY % 100 != 0) || (Year_YYYY % 400 == 0));
	//Days_in_Year = Flag_LeapYear ? 366 : 365; determines days based on ternary
	int Days_in_Year = Flag_LeapYear ? 366 : 365;

	//Date_YYYYMMDDHH_str is string created from SimulationDate_Output_GDH (YYYYMMDDHH) of timeStep, vector defined in readWeatherFile
	string Date_YYYYMMDDHH_str = to_string(input->SimulationDate_Output_GDH[timeStep]);
	//Date_YYYYMMDD_str is string created from SimulationDate_Output_GD (YYYYMMDD) of timeStep, vector defined in readWeatherFile
	string Date_YYYYMMDD_str = to_string(input->SimulationDate_Output_GD[timeStep]);
	//Time_HH_int (HH) is integer extracted from Date_YYYYMMDDHH_str string in 8th position, 2 digits
	Time_HH_int = atoi(Date_YYYYMMDDHH_str.substr(8, 2).c_str());

	//Day_JulianDay is SimulationDate_Output_GD[timeStep] Julian Day extracted from Inputs::Date_YYYYMMDD_to_JulianDay function
	int Day_JulianDay = input->Date_YYYYMMDD_to_JulianDay(input->SimulationDate_Output_GD[timeStep]);
	//Day_JulianDay_TS_Next initiated as SimulationDate_Output_GD[timeStep] Julian Day extracted from Inputs::Date_YYYYMMDD_to_JulianDay function
	int Day_JulianDay_TS_Next = input->Date_YYYYMMDD_to_JulianDay(input->SimulationDate_Output_GD[timeStep]);
	//Day_JulianDay_TS_Prior initiated as SimulationDate_Output_GD[timeStep] Julian Day extracted from Inputs::Date_YYYYMMDD_to_JulianDay function
	int Day_JulianDay_TS_Prior = input->Date_YYYYMMDD_to_JulianDay(input->SimulationDate_Output_GD[timeStep]);

	//If timeStep + 1 does not exceed SimulationDate_Output_GD.size then extract the next date in vector
	if (timeStep + 1 < input->SimulationDate_Output_GD.size()) {
		//Day_JulianDay_TS_Next is SimulationDate_Output_GD[timeStep + 1] Julian Day extracted from Inputs::Date_YYYYMMDD_to_JulianDay function
		Day_JulianDay_TS_Next = input->Date_YYYYMMDD_to_JulianDay(input->SimulationDate_Output_GD[timeStep + 1]);
	}
	//Else timeStep + 1 exceeds SimulationDate_Output_GD.size then at last time step for simulation and hence day
	else {
		//Flag_TS_LastForDay set to true
		Flag_TS_LastForDay = true;
	}

	//SimulationDay_Duration_sec incremented by input->SimulationTimeStep_Duration_sec[timeStep] to contain all seconds in simulated day
	//Note: Some simulations will only simulate partial days, some will use irregular time step durations
	input->SimulationDay_Duration_sec = input->SimulationDay_Duration_sec + input->SimulationTimeStep_Duration_sec[timeStep];

	//If (timeStep - 1 >= 0) then vector index exists
	if ((timeStep - 1) >= 0) {
		//Day_JulianDay_TS_Prior is SimulationDate_Output_GD[timeStep - 1] Julian Day extracted from Inputs::Date_YYYYMMDD_to_JulianDay function
		Day_JulianDay_TS_Prior = input->Date_YYYYMMDD_to_JulianDay(input->SimulationDate_Output_GD[timeStep - 1]);
	}

	//If (Day_JulianDay_TS_Next - Day_JulianDay == 1) then 1 day separates timeSteps and this is last time step of day, in same year
	//Note: Flag_TS_LastForDay initiated to false
	if (Day_JulianDay_TS_Next - Day_JulianDay == 1) {
		//Flag_TS_LastForDay set to true
		Flag_TS_LastForDay = true;
	}
	//Else If (Day_JulianDay - Day_JulianDay_TS_Next >= (Days_in_Year - 1)) then New Year's Eve separates timeSteps and this is last time step in year
	else if (Day_JulianDay - Day_JulianDay_TS_Next >= (Days_in_Year - 1)) {
		//Flag_TS_LastForDay set to true
		Flag_TS_LastForDay = true;
	}

	//If (Day_JulianDay - Day_JulianDay_TS_Prior == 1) then 1 day separates timeSteps and this is first time step of new day, in same year
	if (Day_JulianDay - Day_JulianDay_TS_Prior == 1) {
		//SimulationDay_Duration_sec is input->SimulationTimeStep_Duration_sec[timeStep]
		input->SimulationDay_Duration_sec = input->SimulationTimeStep_Duration_sec[timeStep];
	}
	//Else If (Day_JulianDay_TS_Prior - Day_JulianDay >= (Days_in_Year - 1)) then New Year's Eve separates timeSteps and this is first time step of a new year
	else if (Day_JulianDay_TS_Prior - Day_JulianDay >= (Days_in_Year - 1)) {
		//SimulationDay_Duration_sec is input->SimulationTimeStep_Duration_sec[timeStep]
		input->SimulationDay_Duration_sec = input->SimulationTimeStep_Duration_sec[timeStep];
	}


	//Flag_needDailyBlockGroupFiles is set to true if Flag_PrintBlockGroupDailyAndHourly_str.find is 1 or 2, not npos
	//Note: PrintBlockGroupDailyAndHourly options: 0=no output; 1=day and hour ouput; 2=day only output; 3=hour only output
	bool Flag_needDailyBlockGroupFiles =
		(input->Flag_PrintBlockGroupDailyAndHourly_str.find('1') != string::npos ||
			input->Flag_PrintBlockGroupDailyAndHourly_str.find('2') != string::npos);
	//Flag_needHourlyBlockGroupFiles is set to true if Flag_PrintBlockGroupDailyAndHourly_str.find is 1 or 3, not npos
	//Note: PrintBlockGroupDailyAndHourly options: 0=no output; 1=day and hour ouput; 2=day only output; 3=hour only output
	bool Flag_needHourlyBlockGroupFiles =
		(input->Flag_PrintBlockGroupDailyAndHourly_str.find('1') != string::npos ||
			input->Flag_PrintBlockGroupDailyAndHourly_str.find('3') != string::npos);

	//For auto blockGroupIter : BlockGroupIDToItsFolder; loop through all BlockGroupIDToItsFolder
	for (auto blockGroupIter : BlockGroupIDToItsFolder)	{

		int blockGroupNum = blockGroupIter.first;
		
		//If timeStep is 0 then
		if (timeStep == 0) {
			//ResefolderDailyValues function called to clear daily vectors, BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMax_K"]
			ResefolderDailyValues(blockGroupNum);
		}

		//if (PrintBlockGroupDailyAndHourly) true then call writeToBlockGroupHourlyFile
		if (Flag_needHourlyBlockGroupFiles) {
			//Call writeToBlockGroupHourlyFile(input->SimulationStringParams["OutputFolder_Path"], Date_YYYYMMDDHH_str, BlockGroupIDToItsFolder[blockGroupNum])
			TemperatureOutputWriter::writeToBlockGroupHourlyFile(input->SimulationStringParams["OutputFolder_Path"], Date_YYYYMMDDHH_str, BlockGroupIDToItsFolder[blockGroupNum]);
		}

		//DailyCal function accumulates blockgroup values into daily statistics; perhaps send counter
		//Note: Each call updates the MAX value as the current time step or prior
		DailyCal(input, blockGroupNum, timeStep);

		//Resefolder function called to clear hourly vectors, BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"]
		Resefolder(blockGroupNum);

		//If Flag_TS_LastForDay and Flag_needDailyBlockGroupFiles true, then call writeToBlockGroupDailyFile
		//Note: writeToBlockGroupDailyFile is called for any last hour of the simulation, even if a day has not fully completed at hour 23
		if (Flag_TS_LastForDay && Flag_needDailyBlockGroupFiles) {
			//TemperatureOutputWriter::writeToBlockGroupDailyFile function called
			TemperatureOutputWriter::writeToBlockGroupDailyFile(input, input->SimulationStringParams["OutputFolder_Path"],
				Date_YYYYMMDD_str, BlockGroupIDToItsFolder[blockGroupNum], timeStep);

			//ResefolderDailyValues function called to clear daily vectors, BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMax_K"]
			ResefolderDailyValues(blockGroupNum);
		}
	}
}

BlockGroupHandler::BlockGroupHandler(Inputs *input)
{
	//TemperatureCalculationParams["RowColOutputAllTimeSteps"] && block group >-9999
	for (int i = 0; i < input->BlockGroup_ID.size(); ++i) {

		bool createBlockGroupfolder = (input->TemperatureCalculationParams["PrintAllBlockGroups"] && 
										(input->BlockGroup_ID[i] != Inputs::NODATA));

		if (!createBlockGroupfolder) {
			createBlockGroupfolder = PrintBlockGroupCheck(i, input);
		}
		if (!createBlockGroupfolder) {
			continue;
		}
		if (BlockGroupIDToItsFolder[input->BlockGroup_ID[i]] == NULL) {
			//bgFolder is pointer to new DataFolder class for SpatialTemperatureHydro model
			DataFolder *bgFolder = new DataFolder;
			double limit = 10000;
			bgFolder->VarDict["WindChill_blockGroupMin_K"] = limit;
			bgFolder->VarDict["Tair_blockGroupMin_K"] = limit;
			BlockGroupIDToItsFolder[input->BlockGroup_ID[i]] = bgFolder;
			BlockGroupIDToItsFolder[input->BlockGroup_ID[i]]->ParamDict["blockGroupIndex"] = input->BlockGroup_ID[i];
		}
		BlockGroupIDToItsFolder[input->BlockGroup_ID[i]]->ParamDict["folderCount"] += 1;
	}
}

BlockGroupHandler::~BlockGroupHandler()
{
	for (auto b : BlockGroupIDToItsFolder) {
		delete b.second;
	}

}

void BlockGroupHandler::AddDEMLocationFolderValuesToBlockGroupFolder(DataFolder *folder, int blockGroupID)
{
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["ImpNR_W_p_m2"] += folder->VarDict["ImpNR_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["ImpH_W_p_m2"] += folder->VarDict["ImpH_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["ImpLE_W_p_m2"] += folder->VarDict["ImpLE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeNR_W_p_m2"] += folder->VarDict["TreeNR_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeH_W_p_m2"] += folder->VarDict["TreeH_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeLE_W_p_m2"] += folder->VarDict["TreeLE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeLEE_W_p_m2"] += folder->VarDict["TreeLEE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeLET_W_p_m2"] += folder->VarDict["TreeLET_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegNR_W_p_m2"] += folder->VarDict["SVegNR_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegH_W_p_m2"] += folder->VarDict["SVegH_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegLE_W_p_m2"] += folder->VarDict["SVegLE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegLEE_W_p_m2"] += folder->VarDict["SVegLEE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegLET_W_p_m2"] += folder->VarDict["SVegLET_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SoilNR_W_p_m2"] += folder->VarDict["SoilNR_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SoilH_W_p_m2"] += folder->VarDict["SoilH_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SoilLE_W_p_m2"] += folder->VarDict["SoilLE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WaterNR_W_p_m2"] += folder->VarDict["WaterNR_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WaterH_W_p_m2"] += folder->VarDict["WaterH_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WaterLE_W_p_m2"] += folder->VarDict["WaterLE_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["H_total_W_p_m2"] += folder->VarDict["H_total_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["LE_total_W_p_m2"] += folder->VarDict["LE_total_W_p_m2"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] += folder->VarDict["Tair_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_K"] += folder->VarDict["Tdew_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_K"] += folder->VarDict["HeatIndex_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_K"] += folder->VarDict["Humidex_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_K"] += folder->VarDict["Twetbulbglobe_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_K"] += folder->VarDict["Twetbulbnatural_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_K"] += folder->VarDict["Twetbulbpsychrometric_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_K"] += folder->VarDict["UTCI_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_K"] += folder->VarDict["WindChill_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["AbsHumidity_kg_p_m3"] += folder->VarDict["AbsHumidity_kg_p_m3"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_mesoScale_K"] += folder->VarDict["Tair_mesoScale_final_K"];
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["AbsHumidity_mesoScale_kg_p_m3"] += folder->VarDict["AbsHumidity_mesoScale_final_kg_p_m3"];
}

void BlockGroupHandler::ResefolderDailyValues(int blockGroupID)
{
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMax_K"] = 0;
	double limit = 10000;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMin_K"] = limit;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_Lin_blockGroupMax_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_Lin_blockGroupAverage_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_blockGroupMin_K"] = limit;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_blockGroupAverage_K"] = 0;
}


void BlockGroupHandler::Resefolder(int blockGroupID)
{
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["ImpNR_W_p_m2"] =0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["ImpH_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["ImpLE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeNR_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeH_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeLE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeLEE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["TreeLET_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegNR_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegH_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegLE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegLEE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SVegLET_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SoilNR_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SoilH_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["SoilLE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WaterNR_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WaterH_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WaterLE_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["H_total_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["LE_total_W_p_m2"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["AbsHumidity_kg_p_m3"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_mesoScale_final_K"] = 0;
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["AbsHumidity_mesoScale_final_kg_p_m3"] = 0;
}

//DailyCal function accumulates blockgroup values into daily statistics
//Note: Called each timestep to find average value for blockgroup and update maximum value for day
void BlockGroupHandler::DailyCal(Inputs* input, int blockGroupID, int timeStep)
{
	//Tair_K (C) is derived as quotient of blockGroupID summed Tair_K (K) and blockGroupID folderCount
	double Tair_K = BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"];
	//Tdew_K (C) is derived as quotient of blockGroupID summed Tdew_K (K) and blockGroupID folderCount
	double Tdew_K = BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"];
	//Tair_C (C) is derived as quotient of blockGroupID summed Tair_K (K) and blockGroupID folderCount, converted to degrees C by subtracting 273.15
	double Tair_C = BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] - 273.15;
	//Tdew_C (C) is derived as quotient of blockGroupID summed Tdew_K (K) and blockGroupID folderCount, converted to degrees C by subtracting 273.15
	double Tdew_C = BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] - 273.15;
	//Tair_F (F) is converted from Tair_C using standard equation, Tair_C * 9/5 + 32
	double Tair_F = Tair_C * 9.0 / 5.0 + 32;

	//HeatIndex_Lin_C defined using Eq text on bottom of pg 739 in Lin et al. (2009) 
	//References: 
	//Lin, S., M. Luo, R. J. Walker, X. Liu, S. A. Hwang and R. Chinery (2009). "Extreme high temperatures and hospital admissions for respiratory and cardiovascular diseases." Epidemiology 20(5): 738-746.
	//Basu, R., W.Y.Feng and B.D.Ostro(2008). "Characterizing temperature and mortality in nine California counties." Epidemiology 19(1) : 138 - 145.
	double HeatIndex_Lin_C = -2.653 + (0.994 * Tair_C) + 0.0153 * pow(Tdew_C,2);

	//Wind Chill Index Method, NOAA NWS
	//WindSpeed_miles_p_hour (miles/hr) derived from WindSpd_mps[timeStep] by multiplying by 3600 seconds/hr and dividing my 1609.34 meters/mile
	//The National Weather Service Wind Chill Temperature Index updated in 2001, https://www.weather.gov/safety/cold-wind-chill-chart
	double WindSpeed_miles_p_hour = input->WindSpd_mps[timeStep] * 3600.0 / 1609.34;
	double WindChill_F;
	//If Tair_F < 50 F and WindSpeed_miles_p_hour > 3.0 miles p hour then use WindChill formula
	if (Tair_F < 50.0 && WindSpeed_miles_p_hour > 3.0)	{
		//WindChill_F (F) computed using NWS equaiton on bottom of chart showing relation between wind speed and air temperature
		//Note: Formula for wind chill is on bottom of chart, and windspeed is mph or miles per hour ...
		//Note: WindChill_F = 35.74 + 0.6215(Tair_K) - [35.75 * (WindSpeed_miles_p_hour ^ 0.16)] + [(0.4275 * Tair_K * (WindSpeed_miles_p_hour ^ 0.16)]
		WindChill_F = 35.74 + (0.6215 * Tair_F) - (35.75 * pow(WindSpeed_miles_p_hour, 0.16)) + (0.4275 * Tair_F * pow(WindSpeed_miles_p_hour, 0.16));
	}
	else {
		WindChill_F = Tair_F;
	}

	//Tair_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];
	
	//Tdew_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tdew_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];
	
	//HeatIndex_blockGroupAverage_K is summed to contain all values from Blockgroup for day, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//Humidex_blockGroupAverage_K is summed to contain all values from Blockgroup for day, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//Twetbulbglobe_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//Twetbulbnatural_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//Twetbulbpsychrometric_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//UTCI_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//WindChill_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_blockGroupAverage_K"] += BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"] * input->SimulationTimeStep_Duration_sec[timeStep];

	//HeatIndex_Lin_blockGroupAverage_K is summed to contain all values from Blockgroup, weighted by * SimulationTimeStep_Duration_sec in this routine
	//Note: Weighting of daily averages divides by SimulationDay_Duration_sec in TemperatureOutputWriter to handle daily variation of timeStep count or duration
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_Lin_blockGroupAverage_K"] += (HeatIndex_Lin_C + 273.15) * input->SimulationTimeStep_Duration_sec[timeStep];

	//Tair_blockGroupMax_K is MAX of Tair_blockGroupMax_K or Tair_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//Tair_blockGroupMin_K is MIN of Tair_blockGroupMin_K or Tair_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is lesser than the prior minimum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMin_K"] = MIN(BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_blockGroupMin_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["Tair_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//HeatIndex_blockGroupMax_K is MAX of HeatIndex_blockGroupMax_K or HeatIndex_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//Humidex_blockGroupMax_K is MAX of Humidex_blockGroupMax_K or Humidex_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["Humidex_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//Twetbulbglobe_blockGroupMax_K is MAX of Twetbulbglobe_blockGroupMax_K or Twetbulbglobe_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbglobe_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//Twetbulbnatural_blockGroupMax_K is MAX of Twetbulbnatural_blockGroupMax_K or Twetbulbnatural_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbnatural_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//Twetbulbpsychrometric_blockGroupMax_K is MAX of Twetbulbpsychrometric_blockGroupMax_K or Twetbulbpsychrometric_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["Twetbulbpsychrometric_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//UTCI_blockGroupMax_K is MAX of UTCI_blockGroupMax_K or UTCI_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is larger than the prior maximum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_blockGroupMax_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["UTCI_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);

	//HeatIndex_Lin_blockGroupMax_K is MAX of HeatIndex_Lin_blockGroupMax_K or HeatIndex_Lin_C / folderCount
	//Note: Lin provides separate method for HeatIndex that was used during a USFS NUCFAC research project with Sinha et al (2022)
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_Lin_blockGroupMax_K"] = MAX(BlockGroupIDToItsFolder[blockGroupID]->VarDict["HeatIndex_Lin_blockGroupMax_K"], (HeatIndex_Lin_C + 273.15));
	
	//WindChill_blockGroupMin_K is MIN of WindChill_blockGroupMin_K or WindChill_K / folderCount
	//Note: Updated each timestep to determine if the new blockbgroup average is smaller than the prior minimum value 
	BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_blockGroupMin_K"] = MIN(BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_blockGroupMin_K"], BlockGroupIDToItsFolder[blockGroupID]->VarDict["WindChill_K"] / BlockGroupIDToItsFolder[blockGroupID]->ParamDict["folderCount"]);;
}
