﻿#include "HeatFluxCal.h"
//include "HeatMetrics_Calc.h" to access HeatMetrics_Calc.cpp functionality 
#include "HeatMetrics_Calc.h"

//Note: Refactor CoolAir energy balance
//Note: Update stability theory for resistance using USEPA (2000) Turner's Method page 6-12, or using algorithm for WBGT in HeatMetrics_Calc
//Note: Allow separate soil moisture for reference station
//Note: Explore IC = 0 for reference station 
//Note: Clarify land cover height vs as roughness length; roughness length = 0.123 or 0.0123 * height
//Note: Consider Masson (2000) Eq 13 to 17, adjusting solar radiation between roof, wall, road based on canyon orientation; OHM a1 to a3 based on canyon orientation
//Note: Temporal and seasonal variation in OHM a1 to a3 coefficients as in Sun et al. (2017) or Ward et al. (2016), and combined impervious and building values for urban as in Ward (2016)
//Note: Holstag et al. (1988) evaluated nighttime energy budget which may help us obtain additional cooling
//Note: Add Qadvection term to energy balance as in first equation of Kotthaus et al. (2014)
//Note: Add time varying snow albedo of NREL (1995) p 78; track snow depth and days since snow for each weather station
//Note: Add output Daylight Saving output option within WriteOutputs.cpp
//Note: Add thermal comfort with UTCI algorithm and mean radiant temperature from WBGT algorithm

//Note: Theory of Cool Air model captured by following statement of McNaughton et al. (1995): a) complex model: b) single layer big-leaf model; c) and two-layer model:
//Note: ... Models that describe small-scale processes very well are frequently unsuitable for large-scale applications because the generalized data available on the large scale do not justify precise calculations. For this reason, there often exists a hierarchy of models for the same processes, each with its own data requirements and each with its own degree of approximation to the real world. Canopy energy balance models are no exception. 
//Note: ... A. At the complex end of the range are canopy models that require a detailed description of canopy structure and state and employ a detailed description of turbulent transport processes. The transport processes can be based on diffusion ('K') theory (e.g., Philip, 1965; Waggoner and Reifsnyder, 1968), higher-order closure theory (e.g., Meyers and Paw U, 1987) or Lagrangian theory (e.g., Wilson, 1981; Raupach, 1989a). Of these, K theory is now considered inadequate for precise work because eddy size scales in canopies are large compared to the height intervals over which gradients change appreciably (Raupach and Thom, 1981). /f-theory models, for example, won't allow fluxes 'up' concentration gradients, even though this phenomenon is well documented in nature (Denmead and Bradley, 1985). Whatever turbulence theory is used, all detailed models require too much information on canopy and turbulence structure to be worth using in large-scale applications such as catchment hydrology or representing vegetation in global climate models. 
//Note: ... B. At the simple end of the range is the 'big-leaf' model of Penman and Monteith (Monteith, 1965). This model requires no information on canopy structure apart from the aerodynamic roughness length and displacement height, and it represents turbulent transport by the assumption that air within the canopy is well mixed, with each leaf 'seeing' the same temperature and humidity as all others. It provides no information at all on how microclimate varies within the canopy. The PenmanMonteith model has been very popular in large-scale applications. 
//Note: ... C. Between these extremes lie the two-layer models (e.g., Shuttleworth and Wallace, 1985; Choudhury and Monteith, 1988). These models go one step beyond the simple Penman-Monteith model by giving separate account of conditions at the ground and within the overstorey. The motive for this complication of the basic Penman-Monteith model is that evaporation from soil or shallow-rooted ground vegetation often behaves in a very different way to evaporation from deeper-rooted overstorey vegetation. This can significantly affect the overall energy balance of sparse canopies where a substantial fraction of the incoming solar radiation reaches the ground. The task of a two-layer model is to differentiate the environment near the ground from that within the overstorey canopy. Turbulent transport between the ground and the overstorey canopy is represented explicitly in these models even though the air within the overstorey is still treated as well mixed. 


//Energy balance determines surface heat flux and evaporative cooling or heating of air, as described by Yang et al. (2013)
//NR = H_total_W_p_m2 + LE_total_W_p_m2 + DeltaQ_W_p_m2, where NR is net radiation of shortwave and longwave terms, H_total_W_p_m2 is sensible heat flux, LE_total_W_p_m2 is latent heat flux, and DeltaQ_W_p_m2 is ground storage heat flux

//References:
//Arnfield, A. J. and C. S. B. Grimmond (1998). "An urban canyon energy budget model and its application to urban storage heat flux modeling." Energy and Buildings 27(1): 61-68.
//Bradley, A., Thornes, J., Chapman, L., Unwin, D., & Roy, M. (2002). Modeling spatial and temporal road thermal climatology in rural and urban areas using a GIS. Climate Research - CLIMATE RES, 22, 41-55. doi:10.3354/cr022041
//Camuffo, D. and A. Bernardi (1982). "An observational study of heat fluxes and their relationships with net radiation." Boundary-Layer Meteorology 23(3): 359-368.
//Grimmond, C. S. B., & Oke, T. R. (1999). Heat storage in urban areas: Local-scale observations and evaluation of a simple model. Journal of Applied Meteorology, 38(7), 922-940. doi:10.1175/1520-0450(1999)038<0922:hsiual>2.0.co;2
//Holtslag, A. A. M., & De Bruin, H. A. R. (1988). Applied Modeling of the Nighttime Surface Energy Balance over Land. Journal of Applied Meteorology and Climatology, 27(6), 689-704. doi:10.1175/1520-0450(1988)027<0689:Amotns>2.0.Co;2
//Järvi, L., C. S. B. Grimmond and A. Christen (2011). "The Surface Urban Energy and Water Balance Scheme (SUEWS): Evaluation in Los Angeles and Vancouver." Journal of Hydrology 411(3): 219-237.
//Kotthaus, S. and C. S. B. Grimmond (2014). "Energy exchange in a dense urban environment – Part I: Temporal variability of long-term observations in central London." Urban Climate 10: 261-280
//Kumar, L., Skidmore, A. K., & Knowles, E. (1997). Modelling topographic variation in solar radiation in a GIS environment. International Journal of Geographical Information Science, 11(5), 475-497. doi:10.1080/136588197242266
//Kusaka et al. (2001), A simple single - layer urban canopy model for atmospheric models : Comparison with multi - layer and slab models, Boundary Layer Meteorol., 101(3), 329–358.
//Li, D., E. Bou-Zeid and M. Oppenheimer (2014). "The effectiveness of cool and green roofs as urban heat island mitigation strategies." Environmental Research Letters 9(5): 055002.
//McNaughton, K. G., & Van Den Hurk, B. J. J. M. (1995). A ‘Lagrangian’ revision of the resistors in the two-layer model for calculating the energy budget of a plant canopy. Boundary-Layer Meteorology, 74(3), 261-288. doi:10.1007/BF00712121
//Masson (2000), A Physically based scheme for the urban energy budget in atmospheric models, Boundary Layer Meteorology, 94: 357-397.
//Offerle, B., Grimmond, C. S. B., & Oke, T. R. (2003). Parameterization of net all-wave radiation for urban areas. Journal of Applied Meteorology, 42(8), 1157-1173. doi:10.1175/1520-0450(2003)042<1157:ponarf>2.0.co;2
//Oke, T. R., Mills, G., Christen, A., & Voogt, J. A. (2017). Radiation. In Urban Climates (pp. 122-155). Cambridge: Cambridge University Press. DOI: https://doi.org/10.1017/9781139016476.007
//Reda, I., & Andreas, A. (2003). Solar Position Algorithm for Solar Radiation Applications National Renewable Energy Laboratory, NREL/TP-560-34302. Revised 2008, 55. 
//Shuttleworth, J. W. (1993). Evaporation. In D. R. Maidment (Ed.), Handbook of Hydrology (pp. 4.1-4.5.3). New York: McGraw-Hill.
//Stull, R. (2017) "Practical Meteorology: An Algebra-based Survey of Atmospheric Science" -version 1.02b.  Univ. of British Columbia.  940 pages.  isbn 978-0-88865-283-6. https://www.eoas.ubc.ca/books/Practical_Meteorology/
//Stull, R. (2000). Meteorology for Scientists and Engineers (2nd ed.). Pacific Grove, CA: Brooks/Cole. 
//Stull, R. (2003). An Introduction to Boundary Layer Meteorology. Kluwer Academic Publishers. ISBN-13: 978-94-009-3027-8
//Sun, T., Wang, Z. H., Oechel, W. C., & Grimmond, S. (2017). The Analytical Objective Hysteresis Model (AnOHM v1.0): methodology to determine bulk storage heat flux coefficients. Geosci. Model Dev., 10(7), 2875-2890. doi:10.5194/gmd-10-2875-2017
//Sun, T., Z.-H. Wang and G.-H. Ni (2013). "Revisiting the hysteresis effect in surface energy budgets." Geophysical Research Letters 40(9): 1741-1747.
//USEPA (2000). Meteorological Monitoring Guidance for Regulatory Modeling Applications. US Environmental Protection Agency, Office of Air Quality. Washington, DC. EPA-454/R-99-005. 
//Vanek, F., Albright, L., & Angenent, L. (2012). Energy Systems Engineering: Evaluation and Implementation, Second Edition (2nd ed.). New York: McGraw-Hill
//Ward, H.C., Kotthaus, S., Järvi, L., & Grimmond, C.S.B. (2016). Surface Urban Energy and Water Balance Scheme (SUEWS): Development and evaluation at two UK sites.Urban Climate, 18, 1 - 32. doi : https ://doi.org/10.1016/j.uclim.2016.05.001
//Yang, F., K. Mitchell, Y.-T. Hou, Y. Dai, X. Zeng, Z. Wang and X.-Z. Liang (2008). "Dependence of Land Surface Albedo on Solar Zenith Angle: Observations and Model Parameterization." Journal of Applied Meteorology and Climatology 47(11): 2963-2982.
//Yang, Y., Endreny, T. A., & Nowak, D. J. (2013). A physically based analytical spatial air temperature and humidity model. Journal of Geophysical Research-Atmospheres, 118(18), 10449-10463. doi:10.1002/jgrd.50803

//Theory: Albedo is assigned by cell land cover type 21 to 24 based on the assumption impervious cover increases from ~20 to 80%.
//Note: A strategic alternative is albedo is assigned by actual cell land cover fraction exposed to incoming shortwave radiation.
//Note: Land cover albedo values are in Table 2-3 in Stull, 2000. Meteorology for Scientists and Engineers, 2nd Edition, Brooks Cole.
//Note: Road albedo = 5 to 15; Building albedo = 9; Concrete albedo = 15 to 37
//Note: Tree albedo changes type (max albedo = 15 coniferous vs 25 deciduous, min albedo = 5 coniferous vs 10 deciduous) 
//Note: Tree albedo deciduous should change with leaf on dates
//Note: Green grass albedo = 26; Agricultural cover emerges as gap area (TC+IC<100%) for NCLD LC types, and albedo ranges 5 to 32
//Note: Soil albedo changes with wetness (albedo = 7 wet vs 17 dry) and percent clay (100% clay is albedo = 16 wet vs 23 dry) 
//Note: Snow albedo might be simulated (old snow albedo = 35 to 70; new snow albedo = 75 to 90)

//Theory: Emissivity is assigned as a constant for cell land cover type 21 to 24 and then by land cover type tree, short veg, soil, and water
//Note: A strategic alternative is emissivity is assigned by actual cell impervious cover fraction exposed for outgoing longwave radiation
//Note: Land cover emissivity values are in Table 2-2 in Stull, 2000. Meteorology for Scientists and Engineers, 2nd Edition, Brooks Cole.
//Note: Asphalt  = 0.95; Concrete = 0.71-0.9; Bricks (red) = 0.92; Soils = 0.9-0.98; Desert = 0.84-0.91; Grass lawn = 0.97; Grass = 0.9-0.95; 
//Note: Tree Coniferous = 0.97; Tree Deciduous = 0.95; Snow (old to new) = 0.82-0.99; Shrubs = 0.9; Alfalfa = 0.95; Leaf 10um = 0.97-0.98				

//Theory: Ground heat storage, DeltaQ_W_p_m2, from Camuffo et al. (1982) exploring hystereis of DeltaQ with NetRadiation; used a1, a2, and a3 parameters 
//Note: Sun et al. (2013) present rational for analytically derived OHM coefficients a1 to a3 with phase adjustment, based on Comuffo et al. (1982)
//Note: Grimmond and Oke (1999) coin objective hysteresis model (OHM) and present values for a1 to a3 based on surface properties of the site
//Note: Arfield et al. (1998) in 2nd Equation explains dNR/dt was mid-point based, but not feasible given NR is not known for future time step
//Note: Ward et al. (2016) in Table 4 establish representative values of a1 to a3 for paved and building surfaces, explained on p 15 of manuscript.
//Note: Influence of DeltaQ_W_p_m2 to a1, a2, and a3 explained by Fig 1 and paragraph on p 362 of Camuffo et al. (1982)
//Note:	... Parameter a1 (unitless) determines the magnitude of influence of NR on DeltaQ_W_p_m2.
//Note: ... Parameter a2 (hr) determines the phase of influence of d(NR) / dt on DeltaQ_W_p_m2.When a2 = 0, the DeltaQ_W_p_m2 curve matches, is in phase with, the NR curve, as a2 gets more positive, the DeltaQ_W_p_m2 curve precedes the NR curve, peaking sooner.
//Note: ... Parameter a3 (W/m^2) determines the size of DeltaQ_W_p_m2 when NR becomes negative.A large negative a3 creates a negative DeltaQ_W_p_m2 before NR is negative.
/*
Coefficients for Objective Hysteresis Model: Table 4 of Ward et al. (2016); see OHM monthly diurnal patterns in Sun et al. (2017)
	Land Cover				a1 (frac),a2 (hr),a3 (W/m2)
	Paved					0.676,	0.300,	-42.4
	Bldgs					0.477,	0.337,	-33.9
	EveTr					0.215,	0.325,	−19.9
	DecTr					0.215,	0.325,	−19.9
	Grass					0.215,	0.325,	−19.9
	BSoil					0.355,	0.335,	−35.3
	Water shallow, turbid	0.500,	0.210,	−39.1
	//Water values for NLCD 11 from Endreny based on a1 < Mixed forest, a2 < Bare soil-wet, a3 < Mixed forest
	//Note: Endreny's values for water a1, a2, a3 create water hotter in day, cooler in night compared with values of GY&O
	Water deep, NLCD=11		0.050,	0.050,	−05.0
Coefficients for Objective Hysteresis Model: Table 4 of Grimmond and Oke (1999)
	Land Cover				a1 (frac),a2 (hr),a3 (W/m2)
	Mixed forest			0.11,	0.11,	-12.3
	Short grass				0.32,	0.54,	-27.4
	Bare soil				0.38,	0.56,	-27.3
	Bare soil—wet			0.33,	0.07,	-34.9
	Bare soil—dry			0.35,	0.43,	-36.5
	Soil 					0.36,	0.27,	-42.4
	Water—shallow turbid	0.5,	0.21,	-39.1
	Rooftop Vancouver		0.17,	0.1,	-17
	Rooftop Uppsala			0.44,	0.57,	-28.9
	Rooftop Kyoto			0.82,	0.34,	-55.7
	Concrete Doll			0.81,	0.48,	-79.9
	Concrete Asaeada		0.85,	0.32,	-28.5
	Asphalt Narita			0.36,	0.23,	-19.3
	Asphalt Asaeada			0.64,	0.32,	-43.6
	Asphalt Anandakumar		0.82,	0.68,	-20.1
	UrbanCanyon North-South	0.32,	0.01,	-27.7
	UrbanCanyon Easth-West	0.71	0.04,	-39.7
*/

//Define and initialize the static variables:
vector<double> HeatFluxCal::AH_total_W_p_m2_vec, HeatFluxCal::Imp_AH_W_p_m2_vec, HeatFluxCal::Tree_AH_W_p_m2_vec, HeatFluxCal::SVeg_AH_W_p_m2_vec, HeatFluxCal::Soil_AH_W_p_m2_vec, HeatFluxCal::Water_AH_W_p_m2_vec, HeatFluxCal::NR_total_W_p_m2_vec, HeatFluxCal::DeltaQ_W_p_m2_vec, HeatFluxCal::WaterDeltaQ_W_p_m2_vec, HeatFluxCal::SoilDeltaQ_W_p_m2_vec, HeatFluxCal::SVegDeltaQ_W_p_m2_vec, HeatFluxCal::TreeDeltaQ_W_p_m2_vec, HeatFluxCal::ImpDeltaQ_W_p_m2_vec;
//Note: static property will define vectors on first call and then not reset on future calls 
vector<double> HeatFluxCal::ImpNR_priorTS_W_p_m2_vec, HeatFluxCal::TreeNR_priorTS_W_p_m2_vec, HeatFluxCal::SVegNR_priorTS_W_p_m2_vec, HeatFluxCal::SoilNR_priorTS_W_p_m2_vec, HeatFluxCal::WaterNR_priorTS_W_p_m2_vec, HeatFluxCal::RoofNR_priorTS_W_p_m2_vec;
bool HeatFluxCal::vectors_initialized = false;

//Constructor for HeatFluxCal with both SolarCalculator and Inputs pointer
//Note: Initialization list : or colon after function name initializes SolarCalculator and resistanceVar objects
//Note: ResistanceCal constructor of the ResistanceCal class initializes variables shared across void functions, shared globally
HeatFluxCal::HeatFluxCal(Inputs* input, SolarCalculator* solarCalc)
	: input(input), solarCalculator(solarCalc), resistanceVar(nullptr) {
	// If resistanceVar needs to be initialized based on input
	resistanceVar = new ResistanceCal(input);
}

//HeatFluxCal::~HeatFluxCal Destructor: clean up dynamically allocated memory
HeatFluxCal::~HeatFluxCal() {
	delete resistanceVar; // Free memory allocated for resistanceVar
	resistanceVar = nullptr; // Set pointer to nullptr for safety
}

//Load_TemperatureCalculationParams function initializes HydroPlusConfig TemperatureCalculationParams for folders with and without reference station 
//Note: When TemperatureCalculationParams_ReferenceStation is in HydroPlusConfig.xml the reference station folder takes its values
void HeatFluxCal::Load_TemperatureCalculationParams(Inputs* input, DataFolder* folder)
{
	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally
	//Note: RunTemperatureSpatialModel 1st sends MapPixel_ReferenceStation_IDs, regardless of indexing, then sends MapPixel_ID of 0
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//auto& TemperatureCalculationParams_pointer = (isReferenceStationPixel && !input->TemperatureCalculationParams_ReferenceStation.empty()
	//Note: Create pointer to TemperatureCalculationParams_ReferenceStation or TemperatureCalculationParams based on isReferenceStation
	auto& TemperatureCalculationParams_pointer = (input->isReferenceStationPixel(MapPixel_ID) && !input->TemperatureCalculationParams_ReferenceStation.empty())
		? input->TemperatureCalculationParams_ReferenceStation
		: input->TemperatureCalculationParams;

	//Flag_UrbanCanyonResistance is flag (1 or 0) from HydroPlusConfig.xml for Uban21 to Urban24, where 0 provides greater range of temperatures
	//Note: 1 = simulate urban canyon geometry for radiation balance, 0 = neglect urban canyon geometry
	Flag_UrbanCanyonResistance = TemperatureCalculationParams_pointer["Flag_UrbanCanyonResistance"];

	//Tree parameters
	//a1_OHM_Tree_frac, a2_OHM_Tree_hr, a3_OHM_Tree_W_p_m2 are a1 to a3 coefficients of Eq 3 in Grimmond and Oke (1999) objective hysteresis model (OHM)
	a1_OHM_Tree_frac = TemperatureCalculationParams_pointer["a1_OHM_Tree_frac"];
	a2_OHM_Tree_hr = TemperatureCalculationParams_pointer["a2_OHM_Tree_hr"];
	a3_OHM_Tree_W_p_m2 = TemperatureCalculationParams_pointer["a3_OHM_Tree_W_p_m2"];

	//Height_Tree_m (m) is average height of feature for roughness in turbulence exchange calculations
	Height_Tree_m = input->InputXml["Height_Avg_TreeCover_m"];
	//Albedo_Tree_frac (fraction) is albedo of feature for reflecting shortwave radiation
	Albedo_Tree_frac = TemperatureCalculationParams_pointer["Albedo_Tree_frac"];
	//Emissivity_Tree_frac (fraction) is emissivity of feature  for emitting longwave radiation
	Emissivity_Tree_frac = TemperatureCalculationParams_pointer["Emissivity_Tree_frac"];

	//Short vegetation parameters
	//a1_OHM_SVeg_frac, a2_OHM_SVeg_hr, a3_OHM_SVeg_W_p_m2 are a1 to a3 coefficients of Eq 3 in Grimmond and Oke (1999) objective hysteresis model (OHM)
	a1_OHM_SVeg_frac = TemperatureCalculationParams_pointer["a1_OHM_SVeg_frac"];
	a2_OHM_SVeg_hr = TemperatureCalculationParams_pointer["a2_OHM_SVeg_hr"];
	a3_OHM_SVeg_W_p_m2 = TemperatureCalculationParams_pointer["a3_OHM_SVeg_W_p_m2"];

	//Height_SVeg_m (m) is average height of feature for roughness in turbulence exchange calculations
	Height_SVeg_m = input->InputXml["Height_Avg_SVegCover_m"];
	//Albedo_SVeg_frac (fraction) is albedo of feature for reflecting shortwave radiation
	Albedo_SVeg_frac = TemperatureCalculationParams_pointer["Albedo_SVeg_frac"];
	//Emissivity_SVeg_frac (fraction) is emissivity of feature for emitting longwave radiation
	Emissivity_SVeg_frac = TemperatureCalculationParams_pointer["Emissivity_SVeg_frac"];

	//Soil parameters
	//a1_OHM_Soil_frac, a2_OHM_Soil_hr, a3_OHM_Soil_W_p_m2 are a1 to a3 coefficients of Eq 3 in Grimmond and Oke (1999) objective hysteresis model (OHM)
	a1_OHM_Soil_frac = TemperatureCalculationParams_pointer["a1_OHM_Soil_frac"];
	a2_OHM_Soil_hr = TemperatureCalculationParams_pointer["a2_OHM_Soil_hr"];
	a3_OHM_Soil_W_p_m2 = TemperatureCalculationParams_pointer["a3_OHM_Soil_W_p_m2"];

	Height_Soil_m = TemperatureCalculationParams_pointer["Height_Avg_SoilCover_m"];
	Albedo_Soil_frac = TemperatureCalculationParams_pointer["Albedo_Soil_frac"];
	Emissivity_Soil_frac = TemperatureCalculationParams_pointer["Emissivity_Soil_frac"];

	//Water parameters
	//a1_OHM_Water_frac, a2_OHM_Water_hr, a3_OHM_Water_W_p_m2 are a1 to a3 coefficients of Eq 3 in Grimmond and Oke (1999) objective hysteresis model (OHM)
	//Note: OHM Table 4 of Ward et al. (2016), Water shallow, turbid a1=0.500, a2=0.210, a3=−39.1
	//Note: OHM Yang 2013 and in PASATH was a1=0.8 a2=0.7, a3=-100
	//Note: OHM Endreny for NLCD Class 11, a1=0.05, a2=0.05, a3=−5.0, so a1 < Mixed forest, a2 < Bare soil-wet, a3 < Mixed forest
	a1_OHM_Water_frac = TemperatureCalculationParams_pointer["a1_OHM_Water_frac"];
	a2_OHM_Water_hr = TemperatureCalculationParams_pointer["a2_OHM_Water_hr"];
	a3_OHM_Water_W_p_m2 = TemperatureCalculationParams_pointer["a3_OHM_Water_W_p_m2"];

	//Height_Water_m (m) is average height of feature for roughness in turbulence exchange calculations
	Height_Water_m = TemperatureCalculationParams_pointer["Height_Avg_WaterCover_m"];
	//Albedo_Water_frac (fraction) is albedo of feature for reflecting shortwave radiation
	Albedo_Water_frac = TemperatureCalculationParams_pointer["Albedo_Water_frac"];
	//Emissivity_Water_frac (fraction) is emissivity of surface feature for emitting longwave radiation
	Emissivity_Water_frac = TemperatureCalculationParams_pointer["Emissivity_Water_frac"];

	//Urban parameters
	//a1_OHM_Pavement_frac, a2_OHM_Pavement_hr, a3_OHM_Pavement_W_p_m2 are a1 to a3 coefficients of Eq 3 in Grimmond and Oke (1999) objective hysteresis model (OHM)
	a1_OHM_Pavement_frac = TemperatureCalculationParams_pointer["a1_OHM_Pavement_frac"];
	a2_OHM_Pavement_hr = TemperatureCalculationParams_pointer["a2_OHM_Pavement_hr"];
	a3_OHM_Pavement_W_p_m2 = TemperatureCalculationParams_pointer["a3_OHM_Pavement_W_p_m2"];

	//a1_OHM_Building_frac, a2_OHM_Building_hr, a3_OHM_Building_W_p_m2 are a1 to a3 coefficients of Eq 3 in Grimmond and Oke (1999) objective hysteresis model (OHM)
	a1_OHM_Building_frac = TemperatureCalculationParams_pointer["a1_OHM_Building_frac"];
	a2_OHM_Building_hr = TemperatureCalculationParams_pointer["a2_OHM_Building_hr"];
	a3_OHM_Building_W_p_m2 = TemperatureCalculationParams_pointer["a3_OHM_Building_W_p_m2"];

	//Albedo_Pavement_frac (fraction) is TemperatureCalculationParams["Albedo_Pavement_frac"] from HydroPlusConfig.xml
	//Note: Albedo_Pavement_frac is 0.14 if comprised of 75% Albedo_Asphalt_frac (=0.1) + 25% Albedo_Concrete_frac (=0.26)
	//Note: The weighting of 75% asphalt and 25% concrete was used to represent roads and sidewalks, but can be changed
	//Albedo_Asphalt_frac is average asphalt albedo (0.05 to 0.15) from Table 2.3 Stull (2000); ~ Paved in Table 2 Ward et al. (2016)
	//Albedo_Concrete_frac is average albedo for concrete (0.15 to 0.37) from Table 2.3 Stull (2000)
	Albedo_Pavement_frac = TemperatureCalculationParams_pointer["Albedo_Pavement_frac"];

	//Emissivity_Pavement_frac (fraction) is emissivity of surface feature for emitting longwave radiation, 1 = perfect emission, 0 = no emission
	//Emissivity_Pavement_frac (fraction) is TemperatureCalculationParams["Emissivity_Pavement_frac"] from HydroPlusConfig.xml
	//Note: Emissivity_Pavement_frac is 0.14 if comprised of 75% Emissivity_Asphalt_frac (=0.95) + 25% Emissivity_Concrete_frac (=0.9)
	//Note: The weighting of 75% asphalt and 25% concrete was used to represent roads and sidewalks, but can be changed
	//Emissivity_Asphalt_frac is asphalt emissivity (0.95) from Table 2.2 Stull (2000)
	//Emissivity_Concrete_frac is maximum emissivity for concrete (0.71 to 0.9) from Table 2.3 Stull (2000), matching other sources
	Emissivity_Pavement_frac = TemperatureCalculationParams_pointer["Emissivity_Pavement_frac"];

	//Albedo_Roof_frac (fraction) is albedo of surface feature for reflecting shortwave radiation; 1 = perfect reflector, 0 = perfect absorber
	Albedo_Roof_frac = TemperatureCalculationParams_pointer["Albedo_Roof_frac"];
	//Emissivity_Roof_frac (fraction) is emissivity of surface feature for emitting longwave radiation, 1 = perfect emission, 0 = no emission
	Emissivity_Roof_frac = TemperatureCalculationParams_pointer["Emissivity_Roof_frac"];
	//Albedo_Wall_frac is 0.09 albedo for buildings wall from Table 2.3 Stull (2000)
	Albedo_Wall_frac = TemperatureCalculationParams_pointer["Albedo_Wall_frac"];
	//Emissivity_Wall_frac is 0.91 avg emissivity for bricks and lumber from Table 2.2 Stull (2000); ~Buildings in Table 2 Ward et al. (2016)
	Emissivity_Wall_frac = TemperatureCalculationParams_pointer["Emissivity_Wall_frac"];

	//Note: Albedo, Emissivity, SkyViewFactor values used in radiation shortwave direct adjustments below
	//Albedo_Road_frac is Albedo_Pavement_frac, where road is treated in TEB of Masson (2000) as ground surfaces with pavement
	Albedo_Road_frac = Albedo_Pavement_frac;
	//Emissivity_Road_frac is Emissivity_Pavement_frac, where road is treated in TEB of Masson (2000) as ground surfaces with pavement
	Emissivity_Road_frac = Emissivity_Pavement_frac;
	//Albedo_Window_frac is albedo for Glass from Table 2 Bradley et al. (2002)
	Albedo_Window_frac = 0.305;
	//Emissivity_Wall_frac is average emissivity for glass from Table 2.2 Stull (2000); ~Glass in Table 2 Bradley et al. (2002)
	Emissivity_Window_frac = 0.905;
	//SkyViewFactor_Ground_Open_frac is 1 for open areas, beyond urban canyons
	SkyViewFactor_Ground_Open_frac = 1;
	//SkyViewFactor_Ground_Wall_frac is 0.5 for open areas, beyond urban canyons, and is multiplied by 2 for both walls
	SkyViewFactor_Ground_Wall_frac = 0.5;
}

//EnergyFlux_WaterFlux_DefineTerms function calculates energy balance once for each folder at each timeStep, before heading into the water balance calculations
void HeatFluxCal::EnergyFlux_WaterFlux_DefineTerms(Inputs* input, DataFolder* folder, bool Flag_simulateReferenceStation, int timeStep, WeatherProcessor* WeatherPro, string StationID_string)
{
	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally
	//Note: RunTemperatureSpatialModel 1st sends MapPixel_ReferenceStation_IDs, regardless of indexing, then sends MapPixel_ID of 0
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions with sending solarCalculator
	HeatMetrics_Calc HeatMetricsCalc(input);

	//If Flag_MultipleStations is true and Flag_simulateReferenceStation is true then read in WeatherPro->WeatherMap ReferenceStation values
	//Note: Conditional obtains reference station Tair_K and Tdew_K (K) for Flag_MultipleStations = 1
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1 && Flag_simulateReferenceStation == 1) {
		//StationMapID_string equals StationID_string
		string StationMapID_string = StationID_string;
		string keyname_Radiation_Longwave_Downwelling_Wpm2 = StationMapID_string + "_" + "Radiation_Longwave_Downwelling_Wpm2";
		string keyname_Radiation_Longwave_Upwelling_Wpm2 = StationMapID_string + "_" + "Radiation_Longwave_Upwelling_Wpm2";
		string keyname_WindSp_mps = StationMapID_string + "_" + "WindSpd_mps";
		string keyname_Tair_F = StationMapID_string + "_" + "Tair_F";
		string keyname_Tdew_F = StationMapID_string + "_" + "Tdew_F";

		//Radiation_Longwave_Down_W_p_m2 (W/m2) is from reference station
		//Note: Radiation is positive downward, w/ Radiation_Longwave_Down_W_p_m2 (W/m2) added within net radiation balances
		Radiation_Longwave_Down_W_p_m2 = WeatherPro->RadiationMap[keyname_Radiation_Longwave_Downwelling_Wpm2][timeStep];
		//Radiation_Longwave_Down_W_p_m2 (W/m2) is from reference station
		//Note: Radiation is positive downward, w/ Radiation_Longwave_Up_W_p_m2 (W/m2) subtracted within net radiation balances
		Radiation_Longwave_Up_W_p_m2 = WeatherPro->RadiationMap[keyname_Radiation_Longwave_Upwelling_Wpm2][timeStep];
		//WindSpeed_Reference_m_p_s (m/w) is the measured wind speed at the reference station
		WindSpeed_Reference_m_p_s = WeatherPro->WeatherMap[keyname_WindSp_mps][timeStep];
		//Tair_localCell_K (K) is converted from Tair_F (F), Fahrenheit, K = (F-32) * 5/9 + 273.15
		Tair_localCell_K = (WeatherPro->WeatherMap[keyname_Tair_F][timeStep] - 32) * 5.0 / 9.0 + 273.15;
		//Tair_referenceCell_K (K) set to Tair_localCell_K (K) for error handling
		Tair_referenceCell_K = Tair_localCell_K;
		//Tdew_localCell_K (K) is converted from Tdew_F (F), Fahrenheit, K = (F-32) * 5/9 + 273.15	
		Tdew_localCell_K = (WeatherPro->WeatherMap[keyname_Tdew_F][timeStep] - 32) * 5.0 / 9.0 + 273.15;
	}
	//Else If Flag_MultipleStations is true and Flag_simulateReferenceStation is false then read in WeatherPro->WeightedInputMap values
	//Note: Conditional either obtains local pixel Tair_K and Tdew_K (K) for Flag_MultipleStations = 0
	else if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1 && Flag_simulateReferenceStation != 1) {
		//MapPixel_ID_string equals StationID_string
		string MapPixel_ID_string = to_string(MapPixel_ID);

		//Radiation_Longwave_Down_W_p_m2 (W/m2) is from reference station
		//Note: Radiation is positive downward, w/ Radiation_Longwave_Down_W_p_m2 (W/m2) added within net radiation balances
		Radiation_Longwave_Down_W_p_m2 = WeatherPro->WeightedInputMap[MapPixel_ID]["Radiation_Longwave_Downwelling_Wpm2"][timeStep];
		//Radiation_Longwave_Down_W_p_m2 (W/m2) is from reference station
		//Note: Radiation is positive downward, w/ Radiation_Longwave_Up_W_p_m2 (W/m2) subtracted within net radiation balances
		Radiation_Longwave_Up_W_p_m2 = WeatherPro->WeightedInputMap[MapPixel_ID]["Radiation_Longwave_Upwelling_Wpm2"][timeStep];
		//WindSpeed_Reference_m_p_s (m/w) is the measured wind speed at the reference station
		WindSpeed_Reference_m_p_s = WeatherPro->WeightedInputMap[MapPixel_ID]["WindSpd_mps"][timeStep];
		//Tair_localCell_K (K) is WeatherPro->WeightedInputMap[MapPixel_ID]["Tair_F"][timeStep] - 32) * 5.0/9.0 + 273.15, not lapse rate adjsuted
		Tair_localCell_K = (WeatherPro->WeightedInputMap[MapPixel_ID]["Tair_F"][timeStep] - 32) * 5.0/9.0 + 273.15;
		//Tair_referenceCell_K (K) set to Tair_localCell_K (K) for error handling
		Tair_referenceCell_K = Tair_localCell_K;
		//Tdew_localCell_K (K) is (WeatherPro->WeightedInputMap[MapPixel_ID]["Tdew_F"][timeStep] - 32) * 5.0 / 9.0 + 273.15, not lapse rate adjsuted
		Tdew_localCell_K = (WeatherPro->WeightedInputMap[MapPixel_ID]["Tdew_F"][timeStep] - 32) * 5.0 / 9.0 + 273.15;
	}
	//Else If Flag_MultipleStations and Flag_simulateReferenceStation are false then read in input-> meteorological values
	//Note: Conditional either obtains reference station and local pixel Tair_K and Tdew_K (K) for Flag_MultipleStations = 0
	//Note: if Flag_MultipleStations false then input-> meteorological values are from single instance of Weather.csv, Radiation.csv 
	//Note: if Flag_MultipleStations true then input-> meteorological values are from weighted instances of Weather.csv, Radiation.csv 
	else {
		//Radiation_Longwave_Down_W_p_m2 (W/m2) is from reference station
		//Note: Radiation is positive downward, w/ Radiation_Longwave_Down_W_p_m2 (W/m2) added within net radiation balances
		Radiation_Longwave_Down_W_p_m2 = input->Radiation_Longwave_Downwelling_Wpm2[timeStep];
		//Radiation_Longwave_Down_W_p_m2 (W/m2) is from reference station
		//Note: Radiation is positive downward, w/ Radiation_Longwave_Up_W_p_m2 (W/m2) subtracted within net radiation balances
		Radiation_Longwave_Up_W_p_m2 = input->Radiation_Longwave_Upwelling_Wpm2[timeStep];
		//WindSpeed_Reference_m_p_s (m/w) is the measured wind speed at the reference station
		WindSpeed_Reference_m_p_s = input->WindSpd_mps[timeStep];
		//Tair_localCell_K (K) is converted from Tair_F (F), Fahrenheit, K = (F-32) * 5/9 + 273.15
		Tair_localCell_K = (input->Tair_F[timeStep] - 32) * 5.0 / 9.0 + 273.15;
		//Tair_referenceCell_K (K) set to Tair_localCell_K (K) for error handling
		Tair_referenceCell_K = Tair_localCell_K;
		//Tdew_localCell_K (K) is converted from Tdew_F (F), Fahrenheit, K = (F-32) * 5/9 + 273.15
		Tdew_localCell_K = (input->Tdew_F[timeStep] - 32) * 5.0 / 9.0 + 273.15;
	}

	//If Tdew_localCell_K > Tair_localCell_K then set equal to handle mistaken weather inputs caused by sensor or rounding errors
	if (Tdew_localCell_K > Tair_localCell_K) { Tdew_localCell_K = Tair_localCell_K; }

	//Tdew_localCell_C (C) is converted from Tdew_localCell_K (K) to degrees C by subtracting 273.15
	double Tdew_localCell_C = Tdew_localCell_K - 273.15;
	//VaporPressure_Actual_kPa (kPa) as Tetens' formula in Eq 4.2.2 Shuttleworth (1993), e_kPa=0.6108*exp[(17.27*T_C)/(237.3+T_C)]
	//Note: Used in UrbanCanyon_RadiationLongwave calculation
	VaporPressure_Actual_kPa = 0.6108 * exp((17.27 * Tdew_localCell_C) / (237.3 + Tdew_localCell_C));

	//LandCover_NLCD_Class (Class) defined with input->LandCover_NLCD_Class for MapPixel_ID
	LandCover_NLCD_Class = input->LandCover_NLCD_Class[MapPixel_ID];
	//TreeCover_frac (frac) defined with folder->ParamDict["TreeCover_frac"], adjusted for each MapPixel_ID
	TreeCover_frac = folder->ParamDict["TreeCover_frac"];
	ImperviousCover_frac = folder->ParamDict["ImperviousCover_frac"];
	SoilCover_noTreeCanopy_frac = folder->ParamDict["SoilCover_noTreeCanopy_frac"];
	PermeablePavementCover_noTreeCanopy_frac = folder->ParamDict["PermeablePavementCover_noTreeCanopy_frac"];
	ShortVegCover_noTreeCanopy_frac = folder->ParamDict["ShortVegCover_noTreeCanopy_frac"];
	ImperviousCover_noTreeCanopy_frac = folder->ParamDict["ImperviousCover_noTreeCanopy_frac"];
	WaterCover_noTreeCanopy_frac = folder->ParamDict["WaterCover_noTreeCanopy_frac"];

	//RoadArea_frac initialized to 0
	double RoadArea_frac = 0;
	//LAI_BAI_Tree_m2pm2 is LAI + BAI tree and has time varying impact on water storage 
	LAI_BAI_Tree_m2pm2 = input->LAI_BAI_Tree_m2_p_m2[timeStep];
	//LAI_BAI_SVeg_m2pm2  is LAI + BAI short vegetation and has time varying impact on water storage 
	LAI_BAI_SVeg_m2pm2 = input->LAI_BAI_SVeg_m2_p_m2[timeStep];
	//LAI_BAI_Tree_m2pm2 is LAI tree and has time varying impact on water storage 
	LAI_Tree_m2pm2 = input->LAI_Tree_m2_p_m2[timeStep];
	//LAI_BAI_SVeg_m2pm2  is LAI short vegetation and has time varying impact on water storage 
	LAI_SVeg_m2pm2 = input->LAI_SVeg_m2_p_m2[timeStep];

	//LAI_Tree_1_m2pm2 defined as 1, the threshold below which tree canopy opacity allows shortwave radiation to the ground
	double LAI_Tree_1_m2pm2 = 1.0;
	//if (input->LAI_Tree_m2_p_m2[timeStep] < LAI_Tree_1_m2pm2 ) then adjust tree canopy opacity to let light to lower pavement
	if (input->LAI_Tree_m2_p_m2[timeStep] < LAI_Tree_1_m2pm2) {
		//LAI_BAI_Tree_min_m2pm2 is LeafAreaIndex_Tree_m2pm2 * TreeCover_Evergreen_frac + BarkAreaIndex_Tree_m2pm2 + LAI_Tree_1_m2pm2
		double LAI_BAI_Tree_min_m2pm2 = input->InputXml["LeafAreaIndex_Tree_m2pm2"] * input->InputXml["TreeCover_Evergreen_frac"] + input->InputXml["BarkAreaIndex_Tree_m2pm2"] + LAI_Tree_1_m2pm2;
		//LAI_BAI_Tree_TS_m2pm2 is LAI_Tree_m2_p_m2[timeStep] + BarkAreaIndex_Tree_m2pm2
		double LAI_BAI_Tree_TS_m2pm2 = input->LAI_Tree_m2_p_m2[timeStep] + input->InputXml["BarkAreaIndex_Tree_m2pm2"];
		//TreeCanopy_Opacity_frac = LAI_BAI_Tree_TS_m2pm2 / LAI_BAI_Tree_min_m2pm2
		//Note: This is relative to the bark area, presuming it alone allows light to penetrate the canopy, even if BAI > 1
		double TreeCanopy_Opacity_frac = LAI_BAI_Tree_TS_m2pm2 / LAI_BAI_Tree_min_m2pm2;
		//TreeCover_frac = TreeCover_frac * TreeCanopy_Opacity_frac; scaled to reduce its presence when LAI < 1
		TreeCover_frac = TreeCover_frac * TreeCanopy_Opacity_frac;
		//ImperviousCover_noTreeCanopy_frac = ImperviousCover_noTreeCanopy_frac * (1 - TreeCanopy_Opacity_frac); increase when LAI < 1
		ImperviousCover_noTreeCanopy_frac = ImperviousCover_noTreeCanopy_frac + TreeCover_frac * (1 - TreeCanopy_Opacity_frac);
		PermeablePavementCover_noTreeCanopy_frac = PermeablePavementCover_noTreeCanopy_frac + TreeCover_frac * (1 - TreeCanopy_Opacity_frac);
	}

	//Consider Refactor to allow smaller depths of ponding for full evaporation, using concept such as Storage_PerviousSurfaceAdheringWater_Max_m
	//Storage_Depression_Max_Pervious_m (m) is depth of storage for ponded water in folder surface, scaled from mm to m with Ratio_m_to_mm
	Storage_Depression_Max_Pervious_m = folder->ParamDict["PerviousDepressionStorage_mm"] * Ratio_m_to_mm;
	//Saturation_WaterStorage_frac (frac) defined as 1, as the value of the evaporative flux coefficient
	Saturation_WaterStorage_frac = 1;

	//Annual_Month_MM of 0 to 11 from ((SimulationDate_GD[timeStep] / 100) % 100) -1; where /100 removes days, %100 removes years, -1 indexes at 0
	Annual_Month_MM = ((input->SimulationDate_GD[timeStep] / 100) % 100) - 1;
	//Diurnal_Hour_HH of 0 to 23 hours from SimulationTime_HMS[] vector w stoi and substr functions, starting at 0, selecting 2 places	
	Diurnal_Hour_HH = stoi(input->SimulationTime_HMS[timeStep].substr(0, 2));


	//If Flag_CoolAir_AnthropogenicHeat_Flux, Flag_AH_Flux_Qcr_Qncr_not_Qtot and Flag_CoolAir_AnthropogenicHeat_Flux are 1 then enter
	//Note: Flag_AH_Flux_Qcr_Qncr_not_Qtot with Qcr is commercial residential AH flux, and Qncr is non commercial residential AH flux
	//Note: Järvi et al. (2011) Eq 3 has daily cooling degree and heating degree adjustment for AH Flux values, for weekday and weekend
	//Note: AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 and AnthropogenicHeat_Flux_Qncr_Avg_Wpm2 are read
	if (input->SimulationNumericalParams["Flag_CoolAir_AnthropogenicHeat_Flux"] == 1 && input->SimulationNumericalParams["Flag_AH_Flux_Qcr_Qncr_not_Qtot"] == 1 && input->SimulationNumericalParams["Flag_CoolAir_AnthropogenicHeat_Flux"] == 1) {
		//AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] = AnthropogenicHeat_Flux_Qcr_Avg_Wpm2[MapPixel_ID] + AnthropogenicHeat_Flux_Qncr_Avg_Wpm2[MapPixel_ID]
		//Note: AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 is sum of AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 and AnthropogenicHeat_Flux_Qncr_Avg_Wpm2 
		//Note: AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 and AnthropogenicHeat_Flux_Qncr_Avg_Wpm2 set to 0 if Flag_CoolAir_AnthropogenicHeat_Flux != 1
		input->AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] = input->AnthropogenicHeat_Flux_Qcr_Avg_Wpm2[MapPixel_ID] + input->AnthropogenicHeat_Flux_Qncr_Avg_Wpm2[MapPixel_ID];
		//If Flag_DegreeHour_Dependent_AH_Flux_Qcr then redefine AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 based on degree hour generating building waste energy 
		//Note: Effect of degree hour on building waste energy is a simplification of detailed system behavior used to capture temporal variability in AHE
		if (input->SimulationNumericalParams["Flag_DegreeHour_Dependent_AH_Flux_Qcr"] == 1) {
			//AnthropogenicHeat_Qcr_Climate_Wpm2 (W/m2) is AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 * AH_Flux_Hour_Ratio[Diurnal_Hour_HH] * AH_Flux_Month_Ratio[Annual_Month_MM]
			//Note: AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 is annual average value, and AH_Flux_Hour_Ratio scales it to hour, AH_Flux_Month_Ratio scales it to month
			//Note: AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 is primarily commercial and residential heating and cooling, which can be influenced by tree cover scenario
			//Note: AnthropogenicHeat_Flux_Qcr_Avg_Wpm2 set to 0 if Flag_CoolAir_AnthropogenicHeat_Flux != 1
			//Note: Diurnal_Hour_HH ranges from 0 to 23, and [Diurnal_Month_MM] ranges from 0 to 11 when extracted above by using -1 
			double AnthropogenicHeat_Qcr_Climate_Wpm2 = input->AnthropogenicHeat_Flux_Qcr_Avg_Wpm2[MapPixel_ID] * input->AH_Flux_Hour_Ratio[Diurnal_Hour_HH] * input->AH_Flux_Month_Ratio[Annual_Month_MM];
			//Tair_threshold_K (K) is typiclly 291.483 K, based on 65 F (or 18.333 C) as average daily temperature for cooling and heating degree day threshold
			double DegreeHour_Threshold_K = input->SimulationNumericalParams["DegreeHour_Threshold_K"];
			//Climate_Year_index of 0 to # from int (SimulationDate_GD[timeStep] /10000) - Climate_Year_Start_YYYY
			int Climate_Year_index = int(input->SimulationDate_GD[timeStep] / 10000) - WeatherProcessor::Climate_Year_Start_YYYY;
			
			//Annual_Month_MM of 0 to 11 from ((SimulationDate_GD[timeStep] / 100) % 100) -1; 
			//Note: where /100 removes days, %100 removes years, -1 indexes at 0
			Annual_Month_MM = ((input->SimulationDate_GD[timeStep] / 100) % 100) - 1;
			//Diurnal_Hour_HH of 0 to 23 hours from SimulationTime_HMS[] vector, starting at 0, selecting 2 places
			Diurnal_Hour_HH = stoi(input->SimulationTime_HMS[timeStep].substr(0, 2));
			//If timeStep > 0 then use prior timeStep to align with calculation of DegreeHour_Hour_Scenario
			if (timeStep > 0) {
				//Annual_Month_MM of 0 to 11 from ((SimulationDate_GD[timeStep] / 100) % 100) -1; 
				//Note: where /100 removes days, %100 removes years, -1 indexes at 0
				Annual_Month_MM = ((input->SimulationDate_GD[timeStep - 1] / 100) % 100) - 1;
				//Diurnal_Hour_HH of 0 to 23 hours from SimulationTime_HMS[] vector, starting at 0, selecting 2 places	
				Diurnal_Hour_HH = stoi(input->SimulationTime_HMS[timeStep - 1].substr(0, 2));
			}
			//Tair_priorTS_K is ternary function of timeStep, if not 0 then folder Tair_K, else Tair_localCell_K 
			//Note: Tair_priorTS_K is for the prior timestep, to represent how it affected subsequent anthropogenic heat flux
			double Tair_priorTS_K = (timeStep != 0) ? folder->VarDict["Tair_K"] : Tair_localCell_K;
			double Tdew_priorTS_K = (timeStep != 0) ? folder->VarDict["Tdew_K"] : Tdew_localCell_K;

			//VaporPres_actual_priorTS_mb = HeatMetricsCalc.VaporPressure_haPa_Calc(Tdew_localCell_K, 0); 0 signifies liquid water
			double VaporPres_actual_priorTS_mb = HeatMetricsCalc.VaporPressure_haPa_Calc(Tdew_localCell_K, 0);
			//VaporPres_saturation_priorTS_mb = HeatMetricsCalc.VaporPressure_haPa_Calc(Tair_localCell_K, 0); 0 signifies liquid water
			double VaporPres_saturation_priorTS_mb = HeatMetricsCalc.VaporPressure_haPa_Calc(Tair_localCell_K, 0);
			//RelativeHumidity_priorTS_frac = VaporPres_actual_priorTS_mb / VaporPres_saturation_priorTS_mb; fraction units needed
			double RelativeHumidity_priorTS_frac = VaporPres_actual_priorTS_mb / VaporPres_saturation_priorTS_mb;
			//Humidity_Effect_Coeff initialized to 1; adjusted based on RelativeHumidity_priorTS_frac and cooling or heating season
			double Humidity_Effect_Coeff = 1; 
			//Humidity_Cooling_Base set to 0.4, to avoid humidity extremes turning off all cooling
			double Humidity_Cooling_Base = 0.4;
			//Humidity_Heating_Base set to 0.6, to avoid humidity extremes turning off all heating
			double Humidity_Heating_Base = 0.6;
			//Humidity_Effect_Power = 1.5, to increase effect with larger humidity extremes
			double Humidity_Effect_Power = 1.5;

			//DegreeHour_Hour_Scenario = Tair_priorTS_K - DegreeHour_Threshold_K; use HeatIndex to capture effect of humidity
			//Note: Tair_priorTS_K is from last time step; avoids implicit calculation given it is function of AH Flux 
			//Note: Hourly rather than daily values are used to modify AH Flux; greater temporal resolution on heating and cooling
			double DegreeHour_Hour_Scenario = Tair_priorTS_K - DegreeHour_Threshold_K;

			//DegreeHour_Hour_Climate is initialized to zero, updated to prior time step based on climate data
			double DegreeHour_Hour_Climate = 0;
			//If DegreeHour_Hour_Scenario > 0 then Cooling with Tair_priorTS_K > DegreeHour_Threshold_K
			if (DegreeHour_Hour_Scenario > 0) {
				//DegreeHour_Hour_Climate = DegreeHour_Year_Cooling_Ratio * DegreeHour_Month_Cooling_Ratio * DegreeHour_Hour_Cooling_K
				//Note: Equation developed by Endreny after analyzing 10 yrs of hourly temperature data with the DegreeHour_Threshold_K
				DegreeHour_Hour_Climate = WeatherProcessor::DegreeHour_Year_Cooling_Ratio_vec[Climate_Year_index] * WeatherProcessor::DegreeHour_Month_Cooling_Ratio_vec[Annual_Month_MM] * WeatherProcessor::DegreeHour_Hour_Cooling_K_vec[Diurnal_Hour_HH];
				//Humidity_Effect_Coeff = Humidity_Cooling_Base + pow(RelativeHumidity_priorTS_frac, Humidity_Effect_Power)
				//Note: RelativeHumidity_priorTS_frac grows as power function to capture increased cooling cost with higher humidity
				Humidity_Effect_Coeff = Humidity_Cooling_Base + pow(RelativeHumidity_priorTS_frac, Humidity_Effect_Power);
			}
			//Else If DegreeHour_Hour_Scenario < 0 then Heating with Tair_priorTS_K < DegreeHour_Threshold_K
			else if (DegreeHour_Hour_Scenario < 0) {
				//DegreeHour_Hour_Climate = DegreeHour_Year_Heating_Ratio * DegreeHour_Month_Heating_Ratio * DegreeHour_Hour_Heating_K
				//Note: Equation developed by Endreny after analyzing 10 yrs of hourly temperature data with the DegreeHour_Threshold_K
				DegreeHour_Hour_Climate = WeatherProcessor::DegreeHour_Year_Heating_Ratio_vec[Climate_Year_index] * WeatherProcessor::DegreeHour_Month_Heating_Ratio_vec[Annual_Month_MM] * WeatherProcessor::DegreeHour_Hour_Heating_K_vec[Diurnal_Hour_HH];
				//Humidity_Effect_Coeff = Humidity_Heating_Base + pow(1.0 - RelativeHumidity_priorTS_frac, Humidity_Effect_Power)
				//Note: (1-RelativeHumidity_priorTS_frac) grows as power function to capture decreased heating cost with higher humidity
				Humidity_Effect_Coeff = Humidity_Heating_Base + pow(1.0 - RelativeHumidity_priorTS_frac, Humidity_Effect_Power);
			}
			//Qcr_elevated_Wpm2; baseline AH flux to normalize against, approaching elevated residential, commercial values
			double Qcr_elevated_Wpm2 = 50.0;
			//Qcr_min_scaling bounds based on climate AH flux
			double Qcr_min_scaling = 0.01;
			//Qcr_max_scaling = Qcr_elevated_Wpm2 / max(AnthropogenicHeat_Qcr_Climate_Wpm2, 1.0); 
			//Note: AnthropogenicHeat_Qcr_Climate_Wpm2 dynamically sets maximum limit, to scale for reasonable final AH flux
			double Qcr_max_scaling = Qcr_elevated_Wpm2 / max(AnthropogenicHeat_Qcr_Climate_Wpm2, 1.0);
			//Qcr_max_scaling = clamp(Qcr_max_scaling, 2.0, Qcr_elevated_Wpm2); allows doubling of large AH Flux values
			Qcr_max_scaling = clamp(Qcr_max_scaling, 2.0, Qcr_elevated_Wpm2);
			//AnthropogenicHeat_Qcr_Scenario_Wpm2 (W/m2) initialized to AnthropogenicHeat_Qcr_Climate_Wpm2
			double AnthropogenicHeat_Qcr_Scenario_Wpm2 = AnthropogenicHeat_Qcr_Climate_Wpm2;
			//DegreeHour_Ratio = abs(input->SafeDivide(DegreeHour_Hour_Scenario, DegreeHour_Hour_Climate, 0.0));
			//Note: Use abs to ensure scaling ratio and AH Flux remain positive; set to 0 if either DegreeHour is 0
			double DegreeHour_Ratio = abs(input->SafeDivide(DegreeHour_Hour_Scenario, DegreeHour_Hour_Climate, 0.0));
			//DegreeHour_Humidity_Scaling_Ratio = Humidity_Effect_Coeff * DegreeHour_Ratio
			double DegreeHour_Humidity_Scaling_Ratio = Humidity_Effect_Coeff * DegreeHour_Ratio;
			//DegreeHour_Humidity_Scaling_Ratio = clamp(DegreeHour_Scaling_Ratio, Qcr_min_scaling, Qcr_max_scaling)
			//Note: Constrained adjustment of AnthropogenicHeat_Qcr_Climate_Wpm2 to maintain model stability
			DegreeHour_Humidity_Scaling_Ratio = clamp(DegreeHour_Humidity_Scaling_Ratio, Qcr_min_scaling, Qcr_max_scaling);
			//AnthropogenicHeat_Qcr_Scenario_Wpm2 = DegreeHour_Humidity_Scaling_Ratio * AnthropogenicHeat_Qcr_Climate_Wpm2
			//Note: Weighted scaling of AH Flux inspired by Järvi et al. (2011) Eq 3 has daily cooling degree and heating degree adjustment
			AnthropogenicHeat_Qcr_Scenario_Wpm2 = DegreeHour_Humidity_Scaling_Ratio * AnthropogenicHeat_Qcr_Climate_Wpm2;

			//AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] = AnthropogenicHeat_Qcr_Scenario_Wpm2 + AnthropogenicHeat_Flux_Qncr_Avg_Wpm2[MapPixel_ID]
			//Note: AnthropogenicHeat_Flux_Qncr_Avg_Wpm2 set to 0 if Flag_CoolAir_AnthropogenicHeat_Flux != 1
			input->AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] = AnthropogenicHeat_Qcr_Scenario_Wpm2 + input->AnthropogenicHeat_Flux_Qncr_Avg_Wpm2[MapPixel_ID];

			///If (DegreeHour_Hour_Scenario > (DegreeHour_Hour_Climate * 30)) then alert user of large difference
			//Note: Consider refactor to alert user when DegreeHour_Hour_Scenario >> (DegreeHour_Hour_Climate and AHE is overadjusted
			if (DegreeHour_Hour_Scenario > (DegreeHour_Hour_Climate * 30)) {
				cout << "Note: HydroPlus Flag_DegreeHour_Dependent_AH_Flux_Qcr has led to an instability with anthropogenic Heat." << endl;
				cout << "  This occurred at timestep " << timeStep << " and date hour " << input->SimulationDate_GDH[timeStep] << endl;
				cout << "  This occurred at MapPixel_ID " << MapPixel_ID << endl;
				cout << "  Climate_Year_index= " << Climate_Year_index << endl;
				cout << "  Annual_Month_MM= " << Annual_Month_MM << endl;
				cout << "  Diurnal_Hour_HH= " << Diurnal_Hour_HH << endl;
				cout << "  DegreeHour_Hour_Scenario= " << DegreeHour_Hour_Scenario << endl;
				cout << "  DegreeHour_Hour_Climate= " << DegreeHour_Hour_Climate << endl;
				cout << "  DegreeHour_Year_Cooling_Ratio_vec= " << WeatherProcessor::DegreeHour_Year_Cooling_Ratio_vec[Climate_Year_index] << endl;
				cout << "  DegreeHour_Month_Cooling_Ratio_vec= " << WeatherProcessor::DegreeHour_Month_Cooling_Ratio_vec[Annual_Month_MM] << endl;
				cout << "  DegreeHour_Hour_Cooling_K_vec= " << WeatherProcessor::DegreeHour_Hour_Cooling_K_vec[Diurnal_Hour_HH] << endl;
				cout << "  AnthropogenicHeat_Qcr_Scenario_Wpm2= " << AnthropogenicHeat_Qcr_Scenario_Wpm2 << endl;
				cout << "  AnthropogenicHeat_Qcr_Climate_Wpm2= " << AnthropogenicHeat_Qcr_Climate_Wpm2 << endl;
				cout << "  AnthropogenicHeat_Flux_Qcr_Avg_Wpm2= " << input->AnthropogenicHeat_Flux_Qcr_Avg_Wpm2[MapPixel_ID] << endl;
				cout << "  AH_Flux_Hour_Ratio= " << input->AH_Flux_Hour_Ratio[Diurnal_Hour_HH] << endl;
				cout << "  AH_Flux_Month_Ratio= " << input->AH_Flux_Month_Ratio[Annual_Month_MM] << endl;
				cout << "  Tair_localCell_K = " << Tair_localCell_K << endl;
				cout << "  Tair_referenceCell_K = " << Tair_referenceCell_K << endl;
				cout << "  Tair_mesoScale_K= " << Tair_mesoScale_K << endl;
				cout << "  Tair_priorTS_K= " << Tair_priorTS_K << endl;
				cout << "  DegreeHour_Threshold_K= " << DegreeHour_Threshold_K << endl;
				cout << "  RelativeHumidity_priorTS_frac= " << RelativeHumidity_priorTS_frac << endl;
				cout << "  Humidity_Effect_Coeff= " << Humidity_Effect_Coeff << endl;
				cout << "  DegreeHour_Ratio= " << DegreeHour_Ratio << endl;
				cout << "  DegreeHour_Humidity_Scaling_Ratio= " << DegreeHour_Humidity_Scaling_Ratio << endl;
				cout << endl;
			}
		}
	}

	//AnthropogenicHeat_Flux_Total_Wpm2 (W/m2) is AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 * AH_Flux_Hour_Ratio[Diurnal_Hour_HH] * AH_Flux_Month_Ratio[Annual_Month_MM]
	//Note: AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 is annual average hourly value
	//Note: AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 scaled by AH_Flux_Hour_Ratio to hour, by AH_Flux_Month_Ratio scales to month
	//Note: AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 contains Qcr (commercial and residential) and Qncr (non-cr), for total AH Flux
	//Note: Diurnal_Hour_HH ranges from 0 to 23, and [Diurnal_Month_MM] ranges from 0 to 11 when extracted above by using -1 
	//Note: AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 set to 0 if Flag_CoolAir_AnthropogenicHeat_Flux != 1
	//Note: Anthropogenic heat emission (W/m2) is generated and associated with land where vehicles, people, and buildings are placed
	//Note: Järvi et al. (2011) Table 3 has diurnal pattern for AH Flux values, generated 0.308 Wpm2 per capita for weekday and weekend 
	//Note: Ward et al. (2016) demonstrated how anthropogenic heat is added to last coefficient in OHM w 
	//Note: Ward et al. (2016) developed Eq 5 with Table 3 coefficient values to predict annual variation in anthropogenic heat.
	AnthropogenicHeat_Flux_Total_Wpm2 = input->AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] * input->AH_Flux_Hour_Ratio[Diurnal_Hour_HH] * input->AH_Flux_Month_Ratio[Annual_Month_MM];

	//NLCD Class is 21
	if (LandCover_NLCD_Class == 21) {
		//Height_UrbanBuilding_m (m) is average height of surface feature, transformed later for roughness in turbulence exchange calculations
		Height_UrbanBuilding_m = input->TemperatureCalculationParams["Height_Avg_Urban21_m"];
		//Width_UrbanCanyon_m (m) is average width of surface feature for urban canyon calculations
		Width_UrbanCanyon_m = input->TemperatureCalculationParams["Width_Road_Urban21_m"];
		//Building_Area_frac (fraction) is area of folder surface with buildings vs road; if TC > (1-Building_Area_frac), then TC is on roof
		Building_Area_frac = input->TemperatureCalculationParams["Area_Roof_Urban21_frac"];
		//a1_OHM_Urban_frac, a2_OHM_Urban_hr, a3_OHM_Urban_W_p_m2 are terms for NLCD 21 to 24, based on weighted Pavement and Building a1 to a3 values 
	}
	//NLCD Class is 22
	if (LandCover_NLCD_Class == 22) {
		//Height_UrbanBuilding_m (m) is average height of surface feature, transformed later for roughness in turbulence exchange calculations
		Height_UrbanBuilding_m = input->TemperatureCalculationParams["Height_Avg_Urban22_m"];
		//Width_UrbanCanyon_m (m) is average width of surface feature for urban canyon calculations
		Width_UrbanCanyon_m = input->TemperatureCalculationParams["Width_Road_Urban22_m"];
		//Building_Area_frac (fraction) is area of folder surface with buildings vs road; if TC > (1-Building_Area_frac), then TC is on roof
		Building_Area_frac = input->TemperatureCalculationParams["Area_Roof_Urban22_frac"];
		//a1_OHM_Urban_frac, a2_OHM_Urban_hr, a3_OHM_Urban_W_p_m2 are terms for NLCD 21 to 24, based on weighted Pavement and Building a1 to a3 values 
	}
	//NLCD Class is 23
	if (LandCover_NLCD_Class == 23) {
		//Height_UrbanBuilding_m (m) is average height of surface feature, transformed later for roughness in turbulence exchange calculations
		Height_UrbanBuilding_m = input->TemperatureCalculationParams["Height_Avg_Urban23_m"];
		//Width_UrbanCanyon_m (m) is average width of surface feature for urban canyon calculations
		Width_UrbanCanyon_m = input->TemperatureCalculationParams["Width_Road_Urban23_m"];
		//Building_Area_frac (fraction) is area of folder surface with buildings vs road; if TC > (1-Building_Area_frac), then TC is on roof
		Building_Area_frac = input->TemperatureCalculationParams["Area_Roof_Urban23_frac"];
		//a1_OHM_Urban_frac, a2_OHM_Urban_hr, a3_OHM_Urban_W_p_m2 are terms for NLCD 21 to 24, based on weighted Pavement and Building a1 to a3 values 
	}
	//NLCD Class is 24
	if (LandCover_NLCD_Class == 24) {
		//Height_UrbanBuilding_m (m) is average height of surface feature, transformed later for roughness in turbulence exchange calculations
		Height_UrbanBuilding_m = input->TemperatureCalculationParams["Height_Avg_Urban24_m"];
		//Width_UrbanCanyon_m (m) is average width of surface feature for urban canyon calculations
		Width_UrbanCanyon_m = input->TemperatureCalculationParams["Width_Road_Urban24_m"];
		//Building_Area_frac (fraction) is area of folder surface with buildings vs road; if TC > (1-Building_Area_frac), then TC is on roof
		Building_Area_frac = input->TemperatureCalculationParams["Area_Roof_Urban24_frac"];
		//a1_OHM_Urban_frac, a2_OHM_Urban_hr, a3_OHM_Urban_W_p_m2 are terms for NLCD 21 to 24, based on weighted Pavement and Building a1 to a3 values 
	}
		
	//If NLCD Class < 21 or > 24 then not urban then set parameters for impervious without buildings
	//Note: Consider refactor to define for reference station the Height_UrbanBuilding_m, Width_UrbanCanyon_m and Building_Area_frac when Flag_simulateReferenceStation == 1
	if ((LandCover_NLCD_Class < 21 || LandCover_NLCD_Class > 24)) {
		//Height_UrbanBuilding_m (m) is set H = 0.25 m, height for impervious cover without buildings
		//Note: Used to compute Height_LandCover_Average_m, and can be changed as needed by developers
		Height_UrbanBuilding_m = 0.25;
		//Width_UrbanCanyon_m (m) is average width of surface feature for urban canyon calculations
		Width_UrbanCanyon_m = input->TemperatureCalculationParams["Width_Road_Urban21_m"];
		//Building_Area_frac (fraction) is set to 0.1 for impervious cover without buildings
		Building_Area_frac = 0.0;
	}

	//Storage_Depression_Max_Impervious_m (m) is depth of storage for ponded water in folder surface, scaled from mm to m with Ratio_m_to_mm
	//Note: ParamDict["ImperviousDepressionStorage_mm"] was created as weighted average in BuildDataOrganizer.cpp
	Storage_Depression_Max_Impervious_m = folder->ParamDict["ImperviousDepressionStorage_mm"] * Ratio_m_to_mm;

	//Albedo_Urban_frac is Building_Area_frac * Albedo_Roof_frac + (1 - Building_Area_frac) * Albedo_Pavement_frac
	//Note: Albedo_Urban_frac used when NLCD Class not urban, and UrbanCanyon_RadiationLongwave not called
	//Note: Albedo_Urban_frac represents the horizontal impervious surfaces of roof and pavement; vertical surfaces handled elsewhere
	//Note: Li et al. (2014) showed roofs contribute to surface and near-surface urban heat island effects alongside pavement
	Albedo_Urban_frac = Building_Area_frac * Albedo_Roof_frac + (1 - Building_Area_frac) * Albedo_Pavement_frac;
	//Emissivity_Urban_frac is Building_Area_frac * Emissivity_Roof_frac + (1 - Building_Area_frac) * Emissivity_Pavement_frac
	//Note: Emissivity_Urban_frac used when NLCD Class not urban, and UrbanCanyon_RadiationLongwave not called
	//Note: Emissivity_Urban_frac represents the horizontal impervious surfaces of roof and pavement; vertical surfaces handled elsewhere
	//Note: Li et al. (2014) showed roofs contribute to surface and near-surface urban heat island effects alongside pavement
	Emissivity_Urban_frac = Building_Area_frac * Emissivity_Roof_frac + (1 - Building_Area_frac) * Emissivity_Pavement_frac;

	//a1_OHM_Urban_frac = a1_OHM_Pavement_frac * (1.0 - Building_Area_frac) + a1_OHM_Building_frac * Building_Area_frac
	//Note: Equation developed by Endreny based on Ward et al. (2016) Table 4 values and fitting to NLCD Class 21 to 24 values 
	a1_OHM_Urban_frac = a1_OHM_Pavement_frac * (1.0 - Building_Area_frac) + a1_OHM_Building_frac * Building_Area_frac;
	//a2_OHM_Urban_hr = a2_OHM_Pavement_hr * (1.0 - Building_Area_frac) + a2_OHM_Building_hr * Building_Area_frac
	//Note: Equation developed by Endreny based on Ward et al. (2016) Table 4 values and fitting to NLCD Class 21 to 24 values 
	a2_OHM_Urban_hr = a2_OHM_Pavement_hr * (1.0 - Building_Area_frac) + a2_OHM_Building_hr * Building_Area_frac;
	//a3_OHM_Urban_W_p_m2 = a3_OHM_Pavement_W_p_m2 * (1.0 - Building_Area_frac) + a3_OHM_Building_W_p_m2 * Building_Area_frac
	//Note: Equation developed by Endreny based on Ward et al. (2016) Table 4 values and fitting to NLCD Class 21 to 24 values 
	a3_OHM_Urban_W_p_m2 = a3_OHM_Pavement_W_p_m2 * (1.0 - Building_Area_frac) + a3_OHM_Building_W_p_m2 * Building_Area_frac;

	//If (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) > Building_Area_frac then
	if ((ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) > Building_Area_frac) {
		//RoadArea_frac = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) - Building_Area_frac;
		//Note: Algorithm is creating RoadArea_frac from impervious cover not occupied by Building_Area_frac
		RoadArea_frac = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) - Building_Area_frac;
	}
	//Else Building_Area_frac occupies all impervious cover and 
	else {
		//RoadArea_frac = 0.0
		RoadArea_frac = 0.0;
	}

	//If Building_Area_frac < 1 then proceed to define ratios of ground based on cover
	if (Building_Area_frac < 1) {
		//Ratio_Road_to_Ground defined as RoadArea_frac / (1.0 - Building_Area_frac)
		//Ratio_Road_to_Ground is defined by TEB as ZRD = T%XROAD(JJ) / (T%XROAD(JJ)+T%XGARDEN(JJ))
		Ratio_Road_to_Ground = RoadArea_frac / (1.0 - Building_Area_frac);
		//Ratio_Garden_to_Ground defined as 1 - Ratio_Road_to_Ground
		Ratio_Garden_to_Ground = 1 - Ratio_Road_to_Ground;
	}
	//If Building_Area_frac = 1 then proceed to define ratios of ground as zero
	else {
		//Ratio_Road_to_Ground defined as zero given there is no ground
		Ratio_Road_to_Ground = 0;
		//Ratio_Garden_to_Ground defined as zero given there is no ground
		Ratio_Garden_to_Ground = 0;
	}	

	//If (TreeCover_frac + ShortVegCover_noTreeCanopy_frac) > 0 then enter for division
	if ((TreeCover_frac + ShortVegCover_noTreeCanopy_frac + WaterCover_noTreeCanopy_frac + SoilCover_noTreeCanopy_frac) > 0) {
		//Albedo_Garden_frac presumed weighted fraction of albedo for all non-impervious surfaces, but this could be defined in HydroPlusConfig.xml
		Albedo_Garden_frac = (TreeCover_frac * Albedo_Tree_frac + ShortVegCover_noTreeCanopy_frac * Albedo_SVeg_frac + WaterCover_noTreeCanopy_frac * Albedo_Water_frac + SoilCover_noTreeCanopy_frac * Albedo_Soil_frac) /
			(TreeCover_frac + ShortVegCover_noTreeCanopy_frac + WaterCover_noTreeCanopy_frac + SoilCover_noTreeCanopy_frac);
		//Emissivity_Garden_frac presumed fraction of emissivity for all non-impervious surfaces, but this could be defined in HydroPlusConfig.xml
		Emissivity_Garden_frac = (TreeCover_frac * Emissivity_Tree_frac + ShortVegCover_noTreeCanopy_frac * Emissivity_SVeg_frac + WaterCover_noTreeCanopy_frac * Emissivity_Water_frac + SoilCover_noTreeCanopy_frac * Emissivity_Soil_frac) /
			(TreeCover_frac + ShortVegCover_noTreeCanopy_frac + WaterCover_noTreeCanopy_frac + SoilCover_noTreeCanopy_frac);
	}
	//Else If (TreeCover_frac + ShortVegCover_noTreeCanopy_frac) <= 0 then define terms without division by zero
	else {
		//Albedo_Garden_frac presumed equal parts non-impervious cover, but this could be defined in HydroPlusConfig.xml
		Albedo_Garden_frac = (Albedo_Tree_frac + Albedo_SVeg_frac + Albedo_Water_frac + Albedo_Soil_frac) / 4;
		//Emissivity_Garden_frac presumed equal parts non-impervious cover, but this could be defined in HydroPlusConfig.xml
		Emissivity_Garden_frac = (Emissivity_Tree_frac + Emissivity_SVeg_frac + Emissivity_Water_frac + Emissivity_Soil_frac) / 4;
	}

	//Height_LandCover_Min_m (m) defined as 0.001 m
	double Height_LandCover_Min_m = 0.001;
	//Height_LandCover_Average_m (m) is weighted sum of tree cover and all other land cover types without tree cover
	//Note: Height_LandCover_Average_m is computed as simple average, not log based which would likely overemphasize influence of small heights
	//Note: Refactor Option: Create Height_PermeablePavement_m for PermeablePavementCover_noTreeCanopy_frac and ...
	//Note: Refactor Option: Create Height_ImpermeableCover_m as alternative to Height_UrbanBuilding_m
	Height_LandCover_Average_m = TreeCover_frac * Height_Tree_m + ShortVegCover_noTreeCanopy_frac * Height_SVeg_m +
		SoilCover_noTreeCanopy_frac * Height_Soil_m + PermeablePavementCover_noTreeCanopy_frac * Height_Soil_m + WaterCover_noTreeCanopy_frac * Height_Water_m + ImperviousCover_noTreeCanopy_frac * Height_UrbanBuilding_m;

	//If Height_LandCover_Average_m < Height_LandCover_Min_m (e.g., 0.001 m) then Height_LandCover_Average_m = Height_LandCover_Min_m
	//RoughnessLength_Water_m = 0.00137 m, based on roughness length (m) for water set to z0 = 0.00137 m Chin (2021) and Thom and Oliver (1977)
	//Note: Chin, D. A. (2021). Water Resources Engineering, Fourth Edition. Hoboken, NJ: Pearson Education.
	//Note: Thom, A. S., & Oliver, H. R. (1977). On Penman's equation for estimating regional evaporation. Quarterly Journal of the Royal Meteorological Society, 103(436), 345-357. doi:https://doi.org/10.1002/qj.49710343610
	if (Height_LandCover_Average_m < Height_LandCover_Min_m) { Height_LandCover_Average_m = Height_LandCover_Min_m; }

	//Storage_per_LAI_BAI_Tree_m (m) = LeafAreaStorage_Tree_mm * Ratio_m_to_mm * LAI_Tree_m2_p_m2 / LAI_BAI_Tree_m2_p_m2 + BarkAreaStorage_Tree_mm * Ratio_m_to_mm * (LAI_BAI_Tree_m2_p_m2 - LAI_Tree_m2_p_m2) / LAI_BAI_Tree_m2_p_m2
	//Note: Storage_per_LAI_BAI_Tree_m is per unit area of tree, not the total depth in the tree
	Storage_per_LAI_BAI_Tree_m = input->InputXml["LeafAreaStorage_Tree_mm"] * Ratio_m_to_mm * (input->LAI_Tree_m2_p_m2[timeStep] / input->LAI_BAI_Tree_m2_p_m2[timeStep]) + input->InputXml["BarkAreaStorage_Tree_mm"] * Ratio_m_to_mm * ((input->LAI_BAI_Tree_m2_p_m2[timeStep] - input->LAI_Tree_m2_p_m2[timeStep]) / input->LAI_BAI_Tree_m2_p_m2[timeStep]);

	//Component (S/Smax) of Eq 22 in Yang et al. (2013)
	//If Storage_per_LAI_BAI_Tree_m * LAI_BAI_Tree_m2pm2 > 0 then enter for division
	if (Storage_per_LAI_BAI_Tree_m * LAI_BAI_Tree_m2pm2 > 0) {
		//Saturation_TreeStorage_frac is function of Storage_per_LAI_BAI_Tree_m (LAI and BAI unit storage depth) and LAI_BAI_Tree_m2pm2
		Saturation_TreeStorage_frac = folder->VarDict["Storage_Rain_TreeCanopy_m"] / (Storage_per_LAI_BAI_Tree_m * LAI_BAI_Tree_m2pm2);
		//Saturation_TreeStorage_frac precision = 3 to maintain constant value for same weather station across quads
		//Note: Without precision=3 a rounding errors cause changes in subsequent LE and H calculations when Flag_MultipleStations=1
		Saturation_TreeStorage_frac = Inputs::roundToDecimalPlaces(Saturation_TreeStorage_frac, 3);
	}
	//Else If Storage_per_LAI_BAI_Tree_m * LAI_BAI_Tree_m2pm2 = 0 then there is no storage potential and set Saturation_TreeStorage_frac = 0
	else {
		Saturation_TreeStorage_frac = 0;
	}

	//Storage_per_LAI_BAI_SVeg_m (m) = LeafAreaStorage_SVeg_mm * Ratio_m_to_mm * LAI_SVeg_m2_p_m2 / LAI_BAI_SVeg_m2_p_m2 + BarkAreaStorage_SVeg_mm * Ratio_m_to_mm * (LAI_BAI_SVeg_m2_p_m2 - LAI_SVeg_m2_p_m2) / LAI_BAI_SVeg_m2_p_m2
	//Note: Storage_per_LAI_BAI_SVeg_m is per unit area of short veg, not the total depth in the short veg
	Storage_per_LAI_BAI_SVeg_m = input->InputXml["LeafAreaStorage_SVeg_mm"] * Ratio_m_to_mm * (input->LAI_SVeg_m2_p_m2[timeStep] / input->LAI_BAI_SVeg_m2_p_m2[timeStep]) + input->InputXml["BarkAreaStorage_SVeg_mm"] * Ratio_m_to_mm * ((input->LAI_BAI_SVeg_m2_p_m2[timeStep] - input->LAI_SVeg_m2_p_m2[timeStep]) / input->LAI_BAI_SVeg_m2_p_m2[timeStep]);

	//Component (S/Smax) of Eq 22 in Yang et al. (2013)
	//If Storage_per_LAI_BAI_SVeg_m * LAI_BAI_SVeg_m2pm2 > 0 then enter for division
	if (Storage_per_LAI_BAI_SVeg_m * LAI_BAI_SVeg_m2pm2 > 0) {
		//Saturation_SVegStorage_frac is function of Storage_per_LAI_BAI_SVeg_m (LAI and BIA unit storage depth) and LAI_BAI_SVeg_m2pm2
		Saturation_SVegStorage_frac = folder->VarDict["Storage_Rain_SVegCanopy_m"] / (Storage_per_LAI_BAI_SVeg_m * LAI_BAI_SVeg_m2pm2);
		//Saturation_SVegStorage_frac precision = 3 to maintain constant value for same weather station across quads
		//Note: Without precision=3 a rounding errors cause changes in subsequent LE and H calculations when Flag_MultipleStations=1
		Saturation_SVegStorage_frac = Inputs::roundToDecimalPlaces(Saturation_SVegStorage_frac, 3);
	}
	//Else If Storage_per_LAI_BAI_Tree_m * LAI_BAI_Tree_m2pm2 = 0 then there is no storage potential and set Saturation_SVegStorage_frac = 0
	else {
		Saturation_SVegStorage_frac = 0;
	}															

	//Component [1-(D/Dmax)] of Eq 20 in Yang et al. (2013); available capillary held water in the soil for evaporation
	//If StorageDeficit_FieldCapacity_SoilEvapZone_Max_m (m) > zero then enter for division
	if (folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"] > 0) {
		//Saturation_SoilStorage_frac (frac) ranges from 0=dry, 1=saturated; available water in the soil to evaporate; 
		//Note: Saturation_SoilStorage_frac equals [1-(D/Dmax)] in Eq 20 of Yang et al. (2013)
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_m ranges frm 0=saturated to >0 dry; available storage up to field capacity
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_m is constructed from StorageDeficit_FieldCapacity_SoilEvapZone_TI_m[ia] in ...
		//Note: ... RootZoneEvapoTranspirationCalc.cpp, where ia is sorted from 0 = wettest to N = driest and N = 1 for CoolAir, therefore ...
		//Note: ... StorageDeficit_FieldCapacity_SoilEvapZone_m = StorageDeficit_FieldCapacity_SoilEvapZone_TI_m[0] in CoolAir. 
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_TS_first_m is function of Soil_MoistureInitial_m3pm3 in BuildDataOrganizer.
		Saturation_SoilStorage_frac = 1 - (folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_m"] / folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"]);
		//Saturation_SoilStorage_frac precision = 3 to maintain constant value for same weather station across quads
		//Note: Without precision=3 a rounding errors cause changes in subsequent LE and H calculations when Flag_MultipleStations=1
		Saturation_SoilStorage_frac = Inputs::roundToDecimalPlaces(Saturation_SoilStorage_frac, 3);
	}
	//Else StorageDeficit_FieldCapacity_SoilEvapZone_Max_m (m) is zero, then no deficit and Saturation_SoilStorage_frac is 1
	else {
		//Saturation_SoilStorage_frac (frac) ranges from 0=dry, 1=saturated; available water in the soil to evaporate
		Saturation_SoilStorage_frac = 1;
	}

	//a1_OHM_Tree_ts_frac, a2_OHM_Tree_ts_hr, a3_OHM_Tree_ts_W_p_m2 initalized for ts, timestep
	double a1_OHM_Urban_ts_frac, a2_OHM_Urban_ts_hr, a3_OHM_Urban_ts_W_p_m2;
	double a1_OHM_Tree_ts_frac, a2_OHM_Tree_ts_hr, a3_OHM_Tree_ts_W_p_m2;
	double a1_OHM_SVeg_ts_frac, a2_OHM_SVeg_ts_hr, a3_OHM_SVeg_ts_W_p_m2;
	double a1_OHM_Soil_ts_frac, a2_OHM_Soil_ts_hr, a3_OHM_Soil_ts_W_p_m2;

	//a1_OHM_Urban_ts_frac is a1_OHM_Urban_frac, adjusted for time or season as needed
	a1_OHM_Urban_ts_frac = a1_OHM_Urban_frac;
	a2_OHM_Urban_ts_hr = a2_OHM_Urban_hr;
	a3_OHM_Urban_ts_W_p_m2 = a3_OHM_Urban_W_p_m2;

	//If Saturation_SoilStorage_frac > 0 then take 0.5 power to update OHM coefficients a1, a2, a3
	//Note: Adjust a1, a2, and a3 coefficients in Objective Hysteresis Model so ground heat flux depends on soil moisture
	//Note: Without adjustment in Naples, Italy along hippodrome on 2017071512 the soil moisture approached wilting point and ...
	//Note: ... the ground heat flux was too low when a1, a2, a3 for soil, sveg and tree did not vary with soil moisture, ...
	//Note: ... During drought ground heat flux for soil, sveg and tree was much lower than ground heat flux for impervious cover, ...
	//Note: ... and this led to air temperature for soil, sveg and tree cover being higher than air temperature for impervious cover
	//Note: Adjustment of coefficients a1, a2, a3 for soil, sveg and tree with lower soil moisture approaches that of impervious 
	//Note: The rate of change in coefficients a1, a2, a3 is taken as Saturation_SoilStorage_frac ^ 0.5, a concave down curve, and ...
	//Note: ... small changes in a1, a2, a3 occur when the soil is > 50% wet, large changes as soil moisture approaches 0% wet
	//Note: Adjustment to a1, a2, and a3 for soil, sveg, and tree cover are identical, yet these could vary in a future refactor 
	if (Saturation_SoilStorage_frac > 0) {
		//a1_OHM_Tree_ts_frac = Saturation_SoilStorage_frac^0.5 * a1_OHM_Tree_frac + (1-Saturation_SoilStorage_frac^0.5) * a1_OHM_Urban_ts_frac
		a1_OHM_Tree_ts_frac = pow(Saturation_SoilStorage_frac, 0.5) * a1_OHM_Tree_frac + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a1_OHM_Urban_ts_frac;
		//a2_OHM_Tree_ts_hr = Saturation_SoilStorage_frac^0.5 * a2_OHM_Tree_ts_hr + (1-Saturation_SoilStorage_frac^0.5) * a2_OHM_Urban_ts_hr
		a2_OHM_Tree_ts_hr = pow(Saturation_SoilStorage_frac, 0.5) * a2_OHM_Tree_hr + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a2_OHM_Urban_ts_hr;
		//a3_OHM_Tree_ts_W_p_m2 = Saturation_SoilStorage_frac^0.5 * a3_OHM_Tree_W_p_m2 + (1-Saturation_SoilStorage_frac^0.5) * a3_OHM_Urban_ts_W_p_m2
		a3_OHM_Tree_ts_W_p_m2 = pow(Saturation_SoilStorage_frac, 0.5) * a3_OHM_Tree_W_p_m2 + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a3_OHM_Urban_ts_W_p_m2;

		//a1_OHM_SVeg_ts_frac = Saturation_SoilStorage_frac^0.5 * a1_OHM_SVeg_frac + (1-Saturation_SoilStorage_frac^0.5) * a1_OHM_Urban_ts_frac
		a1_OHM_SVeg_ts_frac = pow(Saturation_SoilStorage_frac, 0.5) * a1_OHM_SVeg_frac + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a1_OHM_Urban_ts_frac;
		//a2_OHM_SVeg_ts_hr = Saturation_SoilStorage_frac^0.5 * a2_OHM_SVeg_ts_hr + (1-Saturation_SoilStorage_frac^0.5) * a2_OHM_Urban_ts_hr
		a2_OHM_SVeg_ts_hr = pow(Saturation_SoilStorage_frac, 0.5) * a2_OHM_SVeg_hr + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a2_OHM_Urban_ts_hr;
		//a3_OHM_SVeg_ts_W_p_m2 = Saturation_SoilStorage_frac^0.5 * a3_OHM_SVeg_W_p_m2 + (1-Saturation_SoilStorage_frac^0.5) * a3_OHM_Urban_ts_W_p_m2
		a3_OHM_SVeg_ts_W_p_m2 = pow(Saturation_SoilStorage_frac, 0.5) * a3_OHM_SVeg_W_p_m2 + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a3_OHM_Urban_ts_W_p_m2;

		//a1_OHM_Soil_ts_frac = Saturation_SoilStorage_frac^0.5 * a1_OHM_Soil_frac + (1-Saturation_SoilStorage_frac^0.5) * a1_OHM_Urban_ts_frac
		a1_OHM_Soil_ts_frac = pow(Saturation_SoilStorage_frac, 0.5) * a1_OHM_Soil_frac + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a1_OHM_Urban_ts_frac;
		//a2_OHM_Soil_ts_hr = Saturation_SoilStorage_frac^0.5 * a2_OHM_Soil_ts_hr + (1-Saturation_SoilStorage_frac^0.5) * a2_OHM_Urban_ts_hr
		a2_OHM_Soil_ts_hr = pow(Saturation_SoilStorage_frac, 0.5) * a2_OHM_Soil_hr + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a2_OHM_Urban_ts_hr;
		//a3_OHM_Soil_ts_W_p_m2 = Saturation_SoilStorage_frac^0.5 * a3_OHM_Soil_W_p_m2 + (1-Saturation_SoilStorage_frac^0.5) * a3_OHM_Urban_ts_W_p_m2
		a3_OHM_Soil_ts_W_p_m2 = pow(Saturation_SoilStorage_frac, 0.5) * a3_OHM_Soil_W_p_m2 + (1 - pow(Saturation_SoilStorage_frac, 0.5)) * a3_OHM_Urban_ts_W_p_m2;
	}
	//Else Saturation_SoilStorage_frac = 0 and do not take 0.5 power, but use Saturation_SoilStorage_frac to update OHM coefficients a1, a2, a3
	//Note: This makes the tree, sveg, and soil OHM coefficients a1, a2, and a3 equate to urban OHM coefficients a1, a2, and a3 
	else {
		//a1_OHM_Tree_ts_frac = Saturation_SoilStorage_frac * a1_OHM_Tree_frac + (1-Saturation_SoilStorage_frac) * a1_OHM_Urban_ts_frac
		a1_OHM_Tree_ts_frac = Saturation_SoilStorage_frac * a1_OHM_Tree_frac + (1 - Saturation_SoilStorage_frac) * a1_OHM_Urban_ts_frac;
		//a2_OHM_Tree_ts_hr = Saturation_SoilStorage_frac * a2_OHM_Tree_ts_hr + (1-Saturation_SoilStorage_frac) * a2_OHM_Urban_ts_hr
		a2_OHM_Tree_ts_hr = Saturation_SoilStorage_frac * a2_OHM_Tree_hr + (1 - Saturation_SoilStorage_frac) * a2_OHM_Urban_ts_hr;
		//a3_OHM_Tree_ts_W_p_m2 = Saturation_SoilStorage_frac * a3_OHM_Tree_W_p_m2 + (1-Saturation_SoilStorage_frac) * a3_OHM_Urban_ts_W_p_m2
		a3_OHM_Tree_ts_W_p_m2 = Saturation_SoilStorage_frac * a3_OHM_Tree_W_p_m2 + (1 - Saturation_SoilStorage_frac) * a3_OHM_Urban_ts_W_p_m2;

		//a1_OHM_SVeg_ts_frac = Saturation_SoilStorage_frac * a1_OHM_SVeg_frac + (1-Saturation_SoilStorage_frac) * a1_OHM_Urban_ts_frac
		a1_OHM_SVeg_ts_frac = Saturation_SoilStorage_frac * a1_OHM_SVeg_frac + (1 - Saturation_SoilStorage_frac) * a1_OHM_Urban_ts_frac;
		//a2_OHM_SVeg_ts_hr = Saturation_SoilStorage_frac * a2_OHM_SVeg_ts_hr + (1-Saturation_SoilStorage_frac) * a2_OHM_Urban_ts_hr
		a2_OHM_SVeg_ts_hr = Saturation_SoilStorage_frac * a2_OHM_SVeg_hr + (1 - Saturation_SoilStorage_frac) * a2_OHM_Urban_ts_hr;
		//a3_OHM_SVeg_ts_W_p_m2 = Saturation_SoilStorage_frac * a3_OHM_SVeg_W_p_m2 + (1-Saturation_SoilStorage_frac) * a3_OHM_Urban_ts_W_p_m2
		a3_OHM_SVeg_ts_W_p_m2 = Saturation_SoilStorage_frac * a3_OHM_SVeg_W_p_m2 + (1 - Saturation_SoilStorage_frac) * a3_OHM_Urban_ts_W_p_m2;

		//a1_OHM_Soil_ts_frac = Saturation_SoilStorage_frac * a1_OHM_Soil_frac + (1-Saturation_SoilStorage_frac) * a1_OHM_Urban_ts_frac
		a1_OHM_Soil_ts_frac = Saturation_SoilStorage_frac * a1_OHM_Soil_frac + (1 - Saturation_SoilStorage_frac) * a1_OHM_Urban_ts_frac;
		//a2_OHM_Soil_ts_hr = Saturation_SoilStorage_frac * a2_OHM_Soil_ts_hr + (1-Saturation_SoilStorage_frac) * a2_OHM_Urban_ts_hr
		a2_OHM_Soil_ts_hr = Saturation_SoilStorage_frac * a2_OHM_Soil_hr + (1 - Saturation_SoilStorage_frac) * a2_OHM_Urban_ts_hr;
		//a3_OHM_Soil_ts_W_p_m2 = Saturation_SoilStorage_frac * a3_OHM_Soil_W_p_m2 + (1-Saturation_SoilStorage_frac) * a3_OHM_Urban_ts_W_p_m2
		a3_OHM_Soil_ts_W_p_m2 = Saturation_SoilStorage_frac * a3_OHM_Soil_W_p_m2 + (1 - Saturation_SoilStorage_frac) * a3_OHM_Urban_ts_W_p_m2;
	}

	//Note: NOAA buoy data from stations below seem to report effects of offshore and onshore breezes, e.g., land and sea breeze, ....
	//Note: ... with air temperature above water warmer than land around sunrise and sunset, and air temperature above water cooler than land around solar noon
	//Note: NOAA buoy Station WASD2 8594900 data for Washington, DC, Potomac River, Coordinates: 38.873 N 77.021 W; 
	//Note: ... Air temp height: 2.6 m above site elevation, August 28, 2018
	//Note: NOAA buoy Station BLTM2 8574680 data for Baltimore, MD, Patapsco River, Coordinates: 39.267 N 76.579 W; 
	//Note: ... Air temp height: 6.4 m above site elevation, August 29, 2018
	//Note: NOAA buoy Station FSKM2 8574728 data for Baltimore, MD, Baltimore Harbor, Coordinates: 39.219 N 76.528 W; 
	//Note: ... Air temp height: 6.1 m above site elevation, August 29, 2018

	//Saturation_ImperviousStorage_frac (S/Smax) of Eq 19 in Yang et al. (2013)
	//Note: Reformulated to reduce overestimates of LE flux when wet, as observed for Karlsruhe, Germany July 7, 2017 at 16:00
	//Note: If Coefficient_ConvertStorageToFluxOnImpArea = 0.02 used, then scenarios where IC is changed should avoid Tair_IC_0% > Tair_IC_2% 
	//Saturation_ImperviousStorage_frac (fraction) is Coefficient_ConvertStorageToFluxOnImpArea * S/Smax of Eq 19 in Yang et al. (2013)
	double Coefficient_ConvertStorageToFluxOnImpArea = 0.02;
	Saturation_ImperviousStorage_frac = Coefficient_ConvertStorageToFluxOnImpArea * input->SafeDivide(folder->VarDict["Storage_ImperviousPondedWater_m"], Storage_Depression_Max_Impervious_m);
	//Note: Original Eq used in PASATH and i-Tree Cool Air through June 2023 is archived on the next lines
	//Saturation_ImperviousStorage_frac (fraction) is variable S/Smax of Eq 19 in Yang et al. (2013)
	//Saturation_ImperviousStorage_frac = folder->VarDict["Storage_ImperviousPondedWater_m"] / Storage_Depression_Max_Impervious_m;

	//If Storage_PerviousPondedWater_m <= Storage_Depression_Max_Pervious_m 
	if (folder->VarDict["Storage_PerviousPondedWater_m"] <= Storage_Depression_Max_Pervious_m) {
		//Saturation_PerviousStorage_frac (fraction) is variable S/Smax of Eq 19 in Yang et al. (2013)
		Saturation_PerviousStorage_frac = input->SafeDivide(folder->VarDict["Storage_PerviousPondedWater_m"], Storage_Depression_Max_Pervious_m);
	}
	//Else Storage_PerviousPondedWater_m > Storage_Depression_Max_Pervious_m then Saturation_PerviousStorage_frac = 1
	else {
		//Saturation_PerviousStorage_frac (fraction) is variable S/Smax of Eq 19 in Yang et al. (2013)
		Saturation_PerviousStorage_frac = 1;
	}

	//AerodynamicResistance_OpenArea function calculate rural canopy flux resistance
	//Note: resistanceVar is the ResistanceCal class and new instance of this class: Resistance_Aerodynamic_Tree_s_p_m, Resistance_Aerodynamic_SVeg_s_p_m, Resistance_Aerodynamic_Impervious_s_p_m, Resistance_Aerodynamic_Soil_s_p_m, Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m, Resistance_Surface_Soil_s_p_m, Resistance_Surface_Tree_s_p_m, Resistance_Surface_SVeg_s_p_m
	resistanceVar->AerodynamicResistance_OpenArea(input, folder, MapPixel_ID, WindSpeed_Reference_m_p_s, LAI_Tree_m2pm2, LAI_SVeg_m2pm2, Height_LandCover_Average_m, timeStep);

	//SurfaceResistance_LandCover function calculate flux resistance that depends on water content
	//Note: resistanceVar is the ResistanceCal class and new instance of this class: Resistance_Aerodynamic_Tree_s_p_m, Resistance_Aerodynamic_SVeg_s_p_m, Resistance_Aerodynamic_Impervious_s_p_m, Resistance_Aerodynamic_Soil_s_p_m, Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m, Resistance_Surface_Soil_s_p_m, Resistance_Surface_Tree_s_p_m, Resistance_Surface_SVeg_s_p_m
	resistanceVar->SurfaceResistance_LandCover(folder, MapPixel_ID, Saturation_SoilStorage_frac, LAI_Tree_m2pm2, LAI_SVeg_m2pm2);

	//solarCalculator->SolarCalculator_Inputs computes Cos_IncidenceAngle_Solar, ZenithAngle_Solar_rad, AzimuthAngle_Solar_N_0_rad 
	solarCalculator->SolarCalculator_Inputs(input, folder, WeatherPro, StationID_string, MapPixel_ID, timeStep, Flag_simulateReferenceStation);
	//Define Cos_IncidenceAngle_Solar, ZenithAngle_Solar_rad, AzimuthAngle_Solar_N_0_rad using folder-> terms from SolarCalculator_Inputs
	//Note: Terms are specific to MapPixel_ID when Flag_LatitudeLongitude_Computed is true
	Cos_IncidenceAngle_Solar = folder->VarDict["Cos_IncidenceAngle_Solar"];
	ZenithAngle_Solar_rad = folder->VarDict["ZenithAngle_Solar_rad"];
	AzimuthAngle_Solar_N_0_rad = folder->VarDict["AzimuthAngle_Solar_N_0_rad"];
	//Cos_ZenithAngle_Solar is cos(ZenithAngle_Solar_rad) for use with direct radiation albedo
	Cos_ZenithAngle_Solar = cos(ZenithAngle_Solar_rad);

	//Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 is solarCalculator->Ratio_AlbedoDirectCZA_to_AlbedoDirectCZA60(Cos_ZenithAngle_Solar)
	//Note: Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 is the left hand side of Eq 15 of Yang et al. (2008)
	Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 = solarCalculator->Ratio_AlbedoDirectCZA_to_AlbedoDirectCZA60(Cos_ZenithAngle_Solar);


	//keynamepart1 and related strings used with StationInfoMap variables across Multiple Weather Stations 
	string keynamepart1{};
	string keyname_Radiation_ShortWave_Direct_W_p_m2{};
	string keyname_Radiation_Shortwave_Diffuse_W_p_m2{};
	string keyname_ImpNR_priorTS_W_p_m2{};
	string keyname_TreeNR_priorTS_W_p_m2{};
	string keyname_SVegNR_priorTS_W_p_m2{};
	string keyname_SoilNR_priorTS_W_p_m2{};
	string keyname_WaterNR_priorTS_W_p_m2{};
	string keyname_RoofNR_priorTS_W_p_m2{};

	//NR_vec_size is the net radiation vector size to store prior NR values used in computing an average
	int NR_vec_size = 1;
	//If Flag_MultipleStations is true and Flag_simulateReferenceStation is true then read in WeatherPro-> reference station variables for flux calculations
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1 && Flag_simulateReferenceStation == 1) {
		keynamepart1 = StationID_string;//station ID
		keyname_Radiation_ShortWave_Direct_W_p_m2 = keynamepart1 + "_" + "Radiation_Shortwave_Direct_Wpm2";
		keyname_Radiation_Shortwave_Diffuse_W_p_m2 = keynamepart1 + "_" + "Radiation_Shortwave_Diffuse_Wpm2";
		//NR_priorTS_W_p_m2_Map updated by keyname_ImpNR_priorTS_W_p_m2 value for ImpNR_W_p_m2, Tree, SVeg, Soil, Water, Roof
		//Note: RoofNR_W_p_m2 not explicitly used in current version of HydroPlus, and is incorporated into ImpNR_W_p_m2
		keyname_ImpNR_priorTS_W_p_m2 = keynamepart1 + "_" + "ImpNR_priorTS_W_p_m2";
		keyname_TreeNR_priorTS_W_p_m2 = keynamepart1 + "_" + "TreeNR_priorTS_W_p_m2";
		keyname_SVegNR_priorTS_W_p_m2 = keynamepart1 + "_" + "SVegNR_priorTS_W_p_m2";
		keyname_SoilNR_priorTS_W_p_m2 = keynamepart1 + "_" + "SoilNR_priorTS_W_p_m2";
		keyname_WaterNR_priorTS_W_p_m2 = keynamepart1 + "_" + "WaterNR_priorTS_W_p_m2";
		keyname_RoofNR_priorTS_W_p_m2 = keynamepart1 + "_" + "RoofNR_priorTS_W_p_m2";

		//Radiation_Shortwave_W_p_m2 is WeatherPro->RadiationMap[keyname_Radiation_ShortWave_Direct_W_p_m2][timeStep] * Cos_IncidenceAngle_Solar + WeatherPro->RadiationMap[keyname_Radiation_Shortwave_Diffuse_W_p_m2][timeStep]
		//Note: From Eq 25 in Yang et al. (2013) and Kumar et al. (1997)
		Radiation_Shortwave_W_p_m2 = WeatherPro->RadiationMap[keyname_Radiation_ShortWave_Direct_W_p_m2][timeStep] * Cos_IncidenceAngle_Solar + WeatherPro->RadiationMap[keyname_Radiation_Shortwave_Diffuse_W_p_m2][timeStep];
		//Radiation_Shortwave_Direct_W_p_m2 is WeatherPro->RadiationMap[keyname_Radiation_ShortWave_Direct_W_p_m2][timeStep] * Cos_IncidenceAngle_Solar
		Radiation_Shortwave_Direct_W_p_m2 = WeatherPro->RadiationMap[keyname_Radiation_ShortWave_Direct_W_p_m2][timeStep] * Cos_IncidenceAngle_Solar;
		//Radiation_Shortwave_Diffuse_W_p_m2 is WeatherPro->RadiationMap[keyname_Radiation_Shortwave_Diffuse_W_p_m2][timeStep]
		Radiation_Shortwave_Diffuse_W_p_m2 = WeatherPro->RadiationMap[keyname_Radiation_Shortwave_Diffuse_W_p_m2][timeStep];

		//ImpNR_priorTS_W_p_m2 is NetRadiation_Prior_Calc(timeStep, NR_vec_size, ImpNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_ImpNR_priorTS_W_p_m2])
		//Note: NetRadiation_Prior_Calc function will return prior NR value based on average of vector length 
		//Note: RoofNR_W_p_m2 not explicitly used in current version of HydroPlus, and is incorporated into ImpNR_W_p_m2
		ImpNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, ImpNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_ImpNR_priorTS_W_p_m2]);
		TreeNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, TreeNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_TreeNR_priorTS_W_p_m2]);
		SVegNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, SVegNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_SVegNR_priorTS_W_p_m2]);
		SoilNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, SoilNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_SoilNR_priorTS_W_p_m2]);
		WaterNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, WaterNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_WaterNR_priorTS_W_p_m2]);
		RoofNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, RoofNR_priorTS_W_p_m2_vec, NR_priorTS_W_p_m2_Map[keyname_RoofNR_priorTS_W_p_m2]);
	}
	//Else If Flag_MultipleStations and Flag_simulateReferenceStation are false then read in in input-> reference station meteorological variables for flux calculations
	else {
		//Radiation_Shortwave_W_p_m2 (W/m2) sum of Radiation_Shortwave_Direct_Wpm2 * Cos_IncidenceAngle_Solar and Radiation_Shortwave_Diffuse_Wpm2
		//Note: From Eq 25 in Yang et al. (2013) and Kumar et al. (1997)
		Radiation_Shortwave_W_p_m2 = input->Radiation_Shortwave_Direct_Wpm2[timeStep] * Cos_IncidenceAngle_Solar + input->Radiation_Shortwave_Diffuse_Wpm2[timeStep];
		Radiation_Shortwave_Direct_W_p_m2 = input->Radiation_Shortwave_Direct_Wpm2[timeStep] * Cos_IncidenceAngle_Solar;
		Radiation_Shortwave_Diffuse_W_p_m2 = input->Radiation_Shortwave_Diffuse_Wpm2[timeStep];

		//ImpNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, ImpNR_priorTS_W_p_m2_vec, folder->VarDict["ImpNR_W_p_m2"]);
		//Note: NetRadiation_Prior_Calc function will return prior NR value based on average of vector length 
		ImpNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, ImpNR_priorTS_W_p_m2_vec, folder->VarDict["ImpNR_W_p_m2"]);
		TreeNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, TreeNR_priorTS_W_p_m2_vec, folder->VarDict["TreeNR_W_p_m2"]);
		SVegNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, SVegNR_priorTS_W_p_m2_vec, folder->VarDict["SVegNR_W_p_m2"]);
		SoilNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, SoilNR_priorTS_W_p_m2_vec, folder->VarDict["SoilNR_W_p_m2"]);
		WaterNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, WaterNR_priorTS_W_p_m2_vec, folder->VarDict["WaterNR_W_p_m2"]);
		RoofNR_priorTS_W_p_m2 = NetRadiation_Prior_Calc(timeStep, NR_vec_size, RoofNR_priorTS_W_p_m2_vec, folder->VarDict["RoofNR_W_p_m2"]);
	}

	//Control for negative values
	if (Radiation_Shortwave_W_p_m2 < 0) { Radiation_Shortwave_W_p_m2 = 0; }
	if (Radiation_Shortwave_Direct_W_p_m2 < 0) { Radiation_Shortwave_Direct_W_p_m2 = 0; }
	if (Radiation_Shortwave_Diffuse_W_p_m2 < 0) { Radiation_Shortwave_Diffuse_W_p_m2 = 0; }

	//If NLCD Class > 20 or < 25 then urban land cover classes with parameters for urban canyon geometry
	//Note: Urban class 21 condition when Flag_simulateReferenceStation > 0 addressed below
	if ((LandCover_NLCD_Class > 20) && (LandCover_NLCD_Class < 25)) {
		//Count_CanyonWalls defined as two for urban canyon
		Count_CanyonWalls = 2;
		//AspectRatio_Canyon_frac is urban height to width ratio of urban canyon, affecting energy balance
		AspectRatio_Canyon_frac = Height_UrbanBuilding_m / Width_UrbanCanyon_m;

		//If AspectRatio_Canyon_frac > 0 then divide by term
		if (AspectRatio_Canyon_frac > 0) {
			//SkyViewFactor_Road_frac or SkyViewFactor_Garden_frac from Eq 9 of Masson (2000), Eq 30b Lee and Park (2008)
			//Note: AspectRatio_Canyon_frac and SkyViewFactor_Road_frac pairs: 0.1->0.90, 0.5->0.62, 0.8->0.48, 1.2->0.36
			SkyViewFactor_Road_frac = sqrt(AspectRatio_Canyon_frac * AspectRatio_Canyon_frac + 1) - AspectRatio_Canyon_frac;
			SkyViewFactor_Garden_frac = SkyViewFactor_Road_frac;
			//SkyViewFactor_Wall_frac from Eq 10 of Masson (2000), Eq 30a Lee and Park (2008)
			//Note: AspectRatio_Canyon_frac and SkyViewFactor_Wall_frac pairs: 0.1->0.47, 0.5->0.38, 0.8->0.32, 1.2->0.27
			SkyViewFactor_Wall_frac = 0.5 * (AspectRatio_Canyon_frac + 1 - sqrt(AspectRatio_Canyon_frac * AspectRatio_Canyon_frac + 1)) / AspectRatio_Canyon_frac;
		}
		//Else If AspectRatio_Canyon_frac <= 0 then define Sky View Factors by open area
		else {
			//SkyViewFactor_Wall_frac equals SkyViewFactor_Ground_Wall_frac
			SkyViewFactor_Wall_frac = SkyViewFactor_Ground_Wall_frac;
			//SkyViewFactor_Road_frac equals SkyViewFactor_Ground_Open_frac
			SkyViewFactor_Road_frac = SkyViewFactor_Ground_Open_frac;
			//SkyViewFactor_Road_frac equals SkyViewFactor_Ground_Open_frac
			SkyViewFactor_Garden_frac = SkyViewFactor_Ground_Open_frac;
		}

		//Call HeatFluxCal::UrbanCanyon_RadiationShortwave to compute urban canyon effects on direct and diffuse shortwave radiation balance
		HeatFluxCal::UrbanCanyon_RadiationShortwave(input, folder, Flag_UrbanCanyonResistance, timeStep);
		//Call HeatFluxCal::UrbanCanyon_RadiationLongwave to compute urban canyon effects on longwave radiation balance
		HeatFluxCal::UrbanCanyon_RadiationLongwave(input, folder, Flag_UrbanCanyonResistance, timeStep);

		//If Flag_UrbanCanyonResistance equals 1 then define resistances using AerodynamicResistance_UrbanCanyon function
		//Note: This function may erroneously constrain range of air temperature across tree cover and impervious cover gradients
		//Note: Consider refactor to enter this calculation, as aspect ratio controls the resistances
		if (Flag_UrbanCanyonResistance == 1) {
			//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions without sending solarCalculator
			HeatMetrics_Calc HeatMetricsCalc(input);
			//AtmPres_kPa = input->AtmPres_kPa[timeStep]
			double AtmPres_kPa = input->AtmPres_kPa[timeStep];
			double SpecificHeat_HumidAir_JpkgK = HeatMetricsCalc.SpecificHeat_HumidAir_JpkgK_Calc(Tair_localCell_K, Tdew_localCell_K, AtmPres_kPa);

			//AerodynamicResistance_UrbanCanyon function calculate urban canopy flux resistance
			//Note: resistanceVar is the ResistanceCal class and new instance of this class: Resistance_Aerodynamic_Tree_s_p_m, Resistance_Aerodynamic_SVeg_s_p_m, Resistance_Aerodynamic_Impervious_s_p_m, Resistance_Aerodynamic_Soil_s_p_m, Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m, Resistance_Surface_Soil_s_p_m, Resistance_Surface_Tree_s_p_m, Resistance_Surface_SVeg_s_p_m
			resistanceVar->AerodynamicResistance_UrbanCanyon(input, folder, MapPixel_ID, WindSpeed_Reference_m_p_s, LAI_Tree_m2pm2, LAI_SVeg_m2pm2, AspectRatio_Canyon_frac, Height_LandCover_Average_m, SpecificHeat_HumidAir_JpkgK);
		}

		//Radiation_Shortwave_Direct_Diffuse_Pavement_and_Roof_UrbanCanyon_Absorbed_W_p_m2 is Building_Area_frac * Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2 + (1 - Building_Area_frac) * Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2
		//Note: It is absorbed radiation, already processed for a time variable albedo 
		//Note: Li et al. (2014) showed roofs contribute to surface and near-surface urban heat island effects alongside pavement
		double Radiation_Shortwave_Direct_Diffuse_Pavement_and_Roof_UrbanCanyon_Absorbed_W_p_m2 = Building_Area_frac * Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2 + (1 - Building_Area_frac) * Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2;

		//ImpNR_W_p_m2 = (Radiation_Shortwave_Direct_Diffuse_Pavement_and_Roof_UrbanCanyon_Absorbed_W_p_m2 + Count_CanyonWalls * Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2) + Radiation_Longwave_Net_Pavement_W_p_m2 + Radiation_Longwave_Net_Wall_W_p_m2;
		//Note: Modified form of Eq 24 from Yang et al (2013) to use urban canyon and emissivity adjusted radiation
		//Note: Radiation_Shortwave_Direct_Diffuse_Pavement_and_Roof_UrbanCanyon_Absorbed_W_p_m2 is absorbed, adjusted for albedo ...
		//Note: ... via Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation net added indicate downward flux when positive
		//Note: Radiation_Longwave_Net_Roof_W_p_m2 not in TEB of Masson (2000); does not radiate into urban canopy air volumes 
		ImpNR_W_p_m2 = (Radiation_Shortwave_Direct_Diffuse_Pavement_and_Roof_UrbanCanyon_Absorbed_W_p_m2 + Count_CanyonWalls * Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2) + Radiation_Longwave_Net_Pavement_W_p_m2 + Radiation_Longwave_Net_Wall_W_p_m2;

		//Note: Alternative for ImpNR_W_p_m2 would use simple Oke et al. (2017) Table 5.4 as algorithm to adjust for day and night
		//Note: ... Uses ratio of wall and floor to roof for direct, longwave down and up from Oke et al. (2017) Fig 5.13 and 5.14 w H:W = 1 
		//Note: ... ImpNR_W_p_m2 increases with urban canyon aspect ratio and orientation, as in Fig 5.14 and Table 5.4 of Oke et al. (2017) 
		//Note: ... Daytime relation between net radiation (NR, fraction) and aspect ratio (AR=H:W): 
		//Note: ...  AR=0.5 NR=0.11, AR=2.0 NR=0.20, AR=2.0 NR=0.24, AR=4.0 NR=0.28
		//Note: ... Daytime trendline function NR_frac = 0.076 * ln(AR) + 0.182, has R^2=0.98, derived with data point of AR=0.1 NR=0.01
		//Note: ... Nighttime relation between net longwave radiation (NLW, fraction) and aspect ratio (AR=H:W): 
		//Note: ...  AR=0.5 NLW=0.25, AR=1.0 NLW=0.31, AR=2.0 NLW=0.31, AR=4.0 NLW=0.38		
		//Note: ... Nighttime trendline function NLW = 0.097 * ln(AR) + 0.27, has R^2=0.92, derived data point of AR=0.1 NLW=0.01

	}
	//Else If LandCover_NLCD_Class not urban or Flag_simulateReferenceStation > 0 then set parameters without urban canyon effect
	else if ((LandCover_NLCD_Class <= 20) || (LandCover_NLCD_Class >= 25) || (Flag_simulateReferenceStation > 0)) {
		//ImpNR_W_p_m2 = (Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Urban_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Urban_frac)) + Emissivity_Urban_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Urban_frac * Radiation_Longwave_Up_W_p_m2
		//Note: Modified form of Eq 24 from Yang et al (2013) to explicitly use emissivity adjusted longwave radiation
		//Note: Radiation_Shortwave_Diffuse_W_p_m2 constant albedo, Radiation_Shortwave_Direct_W_p_m2 adjusted albedo via ... 
		//Note: ... Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation_Longwave_Up_W_p_m2 subtracted to indicate outward flux
		ImpNR_W_p_m2 = (Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Urban_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Urban_frac)) + Emissivity_Urban_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Urban_frac * Radiation_Longwave_Up_W_p_m2;
	}

	//Imp_AH_W_p_m2 (W/m2) is impervious cover term for AnthropogenicHeat_Flux_Total_Wpm2, which is uniform in pixel
	//Note: Anthropogenic heat is not summed directly into net radiation in order to track each term separately
	Imp_AH_W_p_m2 = AnthropogenicHeat_Flux_Total_Wpm2;

	//ImpDeltaQ_W_p_m2 (W/m2) defined in Eq 29 in Yang, Endreny, Nowak (2013), 
	//Note: Modifying Radiation_Longwave_Down_W_p_m2 and Radiation_Longwave_Up_W_p_m2 by emissivity
	//Note: Parameters a1, a2, a3 originally modified from mixed forest in Table 4 of Grimmond et al. (1999)
	//Note: d(ImpNR)/d(t) is represented as (ImpNR_W_p_m2 - ImpNR_priorTS_W_p_m2) / (SimulationTimeStep_Duration_sec * Ratio_Hour_to_Second)
	ImpDeltaQ_W_p_m2 = a1_OHM_Urban_ts_frac * ImpNR_W_p_m2 + a2_OHM_Urban_ts_hr * (ImpNR_W_p_m2 - ImpNR_priorTS_W_p_m2) / (input->SimulationTimeStep_Duration_sec[timeStep] * Ratio_Hour_to_Second) + a3_OHM_Urban_ts_W_p_m2;

	//ImpC = pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m
	//Note: ImpC (frac / s/m) optionally defined to simplify combination of Eq 9 and 19 in Yang et al. (2013)
	ImpC = pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m;

	//If (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) (fraction) < 0.0001 then set terms to zero
	if ((ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) < Constant_1E_negative4) {
		//All energy balance variables set to zero for this land cover type
		ImpNR_W_p_m2 = 0; ImpDeltaQ_W_p_m2 = 0; ImpC = 0;
	}

	//If TreeCover_frac (fraction) greater than 0 then proceed with tree calculations
	if (TreeCover_frac > Constant_1E_negative4) {
		//TreeNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Tree_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Tree_frac) + Emissivity_Tree_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Tree_frac * Radiation_Longwave_Up_W_p_m2
		//Note: Modified form of Eq 24 from Yang et al (2013) to explicitly use emissivity adjusted longwave radiation
		//Note: Radiation_Shortwave_Diffuse_W_p_m2 constant albedo, Radiation_Shortwave_Direct_W_p_m2 adjusted albedo via ... 
		//Note: ... Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation_Longwave_Up_W_p_m2 subtracted to indicate outward flux
		TreeNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Tree_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Tree_frac) + Emissivity_Tree_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Tree_frac * Radiation_Longwave_Up_W_p_m2;
		//If (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 || ...
		//... (Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))
		//Note: Urban canyon radiation algorithm active for NLCD Class 21 to 24 when not reference station or when reference station is urban canyon
		//Note: Airport weather reference stations are typically not considered to have urban canyon radiation effects, but ...
		//Note: ... If RefWeatherLocationUrbanCanyon is 0 then reference station energy balance will not match non-reference station balance
		if (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 ||
			(Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))) {
			//TreeNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Tree_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Tree_W_p_m2
			//Note: Modified form of Eq 24 from Yang et al (2013) to use urban canyon and emissivity adjusted radiation
			//Note: Radiation_Shortwave_Direct_Diffuse_Tree_UrbanCanyon_Absorbed_W_p_m2 is absorbed, adjusted for albedo ...
			//Note: ... via Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
			//Note: Radiation is positive downward with Radiation net added indicate downward flux when positive
			TreeNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Tree_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Tree_W_p_m2;
		}

		//Tree_AH_W_p_m2 (W/m2) is tree cover term for AnthropogenicHeat_Flux_Total_Wpm2, which is uniform in pixel
		//Note: Anthropogenic heat is not summed directly into net radiation in order to track each term separately
		Tree_AH_W_p_m2 = AnthropogenicHeat_Flux_Total_Wpm2;

		//TreeDeltaQ_W_p_m2 (W/m2) defined in Eq 29 in Yang, Endreny, Nowak (2013) 
		//Note: Parameters a1, a2, a3 originally modified from mixed forest in Table 4 of Grimmond et al. (1999)
		TreeDeltaQ_W_p_m2 = a1_OHM_Tree_ts_frac * TreeNR_W_p_m2 + a2_OHM_Tree_ts_hr * (TreeNR_W_p_m2 - TreeNR_priorTS_W_p_m2) / (input->SimulationTimeStep_Duration_sec[timeStep] * Ratio_Hour_to_Second) + a3_OHM_Tree_ts_W_p_m2;

		//TreeCE = pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m
		//Note: TreeCE (frac / s/m) optionally defined to simplify combination of Eq 9 and 22 in Yang et al. (2013)
		TreeCE = pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m;
		//TreeCT = (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m)
		//Note: TreeCT (frac / s/m) optionally defined to simplify combination of Eq 9 and 23 in Yang et al. (2013)
		TreeCT = (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m);
		//TreeC (frac/(s/m))= TreeCT + TreeCE
		//Note: SVegC combines transpiration (TreeCT) and evaporation (TreeCE) components
		TreeC = TreeCT + TreeCE;
	}
	//Else TreeCover_frac (fraction) < 0.0001 and proceed without tree calculations
	else {
		//All energy balance variables set to zero for this land cover type
		TreeNR_W_p_m2 = 0; TreeDeltaQ_W_p_m2 = 0; TreeCE = 0; TreeCT = 0; TreeC = 0;
	}

	//If ShortVegCover_noTreeCanopy_frac (fraction) greater than 0 then proceed with short vegetation calculations
	if (ShortVegCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//SVegNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_SVeg_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_SVeg_frac) + Emissivity_SVeg_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_SVeg_frac * Radiation_Longwave_Up_W_p_m2
		//Note: Modified form of Eq 24 from Yang et al (2013) to explicitly use emissivity adjusted longwave radiation
		//Note: Radiation_Shortwave_Diffuse_W_p_m2 constant albedo, Radiation_Shortwave_Direct_W_p_m2 adjusted albedo via ... 
		//Note: ... Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation_Longwave_Up_W_p_m2 subtracted to indicate outward flux
		SVegNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_SVeg_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_SVeg_frac) + Emissivity_SVeg_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_SVeg_frac * Radiation_Longwave_Up_W_p_m2;
		//If (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 || ...
		//... (Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))
		//Note: Urban canyon radiation algorithm active for NLCD Class 21 to 24 when not reference station or when reference station is urban canyon
		//Note: Airport weather reference stations are typically not considered to have urban canyon radiation effects, but ...
		//Note: ... If RefWeatherLocationUrbanCanyon is 0 then reference station energy balance will not match non-reference station balance
		if (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 ||
			(Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))) {
			//SVegNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_SVeg_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_SVeg_W_p_m2
			//Note: Modified form of Eq 24 from Yang et al (2013) to use urban canyon and emissivity adjusted radiation
			//Note: Radiation_Shortwave_Direct_Diffuse_SVeg_UrbanCanyon_Absorbed_W_p_m2 is absorbed, adjusted for albedo ...
			//Note: ... via Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
			//Note: Radiation is positive downward with Radiation net added indicate downward flux when positive
			SVegNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_SVeg_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_SVeg_W_p_m2;
		}

		//SVeg_AH_W_p_m2 (W/m2) is short vegetation cover term for AnthropogenicHeat_Flux_Total_Wpm2, which is uniform in pixel
		//Note: Anthropogenic heat is not summed directly into net radiation in order to track each term separately
		SVeg_AH_W_p_m2 = AnthropogenicHeat_Flux_Total_Wpm2;

		//SVegDeltaQ_W_p_m2 (W/m2) defined in Eq 29 in Yang et al. (2013) 
		//Note: Parameters a1, a2, a3 originally from short grass in Table 4 of Grimmond et al. (1999)
		SVegDeltaQ_W_p_m2 = a1_OHM_SVeg_ts_frac * SVegNR_W_p_m2 + a2_OHM_SVeg_ts_hr * (SVegNR_W_p_m2 - SVegNR_priorTS_W_p_m2) / (input->SimulationTimeStep_Duration_sec[timeStep] * Ratio_Hour_to_Second) + a3_OHM_SVeg_ts_W_p_m2;

		//SVegCE = pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m
		//Note: SVegCE (frac / s/m) optionally defined to simplify combination of Eq 9 and 22 in Yang et al. (2013)
		SVegCE = pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m;
		//SVegCT = (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m)
		//Note: SVegCT (frac / s/m) optionally defined to simplify combination of Eq 9 and 23 in Yang et al. (2013)
		SVegCT = (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m);
		//SVegC (frac/(s/m))= SVegCT + SVegCE
		//Note: SVegC combines transpiration (SVegCT) and evaporation (SVegCE) components
		SVegC = SVegCT + SVegCE;
	}
	//Else ShortVegCover_noTreeCanopy_frac (fraction) < 0.0001 and proceed without short vegetation calculations
	else {
		//All energy balance variables set to zero for this land cover type
		SVegNR_W_p_m2 = 0; SVegDeltaQ_W_p_m2 = 0; SVegCE = 0; SVegCT = 0; SVegC = 0;
	}

	//If SoilCover_noTreeCanopy_frac (fraction) greater than 0 then proceed with soil calculations
	if (SoilCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//SoilNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Soil_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Soil_frac) + Emissivity_Soil_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Soil_frac * Radiation_Longwave_Up_W_p_m2
		//Note: Modified form of Eq 24 from Yang et al (2013) to explicitly use emissivity adjusted longwave radiation
		//Note: Radiation_Shortwave_Diffuse_W_p_m2 constant albedo, Radiation_Shortwave_Direct_W_p_m2 adjusted albedo via ... 
		//Note: ... Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation_Longwave_Up_W_p_m2 subtracted to indicate outward flux
		SoilNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Soil_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Soil_frac) + Emissivity_Soil_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Soil_frac * Radiation_Longwave_Up_W_p_m2;
		//If (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 || ...
		//... (Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))
		//Note: Urban canyon radiation algorithm active for NLCD Class 21 to 24 when not reference station or when reference station is urban canyon
		//Note: Airport weather reference stations are typically not considered to have urban canyon radiation effects, but ...
		//Note: ... If RefWeatherLocationUrbanCanyon is 0 then reference station energy balance will not match non-reference station balance
		if (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 ||
			(Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))) {
			//SoilNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Soil_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Soil_W_p_m2
			//Note: Modified form of Eq 24 from Yang et al (2013) to use urban canyon and emissivity adjusted radiation
			//Note: Radiation_Shortwave_Direct_Diffuse_Soil_UrbanCanyon_Absorbed_W_p_m2 is absorbed, adjusted for albedo ...
			//Note: ... via Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
			//Note: Radiation is positive downward with Radiation net added indicate downward flux when positive
			SoilNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Soil_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Soil_W_p_m2;
		}

		//Soil_AH_W_p_m2 (W/m2) is soil cover term for AnthropogenicHeat_Flux_Total_Wpm2, which is uniform in pixel
		//Note: Anthropogenic heat is not summed directly into net radiation in order to track each term separately
		Soil_AH_W_p_m2 = AnthropogenicHeat_Flux_Total_Wpm2;

		//SoilDeltaQ_W_p_m2 (W/m2) defined in Eq 29 in Yang, Endreny, Nowak (2013) 
		//Note: Parameters a1, a2, a3 originally from bare soil in Table 4 of Grimmond et al. (1999)
		SoilDeltaQ_W_p_m2 = a1_OHM_Soil_ts_frac * SoilNR_W_p_m2 + a2_OHM_Soil_ts_hr * (SoilNR_W_p_m2 - SoilNR_priorTS_W_p_m2) / (input->SimulationTimeStep_Duration_sec[timeStep] * Ratio_Hour_to_Second) + a3_OHM_Soil_ts_W_p_m2;

		//SoilC = Saturation_SoilStorage_frac / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m)
		//Note: SoilC (frac / s/m) optionally defined to simplify combination of Eq 9 and 20 in Yang et al. (2013)
		SoilC = Saturation_SoilStorage_frac / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m);
	}
	//Else SoilCover_noTreeCanopy_frac (fraction) < 0.0001 and proceed without soil calculations
	else {
		//All energy balance variables set to zero for this land cover type
		SoilNR_W_p_m2 = 0; SoilDeltaQ_W_p_m2 = 0; SoilC = 0;
	}

	//If WaterCover_noTreeCanopy_frac (fraction) greater than 0 then proceed with water calculations
	if (WaterCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//WaterNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Water_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Water_frac) + Emissivity_Water_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Water_frac * Radiation_Longwave_Up_W_p_m2
		//Note: Modified form of Eq 24 from Yang et al (2013) to explicitly use emissivity adjusted longwave radiation
		//Note: Radiation_Shortwave_Diffuse_W_p_m2 constant albedo, Radiation_Shortwave_Direct_W_p_m2 adjusted albedo via ... 
		//Note: ... Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation_Longwave_Up_W_p_m2 subtracted to indicate outward flux
		WaterNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Water_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Water_frac) + Emissivity_Water_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Water_frac * Radiation_Longwave_Up_W_p_m2;
		//If (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 || ...
		//... (Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))
		//Note: Urban canyon radiation algorithm active for NLCD Class 21 to 24 when not reference station or when reference station is urban canyon
		//Note: Airport weather reference stations are typically not considered to have urban canyon radiation effects, but ...
		//Note: ... If RefWeatherLocationUrbanCanyon is 0 then reference station energy balance will not match non-reference station balance
		if (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 ||
			(Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))) {
			//WaterNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Water_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Water_W_p_m2
			//Note: Modified form of Eq 24 from Yang et al (2013) to use urban canyon and emissivity adjusted radiation
			//Note: Radiation_Shortwave_Direct_Diffuse_Water_UrbanCanyon_Absorbed_W_p_m2 is absorbed, adjusted for albedo ...
			//Note: ... via Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
			//Note: Radiation is positive downward with Radiation net added indicate downward flux when positive
			WaterNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Water_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Water_W_p_m2;
		}

		//Water_AH_W_p_m2 (W/m2) is water cover term for AnthropogenicHeat_Flux_Total_Wpm2, which is uniform in pixel
		//Note: Anthropogenic heat is not summed directly into net radiation in order to track each term separately
		Water_AH_W_p_m2 = AnthropogenicHeat_Flux_Total_Wpm2;

		//SoilDeltaQ_W_p_m2 (W/m2) defined in Eq 29 in Yang, Endreny, Nowak (2013) 
		//Note: Parameters a1, a2, a3 originally from water in Table 4 of Grimmond et al. (1999)
		WaterDeltaQ_W_p_m2 = a1_OHM_Water_frac * WaterNR_W_p_m2 + a2_OHM_Water_hr * (WaterNR_W_p_m2 - WaterNR_priorTS_W_p_m2) / (input->SimulationTimeStep_Duration_sec[timeStep] * Ratio_Hour_to_Second) + a3_OHM_Water_W_p_m2;

		//WaterC = Saturation_WaterStorage_frac / resistanceVar->Resistance_Aerodynamic_Soil_s_p_m
		//Note: WaterC (frac / s/m) optionally defined to simplify combination of Eq 9 and 20 in Yang et al. (2013),
		//Note: ... where Eq 20 modified from soil to water by dropping ground resistance
		//Note: WaterC is resistance term within Eq 14 in Yang et al. (2013) to estimate free ground water surface evaporation
		WaterC = Saturation_WaterStorage_frac / resistanceVar->Resistance_Aerodynamic_Soil_s_p_m;
	}
	//Else WaterCover_noTreeCanopy_frac (fraction) < 0.0001 and proceed without water calculations
	else {
		//All energy balance variables set to zero for this land cover type
		WaterNR_W_p_m2 = 0; WaterDeltaQ_W_p_m2 = 0; WaterC = 0;
	}

	//If Building_Area_frac is > 0 then define roof parameters
	//Note: RoofNR_W_p_m2 not explicitly used in current version of HydroPlus, and is incorporated into ImpNR_W_p_m2
	if (Building_Area_frac > 0) {
		//RoofNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Roof_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Roof_frac) + Emissivity_Roof_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Roof_frac * Radiation_Longwave_Up_W_p_m2
		//Note: Modified form of Eq 24 from Yang et al (2013) to explicitly use emissivity adjusted longwave radiation
		//Note: Radiation_Shortwave_Diffuse_W_p_m2 constant albedo, Radiation_Shortwave_Direct_W_p_m2 adjusted albedo via ... 
		//Note: ... Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
		//Note: Radiation is positive downward with Radiation_Longwave_Up_W_p_m2 subtracted to indicate outward flux
		RoofNR_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Roof_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) + Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Roof_frac) + Emissivity_Roof_frac * Radiation_Longwave_Down_W_p_m2 - Emissivity_Roof_frac * Radiation_Longwave_Up_W_p_m2;
		//If (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 || ...
		//... (Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))
		//Note: Urban canyon radiation algorithm active for NLCD Class 21 to 24 when not reference station or when reference station is urban canyon
		//Note: Airport weather reference stations are typically not considered to have urban canyon radiation effects, but ...
		//Note: ... If RefWeatherLocationUrbanCanyon is 0 then reference station energy balance will not match non-reference station balance
		if (LandCover_NLCD_Class > 20 && LandCover_NLCD_Class < 25 && (Flag_simulateReferenceStation < 1 ||
			(Flag_simulateReferenceStation == 1 && input->TemperatureExecutionParams["RefWeatherLocationUrbanCanyon"] == "1"))) {
			//RoofNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Pavement_W_p_m2
			//Note: Modified form of Eq 24 from Yang et al (2013) to use urban canyon and emissivity adjusted radiation
			//Note: Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2  is absorbed, adjusted for albedo ...
			//Note: ... via Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 time varying modifier of albedo from Eq 15 of Yang et al. (2008) 
			//Note: Radiation is positive downward with Radiation net added indicate downward flux when positive
			//Note: Radiation_Longwave_Net_Pavement_W_p_m2 is not explicitly for roof; TEB of Masson (2000) has no LW roof term
			RoofNR_W_p_m2 = Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2 + Radiation_Longwave_Net_Pavement_W_p_m2;
		}
		//RoofDeltaQ_W_p_m2 could be set equal to some fraction of SVegDeltaQ_W_p_m2 and ImpDeltaQ_W_p_m2
		RoofDeltaQ_W_p_m2 = ImpDeltaQ_W_p_m2;
		//RoofC could be set equal to ImpC or SVegC
		RoofC = ImpC;
	}
	//Else NLCD Class for roof not present and proceed without roof calculationsNote: 
	else {
		//All energy balance variables set to zero for this land cover type
		RoofNR_W_p_m2 = 0; RoofDeltaQ_W_p_m2 = 0; RoofC = 0;
	}
	//If Flag_MultipleStations and Flag_simulateReferenceStation true then read in reference station meteorological variables
	//Note: Required step to save as prior time step value for DeltQ inputs for each weather station specific and NR land cover
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1 && Flag_simulateReferenceStation == 1) {
		//NR_priorTS_W_p_m2_Map updated by keyname_ImpNR_priorTS_W_p_m2 value for ImpNR_W_p_m2, Tree, SVeg, Soil, Water, Roof
		//Note: RoofNR_W_p_m2 not explicitly used in current version of HydroPlus, and is incorporated into ImpNR_W_p_m2
		NR_priorTS_W_p_m2_Map[keyname_ImpNR_priorTS_W_p_m2] = ImpNR_W_p_m2;
		NR_priorTS_W_p_m2_Map[keyname_TreeNR_priorTS_W_p_m2] = TreeNR_W_p_m2;
		NR_priorTS_W_p_m2_Map[keyname_SVegNR_priorTS_W_p_m2] = SVegNR_W_p_m2;
		NR_priorTS_W_p_m2_Map[keyname_SoilNR_priorTS_W_p_m2] = SoilNR_W_p_m2;
		NR_priorTS_W_p_m2_Map[keyname_WaterNR_priorTS_W_p_m2] = WaterNR_W_p_m2;
		NR_priorTS_W_p_m2_Map[keyname_RoofNR_priorTS_W_p_m2] = RoofNR_W_p_m2;
	}
}

//UrbanCanyon_RadiationLongwave function calculates urban canyon longwave energy balance once for each folder at each timeStep
void HeatFluxCal::UrbanCanyon_RadiationLongwave(Inputs* input, DataFolder* folder, bool Flag_UrbanCanyonResistance, int timeStep)
{

	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	//Note: RunTemperatureSpatialModel 1st sends MapPixel_ReferenceStation_IDs, regardless of indexing, then sends MapPixel_ID of 0
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//Coefficient_w_Eq09_Offerle_2003 from below Eq 9 of Offerle et al. (2003), where VaporPressure_Actual_kPa (kPa) gives w values ranging from 0 to 0.62
	double Coefficient_w_Eq09_Offerle_2003 = 46.5 * VaporPressure_Actual_kPa / Tair_localCell_K;

	//Emissivity_ClearSky_frac (fraction) from Eq 9 of Offerle et al. (2003) 
	double Emissivity_ClearSky_frac = 1 - (1 + Coefficient_w_Eq09_Offerle_2003) * exp(-1 * pow(1.2 + 3.0 * Coefficient_w_Eq09_Offerle_2003, 0.5));
	//CloudCover_frac (fraction) computed to replace Eq 11 of Offerle et al. (2003) using difference in Tdew_localCell_K and Tair_localCell_K
	double CloudCover_frac = 0;
	//If (Tair_localCell_K - Tdew_localCell_K) <= 1) then moisture content is relatively high and clouds are likely
	if ((Tair_localCell_K - Tdew_localCell_K) <= 1) {
		CloudCover_frac = (Tair_localCell_K - Tdew_localCell_K);
	}
	//Emissivity_Sky_frac (frac) from Eq 12 of Offerle et al. (2003) using new form of CloudCover_frac based on dew point
	//Caution: Emissivity_Sky_frac calculation is not proven to work, and seems to generate sky temperatures that are too large
	//Note: Temperature_Atmosphere_K_Power04 computed from mesoScale temperature, not from Emissivity_Sky_frac
	Emissivity_Sky_frac = Emissivity_ClearSky_frac + (1.0 - Emissivity_ClearSky_frac) * pow(CloudCover_frac, 2);

	//Prevent zero value emissivity to avoid division by zero
	if (Emissivity_Sky_frac <= 0) { Emissivity_Sky_frac = 0.01; }
	if (Emissivity_Wall_frac <= 0) { Emissivity_Wall_frac = 0.01; }
	if (Emissivity_Road_frac <= 0) { Emissivity_Road_frac = 0.01; }
	if (Emissivity_Tree_frac <= 0) { Emissivity_Tree_frac = 0.01; }
	if (Emissivity_SVeg_frac <= 0) { Emissivity_SVeg_frac = 0.01; }
	if (Emissivity_Soil_frac <= 0) { Emissivity_Soil_frac = 0.01; }
	if (Emissivity_SVeg_frac <= 0) { Emissivity_SVeg_frac = 0.01; }

	//Temperature_Sky_K estimated as Tair_localCell_K
	double Temperature_Sky_K = Tair_localCell_K;
	//get the Tair_mesoScale_weighted_K for multiple station case
	double Tair_mesoScale_K_temp;
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1) {
		Tair_mesoScale_K_temp = WeatherProcessor::Tair_mesoScale_weighted_K;
	}else{//get the Tair_mesoScale_K for single station case
		Tair_mesoScale_K_temp = folder->VarDict["Tair_mesoScale_K"];
	}
	//If Tair_mesoScale_K defined then use to update estimate of Tair_mesoScale_K_temp
	if (Tair_mesoScale_K_temp > 0 && !isnan(Tair_mesoScale_K_temp)) {
		Temperature_Sky_K = Tair_mesoScale_K_temp;
	};

	//Temperature_Atmosphere_K_Power04 (K^4) estimated based on Tair_localCell_K (K), and used in a function taking temperature to the 4th power
	double Temperature_Atmosphere_K_Power04 = pow(Temperature_Sky_K, 4);
	//Alternatively, Temperature_Atmosphere_K_Power04 (K^4) rearranged from Eq 26 of Yang et al. (2013), and from Eq 8 of Offerle et al. (2003)
	//Note: This alternative equation not used due to it overestimating sky temperature by 20 C; need better value for Emissivity_Sky_frac
	//double Temperature_Atmosphere_K_Power04 = Radiation_Longwave_Down_W_p_m2 / (Emissivity_Sky_frac * Sigma_Stefan_Boltzmann);

	//Temperature_UrbanCanyonWall_K_Power04 (K^4) from Eq 28 of Yang et al. (2013), based on from Eq 14 to 16 of Offerle et al. (2003)
	//Note: Offerle et al. (2003) explains the assumption of Tsurface = Tair_K at night holds well for urban area
	//Note: Coefficient_c_Eq28_Yang_etal_2013 = c = 0.08 parameter in Eq 28 of Yang et al. (2013), but also given w/ Eq 15 of Offerle et al. (2003), and attributed to Holtslag and van Ulden (1983) https://doi.org/10.1175/1520-0450(1983)022<0517:ASSFDE>2.0.CO;2
	double Temperature_UrbanCanyonWall_K_Power04 = Temperature_Atmosphere_K_Power04 + (Coefficient_c_Eq28_Yang_etal_2013 * Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2 * (1 - Albedo_Wall_frac)) / (Emissivity_Wall_frac * Sigma_Stefan_Boltzmann);
	//Temperature_UrbanCanyonWall_K (K) is Temperature_UrbanCanyonWall_K_Power04 ^ 0.25
	Temperature_UrbanCanyonWall_K = pow(Temperature_UrbanCanyonWall_K_Power04, 0.25);

	//Temperature_UrbanCanyonRoad_K_Power04 (K^4) from Eq 28 of Yang et al. (2013), based on from Eq 14 to 16 of Offerle et al. (2003)
	//Note: Offerle et al. (2003) explains the assumption of Tsurface = Tair_K at night holds well for urban area
	//Note: Coefficient_c_Eq28_Yang_etal_2013 = c = 0.08 parameter in Eq 28 of Yang et al. (2013), but also given w/ Eq 15 of Offerle et al. (2003), and attributed to Holtslag and van Ulden (1983) https://doi.org/10.1175/1520-0450(1983)022<0517:ASSFDE>2.0.CO;2
	double Temperature_UrbanCanyonRoad_K_Power04 = Temperature_Atmosphere_K_Power04 + (Coefficient_c_Eq28_Yang_etal_2013 * Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2 * (1 - Albedo_Road_frac)) / (Emissivity_Road_frac * Sigma_Stefan_Boltzmann);
	//Temperature_UrbanCanyonWall_K (K) is Temperature_UrbanCanyonRoad_K_Power04 ^ 0.25
	Temperature_UrbanCanyonRoad_K = pow(Temperature_UrbanCanyonRoad_K_Power04, 0.25);

	//Temperature_UrbanCanyonTree_K_Power04 (K^4) from Eq 28 of Yang et al. (2013), based on from Eq 14 to 16 of Offerle et al. (2003)
	//Note: Offerle et al. (2003) explains the assumption of Tsurface = Tair_K at night holds well for urban area
	//Note: Coefficient_c_Eq28_Yang_etal_2013 = c = 0.08 parameter in Eq 28 of Yang et al. (2013), but also given w/ Eq 15 of Offerle et al. (2003), and attributed to Holtslag and van Ulden (1983) https://doi.org/10.1175/1520-0450(1983)022<0517:ASSFDE>2.0.CO;2
	double Temperature_UrbanCanyonTree_K_Power04 = Temperature_Atmosphere_K_Power04 + (Coefficient_c_Eq28_Yang_etal_2013 * Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2 * (1 - Albedo_Tree_frac)) / (Emissivity_Tree_frac * Sigma_Stefan_Boltzmann);
	//Temperature_UrbanCanyonTree_K (K) is Temperature_UrbanCanyonGarden_K_Power04 ^ 0.25
	Temperature_UrbanCanyonTree_K = pow(Temperature_UrbanCanyonTree_K_Power04, 0.25);

	//Temperature_UrbanCanyonSVeg_K_Power04 (K^4) from Eq 28 of Yang et al. (2013), based on from Eq 14 to 16 of Offerle et al. (2003)
	//Note: Offerle et al. (2003) explains the assumption of Tsurface = Tair_K at night holds well for urban area
	//Note: Coefficient_c_Eq28_Yang_etal_2013 = c = 0.08 parameter in Eq 28 of Yang et al. (2013), but also given w/ Eq 15 of Offerle et al. (2003), and attributed to Holtslag and van Ulden (1983) https://doi.org/10.1175/1520-0450(1983)022<0517:ASSFDE>2.0.CO;2
	double Temperature_UrbanCanyonSVeg_K_Power04 = Temperature_Atmosphere_K_Power04 + (Coefficient_c_Eq28_Yang_etal_2013 * Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2 * (1 - Albedo_SVeg_frac)) / (Emissivity_SVeg_frac * Sigma_Stefan_Boltzmann);
	//Temperature_UrbanCanyonSVeg_K (K) is Temperature_UrbanCanyonSVeg_K_Power04 ^ 0.25
	Temperature_UrbanCanyonSVeg_K = pow(Temperature_UrbanCanyonSVeg_K_Power04, 0.25);

	//Temperature_UrbanCanyon_Garden_K (K) is weighted average of Temperature_UrbanCanyonTree_K + Temperature_UrbanCanyonSVeg_K
	double Temperature_UrbanCanyon_Garden_K = TreeCover_frac * Temperature_UrbanCanyonTree_K + ShortVegCover_noTreeCanopy_frac * Temperature_UrbanCanyonSVeg_K;

	//Temperature_UrbanCanyonSoil_K_Power04 (K^4) from Eq 28 of Yang et al. (2013), based on from Eq 14 to 16 of Offerle et al. (2003)
	//Note: Offerle et al. (2003) explains the assumption of Tsurface = Tair_K at night holds well for urban area
	//Note: Coefficient_c_Eq28_Yang_etal_2013 = c = 0.08 parameter in Eq 28 of Yang et al. (2013), but also given w/ Eq 15 of Offerle et al. (2003), and attributed to Holtslag and van Ulden (1983) https://doi.org/10.1175/1520-0450(1983)022<0517:ASSFDE>2.0.CO;2
	double Temperature_UrbanCanyonSoil_K_Power04 = Temperature_Atmosphere_K_Power04 + (Coefficient_c_Eq28_Yang_etal_2013 * Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2 * (1 - Albedo_Soil_frac)) / (Emissivity_Soil_frac * Sigma_Stefan_Boltzmann);
	//Temperature_UrbanCanyonSoil_K (K) is Temperature_UrbanCanyonSoil_K_Power04 ^ 0.25
	Temperature_UrbanCanyonSoil_K = pow(Temperature_UrbanCanyonSoil_K_Power04, 0.25);

	//Temperature_UrbanCanyonWater_K (K) is estimated as Tdew_localCell_K (k), dew point temperature
	Temperature_UrbanCanyonWater_K = Tdew_localCell_K;

	//Radiation_Longwave_from_Wall_W_p_m2 (W/m2) from Eq 5.1 of Oke et al. (2017), Emittance (W/m2) = Emissivity * Sigma_Stefan_Boltzmann * Temperature_Surface_K^4 
	double Radiation_Longwave_from_Wall_W_p_m2 = Emissivity_Wall_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4);
	//Radiation_Longwave_from_Ground_W_p_m2  (W/m2) from Eq 5.1 of Oke et al. (2017), Emittance (W/m2) = Emissivity * Sigma_Stefan_Boltzmann * Temperature_Surface_K^4 
	double Radiation_Longwave_from_Ground_W_p_m2 = (Emissivity_Road_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonRoad_K, 4)) * Ratio_Road_to_Ground + (Emissivity_Garden_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyon_Garden_K, 4)) * Ratio_Garden_to_Ground;

	//Radiation_Longwave_Down_to_Ground_W_p_m2 (W/m2) is weighted average of Radiation_Longwave_from_Wall_W_p_m2 from wall and Radiation_Longwave_Down_W_p_m2 from sky
	//Note: SkyViewFactor_Road_frac determines amount of radiation from sky and (1-SkyViewFactor_Road_frac) determines radiation from wall
	Radiation_Longwave_Down_to_Ground_W_p_m2 = (1 - SkyViewFactor_Road_frac) * Radiation_Longwave_from_Wall_W_p_m2 + SkyViewFactor_Road_frac * Radiation_Longwave_Down_W_p_m2;
	//Radiation_Longwave_Down_to_Wall_W_p_m2 (W/m2) is weighted average of Radiation_Longwave_from_Ground_W_p_m2 from ground and Radiation_Longwave_Down_W_p_m2 from sky
	//Note: Radiation_Longwave_Net_Wall_W_p_m2 will add the radiation from the urban canyon walls; if added to Radiation_Longwave_Down_to_Wall_W_p_m2 then double counting
	Radiation_Longwave_Down_to_Wall_W_p_m2 = (1 - SkyViewFactor_Road_frac) * Radiation_Longwave_from_Ground_W_p_m2 + Count_CanyonWalls * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_W_p_m2;

	//Radiation_Longwave_Net_Wall_W_p_m2 (W/m2) from Eq 12 of Masson (2000), and Eq 39 of Lee and Park (2008)
	//Note: Town Energy Budget (TEB) model https://github.com/TEB-model/teb may explicitly compute surface longwave fluxes and temperature
	Radiation_Longwave_Net_Wall_W_p_m2 = Emissivity_Wall_frac * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Wall_W_p_m2 - Emissivity_Wall_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4) + Emissivity_Wall_frac * SkyViewFactor_Wall_frac * Emissivity_Wall_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4) + pow(Emissivity_Wall_frac, 2) * (1 - 2 * SkyViewFactor_Wall_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4) + Emissivity_Wall_frac * (1 - Emissivity_Wall_frac) * SkyViewFactor_Road_frac * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Wall_W_p_m2 + Emissivity_Wall_frac * (1 - Emissivity_Wall_frac) * SkyViewFactor_Wall_frac * (1 - 2 * SkyViewFactor_Wall_frac) * Radiation_Longwave_Down_to_Wall_W_p_m2 + pow(Emissivity_Wall_frac, 2) * (1 - Emissivity_Wall_frac) * (1 - 2 * SkyViewFactor_Wall_frac) * (1 - 2 * SkyViewFactor_Wall_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4) + pow(Emissivity_Wall_frac, 2) * (1 - Emissivity_Wall_frac) * SkyViewFactor_Wall_frac * (1 - SkyViewFactor_Road_frac) *	Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4) + Emissivity_Wall_frac * (1 - Emissivity_Wall_frac) * SkyViewFactor_Wall_frac * (1 - 2 * SkyViewFactor_Wall_frac) * Emissivity_Wall_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWall_K, 4);

	//Radiation_Longwave_Net_Pavement_W_p_m2 (W/m2) absorbed by snow-free road, from Eq 11 of Masson (2000), and Eq 40 of Lee and Park (2008)
	//Note: Radiation_Longwave_Net_Roof_W_p_m2 not in TEB of Masson (2000); does not radiate into urban canopy air volumes 
	//Note: Town Energy Budget (TEB) model https://github.com/TEB-model/teb may may explicitly compute surface longwave fluxes and temperature
	Radiation_Longwave_Net_Pavement_W_p_m2 = Emissivity_Pavement_frac * SkyViewFactor_Road_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 -Emissivity_Pavement_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonRoad_K, 4) + pow(Emissivity_Pavement_frac, 2) * (1 - SkyViewFactor_Road_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonRoad_K, 4) + Emissivity_Pavement_frac * (1 - Emissivity_Pavement_frac) * (1 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 + pow(Emissivity_Pavement_frac, 2) * (1 - Emissivity_Pavement_frac) * (1 - SkyViewFactor_Road_frac) * (1 - 2 * SkyViewFactor_Wall_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonRoad_K, 4) + Emissivity_Pavement_frac * (1 - Emissivity_Pavement_frac) * (1 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Emissivity_Pavement_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonRoad_K, 4);

	//Radiation_Longwave_Net_Tree_W_p_m2 (W/m2) absorbed by snow-free road, modified from Eq 11 of Masson (2000), and Eq 40 of Lee and Park (2008)
	Radiation_Longwave_Net_Tree_W_p_m2 = Emissivity_Tree_frac * SkyViewFactor_Garden_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 -Emissivity_Tree_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonTree_K, 4) +	pow(Emissivity_Tree_frac, 2) * (1 - SkyViewFactor_Garden_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonTree_K, 4) + Emissivity_Tree_frac * (1 - Emissivity_Tree_frac) * (1 - SkyViewFactor_Garden_frac) * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 +	pow(Emissivity_Tree_frac, 2) * (1 - Emissivity_Tree_frac) * (1 - SkyViewFactor_Garden_frac) * (1 - 2 * SkyViewFactor_Wall_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonTree_K, 4) + Emissivity_Tree_frac * (1 - Emissivity_Tree_frac) * (1 - SkyViewFactor_Garden_frac) * SkyViewFactor_Wall_frac * Emissivity_Tree_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonTree_K, 4);

	//Radiation_Longwave_Net_SVeg_W_p_m2 (W/m2) absorbed by snow-free road, modified from Eq 11 of Masson (2000), and Eq 40 of Lee and Park (2008)
	Radiation_Longwave_Net_SVeg_W_p_m2 = Emissivity_SVeg_frac * SkyViewFactor_Garden_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 -Emissivity_SVeg_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSVeg_K, 4) +	pow(Emissivity_SVeg_frac, 2) * (1 - SkyViewFactor_Garden_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSVeg_K, 4) + Emissivity_SVeg_frac * (1 - Emissivity_SVeg_frac) * (1 - SkyViewFactor_Garden_frac) * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 + pow(Emissivity_SVeg_frac, 2) * (1 - Emissivity_SVeg_frac) * (1 - SkyViewFactor_Garden_frac) * (1 - 2 * SkyViewFactor_Wall_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSVeg_K, 4) + Emissivity_SVeg_frac * (1 - Emissivity_SVeg_frac) * (1 - SkyViewFactor_Garden_frac) * SkyViewFactor_Wall_frac * Emissivity_SVeg_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSVeg_K, 4);

	//Radiation_Longwave_Net_Soil_W_p_m2 (W/m2) absorbed by snow-free road, modified from Eq 11 of Masson (2000), and Eq 40 of Lee and Park (2008)
	//Note: SkyViewFactor_Road_frac is used for Radiation_Longwave_Net_Soil_W_p_m2; think rural
	Radiation_Longwave_Net_Soil_W_p_m2 = Emissivity_Soil_frac * SkyViewFactor_Road_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 -Emissivity_Soil_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSoil_K, 4) +	pow(Emissivity_Soil_frac, 2) * (1 - SkyViewFactor_Road_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSoil_K, 4) +	Emissivity_Soil_frac * (1 - Emissivity_Soil_frac) * (1 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 +	pow(Emissivity_Soil_frac, 2) * (1 - Emissivity_Soil_frac) * (1 - SkyViewFactor_Road_frac) * (1 - 2 * SkyViewFactor_Wall_frac) *	Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSoil_K, 4) + Emissivity_Soil_frac * (1 - Emissivity_Soil_frac) * (1 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Emissivity_Soil_frac *	Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonSoil_K, 4);

	//Radiation_Longwave_Down_to_Ground_W_p_m2 (W/m2) from Eq 5.1 of Oke et al. (2017), Emittance (W/m2) = Emissivity * Sigma_Stefan_Boltzmann * Tair_localCell_K^4 
	//Note: Tair_localCell_K used for Radiation_Longwave_Down_to_Ground_W_p_m2 over water to represent the negative net radiation 
	Radiation_Longwave_Down_to_Ground_W_p_m2 = Emissivity_Water_frac * Sigma_Stefan_Boltzmann * pow(Temperature_Sky_K, 4);

	//Radiation_Longwave_Net_Water_W_p_m2 (W/m2) absorbed by snow-free road, modified from Eq 11 of Masson (2000), and Eq 40 of Lee and Park (2008)
	//Note: SkyViewFactor_Road_frac is used for Radiation_Longwave_Net_Water_W_p_m2; think Venice
	Radiation_Longwave_Net_Water_W_p_m2 = Emissivity_Water_frac * SkyViewFactor_Road_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 -Emissivity_Water_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWater_K, 4) + pow(Emissivity_Water_frac, 2) * (1 - SkyViewFactor_Road_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWater_K, 4) + Emissivity_Water_frac * (1 - Emissivity_Water_frac) * (1 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Radiation_Longwave_Down_to_Ground_W_p_m2 + pow(Emissivity_Water_frac, 2) * (1 - Emissivity_Water_frac) * (1 - SkyViewFactor_Road_frac) * (1 - 2 * SkyViewFactor_Wall_frac) * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWater_K, 4) + Emissivity_Water_frac * (1 - Emissivity_Water_frac) * (1 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Emissivity_Water_frac * Sigma_Stefan_Boltzmann * pow(Temperature_UrbanCanyonWater_K, 4);
}

//UrbanCanyon_RadiationShortwave function calculates urban canyon direct and diffuse shortwave energy balance once for each folder at each timeStep
//Note: Developed from TEB code SUBROUTINE urban_solar_abs.F90, Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier ...
//Note: ... This is part of the SURFEX software governed by the CeCILL version 2.1
void HeatFluxCal::UrbanCanyon_RadiationShortwave(Inputs* input, DataFolder* folder, bool Flag_UrbanCanyonResistance, int timeStep)
{
	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	double Denominator_Reflection = 0;
	double Reflection_Road_to_Wall_frac = 0;
	double Reflection_Garden_to_Wall_frac = 0;
	double Tangent_Angle_Zenith_Solar = 0;
	double Angle_SolarZenith_to_BuildingHeight_rad = 0;
	double GardenArea_frac = 0;
	double Radiation_Shortwave_Direct_Road_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_Road_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Tree_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_Tree_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_SVeg_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_SVeg_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Soil_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_Soil_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Water_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_Water_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Garden_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_Garden_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Wall_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Wall_A_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_Wall_B_W_p_m2 = 0;
	double Radiation_Shortwave_Diffuse_Wall_W_p_m2 = 0;

	//Radiation_Shortwave_Direct_Roof_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: Albedo_Roof_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 for direct radiation implements Eq 15 of Yang et al. (2008)
	//Note: Algorithm is simple, and in this model version is not attempting to correct cosine zenith angle (CZA) for roof tilt
	//Note: Consider refactor for fraction of solar panel area: ZABS_DIR_SW_RF   (JJ) = ZDIR_SW(JJ) * (1. - T%XALB_ROOF     (JJ)) * (1.-PFRAC_PANEL(JJ))
	double Radiation_Shortwave_Direct_Roof_W_p_m2 = Radiation_Shortwave_Direct_W_p_m2 * (1 - Albedo_Roof_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60);
	//Radiation_Shortwave_Diffuse_Garden_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: Consider refactor for fraction of solar panel area: ZABS_SCA_SW_RF   (JJ) = ZSCA_SW(JJ) * (1. - T%XALB_ROOF     (JJ)) * (1.-PFRAC_PANEL(JJ))
	double Radiation_Shortwave_Diffuse_Roof_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * (1 - Albedo_Roof_frac);

	//Direction_Road_rad when < 0 is uniform, no specific direction, using compass direction (0 = North, increases clockwise)
	Direction_Road_rad = -1;
	//Symmetry_Wall_int is 1 when symmetrical, 0 when not symmetrical
	Symmetry_Wall_int = 1;
	Glazing_Window = 1;
	double Albedo_WindowWall_frac = Glazing_Window * Albedo_Window_frac + (1.0 - Glazing_Window) * Albedo_Wall_frac;
	//Note: ZAALB_WL(:) =  B%XGR(:) * B%XALB_WIN(:) + (1.-B%XGR(:)) * T%XALB_WALL(:)

	//Start Algorithm for Roads and Walls ---------------------------------------------------------------------------------------------------------------------------------------
	//If Pi/2 - ZenithAngle_Solar_rad < Constant_1E_negative6 then
	if (abs(0.5 * M_PI - ZenithAngle_Solar_rad) < Constant_1E_negative6) {
		//If Pi/2 - ZenithAngle_Solar_rad > 0 then
		if (abs(0.5 * M_PI - ZenithAngle_Solar_rad) > 0.0) {
			//Tangent_Angle_Zenith_Solar is tan(0.5 * M_PI - Constant_1E_negative6)
			//Note: ZTANZEN(JJ)=TAN(0.5*XPI-1.E-6)
			Tangent_Angle_Zenith_Solar = tan(0.5 * M_PI - Constant_1E_negative6);
		}
		//If Pi/2 - ZenithAngle_Solar_rad <= 0 then
		if (abs(0.5 * M_PI - ZenithAngle_Solar_rad) > 0.0) {
			//Tangent_Angle_Zenith_Solar is tan(0.5 * M_PI + Constant_1E_negative6)
			//Note: ZTANZEN(JJ)=TAN(0.5*XPI+1.E-6)
			Tangent_Angle_Zenith_Solar = tan(0.5 * M_PI + Constant_1E_negative6);
		}
	}
	//Else If abs(ZenithAngle_Solar_rad < Constant_1E_negative6 then
	//Note: ZTANZEN(JJ)=SIGN(1.,PZENITH(JJ))*TAN(1.E-6)
	else if (abs(ZenithAngle_Solar_rad < Constant_1E_negative6)) {
		//if ZenithAngle_Solar_rad > 0  
		if (ZenithAngle_Solar_rad > 0) {
			//Tangent_Angle_Zenith_Solar is positive tan(Constant_1E_negative6)
			Tangent_Angle_Zenith_Solar = tan(Constant_1E_negative6);
		}
		else if (ZenithAngle_Solar_rad < 0) {
			//Tangent_Angle_Zenith_Solar is negative tan(Constant_1E_negative6)
			Tangent_Angle_Zenith_Solar = -1 * tan(Constant_1E_negative6);
		}
		else {
			Tangent_Angle_Zenith_Solar = 0;
		}
	}
	else {
		//Tangent_Angle_Zenith_Solar is tan(ZenithAngle_Solar_rad)
		Tangent_Angle_Zenith_Solar = tan(ZenithAngle_Solar_rad);
		//ZTANZEN(JJ) = TAN(PZENITH(JJ))
	}

	//Algorithm for radiation coefficients
	//If Building_Area_frac > 0, and AspectRatio_Canyon_frac  and Tangent_Angle_Zenith_Solar > 0 then proceed and divide by those terms
	if (Building_Area_frac > 0 && AspectRatio_Canyon_frac > 0 && Tangent_Angle_Zenith_Solar > 0) {
		//Algorithm for direct solar radiation on road
		//If Direction_Road_rad < 0 then no specific orientation, equivalent to IF (TOP%CROAD_DIR=='UNIF'), then
		//Note: TEB uses UNIF or ORIE, and ORIE allows for 1 direction per TEB patch
		if (Direction_Road_rad < 0) {
			//Angle_SolarZenith_to_BuildingHeight_rad (rad) function from TEB code SUBROUTINE urban_solar_abs.F90
			//Note: ZTHETA0(JJ) = ASIN( MIN(ABS( 1./ZTANZEN(JJ))/T%XCAN_HW_RATIO(JJ), 1. ) )
			Angle_SolarZenith_to_BuildingHeight_rad = asin(MIN(abs(1.0 / Tangent_Angle_Zenith_Solar) * 1 / AspectRatio_Canyon_frac, 1));

			//Radiation_Shortwave_Direct_Road_W_p_m2 (rad) function from TEB code SUBROUTINE urban_solar_abs.F90
			//Note: ZDIR_SW_RD (JJ) = (ZDIR_SW(JJ) * 2. * ZTHETA0(JJ) / XPI - ZDIR_SW(JJ) * 2. * ZTANZEN(JJ) / XPI * T%XCAN_HW_RATIO(JJ) * (1.-COS(ZTHETA0(JJ))))
			Radiation_Shortwave_Direct_Road_W_p_m2 = (Radiation_Shortwave_W_p_m2 * 2.0 * Angle_SolarZenith_to_BuildingHeight_rad / M_PI- Radiation_Shortwave_W_p_m2 * 2.0 * Tangent_Angle_Zenith_Solar / M_PI * AspectRatio_Canyon_frac * (1 - cos(Angle_SolarZenith_to_BuildingHeight_rad)));
		}
		//Else If Direction_Road_rad >= 0 then specific orientation then
		else {
			//Radiation_Shortwave_Direct_Road_W_p_m2 (rad) function from TEB code SUBROUTINE urban_solar_abs.F90
			//Note: ZDIR_SW_RD (JJ) = ZDIR_SW(JJ) * max(0.,1.-T%XCAN_HW_RATIO(JJ)*ZTANZEN(JJ)*ABS(SIN(PAZIM(JJ)-ZRD_DIR(JJ))))
			Radiation_Shortwave_Direct_Road_W_p_m2 = Radiation_Shortwave_W_p_m2 * max(0.0, 1.0 - AspectRatio_Canyon_frac * Tangent_Angle_Zenith_Solar * abs(sin(AzimuthAngle_Solar_N_0_rad - Direction_Road_rad)));

		}
		//Radiation_Shortwave_Direct_Garden_W_p_m2 and other horizontal surfaces equals Radiation_Shortwave_Direct_Road_W_p_m2 according to TEB
		Radiation_Shortwave_Direct_Tree_W_p_m2 = Radiation_Shortwave_Direct_Road_W_p_m2;
		Radiation_Shortwave_Direct_SVeg_W_p_m2 = Radiation_Shortwave_Direct_Road_W_p_m2;
		Radiation_Shortwave_Direct_Soil_W_p_m2 = Radiation_Shortwave_Direct_Road_W_p_m2;
		Radiation_Shortwave_Direct_Water_W_p_m2 = Radiation_Shortwave_Direct_Road_W_p_m2;
		Radiation_Shortwave_Direct_Garden_W_p_m2 = Radiation_Shortwave_Direct_Road_W_p_m2;

		//Algorithm for direct solar radiation on walls
		//Radiation_Shortwave_Direct_Wall_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
		//Note: ZDIR_SW_WL(JJ) = (ZDIR_SW(JJ) - (ZDIR_SW_RD(JJ)*ZRD(JJ)+ZDIR_SW_GD(JJ)*ZGD(JJ))) * 0.5 / T%XCAN_HW_RATIO(JJ)
		Radiation_Shortwave_Direct_Wall_W_p_m2 = (Radiation_Shortwave_W_p_m2 - (Radiation_Shortwave_Direct_Road_W_p_m2 * Ratio_Road_to_Ground + Radiation_Shortwave_Direct_Garden_W_p_m2 * Ratio_Garden_to_Ground)) * 0.5 * AspectRatio_Canyon_frac;
	}
	//Else if Building_Area_frac <= 0 then
	else {
		//Set radiation to standard values for no urban canyon
		Radiation_Shortwave_Direct_Road_W_p_m2 = Radiation_Shortwave_W_p_m2;
		Radiation_Shortwave_Direct_Garden_W_p_m2 = Radiation_Shortwave_W_p_m2;
		Radiation_Shortwave_Direct_Wall_W_p_m2 = 0;
	}

	//If Direction_Road_rad < 0 then road has no specified direction and If Symmetry_Wall_int > 0 then walls symmetrical
	if (Direction_Road_rad < 0 || Symmetry_Wall_int > 0) {
		//Radiation_Shortwave_Direct_Wall_A_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
		Radiation_Shortwave_Direct_Wall_A_W_p_m2 = Radiation_Shortwave_Direct_Wall_W_p_m2;
		Radiation_Shortwave_Direct_Wall_B_W_p_m2 = Radiation_Shortwave_Direct_Wall_W_p_m2;
	}
	//Else If Direction_Road_rad >= 0 then road has specified direction and If Symmetry_Wall_int < 0 then walls not symmetrical
	//Note: For N-S road Wall_A is west wall and east facing, Wall_B is west facing
	//Note: For E-W road Wall_A is north wall and south facing, Wall_B is north facing
	else {
		if (sin(AzimuthAngle_Solar_N_0_rad - Direction_Road_rad) > 0) {
			//Radiation_Shortwave_Direct_Wall_A_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
			Radiation_Shortwave_Direct_Wall_A_W_p_m2 = 2.0 * Radiation_Shortwave_Direct_Wall_W_p_m2;
			//Radiation_Shortwave_Direct_Wall_B_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
			Radiation_Shortwave_Direct_Wall_B_W_p_m2 = 0;
		}
		else {
			//Radiation_Shortwave_Direct_Wall_A_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
			Radiation_Shortwave_Direct_Wall_A_W_p_m2 = 0;
			//Radiation_Shortwave_Direct_Wall_B_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
			Radiation_Shortwave_Direct_Wall_B_W_p_m2 = 2.0 * Radiation_Shortwave_Direct_Wall_W_p_m2;
		}
	}

	//Diffuse radiation
	//Radiation_Shortwave_Diffuse_Road_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
	Radiation_Shortwave_Diffuse_Road_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Road_frac;
	//Radiation_Shortwave_Diffuse_Garden_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
	Radiation_Shortwave_Diffuse_Garden_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Garden_frac;
	//Radiation_Shortwave_Diffuse_Wall_W_p_m2 (W/m2) function from TEB code SUBROUTINE urban_solar_abs.F90
	Radiation_Shortwave_Diffuse_Wall_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Wall_frac;
	//Radiation_Shortwave_Diffuse_Tree_W_p_m2, SVeg, Soil, Water new, and share SkyViewFactor_Garden_frac; insired by TEB code
	Radiation_Shortwave_Diffuse_Tree_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Garden_frac;
	Radiation_Shortwave_Diffuse_SVeg_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Garden_frac;
	Radiation_Shortwave_Diffuse_Soil_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Garden_frac;
	Radiation_Shortwave_Diffuse_Water_W_p_m2 = Radiation_Shortwave_Diffuse_W_p_m2 * SkyViewFactor_Garden_frac;

	//Solar Reflections for Direct _____________________________________________________________________________________________
	//First Reflection
	double Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 = Albedo_Road_frac * Radiation_Shortwave_Direct_Road_W_p_m2;
	double Radiation_Shortwave_Direct_ReflectionFirst_Garden_W_p_m2 = Albedo_Garden_frac * Radiation_Shortwave_Direct_Garden_W_p_m2;
	double Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 = Albedo_Wall_frac * Radiation_Shortwave_Direct_Wall_W_p_m2;
	//Radiation_Shortwave_Direct_ReflectionFirst_Tree_W_p_m2, SVeg, Soil, Water new; insired by TEB code
	double Radiation_Shortwave_Direct_ReflectionFirst_Tree_W_p_m2 = Albedo_Tree_frac * Radiation_Shortwave_Direct_Tree_W_p_m2;
	double Radiation_Shortwave_Direct_ReflectionFirst_SVeg_W_p_m2 = Albedo_SVeg_frac * Radiation_Shortwave_Direct_SVeg_W_p_m2;
	double Radiation_Shortwave_Direct_ReflectionFirst_Soil_W_p_m2 = Albedo_Soil_frac * Radiation_Shortwave_Direct_Soil_W_p_m2;
	double Radiation_Shortwave_Direct_ReflectionFirst_Water_W_p_m2 = Albedo_Water_frac * Radiation_Shortwave_Direct_Water_W_p_m2;

	//Sum of Solar Reflections
	//Denominator_Reflection function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: Albedo_Road_frac, Albedo_Wall_frac, Albedo_Garden_frac are used in separately in this equation, documented below
	//Note: ZDENOM(JJ) = 1. - (1.-2.*T%XSVF_WALL(JJ)) * T%XALB_WALL(JJ) - (1. - T%XSVF_ROAD(JJ))* T%XSVF_WALL(JJ)*T%XALB_WALL(JJ)*ZAALB_RD(JJ)*ZRD(JJ) - (1. - PSVF_GD(JJ))* T%XSVF_WALL(JJ)*T%XALB_WALL(JJ)*PALB_GD(JJ)*ZGD(JJ)
	Denominator_Reflection = 1.0 - (1.0 - 2 * SkyViewFactor_Wall_frac) * Albedo_Wall_frac -	(1.0 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Albedo_Wall_frac * Albedo_Road_frac * Ratio_Road_to_Ground - (1.0 - SkyViewFactor_Garden_frac) * SkyViewFactor_Wall_frac * Albedo_Wall_frac * Albedo_Garden_frac * Ratio_Garden_to_Ground;
	//Reflection_Road_to_Wall_frac function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZWORK1(JJ) = T%XSVF_WALL(JJ) * T%XALB_WALL(JJ) * ZRD(JJ)
	Reflection_Road_to_Wall_frac = SkyViewFactor_Wall_frac * Albedo_Wall_frac * Ratio_Road_to_Ground;
	//Reflection_Garden_to_Wall_frac function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZWORK2(JJ) = T%XSVF_WALL(JJ) * T%XALB_WALL(JJ) * ZGD(JJ)
	Reflection_Garden_to_Wall_frac = SkyViewFactor_Wall_frac * Albedo_Wall_frac * Ratio_Garden_to_Ground;
	//Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZSREF_SW_WL(JJ) = (  ZREF0_SW_WL(JJ) + ZWORK1(JJ) *ZREF0_SW_RD(JJ)+ ZWORK2(JJ) *ZREF0_SW_GD(JJ)) / ZDENOM(JJ)
	double Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 = (Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Garden_W_p_m2) / Denominator_Reflection;
	//Radiation_Shortwave_Direct_ReflectionSum_Road_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: Albedo_Road_frac is used in all parts of this equation, documented below
	//Note: ZSREF_SW_RD(JJ) = ((1. - T % XSVF_ROAD(JJ)) * ZAALB_RD(JJ) * ZREF0_SW_WL(JJ) + (1. - T % XSVF_ROAD(JJ)) * ZAALB_RD(JJ) * ZWORK1(JJ) * ZREF0_SW_RD(JJ) + (1. - T % XSVF_ROAD(JJ)) * ZAALB_RD(JJ) * ZWORK2(JJ) * ZREF0_SW_GD(JJ)) / ZDENOM(JJ) + ZREF0_SW_RD(JJ)
	double Radiation_Shortwave_Direct_ReflectionSum_Road_W_p_m2 = ((1.0 - SkyViewFactor_Road_frac) * Albedo_Road_frac * Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Road_frac) * Albedo_Road_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Road_frac) * Albedo_Road_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Garden_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2;
	//Radiation_Shortwave_Direct_ReflectionSum_Garden_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZSREF_SW_GD(JJ) = ((1.- PSVF_GD(JJ)) * PALB_GD(JJ) * ZREF0_SW_WL(JJ)+(1.- PSVF_GD(JJ)) * PALB_GD(JJ) * ZWORK1(JJ) * ZREF0_SW_RD(JJ)+(1.- PSVF_GD(JJ)) * PALB_GD(JJ) * ZWORK2(JJ) * ZREF0_SW_GD(JJ) )/ ZDENOM(JJ) + ZREF0_SW_GD(JJ)
	double Radiation_Shortwave_Direct_ReflectionSum_Garden_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Garden_frac * Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Garden_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Garden_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Garden_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Direct_ReflectionFirst_Garden_W_p_m2;
	//Radiation_Shortwave_Direct_ReflectionSum_Tree_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Direct_ReflectionSum_Tree_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Tree_frac * Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Tree_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Tree_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Tree_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Direct_ReflectionFirst_Tree_W_p_m2;
	//Radiation_Shortwave_Direct_ReflectionSum_SVeg_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Direct_ReflectionSum_SVeg_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_SVeg_frac * Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_SVeg_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_SVeg_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_SVeg_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Direct_ReflectionFirst_SVeg_W_p_m2;
	//Radiation_Shortwave_Direct_ReflectionSum_Soil_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Direct_ReflectionSum_Soil_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Soil_frac * Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Soil_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Soil_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Soil_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Direct_ReflectionFirst_Soil_W_p_m2;
	//Radiation_Shortwave_Direct_ReflectionSum_Water_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Direct_ReflectionSum_Water_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Water_frac * Radiation_Shortwave_Direct_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Water_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Water_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Direct_ReflectionFirst_Water_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Direct_ReflectionFirst_Water_W_p_m2;
	//Total Radiation Received
	//Note: Snow will change Albedo_Road_frac to Albedo_Road_w_Snow_frac
	//Radiation_Shortwave_Direct_w_Reflected_Road_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_RD(JJ) = (1.-T%XALB_ROAD(JJ)) * (ZSW_RD(JJ) + ZSREF_SW_WL(JJ) * (1.- T%XSVF_ROAD(JJ)))
	//Note: Albedo_Road_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 for direct radiation implements Eq 15 of Yang et al. (2008)
	double Radiation_Shortwave_Direct_w_Reflected_Road_W_p_m2 = (1.0 - Albedo_Road_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_Road_W_p_m2 + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Road_frac);
	//Radiation_Shortwave_Direct_w_Reflected_Garden_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_GD(JJ) = (1# - PALB_GD(JJ)) * (ZSW_GD(JJ) + ZSREF_SW_WL(JJ) * (1# - PSVF_GD(JJ)))
	//Note: Albedo_Garden_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 for direct radiation implements Eq 15 of Yang et al. (2008)
	double Radiation_Shortwave_Direct_w_Reflected_Garden_W_p_m2 = (1.0 - Albedo_Garden_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_Garden_W_p_m2+ Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Direct_w_Reflected_Tree_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Direct_w_Reflected_Tree_W_p_m2 = (1.0 - Albedo_Tree_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_Tree_W_p_m2 + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Direct_w_Reflected_SVeg_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Direct_w_Reflected_SVeg_W_p_m2 = (1.0 - Albedo_SVeg_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_SVeg_W_p_m2 + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Direct_w_Reflected_Soil_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Direct_w_Reflected_Soil_W_p_m2 = (1.0 - Albedo_Soil_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_Soil_W_p_m2 + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Direct_w_Reflected_Water_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Direct_w_Reflected_Water_W_p_m2 = (1.0 - Albedo_Water_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_Water_W_p_m2 + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_WL(JJ) = (1.-T%XALB_WALL(JJ))* (ZSW_WL(JJ)+ ZSREF_SW_RD(JJ) * T%XSVF_WALL(JJ)*ZRD(JJ) + ZSREF_SW_GD(JJ) * T%XSVF_WALL(JJ)*ZGD(JJ) + ZSREF_SW_WL(JJ) * (1.-2.*T%XSVF_WALL(JJ)))
	//Note: Albedo_Wall_frac * (1/Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) vertical surface direct radiation implements Eq 15 of Yang et al. (2008)
	//Note: Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 is for horizontal surfaces; for walls ratio_vertical = 1/Ratio_AlbedoDirCZA_to_AlbedoDirCZA60
	double Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 = (1.0 - Albedo_Wall_frac * (1/Ratio_AlbedoDirCZA_to_AlbedoDirCZA60)) * Radiation_Shortwave_Direct_Wall_W_p_m2 + Radiation_Shortwave_Direct_ReflectionSum_Road_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Road_to_Ground + Radiation_Shortwave_Direct_ReflectionSum_Garden_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Garden_to_Ground + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - 2.0 * SkyViewFactor_Wall_frac);
	//Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_WIN(JJ) = (1.-T%XALB_WIN(JJ))* (ZSW_WL(JJ)+ ZSREF_SW_RD(JJ) * T%XSVF_WALL(JJ)*ZRD(JJ) + ZSREF_SW_GD(JJ) * T%XSVF_WALL(JJ)*ZGD(JJ) + ZSREF_SW_WL(JJ) * (1.-2.*T%XSVF_WALL(JJ)))
	//Note: Albedo_Window_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60 for direct radiation implements Eq 15 of Yang et al. (2008)
	double Radiation_Shortwave_Direct_w_Reflected_Window_W_p_m2 = (1.0 - Albedo_Window_frac * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60) * Radiation_Shortwave_Direct_Wall_W_p_m2+ Radiation_Shortwave_Direct_ReflectionSum_Road_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Road_to_Ground + Radiation_Shortwave_Direct_ReflectionSum_Garden_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Garden_to_Ground + Radiation_Shortwave_Direct_ReflectionSum_Wall_W_p_m2 * (1.0 - 2.0 * SkyViewFactor_Wall_frac);
	//End Solar Reflections for Direct _____________________________________________________________________________________________


	//Solar Reflections for Diffuse _____________________________________________________________________________________________
	//Note: First Reflection, where reflection is from Albedo_Road_frac, and absorption is from (1 - Albedo_Road_frac)
	//Note: ZREF0_SW_RD(JJ) = ZAALB_RD(JJ) * ZSW_RD(JJ); ZREF0_SW_GD(JJ) = PALB_GD(JJ)  * ZSW_GD(JJ); ZREF0_SW_WL(JJ) = ZAALB_WL(JJ) * ZSW_WL(JJ)
	double Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 = Albedo_Road_frac * Radiation_Shortwave_Diffuse_Road_W_p_m2;
	double Radiation_Shortwave_Diffuse_ReflectionFirst_Garden_W_p_m2 = Albedo_Garden_frac * Radiation_Shortwave_Diffuse_Garden_W_p_m2;
	double Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2 = Albedo_Wall_frac * Radiation_Shortwave_Diffuse_Wall_W_p_m2;
	//Radiation_Shortwave_Diffuse_ReflectionFirst_Tree_W_p_m2, SVeg, Soil, Water new; insired by TEB code
	double Radiation_Shortwave_Diffuse_ReflectionFirst_Tree_W_p_m2 = Albedo_Tree_frac * Radiation_Shortwave_Diffuse_Tree_W_p_m2;
	double Radiation_Shortwave_Diffuse_ReflectionFirst_SVeg_W_p_m2 = Albedo_SVeg_frac * Radiation_Shortwave_Diffuse_SVeg_W_p_m2;
	double Radiation_Shortwave_Diffuse_ReflectionFirst_Soil_W_p_m2 = Albedo_Soil_frac * Radiation_Shortwave_Diffuse_Soil_W_p_m2;
	double Radiation_Shortwave_Diffuse_ReflectionFirst_Water_W_p_m2 = Albedo_Water_frac * Radiation_Shortwave_Diffuse_Water_W_p_m2;

	//Sum of Solar Reflections
	//Denominator_Reflection function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZDENOM(JJ) = 1. - (1.-2.*T%XSVF_WALL(JJ)) * T%XALB_WALL(JJ) - (1. - T%XSVF_ROAD(JJ))* T%XSVF_WALL(JJ)*T%XALB_WALL(JJ)*ZAALB_RD(JJ)*ZRD(JJ) - (1. - PSVF_GD(JJ))* T%XSVF_WALL(JJ)*T%XALB_WALL(JJ)*PALB_GD(JJ)*ZGD(JJ)
	Denominator_Reflection = 1.0 - (1.0 - 2 * SkyViewFactor_Wall_frac) * Albedo_Wall_frac - (1.0 - SkyViewFactor_Road_frac) * SkyViewFactor_Wall_frac * Albedo_Wall_frac * Albedo_Road_frac * Ratio_Road_to_Ground - (1.0 - SkyViewFactor_Garden_frac) * SkyViewFactor_Wall_frac * Albedo_Wall_frac * Albedo_Garden_frac * Ratio_Garden_to_Ground;
	//Reflection_Road_to_Wall_frac function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZWORK1(JJ) = T%XSVF_WALL(JJ) * T%XALB_WALL(JJ) * ZRD(JJ)
	Reflection_Road_to_Wall_frac = SkyViewFactor_Wall_frac * Albedo_Wall_frac * Ratio_Road_to_Ground;
	//Reflection_Garden_to_Wall_frac function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZWORK2(JJ) = T%XSVF_WALL(JJ) * T%XALB_WALL(JJ) * ZGD(JJ)
	Reflection_Garden_to_Wall_frac = SkyViewFactor_Wall_frac * Albedo_Wall_frac * Ratio_Garden_to_Ground;
	//Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZSREF_SW_WL(JJ) = (  ZREF0_SW_WL(JJ) + ZWORK1(JJ) *ZREF0_SW_RD(JJ)+ ZWORK2(JJ) *ZREF0_SW_GD(JJ)) / ZDENOM(JJ)
	double Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 = (Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2	+ Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Garden_W_p_m2) / Denominator_Reflection;
	//Radiation_Shortwave_Diffuse_ReflectionSum_Road_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZSREF_SW_RD(JJ) = ((1. - T % XSVF_ROAD(JJ)) * ZAALB_RD(JJ) * ZREF0_SW_WL(JJ) + (1. - T % XSVF_ROAD(JJ)) * ZAALB_RD(JJ) * ZWORK1(JJ) * ZREF0_SW_RD(JJ) + (1. - T % XSVF_ROAD(JJ)) * ZAALB_RD(JJ) * ZWORK2(JJ) * ZREF0_SW_GD(JJ)) / ZDENOM(JJ) + ZREF0_SW_RD(JJ)
	double Radiation_Shortwave_Diffuse_ReflectionSum_Road_W_p_m2 = ((1.0 - SkyViewFactor_Road_frac) * Albedo_Road_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2	+ (1.0 - SkyViewFactor_Road_frac) * Albedo_Road_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Road_frac) * Albedo_Road_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Garden_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2;
	//Radiation_Shortwave_Diffuse_ReflectionSum_Garden_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZSREF_SW_GD(JJ) = ((1.- PSVF_GD(JJ)) * PALB_GD(JJ) * ZREF0_SW_WL(JJ)+(1.- PSVF_GD(JJ)) * PALB_GD(JJ) * ZWORK1(JJ) * ZREF0_SW_RD(JJ)+(1.- PSVF_GD(JJ)) * PALB_GD(JJ) * ZWORK2(JJ) * ZREF0_SW_GD(JJ) )/ ZDENOM(JJ) + ZREF0_SW_GD(JJ)
	double Radiation_Shortwave_Diffuse_ReflectionSum_Garden_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Garden_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2	+ (1.0 - SkyViewFactor_Garden_frac) * Albedo_Garden_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Garden_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Garden_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Diffuse_ReflectionFirst_Garden_W_p_m2;
	//Radiation_Shortwave_Diffuse_ReflectionSum_Tree_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_ReflectionSum_Tree_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Tree_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Tree_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Tree_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Tree_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Diffuse_ReflectionFirst_Tree_W_p_m2;
	//Radiation_Shortwave_Diffuse_ReflectionSum_SVeg_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_ReflectionSum_SVeg_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_SVeg_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_SVeg_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_SVeg_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_SVeg_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Diffuse_ReflectionFirst_SVeg_W_p_m2;
	//Radiation_Shortwave_Diffuse_ReflectionSum_Soil_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_ReflectionSum_Soil_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Soil_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Soil_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Soil_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Soil_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Diffuse_ReflectionFirst_Soil_W_p_m2;
	//Radiation_Shortwave_Diffuse_ReflectionSum_Water_W_p_m2 function is new, uses SkyViewFactor_Garden_frac, Reflection_Garden_to_Wall_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_ReflectionSum_Water_W_p_m2 = ((1.0 - SkyViewFactor_Garden_frac) * Albedo_Water_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Wall_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Water_frac * Reflection_Road_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Road_W_p_m2 + (1.0 - SkyViewFactor_Garden_frac) * Albedo_Water_frac * Reflection_Garden_to_Wall_frac * Radiation_Shortwave_Diffuse_ReflectionFirst_Water_W_p_m2) / Denominator_Reflection + Radiation_Shortwave_Diffuse_ReflectionFirst_Water_W_p_m2;

	//Total Radiation Received
	//Note: Snow will change Albedo_Road_frac to Albedo_Road_w_Snow_frac
	//Radiation_Shortwave_Diffuse_w_Reflected_Road_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_RD(JJ) = (1.-T%XALB_ROAD(JJ)) * (ZSW_RD(JJ) + ZSREF_SW_WL(JJ) * (1.- T%XSVF_ROAD(JJ)))
	double Radiation_Shortwave_Diffuse_w_Reflected_Road_W_p_m2 = (1.0 - Albedo_Road_frac) * Radiation_Shortwave_Diffuse_Road_W_p_m2	+ Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Road_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_Garden_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_GD(JJ) = (1# - PALB_GD(JJ)) * (ZSW_GD(JJ) + ZSREF_SW_WL(JJ) * (1# - PSVF_GD(JJ)))
	double Radiation_Shortwave_Diffuse_w_Reflected_Garden_W_p_m2 = (1.0 - Albedo_Garden_frac) * Radiation_Shortwave_Diffuse_Garden_W_p_m2 + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_Tree_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_w_Reflected_Tree_W_p_m2 = (1.0 - Albedo_Tree_frac) * Radiation_Shortwave_Diffuse_Tree_W_p_m2 + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_SVeg_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_w_Reflected_SVeg_W_p_m2 = (1.0 - Albedo_SVeg_frac) * Radiation_Shortwave_Diffuse_SVeg_W_p_m2 + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_Soil_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_w_Reflected_Soil_W_p_m2 = (1.0 - Albedo_Soil_frac) * Radiation_Shortwave_Diffuse_Soil_W_p_m2 + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_Water_W_p_m2 function is new, uses SkyViewFactor_Garden_frac; insired by TEB code
	double Radiation_Shortwave_Diffuse_w_Reflected_Water_W_p_m2 = (1.0 - Albedo_Water_frac) * Radiation_Shortwave_Diffuse_Water_W_p_m2 + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - SkyViewFactor_Garden_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_Wall_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_WL(JJ) = (1.-T%XALB_WALL(JJ))* (ZSW_WL(JJ)+ ZSREF_SW_RD(JJ) * T%XSVF_WALL(JJ)*ZRD(JJ) + ZSREF_SW_GD(JJ) * T%XSVF_WALL(JJ)*ZGD(JJ) + ZSREF_SW_WL(JJ) * (1.-2.*T%XSVF_WALL(JJ)))
	double Radiation_Shortwave_Diffuse_w_Reflected_Wall_W_p_m2 = (1.0 - Albedo_Wall_frac) * Radiation_Shortwave_Diffuse_Wall_W_p_m2	+ Radiation_Shortwave_Diffuse_ReflectionSum_Road_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Road_to_Ground + Radiation_Shortwave_Diffuse_ReflectionSum_Garden_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Garden_to_Ground + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - 2.0 * SkyViewFactor_Wall_frac);
	//Radiation_Shortwave_Diffuse_w_Reflected_Wall_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
	//Note: ZABS_SW_WIN(JJ) = (1.-T%XALB_WIN(JJ))* (ZSW_WL(JJ)+ ZSREF_SW_RD(JJ) * T%XSVF_WALL(JJ)*ZRD(JJ) + ZSREF_SW_GD(JJ) * T%XSVF_WALL(JJ)*ZGD(JJ) + ZSREF_SW_WL(JJ) * (1.-2.*T%XSVF_WALL(JJ)))
	double Radiation_Shortwave_Diffuse_w_Reflected_Window_W_p_m2 = (1.0 - Albedo_Window_frac) * Radiation_Shortwave_Diffuse_Wall_W_p_m2+ Radiation_Shortwave_Diffuse_ReflectionSum_Road_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Road_to_Ground + Radiation_Shortwave_Diffuse_ReflectionSum_Garden_W_p_m2 * SkyViewFactor_Wall_frac * Ratio_Garden_to_Ground + Radiation_Shortwave_Diffuse_ReflectionSum_Wall_W_p_m2 * (1.0 - 2.0 * SkyViewFactor_Wall_frac);
	//End Solar Reflections for Diffuse _____________________________________________________________________________________________

	//Adjustment for Direct Radiation (not Diffuse) _________________________________________________________________________________
	double Radiation_Shortwave_Direct_w_Reflected_Wall_A_W_p_m2 = 0;
	double Radiation_Shortwave_Direct_w_Reflected_Wall_B_W_p_m2 = 0;
	//If Direction_Road_rad < 0 then road has no specified direction and If Symmetry_Wall_int > 0 then walls symmetrical
	if (Direction_Road_rad < 0 || Symmetry_Wall_int > 0) {
		//Note: Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 is absorbed, already adjusted for (1 - albedo * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60)
		Radiation_Shortwave_Direct_w_Reflected_Wall_A_W_p_m2 = Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2;
		Radiation_Shortwave_Direct_w_Reflected_Wall_B_W_p_m2 = Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2;
	}
	//Else If Direction_Road_rad >= 0 then road has specified direction or If Symmetry_Wall_int < 0 then walls are not symmetrical
	//Note: For N-S road Wall_A is west wall and east facing, Wall_B is west facing
	//Note: For E-W road Wall_A is north wall and south facing, Wall_B is north facing
	else {
		//Radiation_Shortwave_Direct_w_Reflected_Wall_Extra_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
		//Note: ZDW = (1.-T%XALB_WALL(:)) * ZAALB_WL(:) * (1.-2.*T%XSVF_WALL(:)) / (1.+ZAALB_WL(:)*(1.-2.*T%XSVF_WALL(:)))* 0.5 * (ZDIR_SW_WL_A(:)-ZDIR_SW_WL_B(:))+ 0.5 * (1.-T%XALB_WALL(:)) * (ZDIR_SW_WL_A-ZDIR_SW_WL_B)
		double Radiation_Shortwave_Direct_w_Reflected_Wall_Extra_W_p_m2 = (1.0 - Albedo_Wall_frac) * Albedo_WindowWall_frac * (1.0 - 2.0 * SkyViewFactor_Wall_frac) / (1.0 + Albedo_WindowWall_frac * (1.0 - 2.0 * SkyViewFactor_Wall_frac))	* 0.5 * (Radiation_Shortwave_Direct_Wall_A_W_p_m2 - Radiation_Shortwave_Direct_Wall_B_W_p_m2) + 0.5 * (1.0 - Albedo_Wall_frac) * (Radiation_Shortwave_Direct_Wall_A_W_p_m2 - Radiation_Shortwave_Direct_Wall_B_W_p_m2);
		//Radiation_Shortwave_Direct_w_Reflected_Wall_A_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
		//Note: ZABS_DIR_SW_WL_A = ZABS_DIR_SW_WL + ZDW
		//Note: Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 is absorbed, already adjusted for (1 - albedo * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60)
		Radiation_Shortwave_Direct_w_Reflected_Wall_A_W_p_m2 = Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 + Radiation_Shortwave_Direct_w_Reflected_Wall_Extra_W_p_m2;
		//Radiation_Shortwave_Direct_w_Reflected_Wall_B_W_p_m2 function from TEB code SUBROUTINE urban_solar_abs.F90
		//Note: ZABS_DIR_SW_WL_B = ZABS_DIR_SW_WL - ZDW
		//Note: Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 is absorbed, already adjusted for (1 - albedo * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60)
		Radiation_Shortwave_Direct_w_Reflected_Wall_B_W_p_m2 = Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 - Radiation_Shortwave_Direct_w_Reflected_Wall_Extra_W_p_m2;
	}
	//End Adjustment for Direct Radiation (not Diffuse) _________________________________________________________________________________
	//End Algorithm for Roads and Walls ---------------------------------------------------------------------------------------------------------------------------------------

	//Limit canyon facet direct radiation to no larger than direct radiation, Radiation_Shortwave_Direct_W_p_m2
	Radiation_Shortwave_Direct_Road_W_p_m2 = min(Radiation_Shortwave_Direct_Road_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Garden_W_p_m2 = min(Radiation_Shortwave_Direct_Garden_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Tree_W_p_m2 = min(Radiation_Shortwave_Direct_Tree_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_SVeg_W_p_m2 = min(Radiation_Shortwave_Direct_SVeg_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Soil_W_p_m2 = min(Radiation_Shortwave_Direct_Soil_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Water_W_p_m2 = min(Radiation_Shortwave_Direct_Water_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Wall_W_p_m2 = min(Radiation_Shortwave_Direct_Wall_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Wall_A_W_p_m2 = min(Radiation_Shortwave_Direct_Wall_A_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);
	Radiation_Shortwave_Direct_Wall_B_W_p_m2 = min(Radiation_Shortwave_Direct_Wall_B_W_p_m2, Radiation_Shortwave_Direct_W_p_m2);

	//Total of direct and diffuse radiation
	//Note: Radiation_Shortwave_Direct_w_Reflected_Road_W_p_m2 is absorbed, already adjusted for (1 - albedo * Ratio_AlbedoDirCZA_to_AlbedoDirCZA60)
	//Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_Road_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_Road_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_Garden_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_Garden_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_Tree_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Tree_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_Tree_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_Tree_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_SVeg_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_SVeg_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_SVeg_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_SVeg_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_Soil_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Soil_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_Soil_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_Soil_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_Water_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Water_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_Water_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_Water_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_w_Reflected_Wall_W_p_m2 + Radiation_Shortwave_Diffuse_w_Reflected_Wall_W_p_m2);
	//Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2 defined by direct and diffuse, to 0 or larger
	Radiation_Shortwave_Direct_Diffuse_Roof_UrbanCanyon_Absorbed_W_p_m2 =
		max(0.0, Radiation_Shortwave_Direct_Roof_W_p_m2 + Radiation_Shortwave_Diffuse_Roof_W_p_m2);

	//Radiation_Shortwave_Direct_Diffuse_TotalCanyon_Absorbed_W_p_m2 is Ratio_Road_to_Ground * Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2 + Ratio_Garden_to_Ground * Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2	+ Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2
	//Note: Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2 computed as weighted sum of Tree, SVeg, Soil, Water
	//Note: Radiation_Shortwave_Direct_Diffuse_TotalCanyon_Absorbed_W_p_m2 is not used
	Radiation_Shortwave_Direct_Diffuse_TotalCanyon_Absorbed_W_p_m2 = Ratio_Road_to_Ground * Radiation_Shortwave_Direct_Diffuse_Pavement_UrbanCanyon_Absorbed_W_p_m2 + Ratio_Garden_to_Ground * Radiation_Shortwave_Direct_Diffuse_Garden_UrbanCanyon_Absorbed_W_p_m2	+ Radiation_Shortwave_Direct_Diffuse_Wall_UrbanCanyon_Absorbed_W_p_m2;
}


//Consider refactor to include roof in temperature calculations if at height where appropriate for urban canopy
//Tair_AbsHumidity_LayerBalance function computes mesoscale air temperature and absolute humidity based on urban canopy air temperature, humidity and resistances
//Note: If reference station, then called from SimulationCoordinator::RunTemperatureSpatialModel and ...
//Note: ... iteration will find mesoscale air and humidity without numerical method
//Note: If local station, then called from HeatFluxCal::Tair_AbsHumidity_LayerBalance_NumericalMethod and 
//Note: ... iteration will find local grid canopy air and humidity using Ta_est_K
//Note: Based on Yang et al. (2013) there are 10 equations, (1), (2), (9), (10), (11), (12), (13), (19), (20), and (21) 
//Note: Based on Yang et al. (2013) there are 10 unknowns, H_total_W_p_m2, LE_total_W_p_m2, H1, H2, H3, LE1, LE2, LE3, Tair_localCell_K, Tdew_localCell_K.
//Note: All Energy Flux Values of H and LE computed below are adjusted in RescaleVariables_CoolAir function
void HeatFluxCal::Tair_AbsHumidity_LayerBalance(Inputs* input, DataOrganizer* organizer, WeatherProcessor* WeatherPro, DataFolder* folder, int timeStep)
{
	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions with sending solarCalculator
	HeatMetrics_Calc HeatMetricsCalc(input);

	//AtmPres_kPa = input->AtmPres_kPa[timeStep]
	double AtmPres_kPa = input->AtmPres_kPa[timeStep];
	//SpecificHeat_HumidAir_JpkgK = HeatMetricsCalc.SpecificHeat_HumidAir_JpkgK_Calc(Tair_localCell_K, Tdew_localCell_K, AtmPres_kPa)
	double SpecificHeat_HumidAir_JpkgK = HeatMetricsCalc.SpecificHeat_HumidAir_JpkgK_Calc(Tair_localCell_K, Tdew_localCell_K, AtmPres_kPa);
	//LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(Tair_localCell_K)
	double LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(Tair_localCell_K);


	//Tair_mesoScale_K (K), air temperature of shared upper mesoscale layer initialized to Tair_localCell_K (K), air temperature at urban canopy layer
	//Note: Tair_mesoScale_K (K) is updated in Tair_AbsHumidity_LayerBalance function using Eq 1 from Yang et al. (2013)
	//Note: If reference station, then Tair_localCell_K (K) defined by input time series,  
	//Note: If local station, then Tair_localCell_K defined by Ta_est_K (K) in Tair_AbsHumidity_LayerBalance_NumericalMethod
	Tair_mesoScale_K = Tair_localCell_K;

	//Tair_localCell_C (C) is converted from Tair_localCell_K (K) to degrees C by subtracting 273.15
	double Tair_localCell_C = Tair_localCell_K - 273.15;
	//Tdew_localCell_C (C) is converted from Tdew_localCell_K (K) to degrees C by subtracting 273.15
	double Tdew_localCell_C = Tdew_localCell_K - 273.15;
	//VaporPressure_Saturated_kPa (kPa) as Tetens' formula in Eq 4.2.2 Shuttleworth (1993), e_kPa=0.6108*exp[(17.27*T_C)/(237.3+T_C)]
	double VaporPressure_Saturated_kPa = 0.6108 * exp((17.27 * Tair_localCell_C) / (237.3 + Tair_localCell_C));
	//VaporPressure_Actual_kPa (kPa) as Tetens' formula in Eq 4.2.2 Shuttleworth (1993), e_kPa=0.6108*exp[(17.27*T_C)/(237.3+T_C)]
	double VaporPressure_Actual_kPa = 0.6108 * exp((17.27 * Tdew_localCell_C) / (237.3 + Tdew_localCell_C));

	//AbsHumidity_mesoScale_kg_p_m3 (kg/m3) is absolute humidity actual of mesoscale layer from Eq 4.10 of Stull (2017), AH = e/(Rv*T_K)
	//Note: Initialized to the absolute humidity at urban canopy layer and updated in Tair_AbsHumidity_LayerBalance function using Eq 2 from Yang et al. (2013)
	//Note: If reference station, then AbsHumidity_mesoScale_kg_p_m3 (kg/m3) defined by input time series or 
	//Note: If local station, then AbsHumidity_mesoScale_kg_p_m3 defined by Td_est_K (K) in Tair_AbsHumidity_LayerBalance_NumericalMethod
	//Note: Absolute humidity Eq 4.10 of Stull (2017) explicitly uses vapor pressure, e, and is equivalent to Eq 18 of Yang et al. (2013) as follows ...
	//Note: ... e in Eq 4.2.2 Shuttleworth (1992) and Eq 4.2 Stull (2017) equivalent: Tair_K - 35.85 = 237.3 + Tair_C | 17.27 * (Tair_K - 273.15) = 17.27 * Tair_C
	//Note: ... Coefficient 1.324 in Eq 18 is derived as 1.324 (kg/m3) = e0 / Rv = 0.6108 kPa / 4.61x10–1 kPa·K–1·m3·kg–1
	AbsHumidity_mesoScale_kg_p_m3 = VaporPressure_Actual_kPa / (Gas_constant_water_vapor_kPa_p_K_m3_kg * Tdew_localCell_K);
	//AerodynamicResistance_CanopyToMesoLayer function called to calculate the aerodynamic resistance factors
	resistanceVar->AerodynamicResistance_CanopyToMesoLayer(input, organizer, WeatherPro, folder, MapPixel_ID, WindSpeed_Reference_m_p_s, Height_LandCover_Average_m, Radiation_Shortwave_Direct_W_p_m2, Radiation_Shortwave_Diffuse_W_p_m2, timeStep);

	//AbsHumiditySat_localCell_kg_p_m3 (kg/m3) is absolute humidity saturated of local cell from Eq 4.10 of Stull (2017), AH = e/(Rv*T_K)
	AbsHumiditySat_localCell_kg_p_m3 = VaporPressure_Saturated_kPa / (Gas_constant_water_vapor_kPa_p_K_m3_kg * Tair_localCell_K);
	//AbsHumidityAct_localCell_kg_p_m3 (kg/m3) is absolute humidity actual of local cell from Eq 4.10 of Stull (2017), AH = e/(Rv*T_K)
	AbsHumidityAct_localCell_kg_p_m3 = VaporPressure_Actual_kPa / (Gas_constant_water_vapor_kPa_p_K_m3_kg * Tdew_localCell_K);

	//Gradient of saturated absolute humidity, given above Eq 18 in Yang et al (2013); attributed by Yang et al. (2013) to Lawrence, 2005.
	//Note: Tair_K - 35.85 = 237.3 + Tair_C
	AbsHumidityGradient = -AbsHumiditySat_localCell_kg_p_m3 * (-4169.56 + Tair_localCell_K) * (-0.308 + Tair_localCell_K) / Tair_localCell_K /
		(-35.85 + Tair_localCell_K) / (-35.85 + Tair_localCell_K);

	//If ImperviousCover_noTreeCanopy_frac > Constant_1E_negative4 then calculate ImpH_W_p_m2 and ImpLE_W_p_m2 for use in areas with ImperviousCover_noTreeCanopy_frac 
	if (ImperviousCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//Note: Derivation combines Eq 9 and 19 from Yang, Endreny, Nowak (2013), expand, combine ImpH_W_p_m2 terms on LHS, and isolate ImpH_W_p_m2 
		//Note: Eq 19 rImpH/rImpLE = 1 when rImpH = rImpLE; Eq 19 ImpC = 1 = (S/S_max)^(2/3).
		//1. ImpH_W_p_m2 = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - ImpLE_W_p_m2 = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - [(LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m + (LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * ImpH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)] * pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0)
		//2. ImpH_W_p_m2 = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m - pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * ImpH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)
		//3. ImpH_W_p_m2 + (pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * ImpH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m 
		//4. ImpH_W_p_m2 * ((pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1) = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m
		//5. ImpH_W_p_m2 = (ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) / ((pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1)

		//ImpH_W_p_m2 (W/m2) as impervious sensible heat derived above from Eq 9 and 20 of Yang et al. (2013) 
		//Note: H is function of terms comprising LE, and H must be computed before LE can be computed as function of H
		//Note: H can become a large negative value when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and ...
		//Note: ... Resistance_Aerodynamic_Impervious_s_p_m is small	 
		ImpH_W_p_m2 = (ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) /	((pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + 1);
		//Note: Below is ImpH_W_p_m2 (W/m2) as impervious sensible heat in optional simplified equation w/ ImpC:
		//Note: If ImpC = pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m then simplify above to:
		//ImpH_W_p_m2 = (ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - ImpC * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3)) / (ImpC * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m / SpecificHeat_DryAir_J_p_kg_K / Density_Air_kg_p_m3 + 1);

		//ImpLE_W_p_m2 (W/m2) is impervious latent heat in full equation based on Eq 19 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Impervious_s_p_m is small
		ImpLE_W_p_m2 = (LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * ImpH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m / (resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK)) * pow(Saturation_ImperviousStorage_frac, 2.0 / 3.0);
		//Note: Below is ImpLE_W_p_m2 (W/m2) in simplified energy balance Eq 9 of Yang et al. (2013)
		//ImpLE_W_p_m2 = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2 - ImpH_W_p_m2;
		//Note: Below is ImpLE_W_p_m2 (W/m2) as impervious latent heat in optional simplified equation w/ ImpC:
		//ImpLE_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * ImpH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Impervious_s_p_m / Density_Air_kg_p_m3 / SpecificHeat_DryAir_J_p_kg_K) * ImpC;
	}
	//Else If there is no impervious cover
	else {
		//Imp_AH_W_p_m2 (W/m2) is impervious fraction of AnthropogenicHeat_Flux_Total_Wpm2 on pixel
		Imp_AH_W_p_m2 = 0;
		//Set values to zero if there is no cover
		ImpH_W_p_m2 = 0.0; ImpLE_W_p_m2 = 0.0;
	}

	//If TreeCover_frac >  Constant_1E_negative4 cover threshold
	if (TreeCover_frac > Constant_1E_negative4) {
		//Note: Derivation combines Eq 9, 22 and 23 from Yang et al. (2013), expand, combine TreeH_W_p_m2 terms on LHS, and isolate TreeH_W_p_m2 
		//Note: Eq 22 & 23 rTreeH/rTreeLE = 1 when rTreeH = rTreeLE; Eq 22 (S/Smax)^(2/3)/Resistance_Aerodynamic_Tree_s_p_m = TreeCE; Eq 23 [1-(S/Smax)^(2/3)]/(resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) = TreeCT
		//1. TreeH_W_p_m2 = TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - TreeLE_W_p_m2 = TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m  / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) - (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m  / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0))
		//2. TreeH_W_p_m2 = TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m - (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) - pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) - (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)
		//3. TreeH_W_p_m2 + pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) = (TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - ((pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m)))
		//4. TreeH_W_p_m2 * (pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1) = (TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - ((pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m)))
		//5. TreeH_W_p_m2 = (TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - ((pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m))) / (pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1)

		//TreeH_W_p_m2 (W/m2) as tree sensible heat derived above from Eq 9, 22, and 23 of Yang et al. (2013); note LE embedded with RHS radiation terms
		//Note: H is function of terms comprising LE, and H must be computed before LE can be computed as function of H
		//Note: H can become a large negative value when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and ...
		//Note: ... Resistance_Aerodynamic_Impervious_s_p_m is small
		TreeH_W_p_m2 = (TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - ((pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK *	(AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m))) / (pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m /	(resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient *	resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + 1);
		//Note: Below is TreeH_W_p_m2 (W/m2) as tree sensible heat in optional simplified equation w/ TreeC:
		//Note: If TreeC = TreeCE + TreeCT = pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) / resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + 
		//Note: ... (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) then simplify above to:
		//TreeH_W_p_m2 = (TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2 - TreeC * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3)) / (TreeC * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1);

		//TreeLEE_W_p_m2 (W/m2) is tree evaporation latent heat in full equation based on Eq 22 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Impervious_s_p_m is small
		TreeLEE_W_p_m2 = LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) /resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * pow(Saturation_TreeStorage_frac, (2.0 / 3.0)) + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (resistanceVar->Resistance_Aerodynamic_Tree_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) * pow(Saturation_TreeStorage_frac, (2.0 / 3.0));
		//Note: Below is TreeLEE_W_p_m2 (W/m2) as tree evaporation latent heat in optional simplified equation w/ TreeCE:
		//TreeLEE_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient *	TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * TreeCE;

		//TreeLET_W_p_m2 (W/m2) is tree transpiration latent heat in full equation based on Eq 23 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Impervious_s_p_m is small
		//Note: Resistance_Surface_Tree_s_p_m (s/m) is TreeMinimumResistance_s_p_m for no impervious cover, and is increased linearly with impervious cover
		TreeLET_W_p_m2 = LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) /(resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0)) + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Tree_s_p_m + resistanceVar->Resistance_Surface_Tree_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) * (1 - pow(Saturation_TreeStorage_frac, 2.0 / 3.0));
		//Note: Below is TreeLET_W_p_m2 (W/m2) as tree transpiration latent heat in optional simplified equation w/ TreeCT:
		//TreeLET_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * TreeH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Tree_s_p_m / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * TreeCT;
		
		//If Flag_Scenario_CoolAir_LandCover, Flag_Scenario_CoolAir_NoTreeTranspiration & Scenario_CoolAir_NoTreeTranspiration_LandCover_Class are present
		//... and input->isReferenceStationPixel(folder->ParamDict["MapPixel_ID"]) is false and not a reference station
		if (input->SimulationScenarios.count("Flag_Scenario_CoolAir_LandCover") &&
			input->SimulationScenarios.count("Flag_Scenario_CoolAir_NoTreeTranspiration") &&
			input->SimulationScenarios.count("Scenario_CoolAir_NoTreeTranspiration_LandCover_Class") &&
			!input->isReferenceStationPixel(folder->ParamDict["MapPixel_ID"])) {

			//MapPixel_ID defined as folder->ParamDict["MapPixel_ID"]
			int MapPixel_ID = static_cast<int>(folder->ParamDict["MapPixel_ID"]);
			//Flag_IsAppropriateLandCover = input->IsAppropriateLandCoverForScenario(MapPixel_ID,Scenario_CoolAir_NoTreeTranspiration_LandCover_Class)
			//Note: Determine whether to suppress transpiration based on land cover rule
			bool Flag_IsAppropriateLandCover = input->IsAppropriateLandCoverForScenario(MapPixel_ID, input->SimulationScenarios["Scenario_CoolAir_NoTreeTranspiration_LandCover_Class"]);

			//If Flag_IsAppropriateLandCover == true then appropriate land cover for scenario
			if (Flag_IsAppropriateLandCover == true) {
				//If Flag_Scenario_CoolAir_LandCover is base case (<=1) and Flag_Scenario_CoolAir_NoTreeTranspiration is 1 then no Tree ET for base case
				if (input->SimulationScenarios["Flag_Scenario_CoolAir_LandCover"] <= 1 &&
					input->SimulationScenarios["Flag_Scenario_CoolAir_NoTreeTranspiration"] == 1) {
					//TreeLET_W_p_m2 = 0; set to zero to represent no Transpiration, shade only
					TreeLET_W_p_m2 = 0;
				}

				//If Flag_Scenario_CoolAir_LandCover is alternative case (>=2) and Flag_Scenario_CoolAir_NoTreeTranspiration is 2 then enter for alternative case
				if (input->SimulationScenarios["Flag_Scenario_CoolAir_LandCover"] >= 2 &&
					input->SimulationScenarios["Flag_Scenario_CoolAir_NoTreeTranspiration"] == 2) {

					//If TreeCover_frac > input->TreeCover_base_frac[MapPixel_ID] then only new tree cover has no Transpiration, shade only
					if (TreeCover_frac > input->TreeCover_base_frac[MapPixel_ID]) {
						//TreeLET_W_p_m2 = TreeLET_W_p_m2 * (input->TreeCover_base_frac[MapPixel_ID] / TreeCover_frac)
						//Note: Taking base fraction of TreeLET_W_p_m2, scaled to TC_base/TC_alternative, which removes ET from new TC
						TreeLET_W_p_m2 = TreeLET_W_p_m2 * (input->TreeCover_base_frac[MapPixel_ID] / TreeCover_frac);
					}
					//Else If TreeCover_frac < input->TreeCover_base_frac[MapPixel_ID] then all tree cover has no Transpiration, shade only
					else if (TreeCover_frac < input->TreeCover_base_frac[MapPixel_ID]) {
						//TreeLET_W_p_m2 = 0; set to zero to represent no Transpiration, shade only
						TreeLET_W_p_m2 = 0;
					}
				}
				//If Flag_Scenario_CoolAir_LandCover is alternative case (>=2) and Flag_Scenario_CoolAir_NoTreeTranspiration is 3 then enter for all cases
				if (input->SimulationScenarios["Flag_Scenario_CoolAir_LandCover"] >= 2 &&
					input->SimulationScenarios["Flag_Scenario_CoolAir_NoTreeTranspiration"] == 3) {
					//TreeLET_W_p_m2 = 0; set to zero to represent no Transpiration, shade only
					TreeLET_W_p_m2 = 0;
				}
			}
		}
		//End Scenarios
		
		//TreeLE_W_p_m2 (W/m2) Tree latent energy total as Eq 21 of Yang et al. (2013)
		TreeLE_W_p_m2 = TreeLEE_W_p_m2 + TreeLET_W_p_m2;
	}
	//Else If there is no tree cover
	else {
		//Tree_AH_W_p_m2 (W/m2) is tree fraction of AnthropogenicHeat_Flux_Total_Wpm2 on pixel
		Tree_AH_W_p_m2 = 0;
		//Set values to zero if there is no cover
		TreeH_W_p_m2 = 0.0; TreeLEE_W_p_m2 = 0.0; TreeLET_W_p_m2 = 0.0; TreeLE_W_p_m2 = 0.0;
	}

	//If ShortVegCover_noTreeCanopy_frac >  Constant_1E_negative4 cover threshold
	if (ShortVegCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//Note: Derivation combines Eq 9, 22 and 23 from Yang et al. (2013), expand, combine SVegH_W_p_m2 terms on LHS, and isolate SVegH_W_p_m2 
		//Note: Eq 22 & 23 rSVegH/rSVegLE = 1 when rSVegH = rSVegLE; Eq 22 (S/Smax)^(2/3)/Resistance_Aerodynamic_SVeg_s_p_m = SVegCE; Eq 23 [1-(S/Smax)^(2/3)]/(resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) = SVegCT
		//1. SVegH_W_p_m2 = SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - SVegLE_W_p_m2 = SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m  / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) - (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m  / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0))
		//2. SVegH_W_p_m2 = SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m - (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) - pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) - (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)
		//3. SVegH_W_p_m2 + pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) = (SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - ((pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m)))
		//4. SVegH_W_p_m2 * (pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1) = (SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - ((pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m)))
		//5. SVegH_W_p_m2 = (SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - ((pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m))) / (pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1)

		//SVegH_W_p_m2 (W/m2) as SVeg sensible heat derived above from Eq 9, 22, and 23 of Yang et al. (2013); note LE embedded with RHS radiation terms
		//Note: H is function of terms comprising LE, and H must be computed before LE can be computed as function of H
		//Note: H can become a large negative value when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and ...
		//Note: ... Resistance_Aerodynamic_Impervious_s_p_m is small
		SVegH_W_p_m2 = (SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - ((pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m))) / (pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m /	(resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + 1);
		//Note: Below is SVegH_W_p_m2 (W/m2) as SVeg sensible heat in optional simplified equation w/ SVegC:
		//Note: If SVegC = SVegCE + SVegCT = pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + 
		//Note: ... (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) then simplify above to:
		//SVegH_W_p_m2 = (SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2 - SVegC * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3)) / (SVegC * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1);

		//SVegLEE_W_p_m2 (W/m2) is SVeg evaporation latent heat in full equation based on Eq 22 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Impervious_s_p_m is small
		SVegLEE_W_p_m2 = LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m *	pow(Saturation_SVegStorage_frac, (2.0 / 3.0)) + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) * pow(Saturation_SVegStorage_frac, (2.0 / 3.0));
		//Note: Below is SVegLEE_W_p_m2 (W/m2) as SVeg evaporation latent heat in optional simplified equation w/ SVegCE:
		//SVegLEE_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient *	SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * SVegCE;

		//SVegLET_W_p_m2 (W/m2) is SVeg transpiration latent heat in full equation based on Eq 23 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Impervious_s_p_m is small
		SVegLET_W_p_m2 = LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0)) + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / ((resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m + resistanceVar->Resistance_Surface_SVeg_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) * (1 - pow(Saturation_SVegStorage_frac, 2.0 / 3.0));
		//Note: Below is SVegLET_W_p_m2 (W/m2) as SVeg transpiration latent heat in optional simplified equation w/ SVegCT:
		//SVegLET_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SVegH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_SVeg_s_p_m / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)) * SVegCT;

		//SVegLE_W_p_m2 (W/m2) SVeg latent energy total as Eq 21 of Yang et al. (2013)
		SVegLE_W_p_m2 = SVegLEE_W_p_m2 + SVegLET_W_p_m2;
	}
	//Else If there is no short vegetation cover
	else {
		//SVeg_AH_W_p_m2 (W/m2) is short vegetation fraction of AnthropogenicHeat_Flux_Total_Wpm2 on pixel
		SVeg_AH_W_p_m2 = 0;
		//Set values to zero if there is no cover
		SVegH_W_p_m2 = 0.0; SVegLEE_W_p_m2 = 0.0; SVegLET_W_p_m2 = 0.0; SVegLE_W_p_m2 = 0.0;
	}

	//If SoilCover_noTreeCanopy_frac >  Constant_1E_negative4 cover threshold
	//Consider Refactor to allow SoilH and SoilLE to represent surface layer when ponded water is present
	if (SoilCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//Note: Derivation combines Eq 9 and 20 from Yang, Endreny, Nowak (2013), expand, combine SoilH_W_p_m2 terms on LHS, and isolate SoilH_W_p_m2 
		//Note: Eq 20 rSoilH/rSoilLE = 1 when rSoilH = rSoilLE; Eq 20 (1-D/Dmax) = Saturation_SoilStorage_frac
		//1. SoilH_W_p_m2 = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - SoilLE_W_p_m2 = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - [(LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) + (LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SoilH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m) / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)] * Saturation_SoilStorage_frac
		//2. SoilH_W_p_m2 = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - Saturation_SoilStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) - Saturation_SoilStorage_frac * (LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SoilH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m) / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)
		//3. SoilH_W_p_m2 + (LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * SoilH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m) / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - Saturation_SoilStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) 
		//4. SoilH_W_p_m2 * ((LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m) / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1) = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - Saturation_SoilStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m)
		//5. SoilH_W_p_m2 = (SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - Saturation_SoilStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m)) / ((LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m) / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1)

		//SoilH_W_p_m2 (W/m2) as soil sensible heat derived above from Eq 9 and 20 of Yang et al. (2013); note LE embedded with RHS radiation terms
		//Note: H is function of terms comprising LE, and H must be computed before LE can be computed as function of H
		//Note: H can become a large negative value when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Soil_s_p_m is small

		SoilH_W_p_m2 = (SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - Saturation_SoilStorage_frac * LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m)) / ((Saturation_SoilStorage_frac * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m) / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + 1);
		//Note: Below is SoilH_W_p_m2 (W/m2) as soil sensible heat in optional simplified equation w/ SoilC:
		//Note: If SoilC = Saturation_SoilStorage_frac / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) then simplify above to:
		//SoilH_W_p_m2 = (SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - SoilC * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3)) / (SoilC * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m / (SpecificHeat_DryAir_J_p_kg_K * Density_Air_kg_p_m3) + 1);

		//SoilLE_W_p_m2 (W/m2) is soil latent heat in full equation based on Eq 20 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Soil_s_p_m is small
		SoilLE_W_p_m2 = (LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / (resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * SoilH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m / ((resistanceVar->Resistance_Aerodynamic_Soil_s_p_m + resistanceVar->Resistance_Surface_Soil_s_p_m) * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK)) * Saturation_SoilStorage_frac;
		//Note: Below is SoilLE_W_p_m2 (W/m2) in simplified energy balance Eq 9 of Yang et al. (2013)
		//SoilLE_W_p_m2 = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2 - SoilH_W_p_m2;
		//Note: Below is SoilLE_W_p_m2 (W/m2) as soil latent heat in optional simplified equation w/ SoilC:
		//SoilLE_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient*SoilH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Soil_s_p_m / Density_Air_kg_p_m3 / SpecificHeat_DryAir_J_p_kg_K) * SoilC;
	}
	//Else If there is no soil cover
	else {
		//Soil_AH_W_p_m2 (W/m2) is soil fraction of AnthropogenicHeat_Flux_Total_Wpm2 on pixel
		Soil_AH_W_p_m2 = 0;
		//Set values to zero if there is no cover
		SoilH_W_p_m2 = 0.0; SoilLE_W_p_m2 = 0.0;
	}

	//If WaterCover_noTreeCanopy_frac >  Constant_1E_negative4 cover threshold
	if (WaterCover_noTreeCanopy_frac > Constant_1E_negative4) {
		//Note: Derivation combines Eq 9 and 20 (modified for water by removing rSoilLE from 1st RHS term) from Yang, Endreny, Nowak (2013), expand, combine WaterH_W_p_m2 terms on LHS, and isolate WaterH_W_p_m2 
		//Note: Eq 20 rWaterH/rWaterLE = 1 when rWaterH = rWaterLE; Eq 20 WaterC = 1 = (1 - D/D_max) = (1 - 0/D_max); D = deficit = 0 for water.
		//1. WaterH_W_p_m2 = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - WaterLE_W_p_m2 = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - [(LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m + (LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * WaterH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Water_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)] * Saturation_WaterStorage_frac
		//2. WaterH_W_p_m2 = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m - Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * WaterH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Water_s_p_m / (resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K)
		//3. WaterH_W_p_m2 + (Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * WaterH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Water_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m 
		//4. WaterH_W_p_m2 * ((Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Water_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1) = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m
		//5. WaterH_W_p_m2 = (WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m) / ((Saturation_WaterStorage_frac * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Water_s_p_m) / (resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K) + 1)

		//WaterH_W_p_m2 (W/m2) as water sensible heat derived above from Eq 9 and 20 of Yang et al. (2013); note LE embedded with RHS radiation terms
		WaterH_W_p_m2 = (WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - Saturation_WaterStorage_frac * LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m) / ((Saturation_WaterStorage_frac * LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Water_s_p_m) /	(resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) + 1);
		//Note: Below is WaterH_W_p_m2 (W/m2) as water sensible heat in optional simplified equation w/ WaterC:
		//Note: If WaterC = Saturation_WaterStorage_frac / resistanceVar->Resistance_Aerodynamic_Water_s_p_m then simplify above to:
		//WaterH_W_p_m2 = (WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - WaterC * LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3)) / (WaterC * LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * resistanceVar->Resistance_Aerodynamic_Water_s_p_m / SpecificHeat_DryAir_J_p_kg_K / Density_Air_kg_p_m3 + 1);

		//WaterLE_W_p_m2 (W/m2) is water latent heat in full equation based on Eq 19 of Yang et al. (2013)
		//Note: LE is function of H, so H must be computed prior to LE
		//Note: LE can become large when AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3 is large and Resistance_Aerodynamic_Impervious_s_p_m is small
		WaterLE_W_p_m2 = (LatentHeat_Vaporization_JpkgK * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) / resistanceVar->Resistance_Aerodynamic_Water_s_p_m + LatentHeat_Vaporization_JpkgK * AbsHumidityGradient * WaterH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Water_s_p_m / (resistanceVar->Resistance_Aerodynamic_Water_s_p_m * Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK)) * Saturation_WaterStorage_frac;
		//Note: Below is WaterLE_W_p_m2 (W/m2) in simplified energy balance Eq 9 of Yang et al. (2013)
		//WaterLE_W_p_m2 = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2 - WaterH_W_p_m2;
		//Note: Below is WaterLE_W_p_m2 (W/m2) as water latent heat in optional simplified equation w/ WaterC:
		//WaterLE_W_p_m2 = (LatentHeatVaporization_Water_J_p_kg * (AbsHumiditySat_localCell_kg_p_m3 - AbsHumidityAct_localCell_kg_p_m3) + LatentHeatVaporization_Water_J_p_kg * AbsHumidityGradient * WaterH_W_p_m2 * resistanceVar->Resistance_Aerodynamic_Water_s_p_m / Density_Air_kg_p_m3 / SpecificHeat_DryAir_J_p_kg_K) * WaterC;

	}
	//Else If there is no water cover
	else {
		//Water_AH_W_p_m2 (W/m2) is water fraction of AnthropogenicHeat_Flux_Total_Wpm2 on pixel
		Water_AH_W_p_m2 = 0;
		//Set values to zero if there is no cover
		WaterH_W_p_m2 = 0.0; WaterLE_W_p_m2 = 0.0;
	}

	//H_total_W_p_m2 (W/m2) is Eq 12 of Yang et al. (2013), total sensible heat flux (W/m^2), sum of fractional components from each cover type
	//Note: Theory of Yang et al. (2013) above Eq 12, H_localAir_to_mesoAir = H_ground_to_localAir in steady state to maintain energy balance
	H_total_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * ImpH_W_p_m2 + TreeCover_frac * TreeH_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVegH_W_p_m2 + SoilCover_noTreeCanopy_frac * SoilH_W_p_m2 + WaterCover_noTreeCanopy_frac * WaterH_W_p_m2;
	//LE_total_W_p_m2 (W/m2) is Eq 13 of Yang et al. (2013), total latent heat flux (W/m^2), sum of fractional components from each cover type
	//Note: Theory of Yang et al. (2013) above Eq 12, LE_localAir_to_mesoAir = LE_ground_to_localAir in steady state to maintain energy balance
	LE_total_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * ImpLE_W_p_m2 + TreeCover_frac * TreeLE_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVegLE_W_p_m2 + SoilCover_noTreeCanopy_frac * SoilLE_W_p_m2 + WaterCover_noTreeCanopy_frac * WaterLE_W_p_m2;

	//NR_total_W_p_m2 (W/m2) is total net radiation (W/m^2), sum of fractional components from each cover type net radiation
	NR_total_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * ImpNR_W_p_m2 + TreeCover_frac * TreeNR_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVegNR_W_p_m2 + SoilCover_noTreeCanopy_frac * SoilNR_W_p_m2 + WaterCover_noTreeCanopy_frac * WaterNR_W_p_m2;
	//DeltaQ_W_p_m2 (W/m2) is total ground heat flux (W/m^2), sum of fractional components from each cover type ground heat flux
	DeltaQ_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * ImpDeltaQ_W_p_m2 + TreeCover_frac * TreeDeltaQ_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVegDeltaQ_W_p_m2 + SoilCover_noTreeCanopy_frac * SoilDeltaQ_W_p_m2 + WaterCover_noTreeCanopy_frac * WaterDeltaQ_W_p_m2;
	//AH_total_W_p_m2 (W/m2) is total anthropogenic radiation (W/m^2), sum of fractional components from each cover type anthropogenic radiation
	AH_total_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * Imp_AH_W_p_m2 + TreeCover_frac * Tree_AH_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVeg_AH_W_p_m2 + SoilCover_noTreeCanopy_frac * Soil_AH_W_p_m2 + WaterCover_noTreeCanopy_frac * Water_AH_W_p_m2;

	//Tair_mesoScale_K (K) is based on rearranged form of Eq 1 from Yang et al. (2013), where Tb = Ta - (H*ra)/(rho*Cp), ...
	//Note: ... Tb = Tair_mesoScale_K, Ta = Tair_localCell_K, ra = Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m, ...
	//Note: ... H = H_total_W_p_m2 (W/m2), rho = Density_Air_kg_p_m3, Cp = SpecificHeat_DryAir_J_p_kg_K  
	//Note: Unit balance right hand side = (W/m2) * (s/m) / [(kg/m3) * (J/kg/K)] = (J/s/m2) * (s/m) / (J/m3/K) = (J/m3) / (J/m3/K) = K
	Tair_mesoScale_K = Tair_localCell_K - (H_total_W_p_m2 * resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m) / (Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK);
	
	//AbsHumidity_mesoScale_kg_p_m3 (kg/m^3) is based on rearranged form of Eq 2 from Yang et al. (2013), where AHb = AHa - (LE*ra)/Lambda, ...
	//Note: ... AHb = AbsHumidity_mesoScale_kg_p_m3, AHa = AbsHumidityAct_localCell_kg_p_m3, LE = LE_total_W_p_m2, ...
	//Note: ... Lambda = LatentHeatVaporization_Water_J_p_kg, ra = Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m
	//Note: Unit balance right hand side = (W/m2) * (s/m) / (J/kg) = (J/s/m2) * (s/m) / (J/kg) = (J/m3) / (J/kg) = kg/m3
	AbsHumidity_mesoScale_kg_p_m3 = AbsHumidityAct_localCell_kg_p_m3 - (LE_total_W_p_m2 * resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m) / LatentHeat_Vaporization_JpkgK;

	//Note: All Energy Flux Values of H and LE are adjusted in RescaleVariables_CoolAir function
}

//Tair_AbsHumidity_LayerBalance_NumericalMethod function uses numerical method and calls to Tair_AbsHumidity_LayerBalance function
//Note: Based on Levenberg–Marquardt algorithm (LMA or LM) minimizes error in least squares curve fitting, ...
//Note: ... which mixes Gauss–Newton algorithm and the method of gradient descent
//Note: Called from SimulationCoordinator::RunTemperatureSpatialModel function for non-reference station grids
//Note: LMA for loop is set to default value of Iteration_Maximum, typically set to 40, but 
//Note: ... will break from for loop to save computational time if criteria are satisified
void HeatFluxCal::Tair_AbsHumidity_LayerBalance_NumericalMethod(Inputs* input, DataOrganizer* organizer, WeatherProcessor* WeatherPro, DataFolder* folder, int timeStep, int Iteration_Maximum) {

	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//Tb_obs_K (K) and eb_obs_kgpm3 (kg/m3) are mesoscale air temperature and absolute humidity at start of time step
	//Note: If multiple station is true then this is for pixel, else it is from reference station
	double Tb_obs_K, eb_obs_kgpm3;
	//Ta_est_K (K) and Td_est_K (K) are initial estimates of local air and dew point temperatures, initially set to ref station
	double Ta_est_K, Td_est_K;
	//Tair_mesoScale_weighted_K will contain a weighted value from multiple stations 
	double Tair_mesoScale_weighted_K = 0;
	//AbsHumidity_mesoScale_weighted_kg_p_m3 will contain a weighted value from multiple stations 
	double AbsHumidity_mesoScale_weighted_kg_p_m3 = 0;
	//Tair_weighted_K will contain a weighted value from multiple stations 
	double Tair_weighted_K = 0;
	//Tdew_weighted_K will contain a weighted value from multiple stations 
	double Tdew_weighted_K = 0;
	//keyname_Tair_F and keyname_Tdew_F initialized
	string keyname_Tair_F, keyname_Tdew_F;

	//If Flag_MultipleStations equals 1 then create WeatherPro-> weighted averages for mesoscale and canopy layer temperatures and humidity
	//Note: WeatherProcessor was used to lapse rate adjust the following variables in this contion so they work from the same relative elevation: ...
	//Note: ... Tair_mesoScale_weighted_K (K), AbsHumidity_mesoScale_weighted_kg_p_m3 (kg/m3), Tair_weighted_K (K) and Tdew_weighted_K (K)
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1) {
		//Terms are defined by weather station weighted averages at folder
		//Tb_obs_K (K) is the weighted mesoscale air temperature,Tair_mesoScale_weighted_K (K)
		Tb_obs_K = WeatherPro->Tair_mesoScale_weighted_K;
		//eb_obs_kgpm3 (kg/m3) is the weighted mesoscale air absolute humidity, AbsHumidity_mesoScale_weighted_kg_p_m3 (kg/m3)
		eb_obs_kgpm3 = WeatherPro->AbsHumidity_mesoScale_weighted_kg_p_m3;
		//Ta_est_K (K) is the weighted canopy air temperature, Tair_weighted_K (K)
		Ta_est_K = WeatherPro->Tair_weighted_K;
		//Td_est_K (K) is weighted canopy dew point temperature, Tdew_weighted_K (K)
		Td_est_K = WeatherPro->Tdew_weighted_K;
	}
	//Else If Flag_MultipleStations not equal to 1 then enter
	else {
		//Ref_MapPixel_ID = input->MapPixel_ReferenceStation_IDs[0]; Ref_MapPixel_ID updated to 1st index when Flag_MultipleStations = false
		int Ref_MapPixel_ID = input->MapPixel_ReferenceStation_IDs[0];
		//DataFolder* ReferenceFolder = organizer->DataDrawers[Ref_MapPixel_ID][0]; ReferenceFolder updated
		DataFolder* ReferenceFolder = organizer->DataDrawers[Ref_MapPixel_ID][0];

		//Terms are defined by weather station values at ReferenceFolder
		//Tb_obs_K (K) is the calculated mesoscale air temperature, from CollectVariables_CoolAir function
		Tb_obs_K = ReferenceFolder->VarDict["Tair_mesoScale_K"];
		//eb_obs_kgpm3 (kg/m3) is the calculated mesoscale air absolute humidity, from CollectVariables_CoolAir function
		eb_obs_kgpm3 = ReferenceFolder->VarDict["AbsHumidity_mesoScale_kg_p_m3"];
		//Ta_est_K (K) is the initial value of the estimated air temperature of a folder
		Ta_est_K = ReferenceFolder->VarDict["Tair_K"];
		//Td_est_K (K) is the initial value of the estimated dew point temperature of a folder
		Td_est_K = ReferenceFolder->VarDict["Tdew_K"];
	}

	//If Flag_converged = false then alert user
	if (Tb_obs_K == 0 || Ta_est_K == 0 || Td_est_K == 0 ||
		isnan(Tb_obs_K) || isnan(Ta_est_K) || isnan(Td_est_K)) {
		cout << "Warning: The i-Tree Cool Air model identified an absolute zero Kelvin or Nan air temperature." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Explanation: This condition suggests errors with input maps, parameters, calculated resistances, etc.." << endl;
		cout << "  This occurred at timestep " << timeStep << " and date hour " << input->SimulationDate_GDH[timeStep] << endl;
		cout << "  This occurred at MapPixel_ID " << MapPixel_ID << endl;
		cout << "  Tair_mesoScale_K " << Tb_obs_K << endl;
		cout << "  Tair_K " << Ta_est_K << endl;
		cout << "  Tdew_K " << Td_est_K << endl;
		cout << "  LandCover_Class " << input->LandCover_NLCD_Class[MapPixel_ID] << endl;
		cout << "  TreeCover_frac " << input->TreeCover_frac[MapPixel_ID] << endl;
		cout << "  ImperviousCover_frac " << input->ImperviousCover_frac[MapPixel_ID] << endl;
		cout << "  DEM_m " << input->Elevation_DEM_m[MapPixel_ID] << endl;
		cout << "  WindSpeed_mps " << WindSpeed_Reference_m_p_s << endl;
		//Note: Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m is nan for non reference station pixels until call to Tair_AbsHumidity_LayerBalance 
		cout << "Explanation: Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m " << resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m << endl;
		cout << "Explanation: AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 " << input->AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] << endl;
		cout << "Correction: Improve inputs or modify code to allow for such large Tair variation." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}

	//alamda is the lambda term in LM algorithm
	double alamda = 0.1;
	//w1 is the weight for eb air absolute humidity
	double w1 = 10000;
	//W is the weighting matrix
	double W[2][2] = { 0 };
	double J[2][2] = { 0 };
	double H[2][2] = { 0 };
	double H_lm[2][2] = { 0 };

	//Epsilon_ErrorThreshold is initialized to zero
	double Epsilon_ErrorThreshold = 0.0;
	double EstimateError_mesoScale_Tair_K = 0.0;
	double EstimateError_mesoScale_AbsHumidity_kg_p_m3 = 0.0;

	//Weighting matrix is assigned values of 1, 0, 0, and w
	W[0][0] = 1;
	W[0][1] = 0;
	W[1][0] = 0;
	W[1][1] = w1;

	int updateJ = 1;
	//dTa is incremental change in air temperature
	double dTa = 0.01;
	//dTd is incremental change in dew point temperature
	double dTd = 0.01;
	//Iteration_Counter initialized to zero
	int Iteration_Counter = 0;
	//Stability_Counter initialized to zero
	int Stability_Counter = 0;

	//Initialize variabels for numerical method
	double d_lm1, d_lm2, e_lm, Tb_est_lm, eb_est_lm;


	//For Loop through Iteration_Maximum, which is HydroPlusConfig.xml parameter IterationCount
	//Note: Break from for loop is in place below if stability and temperature closure criteria are satisified
	for (Iteration_Counter; Iteration_Counter < Iteration_Maximum; Iteration_Counter++) {
		//If updateJ flag equals 1 then enter calculations, considered LMA conditional statement
		//Note: Make 3 calls to Tair_AbsHumidity_LayerBalance function with different sets of Tair and Tdew 
		if (updateJ == 1) {

			//Tair_localCell_K (K) redefined by Ta_est_K (K), air temperature estimated at start of LM algorithm
			//Note: For Flag_MultipleStations = 1, this is not a redefinition
			Tair_localCell_K = Ta_est_K;
			//Tdew_localCell_K (K) redefined by Td_est_K (K), dew point temperature estimated at start of LM algorithm
			//Note: For Flag_MultipleStations = 1, this is not a redefinition
			Tdew_localCell_K = Td_est_K;

			//Tair_AbsHumidity_LayerBalance function computes Tair_mesoScale_K (K) & AbsHumidity_mesoScale_kg_p_m3 (kg/m3) 
			//Note: The Tair_AbsHumidity_LayerBalance_NumericalMethod function calls this Tair_AbsHumidity_LayerBalance function ...
			//Note: ... until there is closure on the error term or the interation maximum is exceeded
			Tair_AbsHumidity_LayerBalance(input, organizer, WeatherPro, folder, timeStep);

			//Tb0 (K) is mesoScale air temperature estimate updated after each call to Tair_AbsHumidity_LayerBalance
			double Tb0 = Tair_mesoScale_K;
			//eb0 (kg/m3) is updated mesoScale absolute humidity estimate updated after each call to Tair_AbsHumidity_LayerBalance
			double eb0 = AbsHumidity_mesoScale_kg_p_m3;

			//Tair_localCell_K (K) redefined as sum of Ta_est_K + dTa
			Tair_localCell_K = Ta_est_K + dTa;

			//Tair_AbsHumidity_LayerBalance function computes Tair_mesoScale_K (K) & AbsHumidity_mesoScale_kg_p_m3 (kg/m3) 
			//Note: The Tair_AbsHumidity_LayerBalance_NumericalMethod function calls this Tair_AbsHumidity_LayerBalance function ...
			//Note: ... until there is closure on the error term or the interation maximum is exceeded
			Tair_AbsHumidity_LayerBalance(input, organizer, WeatherPro, folder, timeStep);

			//Tb1 (K) is mesoScale air temperature estimate updated after each call to Tair_AbsHumidity_LayerBalance
			double Tb1 = Tair_mesoScale_K;
			//eb1 (kg/m3) is updated mesoScale absolute humidity estimate updated after each call to Tair_AbsHumidity_LayerBalance
			double eb1 = AbsHumidity_mesoScale_kg_p_m3;

			//Tair_localCell_K (K) redefined as sum of Ta_est_K
			Tair_localCell_K = Ta_est_K;
			Tdew_localCell_K = Td_est_K + dTd;

			//Tair_AbsHumidity_LayerBalance function computes Tair_mesoScale_K (K) & AbsHumidity_mesoScale_kg_p_m3 (kg/m3) 
			//Note: The Tair_AbsHumidity_LayerBalance_NumericalMethod function calls this Tair_AbsHumidity_LayerBalance function ...
			//Note: ... until there is closure on the error term or the interation maximum is exceeded
			Tair_AbsHumidity_LayerBalance(input, organizer, WeatherPro, folder, timeStep);

			//Tb2 (K) is mesoScale air temperature estimate updated after each call to Tair_AbsHumidity_LayerBalance
			double Tb2 = Tair_mesoScale_K;
			//eb2 (kg/m3) is updated mesoScale absolute humidity estimate updated after each call to Tair_AbsHumidity_LayerBalance
			double eb2 = AbsHumidity_mesoScale_kg_p_m3;

			//Compute Jacobian matrix with three sets of mesoscale air temperature and absolute humidity actual values
			//J[0][0] is gradient of (Tb1 - Tb0) / dTa 
			J[0][0] = (Tb1 - Tb0) / dTa;
			//J[0][0] is gradient of (Tb2 - Tb0) / dTa 
			J[0][1] = (Tb2 - Tb0) / dTa;
			//J[0][0] is gradient of (eb1 - eb0) / dTd 
			J[1][0] = (eb1 - eb0) / dTd;
			//J[1][1] is gradient of (eb2 - eb0) / dTd 
			J[1][1] = (eb2 - eb0) / dTd;

			//EstimateError_mesoScale_Tair_K is computed as difference of mesoscale air temperatures Tb_obs_K - Tb0
			EstimateError_mesoScale_Tair_K = Tb_obs_K - Tb0;
			//EstimateError_mesoScale_AbsHumidity_kg_p_m3 is computed as difference of mesoscale absolute humidities (eb_obs_kgpm3 - eb0) weighted by w1
			EstimateError_mesoScale_AbsHumidity_kg_p_m3 = (eb_obs_kgpm3 - eb0) * w1;

			//Combine Jacobian matrices
			H[0][0] = J[0][0] * J[0][0] + J[1][0] * J[1][0] * w1;
			H[0][1] = J[0][0] * J[0][1] + J[1][0] * J[1][1] * w1;
			H[1][0] = J[0][1] * J[0][0] + J[1][1] * J[1][0] * w1;
			H[1][1] = J[0][1] * J[0][1] + J[1][1] * J[1][1] * w1;

			//If Iteration_Counter equals zero then compute initial Epsilon_ErrorThreshold 
			//Note: Epsilon_ErrorThreshold is re-computed below when e_lm < Epsilon_ErrorThreshold
			if (Iteration_Counter == 0) {
				//Epsilon_ErrorThreshold is sum of squares of EstimateError_mesoScale_Tair_K and EstimateError_mesoScale_AbsHumidity_kg_p_m3, ensuring it is positive
				Epsilon_ErrorThreshold = EstimateError_mesoScale_Tair_K * EstimateError_mesoScale_Tair_K +					EstimateError_mesoScale_AbsHumidity_kg_p_m3 * EstimateError_mesoScale_AbsHumidity_kg_p_m3;
			}
		}

		//H_lm[0][0] increased by alamda
		H_lm[0][0] = H[0][0] + alamda;
		H_lm[0][1] = H[0][1];
		H_lm[1][0] = H[1][0];
		//H_lm[1][1] increased by alamda
		H_lm[1][1] = H[1][1] + alamda;

		double bb = 1 / (H_lm[0][0] * H_lm[1][1] - H_lm[0][1] * H_lm[1][0]);
		//dp1 (K) is incremental update in air temperature based on LM algorithm 
		double dp1 = bb * (H_lm[1][1] * (J[0][0] * EstimateError_mesoScale_Tair_K + J[1][0] * EstimateError_mesoScale_AbsHumidity_kg_p_m3) -
			H_lm[0][1] * (J[0][1] * EstimateError_mesoScale_Tair_K + J[1][1] * EstimateError_mesoScale_AbsHumidity_kg_p_m3));
		//dp2 (K) is incremental update in dew point temperature based on LM algorithm 
		double dp2 = bb * (-H_lm[1][0] * (J[0][0] * EstimateError_mesoScale_Tair_K + J[1][0] * EstimateError_mesoScale_AbsHumidity_kg_p_m3) + H_lm[0][0] * (J[0][1] * EstimateError_mesoScale_Tair_K + J[1][1] * EstimateError_mesoScale_AbsHumidity_kg_p_m3));

		//Ta_lm (K) is updated estimate of air temperature based on LM algorithm, summing Ta_est_K (K) and dp1
		double Ta_lm = Ta_est_K + dp1;
		//Td_lm (K) is updated estimate of dew point temperature based on LM algorithm, summing Td_est_K (K) and dp2
		double Td_lm = Td_est_K + dp2;

		//Ta_est_Prior and Td_est_Prior archived for comparison
		double Ta_est_Prior = Ta_est_K;
		double Td_est_Prior = Td_est_K;

		//Tair_localCell_K (K) redefined by Ta_lm (K), air temperature estimated with LM algorithm
		Tair_localCell_K = Ta_lm;
		//Tdew_localCell_K (K) redefined by Ta_lm (K), dew point temperature estimated with LM algorithm
		Tdew_localCell_K = Td_lm;

		//Tair_AbsHumidity_LayerBalance function computes Tair_mesoScale_K (K) & AbsHumidity_mesoScale_kg_p_m3 (kg/m3) 
		//Note: The Tair_AbsHumidity_LayerBalance_NumericalMethod function calls this Tair_AbsHumidity_LayerBalance function ...
		//Note: ... until there is closure on the error term or the interation maximum is exceeded
		Tair_AbsHumidity_LayerBalance(input, organizer, WeatherPro, folder, timeStep);

		//Tb_est_lm (K) defined as Tair_mesoScale_K (K)
		Tb_est_lm = Tair_mesoScale_K;
		//eb_est_lm (K) defined as AbsHumidity_mesoScale_kg_p_m3 (K)
		eb_est_lm = AbsHumidity_mesoScale_kg_p_m3;

		//d_lm1 (K) is estimation error for mesoscale layer Tb_obs_K based on LM algorithm
		d_lm1 = Tb_obs_K - Tb_est_lm;
		//d_lm2 (K) is estimation error for mesoscale layer eb_obs_kgpm3 based on LM algorithm
		d_lm2 = (eb_obs_kgpm3 - eb_est_lm) * w1;
		//e_lm is total error term used below in LM algorithm
		e_lm = d_lm1 * d_lm1 + d_lm2 * d_lm2;

		//If e_lm less than Epsilon_ErrorThreshold then decrease alamda and prepare for entry into LMA if conditional above
		//Note: Conditional to find the local minimum by decreasing alamda
		if (e_lm < Epsilon_ErrorThreshold) {
			//alamda decreased by factor of 10
			alamda = alamda / 10;
			//Ta_est_K (K) redefined to Ta_lm (K)
			Ta_est_K = Ta_lm;
			//Td_est_K (K) redefined to Td_lm (K)
			Td_est_K = Td_lm;
			//Epsilon_ErrorThreshold redefined to e_lm
			Epsilon_ErrorThreshold = e_lm;
			//updateJ set to 1
			updateJ = 1;
			//Stability_Counter reset to 0 when algorithm re-enters updateJ = 1 condition
			Stability_Counter = 0;
		}
		//Else If e_lm greater than or equals Epsilon_ErrorThreshold then increase alamda and prepare for no entry into LMA if conditional above
		//Note: Conditional to find the local minimum by increasing alamda
		//Note: Optionally If (EstimateError_mesoScale_Tair_K > (Temperature_ErrorThreshold_K * 10)) force Ta_est_K to satisfy Tb_obs_K or Td_lm
		//Note: Ta_est_K (K) is based on rearranged form of Eq 1 from Yang et al. (2013), where Ta = Ta + (H*ra)/(rho*Cp), ...
		//Note: ... Tb = Tair_mesoScale_K or Tb_obs_K, Ta = Tair_localCell_K or Ta_est_K, ra = Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m, ...
		//Note: ... H = H_total_W_p_m2 (W/m2), rho = Density_Air_kg_p_m3, Cp = SpecificHeat_DryAir_J_p_kg_K  
		//Note: Unit balance right hand side = (W/m2) * (s/m) / [(kg/m3) * (J/kg/K)] = (J/s/m2) * (s/m) / (J/m3/K) = (J/m3) / (J/m3/K) = K
		//Note: Ta_est_K = Tb_obs_K + (H_total_W_p_m2 * resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m) / (Density_Air_kg_p_m3 * SpecificHeat_DryAir_J_p_kg_K);
		else {
			//alamda increased by factor of 10
			alamda = alamda * 10;
			//updateJ set to 0
			updateJ = 0;
			//Stability_Counter increased by 1 when updateJ = 0, but Stability_Counter resets to 0 if LMA re-enters update = 1
			Stability_Counter = Stability_Counter + 1;
		}

		//ErrorEstimate_canopyLayer_Tair_K (K) computes the absolute difference between the prior and current estimate of canopy layer local air temperature
		double ErrorEstimate_canopyLayer_Tair_K = abs(Ta_est_Prior - Ta_lm);
		//ErrorEstimate_canopyLayer_Tdew_K (K) computes the absolute difference between the prior and current estimate of canopy layer local dew point temperature
		double ErrorEstimate_canopyLayer_Tdew_K = abs(Td_est_Prior - Td_lm);
		//If Stability_Counter > 2, and ErrorEstimate_canopyLayer_Tair_K and ErrorEstimate_canopyLayer_Tdew_K estimates are less than Temperature_ErrorThreshold_K
		//Note: Criteria 1 Stability_Counter > 2 is used due to the LM algorithm cycling between updateJ = 1 and 0 to avoid local minima in the least squares fit
		//Note: Criteria 2 ErrorEstimate_canopyLayer_Tair_K < Temperature_ErrorThreshold_K ensures air temperatures are within 0.001 K, and share the same tenth of a degree
		//Note: Criteria 3 ErrorEstimate_canopyLayer_Tdew_K < Temperature_ErrorThreshold_K ensures dew point temperatures are within 0.001 K, and share the same tenth of a degree
		//Note: Consider Refactor to allow HydroPlusConfig.xml to set Temperature_ErrorThreshold_K value
		if (Stability_Counter > 2 && ErrorEstimate_canopyLayer_Tair_K < Temperature_ErrorThreshold_K && ErrorEstimate_canopyLayer_Tdew_K < Temperature_ErrorThreshold_K) {
			//break function leaves for loop
			break;
		}
	}
}

//AdiabaticLapseRates function will adjust air and dew point temperatures based elevation, using adiabatic lapse rate 
//Note: Even if Flag_MultipleStations=1 and WeatherProcessor lapse rate adjusted Tair_weighted_K and Tair_mesoScale_K, this adjustment is needed 
//Note: Explicit testing shows HeatFluxCal::AdiabaticLapseRates is required when flag_MultipleStation=1 to lapse rate adjust Tair_K and Tdew_K.
void HeatFluxCal::AdiabaticLapseRates(Inputs* input, DataFolder* folder, int timeStep, WeatherProcessor* WeatherPro)
{

	//Rationale for use of adiabatic lapse rate adjustments to air temperature:
	//Note: Function calculates the effect of elevation difference with the reference station on local cell air temperature above ground
	//Note: Function modifies Stull (2000) theory of cooling by dry adiabatic lapse rate (ALR) with environmental lapse rate (ELR) of ~5.5 C/km, ...
	//Note: Assumes climbing 1 km up mountain air on mountain cools at ELR, while air lifting 1 km above ground cools at ALR.
	//Note: Stull (2000) gives dry adiabatic lapse rate Gamma_dry_C_p_km = 9.8 C/km; dew point lapse rate Gamma_dew_C_p_km = 1.8 C/km derived Eq 5.8
	//Note: Elevation_lcl_km = 1/(9.8 - 1.8) * (Tair_K - Tdew_K), Elevation_lcl_km = lifting condensation level based on Eq 5.8 Stull (2000)
	//Note: Elevation_lcl_km = 1/(5.5 - 1.8) * (Tair_K - Tdew_K), where Gamma_EnvironmentalLapseRate_C_p_km = 5.5 C/km, based on modification
	//Note: Elevation_delta_km = (Elevation_pixel - Elevation_referenceStation) and sets Tair_K = Tdew_K at Elevation_lcl_km
	//Note: If Elevation_delta_km <= Elevation_lcl_km Then Tair_update = Tair_K - Gamma_dry_C_p_km * Elevation_delta_km, based on Stull (2000)
	//Note: Else Tair_update = Tair_K - (Gamma_dry_C_p_km * Elevation_lcl_km) - [Gamma_dew_C_p_km * (Elevation_delta_km - Elevation_lcl_km)]; Stull (2000)
	//Note: If Elevation_delta_km <= Elevation_lcl_km Then Tair_update = Tair_K - Gamma_EnvironmentalLapseRate_C_p_km * Elevation_delta_km
	//Note: Else Tair_update = Tair_K - (Gamma_EnvironmentalLapseRate_C_p_km * Elevation_lcl_km) - [Gamma_dew_C_p_km * (Elevation_delta_km - Elevation_lcl_km)] based modification
	//Note: Tdew_update = Tdew_K - Gamma_dew_C_p_km * Elevation_delta_km based on Stull (2000) and modification
	//Note: For differencing temperatures, Kelvin can be used in place of Celsius, as they are linearly related
	//References:
	//Stull, R. (2000). Meteorology for Scientists and Engineers, 2nd Edition. Brooks Cole. ISBN 0-534-37214-7

	//Note: Algorithm validated in Colorado Rocky Mountains with Flag_MultipleStations=1 Flag_HottestDaySimulated=1 MultipleStations_NumberOfStationsAveraged=2 for 2021
	//Note: Validation placed reference cell in map upper left corner, and test cells at weather stations (USAF WBAN): 725650-03017 and 720385-00419
	//Note: Station 720385-00419 (Mtn) DEM=3575.5 m, NLCD LC=52, TC=4%, IC=0%. Station 725650-03017 (Denver) DEM=1650 m, NLCD LC=24, TC=0%, IC=56%.
	//Note: Tests used two versions of WeatherStationAttributeTable.csv input file, one with and one without Station 720385-00419.
	//Note: Without Station 720385-00419 in file, 2 nearest stations to the local cell overlapping station 720385-00419 were 724640-93058 and 725640-24018
	//Note: With Station 720385-00419 in file, 2 nearest stations to the local cell overlapping station 720385-00419 were 720385-00419 and 725640-24018 
	//Note: Results found estimated Tair approximated observed Tair at Station 720385-00419 for runs using both versions of WeatherStationAttributeTable.csv
	//Note: Results at 3 pm for local cell at 720385-00419 Tair_modeled_w_station = 289.9K, Tair_modeled_wo_station = 290.1K, and Tair_observed = 290.4K
	//Note: Avoid using adiabatic lapse rate dry in place of environmental lapse rate, as it results in Tair_modeled_wo_station = 276.7K, much too cold. 

	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//Elevation_Station_weighted_m (m) initialized to 0
	double Elevation_Station_weighted_m = 0;
	//If Flag_MultipleStations is true then read in WeatherPro-> reference station meteorological variables for flux calculations
	//Note: Even if Flag_MultipleStations=1 and WeatherProcessor lapse rate adjusted Tair_weighted_K, this adjustment is needed 
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1) {
		//Note: WeightedInputMap[MapPixel_ID]["Elevation_WeatherStations_weighted_m"] is computed for each map pixel, only at timeStep = 0
		Elevation_Station_weighted_m = WeatherPro->WeightedInputMap[MapPixel_ID]["Elevation_WeatherStations_weighted_m"][0];
		//Elevation_delta_km (km) is height of pixel above station, equals Elevation_DEM_m(MapPixel_ID) - Elevation_Station_weighted_m), converted from m to km
		Elevation_delta_km = (input->Elevation_DEM_m[MapPixel_ID] - Elevation_Station_weighted_m) * Ratio_km_to_m;
	}
	//Else If Flag_MultipleStations is false then read in in input-> reference station meteorological variables for flux calculations
	else {
		//Ref_MapPixel_ID = input->MapPixel_ReferenceStation_IDs[0]; Ref_MapPixel_ID updated to 1st index when Flag_MultipleStations = false
		int Ref_MapPixel_ID = input->MapPixel_ReferenceStation_IDs[0];
		//Elevation_delta_km (km) is pixel height above station
		//Note: Elevation_delta_km = Elevation_DEM_m(MapPixel_ID) - Elevation_DEM_m[Ref_MapPixel_ID], convert from m to km
		Elevation_delta_km = (input->Elevation_DEM_m[MapPixel_ID] - input->Elevation_DEM_m[Ref_MapPixel_ID]) * Ratio_km_to_m;
	}

	//Elevation_lcl_km (km) equals 1 / (Gamma_EnvironmentalLapseRate_C_p_km - Gamma_dew_C_p_km) * (Tair_localCell_K - Tdew_localCell_K)
	//Note: Elevation_lcl_km (km) is lifting condensation level, modified from from Eq 5.8 Stull (2000), using the environmental lapse rate
	Elevation_lcl_km = 1 / (Gamma_EnvironmentalLapseRate_C_p_km - Gamma_dew_C_p_km) * (Tair_localCell_K - Tdew_localCell_K);

	//If Elevation_delta_km is below Elevation_lcl_km, then the lifting condensation level was not reached and air remains unsaturated
	if (Elevation_delta_km <= Elevation_lcl_km) {
		//Tair_weighted_K (K) equals Tair_weighted_K - Gamma_EnvironmentalLapseRate_C_p_km * Elevation_delta_km
		//Note: Tair_localCell_K (K) modified from Eq 5.12 Stull (2000) as explained above to use the environmental lapse rate
		Tair_localCell_K = Tair_localCell_K - Gamma_EnvironmentalLapseRate_C_p_km * Elevation_delta_km;
	}
	//Else, Elevation_delta_km > Elevation_lcl_km, so 1st lift air to Elevation_lcl_km, then 2nd lift additional (Elevation_delta_km - Elevation_lcl_km)
	//Note: Tair_K cools at adiabatic lapse rate dry Gamma_dry_C_p_km until Elevation_lcl_km, then becomes saturated and cools at Gamma_dew_C_p_km
	else {
		//Tair_weighted_K (K) equals Tair_localCell_K - Gamma_EnvironmentalLapseRate_C_p_km * Elevation_lcl_km;
		//Note: Tair_weighted_K (K) is initially updated using environmental lapse rate until Elevation_lcl_km, from Eq 5.12 Stull (2000)
		//Note: Tair_localCell_K (K) modified from Eq 5.12 Stull (2000) as explained above to use the environmental lapse rate
		Tair_localCell_K = Tair_localCell_K - Gamma_EnvironmentalLapseRate_C_p_km * Elevation_lcl_km;

		//Tair_localCell_K (K) is subsequently updated using adiabatic lapse rate for dew point air, modified Eq 5.13 Stull (2000)
		Tair_localCell_K = Tair_localCell_K - Gamma_dew_C_p_km * (Elevation_delta_km - Elevation_lcl_km);
	}
	//Tdew_localCell_K (K) is updated based on the adiabatic lapse rate for saturated dew point air, modified Eq 5.13 Stull (2000)
	Tdew_localCell_K = Tdew_localCell_K - Gamma_dew_C_p_km * Elevation_delta_km;
}

//RescaleVariables_CoolAir function will rescale latent and sensible energy fluxes
void HeatFluxCal::Rescale_LE_H_CoolAir(Inputs* input, DataFolder* folder, int timeStep) 
{
	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//If Tair_localCell_K became less than Tdew_localCell_K, set it to Tdew_localCell_K
	if (Tair_localCell_K < Tdew_localCell_K) { 
		Tair_localCell_K = Tdew_localCell_K; 
	}

	//Theory: Rescale latent energy (LE) and sensible heat (H) flux after iteratively computed in Tair_AbsHumidity_LayerBalance
	//Note: Rescaling is not ideal; it recognizes accurate temperature prediction with unrealistic energy exchange. Alternatives are possible.
	//Note: Alternative 1, compute LE and H with larger aerodynamic resistances for canopy and mesoScale, and cooler mesoScale air temperature
	//Note: Alternative 2, allow heat domes with negative daytime H, negative aerodynamic resistance with H yet positive with LE in convective cells
	//Note: Eddy covariance LE and H data from Capodimonte Park, Naples, Italy 2017-2021 used to rescale values from Tair_AbsHumidity_LayerBalance
	//Note: Capodimonte eddy covariance data from Drs. Carlo Calfapietra (carlo.calfapietra@cnr.it) and Terenzio Zenone (terenzio.zenone@isafom.cnr.it)
	//Note: Faulty approach 1: Tried recomputing LE using Penman Monteith evaporation functions, with WeatherPrep type resistances, then ...
	//Note: ... deriving H from energy balance, H = NR - DeltaQ - AH - LE. Obtained reasonable H and LE values but Tair_AbsHumidity_LayerBalance then ...
	//Note: ... generated faulty air temperature values with too high temperatures above NLCD Class 11 and 40s, effectively destroying model fitness.
	//Note: Faulty approach 2: Tried correcting high temperatures above NLCD Class 11 and 40s with algorithm to mix fraction of Tdew and Tair, which ...
	//Note: ... generated spatial patterns of temperature but was an overly parameterized approach. Hence, simple rescaling of H and LE preferred.
	//Note: Faulty approach 3: Tried using larger aerodynamic resistances for canopy and mesoScale as well as negative values, which ...
	//Note: ... generated some promising results with preserving water and forest air temperatures, but losing urban. May need to adjust by cover type.
	//Note: Regarding fit of temperature for NLCD Class 11, Buoy data for Washington, DC  August 28, 2018 provides air temperature above water.
	//Note: Buoy Station WASD2 8594900, Coordinates: 38.873 N 77.021 W; Air temp height: 2.6 m above site elevation.
	//Note: Woolway et. al. (2018) provide Bowen Ratio data for lakes that supports the rescaling approach for computing the LE_fraction

	//Theory from KIT: 
	//Note: Rescale Tair_mesoScale_K (K) flux and Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m after computed in Tair_AbsHumidity_LayerBalance
	//Note: Algorithm allows H_total_W_p_m2 across domain to have positive or negative values, adjusting Tair_mesoScale_K to keep r_aero > 0
	//Note: Tair_mesoScale_K can vary spatially across domain due to heterogeneity within boundary layer, likely during sunrise and sunset transitions
	//Note: Algorithm keeps Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m < 1000 by adjusting Tair_mesoScale_K (K) to allow local heterogeneity
	//Note: If pixel_ref H < 0 then stable atmosphere and set target Tair_mesoScale_K > Tair_localCell_K based on boundary layer thickness and lapse rate
	//Note: If pixel_ref H < 0 & pixel_local H > 0, then allow for local instability with instance of Tair_mesoScale_K < Tair_localCell_K
	//Note: Tair_mesoScale_final_K and AbsHumidity_mesoScale_final_kg_p_m3 represent rescaled values to report with outputs
	//Note: Tair_mesoScale_K and AbsHumidity_mesoScale_kg_p_m3 represent unscaled values needed by Tair_AbsHumidity_LayerBalance_NumericalMethod
	//Note: Algorithm inspired by Dr. Christopher Holst of Kalsruhe Institute of Technology (KIT) Alpine campus in GaPa, where he develops PALM-4U model
	//Note: Dr. Holst and post-doc ChangXing Lan (UW PhD) provided insights May-June 2023 at KIT that helped parameterize this energy balance model
	//Note: Consider refactor to control timing of boundary layer based on diurnal function, such as solar position or hour, with lag as needed
	//Note: Consider refactor to control thickness of boundary layer based on stability, such as USEPA 2000

	//References:
	//Woolway, R. I., ... & Jones, I. D. (2018). Geographic and temporal variations in turbulent heat loss from lakes: A global analysis across 45 lakes. Limnology and Oceanography, 63(6), 2436-2449. doi:https://doi.org/10.1002/lno.10950

	//Routine for daytime adjustments
	//If (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 > 0 then rescale H and LE
	if ((ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) > Constant_1E_negative4 && NR_total_W_p_m2 > 0 && ImpH_W_p_m2 < 0) {
		//H_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled sensible heat flux, using abs function to change from negative to positive
		double H_rescaled_Wpm2 = abs(ImpH_W_p_m2);
		//LE_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled latent energy flux, based on H_rescaled_Wpm2 being scaled by abs function 2 fold, from - to 0 to +
		//Note: If H_rescaled_Wpm2 = abs(H_initial), then LE_rescaled_Wpm2 = LE_initial + 2 * H_rescaled_Wpm2, and ... 
		//Note: ... X = LE_initial - H_initial, then X = LE_rescaled_Wpm2 - H_rescaled_Wpm2
		double LE_rescaled_Wpm2 = ImpLE_W_p_m2 + 2.0 * H_rescaled_Wpm2;
		//LE_frac = SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2)); uses SafeDivide function 
		double LE_frac = input->SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2));
		//H_frac (fraction) is defined as 1 minus LE_frac, where LE_frac + H_frac = unity in a Bowen Ratio
		double H_frac = 1.0 - LE_frac;
		//NR_available_W_p_m2 (W/m2) is total radiation available for LE and H flux, defined as NR_W_p_m2 + AH_W_p_m2 - DeltaQ_W_p_m2
		double NR_available_W_p_m2 = ImpNR_W_p_m2 + Imp_AH_W_p_m2 - ImpDeltaQ_W_p_m2;
		//ImpLE_W_p_m2 (W/m2) is final rescaled latent energy flux estimate, as product of NR_available_W_p_m2 and LE_frac
		ImpLE_W_p_m2 = NR_available_W_p_m2 * LE_frac;
		//ImpH_W_p_m2 (W/m2) is final rescaled sensible heat flux estimate, as product of NR_available_W_p_m2 and H_frac
		ImpH_W_p_m2 = NR_available_W_p_m2 * H_frac;
	}
	//If TreeCover_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 > 0 then rescale H and LE
	if (TreeCover_frac > Constant_1E_negative4 && NR_total_W_p_m2 > 0 && TreeH_W_p_m2 < 0) {
		//H_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled sensible heat flux, using abs function to change from negative to positive
		double H_rescaled_Wpm2 = abs(TreeH_W_p_m2);
		//LE_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled latent energy flux, based on H_rescaled_Wpm2 being scaled by abs function 2 fold, from - to 0 to +
		//Note: If H_rescaled_Wpm2 = abs(H_initial), then LE_rescaled_Wpm2 = LE_initial + 2 * H_rescaled_Wpm2, and ... 
		//Note: ... X = LE_initial - H_initial, then X = LE_rescaled_Wpm2 - H_rescaled_Wpm2
		double LE_rescaled_Wpm2 = TreeLE_W_p_m2 + 2.0 * H_rescaled_Wpm2;
		//LE_frac = SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2)); uses SafeDivide function 
		double LE_frac = input->SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2));
		//H_frac (fraction) is defined as 1 minus LE_frac, where LE_frac + H_frac = unity in a Bowen Ratio
		double H_frac = 1.0 - LE_frac;
		//NR_available_W_p_m2 (W/m2) is total radiation available for LE and H flux, defined as NR_W_p_m2 + AH_W_p_m2 - DeltaQ_W_p_m2
		double NR_available_W_p_m2 = TreeNR_W_p_m2 + Tree_AH_W_p_m2 - TreeDeltaQ_W_p_m2;
		//TreeLE_W_p_m2 (W/m2) is final rescaled latent energy flux estimate, as product of NR_available_W_p_m2 and LE_frac
		TreeLE_W_p_m2 = NR_available_W_p_m2 * LE_frac;
		//TreeH_W_p_m2 (W/m2) is final rescaled sensible heat flux estimate, as product of NR_available_W_p_m2 and H_frac
		TreeH_W_p_m2 = NR_available_W_p_m2 * H_frac;
		//LET_to_LE_ratio = SafeDivide(TreeLET_W_p_m2, (TreeLET_W_p_m2 + TreeLEE_W_p_m2)); uses SafeDivide function
		double LET_to_LE_ratio = input->SafeDivide(TreeLET_W_p_m2, (TreeLET_W_p_m2 + TreeLEE_W_p_m2));
		//TreeLET_W_p_m2 is computed as product of LET_to_LE_ratio and TreeLE_W_p_m2
		TreeLET_W_p_m2 = LET_to_LE_ratio * TreeLE_W_p_m2;
		//TreeLEE_W_p_m2 is computed as product of (1.0 - LET_to_LE_ratio) and TreeLE_W_p_m2
		TreeLEE_W_p_m2 = (1.0 - LET_to_LE_ratio) * TreeLE_W_p_m2;
	}
	//If ShortVegCover_noTreeCanopy_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 > 0 then rescale H and LE
	if (ShortVegCover_noTreeCanopy_frac > Constant_1E_negative4 && NR_total_W_p_m2 > 0 && SVegH_W_p_m2 < 0) {
		//H_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled sensible heat flux, using abs function to change from negative to positive
		double H_rescaled_Wpm2 = abs(SVegH_W_p_m2);
		//LE_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled latent energy flux, based on H_rescaled_Wpm2 being scaled by abs function 2 fold, from - to 0 to +
		//Note: If H_rescaled_Wpm2 = abs(H_initial), then LE_rescaled_Wpm2 = LE_initial + 2 * H_rescaled_Wpm2, and ... 
		//Note: ... X = LE_initial - H_initial, then X = LE_rescaled_Wpm2 - H_rescaled_Wpm2
		double LE_rescaled_Wpm2 = SVegLE_W_p_m2 + 2.0 * H_rescaled_Wpm2;
		//LE_frac = SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2)); uses SafeDivide function 
		double LE_frac = input->SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2));
		//H_frac (fraction) is defined as 1 minus LE_frac, where LE_frac + H_frac = unity in a Bowen Ratio
		double H_frac = 1.0 - LE_frac;
		//NR_available_W_p_m2 (W/m2) is total radiation available for LE and H flux, defined as NR_W_p_m2 + AH_W_p_m2 - DeltaQ_W_p_m2
		double NR_available_W_p_m2 = SVegNR_W_p_m2 + SVeg_AH_W_p_m2 - SVegDeltaQ_W_p_m2;
		//SVegLE_W_p_m2 (W/m2) is final rescaled latent energy flux estimate, as product of NR_available_W_p_m2 and LE_frac
		SVegLE_W_p_m2 = NR_available_W_p_m2 * LE_frac;
		//SVegH_W_p_m2 (W/m2) is final rescaled sensible heat flux estimate, as product of NR_available_W_p_m2 and H_frac
		SVegH_W_p_m2 = NR_available_W_p_m2 * H_frac;
		//LET_to_LE_ratio = SafeDivide(SVegLET_W_p_m2, (SVegLET_W_p_m2 + SVegLEE_W_p_m2)); uses SafeDivide function 
		double LET_to_LE_ratio = input->SafeDivide(SVegLET_W_p_m2, (SVegLET_W_p_m2 + SVegLEE_W_p_m2));
		//SVegLET_W_p_m2 is computed as product of LET_to_LE_ratio and SVegLE_W_p_m2
		SVegLET_W_p_m2 = LET_to_LE_ratio * SVegLE_W_p_m2;
		//SVegLEE_W_p_m2 is computed as product of (1.0 - LET_to_LE_ratio) and SVegLE_W_p_m2
		SVegLEE_W_p_m2 = (1.0 - LET_to_LE_ratio) * SVegLE_W_p_m2;
	}
	//If SoilCover_noTreeCanopy_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 > 0 then rescale H and LE
	if (SoilCover_noTreeCanopy_frac > Constant_1E_negative4 && NR_total_W_p_m2 > 0 && SoilH_W_p_m2 < 0) {
		//H_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled sensible heat flux, using abs function to change from negative to positive
		double H_rescaled_Wpm2 = abs(SoilH_W_p_m2);
		//LE_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled latent energy flux, based on H_rescaled_Wpm2 being scaled by abs function 2 fold, from - to 0 to +
		//Note: If H_rescaled_Wpm2 = abs(H_initial), then LE_rescaled_Wpm2 = LE_initial + 2 * H_rescaled_Wpm2, and ... 
		//Note: ... X = LE_initial - H_initial, then X = LE_rescaled_Wpm2 - H_rescaled_Wpm2
		double LE_rescaled_Wpm2 = SoilLE_W_p_m2 + 2.0 * H_rescaled_Wpm2;
		//LE_frac = SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2)); uses SafeDivide function 
		double LE_frac = input->SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2));
		//H_frac (fraction) is defined as 1 minus LE_frac, where LE_frac + H_frac = unity in a Bowen Ratio
		double H_frac = 1.0 - LE_frac;
		//NR_available_W_p_m2 (W/m2) is total radiation available for LE and H flux, defined as NR_W_p_m2 + AH_W_p_m2 - DeltaQ_W_p_m2
		double NR_available_W_p_m2 = SoilNR_W_p_m2 + Soil_AH_W_p_m2 - SoilDeltaQ_W_p_m2;
		//SoilLE_W_p_m2 (W/m2) is final rescaled latent energy flux estimate, as product of NR_available_W_p_m2 and LE_frac
		SoilLE_W_p_m2 = NR_available_W_p_m2 * LE_frac;
		//SoilH_W_p_m2 (W/m2) is final rescaled sensible heat flux estimate, as product of NR_available_W_p_m2 and H_frac
		SoilH_W_p_m2 = NR_available_W_p_m2 * H_frac;
	}
	//If WaterCover_noTreeCanopy_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 > 0 then rescale H and LE
	if (WaterCover_noTreeCanopy_frac > Constant_1E_negative4 && NR_total_W_p_m2 > 0 && WaterH_W_p_m2 < 0) {
		//H_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled sensible heat flux, using abs function to change from negative to positive
		double H_rescaled_Wpm2 = abs(WaterH_W_p_m2);
		//LE_rescaled_Wpm2 (W/m2) is 1st estimate of rescaled latent energy flux, based on H_rescaled_Wpm2 being scaled by abs function 2 fold, from - to 0 to +
		//Note: If H_rescaled_Wpm2 = abs(H_initial), then LE_rescaled_Wpm2 = LE_initial + 2 * H_rescaled_Wpm2, and ... 
		//Note: ... X = LE_initial - H_initial, then X = LE_rescaled_Wpm2 - H_rescaled_Wpm2
		double LE_rescaled_Wpm2 = WaterLE_W_p_m2 + 2.0 * H_rescaled_Wpm2;
		//LE_frac = SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2)); uses SafeDivide function 
		double LE_frac = input->SafeDivide(LE_rescaled_Wpm2, (LE_rescaled_Wpm2 + H_rescaled_Wpm2));
		//H_frac (fraction) is defined as 1 minus LE_frac, where LE_frac + H_frac = unity in a Bowen Ratio
		double H_frac = 1.0 - LE_frac;
		//NR_available_W_p_m2 (W/m2) is total radiation available for LE and H flux, defined as NR_W_p_m2 + AH_W_p_m2 - DeltaQ_W_p_m2
		double NR_available_W_p_m2 = WaterNR_W_p_m2 + Water_AH_W_p_m2 - WaterDeltaQ_W_p_m2;
		//WaterLE_W_p_m2 (W/m2) is final rescaled latent energy flux estimate, as product of NR_available_W_p_m2 and LE_frac
		WaterLE_W_p_m2 = NR_available_W_p_m2 * LE_frac;
		//WaterH_W_p_m2 (W/m2) is final rescaled sensible heat flux estimate, as product of NR_available_W_p_m2 and H_frac
		WaterH_W_p_m2 = NR_available_W_p_m2 * H_frac;
	}

	//Routine for nighttime adjustments
	//If (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 <= 0 then rescale H and LE
	if ((ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) > Constant_1E_negative4 && NR_total_W_p_m2 <= 0) {
		//If ImpLE_W_p_m2 > 0 then anthropogenic heat goes to latent energy and sensible heat equals net radiation minus ground heat flux
		if (ImpLE_W_p_m2 > 0) {
			ImpLE_W_p_m2 = Imp_AH_W_p_m2;
			ImpH_W_p_m2 = ImpNR_W_p_m2 - ImpDeltaQ_W_p_m2;
		}
		//Else If ImpLE_W_p_m2 = 0 then latent energy is zero and sensible heat equals net radiation minus ground heat flux plus anthropogenic heat
		else {
			ImpH_W_p_m2 = ImpNR_W_p_m2 - ImpDeltaQ_W_p_m2 + Imp_AH_W_p_m2;
		}
	}
	//If TreeCover_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 <= 0 then rescale H and LE
	if (TreeCover_frac > Constant_1E_negative4 && NR_total_W_p_m2 <= 0) {
		//If TreeLE_W_p_m2 > 0 then anthropogenic heat goes to latent energy and sensible heat equals net radiation minus ground heat flux
		if (TreeLE_W_p_m2 > 0) {
			TreeLE_W_p_m2 = Tree_AH_W_p_m2;
			TreeH_W_p_m2 = TreeNR_W_p_m2 - TreeDeltaQ_W_p_m2;
		}
		//Else If TreeLE_W_p_m2 = 0 then latent energy is zero and sensible heat equals net radiation minus ground heat flux plus anthropogenic heat
		else {
			TreeH_W_p_m2 = TreeNR_W_p_m2 - TreeDeltaQ_W_p_m2 + Tree_AH_W_p_m2;
		}
		//LET_to_LE_ratio = SafeDivide(TreeLET_W_p_m2, (TreeLET_W_p_m2 + TreeLEE_W_p_m2)); uses SafeDivide function 
		double LET_to_LE_ratio = input->SafeDivide(TreeLET_W_p_m2, (TreeLET_W_p_m2 + TreeLEE_W_p_m2));
		//TreeLET_W_p_m2 (W/m2) defined as product of LET_to_LE_ratio and TreeLE_W_p_m2
		TreeLET_W_p_m2 = LET_to_LE_ratio * TreeLE_W_p_m2;
		//TreeLEE_W_p_m2 (W/m2) defined as product of (1 - LET_to_LE_ratio) and TreeLE_W_p_m2
		TreeLEE_W_p_m2 = (1.0 - LET_to_LE_ratio) * TreeLE_W_p_m2;
	}
	//If ShortVegCover_noTreeCanopy_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 <= 0 then rescale H and LE
	if (ShortVegCover_noTreeCanopy_frac > Constant_1E_negative4 && NR_total_W_p_m2 <= 0) {
		//If SVegLE_W_p_m2 > 0 then anthropogenic heat goes to latent energy and sensible heat equals net radiation minus ground heat flux
		if (SVegLE_W_p_m2 > 0) {
			SVegLE_W_p_m2 = SVeg_AH_W_p_m2;
			SVegH_W_p_m2 = SVegNR_W_p_m2 - SVegDeltaQ_W_p_m2;
		}
		//Else If SVegLE_W_p_m2 = 0 then latent energy is zero and sensible heat equals net radiation minus ground heat flux plus anthropogenic heat
		else {
			SVegH_W_p_m2 = SVegNR_W_p_m2 - SVegDeltaQ_W_p_m2 + SVeg_AH_W_p_m2;
		}
		//LET_to_LE_ratio = SafeDivide(SVegLET_W_p_m2, (SVegLET_W_p_m2 + SVegLEE_W_p_m2)); uses SafeDivide function 
		double LET_to_LE_ratio = input->SafeDivide(SVegLET_W_p_m2, (SVegLET_W_p_m2 + SVegLEE_W_p_m2));
		//SVegLET_W_p_m2 (W/m2) defined as product of LET_to_LE_ratio and SVegLE_W_p_m2
		SVegLET_W_p_m2 = LET_to_LE_ratio * SVegLE_W_p_m2;
		//SVegLEE_W_p_m2 (W/m2) defined as product of (1 - LET_to_LE_ratio) and SVegLE_W_p_m2
		SVegLEE_W_p_m2 = (1.0 - LET_to_LE_ratio) * SVegLE_W_p_m2;
	}
	//If SoilCover_noTreeCanopy_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 <= 0 then rescale H and LE
	if (SoilCover_noTreeCanopy_frac > Constant_1E_negative4 && NR_total_W_p_m2 <= 0) {
		//If SoilLE_W_p_m2  > 0 then anthropogenic heat goes to latent energy and sensible heat equals net radiation minus ground heat flux
		if (SoilLE_W_p_m2 > 0) {
			SoilLE_W_p_m2 = Soil_AH_W_p_m2;
			SoilH_W_p_m2 = SoilNR_W_p_m2 - SoilDeltaQ_W_p_m2;
		}
		//Else If SoilLE_W_p_m2 = 0 then latent energy is zero and sensible heat equals net radiation minus ground heat flux plus anthropogenic heat
		else {
			SoilH_W_p_m2 = SoilNR_W_p_m2 - SoilDeltaQ_W_p_m2 + Soil_AH_W_p_m2;
		}
	}
	//If WaterCover_noTreeCanopy_frac > Constant_1E_negative4 cover threshold AND NR_total_W_p_m2 <= 0 then rescale H and LE
	if (WaterCover_noTreeCanopy_frac > Constant_1E_negative4 && NR_total_W_p_m2 <= 0) {
		//If WaterLE_W_p_m2 > 0 then anthropogenic heat goes to latent energy and sensible heat equals net radiation minus ground heat flux
		if (WaterLE_W_p_m2 > 0) {
			WaterLE_W_p_m2 = Water_AH_W_p_m2;
			WaterH_W_p_m2 = WaterNR_W_p_m2 - WaterDeltaQ_W_p_m2;
		}
		//Else If WaterLE_W_p_m2 = 0 then latent energy is zero and sensible heat equals net radiation minus ground heat flux plus anthropogenic heat
		else {
			WaterH_W_p_m2 = WaterNR_W_p_m2 - WaterDeltaQ_W_p_m2 + Water_AH_W_p_m2;
		}
	}

	//H_total_W_p_m2 (W/m2) is Eq 12 of Yang et al. (2013), total sensible heat flux (W/m^2), sum of fractional components from each cover type
	//Note: Theory of Yang et al. (2013) above Eq 12, H_localAir_to_mesoAir = H_ground_to_localAir in steady state to maintain energy balance
	H_total_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * ImpH_W_p_m2 + TreeCover_frac * TreeH_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVegH_W_p_m2 + SoilCover_noTreeCanopy_frac * SoilH_W_p_m2 + WaterCover_noTreeCanopy_frac * WaterH_W_p_m2;
	//LE_total_W_p_m2 (W/m2) is Eq 13 of Yang et al. (2013), total latent heat flux (W/m^2), sum of fractional components from each cover type
	//Note: Theory of Yang et al. (2013) above Eq 12, LE_localAir_to_mesoAir = LE_ground_to_localAir in steady state to maintain energy balance
	LE_total_W_p_m2 = (ImperviousCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) * ImpLE_W_p_m2 + TreeCover_frac * TreeLE_W_p_m2 + ShortVegCover_noTreeCanopy_frac * SVegLE_W_p_m2 + SoilCover_noTreeCanopy_frac * SoilLE_W_p_m2 + WaterCover_noTreeCanopy_frac * WaterLE_W_p_m2;
}

//Initialize_AtmosphericBoundaryLayer_param function imposes sine function oscillation on the atmospheric boundary layer (ABL) depth or thickness
//Note: HydroPlus and the i-Tree Cool Air model do not need the ABL thickness, yet estimates it to enable a conversation about it's role
//Note: Analytical and emperical methods exist to estimate ABL depth in unstable, neutral, and stable conditions ...
//Note: Unstable: See Stull (2017) Section 18.4.4. Mixed-layer Temperature, Eq 18.6 for zi, and FHmax from Sample Application Chp 18 p. 694
//Note: Neutral: See Stull (2003) h = Ux * C / f, where Ux is friction velocity, C is 0.2 to 0.3 and f is Coriolis parameter, Ux from Eq 18.1 Stull (2017)
//Note: Stable: See Stull (2017) Section 18.4.3 Stable ABL Temperature, Eq 18.4 for He, and above that, h = 5 * He
void HeatFluxCal::Initialize_AtmosphericBoundaryLayer_param() {
	//AtmosphericBoundaryLayer_Thickness_max_m (m) is the maximum ABL thickness, ranging from 100s to 1000s of m, estimated at 1500m
	//Note: Consider refactor to compute value based on Unstable, Neutral and Stable relations given in function header
	//Note: AbsHumidity_mesoScale_kg_p_m3 (kg/m3) and Tair_mesoScale_K (K) very sensitive to ABL thickness, so adjust to improve performance
	double AtmosphericBoundaryLayer_Thickness_max_m = 1500;
	//AtmosphericBoundaryLayer_Thickness_min_m (m) is the minimum ABL thickness, ranging from 10s to 100s of m, estimated at 50m
	//Note: Consider refactor to compute value based on Unstable, Neutral and Stable relations given in function header
	//Note: AbsHumidity_mesoScale_kg_p_m3 (kg/m3) and Tair_mesoScale_K (K) very sensitive to ABL thickness, so adjust to improve performance
	double AtmosphericBoundaryLayer_Thickness_min_m = 50;
	//AtmosphericBoundaryLayer_Thickness_max_hr (hr) is the maximum thickness time from midnight, hr = 0, estimated at 13hr
	//Note: Consider refactor to compute value based on Unstable, Neutral and Stable relations given in function header
	double AtmosphericBoundaryLayer_Thickness_max_hr = 13;
	//SinWave_Coeff_A_m is the amplitude of the wave, defined as A = (thickness_max - thickness_min) / 2
	double SinWave_Coeff_A_m = (AtmosphericBoundaryLayer_Thickness_max_m - AtmosphericBoundaryLayer_Thickness_min_m) / 2;
	//SineWave_Coeff_B_hr is the frequency of the wave, defined as B = 2π / 24, based on two pi equals 360 degrees, and 24 hours in day
	double SineWave_Coeff_B_hr = 2.0 * M_PI / Ratio_Hour_to_Day;
	//SineWave_Coeff_C_hr is the horizontal shift to align the peak with DiurnalCycle_hr, defined as ...
	// ... C = AtmosphericBoundaryLayer_Thickness_max_hr - (π/ (2 * B))
	double SineWave_Coeff_C_hr = AtmosphericBoundaryLayer_Thickness_max_hr - (M_PI / (2.0 * SineWave_Coeff_B_hr));
	//SineWave_Coeff_D_m is the vertical shift of the wave, defined as D = (height_max + height_min) / 2
	double SineWave_Coeff_D_m = (AtmosphericBoundaryLayer_Thickness_max_m + AtmosphericBoundaryLayer_Thickness_min_m) / 2;
	//AtmosphericBoundaryLayer_Thickness_m (m) is the height of the ABL, defined as AtmosphericBoundaryLayer_Thickness_m = A * sin(B * (hour_int - C)) + D
	double AtmosphericBoundaryLayer_Thickness_m = SinWave_Coeff_A_m * sin(SineWave_Coeff_B_hr * (Diurnal_Hour_HH - SineWave_Coeff_C_hr)) + SineWave_Coeff_D_m;
	//AtmosphericBoundaryLayer_Thickness_km (km) is the height of the ABL converted to km with Ratio_km_to_m
	AtmosphericBoundaryLayer_Thickness_km = AtmosphericBoundaryLayer_Thickness_m * Ratio_km_to_m;
	//ABL_adjust_frac (fraction) reduces the influence of the atmospheric boundary layer thickness on Tair_mesoScale_K to 10% of standard thickness
	//Note: Based very roughly on data discussed with Dr. Holst from KIT, Germany; heterogeneity in the ABL is often layered, with low lying inversions
	ABL_adjust_frac = 0.1;
	//Resistance_Aerodynamic_CanopyToMesoLayer_max_s_p_m (s/m) defined arbitrarily as 100 s/m
	//Note: Consider refactor to compute value based on Unstable, Neutral and Stable relations given in function header
	Resistance_Aerodynamic_CanopyToMesoLayer_max_s_p_m = 100;
}

//RescaleVariables_CoolAir function will rescale latent and sensible energy fluxes, compute corresponding mesoScale values for Tair and AbsoluteHumidity
void HeatFluxCal::RescaleVariables_CoolAir(Inputs* input, DataOrganizer* organizer, DataFolder* folder, int timeStep)
{

	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions without sending solarCalculator
	HeatMetrics_Calc HeatMetricsCalc(input);
	//AtmPres_kPa = input->AtmPres_kPa[timeStep]
	double AtmPres_kPa = input->AtmPres_kPa[timeStep];
	//SpecificHeat_HumidAir_JpkgK = HeatMetricsCalc.SpecificHeat_HumidAir_JpkgK_Calc(Tair_localCell_K, Tdew_localCell_K, AtmPres_kPa)
	double SpecificHeat_HumidAir_JpkgK = HeatMetricsCalc.SpecificHeat_HumidAir_JpkgK_Calc(Tair_localCell_K, Tdew_localCell_K, AtmPres_kPa);
	//LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(Tair_localCell_K)
	double LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(Tair_localCell_K);

	//Call Rescale_LE_H_CoolAir to compute H_total_W_p_m2 and LE_total_W_p_m2
	Rescale_LE_H_CoolAir(input, folder, timeStep);
	//initialize the AtmosphericBoundaryLayer parameters
	Initialize_AtmosphericBoundaryLayer_param();
	
	//If isReferenceStationPixel(MapPixel_ID) then update for reference folder mesoScale values
	if (input->isReferenceStationPixel(MapPixel_ID)) {
		//If not Flag_MultipleStations then call Rescale_Tair_mesoScale_Ref_CoolAir
		if (input->SimulationNumericalParams["Flag_MultipleStations"] != 1) {
			//If H_total_W_p_m2 (W/m2) <= 0 then Tair_mesoScale_K > Tair_localCell_K to maintain Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m > 0
			//Note: Atmospheric stability is characterized by Tair_mesoScale_K > Tair_localCell_K, and H_total_W_p_m2 < 0 is downwelling 
			//Note: Ensure Tair_mesoScale_K > Tair_localCell_K for all pixels by computing maximum potential Tair_localCell_K
			if (H_total_W_p_m2 <= 0) {
				//Tair_mesoScale_final_K (K) defined as Tair_localCell_K + (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km)
				//Note: Approximation of the environmental lapse rate and its effect on air temperature is based on Eq 5.12 in Stull (2000)
				Tair_mesoScale_final_K = Tair_localCell_K + (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km);
			}
			//If H_total_W_p_m2 (W/m2) > 0 then Tair_mesoScale_K < Tair_localCell_K to maintain Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m > 0
			//Note: Atmospheric instability is characterized by Tair_mesoScale_K < Tair_localCell_K, and H_total_W_p_m2 > 0 is upwelling 
			if (H_total_W_p_m2 > 0) {
				//Tair_mesoScale_final_K (K) defined as Tair_localCell_K + (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km)
				//Note: Approximation of the environmental lapse rate and its effect on air temperature is based on Eq 5.12 in Stull (2000)
				Tair_mesoScale_final_K = Tair_localCell_K - (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km);
			}
		}
	}
	//Else If not isReferenceStationPixel(MapPixel_ID) then update for non-reference folder mesoScale values
	else if (!input->isReferenceStationPixel(MapPixel_ID)) {
		//If Flag_MultipleStations is 1 then update Tair_mesoScale_final_pointer_K
		if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1) {
			//Tair_mesoScale_final_weighted_K and H_total_weighted_W_p_m2 has been computed in Simulation Cordinator
			Tair_mesoScale_final_pointer_K = WeatherProcessor::Tair_mesoScale_final_weighted_K;
			H_total_pointer_W_p_m2 = WeatherProcessor::H_total_weighted_W_p_m2;
		}
		//Else If Flag_MultipleStations not equal to 1 then create ReferenceFolder temperature and humidity
		else {
			//Ref_MapPixel_ID = input->MapPixel_ReferenceStation_IDs[0]; Ref_MapPixel_ID updated to 1st index when Flag_MultipleStations = false
			int Ref_MapPixel_ID = input->MapPixel_ReferenceStation_IDs[0];
			//DataFolder* ReferenceFolder = organizer->DataDrawers[Ref_MapPixel_ID][0]; ReferenceFolder updated
			DataFolder* ReferenceFolder = organizer->DataDrawers[Ref_MapPixel_ID][0];

			Tair_mesoScale_final_pointer_K = ReferenceFolder->VarDict["Tair_mesoScale_final_K"];
			H_total_pointer_W_p_m2 = ReferenceFolder->VarDict["H_total_W_p_m2"];
		}

		//If H_total_W_p_m2 (W/m2) & H_total_pointer_W_p_m2 <= 0, then atmosphere is stable and Tair_localCell_K < Tair_mesoScale_final_pointer_K
		if (H_total_W_p_m2 <= 0 && H_total_pointer_W_p_m2 <= 0) {
			//If Tair_localCell_K (K) > Tair_mesoScale_final_K (K) then cell has unstable atmosphere, so increase Tair_mesoScale_final_K
			if (Tair_localCell_K >= Tair_mesoScale_final_pointer_K) {
				//Tair_mesoScale_final_K (K) defined as Tair_localCell_K + ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km)
				//Note: Approximation of the environmental lapse rate and its effect on air temperature is based on Eq 5.12 in Stull (2000)
				Tair_mesoScale_final_K = Tair_localCell_K + ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km);
			}
			//Else Tair_localCell_K (K) < Tair_mesoScale_K when H_total_W_p_m2 < 0, then stable atmosphere with Tair_mesoScale_K
			else {
				//Tair_mesoScale_final_K (K) defined as Tair_mesoScale_final_pointer_K
				Tair_mesoScale_final_K = Tair_mesoScale_final_pointer_K;
			}
		}
		//If H_total_W_p_m2 (W/m2) & H_total_pointer_W_p_m2 > 0, then atmosphere is unstable and Tair_localCell_K > Tair_mesoScale_K
		else if (H_total_W_p_m2 > 0 && H_total_pointer_W_p_m2 > 0) {
			//If Tair_localCell_K (K) < Tair_mesoScale_final_pointer_K (K) then cell has stable atmosphere, so decrease Tair_mesoScale_final_K
			if (Tair_localCell_K < Tair_mesoScale_final_pointer_K) {
				//Tair_mesoScale_final_K (K) defined as Tair_localCell_K - ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km)
				//Note: Approximation of the environmental lapse rate and its effect on air temperature is based on Eq 5.12 in Stull (2000)
				Tair_mesoScale_final_K = Tair_localCell_K - ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km);
			}
			//Else Tair_localCell_K (K) >= Tair_mesoScale_K when H_total_W_p_m2 > 0, then unstable atmosphere with Tair_mesoScale_K
			else {
				//Tair_mesoScale_final_K (K) defined as Tair_mesoScale_final_K
				Tair_mesoScale_final_K = Tair_mesoScale_final_pointer_K;
			}
		}
		//Else If H_total_W_p_m2 (W/m2) & VarDict["H_total_W_p_m2"] are different sign
		//Note: Tair_mesoScale_K varies spatially across domain due to heterogeneity within boundary layer, likely during sunrise and sunset transitions
		else {
			//If H_total_W_p_m2 (W/m2) <= 0 then Tair_mesoScale_K > Tair_localCell_K to maintain Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m > 0
			//Note: Atmospheric stability is characterized by Tair_mesoScale_K > Tair_localCell_K, and H_total_W_p_m2 < 0 is downwelling 
			if (H_total_W_p_m2 <= 0) {
				//Tair_mesoScale_final_K (K) defined as Tair_localCell_K + ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km)
				//Note: Approximation of the environmental lapse rate and its effect on air temperature is based on Eq 5.12 in Stull (2000)
				Tair_mesoScale_final_K = Tair_localCell_K + ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km);
			}
			//If H_total_W_p_m2 (W/m2) > 0 then Tair_mesoScale_K < Tair_localCell_K to maintain Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m > 0
			//Note: Atmospheric instability is characterized by Tair_mesoScale_K < Tair_localCell_K, and H_total_W_p_m2 > 0 is upwelling 
			if (H_total_W_p_m2 > 0) {
				//Tair_mesoScale_final_K (K) defined as Tair_localCell_K + ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km)
				//Note: Approximation of the environmental lapse rate and its effect on air temperature is based on Eq 5.12 in Stull (2000)
				Tair_mesoScale_final_K = Tair_localCell_K - ABL_adjust_frac * (AtmosphericBoundaryLayer_Thickness_km * Gamma_EnvironmentalLapseRate_C_p_km);
			}
		}
	}

	//Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m (s/m) is based on rearranged form of Eq 1 from Yang et al. (2013), where ra = [(rho*Cp)* (Ta - Tb)] / H, ...
	//Note: Eq 1 of Yang et al. (2013), H = [(rho*Cp)* (Ta - Tb)] / ra, and ra = Resistance_Aerodynamic_CanopyToMesoLayer_Fitting_s_p_m
	//Note: ... Tb = Tair_mesoScale_K, Ta = Tair_localCell_K, H = H_total_W_p_m2 (W/m2), rho = Density_Air_kg_p_m3, Cp = SpecificHeat_DryAir_J_p_kg_K
	//Note: Unit balance right hand side = (W/m2) * (s/m) / [(kg/m3) * (J/kg/K)] = (J/s/m2) * (s/m) / (J/m3/K) = (J/m3) / (J/m3/K) = K
	resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m = (Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) *
		(Tair_localCell_K - Tair_mesoScale_final_K) / H_total_W_p_m2;

	//If Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m (s/m) > Resistance_Aerodynamic_CanopyToMesoLayer_max_s_p_m then unreasonable
	//Note: Ratio of (Tair_localCell_K - Tair_mesoScale_K) / H_total_W_p_m2 determines Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m
	//Note: Adjust Tair_mesoScale_K (K) to maintain reasonable Ratio of (Tair_localCell_K - Tair_mesoScale_K) / H_total_W_p_m2 ~ 1/20
	//Note: Consider refactor to improve error handling for large values of Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m 
	if (resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m > Resistance_Aerodynamic_CanopyToMesoLayer_max_s_p_m) {
		//Tair_localCell_K_minus_T_mesoScale_K (K) defined as ~1/20.0 of H_total_W_p_m2, using Diurnal_Hour_HH to add diurnal variability to fraction
		double Tair_localCell_K_minus_T_mesoScale_K_abs_K = 1 / (15.0 + Diurnal_Hour_HH) * abs(H_total_W_p_m2);
		//If H_total_W_p_m2 (W/m2) <= 0 then stable atmosphere 
		if (H_total_W_p_m2 <= 0) {
			//Tair_mesoScale_final_K (K) is Tair_localCell_K plus abs(Tair_localCell_K_minus_T_mesoScale_K) to create stable atmosphere
			Tair_mesoScale_final_K = Tair_localCell_K + Tair_localCell_K_minus_T_mesoScale_K_abs_K;
		}
		//Else If H_total_W_p_m2 (W/m2) > 0 then unstable atmosphere 
		else if (H_total_W_p_m2 > 0) {
			//Tair_mesoScale_final_K (K) is Tair_localCell_K - abs(Tair_localCell_K_minus_T_mesoScale_K) to create unstable atmosphere
			Tair_mesoScale_final_K = Tair_localCell_K - Tair_localCell_K_minus_T_mesoScale_K_abs_K;
		}
		//Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m (s/m) recomputed w rearranged form of Eq 1 from Yang et al. (2013), where ra = [(rho*Cp)* (Ta - Tb)] / H
		resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m = (Density_Air_kg_p_m3 * SpecificHeat_HumidAir_JpkgK) *
			(Tair_localCell_K - Tair_mesoScale_final_K) / H_total_W_p_m2;
	}

	//AbsHumidity_mesoScale_kg_p_m3 (kg/m^3) is based on rearranged form of Eq 2 from Yang et al. (2013), where AHb = AHa - (LE*ra)/Lambda, ...
	//Note: ... AHb = AbsHumidity_mesoScale_kg_p_m3, AHa = AbsHumidityAct_localCell_kg_p_m3, LE = LE_total_W_p_m2, ...
	//Note: ... Lambda = LatentHeatVaporization_Water_J_p_kg, ra = Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m
	//Note: Unit balance right hand side = (W/m2) * (s/m) / (J/kg) = (J/s/m2) * (s/m) / (J/kg) = (J/m3) / (J/kg) = kg/m3
	AbsHumidity_mesoScale_final_kg_p_m3 = AbsHumidityAct_localCell_kg_p_m3 - (LE_total_W_p_m2 * resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m) /
		LatentHeat_Vaporization_JpkgK;
	//If AbsHumidity_mesoScale_final_kg_p_m3 < 0 then set to 0; 
	//Note: This error may emerge due to algorithm using same Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m for heat and humidity flux
	if (AbsHumidity_mesoScale_final_kg_p_m3 < 0) { AbsHumidity_mesoScale_final_kg_p_m3 = 0; }
}

//CollectVariables_CoolAir function will store all values to memory
void HeatFluxCal::CollectVariables_CoolAir(Inputs* input, DataFolder* folder, SolarCalculator* solarCalc, int timeStep)
{
	//MapPixel_ID = folder->ParamDict["MapPixel_ID"] defined to locally, and perhaps MapPixel_ID should not be sent
	int MapPixel_ID = folder->ParamDict["MapPixel_ID"];

	//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions with sending solarCalculator
	HeatMetrics_Calc HeatMetricsCalc(input);

	//If (timeStep == 0 && vectors_initialized == false) then initialize vectors
	if (timeStep == 0 && vectors_initialized == false) {
		//HeatFlux_InitializeVectors(input) and reset flag
		HeatFlux_InitializeVectors(input);
		//vectors_initialized = true to avoid reentry
		vectors_initialized = true;
	}

	//Tair_variation_threshold_K (K) represents maximum likely variation between weather station and pixel values in one time step
	//Note: Used for error handling, to protect against making very unlikely predictions 
	double Tair_variation_threshold_K = 40;
	//If abs(Tair_localCell_K - Tair_referenceCell_K) > 40
	if ((abs(Tair_localCell_K - Tair_referenceCell_K) > Tair_variation_threshold_K) ||
			isnan(Tair_localCell_K) || isnan(Tair_referenceCell_K) || isnan(resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m)) {
		//If an exceedance then explain ...
		if (abs(Tair_localCell_K - Tair_referenceCell_K) > Tair_variation_threshold_K) {
			cout << "Warning: The predicted Tair of " << Tair_localCell_K << " K when observed Tair is " << Tair_referenceCell_K << " K" << endl;
			cout << " ... which exceeds the threshold of " << Tair_variation_threshold_K << ". Details follow." << endl;
			cout << "Correction: Check inputs or modify code to allow for such large Tair variation." << endl;
		}
		//Else if a nan than explain
		else {
			cout << "Warning: A model variable has a value that is not a number. Details follow." << endl;
			cout << "Correction: Check inputs or modify code to avoid not a number conditions." << endl;
		}
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Explanation: This Tair value suggests errors with input maps, parameters, calculated resistances, etc.." << endl;
		cout << "  This occurred at timestep " << timeStep << " and date hour " << input->SimulationDate_GDH[timeStep] << endl;
		cout << "  This occurred at MapPixel_ID " << folder->ParamDict["MapPixel_ID"] << endl;
		cout << "  LandCover_Class " << input->LandCover_NLCD_Class[MapPixel_ID] << endl;
		cout << "  TreeCover_frac " << input->TreeCover_frac[MapPixel_ID] << endl;
		cout << "  ImperviousCover_frac " << input->ImperviousCover_frac[MapPixel_ID] << endl;
		cout << "  DEM_m " << input->Elevation_DEM_m[MapPixel_ID] << endl;
		cout << "  Local Tair_K " << Tair_localCell_K << endl;
		cout << "  Local Tdew_K " << Tdew_localCell_K << endl;
		cout << "  Reference Tair_K " << Tair_referenceCell_K << endl;
		cout << "  Mesoscale Tair_K " << Tair_mesoScale_K << endl;
		cout << "  Mesoscale AbsHumidity_mesoScale_kg_p_m3 " << AbsHumidity_mesoScale_kg_p_m3 << endl;
		cout << "  WindSpeed_mps " << WindSpeed_Reference_m_p_s << endl;
		cout << "  Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m " << resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m << endl;
		cout << "  AnthropogenicHeat_Flux_Qtot_Avg_Wpm2 " << input->AnthropogenicHeat_Flux_Qtot_Avg_Wpm2[MapPixel_ID] << endl;
		//Call abort function, which ends the HydroPlus.exe simulation 
		abort();
	}

	//Tair_K (K) = Inputs::roundToDecimalPlaces(Tair_localCell_K,2), air temperature of the local pixel
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["Tair_K"] = Inputs::roundToDecimalPlaces(Tair_localCell_K, 2);
	//Tdew_K (K) = Inputs::roundToDecimalPlaces(Tdew_localCell_K, 2), dew point temperature of the local pixel
	folder->VarDict["Tdew_K"] = Inputs::roundToDecimalPlaces(Tdew_localCell_K, 2);
	//AbsHumidity_kg_p_m3 (kg/m3) = Inputs::roundToDecimalPlaces(AbsHumidityAct_localCell_kg_p_m3, 4), absolute humidity of the local pixel
	folder->VarDict["AbsHumidity_kg_p_m3"] = Inputs::roundToDecimalPlaces(AbsHumidityAct_localCell_kg_p_m3, 4);
	//Tair_mesoScale_final_K (K) = Inputs::roundToDecimalPlaces(Tair_mesoScale_final_K, 2), air temperature of the mesocale layer
	//Note: Tair_mesoScale_final_K (K), mesoScale air temperature needed for output and RescaleVariables_CoolAir
	//Note: Do not remove this Tair_mesoScale_final_K variable or replace with folder->VarDict["Tair_mesoScale_K"]
	folder->VarDict["Tair_mesoScale_final_K"] = Inputs::roundToDecimalPlaces(Tair_mesoScale_final_K, 2);
	//AbsHumidity_mesoScale_final_kg_p_m3 (kg/m3) = Inputs::roundToDecimalPlaces(AbsHumidity_mesoScale_final_kg_p_m3, 4), absolute humidity of the mesoscale layer
	//AbsHumidity_mesoScale_final_kg_p_m3 (kg/m3), mesoScale absolute humidity needed for output and RescaleVariables_CoolAir
	//Note: Do not remove this AbsHumidity_mesoScale_final_kg_p_m3 variable or replace with folder->VarDict["AbsHumidity_mesoScale_kg_p_m3"]
	folder->VarDict["AbsHumidity_mesoScale_final_kg_p_m3"] = Inputs::roundToDecimalPlaces(AbsHumidity_mesoScale_final_kg_p_m3, 4);
	//Tair_mesoScale_K (K) = Tair_mesoScale_K[i] at full precision, air temperature of the mesocale layer
	//Note: Tair_mesoScale_K (K), mesoScale air temperature needed for energy balance of Tair_AbsHumidity_LayerBalance_NumericalMethod
	//Note: Do not remove this Tair_mesoScale_K variable or replace with folder->VarDict["Tair_mesoScale_final_K"]
	folder->VarDict["Tair_mesoScale_K"] = Tair_mesoScale_K;
	//AbsHumidity_mesoScale_kg_p_m3 (kg/m3) = AbsHumidity_mesoScale_kg_p_m3 at full precision, absolute humidity of the mesoscale layer
	//Note: AbsHumidity_mesoScale_kg_p_m3 (kg/m3), mesoScale absolute humidity needed for energy balance of Tair_AbsHumidity_LayerBalance_NumericalMethod
	//Note: Do not remove this AbsHumidity_mesoScale_kg_p_m3 variable or replace with folder->VarDict["AbsHumidity_mesoScale_final_kg_p_m3"]
	folder->VarDict["AbsHumidity_mesoScale_kg_p_m3"] = AbsHumidity_mesoScale_kg_p_m3;
	//Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m (s/m) is Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m (s/m), using Tair_mesoScale_final_K (K)
	folder->VarDict["Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m"] = resistanceVar->Resistance_Aerodynamic_CanopyToMesoLayer_s_p_m;

	//Note: Net Radiation, Sensible Heat (H), and Latent Energy (LE) terms for each cover type are per meter squared, not scaled to fractional area
	folder->VarDict["ImpNR_W_p_m2"] = ImpNR_W_p_m2;
	Imp_AH_W_p_m2_vec[MapPixel_ID] = Imp_AH_W_p_m2;
	folder->VarDict["ImpH_W_p_m2"] = ImpH_W_p_m2;
	folder->VarDict["ImpLE_W_p_m2"] = ImpLE_W_p_m2;
	ImpDeltaQ_W_p_m2_vec[MapPixel_ID] = ImpDeltaQ_W_p_m2;

	//Note: Net Radiation, Sensible Heat (H), and Latent Energy (LE) terms for each cover type are per meter squared, not scaled to fractional area
	folder->VarDict["TreeNR_W_p_m2"] = TreeNR_W_p_m2;
	Tree_AH_W_p_m2_vec[MapPixel_ID] = Tree_AH_W_p_m2;
	folder->VarDict["TreeH_W_p_m2"] = TreeH_W_p_m2;
	folder->VarDict["TreeLE_W_p_m2"] = TreeLE_W_p_m2;
	folder->VarDict["TreeLEE_W_p_m2"] = TreeLEE_W_p_m2;
	folder->VarDict["TreeLET_W_p_m2"] = TreeLET_W_p_m2;
	TreeDeltaQ_W_p_m2_vec[MapPixel_ID] = TreeDeltaQ_W_p_m2;

	//Note: Net Radiation, Sensible Heat (H), and Latent Energy (LE) terms for each cover type are per meter squared, not scaled to fractional area
	folder->VarDict["SVegNR_W_p_m2"] = SVegNR_W_p_m2;
	SVeg_AH_W_p_m2_vec[MapPixel_ID] = SVeg_AH_W_p_m2;
	folder->VarDict["SVegH_W_p_m2"] = SVegH_W_p_m2;
	folder->VarDict["SVegLE_W_p_m2"] = SVegLE_W_p_m2;
	folder->VarDict["SVegLEE_W_p_m2"] = SVegLEE_W_p_m2;
	folder->VarDict["SVegLET_W_p_m2"] = SVegLET_W_p_m2;
	SVegDeltaQ_W_p_m2_vec[MapPixel_ID] = SVegDeltaQ_W_p_m2;

	//Note: Net Radiation, Sensible Heat (H), and Latent Energy (LE) terms for each cover type are per meter squared, not scaled to fractional area
	folder->VarDict["SoilNR_W_p_m2"] = SoilNR_W_p_m2;
	Soil_AH_W_p_m2_vec[MapPixel_ID] = Soil_AH_W_p_m2;
	folder->VarDict["SoilH_W_p_m2"] = SoilH_W_p_m2;
	folder->VarDict["SoilLE_W_p_m2"] = SoilLE_W_p_m2;
	SoilDeltaQ_W_p_m2_vec[MapPixel_ID] = SoilDeltaQ_W_p_m2;

	//Note: Net Radiation, Sensible Heat (H), and Latent Energy (LE) terms for each cover type are per meter squared, not scaled to fractional area
	folder->VarDict["WaterNR_W_p_m2"] = WaterNR_W_p_m2;
	Water_AH_W_p_m2_vec[MapPixel_ID] = Water_AH_W_p_m2;
	folder->VarDict["WaterH_W_p_m2"] = WaterH_W_p_m2;
	folder->VarDict["WaterLE_W_p_m2"] = WaterLE_W_p_m2;
	WaterDeltaQ_W_p_m2_vec[MapPixel_ID] = WaterDeltaQ_W_p_m2;

	//Note: Net Radiation, Sensible Heat (H), and Latent Energy (LE) terms for total are per meter squared and were scaled by fractional areas
	NR_total_W_p_m2_vec[MapPixel_ID] = NR_total_W_p_m2;
	folder->VarDict["H_total_W_p_m2"] = H_total_W_p_m2;
	folder->VarDict["LE_total_W_p_m2"] = LE_total_W_p_m2;
	DeltaQ_W_p_m2_vec[MapPixel_ID] = DeltaQ_W_p_m2;

	//Store Anthropogenic Heat (W/m2) variables, total and for each land cover type
	AH_total_W_p_m2_vec[MapPixel_ID] = AH_total_W_p_m2;

	//Initialize Twetbulbglobe_C, Twetbulbnatural_C, Twetbulbpsychrometric_C, UTCI_C
	double Twetbulbglobe_C = 0;
	double Twetbulbnatural_C = 0;
	double Twetbulbpsychrometric_C = 0;
	double UTCI_C = 0;

	//Twetbulbglobe_C, Twetbulbnatural_C, Twetbulbpsychrometric_C, UTCI_C from call to HeatMetricsCalc.WetBulbGlobeTemperature_C_Calc
	//Note: UTCI_C is the Universal Thermal Climate Index, which is computed in HeatMetrics_Calc::UniversalThermalClimateIndex_C_Calc
	//Note: Radiation_Shortwave_Direct_W_p_m2 and Radiation_Shortwave_Diffuse_W_p_m2 for single or multiple stations
	//Note: Radiation_Shortwave_Direct_W_p_m2 adjusted for Cos_IncidenceAngle_Solar of terrain azimuth and slope
	//Note: WetBulbGlobeTemperature algorithm from Liljegren et al. (2008) code: https://github.com/mdljts/wbgt/blob/master/src/wbgt.c
	HeatMetricsCalc.WetBulbGlobeTemperature_C_Calc(input, folder, MapPixel_ID, timeStep, Tair_localCell_K, Tdew_localCell_K, Radiation_Shortwave_Direct_W_p_m2, Radiation_Shortwave_Diffuse_W_p_m2, Cos_IncidenceAngle_Solar, Twetbulbglobe_C, Twetbulbnatural_C, Twetbulbpsychrometric_C, UTCI_C);

	//Twetbulbglobe_K = Twetbulbglobe_C + 273.15; convert from C to K
	double Twetbulbglobe_K = Twetbulbglobe_C + 273.15;
	//Twetbulbglobe_K (K) is scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["Twetbulbglobe_K"] = Inputs::roundToDecimalPlaces(Twetbulbglobe_K, 2);

	//Twetbulbnatural_K = Twetbulbnatural_C + 273.15; convert from C to K
	double Twetbulbnatural_K = Twetbulbnatural_C + 273.15;
	//Twetbulbnatural_K (K) is scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["Twetbulbnatural_K"] = Inputs::roundToDecimalPlaces(Twetbulbnatural_K, 2);

	//Twetbulbpsychrometric_K = Twetbulbpsychrometric_C + 273.15; convert from C to K
	double Twetbulbpsychrometric_K = Twetbulbpsychrometric_C + 273.15;
	//Twetbulbpsychrometric_K (K) is scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["Twetbulbpsychrometric_K"] = Inputs::roundToDecimalPlaces(Twetbulbpsychrometric_K, 2);

	//UTCI_K = UTCI_C + 273.15; convert from C to F for Universal Thermal Climate Index 
	double UTCI_K = UTCI_C + 273.15;
	//UTCI_K (K) is scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["UTCI_K"] = Inputs::roundToDecimalPlaces(UTCI_K, 2);

	//HeatIndex_K from call to HeatMetricsCalc.HeatIndex_K_Calc function to compute heat index temperatures 
	double HeatIndex_K = HeatMetricsCalc.HeatIndex_K_Calc(input, folder, Tair_localCell_K, Tdew_localCell_K);
	//HeatIndex_K (K) is equivalent to the NWS calculated HeatIndex_NWS_K, scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["HeatIndex_K"] = Inputs::roundToDecimalPlaces(HeatIndex_K, 2);

	//Humidex_K from call to HeatMetricsCalc.Humidex_K_Calc function to compute humidex temperatures 
	double Humidex_K = HeatMetricsCalc.Humidex_K_Calc(input, folder, Tair_localCell_K, Tdew_localCell_K);
	//Humidex_K (K) is scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["Humidex_K"] = Inputs::roundToDecimalPlaces(Humidex_K, 2);
	
	//Initialize WindChill_K 
	double WindChill_K = 0;

	//WindChill_K returned from call to HeatMetricsCalc.WindChill_K_Calc
	//Note: Computed with Eq 3.64
	HeatMetricsCalc.WindChill_K_Calc(input, folder, timeStep, Tair_localCell_K, Tdew_localCell_K, WindChill_K);
	//Humidex_K (K) is scaled to 2 digits precision
	//Note: RHS function divides by 100 after taking integer of the term that takes product with 100 to round up
	folder->VarDict["WindChill_K"] = Inputs::roundToDecimalPlaces(WindChill_K, 2);

	//Activate WeatherProcessor* WeatherPro to write out WeatherPro variables
	WeatherProcessor* WeatherPro;
	if (input->SimulationNumericalParams["Flag_MultipleStations"] == 1) {
		folder->VarDict["Tair_weighted_K"] = WeatherPro->Tair_weighted_K;
		folder->VarDict["Tdew_weighted_K"] = WeatherPro->Tdew_weighted_K;
		folder->VarDict["Tair_mesoScale_weighted_K"] = WeatherPro->Tair_mesoScale_weighted_K;
		folder->VarDict["AbsHumidity_mesoScale_weighted_kg_p_m3"] = WeatherPro->AbsHumidity_mesoScale_weighted_kg_p_m3;
	}
}

//HeatFlux_InitializeVectors function initializes size of arrays
void HeatFluxCal::HeatFlux_InitializeVectors(Inputs* input) {
	// Initialize with correct size and value
	WaterDeltaQ_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0); 
	SoilDeltaQ_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	DeltaQ_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	NR_total_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	Water_AH_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	Soil_AH_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	SVeg_AH_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	Tree_AH_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	Imp_AH_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	AH_total_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	SVegDeltaQ_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	TreeDeltaQ_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
	ImpDeltaQ_W_p_m2_vec.assign(input->SimulationLocationParams["NumberOfLocations"], 0.0);
}
//NetRadiation_Prior_Calc function to update rolling values and return the new average
double HeatFluxCal::NetRadiation_Prior_Calc(int timeStep, int NR_vec_size, vector<double>& NetRadiation_W_p_m2_vec, double NetRadiation_W_p_m2)
{
	//NR_priorTS_W_p_m2 initialized to 0
	double NR_priorTS_W_p_m2 = 0;
	//If timeStep is zero return NR_priorTS_W_p_m2 as 0
	if (timeStep == 0) {
		return NR_priorTS_W_p_m2;
	}
	//Else If timeStep is 1 populate vector with uniform values
	else if (timeStep == 1) {
		//NetRadiation_W_p_m2_vec.assign(NR_vec_size , NetRadiation_W_p_m2) to assign initial values to vector
		NetRadiation_W_p_m2_vec.assign(NR_vec_size, NetRadiation_W_p_m2);
	}
	//Else for timeStep > 1
	else {
		//NetRadiation_W_p_m2_vec.erase(NetRadiation_W_p_m2_vec.begin()) to erase oldest element in vector
		NetRadiation_W_p_m2_vec.erase(NetRadiation_W_p_m2_vec.begin());
		//NetRadiation_W_p_m2_vec.push_back(NetRadiation_W_p_m2) to add newest element to vector
		NetRadiation_W_p_m2_vec.push_back(NetRadiation_W_p_m2);
	}

	//If not NetRadiation_W_p_m2_vec.empty() then compute and return the average, ensuring no division by zero
	if (!NetRadiation_W_p_m2_vec.empty()) {
		//NR_priorTS_W_p_m2 = accumulate(NetRadiation_W_p_m2_vec.begin(), NetRadiation_W_p_m2_vec.end(), 0.0) / NetRadiation_W_p_m2_vec.size();
		NR_priorTS_W_p_m2 = accumulate(NetRadiation_W_p_m2_vec.begin(), NetRadiation_W_p_m2_vec.end(), 0.0) / NetRadiation_W_p_m2_vec.size();
	}
	//return NR_priorTS_W_p_m2
	return NR_priorTS_W_p_m2;
}

