#include "RootZoneEvapoTranspirationCalc.h"

//Root zone evapotranspiration only removes water between field capacity and wilting point, referred to as capillarity held water
//Note: Root zone refers to soil below all pervious areas and any impervious area with tree cover, not only the soil land cover type.
//Note: Soil below any impervious area with tree cover is considered recharged via the lateral redistribution of groundwater. 

//Note: The model StatisticalHydro uses the input data of potential ET to determine atmospheric demand
//Note:	The model SpatialTemperatureHydro uses latent energy to determine atmospheric demand

//Note: EvapoTranspiration_SoilEvapZone_m (m) is reduced from potential ET by ratio of available soil moisture based on theory, including ...
//Note: ... ET_act = ET_pot * (soil storage)/(soil storage max) based on Saxton (1986) Eq in Fisher et al. (2005)
//Note: ... Fisher et al (2005) show potential ET based on Penmam-Monteith is too high relative to observed actual ET. 
//Note: ... The Eqs of Fisher et al. (2005) are not numbered, the same LHS variable is used for multiple Eqs of potential ET
//Note: ... Fisher et al (2005) give the adjustment in text near citation of Saxton (1986), as f(phi); improvement illustrated in Fig 2 vs Fig 1. 
//References:
//Fisher, J. B., DeBiase, T. A., Qi, Y., Xu, M., & Goldstein, A. H. (2005). Evapotranspiration models compared on a Sierra Nevada forest ecosystem. Environmental Modelling & Software, 20(6), 783-796. doi:https://doi.org/10.1016/j.envsoft.2004.04.009

//RootZoneEvapoTranspirationCalc::Calculate function handles StatisticalHydro and SpatialTemperatureHydro models
void RootZoneEvapoTranspirationCalc::Calculate(Inputs* input, DataFolder* folder, int timeStep)
{
	//initialize VarDict["iA"] as current TI_Value of topographic index; used for passing TI_Value to separate function while in loop
	int ia = int(folder->VarDict["iA"]);

	//Initialize LAI_BAI_Tree_m2_p_m2 and LAI_BAI_SVeg_m2_p_m2 
	double LAI_BAI_Tree_m2_p_m2 = input->LAI_BAI_Tree_m2_p_m2[timeStep];
	double LAI_BAI_SVeg_m2_p_m2 = input->LAI_BAI_SVeg_m2_p_m2[timeStep];
	//Note: If LAI_BAI_Tree_m2_p_m2 < 1, then set to 1 so that its position as denominator does not enlarge TreeLEE_W_p_m2 when / LAI_BAI_Tree_m2_p_m2
	if (LAI_BAI_Tree_m2_p_m2 < 1) { LAI_BAI_Tree_m2_p_m2 = 1; }
	//Note: If LAI_BAI_SVeg_m2_p_m2  < 1, then set to 1 so that its position as denominator does not enlarge SVegLEE_W_p_m2 when / LAI_BAI_SVeg_m2_p_m2 
	if (LAI_BAI_SVeg_m2_p_m2 < 1) { LAI_BAI_SVeg_m2_p_m2 = 1; }

	//EvapoTranspiration_SoilEvapZone_m (m) reset to zero at start of each TI value
	double EvapoTranspiration_SoilEvapZone_m = 0;
	double Storage_EvapoTransZone_m = 0;
	//StorageDeficit_FieldCapacity_SoilEvapZone_Max_m (m) is the maximum soil deficit for capillarity held water for ET; = Evapotranspiration_Depth_m * (Soil_FieldCapacity_m3pm3 - Soil_WiltingPoint_m3pm3)
	double StorageDeficit_FieldCapacity_SoilEvapZone_Max_m = folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"];
	
	//If Model is StatisticalHydro or Model is SpatialTemperatureHydro And timeStep equals 1, then call statistical calculation based on weather input data, given the spatial calculation has not yet been called
	if (input->SimulationStringParams["Model_Selection"] == "StatisticalHydro" || ((input->SimulationStringParams["Model_Selection"] == "SpatialTemperatureHydro" || input->SimulationStringParams["Model_Selection"] == "SpatialBufferwGI") && timeStep == 0)) {
		//EvapoTranspiration_SoilEvapZone_m (m) = potential ET read in from Evaporation.csv input file
		//Note: PotentialEvapotranspiration_TreeCover_m (m) computed for tree, yet used for tree and pervious areas 
		//Note: ... This is a model simplification for the 1st timestep, while in the next time steps cover types are represented
		//Note: Depths of evapotranspiration are relative to the area of analysis, and do not get scaled by fraction of cover type
		EvapoTranspiration_SoilEvapZone_m = input->PotentialEvapotranspiration_TreeCover_m[timeStep];
	}
	//Else If input->SimulationStringParams["Model_Selection"] == "StatisticalHydro" or (Model is "SpatialTemperatureHydro" etc. And timeStep > 0)
	else {
		//Temperature_Air_C (C) is converted from folder Tair_K (K) 
		double Temperature_Air_C = folder->VarDict["Tair_K"] - 273.15;

		//Calculate HeatMetrics with call to HeatMetrics_Calc functions
		//HeatMetrics_Calc HeatMetricsCalc(input) creates pointer to access HeatMetrics_Calc functions with sending input
		HeatMetrics_Calc HeatMetricsCalc(input);

		//DensityWater_kgpm3 = HeatMetricsCalc.Density_Water_kgpm3_Calc(folder->VarDict["Tair_K"])
		double DensityWater_kgpm3 = HeatMetricsCalc.Density_Water_kgpm3_Calc(folder->VarDict["Tair_K"]);

		//LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(Tair_K)
		double LatentHeat_Vaporization_JpkgK = HeatMetricsCalc.LatentHeat_Vaporization_JpkgK_Calc(folder->VarDict["Tair_K"]);

		//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) initialized
		double EvapoTranspiration_SoilEvapZone_m_p_s = 0;

		//If Storage_PerviousPondedWater_m or Storage_ImperviousPondedWater_m > Constant_1E_negative6 then presume TreeLEE_W_p_m2 needs to be reduced due to some used to evaporate depression storage
		if (folder->VarDict["Storage_PerviousPondedWater_m"] > Constant_1E_negative6 || 
			folder->VarDict["Storage_ImperviousPondedWater_m"] > Constant_1E_negative6) {
			//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) derived as latent energy (W/m2) divided by (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3)
			//Note: If depression storage received TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2, then TreeLET_W_p_m2 - (TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) remains
			//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
			//Note: Latent energy is (SoilLE_W_p_m2 * SoilCover_noTreeCanopy_frac + (SVegLET_W_p_m2 - SVegLET_W_p_m2 / LAI_BAI_SVeg_m2_p_m2) * ShortVegCover_noTreeCanopy_frac + ...
			//Note: ... (TreeLET_W_p_m2 - TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) * TreeCover_frac), and ...
			//Note: TreeCover_frac vs TreeCanopyCover_overPervious_frac is used given tree cover over pervious and impervious areas have roots within the soil layer 
			//Note: Soils covered by impervious may receive recharge from groundwater via lateral redistribution.
			//Note: RHS numerator is W/m2 = J/s/m2, and RHS denominator is (J/kg) * (kg/m3) = J/m3, therefore RHS quotient is J/s/m2 / J/m3 = m/s.
			EvapoTranspiration_SoilEvapZone_m_p_s = (folder->VarDict["SoilLE_W_p_m2"] * folder->ParamDict["SoilCover_noTreeCanopy_frac"] + folder->VarDict["ImpLE_W_p_m2"] * folder->ParamDict["PermeablePavementCover_noTreeCanopy_frac"] + (folder->VarDict["SVegLEE_W_p_m2"] - input->SafeDivide(folder->VarDict["SVegLEE_W_p_m2"], LAI_BAI_SVeg_m2_p_m2)) * folder->ParamDict["ShortVegCover_noTreeCanopy_frac"] + (folder->VarDict["TreeLET_W_p_m2"] - input->SafeDivide(folder->VarDict["TreeLET_W_p_m2"], LAI_BAI_Tree_m2_p_m2)) * folder->ParamDict["TreeCover_frac"]) / (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3);
		}
		//Else Storage_PerviousPondedWater_m <= Constant_1E_negative6 and use full amount of TreeLEE_W_p_m2 
		else {
			//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) derived as latent energy (W/m2) divided by (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3)
			//Note: If depression storage received TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2, then TreeLET_W_p_m2 - (TreeLET_W_p_m2 / LAI_BAI_Tree_m2_p_m2) remains
			//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
			//Note: Latent energy is (SoilLE_W_p_m2 * SoilCover_noTreeCanopy_frac + SVegLET_W_p_m2 * ShortVegCover_noTreeCanopy_frac + ...
			//Note: ... TreeLET_W_p_m2 * TreeCover_frac), and ...
			//Note: TreeCover_frac is used given tree cover over pervious and impervious areas have roots within the soil layer 
			//Note: Soils covered by impervious may receive recharge from groundwater via lateral redistribution.
			//Note: RHS numerator is W/m2 = J/s/m2, and RHS denominator is (J/kg) * (kg/m3) = J/m3, therefore RHS quotient is J/s/m2 / J/m3 = m/s.
			EvapoTranspiration_SoilEvapZone_m_p_s = (folder->VarDict["SoilLE_W_p_m2"] * folder->ParamDict["SoilCover_noTreeCanopy_frac"] + folder->VarDict["ImpLE_W_p_m2"] * folder->ParamDict["PermeablePavementCover_noTreeCanopy_frac"] + folder->VarDict["SVegLET_W_p_m2"] * folder->ParamDict["ShortVegCover_noTreeCanopy_frac"] + folder->VarDict["TreeLET_W_p_m2"] * folder->ParamDict["TreeCover_frac"]) / (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3);
		}

		//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) derived as latent energy (W/m2) divided by (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3)
		//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
		//Note: Latent energy is (SoilLE_W_p_m2 * SoilCover_noTreeCanopy_frac + SVegLET_W_p_m2 * ShortVegCover_noTreeCanopy_frac + ...
		//Note: ... TreeLET_W_p_m2 + TreeCover_frac), and ...
		//Note: TreeCover_frac is used given tree cover over pervious and impervious areas have roots within the soil layer 
		//Note: Soils covered by impervious may receive recharge from groundwater via lateral redistribution.
		//Note: RHS numerator is W/m2 = J/s/m2, and RHS denominator is (J/kg) * (kg/m3) = J/m3, therefore RHS quotient is J/s/m2 / J/m3 = m/s.
		EvapoTranspiration_SoilEvapZone_m_p_s = (folder->VarDict["SoilLE_W_p_m2"] * folder->ParamDict["SoilCover_noTreeCanopy_frac"] + folder->VarDict["ImpLE_W_p_m2"] * folder->ParamDict["PermeablePavementCover_noTreeCanopy_frac"] + folder->VarDict["SVegLET_W_p_m2"] * folder->ParamDict["ShortVegCover_noTreeCanopy_frac"] + folder->VarDict["TreeLET_W_p_m2"] * folder->ParamDict["TreeCover_frac"]) / (LatentHeat_Vaporization_JpkgK * DensityWater_kgpm3);

		//EvapoTranspiration_SoilEvapZone_m_p_s (m/s) next divided by pervious area + tree over impervious so it is scaled to cover area
		EvapoTranspiration_SoilEvapZone_m_p_s = input->SafeDivide(EvapoTranspiration_SoilEvapZone_m_p_s, (folder->ParamDict["SoilCover_noTreeCanopy_frac"] + folder->ParamDict["PermeablePavementCover_noTreeCanopy_frac"] + folder->ParamDict["ShortVegCover_noTreeCanopy_frac"] + folder->ParamDict["TreeCover_frac"]));

		//Control for negative values
		EvapoTranspiration_SoilEvapZone_m_p_s = Inputs::forceZeroIfLessThanOrAlmostEqualZero(EvapoTranspiration_SoilEvapZone_m_p_s, Epsilon_Tolerance_1E_negative15);

		//EvapoTranspiration_SoilEvapZone_m (m) is product of EvapoTranspiration_SoilEvapZone_m_p_s (m/s) and SimulationTimeStep_Duration_sec[timeStep]
		EvapoTranspiration_SoilEvapZone_m = EvapoTranspiration_SoilEvapZone_m_p_s * input->SimulationTimeStep_Duration_sec[timeStep];
	}

	//If Type equals GI do not proceed with updates to topographic index (TI) terms
	if (folder->ParamStringDict["Type"] != "BulkArea") {
		//Storage_EvapoTransZone_m (m) = GI water depth available for ET, defined in Inflow_StormwaterDevice.cpp as folder->VarDict["Storage_EvapoTransZone_m"]
		Storage_EvapoTransZone_m = folder->VarDict["Storage_EvapoTransZone_m"];
		//EvapoTranspiration_SoilEvapZone_TI_m (m) equals potential ET read in from Evaporation.csv input file
		folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] = EvapoTranspiration_SoilEvapZone_m;

		//If the EvapoTranspiration_SoilEvapZone_TI_m (m) is greater than Storage_EvapoTransZone_m (m), then reset to available soil evapotranspiration zone water
		if (folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] > Storage_EvapoTransZone_m) {
			//EvapoTranspiration_SoilEvapZone_TI_m (m) is limited to Storage_EvapoTransZone_m (m), available soil evapotranspiration zone water
			folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] = Storage_EvapoTransZone_m;
		}
	}

	//if Type equals BulkArea then 
	if (folder->ParamStringDict["Type"] == "BulkArea") {
		//StorageDeficit_FieldCapacity_SoilEvapZone_Max_m (m) is the maximum soil deficit for capillarity held water for ET, defined in HydroPlusConfig.xml
		StorageDeficit_FieldCapacity_SoilEvapZone_Max_m = folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"];

		//Initialize vectors at start of simulation 
		if (timeStep == 0) {
			//If EvapoTranspiration_SoilEvapZone_TI_m size() is less than or equal to ia then populate vector with zero
			if (folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"].size() <= ia) {
				//EvapoTranspiration_SoilEvapZone_TI_m (m) is soil evapotranspiration, initialized to 0
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"].push_back(0.0);
			}
		}
		//EvapoTranspiration_SoilEvapZone_TI_m (m) reset to zero at start of each TI_Value vector
		folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = 0.0;

		//if EvapoTranspiration_SoilEvapZone_m > 0 then remove water from SoilEvapZone
		if (EvapoTranspiration_SoilEvapZone_m > 0.0) {
			//If StorageDeficit_FieldCapacity_SoilEvapZone_Max_m greater than zero, then enter for division by StorageDeficit_FieldCapacity_SoilEvapZone_Max_m
			if (StorageDeficit_FieldCapacity_SoilEvapZone_Max_m > 0) {
				//EvapoTranspiration_SoilEvapZone_TI_m (m) is equal to EvapoTranspiration_SoilEvapZone_m when StorageDeficit_FieldCapacity_SoilEvapZone_TI_m = 0, no capacity remains
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = EvapoTranspiration_SoilEvapZone_m * (1 - (folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] / StorageDeficit_FieldCapacity_SoilEvapZone_Max_m));
				//Storage_EvapoTransZone_m (m) = StorageDeficit_FieldCapacity_SoilEvapZone_Max_m - StorageDeficit_FieldCapacity_SoilEvapZone_TI_m; 
				//	Example: StorageDeficit_FieldCapacity_SoilEvapZone_Max_m = 0.1 m, StorageDeficit_FieldCapacity_SoilEvapZone_TI_m = 0.08 m, then Storage_EvapoTransZone_m = 0.1 - 0.08 = 0.02 m
				Storage_EvapoTransZone_m = StorageDeficit_FieldCapacity_SoilEvapZone_Max_m - folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia];
			}
			else {
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = 0;
				//Storage_EvapoTransZone_m (m) equals zero
				Storage_EvapoTransZone_m = 0;
			}

			//If the EvapoTranspiration_SoilEvapZone_TI_m (m) is greater than Storage_EvapoTransZone_m (m), then reset to available soil evapotranspiration zone water
			if (folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] > Storage_EvapoTransZone_m) {
				//EvapoTranspiration_SoilEvapZone_TI_m (m) is limited to Storage_EvapoTransZone_m (m), available soil evapotranspiration zone water
				folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] = Storage_EvapoTransZone_m;
			}
		}

		//StorageDeficit_FieldCapacity_SoilEvapZone_TI_m (m) is increased by removal of EvapoTranspiration_SoilEvapZone_TI_m
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_TI_m decreases due to Infiltration_TI_m in DrainageToEvapotranspirationZone.cpp
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_TI_m is a deficit, aka capacity; all TI bins have same thickness
		folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] = folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] + folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia];

		//EvapoTranspiration_SoilEvapZone_m (m) = sum EvapoTranspiration_SoilEvapZone_TI_m for each TI bin for time step, weighted by watershed fraction
		//	HydroPlus weights discrete TI parameter values, i.e, SoilParameter[ia], with TI_binArea_frac, catchment area bounding discrete value
		folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] = folder->VarDict["EvapoTranspiration_SoilEvapZone_m"] + folder->VarVecDict["EvapoTranspiration_SoilEvapZone_TI_m"][ia] * folder->VarDict["TI_binArea_frac"];

		//StorageDeficit_FieldCapacity_SoilEvapZone_m (m) = sum StorageDeficit_FieldCapacity_SoilEvapZone_TI_m 
		//Note: Summing for each TI bin for time step, weighted by watershed fraction, TI_binArea_frac
		//Note: StorageDeficit_FieldCapacity_SoilEvapZone_m ranges frm 0=saturated, 1=dry; available storage up to field capacity
		//HydroPlus weights discrete TI parameter values, i.e, SoilParameter[ia], with TI_binArea_frac, catchment area bounding discrete value
		folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_m"] = folder->VarDict["StorageDeficit_FieldCapacity_SoilEvapZone_m"] + folder->VarVecDict["StorageDeficit_FieldCapacity_SoilEvapZone_TI_m"][ia] * folder->VarDict["TI_binArea_frac"];
	}
	//Ending of BulkArea calculations
}
