#include "WaterBalance_Summary.h"
#include <iomanip>
#include <iostream>

//Note: Snow depths are liquid depths, also known as snow water equivalent

//WaterBalance_Summary::writeWaterBalanceFile function computes water balance terms for all folders in catchment
void WaterBalance_Summary::writeWaterBalanceFile(Inputs* input)
{
	//Create output file for summary of catchment water balance
	string file_path, file_path_csv;
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "Summary_Budget_Water_mm.dat";
	file_path_csv = input->SimulationStringParams["OutputFolder_Path"] + "Summary_Budget_Water_mm.csv";

	//ofstream function creates outfile_WaterBudget_dat to the file path
	//ios_base::app opens in append mode, meaning it sets position indicator to the end before each output operation.
	ofstream outfile_WaterBudget_dat(file_path);  // Use default truncate mode
	ofstream outfile_WaterBudget_csv(file_path_csv);
	//If does not exist outfile_WaterBudget_dat, then 
	if (!outfile_WaterBudget_dat || !outfile_WaterBudget_csv) {
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	
	outfile_WaterBudget_dat.setf(ios::fixed);
	outfile_WaterBudget_dat.setf(ios::showpoint);
	//Folder_BulkArea_ID = 0; default setting
	int Folder_BulkArea_ID = 0;
	//Folder_GI_ID = 1; default setting
	int Folder_GI_ID = 1;
	vector<int> Folder_List;

	//Folder_List is defined to contain Folder_BulkArea_ID
	Folder_List = { Folder_BulkArea_ID };
	//If Flag_GI_Simulated equals 1 then add GI to Folder_List
	if (input->RepoDict["Flag_GI_Simulated"] == 1) {
		//Folder_List is redefined to contain Folder_BulkArea_ID and Folder_GI_ID
		Folder_List = { Folder_BulkArea_ID, Folder_GI_ID };
	}
	
	//For Loop with auto of Folder_Type_ID within Folder_List, as for (auto Folder_Type_ID : Folder_List)
	//Note: Loop can represent values of BulkArea and/or GI, and each writes output in 2 formats of .DAT and .CSV
	//Note: Loop has 2 instances of Summary_Budget_Water_mm for each format (.DAT, .CSV), 1st. Catchment when BA & GI, 2nd. Folder BA &/or GI
	//Note: Alternatively, use for (int Folder_Type_ID = 0; Folder_Type_ID < (Folder_GI_ID + 1); ++Folder_Type_ID ) {
	for (auto Folder_Type_ID : Folder_List) {

		//Zero_Threshold_mm defined to turn values to absolute zero if below that amount
		double Zero_Threshold_mm = 0.001;
		//CA_GI_m2 is GI folder area initialized to zero
		double CA_GI_m2 = 0.0;
		//Name_Folder_Type_str initialized as string
		string Name_Folder_Type_str;
		//Length_Name_Folder_Type initialized as zero
		int Length_Name_Folder_Type = 0;
		//If Folder_Type_ID equals zero and Folder_List.size() == 1 then only BulkArea simulated, so call it CatchmentArea
		if (Folder_Type_ID == 0 && Folder_List.size() == 1) {
			//Name_Folder_Type_str set to Catchment
			Name_Folder_Type_str = "Catchment";
			//Length_Name_Folder_Type set to actual length
			Length_Name_Folder_Type = Name_Folder_Type_str.length();
			//CA_GI_m2 equal to 1 to avoid division by zero
			CA_GI_m2 = 1;
		}
		//If Folder_Type_ID equals zero and Folder_List.size() > 1 then more than BulkArea simulated, so call it BulkArea
		else if (Folder_Type_ID == 0 && Folder_List.size() > 1) {
			//Name_Folder_Type_str set to BulkArea
			Name_Folder_Type_str = "BulkArea";
			//Length_Name_Folder_Type set to actual length
			Length_Name_Folder_Type = Name_Folder_Type_str.length();
			//CA_GI_m2 equal to 1 to avoid division by zero
			CA_GI_m2 = 1;
		}
		//If Folder_Type_ID equals 1 then
		if (Folder_Type_ID == 1) {
			//Name_Folder_Type_str set to BulkArea
			Name_Folder_Type_str = "GreenInfrastructure";
			//Length_Name_Folder_Type set to actual length
			Length_Name_Folder_Type = Name_Folder_Type_str.length();
			//CA_GI_m2 equal to Catchment_Type_Area_m2 as function of Folder_Type_ID = 1
			CA_GI_m2 = input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
		}

		//CA_m2 is catchment area, CatchmentArea_m2 from HydroPlusConfig.xml
		double CA_m2 = input->SimulationNumericalParams["CatchmentArea_m2"];
		//CA_PC_frac is catchment pervious area fraction
		double CA_PC_frac = input->RepoDict["Catchment_TotalPerviousCover_frac"];
		//FA_m2 is folder area, Catchment_Type_Area_m2 as function of Folder_Type_ID
		double FA_m2 = input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
		//PC_m2 is pervious cover fraction of catchment area
		double PC_m2 = input->RepoVecDict["TotalPerviousCover_Type_m2"][Folder_Type_ID];
		//IC_m2 is impervious cover fraction of catchment area
		double IC_m2 = input->RepoVecDict["TotalImperviousCover_Type_m2"][Folder_Type_ID];
		//TC_m2 is tree cover fraction of catchment area
		double TC_m2 = input->RepoVecDict["TotalTreeCover_Type_m2"][Folder_Type_ID];
		//TC_overPC_m2 is tree cover over pervious cover fraction of catchment area
		double TC_overPC_m2 = input->RepoVecDict["TreeCanopyCover_overPervious_Type_m2"][Folder_Type_ID];
		//TC_overIC_m2 is tree cover over impervious cover fraction of catchment area
		double TC_overIC_m2 = input->RepoVecDict["TreeCanopyCover_overImpervious_Type_m2"][Folder_Type_ID];
		//IC_noTC_m2 is impervious cover not under tree cover fraction of catchment area
		double IC_noTC_m2 = input->RepoVecDict["ImperviousCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//PC_noTC_m2 is pervious area without tree cover 
		double PC_noTC_m2 = PC_m2 - TC_overPC_m2;
		//SVeg_noTC_m2 is short vegetation cover not under tree cover fraction of catchment area
		double SVegC_noTC_m2 = input->RepoVecDict["ShortVegCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//Soil_noTC_m2 is soil cover not under tree cover fraction of catchment area
		double SoilC_noTC_m2 = input->RepoVecDict["SoilCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//PermeablePavementC_noTC_m2 is permeable pavement cover not under tree cover area (m2) of catchment area
		double PermeablePavementC_noTC_m2 = input->RepoVecDict["PermeablePavementCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//Water_noTC_m2 is water cover not under tree cover fraction of catchment area
		double WaterC_noTC_m2 = input->RepoVecDict["WaterCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//DCIA_m2 is directly connected impervious cover fraction of impervious area
		double DCIA_m2 = input->RepoVecDict["DCIA_Type_m2"][Folder_Type_ID];
		//InfilExSim_m2 is infiltration excess governed area fraction of catchment area
		double InfilExSim_m2 = input->RepoVecDict["InfiltExcessGovernedArea_Type_m2"][Folder_Type_ID];
		//Macropore_m2 is macropore fraction of catchment area
		double MacroPore_m2 = input->RepoVecDict["MacroPore_Type_m2"][Folder_Type_ID];
		//NoCanopy_m2 is area without tree or short vegetation canopy cover
		double NoCanopy_m2 = SoilC_noTC_m2 + PermeablePavementC_noTC_m2 + IC_noTC_m2 + WaterC_noTC_m2;
		//double NoCanopy_m2 = Folder_Area_m2 - TC_m2 - SVegC_noTC_m2;
	
		double PC_frac = 0.0;
		double IC_frac = 0.0;
		double TC_frac = 0.0;
		double TC_overIC_frac = 0.0;
		double TC_overPC_frac = 0.0;
		double IC_noTC_frac = 0.0;
		double SVegC_noTC_frac = 0.0;
		double SoilC_noTC_frac = 0.0;
		double PermeablePavementC_noTC_frac = 0.0;
		double WaterC_noTC_frac = 0.0;
		double InfilExSim_frac = 0.0;
		double DCIA_frac = 0.0;
		double MacroPore_frac = 0.0;
		double PC_noTC_frac = 0.0;
		double Ratio_TCoverIC_to_TC = 0.0;
		double Ratio_TCoverPC_to_TC = 0.0;
		double Ratio_ICnoTC_to_noCanopy = 0.0;
		double ratio_WC_to_noCanopy = 0.0;
		double Groundwater_surficial_TS_first_frac = 0.0;
		double Groundwater_surficial_TS_last_frac = 0.0;

		//If Catchment_Type_Area_m2 (m2) > 0 then divide by Catchment_Type_Area_m2 (m2) and avoid division by zero
		if (input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID] > 0) {
			//TC_overIC_frac  (frac) = Inputs::roundToDecimalPlaces((TreeCanopyCover_overImpervious_Type_m2 / Catchment_Type_Area_m2), 3)
			//Note: RHS function divides by 1000 after taking integer of the term that takes product with 1000 and sums 0.5 to round up
			TC_overIC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["TreeCanopyCover_overImpervious_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			TC_overPC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["TreeCanopyCover_overPervious_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			IC_noTC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["ImperviousCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			SVegC_noTC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["ShortVegCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			SoilC_noTC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["SoilCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			PermeablePavementC_noTC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["PermeablePavementCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			WaterC_noTC_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["WaterCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);

			//Note: Sum land cover groups after rounding sub-components above to desired precision
			//PC_frac (frac) = TC_overPC_frac + SVegC_noTC_frac + SoilC_noTC_frac + PermeablePavementC_noTC_frac
			PC_frac = TC_overPC_frac + SVegC_noTC_frac + SoilC_noTC_frac + PermeablePavementC_noTC_frac;
			//IC_frac (frac) = TC_overIC_frac + IC_noTC_frac
			IC_frac = TC_overIC_frac + IC_noTC_frac;
			//TC_frac (frac) = TC_overIC_frac + TC_overPC_frac
			TC_frac = TC_overIC_frac + TC_overPC_frac;

			InfilExSim_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["InfiltExcessGovernedArea_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			DCIA_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["DCIA_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			MacroPore_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["MacroPore_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID], 3);
			Groundwater_surficial_TS_first_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["Groundwater_surficial_TS_first_frac"][Folder_Type_ID], 3);
			Groundwater_surficial_TS_last_frac = Inputs::roundToDecimalPlaces(input->RepoVecDict["Groundwater_surficial_TS_last_frac"][Folder_Type_ID], 3);
			//Clamp to ensure variable is between 0 and 1 
			MacroPore_frac = clamp(MacroPore_frac, 0.0, 1.0);
		}

		//PC_noTC_frac (fraction) is pervious area without tree cover, PC_frac minus TC_overPC_frac
		PC_noTC_frac = PC_frac - TC_overPC_frac;
		//If TC_m2 > 0 then divide by TC_m2
		if (TC_m2 > 0) {
			Ratio_TCoverIC_to_TC = input->RepoVecDict["TreeCanopyCover_overImpervious_Type_m2"][Folder_Type_ID] / TC_m2;
			Ratio_TCoverPC_to_TC = input->RepoVecDict["TreeCanopyCover_overPervious_Type_m2"][Folder_Type_ID] / TC_m2;
		}
		//If NoCanopy_m2 > 0 then divide by NoCanopy_m2
		if (NoCanopy_m2 > 0) {
			Ratio_ICnoTC_to_noCanopy = input->RepoVecDict["ImperviousCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / NoCanopy_m2;
			ratio_WC_to_noCanopy = input->RepoVecDict["WaterCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / NoCanopy_m2;
		}

		double TCanCover_WaterBalance_m3, SVegCanCover_WaterBalance_m3, ImperviousCover_WaterBalance_m3, PerviousCover_WaterBalance_m3, WaterCover_WaterBalance_m3;
		//Establish variables for BulkArea water transferred to GI
		//Flux_BA_to_GI_m3 (m3) is sum of volumes transferred from BulkArea to GI
		double Flux_BA_to_GI_m3 = input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID] +
			input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] + input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID];
		//Flux_BA_to_GI_Impervious_m3 (m3) is volume of water transferred from BulkArea directly connected impervious area to GI
		double Flux_BA_to_GI_Impervious_m3 = input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID];
		//Flux_BA_to_GI_Pervious_m3 (m3) is volume of water transferred from BulkArea pervious area to GI
		double Flux_BA_to_GI_Pervious_m3 = input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] + input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID];
		//Flux_BA_to_GI_Water_m3 (m3) is volume of water transferred from BulkArea water area to GI
		double Flux_BA_to_GI_Water_m3 = input->RepoVecDict["Runoff_Water_BAtoGI_Sum_m3"][Folder_Type_ID];

		//Establish variables for GI water storage and fluxes
		double Evaporation_Surface_GI_m3 = input->RepoDict["GI_Evaporation_Surface_TS_Sum_m3"];
		double Evaporation_Subsurface_GI_m3 = input->RepoDict["GI_Evaporation_Pavement_TS_Sum_m3"] + input->RepoDict["GI_Evaporation_Soil_TS_Sum_m3"] +
			input->RepoDict["GI_Evaporation_Vault_TS_Sum_m3"];
		double Storage_Surface_GI_m3 = input->RepoDict["GI_Storage_Surface_TS_Sum_m3"];
		double Storage_Subsurface_GI_m3 = (input->RepoDict["GI_Storage_Soil_TS_last_m3"] - input->RepoDict["GI_Storage_Soil_TS_first_m3"]) + input->RepoDict["GI_PavementWater_TS_Sum_m3"] +
			input->RepoDict["GI_Storage_Vault_TS_Sum_m3"];
		double Runoff_Surface_GI_m3 = input->RepoDict["GI_Runoff_Surface_TS_Sum_m3"];
		double Runoff_Vault_GI_m3 = input->RepoDict["GI_Runoff_VaultDrain_TS_Sum_m3"];
		double Drainage_VadoseZone_GI_m3 = input->RepoDict["GI_Drainage_Exfiltration_TS_Sum_m3"];

		//======================================================
		//Eq_GI_RunonSurface_WaterBalance_mm (m3) includes BulkARea to GI surface transfer
		double GI_Runon_from_BulkArea_WaterCoverRunoff_mm, GI_Runon_from_BulkArea_ImpCoverRunoff_mm, GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm, GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm,
			GI_Runon_from_GI_ImpCoverRunoff_mm, GI_Runon_from_GI_PerCoverRunoff_mm, GI_Runon_from_GI_VaultRunoff_mm, 
			GI_Runon_from_GI_WaterCoverRunoff_mm, GI_Runon_from_BulkArea_Sum_mm, GI_Runon_from_GI_Sum_mm,
			Eq_GI_RunonSurface_WaterBalance_mm;
		GI_Runon_from_BulkArea_WaterCoverRunoff_mm = input->RepoVecDict["Runoff_Water_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_BulkArea_ImpCoverRunoff_mm = input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm = input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm = input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_GI_WaterCoverRunoff_mm = input->RepoVecDict["Runoff_Water_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_GI_ImpCoverRunoff_mm = input->RepoVecDict["Runoff_Impervious_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_GI_PerCoverRunoff_mm = input->RepoVecDict["Runoff_Pervious_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_GI_VaultRunoff_mm = input->RepoVecDict["Runoff_Vault_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_BulkArea_Sum_mm = input->RepoDict["GI_Runon_from_BulkArea_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Runon_from_GI_Sum_mm = GI_Runon_from_GI_WaterCoverRunoff_mm + GI_Runon_from_GI_ImpCoverRunoff_mm + GI_Runon_from_GI_PerCoverRunoff_mm +
			GI_Runon_from_GI_VaultRunoff_mm;
		//Balance is simple input = change in storage + output(s)
		Eq_GI_RunonSurface_WaterBalance_mm = GI_Runon_from_BulkArea_WaterCoverRunoff_mm + GI_Runon_from_BulkArea_ImpCoverRunoff_mm + 
			GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm + GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm + 
			GI_Runon_from_GI_WaterCoverRunoff_mm + GI_Runon_from_GI_ImpCoverRunoff_mm + GI_Runon_from_GI_PerCoverRunoff_mm + 
			GI_Runon_from_GI_VaultRunoff_mm - GI_Runon_from_BulkArea_Sum_mm - GI_Runon_from_GI_Sum_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_GI_RunonSurface_WaterBalance_mm) < Zero_Threshold_mm) { Eq_GI_RunonSurface_WaterBalance_mm = abs(Eq_GI_RunonSurface_WaterBalance_mm); }
		input->RepoDict["Eq_GI_RunonSurface_WaterBalance_mm"] = Eq_GI_RunonSurface_WaterBalance_mm;

		//======================================================
		//Eq_GI_Surface_WaterBalance_mm (mm) includes surface GI components
		double GI_Precipitation_Sum_mm, GI_Infiltration_Sum_mm, GI_RunoffSurface_Sum_mm, GI_Storage_Surface_TS_Last_mm,
			GI_EvaporationSurface_Sum_mm, Eq_GI_Surface_WaterBalance_mm;
		GI_Precipitation_Sum_mm = input->RepoDict["GI_Precipitation_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Infiltration_Sum_mm = input->RepoDict["GI_Infiltration_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_RunoffSurface_Sum_mm = input->RepoDict["GI_Runoff_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Storage_Surface_TS_Last_mm = input->RepoDict["GI_Storage_Surface_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_EvaporationSurface_Sum_mm = input->RepoDict["GI_Evaporation_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)
		Eq_GI_Surface_WaterBalance_mm = GI_Precipitation_Sum_mm + GI_Runon_from_BulkArea_Sum_mm + GI_Runon_from_GI_Sum_mm - GI_Infiltration_Sum_mm - GI_RunoffSurface_Sum_mm - GI_Storage_Surface_TS_Last_mm - GI_EvaporationSurface_Sum_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_GI_Surface_WaterBalance_mm) < Zero_Threshold_mm) { Eq_GI_Surface_WaterBalance_mm = abs(Eq_GI_Surface_WaterBalance_mm); }
		input->RepoDict["Eq_GI_Surface_WaterBalance_mm"] = Eq_GI_Surface_WaterBalance_mm;

		//======================================================
		//Eq_GI_All_Layers_WaterBalance_mm (mm) includes all GI components; 
		//Note: HydroPlusConfig.xml element Soil_MoistureInitial_m3pm3 enables a non-zero initial condition for soil moisture
		//Note: Balance appears to require adding back the average difference of GI_GroundwaterMound_TS_Last_mm and GI_Drainage_VadoseZone_Sum_mm, called GI_GoundwaterMound_DynamicEquilibrium_mm
		double GI_RunoffVaultDrain_Sum_mm, GI_Drainage_VadoseZone_Sum_mm, GI_PavementWater_TS_Last_mm, GI_Storage_Soil_TS_last_mm, GI_Storage_Soil_TS_0th_mm, GI_Storage_Vault_TS_Last_mm, GI_EvaporationPavement_Sum_mm, GI_EvaporationSoil_Sum_mm, GI_EvaporationVault_Sum_mm, GI_GroundwaterMound_TS_0th_mm, GI_GroundwaterMound_TS_Last_mm, 			GI_GoundwaterMound_DynamicEquilibrium_mm, Eq_GI_All_Layers_WaterBalance_mm;
		GI_RunoffVaultDrain_Sum_mm = input->RepoDict["GI_Runoff_VaultDrain_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Drainage_VadoseZone_Sum_mm = input->RepoDict["GI_Drainage_VadoseZone_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_PavementWater_TS_Last_mm = input->RepoDict["GI_PavementWater_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Storage_Soil_TS_last_mm = input->RepoDict["GI_Storage_Soil_TS_last_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Storage_Soil_TS_0th_mm = input->RepoDict["GI_Storage_Soil_TS_0th_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_Storage_Vault_TS_Last_mm = input->RepoDict["GI_Storage_Vault_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_EvaporationPavement_Sum_mm = input->RepoDict["GI_Evaporation_Pavement_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_EvaporationSoil_Sum_mm = input->RepoDict["GI_Evaporation_Soil_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_EvaporationVault_Sum_mm = input->RepoDict["GI_Evaporation_Vault_Sum_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_GroundwaterMound_TS_0th_mm = input->RepoDict["GI_Storage_GroundwaterMound_TS_0th_m3"] / FA_m2 * Ratio_mm_to_m;
		GI_GroundwaterMound_TS_Last_mm = input->RepoDict["GI_Storage_GroundwaterMound_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m;
		//GI_GoundwaterMound_DynamicEquilibrium_mm initialized to zero, and given a larger value if mounding active
		GI_GoundwaterMound_DynamicEquilibrium_mm = 0;
		//If GI_GroundwaterMound_TS_Last_mm > 0 then Eq_GI_All_Layers_WaterBalance_mm balance appears to require adding back GI_GoundwaterMound_DynamicEquilibrium_mm
		if (GI_GroundwaterMound_TS_Last_mm > 0) {
			//GI_GoundwaterMound_DynamicEquilibrium_mm  (mm) determined by analy
			GI_GoundwaterMound_DynamicEquilibrium_mm = (GI_Drainage_VadoseZone_Sum_mm - (GI_GroundwaterMound_TS_Last_mm - GI_GroundwaterMound_TS_0th_mm)) * 0.5;
		}

		//Balance is simple input = change in storage + output(s)
		Eq_GI_All_Layers_WaterBalance_mm = GI_Precipitation_Sum_mm + GI_Runon_from_BulkArea_Sum_mm + GI_Runon_from_GI_Sum_mm - GI_RunoffSurface_Sum_mm - GI_RunoffVaultDrain_Sum_mm - GI_Drainage_VadoseZone_Sum_mm - GI_Storage_Surface_TS_Last_mm - GI_PavementWater_TS_Last_mm - (GI_Storage_Soil_TS_last_mm - GI_Storage_Soil_TS_0th_mm) - GI_Storage_Vault_TS_Last_mm - GI_EvaporationSurface_Sum_mm - GI_EvaporationPavement_Sum_mm - GI_EvaporationSoil_Sum_mm - GI_EvaporationVault_Sum_mm + GI_GoundwaterMound_DynamicEquilibrium_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_GI_All_Layers_WaterBalance_mm) < Zero_Threshold_mm) { Eq_GI_All_Layers_WaterBalance_mm = abs(Eq_GI_All_Layers_WaterBalance_mm); }
		input->RepoDict["Eq_GI_All_Layers_WaterBalance_mm"] = Eq_GI_All_Layers_WaterBalance_mm;

		//======================================================
		//Eq_Tree_Canopy_WaterBalance_mm (mm), relative to Folder area
		double FA_TC_Total_Precip_RainSWE_mm, FA_TC_Total_Irrigation_mm, FA_TC_Evap_mm, FA_TC_Storage_Canopy_TS_last_mm, FA_TC_ThroughFall_mm, Eq_Tree_Canopy_WaterBalance_mm;
		FA_TC_Total_Precip_RainSWE_mm = input->TotalPrecip_m * Ratio_mm_to_m * TC_frac;
		FA_TC_Total_Irrigation_mm = input->RepoVecDict["Irrigation_TreeInterceptionStorage_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_TC_Evap_mm = input->RepoVecDict["EvaporationSublimation_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_TC_Storage_Canopy_TS_last_mm = input->RepoVecDict["StorageRain_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m + 
			input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_TC_ThroughFall_mm = input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)
		Eq_Tree_Canopy_WaterBalance_mm = FA_TC_Total_Precip_RainSWE_mm + FA_TC_Total_Irrigation_mm - FA_TC_Storage_Canopy_TS_last_mm - FA_TC_Evap_mm - FA_TC_ThroughFall_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_Tree_Canopy_WaterBalance_mm) < Zero_Threshold_mm) { Eq_Tree_Canopy_WaterBalance_mm = abs(Eq_Tree_Canopy_WaterBalance_mm); }
		input->RepoDict["Eq_Tree_Canopy_WaterBalance_mm"] = Eq_Tree_Canopy_WaterBalance_mm;

		//======================================================
		//Eq_SVeg_Canopy_WaterBalance_mm (mm), relative to Folder area
		double FA_SV_Total_Precip_RainSWE_mm, FA_SV_Total_Irrigation_mm, FA_SV_Evap_mm, FA_SV_Storage_Canopy_TS_last_mm, FA_SV_ThroughFall_mm, Eq_SVeg_Canopy_WaterBalance_mm;
		FA_SV_Total_Precip_RainSWE_mm = input->TotalPrecip_m * Ratio_mm_to_m * SVegC_noTC_frac;
		FA_SV_Total_Irrigation_mm = input->RepoVecDict["Irrigation_SVegInterceptionStorage_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_SV_Evap_mm = input->RepoVecDict["EvaporationSublimation_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_SV_Storage_Canopy_TS_last_mm = input->RepoVecDict["StorageRain_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m + 
			input->RepoVecDict["StorageSnow_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_SV_ThroughFall_mm = input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)
		Eq_SVeg_Canopy_WaterBalance_mm = FA_SV_Total_Precip_RainSWE_mm + FA_SV_Total_Irrigation_mm - FA_SV_Storage_Canopy_TS_last_mm - FA_SV_Evap_mm - FA_SV_ThroughFall_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_SVeg_Canopy_WaterBalance_mm) < Zero_Threshold_mm) { Eq_SVeg_Canopy_WaterBalance_mm = abs(Eq_SVeg_Canopy_WaterBalance_mm); }
		input->RepoDict["Eq_SVeg_Canopy_WaterBalance_mm"] = Eq_SVeg_Canopy_WaterBalance_mm;

		//======================================================
		//Eq_Snow_WaterBalance_mm (mm), relative to Folder area
		double FA_Precip_Snow_mm, FA_Total_Sublimation_Ground_Snow_mm, FA_Total_Sublimation_Canopy_Snow_mm, FA_Total_Melt_Ground_Snow_mm, FA_Total_Melt_Canopy_Snow_mm, FA_Total_Storage_Ground_Snow_mm, FA_TS_Last_Canopy_Snow_mm, Eq_Snow_WaterBalance_mm;
		FA_Precip_Snow_mm = input->TotalSnow_m * Ratio_mm_to_m;
		FA_Total_Sublimation_Ground_Snow_mm = input->RepoVecDict["SublimationSnow_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["SublimationSnow_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["SublimationSnow_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_Total_Sublimation_Canopy_Snow_mm = input->RepoVecDict["SublimationSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["SublimationSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_Total_Melt_Ground_Snow_mm = input->RepoVecDict["SnowMelt_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["SnowMelt_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["SnowMelt_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_Total_Melt_Canopy_Snow_mm = input->RepoVecDict["SnowMelt_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["SnowMelt_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_Total_Storage_Ground_Snow_mm = input->RepoVecDict["StorageSnow_UnderTree_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["StorageSnow_UnderSVeg_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["StorageSnow_noCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_TS_Last_Canopy_Snow_mm = input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m +
			input->RepoVecDict["StorageSnow_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)   
		Eq_Snow_WaterBalance_mm = FA_Precip_Snow_mm - FA_Total_Sublimation_Ground_Snow_mm - FA_Total_Sublimation_Canopy_Snow_mm - FA_Total_Melt_Ground_Snow_mm - FA_Total_Melt_Canopy_Snow_mm - FA_Total_Storage_Ground_Snow_mm - FA_TS_Last_Canopy_Snow_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_Snow_WaterBalance_mm) < Zero_Threshold_mm) { Eq_Snow_WaterBalance_mm = abs(Eq_Snow_WaterBalance_mm); }
		input->RepoDict["Eq_Snow_WaterBalance_mm"] = Eq_Snow_WaterBalance_mm;

		//======================================================
		//Eq_WC_SurfaceInflux_WaterBalance_mm (mm), relative to Folder area
		double FA_WC_Precip_mm, FA_WC_Sublimation_mm, FA_WC_Storage_Ground_Snow_TS_last_mm, FA_WC_FluxRainSnowMelt_mm, Eq_WC_SurfaceInflux_WaterBalance_mm;
		FA_WC_Precip_mm = input->TotalPrecip_m * Ratio_mm_to_m * WaterC_noTC_frac;
		FA_WC_Sublimation_mm = input->RepoVecDict["SublimationSnow_WaterNoCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_WC_Storage_Ground_Snow_TS_last_mm = input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_WC_FluxRainSnowMelt_mm = input->RepoVecDict["Flux_to_WaterArea_Rain_SnowMelt_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s) [Snow melt is included in Fux_Rain_SnowMelt]
		Eq_WC_SurfaceInflux_WaterBalance_mm = FA_WC_Precip_mm - FA_WC_Sublimation_mm - FA_WC_Storage_Ground_Snow_TS_last_mm - FA_WC_FluxRainSnowMelt_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_WC_SurfaceInflux_WaterBalance_mm) < Zero_Threshold_mm) { Eq_WC_SurfaceInflux_WaterBalance_mm = abs(Eq_WC_SurfaceInflux_WaterBalance_mm); }

		//======================================================
		//Eq_WC_SnowMelt_WaterBalance_mm (mm), relative to Folder area
		double FA_WC_Rain_mm, FA_WC_SnowMelt_mm, Eq_WC_SnowMelt_WaterBalance_mm;
		FA_WC_Rain_mm = input->TotalRain_m * Ratio_mm_to_m * WaterC_noTC_frac;
		FA_WC_SnowMelt_mm = input->RepoVecDict["SnowMelt_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * ratio_WC_to_noCanopy;
		//Balance is simple input = change in storage + output(s) [Snow melt is included in Fux_Rain_SnowMelt]
		Eq_WC_SnowMelt_WaterBalance_mm = FA_WC_FluxRainSnowMelt_mm - FA_WC_Rain_mm - FA_WC_SnowMelt_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_WC_SnowMelt_WaterBalance_mm) < Zero_Threshold_mm) { Eq_WC_SnowMelt_WaterBalance_mm = abs(Eq_WC_SnowMelt_WaterBalance_mm); }

		//======================================================
		//Eq_WC_SurfaceRunoff_WaterBalance_mm (mm), relative to Folder area
		double FA_WC_Storage_Depression_TS_last_mm, FA_WC_Evaporation_Depression_mm, FA_WC_Runoff_mm, FA_WC_Flux_BA_to_GI_mm, Eq_WC_SurfaceRunoff_WaterBalance_mm;
		FA_WC_Storage_Depression_TS_last_mm = input->RepoVecDict["Storage_WaterPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_WC_Evaporation_Depression_mm = input->RepoVecDict["Evaporation_WaterPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_WC_Runoff_mm = input->RepoVecDict["Runoff_Water_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_WC_Flux_BA_to_GI_mm = input->RepoVecDict["Runoff_Water_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s) [Snow melt is included in Fux_Rain_SnowMelt]
		Eq_WC_SurfaceRunoff_WaterBalance_mm = FA_WC_FluxRainSnowMelt_mm - FA_WC_Storage_Depression_TS_last_mm - FA_WC_Evaporation_Depression_mm - FA_WC_Runoff_mm - FA_WC_Flux_BA_to_GI_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_WC_SurfaceRunoff_WaterBalance_mm) < Zero_Threshold_mm) { Eq_WC_SurfaceRunoff_WaterBalance_mm = abs(Eq_WC_SurfaceRunoff_WaterBalance_mm); }

		//======================================================
		//Eq_IC_SurfaceInflux_WaterBalance_mm (mm), relative to Folder area
		double FA_IC_noTC_Precip_mm, FA_IC_TC_Throughfall_mm, FA_IC_TC_Sublimation_mm, FA_IC_TC_Storage_Ground_Snow_TS_last_mm, FA_IC_noTC_Sublimation_mm, FA_IC_noTC_Storage_Ground_Snow_TS_last_mm, FA_IC_FluxRainSnowMeltIrrigation_mm, FA_IC_Total_Irrigation_mm, Eq_IC_SurfaceInflux_WaterBalance_mm;
		FA_IC_noTC_Precip_mm = input->TotalPrecip_m * Ratio_mm_to_m * IC_noTC_frac;
		FA_IC_TC_Throughfall_mm = input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverIC_to_TC;
		FA_IC_Total_Irrigation_mm = input->RepoVecDict["Irrigation_ImperviousDepressionStorage_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_IC_TC_Sublimation_mm = input->RepoVecDict["SublimationSnow_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverIC_to_TC;
		FA_IC_TC_Storage_Ground_Snow_TS_last_mm = input->RepoVecDict["StorageSnow_UnderTree_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverIC_to_TC;
		FA_IC_noTC_Sublimation_mm = input->RepoVecDict["SublimationSnow_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_ICnoTC_to_noCanopy;
		FA_IC_noTC_Storage_Ground_Snow_TS_last_mm = input->RepoVecDict["StorageSnow_noCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_ICnoTC_to_noCanopy;
		FA_IC_FluxRainSnowMeltIrrigation_mm = input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s) [Snow melt is included in FA_IC_FluxRainSnowMeltIrrigation_mm]
		//Note: snow depths are liquid equivalent; precip, throughfall, and storage are frozen, sublimation is vapor, and the difference is snowmelt
		Eq_IC_SurfaceInflux_WaterBalance_mm = FA_IC_noTC_Precip_mm + FA_IC_TC_Throughfall_mm + FA_IC_Total_Irrigation_mm - FA_IC_TC_Sublimation_mm - FA_IC_TC_Storage_Ground_Snow_TS_last_mm - FA_IC_noTC_Sublimation_mm - FA_IC_noTC_Storage_Ground_Snow_TS_last_mm - FA_IC_FluxRainSnowMeltIrrigation_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_IC_SurfaceInflux_WaterBalance_mm) < Zero_Threshold_mm) { Eq_IC_SurfaceInflux_WaterBalance_mm = abs(Eq_IC_SurfaceInflux_WaterBalance_mm); }
		input->RepoDict["Eq_IC_SurfaceInflux_WaterBalance_mm"] = Eq_IC_SurfaceInflux_WaterBalance_mm;

		//======================================================
		//Eq_IC_SnowMelt_WaterBalance_mm (mm), relative to Folder area
		double FA_IC_noTC_Rain_mm, FA_IC_TC_ThroughfallRain_mm, FA_IC_noTC_SnowMelt_mm, FA_IC_underTC_SnowMelt_mm, Eq_IC_SnowMelt_WaterBalance_mm;
		FA_IC_noTC_Rain_mm = input->TotalRain_m * Ratio_mm_to_m * IC_noTC_frac;
		FA_IC_TC_ThroughfallRain_mm = input->RepoVecDict["ThroughFallRain_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverIC_to_TC;
		FA_IC_noTC_SnowMelt_mm = input->RepoVecDict["SnowMelt_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_ICnoTC_to_noCanopy;
		FA_IC_underTC_SnowMelt_mm = input->RepoVecDict["SnowMelt_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverIC_to_TC;
		//Balance is simple input = change in storage + output(s)
		Eq_IC_SnowMelt_WaterBalance_mm = FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_noTC_Rain_mm - FA_IC_TC_ThroughfallRain_mm - FA_IC_noTC_SnowMelt_mm - FA_IC_underTC_SnowMelt_mm - FA_IC_Total_Irrigation_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_IC_SnowMelt_WaterBalance_mm) < Zero_Threshold_mm) { Eq_IC_SnowMelt_WaterBalance_mm = abs(Eq_IC_SnowMelt_WaterBalance_mm); }
		input->RepoDict["Eq_IC_SnowMelt_WaterBalance_mm"] = Eq_IC_SnowMelt_WaterBalance_mm;

		//======================================================
		//Eq_IC_SurfaceRunoff_WaterBalance_mm (mm), relative to Folder area
		double FA_IC_Storage_Depression_TS_last_mm, FA_IC_Evaporation_Depression_mm, FA_IC_Flux_to_PC_mm, FA_IC_Runoff_mm, FA_IC_Flux_BA_to_GI_mm, Eq_IC_SurfaceRunoff_WaterBalance_mm;
		FA_IC_Storage_Depression_TS_last_mm = input->RepoVecDict["Storage_ImperviousPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_IC_Evaporation_Depression_mm = input->RepoVecDict["Evaporation_ImperviousPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//total_Flux_Impervious_to_Pervious_m (m) as pervious cover depth
		FA_IC_Flux_to_PC_mm = input->RepoVecDict["Flux_Impervious_to_Pervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//total_Flux_Impervious_to_Pervious_m (m) as impervious cover depth
		FA_IC_Runoff_mm = input->RepoVecDict["Runoff_Impervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//total_Runoff_Impervious_BAtoGI_m (m) is folder area depth
		FA_IC_Flux_BA_to_GI_mm = input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s); snowmelt is included in WaterOn.
		Eq_IC_SurfaceRunoff_WaterBalance_mm = FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_Storage_Depression_TS_last_mm - FA_IC_Evaporation_Depression_mm - FA_IC_Flux_to_PC_mm - FA_IC_Runoff_mm - FA_IC_Flux_BA_to_GI_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_IC_SurfaceRunoff_WaterBalance_mm) < Zero_Threshold_mm) { Eq_IC_SurfaceRunoff_WaterBalance_mm = abs(Eq_IC_SurfaceRunoff_WaterBalance_mm); }
		input->RepoDict["Eq_IC_SurfaceRunoff_WaterBalance_mm"] = Eq_IC_SurfaceRunoff_WaterBalance_mm;

		//======================================================
		//Eq_PC_SurfaceInflux_WaterBalance_mm (mm), relative to Folder area
		double FA_PC_noTC_noSV_Precip_mm, FA_PC_TC_ThroughFall_mm, FA_PC_FluxRainSnowMelt_mm, FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm, FA_PC_underSV_Sublimation_mm, FA_SV_StorageFinal_Ground_Snow_mm, FA_PC_underTC_Sublimation_mm, FA_PC_TC_StorageFinal_Ground_Snow_mm, FA_PC_noTC_noSV_Sublimation_mm, FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm, Eq_PC_SurfaceInflux_WaterBalance_mm, FA_PC_Flux_IC_to_PC_mm, FA_PC_Total_Irrigation_mm;
		FA_PC_Total_Irrigation_mm = input->RepoVecDict["Irrigation_PerviousDepressionStorage_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_TC_ThroughFall_mm = input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverPC_to_TC;
		FA_SV_ThroughFall_mm = input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_noTC_noSV_Precip_mm = input->TotalPrecip_m * Ratio_mm_to_m * SoilC_noTC_frac;
		FA_PC_FluxRainSnowMelt_mm = input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm = input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_underTC_Sublimation_mm = input->RepoVecDict["SublimationSnow_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverPC_to_TC;
		FA_PC_underSV_Sublimation_mm = input->RepoVecDict["SublimationSnow_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_noTC_noSV_Sublimation_mm = input->RepoVecDict["SublimationSnow_SoilNoCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_TC_StorageFinal_Ground_Snow_mm = input->RepoVecDict["StorageSnow_UnderTree_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverPC_to_TC;
		FA_SV_StorageFinal_Ground_Snow_mm = input->RepoVecDict["StorageSnow_UnderSVeg_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm = (input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_last_m3"][Folder_Type_ID] + input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_last_m3"][Folder_Type_ID])/ FA_m2 * Ratio_mm_to_m;
		FA_PC_Flux_IC_to_PC_mm = input->RepoVecDict["Flux_Impervious_to_Pervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		// Balance is simple input = change in storage + output(s)
		Eq_PC_SurfaceInflux_WaterBalance_mm = FA_PC_TC_ThroughFall_mm + FA_SV_ThroughFall_mm + FA_PC_noTC_noSV_Precip_mm + FA_PC_Flux_IC_to_PC_mm + FA_PC_Total_Irrigation_mm - FA_PC_underTC_Sublimation_mm -	FA_PC_TC_StorageFinal_Ground_Snow_mm - FA_PC_underSV_Sublimation_mm - FA_SV_StorageFinal_Ground_Snow_mm - FA_PC_noTC_noSV_Sublimation_mm - FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm - FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_PC_SurfaceInflux_WaterBalance_mm) < Zero_Threshold_mm) { Eq_PC_SurfaceInflux_WaterBalance_mm = abs(Eq_PC_SurfaceInflux_WaterBalance_mm); }
		input->RepoDict["Eq_PC_SurfaceInflux_WaterBalance_mm"] = Eq_PC_SurfaceInflux_WaterBalance_mm;

		//======================================================
		//Eq_PC_SnowMelt_WaterBalance_mm (mm), relative to Folder area
		double Eq_PC_SnowMelt_WaterBalance_mm, FA_PC_Soil_noCanopy_Rain_mm, FA_PC_TC_ThroughfallRain_mm, FA_PC_SV_ThroughfallRain_mm, FA_PC_Soil_noCanopy_SnowMelt_mm, FA_PC_PermeablePavement_noCanopy_SnowMelt_mm, FA_PC_underTC_SnowMelt_mm, FA_PC_underSV_SnowMelt_mm;
		FA_PC_FluxRainSnowMelt_mm = input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Soil_noCanopy_Rain_mm = input->TotalRain_m * Ratio_mm_to_m * SoilC_noTC_frac;
		FA_PC_TC_ThroughfallRain_mm = input->RepoVecDict["ThroughFallRain_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverPC_to_TC;
		FA_PC_SV_ThroughfallRain_mm = input->RepoVecDict["ThroughFallRain_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Soil_noCanopy_SnowMelt_mm = input->RepoVecDict["SnowMelt_SoilNoCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_PermeablePavement_noCanopy_SnowMelt_mm = input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_underTC_SnowMelt_mm = input->RepoVecDict["SnowMelt_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m * Ratio_TCoverPC_to_TC;
		FA_PC_underSV_SnowMelt_mm = input->RepoVecDict["SnowMelt_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)
		Eq_PC_SnowMelt_WaterBalance_mm = FA_PC_FluxRainSnowMelt_mm - FA_PC_Soil_noCanopy_Rain_mm - FA_PC_TC_ThroughfallRain_mm - FA_PC_SV_ThroughfallRain_mm - FA_PC_Total_Irrigation_mm - FA_PC_Soil_noCanopy_SnowMelt_mm - FA_PC_PermeablePavement_noCanopy_SnowMelt_mm - FA_PC_underTC_SnowMelt_mm - FA_PC_underSV_SnowMelt_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_PC_SnowMelt_WaterBalance_mm) < Zero_Threshold_mm) { Eq_PC_SnowMelt_WaterBalance_mm = abs(Eq_PC_SnowMelt_WaterBalance_mm); }
		input->RepoDict["Eq_PC_SnowMelt_WaterBalance_mm "] = Eq_PC_SnowMelt_WaterBalance_mm;

		//======================================================
		//Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm (mm), relative to Folder area
		double FA_PC_Runoff_InfilExcess_mm, FA_PC_Runoff_SatExcess_mm, FA_PC_Infiltration_InfilExcess_mm, FA_PC_Infiltration_SatExcess_mm, FA_PC_Drainage_Macropore_mm, FA_PC_Evaporation_Depression_mm, FA_PC_InfExRO_Flux_to_GI_mm, FA_PC_SatExRO_Flux_to_GI_mm, FA_PC_Storage_Depression_TS_last_mm, Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm;
		//Note: total_Runoff_InfilEx_m and total_Runoff_SatExcess_m are folder depths (m), like all Runoff variables
		FA_PC_Runoff_InfilExcess_mm = input->RepoVecDict["Runoff_InfilExcess_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Runoff_SatExcess_mm = input->RepoVecDict["Runoff_SatExcess_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Infiltration_InfilExcess_mm = input->RepoVecDict["Infiltration_viaInfilEx_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Infiltration_SatExcess_mm = input->RepoVecDict["Infiltration_viaSatEx_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Drainage_Macropore_mm = input->RepoVecDict["Drainage_macroPore_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Evaporation_Depression_mm = input->RepoVecDict["Evaporation_PerviousPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Storage_Depression_TS_last_mm = input->RepoVecDict["Storage_PerviousPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_InfExRO_Flux_to_GI_mm = input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_SatExRO_Flux_to_GI_mm = input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)
		Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm = FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm - FA_PC_Runoff_InfilExcess_mm - FA_PC_Runoff_SatExcess_mm - FA_PC_Infiltration_InfilExcess_mm - FA_PC_Infiltration_SatExcess_mm - FA_PC_Drainage_Macropore_mm - FA_PC_Evaporation_Depression_mm - FA_PC_Storage_Depression_TS_last_mm - FA_PC_InfExRO_Flux_to_GI_mm - FA_PC_SatExRO_Flux_to_GI_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm) < Zero_Threshold_mm) { Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm = abs(Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm); }
		input->RepoDict["Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm"] = Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm;

		//======================================================
		//Eq_PC_VadoseZone_WaterBalance_mm (mm), relative to Folder area
		//Note: Eq_PC_VadoseZone_WaterBalance_mm balances to zero when EvapoTranspiration_Sum_m3 divided by FA_m2, even though EvapoTranspiration_Sum_m3 is even taken from Tree Canopy over Impervious Cover, it is pulled from the pervious vadose area. 
		double FA_PC_EvapTranspiration_VadoseZone_mm, FA_PC_StorageChange_SoilEvapZone_mm, FA_PC_Drainage_SoilEvapZone_mm, FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm, FA_PC_Drainage_VadoseZone_mm, FA_PC_Storage_VadoseZone_TS_last_mm, Eq_PC_VadoseZone_WaterBalance_mm;
		FA_PC_EvapTranspiration_VadoseZone_mm = input->RepoVecDict["EvapoTranspiration_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm = input->RepoVecDict["Flux_SatExGW_to_SoilFieldCapacity_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_StorageChange_SoilEvapZone_mm = (input->RepoVecDict["Storage_SoilEvapZone_TS_last_m3"][Folder_Type_ID] -
			input->RepoVecDict["Storage_SoilEvapZone_TS_0th_m"][Folder_Type_ID]) / FA_m2 * Ratio_mm_to_m;
		FA_PC_Drainage_SoilEvapZone_mm = input->RepoVecDict["Drainage_SoilEvapZone_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Storage_VadoseZone_TS_last_mm = input->RepoVecDict["Storage_VadoseZone_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		FA_PC_Drainage_VadoseZone_mm = input->RepoVecDict["Drainage_VadoseZone_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m;
		//Balance is simple input = change in storage + output(s)
		Eq_PC_VadoseZone_WaterBalance_mm = FA_PC_Infiltration_InfilExcess_mm + FA_PC_Infiltration_SatExcess_mm + FA_PC_Drainage_Macropore_mm + FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm - FA_PC_StorageChange_SoilEvapZone_mm - FA_PC_EvapTranspiration_VadoseZone_mm - FA_PC_Drainage_VadoseZone_mm - FA_PC_Storage_VadoseZone_TS_last_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_PC_VadoseZone_WaterBalance_mm) < Zero_Threshold_mm) { Eq_PC_VadoseZone_WaterBalance_mm = abs(Eq_PC_VadoseZone_WaterBalance_mm); }
		input->RepoDict["Eq_PC_VadoseZone_WaterBalance_mm"] = Eq_PC_VadoseZone_WaterBalance_mm;

		//======================================================
		//Eq_PC_SatZone_WaterBalance_mm (mm), relative to Catchment area
		//Note: Each time step in StorageDeficit_VadoseZone.cpp, AveSMD_new = AveSMD_old - Drainage_VadoseZone_m + Runoff_Subsurface_m
		//Note: To balance entire simulation period, AveSMD_TS_last = AveSMD_TS_first - Drainage_VadoseZone_m + Runoff_Subsurface_m - Runoff_Subsurface_TS_first_m 
		double CA_PC_StorageDeficit_VadoseZone_TS_first_mm, CA_PC_StorageDeficit_VadoseZone_TS_last_mm, CA_PC_Drainage_VadoseZone_BA_and_GI_mm,
			CA_PC_Runoff_Subsurface_BA_and_GI_mm, CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm, Eq_PC_SatZone_WaterBalance_mm;

		//Catchment_PerviousArea_AffectingStorageDeficit_m2 (m2) contains entire catchment pervious area draining to vadose zone
		//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 influences the catchment variable, StorageDeficit_VadoseZone_m
		double Catchment_PerviousArea_AffectingStorageDeficit_m2 = input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"];

		//CA_PC_Drainage_VadoseZone_BA_and_GI_mm  is input->RepoDict["Drainage_VadoseZone_Catchment_WB_m3"], Catchment_PerviousArea_AffectingStorageDeficit_m2 * Ratio_mm_to_m
		CA_PC_Drainage_VadoseZone_BA_and_GI_mm = input->SafeDivide(input->RepoDict["Drainage_VadoseZone_Catchment_WB_m3"], Catchment_PerviousArea_AffectingStorageDeficit_m2) * Ratio_mm_to_m;
		//CA_PC_Runoff_Subsurface_BA_and_GI_mm 
		CA_PC_Runoff_Subsurface_BA_and_GI_mm = input->SafeDivide(input->RepoDict["Runoff_Subsurface_Catchment_WB_m3"], Catchment_PerviousArea_AffectingStorageDeficit_m2) * Ratio_mm_to_m;
		//CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm 
		CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm = input->SafeDivide(input->RepoDict["Runoff_Subsurface_TS_first_m3"], Catchment_PerviousArea_AffectingStorageDeficit_m2) * Ratio_mm_to_m;
		//CA_PC_StorageDeficit_VadoseZone_TS_first_mm 
		CA_PC_StorageDeficit_VadoseZone_TS_first_mm = input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"], Catchment_PerviousArea_AffectingStorageDeficit_m2) * Ratio_mm_to_m;
		//CA_PC_StorageDeficit_VadoseZone_TS_last_mm 
		CA_PC_StorageDeficit_VadoseZone_TS_last_mm = input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"], Catchment_PerviousArea_AffectingStorageDeficit_m2) * Ratio_mm_to_m;

		//Balance is simple input = change in storage + output(s)   
		Eq_PC_SatZone_WaterBalance_mm = CA_PC_StorageDeficit_VadoseZone_TS_first_mm + CA_PC_Runoff_Subsurface_BA_and_GI_mm - CA_PC_Drainage_VadoseZone_BA_and_GI_mm - CA_PC_StorageDeficit_VadoseZone_TS_last_mm - CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm;
		//If absolute value of Eq variable < Zero_Threshold_mm then write it as absolute to avoid - zero
		if (abs(Eq_PC_SatZone_WaterBalance_mm) < Zero_Threshold_mm) { Eq_PC_SatZone_WaterBalance_mm = abs(Eq_PC_SatZone_WaterBalance_mm); }
		input->RepoDict["Eq_PC_SatZone_WaterBalance_mm"] = Eq_PC_SatZone_WaterBalance_mm;

		//DAT File output starts
		//If Folder_Type_ID = 0 (BulkArea) And Folder_List.size() > 1 then BulkArea and GI simulated and catchment totals should be written 
		//Note: This output section is repeated below for BulkArea and GI, to report their area specific fractions and depths
		//if (Folder_Type_ID == 0 && Folder_List.size() > 1) {
		if (Folder_Type_ID == 0 && Folder_List.size() > 1) {
			//======================================================
			//Section to write to outfile_WaterBudget_dat.dat file for Catchment when both BulkArea & GI are simulated
			//Note: 1st instance of Summary_Budget_Water_mm.dat for Folder_List.size() > 1
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_dat << "Catchment Variable (_units)" << setw(70 - size("Catchment Variable (_units)")) << "Value" << endl;
			outfile_WaterBudget_dat << "Note: Depth is relative to the Area. Volume is conserved. Depth = Volume/Area." << endl;
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
			//precision function sets precision to 3 decimal places for fractions
			outfile_WaterBudget_dat.precision(3);
			outfile_WaterBudget_dat << "Area_Catchment_m2" << setw(70 - size("Area_Catchment_m2")) << CA_m2 << endl;
			outfile_WaterBudget_dat << "Area_Folder_m2" << setw(70 - size("Area_Folder_m2")) << input->RepoVecDict["Catchment_Type_Area_m2"][0] + input->RepoVecDict["Catchment_Type_Area_m2"][1] << endl;
			outfile_WaterBudget_dat << "Area_ImperviousAll_frac" << setw(70 - size("Area_ImperviousAll_frac")) << input->RepoDict["Catchment_TotalImperviousCover_frac"] << endl;
			outfile_WaterBudget_dat << "Area_PerviousAll_frac" << setw(70 - size("Area_PerviousAll_frac")) << input->RepoDict["Catchment_TotalPerviousCover_frac"] << endl;
			outfile_WaterBudget_dat << "Area_TreeOverImpervious_frac" << setw(70 - size("Area_TreeOverImpervious_frac")) << input->RepoDict["Catchment_TreeCanopyCover_overImpervious_frac"] << endl;
			outfile_WaterBudget_dat << "Area_TreeOverPervious_frac" << setw(70 - size("Area_TreeOverPervious_frac")) << input->RepoDict["Catchment_TreeCanopyCover_overPervious_frac"] << endl;
			outfile_WaterBudget_dat << "Area_ShortVegCanopy_frac" << setw(70 - size("Area_ShortVegCanopy_frac")) << input->RepoDict["Catchment_ShortVegCover_noTreeCanopy_frac"] << endl;
			outfile_WaterBudget_dat << "Area_SoilNoCanopy_frac" << setw(70 - size("Area_SoilNoCanopy_frac")) << input->RepoDict["Catchment_SoilCover_noTreeCanopy_frac"] << endl;
			outfile_WaterBudget_dat << "Area_PermeablePavementNoCanopy_frac" << setw(70 - size("Area_PermeablePavementNoCanopy_frac")) << input->RepoDict["Catchment_PermeablePavementCover_noTreeCanopy_frac"] << endl;
			outfile_WaterBudget_dat << "Area_WaterNoCanopy_frac" << setw(70 - size("Area_WaterNoCanopy_frac")) << input->RepoDict["Catchment_WaterCover_noTreeCanopy_frac"] << endl;
			outfile_WaterBudget_dat << "Area_DCIA_frac" << setw(70 - size("Area_DCIA_frac")) << input->RepoDict["Catchment_DCIA_frac"] << endl;
			outfile_WaterBudget_dat << "Area_InfiltExcessGoverned_frac" << setw(70 - size("Area_InfiltExcessGoverned_frac")) << input->RepoDict["Catchment_InfiltExcessGovernedArea_frac"] << endl;
			outfile_WaterBudget_dat << "Area_MacroPore_frac" << setw(70 - size("Area_MacroPore_frac")) << input->RepoDict["Catchment_MacroPore_frac"] << endl;
			outfile_WaterBudget_dat << "TS_First_SurficialGroundwater_frac" << setw(70 - size("TS_first_SurficialGroundwater_frac")) << input->RepoVecDict["Groundwater_surficial_TS_first_frac"][Folder_Type_ID] << endl;
			outfile_WaterBudget_dat << "TS_Last_SurficialGroundwater_frac" << setw(70 - size("TS_Last_SurficialGroundwater_frac")) << input->RepoVecDict["Groundwater_surficial_TS_last_frac"][Folder_Type_ID] << endl;
			//precision function sets precision to 2 decimal places for depths
			outfile_WaterBudget_dat.precision(2);
			outfile_WaterBudget_dat << "TS_First_SurficialGroundwater_mm" << setw(70 - size("TS_first_SurficialGroundwater_mm")) << sumAcrossFolderTypes(input, "Groundwater_surficial_TS_first_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_SurficialGroundwater_mm" << setw(70 - size("TS_Last_SurficialGroundwater_mm")) << sumAcrossFolderTypes(input, "Groundwater_surficial_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Precipitation_mm" << setw(70 - size("Precipitation_mm")) << input->TotalPrecip_m * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "PrecipitationRain_mm" << setw(70 - size("PrecipitationRain_mm")) << input->TotalRain_m * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "PrecipitationSnow_mm" << setw(70 - size("PrecipitationSnow_mm")) << input->TotalSnow_m * Ratio_mm_to_m << endl;

			outfile_WaterBudget_dat << "Irrigation_TreeInterception_mm" << setw(70 - size("Irrigation_TreeInterception_mm")) << sumAcrossFolderTypes(input, "Irrigation_TreeInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Irrigation_SVegInterception_mm" << setw(70 - size("Irrigation_SVegInterception_mm")) << sumAcrossFolderTypes(input, "Irrigation_SVegInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Irrigation_PerviousDepressionStorage_mm" << setw(70 - size("Irrigation_PerviousDepressionStorage_mm")) << sumAcrossFolderTypes(input, "Irrigation_PerviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Irrigation_ImperviousDepressionStorage_mm" << setw(70 - size("Irrigation_ImperviousDepressionStorage_mm")) << sumAcrossFolderTypes(input, "Irrigation_ImperviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;

			outfile_WaterBudget_dat << "InterceptionRainSnow_TreeCanopy_mm" << setw(70 - size("InterceptionRainSnow_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "InterceptionRainSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "InterceptionRainSnow_SVegCanopy_mm" << setw(70 - size("InterceptionRainSnow_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "InterceptionRainSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "EvaporationSublimation_TreeCanopy_mm" << setw(70 - size("EvaporationSublimation_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "EvaporationSublimation_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "EvaporationSublimation_SVegCanopy_mm" << setw(70 - size("EvaporationSublimation_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "EvaporationSublimation_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageRain_TreeCanopy_mm" << setw(70 - size("TS_Last_StorageRain_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "StorageRain_TreeCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageSnow_TreeCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_TreeCanopy_mm")) << (input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][0] + input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][1]) / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageRain_SVegCanopy_mm" << setw(70 - size("TS_Last_StorageRain_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "StorageRain_SVegCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageSnow_SVegCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "StorageSnow_SVegCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "ThroughFallRain_TreeCanopy_mm" << setw(70 - size("ThroughFallRain_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "ThroughFallRain_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "ThroughFallSnow_TreeCanopy_mm" << setw(70 - size("ThroughFallSnow_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "ThroughFallSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "ThroughFallRain_SVegCanopy_mm" << setw(70 - size("ThroughFallRain_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "ThroughFallRain_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "ThroughFallSnow_SVegCanopy_mm" << setw(70 - size("ThroughFallSnow_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "ThroughFallSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageSnow_noCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_noCanopy_mm")) << sumAcrossFolderTypes(input, "StorageSnow_noCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageSnow_underTreeCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_underTreeCanopy_mm")) << sumAcrossFolderTypes(input, "StorageSnow_UnderTree_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageSnow_underSVegCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_underSVegCanopy_mm")) << sumAcrossFolderTypes(input, "StorageSnow_UnderSVeg_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "InterceptionSnow_TreeCanopy_mm" << setw(70 - size("InterceptionSnow_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "InterceptionSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "InterceptionSnow_SVegCanopy_mm" << setw(70 - size("InterceptionSnow_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "InterceptionSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SublimationSnow_TreeCanopy_mm" << setw(70 - size("SublimationSnow_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "SublimationSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SublimationSnow_SVegCanopy_mm" << setw(70 - size("SublimationSnow_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "SublimationSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SublimationSnow_UnderTree_mm" << setw(70 - size("SublimationSnow_UnderTree_mm")) << sumAcrossFolderTypes(input, "SublimationSnow_UnderTree_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SublimationSnow_UnderSVeg_mm" << setw(70 - size("SublimationSnow_UnderSVeg_mm")) << sumAcrossFolderTypes(input, "SublimationSnow_UnderSVeg_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SublimationSnow_noCanopy_mm" << setw(70 - size("SublimationSnow_noCanopy_mm")) << sumAcrossFolderTypes(input, "SublimationSnow_noCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SnowMelt_TreeCanopy_mm" << setw(70 - size("SnowMelt_TreeCanopy_mm")) << sumAcrossFolderTypes(input, "SnowMelt_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "SnowMelt_SVegCanopy_mm" << setw(70 - size("SnowMelt_SVegCanopy_mm")) << sumAcrossFolderTypes(input, "SnowMelt_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Snow_Melt_UnderTree_mm" << setw(70 - size("Snow_Melt_UnderTree_mm")) << sumAcrossFolderTypes(input, "SnowMelt_UnderTree_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Snow_Melt_UnderShortVeg_mm" << setw(70 - size("Snow_Melt_UnderShortVeg_mm")) << sumAcrossFolderTypes(input, "SnowMelt_UnderSVeg_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Snow_Melt_OpenArea_mm" << setw(70 - size("Snow_Melt_OpenArea_mm")) << sumAcrossFolderTypes(input, "SnowMelt_noCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_mm" << setw(70 - size("Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_mm")) << sumAcrossFolderTypes(input, "Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Evaporation_ImperviousPondedWater_mm" << setw(70 - size("Evaporation_ImperviousPondedWater_mm")) << sumAcrossFolderTypes(input, "Evaporation_ImperviousPondedWater_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_Storage_ImperviousPonded_mm" << setw(70 - size("TS_Last_Storage_ImperviousPonded_mm")) << sumAcrossFolderTypes(input, "Storage_ImperviousPondedWater_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_Impervious_to_Pervious_mm" << setw(70 - size("Flux_Impervious_to_Pervious_mm")) << sumAcrossFolderTypes(input, "Flux_Impervious_to_Pervious_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_to_WaterArea_Rain_SnowMelt_mm" << setw(70 - size("Flux_to_WaterArea_Rain_SnowMelt_mm")) << sumAcrossFolderTypes(input, "Flux_to_WaterArea_Rain_SnowMelt_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Evaporation_WaterPondedWater_mm" << setw(70 - size("Evaporation_WaterPondedWater_mm")) << sumAcrossFolderTypes(input, "Evaporation_WaterPondedWater_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_Storage_WaterPonded_mm" << setw(70 - size("TS_Last_Storage_WaterPonded_mm")) << sumAcrossFolderTypes(input, "Storage_WaterPondedWater_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_mm" << setw(70 - size("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_mm")) << sumAcrossFolderTypes(input, "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_mm" << setw(70 - size("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_mm")) << sumAcrossFolderTypes(input, "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Evaporation_PerviousPondedWater_mm" << setw(70 - size("Evaporation_PerviousPondedWater_mm")) << sumAcrossFolderTypes(input, "Evaporation_PerviousPondedWater_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_Storage_PerviousPonded_mm" << setw(70 - size("TS_Last_Storage_PerviousPonded_mm")) << sumAcrossFolderTypes(input, "Storage_PerviousPondedWater_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_First_Storage_SoilEvapZone_mm" << setw(70 - size("TS_first_Storage_SoilEvapZone_mm")) << sumAcrossFolderTypes(input, "Storage_SoilEvapZone_TS_first_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_Storage_SoilEvapZone_mm" << setw(70 - size("TS_Last_Storage_SoilEvapZone_mm")) << sumAcrossFolderTypes(input, "Storage_SoilEvapZone_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_First_Storage_VadoseZone_mm" << setw(70 - size("TS_first_Storage_VadoseZone_mm")) << sumAcrossFolderTypes(input, "Storage_VadoseZone_TS_first_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_Storage_VadoseZone_mm" << setw(70 - size("TS_Last_Storage_VadoseZone_mm")) << sumAcrossFolderTypes(input, "Storage_VadoseZone_TS_last_m3") / CA_m2 * Ratio_mm_to_m << endl;

			outfile_WaterBudget_dat << "TS_First_StorageDeficit_VadoseZone_mm" << setw(70 - size("TS_first_StorageDeficit_VadoseZone_mm")) << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "TS_Last_StorageDeficit_VadoseZone_mm" << setw(70 - size("TS_Last_StorageDeficit_VadoseZone_mm")) << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << endl;

			outfile_WaterBudget_dat << "Drainage_macroPore_mm" << setw(70 - size("Drainage_macroPore_mm")) << sumAcrossFolderTypes(input, "Drainage_macroPore_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Infiltration_mm" << setw(70 - size("Infiltration_mm")) << sumAcrossFolderTypes(input, "Infiltration_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Infiltration_viaInfilEx_mm" << setw(70 - size("Infiltration_viaInfilEx_mm")) << sumAcrossFolderTypes(input, "Infiltration_viaInfilEx_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Infiltration_viaSatEx_mm" << setw(70 - size("Infiltration_viaSatEx_mm")) << sumAcrossFolderTypes(input, "Infiltration_viaSatEx_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "EvapoTranspiration_mm" << setw(70 - size("EvapoTranspiration_mm")) << sumAcrossFolderTypes(input, "EvapoTranspiration_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Drainage_SoilEvapZone_mm" << setw(70 - size("Drainage_SoilEvapZone_mm")) << sumAcrossFolderTypes(input, "Drainage_SoilEvapZone_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Drainage_VadoseZone_mm" << setw(70 - size("Drainage_VadoseZone_mm")) << sumAcrossFolderTypes(input, "Drainage_VadoseZone_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_SatExGW_to_SoilFieldCapacity_mm" << setw(70 - size("Flux_SatExGW_to_SoilFieldCapacity_mm")) << sumAcrossFolderTypes(input, "Flux_SatExGW_to_SoilFieldCapacity_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;

			//Note: Fluxes from BulkArea to GI reported
			outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_WaterCover_Runoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_WaterCover_Runoff_mm")) << sumAcrossFolderTypes(input, "Runoff_Water_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_ImpCover_Runoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_ImpCover_Runoff_mm")) << sumAcrossFolderTypes(input, "Runoff_Impervious_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_PerCover_InfilExRunoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_PerCover_InfilExRunoff_mm")) << sumAcrossFolderTypes(input, "Runoff_InfilExcess_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_PerCover_SatExRunoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_PerCover_SatExRunoff_mm")) << sumAcrossFolderTypes(input, "Runoff_SatExcess_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;

			//Note: Runoff variables are folder depths (m), not cover type depths, and were adjusted to folder depths in RunoffSummation.cpp or ImperviousDepressionStorageCalc.cpp for IC
			outfile_WaterBudget_dat << "Runoff_Impervious_mm" << setw(70 - size("Runoff_Impervious_mm")) << sumAcrossFolderTypes(input, "Runoff_Impervious_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Runoff_Water_mm" << setw(70 - size("Runoff_Water_mm")) << sumAcrossFolderTypes(input, "Runoff_Water_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Runoff_Pervious_mm" << setw(70 - size("Runoff_Pervious_mm")) << sumAcrossFolderTypes(input, "Runoff_Pervious_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Runoff_InfilEx_mm" << setw(70 - size("Runoff_InfilEx_mm")) << sumAcrossFolderTypes(input, "Runoff_InfilExcess_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Runoff_SatExcess_mm" << setw(70 - size("Runoff_SatExcess_mm")) << sumAcrossFolderTypes(input, "Runoff_SatExcess_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Runoff_Subsurface_mm" << setw(70 - size("Runoff_Subsurface_mm")) << sumAcrossFolderTypes(input, "Runoff_Subsurface_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "Runoff_mm" << setw(70 - size("Runoff_mm")) << sumAcrossFolderTypes(input, "Runoff_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << endl;
		}
		//End if (Folder_Type_ID == 0 && Folder_List.size() > 1)

		//======================================================
		//Section to write to outfile_WaterBudget_dat.dat file for each Folder when both BulkArea & GI are simulated
		//Note: 2nd instance of Summary_Budget_Water_mm.dat for Folder_List.size() > 1
		outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
		outfile_WaterBudget_dat << Name_Folder_Type_str << " Variable (_units)" << setw(70 - Length_Name_Folder_Type - size(" Variable (_units)")) << "Value" << endl;
		outfile_WaterBudget_dat << "Note: Depth is relative to the Area. Volume is conserved. Depth = Volume/Area." << endl;
		outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
		//precision function sets precision to 3 decimal places for fractions
		outfile_WaterBudget_dat.precision(3);
		outfile_WaterBudget_dat << "Area_Catchment_m2" << setw(70 - size("Area_Catchment_m2")) << CA_m2 << endl;
		outfile_WaterBudget_dat << "Area_Folder_m2" << setw(70 - size("Area_Folder_m2")) << FA_m2 << endl;
		outfile_WaterBudget_dat << "Area_ImperviousAll_frac" << setw(70 - size("Area_ImperviousAll_frac")) << IC_frac << endl;
		outfile_WaterBudget_dat << "Area_PerviousAll_frac" << setw(70 - size("Area_PerviousAll_frac")) << PC_frac << endl;
		outfile_WaterBudget_dat << "Area_TreeOverImpervious_frac" << setw(70 - size("Area_TreeOverImpervious_frac")) << TC_overIC_frac << endl;
		outfile_WaterBudget_dat << "Area_TreeOverPervious_frac" << setw(70 - size("Area_TreeOverPervious_frac")) << TC_overPC_frac << endl;
		outfile_WaterBudget_dat << "Area_ShortVegCanopy_frac" << setw(70 - size("Area_ShortVegCanopy_frac")) << SVegC_noTC_frac << endl;
		outfile_WaterBudget_dat << "Area_SoilNoCanopy_frac" << setw(70 - size("Area_SoilNoCanopy_frac")) << SoilC_noTC_frac << endl;
		outfile_WaterBudget_dat << "Area_PermeablePavementNoCanopy_frac" << setw(70 - size("Area_PermeablePavementNoCanopy_frac")) << PermeablePavementC_noTC_frac << endl;
		outfile_WaterBudget_dat << "Area_WaterNoCanopy_frac" << setw(70 - size("Area_WaterNoCanopy_frac")) << WaterC_noTC_frac << endl;
		outfile_WaterBudget_dat << "Area_DCIA_frac" << setw(70 - size("Area_DCIA_frac")) << DCIA_frac << endl;
		outfile_WaterBudget_dat << "Area_InfiltExcessGoverned_frac" << setw(70 - size("Area_InfiltExcessGoverned_frac")) << InfilExSim_frac << endl;
		outfile_WaterBudget_dat << "Area_MacroPore_frac" << setw(70 - size("Area_MacroPore_frac")) << MacroPore_frac << endl;
		outfile_WaterBudget_dat << "TS_First_SurficialGroundwater_frac" << setw(70 - size("TS_first_SurficialGroundwater_frac")) << input->RepoVecDict["Groundwater_surficial_TS_first_frac"][Folder_Type_ID] << endl;
		outfile_WaterBudget_dat << "TS_Last_SurficialGroundwater_frac" << setw(70 - size("TS_Last_SurficialGroundwater_frac")) << input->RepoVecDict["Groundwater_surficial_TS_last_frac"][Folder_Type_ID] << endl;
		//precision function sets precision to 2 decimal places for depths
		outfile_WaterBudget_dat.precision(2);
		outfile_WaterBudget_dat << "TS_First_SurficialGroundwater_mm" << setw(70 - size("TS_first_SurficialGroundwater_mm")) << input->RepoVecDict["Groundwater_surficial_TS_first_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_SurficialGroundwater_mm" << setw(70 - size("TS_Last_SurficialGroundwater_mm")) << input->RepoVecDict["Groundwater_surficial_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Precipitation_mm" << setw(70 - size("Precipitation_mm")) << input->TotalPrecip_m * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "PrecipitationRain_mm" << setw(70 - size("PrecipitationRain_mm")) << input->TotalRain_m * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "PrecipitationSnow_mm" << setw(70 - size("PrecipitationSnow_mm")) << input->TotalSnow_m * Ratio_mm_to_m << endl;

		outfile_WaterBudget_dat << "Irrigation_TreeInterception_mm" << setw(70 - size("Irrigation_TreeInterception_mm")) << sumAcrossFolderTypes(input, "Irrigation_TreeInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Irrigation_SVegInterception_mm" << setw(70 - size("Irrigation_SVegInterception_mm")) << sumAcrossFolderTypes(input, "Irrigation_SVegInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Irrigation_PerviousDepressionStorage_mm" << setw(70 - size("Irrigation_PerviousDepressionStorage_mm")) << sumAcrossFolderTypes(input, "Irrigation_PerviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Irrigation_ImperviousDepressionStorage_mm" << setw(70 - size("Irrigation_ImperviousDepressionStorage_mm")) << sumAcrossFolderTypes(input, "Irrigation_ImperviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;

		outfile_WaterBudget_dat << "InterceptionRainSnow_TreeCanopy_mm" << setw(70 - size("InterceptionRainSnow_TreeCanopy_mm")) << input->RepoVecDict["InterceptionRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "InterceptionRainSnow_SVegCanopy_mm" << setw(70 - size("InterceptionRainSnow_SVegCanopy_mm")) << input->RepoVecDict["InterceptionRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "EvaporationSublimation_TreeCanopy_mm" << setw(70 - size("EvaporationSublimation_TreeCanopy_mm")) << input->RepoVecDict["EvaporationSublimation_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "EvaporationSublimation_SVegCanopy_mm" << setw(70 - size("EvaporationSublimation_SVegCanopy_mm")) << input->RepoVecDict["EvaporationSublimation_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageRain_TreeCanopy_mm" << setw(70 - size("TS_Last_StorageRain_TreeCanopy_mm")) << input->RepoVecDict["StorageRain_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageSnow_TreeCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_TreeCanopy_mm")) << input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageRain_SVegCanopy_mm" << setw(70 - size("TS_Last_StorageRain_SVegCanopy_mm")) << input->RepoVecDict["StorageRain_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageSnow_SVegCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_SVegCanopy_mm")) << input->RepoVecDict["StorageSnow_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "ThroughFallRain_TreeCanopy_mm" << setw(70 - size("ThroughFallRain_TreeCanopy_mm")) << input->RepoVecDict["ThroughFallRain_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "ThroughFallSnow_TreeCanopy_mm" << setw(70 - size("ThroughFallSnow_TreeCanopy_mm")) << input->RepoVecDict["ThroughFallSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "ThroughFallRain_SVegCanopy_mm" << setw(70 - size("ThroughFallRain_SVegCanopy_mm")) << input->RepoVecDict["ThroughFallRain_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "ThroughFallSnow_SVegCanopy_mm" << setw(70 - size("ThroughFallSnow_SVegCanopy_mm")) << input->RepoVecDict["ThroughFallSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageSnow_noCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_noCanopy_mm")) << input->RepoVecDict["StorageSnow_noCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageSnow_underTreeCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_underTreeCanopy_mm")) << input->RepoVecDict["StorageSnow_UnderTree_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageSnow_underSVegCanopy_mm" << setw(70 - size("TS_Last_StorageSnow_underSVegCanopy_mm")) << input->RepoVecDict["StorageSnow_UnderSVeg_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "InterceptionSnow_TreeCanopy_mm" << setw(70 - size("InterceptionSnow_TreeCanopy_mm")) << input->RepoVecDict["InterceptionSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "InterceptionSnow_SVegCanopy_mm" << setw(70 - size("InterceptionSnow_SVegCanopy_mm")) << input->RepoVecDict["InterceptionSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SublimationSnow_TreeCanopy_mm" << setw(70 - size("SublimationSnow_TreeCanopy_mm")) << input->RepoVecDict["SublimationSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SublimationSnow_SVegCanopy_mm" << setw(70 - size("SublimationSnow_SVegCanopy_mm")) << input->RepoVecDict["SublimationSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SublimationSnow_UnderTree_mm" << setw(70 - size("SublimationSnow_UnderTree_mm")) << input->RepoVecDict["SublimationSnow_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SublimationSnow_UnderSVeg_mm" << setw(70 - size("SublimationSnow_UnderSVeg_mm")) << input->RepoVecDict["SublimationSnow_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SublimationSnow_noCanopy_mm" << setw(70 - size("SublimationSnow_noCanopy_mm")) << input->RepoVecDict["SublimationSnow_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SnowMelt_TreeCanopy_mm" << setw(70 - size("SnowMelt_TreeCanopy_mm")) << input->RepoVecDict["SnowMelt_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "SnowMelt_SVegCanopy_mm" << setw(70 - size("SnowMelt_SVegCanopy_mm")) << input->RepoVecDict["SnowMelt_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Snow_Melt_UnderTree_mm" << setw(70 - size("Snow_Melt_UnderTree_mm")) << input->RepoVecDict["SnowMelt_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Snow_Melt_UnderShortVeg_mm" << setw(70 - size("Snow_Melt_UnderShortVeg_mm")) << input->RepoVecDict["SnowMelt_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Snow_Melt_OpenArea_mm" << setw(70 - size("Snow_Melt_OpenArea_mm")) << input->RepoVecDict["SnowMelt_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_mm" << setw(70 - size("Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_mm")) << input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Evaporation_ImperviousPondedWater_mm" << setw(70 - size("Evaporation_ImperviousPondedWater_mm")) << input->RepoVecDict["Evaporation_ImperviousPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_Storage_ImperviousPonded_mm" << setw(70 - size("TS_Last_Storage_ImperviousPonded_mm")) << input->RepoVecDict["Storage_ImperviousPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_Impervious_to_Pervious_mm" << setw(70 - size("Flux_Impervious_to_Pervious_mm")) << input->RepoVecDict["Flux_Impervious_to_Pervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_to_WaterArea_Rain_SnowMelt_mm" << setw(70 - size("Flux_to_WaterArea_Rain_SnowMelt_mm")) << input->RepoVecDict["Flux_to_WaterArea_Rain_SnowMelt_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Evaporation_WaterPondedWater_mm" << setw(70 - size("Evaporation_WaterPondedWater_mm")) << input->RepoVecDict["Evaporation_WaterPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_Storage_WaterPonded_mm" << setw(70 - size("TS_Last_Storage_WaterPonded_mm")) << input->RepoVecDict["Storage_WaterPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_mm" << setw(70 - size("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_mm")) << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_mm" << setw(70 - size("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_mm")) << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Evaporation_PerviousPondedWater_mm" << setw(70 - size("Evaporation_PerviousPondedWater_mm")) << input->RepoVecDict["Evaporation_PerviousPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_Storage_PerviousPonded_mm" << setw(70 - size("TS_Last_Storage_PerviousPonded_mm")) << input->RepoVecDict["Storage_PerviousPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_First_Storage_SoilEvapZone_mm" << setw(70 - size("TS_first_Storage_SoilEvapZone_mm")) << input->RepoVecDict["Storage_SoilEvapZone_TS_first_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_Storage_SoilEvapZone_mm" << setw(70 - size("TS_Last_Storage_SoilEvapZone_mm")) << input->RepoVecDict["Storage_SoilEvapZone_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_First_Storage_VadoseZone_mm" << setw(70 - size("TS_first_Storage_VadoseZone_mm")) << input->RepoVecDict["Storage_VadoseZone_TS_first_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_Storage_VadoseZone_mm" << setw(70 - size("TS_Last_Storage_VadoseZone_mm")) << input->RepoVecDict["Storage_VadoseZone_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;

		outfile_WaterBudget_dat << "TS_First_StorageDeficit_VadoseZone_mm" << setw(70 - size("TS_first_StorageDeficit_VadoseZone_mm")) << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "TS_Last_StorageDeficit_VadoseZone_mm" << setw(70 - size("TS_Last_StorageDeficit_VadoseZone_mm")) << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << endl;

		outfile_WaterBudget_dat << "Drainage_macroPore_mm" << setw(70 - size("Drainage_macroPore_mm")) << input->RepoVecDict["Drainage_macroPore_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Infiltration_mm" << setw(70 - size("Infiltration_mm")) << input->RepoVecDict["Infiltration_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Infiltration_viaInfilEx_mm" << setw(70 - size("Infiltration_viaInfilEx_mm")) << input->RepoVecDict["Infiltration_viaInfilEx_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Infiltration_viaSatEx_mm" << setw(70 - size("Infiltration_viaSatEx_mm")) << input->RepoVecDict["Infiltration_viaSatEx_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "EvapoTranspiration_mm" << setw(70 - size("EvapoTranspiration_mm")) << input->RepoVecDict["EvapoTranspiration_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Drainage_SoilEvapZone_mm" << setw(70 - size("Drainage_SoilEvapZone_mm")) << input->RepoVecDict["Drainage_SoilEvapZone_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Drainage_VadoseZone_mm" << setw(70 - size("Drainage_VadoseZone_mm")) << input->RepoVecDict["Drainage_VadoseZone_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_SatExGW_to_SoilFieldCapacity_mm" << setw(70 - size("Flux_SatExGW_to_SoilFieldCapacity_mm")) << input->RepoVecDict["Flux_SatExGW_to_SoilFieldCapacity_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;

		//Note: Fluxes from BulkArea to GI reported
		outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_WaterCover_Runoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_WaterCover_Runoff_mm")) << input->RepoVecDict["Runoff_Water_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_ImpCover_Runoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_ImpCover_Runoff_mm")) << input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_PerCover_InfilExRunoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_PerCover_InfilExRunoff_mm")) << input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Flux_BulkArea_to_GI_PerCover_SatExRunoff_mm" << setw(70 - size("Flux_BulkArea_to_GI_PerCover_SatExRunoff_mm")) << input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;

		//Note: Runoff variables are folder depths (m), not cover type depths, and were adjusted to folder depths in RunoffSummation.cpp or ImperviousDepressionStorageCalc.cpp for IC
		outfile_WaterBudget_dat << "Runoff_Impervious_mm" << setw(70 - size("Runoff_Impervious_mm")) << input->RepoVecDict["Runoff_Impervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Runoff_Water_mm" << setw(70 - size("Runoff_Water_mm")) << input->RepoVecDict["Runoff_Water_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Runoff_Pervious_mm" << setw(70 - size("Runoff_Pervious_mm")) << input->RepoVecDict["Runoff_Pervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Runoff_InfilEx_mm" << setw(70 - size("Runoff_InfilEx_mm")) << input->RepoVecDict["Runoff_InfilExcess_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Runoff_SatExcess_mm" << setw(70 - size("Runoff_SatExcess_mm")) << input->RepoVecDict["Runoff_SatExcess_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Runoff_Subsurface_mm" << setw(70 - size("Runoff_Subsurface_mm")) << input->RepoVecDict["Runoff_Subsurface_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << "Runoff_mm" << setw(70 - size("Runoff_mm")) << input->RepoVecDict["Runoff_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_dat << endl;

		//If Folder_Type_ID equals Folder_BulkArea_ID, then write out BulkArea data
		if (Folder_Type_ID == Folder_BulkArea_ID) {
			outfile_WaterBudget_dat << "Bulk Area Water Balance Summary" << setw(70 - size("Bulk Area Water Balance Summary")) << "Value:" << endl;
			outfile_WaterBudget_dat << "Eq_Tree_Canopy_WaterBalance_mm" << setw(70 - size("Eq_Tree_Canopy_WaterBalance_mm")) << input->RepoDict["Eq_Tree_Canopy_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_SVeg_Canopy_WaterBalance_mm" << setw(70 - size("Eq_SVeg_Canopy_WaterBalance_mm")) << input->RepoDict["Eq_SVeg_Canopy_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_Snow_WaterBalance_mm" << setw(70 - size("Eq_Snow_WaterBalance_mm")) << input->RepoDict["Eq_Snow_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_WC_SurfaceInflux_WaterBalance_mm" << setw(70 - size("Eq_WC_SurfaceInflux_WaterBalance_mm")) << input->RepoDict["Eq_WC_SurfaceInflux_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_WC_SnowMelt_WaterBalance_mm" << setw(70 - size("Eq_WC_SnowMelt_WaterBalance_mm")) << input->RepoDict["Eq_WC_SnowMelt_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_WC_SurfaceRunoff_WaterBalance_mm" << setw(70 - size("Eq_WC_SurfaceRunoff_WaterBalance_mm")) << input->RepoDict["Eq_WC_SurfaceRunoff_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_IC_SurfaceInflux_WaterBalance_mm" << setw(70 - size("Eq_IC_SurfaceInflux_WaterBalance_mm")) << input->RepoDict["Eq_IC_SurfaceInflux_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_IC_SnowMelt_WaterBalance_mm" << setw(70 - size("Eq_IC_SnowMelt_WaterBalance_mm")) << input->RepoDict["Eq_IC_SnowMelt_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_IC_SurfaceRunoff_WaterBalance_mm" << setw(70 - size("Eq_IC_SurfaceRunoff_WaterBalance_mm")) << input->RepoDict["Eq_IC_SurfaceRunoff_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_PC_SurfaceInflux_WaterBalance_mm" << setw(70 - size("Eq_PC_SurfaceInflux_WaterBalance_mm")) << input->RepoDict["Eq_PC_SurfaceInflux_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_PC_SnowMelt_WaterBalance_mm" << setw(70 - size("Eq_PC_SnowMelt_WaterBalance_mm")) << input->RepoDict["Eq_PC_SnowMelt_WaterBalance_mm "] << endl;
			outfile_WaterBudget_dat << "Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm" << setw(70 - size("Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm")) << input->RepoDict["Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_PC_VadoseZone_WaterBalance_mm" << setw(70 - size("Eq_PC_VadoseZone_WaterBalance_mm")) << input->RepoDict["Eq_PC_VadoseZone_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_PC_SatZone_WaterBalance_mm" << setw(70 - size("Eq_PC_SatZone_WaterBalance_mm")) << input->RepoDict["Eq_PC_SatZone_WaterBalance_mm"] << endl;
		}
		//End if (Folder_Type_ID == Folder_BulkArea_ID)

		//If Folder_Type_ID equals Folder_GI_ID, then write out Green Infrastructure data
		if (Folder_Type_ID == Folder_GI_ID) {
			outfile_WaterBudget_dat << "GI_Precipitation_mm" << setw(70 - size("GI_Precipitation_mm")) << input->RepoDict["GI_Precipitation_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Inflow_from_BulkArea_mm" << setw(70 - size("GI_Inflow_from_BulkArea_mm")) << input->RepoDict["GI_Runon_from_BulkArea_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Inflow_from_GreenInfrastructure_mm" << setw(70 - size("GI_Inflow_from_GreenInfrastructure_mm")) << GI_Runon_from_GI_Sum_mm << endl;
			outfile_WaterBudget_dat << "GI_Inflow_from_GI_WaterCover_mm" << setw(70 - size("GI_Inflow_from_GI_WaterCover_mm")) << input->RepoVecDict["Runoff_Water_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Inflow_from_GI_ImpCover_mm" << setw(70 - size("GI_Inflow_from_GI_ImpCover_mm")) << input->RepoVecDict["Runoff_Impervious_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Inflow_from_GI_PerCover_mm" << setw(70 - size("GI_Inflow_from_GI_PerCover_mm")) << input->RepoVecDict["Runoff_Pervious_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Inflow_from_GI_Vault_mm" << setw(70 - size("GI_Inflow_from_GI_Vault_mm")) << input->RepoVecDict["Runoff_Vault_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Infiltration_mm" << setw(70 - size("GI_Infiltration_mm")) << input->RepoDict["GI_Infiltration_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Storage_Surface_TS_Last_mm" << setw(70 - size("GI_Storage_Surface_TS_Last_mm")) << input->RepoDict["GI_Storage_Surface_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_PavementWater_TS_Last_mm" << setw(70 - size("GI_PavementWater_TS_Last_mm")) << input->RepoDict["GI_PavementWater_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Storage_Soil_TS_0th_mm" << setw(70 - size("GI_Storage_Soil_TS_0th_mm")) << input->RepoDict["GI_Storage_Soil_TS_0th_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Storage_Soil_TS_Last_mm" << setw(70 - size("GI_Storage_Soil_TS_Last_mm")) << input->RepoDict["GI_Storage_Soil_TS_last_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_Storage_Vault_TS_Last_mm" << setw(70 - size("GI_Storage_Vault_TS_Last_mm")) << input->RepoDict["GI_Storage_Vault_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_EvaporationSurface_mm" << setw(70 - size("GI_EvaporationSurface_mm")) << input->RepoDict["GI_Evaporation_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_EvaporationPavement_mm" << setw(70 - size("GI_EvaporationPavement_mm")) << input->RepoDict["GI_Evaporation_Pavement_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_EvaporationSoil_mm" << setw(70 - size("GI_EvaporationSoil_mm")) << input->RepoDict["GI_Evaporation_Soil_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_EvaporationVault_mm" << setw(70 - size("GI_EvaporationVault_mm")) << input->RepoDict["GI_Evaporation_Vault_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_DrainageVadoseZone_mm" << setw(70 - size("GI_DrainageVadoseZone_mm")) << input->RepoDict["GI_Drainage_VadoseZone_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_RunoffSurface_mm" << setw(70 - size("GI_RunoffSurface_mm")) << input->RepoDict["GI_Runoff_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_RunoffVaultDrain_mm" << setw(70 - size("GI_RunoffVaultDrain_mm")) << input->RepoDict["GI_Runoff_VaultDrain_Sum_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << "GI_GroundwaterMound_TS_Last_mm" << setw(70 - size("GI_GroundwaterMound_TS_Last_mm")) << input->RepoDict["GI_Storage_GroundwaterMound_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << endl;
			outfile_WaterBudget_dat << "Green Infrastructure Water Balance and Layer Summary" << setw(70 - size("Green Infrastructure Water Balance and Layer Summary")) << "Value:" << endl;
			outfile_WaterBudget_dat << "Eq_GI_Tree_Canopy_WaterBalance_mm" << setw(70 - size("Eq_GI_Tree_Canopy_WaterBalance_mm")) << input->RepoDict["Eq_Tree_Canopy_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_GI_SVeg_Canopy_WaterBalance_mm" << setw(70 - size("Eq_GI_SVeg_Canopy_WaterBalance_mm")) << input->RepoDict["Eq_SVeg_Canopy_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_GI_RunonSurface_WaterBalance_mm" << setw(70 - size("Eq_GI_RunonSurface_WaterBalance_mm")) << input->RepoDict["Eq_GI_RunonSurface_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_GI_Surface_WaterBalance_mm" << setw(70 - size("Eq_GI_Surface_WaterBalance_mm")) << input->RepoDict["Eq_GI_Surface_WaterBalance_mm"] << endl;
			outfile_WaterBudget_dat << "Eq_GI_All_Layers_WaterBalance_mm" << setw(70 - size("Eq_GI_All_Layers_WaterBalance_mm")) << input->RepoDict["Eq_GI_All_Layers_WaterBalance_mm"] << endl;
		}
		//End if (Folder_Type_ID == Folder_GI_ID)

		//Write out specific water balances for each component of vertical system
		outfile_WaterBudget_dat << endl;
		outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
		outfile_WaterBudget_dat << Name_Folder_Type_str << " Water Balance Equations (Eq.)" << endl;
		outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
		outfile_WaterBudget_dat << endl;
		outfile_WaterBudget_dat << " Eq_Tree_Canopy_WaterBalance_mm (mm), relative to Folder area" << endl;
		outfile_WaterBudget_dat << " Eq_Tree_Canopy_WaterBalance_mm = FA_TC_Total_Precip_RainSWE_mm + FA_TC_Total_Irrigation_mm - FA_TC_Storage_Canopy_TS_last_mm - FA_TC_Evap_mm - FA_TC_ThroughFall_mm " << endl;
		outfile_WaterBudget_dat << " = " << FA_TC_Total_Precip_RainSWE_mm << " + " << FA_TC_Total_Irrigation_mm << " - " << FA_TC_Storage_Canopy_TS_last_mm << " - " << FA_TC_Evap_mm << " - " << FA_TC_ThroughFall_mm << endl;
		outfile_WaterBudget_dat << " = " << Eq_Tree_Canopy_WaterBalance_mm << endl;
		outfile_WaterBudget_dat << endl;

		outfile_WaterBudget_dat << " Eq_SVeg_Canopy_WaterBalance_mm (mm), relative to Folder area" << endl;
		outfile_WaterBudget_dat << " Eq_SVeg_Canopy_WaterBalance_mm = FA_SV_Total_Precip_RainSWE_mm + FA_SV_Total_Irrigation_mm- FA_SV_Storage_Canopy_TS_last_mm - FA_SV_Evap_mm - FA_SV_ThroughFall_mm  " << endl;
		outfile_WaterBudget_dat << " = " << FA_SV_Total_Precip_RainSWE_mm << " + " << FA_SV_Total_Irrigation_mm << " - " << FA_SV_Storage_Canopy_TS_last_mm << " - " << FA_SV_Evap_mm << " - " << FA_SV_ThroughFall_mm << endl;
		outfile_WaterBudget_dat << " = " << Eq_SVeg_Canopy_WaterBalance_mm << endl;
		outfile_WaterBudget_dat << endl;

		//If Folder_Type_ID equals Folder_BulkArea_ID, then write out BulkArea data
		if (Folder_Type_ID == Folder_BulkArea_ID) {

			outfile_WaterBudget_dat << " Eq_IC_SurfaceInflux_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_IC_SurfaceInflux_WaterBalance_mm =  FA_IC_noTC_Precip_mm + FA_IC_TC_Throughfall_mm + FA_IC_Total_Irrigation_mm - FA_IC_TC_Sublimation_mm - FA_IC_TC_Storage_Ground_Snow_TS_last_mm - FA_IC_noTC_Sublimation_mm - FA_IC_noTC_Storage_Ground_Snow_TS_last_mm - FA_IC_FluxRainSnowMeltIrrigation_mm" << endl;
			outfile_WaterBudget_dat << " = " << FA_IC_noTC_Precip_mm << " + " << FA_IC_TC_Throughfall_mm << " + " << FA_IC_Total_Irrigation_mm << " - " << FA_IC_TC_Sublimation_mm << " - " << FA_IC_TC_Storage_Ground_Snow_TS_last_mm << " - "	<< FA_IC_noTC_Sublimation_mm << " - " << FA_IC_noTC_Storage_Ground_Snow_TS_last_mm << " - " << FA_IC_FluxRainSnowMeltIrrigation_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_IC_SurfaceInflux_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_IC_SnowMelt_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_IC_SnowMelt_WaterBalance_mm = FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_noTC_Rain_mm - FA_IC_TC_ThroughfallRain_mm - FA_IC_noTC_SnowMelt_mm - FA_IC_underTC_SnowMelt_mm - FA_IC_Total_Irrigation_mm" << endl;
			outfile_WaterBudget_dat << " = " << FA_IC_FluxRainSnowMeltIrrigation_mm << " - " << FA_IC_noTC_Rain_mm << " - " << FA_IC_TC_ThroughfallRain_mm << " - " << FA_IC_noTC_SnowMelt_mm << " - " << FA_IC_underTC_SnowMelt_mm << " - " << FA_IC_Total_Irrigation_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_IC_SnowMelt_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_IC_SurfaceRunoff_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_IC_SurfaceRunoff_WaterBalance_mm = FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_Storage_Depression_TS_last_mm - FA_IC_Evaporation_Depression_mm - FA_IC_Flux_to_PC_mm - FA_IC_Runoff_mm - FA_IC_Flux_BA_to_GI_mm" << endl;
			outfile_WaterBudget_dat << " = " << FA_IC_FluxRainSnowMeltIrrigation_mm << " - " << FA_IC_Storage_Depression_TS_last_mm << " - " << FA_IC_Evaporation_Depression_mm << " - " << FA_IC_Flux_to_PC_mm << " - " << FA_IC_Runoff_mm << " - " << FA_IC_Flux_BA_to_GI_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_IC_SurfaceRunoff_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			//Note: Consider refactor to expand water balance and include water cover as unique type
			/*
			outfile_WaterBudget_dat << " Eq_WC_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " FA_WC_WaterBalance_mm = FA_WC_Precip_mm - FA_WC_Flux_BA_to_GI_mm - FA_WC_Evaporation_Depression_mm - FA_WC_Sublimation_mm - FA_WC_Storage_Depression_TS_last_mm - FA_WC_Storage_Ground_Snow_TS_last_mm - FA_WC_Runoff_mm " << endl;
			outfile_WaterBudget_dat << " = " << FA_WC_Precip_mm << " - " << FA_WC_Flux_BA_to_GI_mm << " - " << FA_WC_Evaporation_Depression_mm << " - " << FA_WC_Sublimation_mm << " - " << FA_WC_Storage_Depression_TS_last_mm << " - " << FA_WC_Storage_Ground_Snow_TS_last_mm << " - " << FA_WC_Runoff_mm << endl;
			outfile_WaterBudget_dat << " = " << FA_WC_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;
			*/

			outfile_WaterBudget_dat << " Eq_PC_SurfaceInflux_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_PC_SurfaceInflux_WaterBalance_mm = FA_PC_TC_ThroughFall_mm + FA_SV_ThroughFall_mm + FA_PC_noTC_noSV_Precip_mm + FA_PC_Flux_IC_to_PC_mm + FA_PC_Total_Irrigation_mm - FA_PC_underTC_Sublimation_mm - FA_PC_TC_StorageFinal_Ground_Snow_mm - FA_PC_underSV_Sublimation_mm - FA_SV_StorageFinal_Ground_Snow_mm - FA_PC_noTC_noSV_Sublimation_mm -  FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm - FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm" << endl;
			outfile_WaterBudget_dat << " = " << FA_PC_TC_ThroughFall_mm << " + " << FA_SV_ThroughFall_mm << " + " << FA_PC_noTC_noSV_Precip_mm << " + " << FA_PC_Flux_IC_to_PC_mm << " + " << FA_PC_Total_Irrigation_mm << " - " << FA_PC_underTC_Sublimation_mm << " - " << FA_PC_TC_StorageFinal_Ground_Snow_mm << " - " << FA_PC_underSV_Sublimation_mm << " - " << FA_SV_StorageFinal_Ground_Snow_mm << " - " << FA_PC_noTC_noSV_Sublimation_mm << " - " << FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm << " - " << FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_PC_SurfaceInflux_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_PC_SnowMelt_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_PC_SnowMelt_WaterBalance_mm = FA_PC_FluxRainSnowMelt_mm - FA_PC_Soil_noCanopy_Rain_mm - FA_PC_TC_ThroughfallRain_mm - FA_PC_SV_ThroughfallRain_mm - FA_PC_Total_Irrigation_mm - FA_PC_Soil_noCanopy_SnowMelt_mm - FA_PC_PermeablePavement_noCanopy_SnowMelt_mm - FA_PC_underTC_SnowMelt_mm - FA_PC_underSV_SnowMelt_mm " << endl;
			outfile_WaterBudget_dat << " = " << FA_PC_FluxRainSnowMelt_mm << " - " << FA_PC_Soil_noCanopy_Rain_mm << " - " << FA_PC_TC_ThroughfallRain_mm << " - " << FA_PC_SV_ThroughfallRain_mm << " - " << FA_PC_Total_Irrigation_mm << " - " << FA_PC_Soil_noCanopy_SnowMelt_mm << " - " << FA_PC_PermeablePavement_noCanopy_SnowMelt_mm << " - " << FA_PC_underTC_SnowMelt_mm << " - " << FA_PC_underSV_SnowMelt_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_PC_SnowMelt_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm = FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm - FA_PC_Runoff_InfilExcess_mm - FA_PC_Runoff_SatExcess_mm - FA_PC_Infiltration_InfilExcess_mm - FA_PC_Infiltration_SatExcess_mm - FA_PC_Drainage_Macropore_mm - FA_PC_Evaporation_Depression_mm - FA_PC_Storage_Depression_TS_last_mm - FA_PC_InfExRO_Flux_to_GI_mm - FA_PC_SatExRO_Flux_to_GI_mm" << endl;
			outfile_WaterBudget_dat << " = " << FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm << " - " << FA_PC_Runoff_InfilExcess_mm << " - " << FA_PC_Runoff_SatExcess_mm << " - " << FA_PC_Infiltration_InfilExcess_mm << " - " << FA_PC_Infiltration_SatExcess_mm << " - " << FA_PC_Drainage_Macropore_mm << " - " << FA_PC_Evaporation_Depression_mm << " - " << FA_PC_Storage_Depression_TS_last_mm << " - " << FA_PC_InfExRO_Flux_to_GI_mm << " - " << FA_PC_SatExRO_Flux_to_GI_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_PC_VadoseZone_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_PC_VadoseZone_WaterBalance_mm = FA_PC_Infiltration_InfilExcess_mm + FA_PC_Infiltration_SatExcess_mm + FA_PC_Drainage_Macropore_mm + FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm - FA_PC_StorageChange_SoilEvapZone_mm - FA_PC_EvapTranspiration_VadoseZone_mm - FA_PC_Drainage_VadoseZone_mm - FA_PC_Storage_VadoseZone_TS_last_mm " << endl;
			outfile_WaterBudget_dat << " = " << FA_PC_Infiltration_InfilExcess_mm << " + " << FA_PC_Infiltration_SatExcess_mm << " + " << FA_PC_Drainage_Macropore_mm << " + " << FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm << " - " << FA_PC_StorageChange_SoilEvapZone_mm << " - " << FA_PC_EvapTranspiration_VadoseZone_mm << " - " << FA_PC_Drainage_VadoseZone_mm << " - " << FA_PC_Storage_VadoseZone_TS_last_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_PC_VadoseZone_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_Snow_WaterBalance_mm (mm), relative to Folder area" << endl;
			outfile_WaterBudget_dat << " Eq_Snow_WaterBalance_mm = FA_Precip_Snow_mm - FA_Total_Sublimation_Ground_Snow_mm - FA_Total_Sublimation_Canopy_Snow_mm - FA_Total_Melt_Ground_Snow_mm - FA_Total_Melt_Canopy_Snow_mm - FA_Total_Storage_Ground_Snow_mm - FA_TS_Last_Canopy_Snow_mm" << endl;
			outfile_WaterBudget_dat << " = " << FA_Precip_Snow_mm << " - " << FA_Total_Sublimation_Ground_Snow_mm << " - " << FA_Total_Sublimation_Canopy_Snow_mm << " - " << FA_Total_Melt_Ground_Snow_mm << " - " << FA_Total_Melt_Canopy_Snow_mm << " - " << FA_Total_Storage_Ground_Snow_mm << " - " << FA_TS_Last_Canopy_Snow_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_Snow_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_PC_SatZone_WaterBalance_mm (mm), relative to Catchment area" << endl;
			outfile_WaterBudget_dat << " Eq_PC_SatZone_WaterBalance_mm = CA_PC_StorageDeficit_VadoseZone_TS_first_mm + CA_PC_Runoff_Subsurface_BA_and_GI_mm - CA_PC_Drainage_VadoseZone_BA_and_GI_mm - CA_PC_StorageDeficit_VadoseZone_TS_last_mm - CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm" << endl;
			outfile_WaterBudget_dat << " = " << CA_PC_StorageDeficit_VadoseZone_TS_first_mm << " + " << CA_PC_Runoff_Subsurface_BA_and_GI_mm << " - " << CA_PC_Drainage_VadoseZone_BA_and_GI_mm << " - " << CA_PC_StorageDeficit_VadoseZone_TS_last_mm << " - " << CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_PC_SatZone_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;
		}
		//If Folder_Type_ID equals Folder_GI_ID, then write out Green Infrastructure data
		if (Folder_Type_ID == Folder_GI_ID) {

			outfile_WaterBudget_dat << " Eq_GI_RunonSurface_WaterBalance_mm (mm), relative to Green Infrastructure area" << endl;
			outfile_WaterBudget_dat << " Eq_GI_RunonSurface_WaterBalance_mm = GI_Runon_from_BulkArea_WaterCoverRunoff_mm + GI_Runon_from_BulkArea_ImpCoverRunoff_mm + GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm + GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm + GI_Runon_from_GI_WaterCoverRunoff_mm + GI_Runon_from_GI_ImpCoverRunoff_mm + GI_Runon_from_GI_PerCoverRunoff_mm + GI_Runon_from_GI_VaultRunoff_mm - GI_Runon_from_BulkArea_Sum_mm - GI_Runon_from_GI_Sum_mm" << endl;
			outfile_WaterBudget_dat << " = " << GI_Runon_from_BulkArea_WaterCoverRunoff_mm << " + " << GI_Runon_from_BulkArea_ImpCoverRunoff_mm << " + " << GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm << " + " << GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm << " + " << GI_Runon_from_GI_WaterCoverRunoff_mm << " + " << GI_Runon_from_GI_ImpCoverRunoff_mm << " + " << GI_Runon_from_GI_PerCoverRunoff_mm << " + " << GI_Runon_from_GI_VaultRunoff_mm << " - " << GI_Runon_from_BulkArea_Sum_mm << " - " << GI_Runon_from_GI_Sum_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_GI_RunonSurface_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_GI_Surface_WaterBalance_mm (mm), relative to Green Infrastructure area" << endl;
			outfile_WaterBudget_dat << " Eq_GI_Surface_WaterBalance_mm = GI_Precipitation_Sum_mm + GI_Runon_from_BulkArea_Sum_mm + GI_Runon_from_GI_Sum_mm - GI_Infiltration_Sum_mm - GI_RunoffSurface_Sum_mm - GI_Storage_Surface_TS_Last_mm - GI_EvaporationSurface_Sum_mm" << endl;
			outfile_WaterBudget_dat << " = " << GI_Precipitation_Sum_mm << " + " << GI_Runon_from_BulkArea_Sum_mm << " + " << GI_Runon_from_GI_Sum_mm << " - " << GI_Infiltration_Sum_mm << " - " << GI_RunoffSurface_Sum_mm << " - " << GI_Storage_Surface_TS_Last_mm << " - " << GI_EvaporationSurface_Sum_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_GI_Surface_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << " Eq_GI_All_Layers_WaterBalance_mm (mm), relative to Green Infrastructure area" << endl;
			outfile_WaterBudget_dat << " Eq_GI_All_Layers_WaterBalance_mm = GI_Precipitation_Sum_mm + GI_Runon_from_BulkArea_Sum_mm + GI_Runon_from_GI_Sum_mm - GI_RunoffSurface_Sum_mm - GI_RunoffVaultDrain_Sum_mm - GI_Drainage_VadoseZone_Sum_mm - GI_Storage_Surface_TS_Last_mm - GI_PavementWater_TS_Last_mm - (GI_Storage_Soil_TS_last_mm - GI_Storage_Soil_TS_0th_mm) - GI_Storage_Vault_TS_Last_mm - GI_EvaporationSurface_Sum_mm - GI_EvaporationPavement_Sum_mm - GI_EvaporationSoil_Sum_mm - GI_EvaporationVault_Sum_mm + GI_GoundwaterMound_DynamicEquilibrium_mm" << endl;
			outfile_WaterBudget_dat << " = " << GI_Precipitation_Sum_mm << " + " << GI_Runon_from_BulkArea_Sum_mm << " + " << GI_Runon_from_GI_Sum_mm << " - " << GI_RunoffSurface_Sum_mm << " - " << GI_RunoffVaultDrain_Sum_mm << " - " << GI_Drainage_VadoseZone_Sum_mm << " - " << GI_Storage_Surface_TS_Last_mm << " - " << GI_PavementWater_TS_Last_mm << " - (" << GI_Storage_Soil_TS_last_mm << " - " << GI_Storage_Soil_TS_0th_mm << ") - " << GI_Storage_Vault_TS_Last_mm << " - " << GI_EvaporationSurface_Sum_mm << " - " << GI_EvaporationPavement_Sum_mm << " - " << GI_EvaporationSoil_Sum_mm << " - " << GI_EvaporationVault_Sum_mm << " + " << GI_GoundwaterMound_DynamicEquilibrium_mm << endl;
			outfile_WaterBudget_dat << " = " << Eq_GI_All_Layers_WaterBalance_mm << endl;
			outfile_WaterBudget_dat << endl;
		}

		//If Folder_Type_ID equals Folder_List.back(), last item in list, then write topographic index data for catchment
		if (Folder_Type_ID == Folder_List.back()) {
			//======================================================
			//Eq. 19 Runoff balance (mm) UnRouted - Routed (mm), relative to Folder area; Note Eq # system was historically used in Excel balances
			double total_Runoff_Impervious_mm, total_Runoff_Pervious_mm, total_Runoff_Water_mm, total_Runoff_Subsurface_mm, total_Runoff_mm, total_Runoff_Impervious_R_mm, total_Runoff_Pervious_R_mm, total_Runoff_Water_R_mm, total_Runoff_Subsurface_R_mm, total_Runoff_R_mm, CA_RunoffRouting_Balance_mm;
			//Catchment runoff divides variables of volume (m3) by catchment area CA_m2 (m2)
			total_Runoff_Impervious_mm = input->RepoDict["Runoff_Impervious_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_Pervious_mm = input->RepoDict["Runoff_Pervious_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_Water_mm = input->RepoDict["Runoff_Water_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_Subsurface_mm = input->RepoDict["Runoff_Subsurface_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_mm = input->RepoDict["Runoff_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			//Routed runoff depths (m) were generated in RoutingDiffusion::ScalingRunoff function
			total_Runoff_Impervious_R_mm = input->RepoDict["Runoff_Impervious_R_m"] * Ratio_mm_to_m;
			total_Runoff_Pervious_R_mm = input->RepoDict["Runoff_Pervious_R_m"] * Ratio_mm_to_m;
			total_Runoff_Water_R_mm = input->RepoDict["Runoff_Water_R_m"] * Ratio_mm_to_m;
			total_Runoff_Subsurface_R_mm = input->RepoDict["Runoff_Subsurface_R_m"] * Ratio_mm_to_m;
			total_Runoff_R_mm = input->RepoDict["Runoff_R_m"] * Ratio_mm_to_m;
			//Balance is simple input = (non-routed - routed) 
			CA_RunoffRouting_Balance_mm = (total_Runoff_Impervious_mm - total_Runoff_Impervious_R_mm) + (total_Runoff_Pervious_mm - total_Runoff_Pervious_R_mm) + (total_Runoff_Water_mm - total_Runoff_Water_R_mm) + (total_Runoff_Subsurface_mm - total_Runoff_Subsurface_R_mm) + (total_Runoff_mm - total_Runoff_R_mm);

			outfile_WaterBudget_dat << endl;
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_dat << "Total Catchment Runoff:" << endl;
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_dat << " CA_RunoffRouting_Balance_mm = (total_Runoff_Impervious_mm - total_Runoff_Impervious_R_mm) + (total_Runoff_Pervious_mm - total_Runoff_Pervious_R_mm) + (total_Runoff_Water_mm - total_Runoff_Water_R_mm) + (total_Runoff_Subsurface_mm - total_Runoff_Subsurface_R_mm) + (total_Runoff_mm - total_Runoff_R_mm)" << endl;
			outfile_WaterBudget_dat << " = " << "(" << total_Runoff_Impervious_mm << " - " << total_Runoff_Impervious_R_mm << ")" << " + " << "(" << total_Runoff_Pervious_mm << " - " << total_Runoff_Pervious_R_mm << ")" << " + " << "(" << total_Runoff_Water_mm << " - " << total_Runoff_Water_R_mm << ")" << " + " << "(" << total_Runoff_Subsurface_mm << " - " << total_Runoff_Subsurface_R_mm << ")" << " + " << "(" << total_Runoff_mm << " - " << total_Runoff_R_mm << ")" << endl;
			outfile_WaterBudget_dat << " = " << CA_RunoffRouting_Balance_mm << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat << endl;
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_dat << "Topographic Index Calculation Approach (Exponential or Power Decay) and Parameters:" << endl;
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;

			//References for Topographic Index theory
			//Famiglietti, J.S., & Wood, E.F. (1994).Multiscale Modeling of Spatially Variable Water and Energy Balance Processes.Water Resources Research, 30(11), 3061 - 3078.
			//Hornberger, G. M., Beven, K. J., Cosby, B. J., & Sappington, D. E. (1985). Shenandoah Watershed Study: Calibration of a Topography-Based, Variable Contributing Area Hydrological Model to a Small Forested Catchment. Water Resources Research, 21(12), 1841-1850.
			//TMOD9502.FOR TOPMODEL DEMONSTRATION PROGRAM VERSION 95.02, By Keith Beven or Eq given as misc.S_mean[0] = - params.m * log(misc.qs0 / misc.qss) in parameter.init.c https://cran.r-project.org/web/packages/topmodel/index.html
			//Wang, J., Endreny, T. A., & Hassett, J. M. (2006). Power function decay of hydraulic conductivity for a TOPMODEL-based infiltration routine Hydrological Processes, 20(18), 3825-3834.
			//Wang, J., Endreny, T. A., & Hassett, J. M. (2005). Flexible Modeling Package for Topographically Based Watershed Hydrology. Journal of Hydrology, 314(1-4), 78-91. 
			//Note: Discharge_Subsurface_Max_mph = SZQ | Discharge_Subsurface_Initial_mph = Q0_m_p_dt | VadoseZone_Transmissivity_Max_m2ph = T0_m2_p_dt | ...
			//Note: StorageDeficit_VadoseZone_m = AveSMD | Parameter_m_KsatExpDecay = m_ScalingParam | Parameter_n_KsatPowerDecay = n_PowerDecayParam
			double Discharge_Subsurface_Max_mph, Discharge_Subsurface_Initial_mph, VadoseZone_Transmissivity_Max_m2ph, TopographicIndex_Avg, StorageDeficit_VadoseZone_m, Parameter_m_KsatExpDecay, Parameter_n_KsatPowerDecay;
			//StorageDeficit_VadoseZone_TI_m defined as vector<double> containing values from input->SimulationNumericalParams["TopographicIndexBins"]
			vector<double> StorageDeficit_VadoseZone_TI_m(input->SimulationNumericalParams["TopographicIndexBins"]);
			Discharge_Subsurface_Initial_mph = input->InputXml["Discharge_Subsurface_Initial_mph"];
			VadoseZone_Transmissivity_Max_m2ph = input->InputXml["VadoseZone_Transmissivity_Max_m2ph"];
			TopographicIndex_Avg = input->TI_Value_Average;
			Parameter_m_KsatExpDecay = input->InputXml["Parameter_m_KsatExpDecay"];

			//Infiltration type for decay function with hydraulic conductivity and computation of topographic index
			//Note: Consider refactor to remove this code and simply write out values computed from Inputs functions
			if (input->SimulationStringParams["Algorithm_SoilKsatDecay"] == "ExponentialDecay") {
				outfile_WaterBudget_dat << "HydraulicConductivityDecay_function" << setw((70 - size("HydraulicConductivityDecay_function"))) << "ExponentialDecay" << endl;
				//Discharge_Subsurface_Max_mph(m) is maximum potential subsurface flow rate, when groundwater table is at the ground surface
				//Eq for Discharge_Subsurface_Max_mph similar to Eq 31b for Q0 in Famiglietti and Wood (1994)
				//Eq for Discharge_Subsurface_Max_mph below Eq 3 of Wang et al. (2005), S_bar = -m * ln(R / T0) - m * Lambda, and Discharge_Subsurface_Max_mph becomes T0* exp(-Lambda), where R = Q0
				Discharge_Subsurface_Max_mph = VadoseZone_Transmissivity_Max_m2ph * exp(-1 * TopographicIndex_Avg);
				//Eq for StorageDeficit_VadoseZone_m derived from Eq 4 in Hornberger et al. (1985) Qb = Discharge_Subsurface_Max_mph * EXP(-S / SZM)
				//Eq for StorageDeficit_VadoseZone_m Eq given as SBAR = -SZM * ALOG(Q0 / Discharge_Subsurface_Max_mph) in TMOD9502.FOR or parameter.init.c
				//Eq for StorageDeficit_VadoseZone_m below Eq 3 of Wang et al. (2005), S_bar = -m * ln(R / T0) - m * Lambda
				//StorageDeficit_VadoseZone_m computation uses Natural log of(Discharge_Subsurface_Initial_mph / Discharge_Subsurface_Max_mph)
				//StorageDeficit_VadoseZone_m adjusted to PC_frac, which is fraction of catchment with soil moisture

				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				if (Discharge_Subsurface_Max_mph > 0 and Discharge_Subsurface_Initial_mph > 0) {
					StorageDeficit_VadoseZone_m = -1 * Parameter_m_KsatExpDecay * log(Discharge_Subsurface_Initial_mph / Discharge_Subsurface_Max_mph) * PC_frac;
				}
				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				else {
					StorageDeficit_VadoseZone_m = 0.0;
				}
			}
			if (input->SimulationStringParams["Algorithm_SoilKsatDecay"] == "PowerDecay") {
				outfile_WaterBudget_dat << "HydraulicConductivityDecay_function" << setw((70 - size("HydraulicConductivityDecay_function"))) << "PowerDecay" << endl;
				Parameter_n_KsatPowerDecay = input->InputXml["Parameter_n_KsatPowerDecay"];
				//Discharge_Subsurface_Max_mph(m) is maximum potential subsurface flow rate, when groundwater table is at the ground surface
				//Eq for Discharge_Subsurface_Max_mph is RHS of Eq for S_bar, StorageDeficit_VadoseZone_m, given in text below Eq 9 of Wang et al. (2006)
				Discharge_Subsurface_Max_mph = pow((Discharge_Subsurface_Initial_mph / VadoseZone_Transmissivity_Max_m2ph), (1 / Parameter_n_KsatPowerDecay)) * TopographicIndex_Avg;

				//Eq for StorageDeficit_VadoseZone_m is below Eq 9 of Wang et al. (2006) and below Eq 5 of Wang(2005)
				//StorageDeficit_VadoseZone_m (m) adjusted to PC_frac, which is fraction of catchment with soil moisture
				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				if (Discharge_Subsurface_Max_mph > 0 and Discharge_Subsurface_Initial_mph > 0) {
					StorageDeficit_VadoseZone_m = Parameter_m_KsatExpDecay * (1 - Discharge_Subsurface_Max_mph) * PC_frac;
				}
				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				else {
					//If Q0 or T0 are entered as zero values in HydroPlusConfig.xml, then set StorageDeficit_VadoseZone_m = 0
					StorageDeficit_VadoseZone_m = 0.0;
				}

			}
			//Timestep for adjusting rate constants from HydroPlusConfig.xml
			outfile_WaterBudget_dat << "TimeStep_sec (s)" << setw((70 - size("TimeStep_sec (s)"))) << input->SimulationNumericalParams["TimeStep_sec"] << endl;
			//Number of topographic index bins
			outfile_WaterBudget_dat << "TopographicIndexBins (#)" << setw((70 - size("TopographicIndexBins (#)"))) << fixed << input->SimulationNumericalParams["TopographicIndexBins"] << endl;
			//Average topographic index value
			outfile_WaterBudget_dat << "TopographicIndex_Avg" << setw((70 - size("TopographicIndex_Avg"))) << fixed << input->TI_Value_Average << endl;
			//Scaling parameter for adjusting soil moisture across topographic index
			outfile_WaterBudget_dat << "Parameter_m_KsatExpDecay" << setw((70 - size("Parameter_m_KsatExpDecay"))) << scientific << input->InputXml["Parameter_m_KsatExpDecay"] << endl;
			//Power decay parameter for adjusting conductivity across topographic index
			outfile_WaterBudget_dat << "Parameter_n_KsatPowerDecay" << setw((70 - size("Parameter_n_KsatPowerDecay"))) << fixed << input->InputXml["Parameter_n_KsatPowerDecay"] << endl;
			//Transmissivity (m^2/dt) at catchment saturation
			outfile_WaterBudget_dat << "VadoseZone_Transmissivity_Max_m2ph (m/h)" << setw((70 - size("VadoseZone_Transmissivity_Max_m2ph (m/h)"))) << scientific << input->InputXml["VadoseZone_Transmissivity_Max_m2ph"] << endl;
			//Subsurface flow (m/dt) at start of simulation
			outfile_WaterBudget_dat << "Discharge_Subsurface_Initial_mph (m/h)" << setw((70 - size("Discharge_Subsurface_Initial_mph (m/h)"))) << scientific << input->InputXml["Discharge_Subsurface_Initial_mph"] << endl;
			//Discharge_Subsurface_Max_mph (m/hr) is maximum discharge from catchment pervious area when soil storage full saturated
			outfile_WaterBudget_dat << "Discharge_Subsurface_Max_mph (m/ts)" << setw((70 - size("Discharge_Subsurface_Max_mph (m/ts)"))) << scientific << Discharge_Subsurface_Max_mph << endl;
			//StorageDeficit_VadoseZone_m (m) is catchment average soil moisture deficit for water storage; multiplied by Ratio_mm_to_m to convert m to mm for output display
			outfile_WaterBudget_dat << "StorageDeficit_VadoseZone_m (mm) TS 1st" << setw((70 - size("StorageDeficit_VadoseZone_m (mm) TS 1st"))) << scientific << StorageDeficit_VadoseZone_m * Ratio_mm_to_m << endl;
			outfile_WaterBudget_dat << endl;

			outfile_WaterBudget_dat.precision(4);

			//If Model equals StatisticalHydro then proceed with TI histogram bins
			//Note Model SpatialTemperatureHydro does not use TI histogram bins
			if (input->SimulationStringParams["Model_Selection"] == "StatisticalHydro" || input->SimulationStringParams["Model_Selection"] == "SpatialBufferwGI") {
				//Loop through all topographic index bins, write out TI values; histogram of TI bins ordered from maximum and wettest to minimum and driest
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
				outfile_WaterBudget_dat << "Topographic Index Values for each Histogram Bin, sorted from Wettest to Driest" << endl;
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
				for (int ia = 0; ia < input->Size_TI_Value_Vector; ia++) {
					outfile_WaterBudget_dat << "TopographicIndex_" << ia << setw((70 - size("TopographicIndex_"))) << fixed << input->TI_Value[ia] << endl;
				}
				outfile_WaterBudget_dat << endl;

				//Loop through all topographic index bins, write out TI value fractional areas; histogram of TI bins ordered from maximum and wettest to minimum and driest
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
				outfile_WaterBudget_dat << "Topographic Index Value Fractional Areas for each Histogram Bin, sorted from Wettest to Driest" << endl;
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
				for (int ia = 0; ia < input->Size_TI_Value_Vector; ia++) {
					outfile_WaterBudget_dat << "TopographicIndexArea_f_" << ia << setw((70 - size("TopographicIndexArea_f_00"))) << scientific << input->TI_Area_frac[ia] << endl;
				}
				outfile_WaterBudget_dat << endl;

				//Loop through all topographic index bins, write out StorageDeficit_VadoseZone_TI_m values; histogram of TI bins ordered from maximum and wettest to minimum and driest
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
				outfile_WaterBudget_dat << "Local Soil Moisture Deficit Values for each Histogram Bin, sorted from Wettest to Driest" << endl;
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
				for (int ia = 0; ia < input->Size_TI_Value_Vector; ia++) {
					StorageDeficit_VadoseZone_TI_m[ia] = StorageDeficit_VadoseZone_m + Parameter_m_KsatExpDecay * (TopographicIndex_Avg - input->TI_Value[ia]);
					outfile_WaterBudget_dat << "StorageDeficit_VadoseZone_TI_m[" << ia << "]" << setw((70 - size("StorageDeficit_VadoseZone_TI_m[00]"))) << scientific << StorageDeficit_VadoseZone_TI_m[ia] << endl;
				}
				outfile_WaterBudget_dat << endl;
				outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;
			}
			//Groundwater_surficial_frac > 0.05 used to determine message about concern for excessive saturated area, using 0.05 as arbitrary threshold
			//Note: input->RepoVecDict["Groundwater_surficial_TS_last_frac"][0] is BulkArea value, with Folder_Type_ID = 0 for BulkArea
			if (input->RepoVecDict["Groundwater_surficial_TS_last_frac"][0] >= 0.05) {
				outfile_WaterBudget_dat << " Note: The simulation ended with " << setprecision(2) << input->RepoVecDict["Groundwater_surficial_TS_last_frac"][0] * 100 << "% of the catchment with groundwater at the soil surface, which is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values for each Topographic Index (TI) bin contains negative values and may erroneously over-represent saturated areas and its runoff." << endl;
			}
			//StorageDeficit_VadoseZone_TI_m[1] and input->TI_Area_frac[0] used to determine message about concern some saturated area, using 0 to 0.0499 as arbitrary range for accurate simulation
			else if (StorageDeficit_VadoseZone_TI_m[1] < 0 and input->TI_Area_frac[1] > 0) {
				outfile_WaterBudget_dat << " Note: The simulation ended with between 0 and 5% of the catchment with groundwater at the soil surface, which is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values for each Topographic Index (TI) bin contains negative values and may accurately represent saturated areas and its runoff." << endl;
			}
			//StorageDeficit_VadoseZone_TI_m[1] used to determine message about no saturated area, presuming some catchment area should be saturated (e.g., river channel area)
			else if (StorageDeficit_VadoseZone_TI_m[1] >= 0) {
				outfile_WaterBudget_dat << " Note: The simulation ended with 0% of the catchment with groundwater at the soil surface, which is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values for each Topographic Index (TI) bin with any fractional area contain no negative values and may erroneously under-represent saturated areas and its runoff." << endl;
			}
			//Potential that StorageDeficit_VadoseZone_TI_m[1] > 0 and StorageDeficit_VadoseZone_TI_m[0] < 0 to determine message about uncertain but likely small saturated area, presuming some catchment area should be saturated (e.g., river channel area)
			else {
				outfile_WaterBudget_dat << " Note: The simulation may have ended with little to none of the catchment with groundwater at the soil surface, which is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values (SMD) for each Topographic Index (TI) bin with any fractional area contain no negative values and may erroneously under-represent saturated areas and its runoff." << endl;
			}
			//Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph
			outfile_WaterBudget_dat << endl;
			outfile_WaterBudget_dat << " Note: If the you wish to adjust the fraction of catchment with surficial groundwater and saturation excess runoff by adjusting StorageDeficit_VadoseZone_TI_m values, options include:" << endl;
			outfile_WaterBudget_dat << "      1. Adjust VadoseZone_Transmissivity_Max_m2ph; a larger VadoseZone_Transmissivity_Max_m2ph input reduces the number TI bins with negative StorageDeficit_VadoseZone_TI_m values, a smaller value increases the number TI bins with negative StorageDeficit_VadoseZone_TI_m values; VadoseZone_Transmissivity_Max_m2ph values can range order of magnitudes (e.g., 1E-3 to 1E3)." << endl;
			outfile_WaterBudget_dat << "      2. Adjust Discharge_Subsurface_Initial_mph; a smaller Discharge_Subsurface_Initial_mph input reduces the number TI bins with negative StorageDeficit_VadoseZone_TI_m values, a larger value increases the number TI bins with negative StorageDeficit_VadoseZone_TI_m values." << endl;
			outfile_WaterBudget_dat << "      3. Adjust TopographicIndexBins; reducing the number of TI bins between the maximum and average TI values reduces number of TI bins with negative StorageDeficit_VadoseZone_TI_m values, increasing has an opposite effect; the total fraction of area with negative or positive StorageDeficit_VadoseZone_TI_m may remain the same." << endl;
			outfile_WaterBudget_dat << "      4. Adjust DEM resolution; a larger DEM grid size may increase the minimum, average, and maximum TI values, and create a larger and deeper area of surficial groundwater; finer resolution DEM grid sizes could have the opposite effect. To maintain the same area and depth of surficial groundwater when coarsening the DEM it may require decreasing the value of the VadoseZone_Transmissivity_Max_m2ph and Discharge_Subsurface_Initial_mph parameters." << endl;
			outfile_WaterBudget_dat << "      5. Adjust Parameter_m_KsatExpDecay scaling parameter; a smaller Parameter_m_KsatExpDecay scaling parameter input reduces the local SMD value magnitude, but not the sign or number of TI bins with negative or positive StorageDeficit_VadoseZone_TI_m values." << endl;
			outfile_WaterBudget_dat << endl;
			outfile_WaterBudget_dat << " --------------------------------------------------------------------------------------------------" << endl;

			outfile_WaterBudget_dat << " References:" << endl;
			outfile_WaterBudget_dat << " Famiglietti, J.S., & Wood, E.F. (1994). Multiscale Modeling of Spatially Variable Water and Energy Balance Processes.Water Resources Research, 30(11), 3061 - 3078.:" << endl;
			outfile_WaterBudget_dat << " Hornberger, G.M., Beven, K.J., Cosby, B.J., & Sappington, D.E. (1985). Shenandoah Watershed Study: Calibration of a Topography-Based, Variable Contributing Area Hydrological Model to a Small Forested Catchment. Water Resources Research, 21(12), 1841-1850." << endl;
			outfile_WaterBudget_dat << " TMOD9502.FOR TOPMODEL DEMONSTRATION PROGRAM VERSION 95.02, By Keith Beven or Eq given as misc.S_mean[0] = - params.m * log(misc.qs0 / misc.qss) in parameter.init.c https://cran.r-project.org/web/packages/topmodel/index.html" << endl;
			outfile_WaterBudget_dat << " Wang, J., Endreny, T.A., & Hassett, J.M. (2006). Power function decay of hydraulic conductivity for a TOPMODEL-based infiltration routine Hydrological Processes, 20(18), 3825-3834. " << endl;
			outfile_WaterBudget_dat << " Wang, J., Endreny, T.A., & Hassett, J.M. (2005). Flexible Modeling Package for Topographically Based Watershed Hydrology. Journal of Hydrology, 314(1-4), 78-91. " << endl << endl;
			outfile_WaterBudget_dat << " Note Variable Alternative Names Between HydroPlus and Literature: " << endl;
			outfile_WaterBudget_dat << " Discharge_Subsurface_Max_mph = SZQ | Discharge_Subsurface_Initial_mph = Q0 | VadoseZone_Transmissivity_Max_m2ph = T0 | StorageDeficit_VadoseZone_m = AveSMD | Parameter_m_KsatExpDecay = m | Parameter_n_KsatPowerDecay = n" << endl;
			//outfile_WaterBudget_dat close
			outfile_WaterBudget_dat.close();
		}
		//End If Folder_Type_ID equals 1
		//DAT File output stops

		//CSV File output starts
		//If Folder_Type_ID = 0 (BulkArea) And Folder_List.size() > 1 then BulkArea and GI simulated and catchment totals should be written 
		//Note: This output section is repeated below for BulkArea and GI, to report their area specific fractions and depths
		//if (Folder_Type_ID == 0 && Folder_List.size() > 1) {
		if (Folder_Type_ID == 0 && Folder_List.size() > 1) {
			//======================================================
			//Section to write to outfile_WaterBudget.csv file for Catchment when both BulkArea & GI are simulated
			//Note: 1st instance of Summary_Budget_Water_mm.csv for Folder_List.size() > 1
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_csv << "Catchment_Variable,Value,Units" << endl;
			outfile_WaterBudget_csv << "Note: Depth is relative to the Area. Volume is conserved. Depth = Volume/Area." << endl;
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
			//precision function sets precision to 10 decimal places for fractions
			outfile_WaterBudget_csv.precision(10);
			outfile_WaterBudget_csv << "Area_Catchment," << CA_m2 << ",m2" << endl;
			outfile_WaterBudget_csv << "Area_Folder," << input->RepoVecDict["Catchment_Type_Area_m2"][0] + input->RepoVecDict["Catchment_Type_Area_m2"][1] << ",m2" << endl;
			//Create 3 digit values for fractions: Inputs::roundToDecimalPlaces(Variable_Value, 3)
			//Note: RHS function divides by 1000 after taking integer of the term that takes product with 1000 to round up
			outfile_WaterBudget_csv << "Area_ImperviousAll," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_TotalImperviousCover_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_PerviousAll," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_TotalPerviousCover_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_TreeOverImpervious," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_TreeCanopyCover_overImpervious_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_TreeOverPervious," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_TreeCanopyCover_overPervious_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_ShortVegNoTree," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_ShortVegCover_noTreeCanopy_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_SoilNoCanopy," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_SoilCover_noTreeCanopy_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_PermeablePavementNoCanopy," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_PermeablePavementCover_noTreeCanopy_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_WaterNoTree," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_WaterCover_noTreeCanopy_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_DCIA," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_DCIA_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_InfiltExcessGoverned," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_InfiltExcessGovernedArea_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "Area_MacroPore," << Inputs::roundToDecimalPlaces(input->RepoDict["Catchment_MacroPore_frac"], 3) << ",frac" << endl;
			outfile_WaterBudget_csv << "TS_First_SurficialGroundwater," << Groundwater_surficial_TS_first_frac << ",frac" << endl;
			outfile_WaterBudget_csv << "TS_Last_SurficialGroundwater," << Groundwater_surficial_TS_last_frac << ",frac" << endl;
			outfile_WaterBudget_csv << "TS_First_SurficialGroundwater," << sumAcrossFolderTypes(input, "Groundwater_surficial_TS_first_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_SurficialGroundwater," << sumAcrossFolderTypes(input, "Groundwater_surficial_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Precipitation," << input->TotalPrecip_m * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "PrecipitationRain," << input->TotalRain_m * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "PrecipitationSnow," << input->TotalSnow_m * Ratio_mm_to_m << ",mm" << endl;

			outfile_WaterBudget_csv << "Irrigation_TreeInterception_mm," << sumAcrossFolderTypes(input, "Irrigation_TreeInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_csv << "Irrigation_SVegInterception_mm," << sumAcrossFolderTypes(input, "Irrigation_SVegInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_csv << "Irrigation_PerviousDepressionStorage_mm," << sumAcrossFolderTypes(input, "Irrigation_PerviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
			outfile_WaterBudget_csv << "Irrigation_ImperviousDepressionStorage_mm," << sumAcrossFolderTypes(input, "Irrigation_ImperviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;

			outfile_WaterBudget_csv << "InterceptionRainSnow_TreeCanopy," << sumAcrossFolderTypes(input, "InterceptionRainSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "InterceptionRainSnow_SVegCanopy," << sumAcrossFolderTypes(input, "InterceptionRainSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "EvaporationSublimation_TreeCanopy," << sumAcrossFolderTypes(input, "EvaporationSublimation_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "EvaporationSublimation_SVegCanopy," << sumAcrossFolderTypes(input, "EvaporationSublimation_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageRain_TreeCanopy," << sumAcrossFolderTypes(input, "StorageRain_TreeCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageSnow_TreeCanopy," << (input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][0] + input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][1]) / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageRain_SVegCanopy," << sumAcrossFolderTypes(input, "StorageRain_SVegCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageSnow_SVegCanopy," << sumAcrossFolderTypes(input, "StorageSnow_SVegCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "ThroughFallRain_TreeCanopy," << sumAcrossFolderTypes(input, "ThroughFallRain_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "ThroughFallSnow_TreeCanopy," << sumAcrossFolderTypes(input, "ThroughFallSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "ThroughFallRain_SVegCanopy," << sumAcrossFolderTypes(input, "ThroughFallRain_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "ThroughFallSnow_SVegCanopy," << sumAcrossFolderTypes(input, "ThroughFallSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageSnow_noCanopy," << sumAcrossFolderTypes(input, "StorageSnow_noCanopy_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageSnow_underTreeCanopy," << sumAcrossFolderTypes(input, "StorageSnow_UnderTree_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageSnow_underSVegCanopy," << sumAcrossFolderTypes(input, "StorageSnow_UnderSVeg_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "InterceptionSnow_TreeCanopy," << sumAcrossFolderTypes(input, "InterceptionSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "InterceptionSnow_SVegCanopy," << sumAcrossFolderTypes(input, "InterceptionSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SublimationSnow_TreeCanopy," << sumAcrossFolderTypes(input, "SublimationSnow_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SublimationSnow_SVegCanopy," << sumAcrossFolderTypes(input, "SublimationSnow_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SublimationSnow_UnderTree," << sumAcrossFolderTypes(input, "SublimationSnow_UnderTree_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SublimationSnow_UnderSVeg," << sumAcrossFolderTypes(input, "SublimationSnow_UnderSVeg_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SublimationSnow_noCanopy," << sumAcrossFolderTypes(input, "SublimationSnow_noCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SnowMelt_TreeCanopy," << sumAcrossFolderTypes(input, "SnowMelt_TreeCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "SnowMelt_SVegCanopy," << sumAcrossFolderTypes(input, "SnowMelt_SVegCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Snow_Melt_UnderTree," << sumAcrossFolderTypes(input, "SnowMelt_UnderTree_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Snow_Melt_UnderShortVeg," << sumAcrossFolderTypes(input, "SnowMelt_UnderSVeg_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Snow_Melt_OpenArea," << sumAcrossFolderTypes(input, "SnowMelt_noCanopy_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_to_ImperviousArea_Rain_SnowMelt," << sumAcrossFolderTypes(input, "Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Evaporation_ImperviousPondedWater," << sumAcrossFolderTypes(input, "Evaporation_ImperviousPondedWater_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_Storage_ImperviousPonded," << sumAcrossFolderTypes(input, "Storage_ImperviousPondedWater_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_Impervious_to_Pervious," << sumAcrossFolderTypes(input, "Flux_Impervious_to_Pervious_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_to_WaterArea_Rain_SnowMelt," << sumAcrossFolderTypes(input, "Flux_to_WaterArea_Rain_SnowMelt_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Evaporation_WaterPondedWater," << sumAcrossFolderTypes(input, "Evaporation_WaterPondedWater_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_Storage_WaterPonded," << sumAcrossFolderTypes(input, "Storage_WaterPondedWater_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_to_PerviousArea_Rain_SnowMelt_ImpRunon," << sumAcrossFolderTypes(input, "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_to_PerviousArea_Rain_SnowMelt," << sumAcrossFolderTypes(input, "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Evaporation_PerviousPondedWater," << sumAcrossFolderTypes(input, "Evaporation_PerviousPondedWater_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_Storage_PerviousPonded," << sumAcrossFolderTypes(input, "Storage_PerviousPondedWater_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_First_Storage_SoilEvapZone," << sumAcrossFolderTypes(input, "Storage_SoilEvapZone_TS_first_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_Storage_SoilEvapZone," << sumAcrossFolderTypes(input, "Storage_SoilEvapZone_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_First_Storage_VadoseZone," << sumAcrossFolderTypes(input, "Storage_VadoseZone_TS_first_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_Storage_VadoseZone," << sumAcrossFolderTypes(input, "Storage_VadoseZone_TS_last_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;

			outfile_WaterBudget_csv << "TS_First_StorageDeficit_VadoseZone," << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "TS_Last_StorageDeficit_VadoseZone," << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << ",mm" << endl;

			outfile_WaterBudget_csv << "Drainage_macroPore," << sumAcrossFolderTypes(input, "Drainage_macroPore_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Infiltration," << sumAcrossFolderTypes(input, "Infiltration_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Infiltration_viaInfilEx," << sumAcrossFolderTypes(input, "Infiltration_viaInfilEx_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Infiltration_viaSatEx," << sumAcrossFolderTypes(input, "Infiltration_viaSatEx_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "EvapoTranspiration," << sumAcrossFolderTypes(input, "EvapoTranspiration_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Drainage_SoilEvapZone," << sumAcrossFolderTypes(input, "Drainage_SoilEvapZone_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Drainage_VadoseZone," << sumAcrossFolderTypes(input, "Drainage_VadoseZone_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_SatExGW_to_SoilFieldCapacity," << sumAcrossFolderTypes(input, "Flux_SatExGW_to_SoilFieldCapacity_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;

			//Note: Fluxes from BulkArea to GI reported
			outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_WaterCover_Runoff," << sumAcrossFolderTypes(input, "Runoff_Water_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_ImpCover_Runoff," << sumAcrossFolderTypes(input, "Runoff_Impervious_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_PerCover_InfilExRunoff," << sumAcrossFolderTypes(input, "Runoff_InfilExcess_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_PerCover_SatExRunoff," << sumAcrossFolderTypes(input, "Runoff_SatExcess_BAtoGI_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;

			//Note: Runoff variables are folder depths (m), not cover type depths, and were adjusted to folder depths in RunoffSummation.cpp or ImperviousDepressionStorageCalc.cpp for IC
			outfile_WaterBudget_csv << "Runoff_Impervious," << sumAcrossFolderTypes(input, "Runoff_Impervious_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Runoff_Water," << sumAcrossFolderTypes(input, "Runoff_Water_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Runoff_Pervious," << sumAcrossFolderTypes(input, "Runoff_Pervious_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Runoff_InfilEx," << sumAcrossFolderTypes(input, "Runoff_InfilExcess_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Runoff_SatExcess," << sumAcrossFolderTypes(input, "Runoff_SatExcess_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Runoff_Subsurface," << sumAcrossFolderTypes(input, "Runoff_Subsurface_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "Runoff," << sumAcrossFolderTypes(input, "Runoff_Sum_m3") / CA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << endl;
		}
		//End if (Folder_Type_ID == 0 && Folder_List.size() > 1)

		//======================================================
		//Section to write to outfile_WaterBudget.csv file for each Folder when both BulkArea & GI are simulated
		//Note: 2nd instance of Summary_Budget_Water_mm.csv for Folder_List.size() > 1
		outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
		outfile_WaterBudget_csv << Name_Folder_Type_str << "_Variable,Value,Units" << endl;
		outfile_WaterBudget_csv << "Note: Depth is relative to the Area. Volume is conserved. Depth = Volume/Area." << endl;
		outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
		//precision function sets precision to 3 decimal places for fractions
		outfile_WaterBudget_csv.precision(10);
		outfile_WaterBudget_csv << "Area_Catchment," << CA_m2 << ",m2" << endl;
		outfile_WaterBudget_csv << "Area_Folder," << FA_m2 << ",m2" << endl;
		outfile_WaterBudget_csv << "Area_ImperviousAll," << IC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_PerviousAll," << PC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_TreeOverImpervious," << TC_overIC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_TreeOverPervious," << TC_overPC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_ShortVegNoTree," << SVegC_noTC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_SoilNoCanopy," << SoilC_noTC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_PermeablePavementNoCanopy," << PermeablePavementC_noTC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_WaterNoTree," << WaterC_noTC_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_DCIA," << DCIA_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_InfiltExcessGoverned," << InfilExSim_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "Area_MacroPore," << MacroPore_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "TS_First_SurficialGroundwater," << Groundwater_surficial_TS_first_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "TS_Last_SurficialGroundwater," << Groundwater_surficial_TS_last_frac << ",frac" << endl;
		outfile_WaterBudget_csv << "TS_First_SurficialGroundwater," << input->RepoVecDict["Groundwater_surficial_TS_first_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_SurficialGroundwater," << input->RepoVecDict["Groundwater_surficial_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Precipitation," << input->TotalPrecip_m * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "PrecipitationRain," << input->TotalRain_m * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "PrecipitationSnow," << input->TotalSnow_m * Ratio_mm_to_m << ",mm" << endl;

		outfile_WaterBudget_csv << "Irrigation_TreeInterception_mm," << sumAcrossFolderTypes(input, "Irrigation_TreeInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_csv << "Irrigation_SVegInterception_mm," << sumAcrossFolderTypes(input, "Irrigation_SVegInterceptionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_csv << "Irrigation_PerviousDepressionStorage_mm," << sumAcrossFolderTypes(input, "Irrigation_PerviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;
		outfile_WaterBudget_csv << "Irrigation_ImperviousDepressionStorage_mm," << sumAcrossFolderTypes(input, "Irrigation_ImperviousDepressionStorage_Sum_m3") / CA_m2 * Ratio_mm_to_m << endl;

		outfile_WaterBudget_csv << "InterceptionRainSnow_TreeCanopy," << input->RepoVecDict["InterceptionRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "InterceptionRainSnow_SVegCanopy," << input->RepoVecDict["InterceptionRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "EvaporationSublimation_TreeCanopy," << input->RepoVecDict["EvaporationSublimation_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "EvaporationSublimation_SVegCanopy," << input->RepoVecDict["EvaporationSublimation_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageRain_TreeCanopy," << input->RepoVecDict["StorageRain_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageSnow_TreeCanopy," << input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageRain_SVegCanopy," << input->RepoVecDict["StorageRain_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageSnow_SVegCanopy," << input->RepoVecDict["StorageSnow_SVegCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "ThroughFallRain_TreeCanopy," << input->RepoVecDict["ThroughFallRain_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "ThroughFallSnow_TreeCanopy," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "ThroughFallRain_SVegCanopy," << input->RepoVecDict["ThroughFallRain_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "ThroughFallSnow_SVegCanopy," << input->RepoVecDict["ThroughFallSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageSnow_noCanopy," << input->RepoVecDict["StorageSnow_noCanopy_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageSnow_underTreeCanopy," << input->RepoVecDict["StorageSnow_UnderTree_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageSnow_underSVegCanopy," << input->RepoVecDict["StorageSnow_UnderSVeg_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "InterceptionSnow_TreeCanopy," << input->RepoVecDict["InterceptionSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "InterceptionSnow_SVegCanopy," << input->RepoVecDict["InterceptionSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SublimationSnow_TreeCanopy," << input->RepoVecDict["SublimationSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SublimationSnow_SVegCanopy," << input->RepoVecDict["SublimationSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SublimationSnow_UnderTree," << input->RepoVecDict["SublimationSnow_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SublimationSnow_UnderSVeg," << input->RepoVecDict["SublimationSnow_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SublimationSnow_noCanopy," << input->RepoVecDict["SublimationSnow_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SnowMelt_TreeCanopy," << input->RepoVecDict["SnowMelt_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "SnowMelt_SVegCanopy," << input->RepoVecDict["SnowMelt_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Snow_Melt_UnderTree," << input->RepoVecDict["SnowMelt_UnderTree_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Snow_Melt_UnderShortVeg," << input->RepoVecDict["SnowMelt_UnderSVeg_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Snow_Melt_OpenArea," << input->RepoVecDict["SnowMelt_noCanopy_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_to_ImperviousArea_Rain_SnowMelt," << input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Evaporation_ImperviousPondedWater," << input->RepoVecDict["Evaporation_ImperviousPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_Storage_ImperviousPonded," << input->RepoVecDict["Storage_ImperviousPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_Impervious_to_Pervious," << input->RepoVecDict["Flux_Impervious_to_Pervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_to_WaterArea_Rain_SnowMelt," << input->RepoVecDict["Flux_to_WaterArea_Rain_SnowMelt_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Evaporation_WaterPondedWater," << input->RepoVecDict["Evaporation_WaterPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_Storage_WaterPonded," << input->RepoVecDict["Storage_WaterPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_to_PerviousArea_Rain_SnowMelt_ImpRunon," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_to_PerviousArea_Rain_SnowMelt," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Evaporation_PerviousPondedWater," << input->RepoVecDict["Evaporation_PerviousPondedWater_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_Storage_PerviousPonded," << input->RepoVecDict["Storage_PerviousPondedWater_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_First_Storage_SoilEvapZone," << input->RepoVecDict["Storage_SoilEvapZone_TS_first_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_Storage_SoilEvapZone," << input->RepoVecDict["Storage_SoilEvapZone_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_First_Storage_VadoseZone," << input->RepoVecDict["Storage_VadoseZone_TS_first_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_Storage_VadoseZone," << input->RepoVecDict["Storage_VadoseZone_TS_last_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;

		outfile_WaterBudget_csv << "TS_First_StorageDeficit_VadoseZone," << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "TS_Last_StorageDeficit_VadoseZone," << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_mm_to_m << ",mm" << endl;

		outfile_WaterBudget_csv << "Drainage_macroPore," << input->RepoVecDict["Drainage_macroPore_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Infiltration," << input->RepoVecDict["Infiltration_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Infiltration_viaInfilEx," << input->RepoVecDict["Infiltration_viaInfilEx_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Infiltration_viaSatEx," << input->RepoVecDict["Infiltration_viaSatEx_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "EvapoTranspiration," << input->RepoVecDict["EvapoTranspiration_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Drainage_SoilEvapZone," << input->RepoVecDict["Drainage_SoilEvapZone_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Drainage_VadoseZone," << input->RepoVecDict["Drainage_VadoseZone_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_SatExGW_to_SoilFieldCapacity," << input->RepoVecDict["Flux_SatExGW_to_SoilFieldCapacity_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;

		//Note: Fluxes from BulkArea to GI reported
		outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_WaterCover_Runoff," << input->RepoVecDict["Runoff_Water_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_ImpCover_Runoff," << input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_PerCover_InfilExRunoff," << input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Flux_BulkArea_to_GI_PerCover_SatExRunoff," << input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;

		//Note: Runoff variables are folder depths (m), not cover type depths, and were adjusted to folder depths in RunoffSummation.cpp or ImperviousDepressionStorageCalc.cpp for IC
		outfile_WaterBudget_csv << "Runoff_Impervious," << input->RepoVecDict["Runoff_Impervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Runoff_Water," << input->RepoVecDict["Runoff_Water_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Runoff_Pervious," << input->RepoVecDict["Runoff_Pervious_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Runoff_InfilEx," << input->RepoVecDict["Runoff_InfilExcess_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Runoff_SatExcess," << input->RepoVecDict["Runoff_SatExcess_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Runoff_Subsurface," << input->RepoVecDict["Runoff_Subsurface_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << "Runoff," << input->RepoVecDict["Runoff_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
		outfile_WaterBudget_csv << endl;

		//If Folder_Type_ID equals Folder_BulkArea_ID, then write out BulkArea data
		if (Folder_Type_ID == Folder_BulkArea_ID) {
			outfile_WaterBudget_csv << "Bulk Area Water Balance Summary Variable,Value,Units" << endl;
			outfile_WaterBudget_csv << "Eq_Tree_Canopy_WaterBalance," << input->RepoDict["Eq_Tree_Canopy_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_SVeg_Canopy_WaterBalance," << input->RepoDict["Eq_SVeg_Canopy_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_Snow_WaterBalance," << input->RepoDict["Eq_Snow_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_WC_SurfaceInflux_WaterBalance," << input->RepoDict["Eq_WC_SurfaceInflux_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_WC_SnowMelt_WaterBalance," << input->RepoDict["Eq_WC_SnowMelt_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_WC_SurfaceRunoff_WaterBalance," << input->RepoDict["Eq_WC_SurfaceRunoff_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_IC_SurfaceInflux_WaterBalance," << input->RepoDict["Eq_IC_SurfaceInflux_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_IC_SnowMelt_WaterBalance," << input->RepoDict["Eq_IC_SnowMelt_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_IC_SurfaceRunoff_WaterBalance," << input->RepoDict["Eq_IC_SurfaceRunoff_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_PC_SurfaceInflux_WaterBalance," << input->RepoDict["Eq_PC_SurfaceInflux_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_PC_SnowMelt_WaterBalance," << input->RepoDict["Eq_PC_SnowMelt_WaterBalance_mm "] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_PC_SurfaceRunoff_Infiltration_WaterBalance," << input->RepoDict["Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_PC_VadoseZone_WaterBalance," << input->RepoDict["Eq_PC_VadoseZone_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_PC_SatZone_WaterBalance," << input->RepoDict["Eq_PC_SatZone_WaterBalance_mm"] << ",mm" << endl;
		}
		//End if (Folder_Type_ID == Folder_BulkArea_ID)

		//If Folder_Type_ID equals Folder_GI_ID, then write out Green Infrastructure data
		if (Folder_Type_ID == Folder_GI_ID) {
			outfile_WaterBudget_csv << "GI_Precipitation," << input->RepoDict["GI_Precipitation_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Inflow_from_BulkArea," << input->RepoDict["GI_Runon_from_BulkArea_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Inflow_from_GreenInfrastructure," << GI_Runon_from_GI_Sum_mm << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Inflow_from_GI_WaterCover," << input->RepoVecDict["Runoff_Water_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Inflow_from_GI_ImpCover," << input->RepoVecDict["Runoff_Impervious_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Inflow_from_GI_PerCover," << input->RepoVecDict["Runoff_Pervious_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Inflow_from_GI_Vault," << input->RepoVecDict["Runoff_Vault_GItoGI_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Infiltration," << input->RepoDict["GI_Infiltration_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Storage_Surface_TS_Last," << input->RepoDict["GI_Storage_Surface_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_PavementWater_TS_Last," << input->RepoDict["GI_PavementWater_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Storage_Soil_TS_0th," << input->RepoDict["GI_Storage_Soil_TS_0th_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Storage_Soil_TS_Last," << input->RepoDict["GI_Storage_Soil_TS_last_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_Storage_Vault_TS_Last," << input->RepoDict["GI_Storage_Vault_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_EvaporationSurface," << input->RepoDict["GI_Evaporation_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_EvaporationPavement," << input->RepoDict["GI_Evaporation_Pavement_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_EvaporationSoil," << input->RepoDict["GI_Evaporation_Soil_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_EvaporationVault," << input->RepoDict["GI_Evaporation_Vault_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_DrainageVadoseZone," << input->RepoDict["GI_Drainage_VadoseZone_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_RunoffSurface," << input->RepoDict["GI_Runoff_Surface_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_RunoffVaultDrain," << input->RepoDict["GI_Runoff_VaultDrain_Sum_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << "GI_GroundwaterMound_TS_Last," << input->RepoDict["GI_Storage_GroundwaterMound_TS_Last_m3"] / FA_m2 * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << endl;
			outfile_WaterBudget_csv << "Green Infrastructure Water Balance and Layer Summary Variable,Value,Units" << endl;
			outfile_WaterBudget_csv << "Eq_GI_Tree_Canopy_WaterBalance," << input->RepoDict["Eq_Tree_Canopy_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_GI_SVeg_Canopy_WaterBalance," << input->RepoDict["Eq_SVeg_Canopy_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_GI_RunonSurface_WaterBalance," << input->RepoDict["Eq_GI_RunonSurface_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_GI_Surface_WaterBalance," << input->RepoDict["Eq_GI_Surface_WaterBalance_mm"] << ",mm" << endl;
			outfile_WaterBudget_csv << "Eq_GI_All_Layers_WaterBalance," << input->RepoDict["Eq_GI_All_Layers_WaterBalance_mm"] << ",mm" << endl;
		}
		//End if (Folder_Type_ID == Folder_GI_ID)

		//Write out specific water balances for each component of vertical system
		outfile_WaterBudget_csv.precision(10);
		outfile_WaterBudget_csv << endl;
		outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
		outfile_WaterBudget_csv << Name_Folder_Type_str << " Water Balance Equations (Eq.)" << endl;
		outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
		outfile_WaterBudget_csv << endl;
		outfile_WaterBudget_csv << " Eq_Tree_Canopy_WaterBalance_mm (mm) relative to Folder area" << endl;
		outfile_WaterBudget_csv << " Eq_Tree_Canopy_WaterBalance_mm = FA_TC_Total_Precip_RainSWE_mm + FA_TC_Total_Irrigation_mm - FA_TC_Storage_Canopy_TS_last_mm - FA_TC_Evap_mm - FA_TC_ThroughFall_mm " << endl;
		outfile_WaterBudget_csv << scientific << " = " << FA_TC_Total_Precip_RainSWE_mm << " + " << FA_TC_Total_Irrigation_mm << " - " << FA_TC_Storage_Canopy_TS_last_mm << " - " << FA_TC_Evap_mm << " - " << FA_TC_ThroughFall_mm << endl;
		outfile_WaterBudget_csv << scientific << " = " << Eq_Tree_Canopy_WaterBalance_mm << endl;
		outfile_WaterBudget_csv << endl;

		outfile_WaterBudget_csv << " Eq_SVeg_Canopy_WaterBalance_mm (mm) relative to Folder area" << endl;
		outfile_WaterBudget_csv << " Eq_SVeg_Canopy_WaterBalance_mm = FA_SV_Total_Precip_RainSWE_mm + FA_SV_Total_Irrigation_mm - FA_SV_Storage_Canopy_TS_last_mm - FA_SV_Evap_mm - FA_SV_ThroughFall_mm  " << endl;
		outfile_WaterBudget_csv << " = " << FA_SV_Total_Precip_RainSWE_mm << " + " << FA_SV_Total_Irrigation_mm << " - " << FA_SV_Storage_Canopy_TS_last_mm << " - " << FA_SV_Evap_mm << " - " << FA_SV_ThroughFall_mm << endl;
		outfile_WaterBudget_csv << " = " << Eq_SVeg_Canopy_WaterBalance_mm << endl;
		outfile_WaterBudget_csv << endl;

		//If Folder_Type_ID equals Folder_BulkArea_ID, then write out BulkArea data
		if (Folder_Type_ID == Folder_BulkArea_ID) {

			outfile_WaterBudget_csv << " Eq_IC_SurfaceInflux_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_IC_SurfaceInflux_WaterBalance_mm =  FA_IC_noTC_Precip_mm + FA_IC_TC_Throughfall_mm + FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_TC_Sublimation_mm - FA_IC_TC_Storage_Ground_Snow_TS_last_mm - FA_IC_noTC_Sublimation_mm - FA_IC_noTC_Storage_Ground_Snow_TS_last_mm" << endl;
			outfile_WaterBudget_csv << " = " << FA_IC_noTC_Precip_mm << " + " << FA_IC_TC_Throughfall_mm << " + " << FA_IC_Total_Irrigation_mm << " - " << FA_IC_TC_Sublimation_mm << " - " << FA_IC_TC_Storage_Ground_Snow_TS_last_mm << " - " << FA_IC_noTC_Sublimation_mm << " - " << FA_IC_noTC_Storage_Ground_Snow_TS_last_mm << " - " << FA_IC_FluxRainSnowMeltIrrigation_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_IC_SurfaceInflux_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_IC_SnowMelt_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_IC_SnowMelt_WaterBalance_mm = FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_noTC_Rain_mm - FA_IC_TC_ThroughfallRain_mm - FA_IC_noTC_SnowMelt_mm - FA_IC_underTC_SnowMelt_mm - FA_IC_Total_Irrigation_mm" << endl;
			outfile_WaterBudget_csv << " = " << FA_IC_FluxRainSnowMeltIrrigation_mm << " - " << FA_IC_noTC_Rain_mm << " - " << FA_IC_TC_ThroughfallRain_mm << " - " << FA_IC_noTC_SnowMelt_mm << " - " << FA_IC_underTC_SnowMelt_mm << " - " << FA_IC_Total_Irrigation_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_IC_SnowMelt_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_IC_SurfaceRunoff_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_IC_SurfaceRunoff_WaterBalance_mm = FA_IC_FluxRainSnowMeltIrrigation_mm - FA_IC_Storage_Depression_TS_last_mm - FA_IC_Evaporation_Depression_mm - FA_IC_Flux_to_PC_mm - FA_IC_Runoff_mm - FA_IC_Flux_BA_to_GI_mm" << endl;
			outfile_WaterBudget_csv << " = " << FA_IC_FluxRainSnowMeltIrrigation_mm << " - " << FA_IC_Storage_Depression_TS_last_mm << " - " << FA_IC_Evaporation_Depression_mm << " - " << FA_IC_Flux_to_PC_mm << " - " << FA_IC_Runoff_mm << " - " << FA_IC_Flux_BA_to_GI_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_IC_SurfaceRunoff_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_PC_SurfaceInflux_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_PC_SurfaceInflux_WaterBalance_mm = FA_PC_TC_ThroughFall_mm + FA_SV_ThroughFall_mm + FA_PC_noTC_noSV_Precip_mm + FA_PC_Flux_IC_to_PC_mm + FA_PC_Total_Irrigation_mm - FA_PC_underTC_Sublimation_mm - FA_PC_TC_StorageFinal_Ground_Snow_mm - FA_PC_underSV_Sublimation_mm - FA_SV_StorageFinal_Ground_Snow_mm - FA_PC_noTC_noSV_Sublimation_mm -  FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm - FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm" << endl;
			outfile_WaterBudget_csv << " = " << FA_PC_TC_ThroughFall_mm << " + " << FA_SV_ThroughFall_mm << " + " << FA_PC_noTC_noSV_Precip_mm << " + " << FA_PC_Flux_IC_to_PC_mm << " + " << FA_PC_Total_Irrigation_mm << " - " << FA_PC_underTC_Sublimation_mm << " - " << FA_PC_TC_StorageFinal_Ground_Snow_mm << " - " << FA_PC_underSV_Sublimation_mm << " - " << FA_SV_StorageFinal_Ground_Snow_mm << " - " << FA_PC_noTC_noSV_Sublimation_mm << " - " << FA_PC_noTC_noSV_StorageFinal_Ground_Snow_mm << " - " << FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_PC_SurfaceInflux_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_PC_SnowMelt_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_PC_SnowMelt_WaterBalance_mm = FA_PC_FluxRainSnowMelt_mm - FA_PC_Soil_noCanopy_Rain_mm - FA_PC_TC_ThroughfallRain_mm - FA_PC_SV_ThroughfallRain_mm - FA_PC_Total_Irrigation_mm - FA_PC_Soil_noCanopy_SnowMelt_mm - FA_PC_PermeablePavement_noCanopy_SnowMelt_mm - FA_PC_underTC_SnowMelt_mm - FA_PC_underSV_SnowMelt_mm " << endl;
			outfile_WaterBudget_csv << " = " << FA_PC_FluxRainSnowMelt_mm << " - " << FA_PC_Soil_noCanopy_Rain_mm << " - " << FA_PC_TC_ThroughfallRain_mm << " - " << FA_PC_SV_ThroughfallRain_mm << " - " << FA_PC_Total_Irrigation_mm << " - " << FA_PC_Soil_noCanopy_SnowMelt_mm << " - " << FA_PC_PermeablePavement_noCanopy_SnowMelt_mm << " - " << FA_PC_underTC_SnowMelt_mm << " - " << FA_PC_underSV_SnowMelt_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_PC_SnowMelt_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm = FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm - FA_PC_Runoff_InfilExcess_mm - FA_PC_Runoff_SatExcess_mm - FA_PC_Infiltration_InfilExcess_mm - FA_PC_Infiltration_SatExcess_mm - FA_PC_Drainage_Macropore_mm - FA_PC_Evaporation_Depression_mm - FA_PC_Storage_Depression_TS_last_mm - FA_PC_InfExRO_Flux_to_GI_mm - FA_PC_SatExRO_Flux_to_GI_mm" << endl;
			outfile_WaterBudget_csv << " = " << FA_PC_FluxRainSnowMelt_Irrigation_ImpRunon_mm << " - " << FA_PC_Runoff_InfilExcess_mm << " - " << FA_PC_Runoff_SatExcess_mm << " - " << FA_PC_Infiltration_InfilExcess_mm << " - " << FA_PC_Infiltration_SatExcess_mm << " - " << FA_PC_Drainage_Macropore_mm << " - " << FA_PC_Evaporation_Depression_mm << " - " << FA_PC_Storage_Depression_TS_last_mm << " - " << FA_PC_InfExRO_Flux_to_GI_mm << " - " << FA_PC_SatExRO_Flux_to_GI_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_PC_SurfaceRunoff_Infiltration_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_PC_VadoseZone_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_PC_VadoseZone_WaterBalance_mm = FA_PC_Infiltration_InfilExcess_mm + FA_PC_Infiltration_SatExcess_mm + FA_PC_Drainage_Macropore_mm + FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm - FA_PC_StorageChange_SoilEvapZone_mm - FA_PC_EvapTranspiration_VadoseZone_mm - FA_PC_Drainage_VadoseZone_mm - FA_PC_Storage_VadoseZone_TS_last_mm " << endl;
			outfile_WaterBudget_csv << " = " << FA_PC_Infiltration_InfilExcess_mm << " + " << FA_PC_Infiltration_SatExcess_mm << " + " << FA_PC_Drainage_Macropore_mm << " + " << FA_PC_Flux_SatExGW_to_SoilFieldCapacity_mm << " - " << FA_PC_StorageChange_SoilEvapZone_mm << " - " << FA_PC_EvapTranspiration_VadoseZone_mm << " - " << FA_PC_Drainage_VadoseZone_mm << " - " << FA_PC_Storage_VadoseZone_TS_last_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_PC_VadoseZone_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_Snow_WaterBalance_mm (mm) relative to Folder area" << endl;
			outfile_WaterBudget_csv << " Eq_Snow_WaterBalance_mm = FA_Precip_Snow_mm - FA_Total_Sublimation_Ground_Snow_mm - FA_Total_Sublimation_Canopy_Snow_mm - FA_Total_Melt_Ground_Snow_mm - FA_Total_Melt_Canopy_Snow_mm - FA_Total_Storage_Ground_Snow_mm - FA_TS_Last_Canopy_Snow_mm" << endl;
			outfile_WaterBudget_csv << " = " << FA_Precip_Snow_mm << " - " << FA_Total_Sublimation_Ground_Snow_mm << " - " << FA_Total_Sublimation_Canopy_Snow_mm << " - " << FA_Total_Melt_Ground_Snow_mm << " - " << FA_Total_Melt_Canopy_Snow_mm << " - " << FA_Total_Storage_Ground_Snow_mm << " - " << FA_TS_Last_Canopy_Snow_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_Snow_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_PC_SatZone_WaterBalance_mm (mm) relative to Catchment area" << endl;
			outfile_WaterBudget_csv << " Eq_PC_SatZone_WaterBalance_mm = CA_PC_StorageDeficit_VadoseZone_TS_first_mm + CA_PC_Runoff_Subsurface_BA_and_GI_mm - CA_PC_Drainage_VadoseZone_BA_and_GI_mm - CA_PC_StorageDeficit_VadoseZone_TS_last_mm - CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm" << endl;
			outfile_WaterBudget_csv << " = " << CA_PC_StorageDeficit_VadoseZone_TS_first_mm << " + " << CA_PC_Runoff_Subsurface_BA_and_GI_mm << " - " << CA_PC_Drainage_VadoseZone_BA_and_GI_mm << " - " << CA_PC_StorageDeficit_VadoseZone_TS_last_mm << " - " << CA_PC_Runoff_Subsurface_BA_and_GI_TS_first_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_PC_SatZone_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;
		}
		//End if (Folder_Type_ID == Folder_BulkArea_ID)

		//If Folder_Type_ID equals Folder_GI_ID, then write out Green Infrastructure data
		if (Folder_Type_ID == Folder_GI_ID) {

			outfile_WaterBudget_csv << " Eq_GI_RunonSurface_WaterBalance_mm (mm) relative to Green Infrastructure area" << endl;
			outfile_WaterBudget_csv << " Eq_GI_RunonSurface_WaterBalance_mm = GI_Runon_from_BulkArea_WaterCoverRunoff_mm + GI_Runon_from_BulkArea_ImpCoverRunoff_mm + GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm + GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm + GI_Runon_from_GI_WaterCoverRunoff_mm + GI_Runon_from_GI_ImpCoverRunoff_mm + GI_Runon_from_GI_PerCoverRunoff_mm + GI_Runon_from_GI_VaultRunoff_mm - GI_Runon_from_BulkArea_Sum_mm - GI_Runon_from_GI_Sum_mm" << endl;
			outfile_WaterBudget_csv << " = " << GI_Runon_from_BulkArea_WaterCoverRunoff_mm << " + " << GI_Runon_from_BulkArea_ImpCoverRunoff_mm << " + " << GI_Runon_from_BullkArea_PervCover_InfilExRunoff_mm << " + " << GI_Runon_from_BulkArea_PervCover_SatExRunoff_mm << " + " << GI_Runon_from_GI_WaterCoverRunoff_mm << " + " << GI_Runon_from_GI_ImpCoverRunoff_mm << " + " << GI_Runon_from_GI_PerCoverRunoff_mm << " + " << GI_Runon_from_GI_VaultRunoff_mm << " - " << GI_Runon_from_BulkArea_Sum_mm << " - " << GI_Runon_from_GI_Sum_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_GI_RunonSurface_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_GI_Surface_WaterBalance_mm (mm) relative to Green Infrastructure area" << endl;
			outfile_WaterBudget_csv << " Eq_GI_Surface_WaterBalance_mm = GI_Precipitation_Sum_mm + GI_Runon_from_BulkArea_Sum_mm + GI_Runon_from_GI_Sum_mm - GI_Infiltration_Sum_mm - GI_RunoffSurface_Sum_mm - GI_Storage_Surface_TS_Last_mm - GI_EvaporationSurface_Sum_mm" << endl;
			outfile_WaterBudget_csv << " = " << GI_Precipitation_Sum_mm << " + " << GI_Runon_from_BulkArea_Sum_mm << " + " << GI_Runon_from_GI_Sum_mm << " - " << GI_Infiltration_Sum_mm << " - " << GI_RunoffSurface_Sum_mm << " - " << GI_Storage_Surface_TS_Last_mm << " - " << GI_EvaporationSurface_Sum_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_GI_Surface_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << " Eq_GI_All_Layers_WaterBalance_mm (mm), relative to Green Infrastructure area" << endl;
			outfile_WaterBudget_csv << " Eq_GI_All_Layers_WaterBalance_mm = GI_Precipitation_Sum_mm + GI_Runon_from_BulkArea_Sum_mm + GI_Runon_from_GI_Sum_mm - GI_RunoffSurface_Sum_mm - GI_RunoffVaultDrain_Sum_mm - GI_Drainage_VadoseZone_Sum_mm - GI_Storage_Surface_TS_Last_mm - GI_PavementWater_TS_Last_mm - (GI_Storage_Soil_TS_last_mm - GI_Storage_Soil_TS_0th_mm) - GI_Storage_Vault_TS_Last_mm - GI_EvaporationSurface_Sum_mm - GI_EvaporationPavement_Sum_mm - GI_EvaporationSoil_Sum_mm - GI_EvaporationVault_Sum_mm + GI_GoundwaterMound_DynamicEquilibrium_mm" << endl;
			outfile_WaterBudget_csv << " = " << GI_Precipitation_Sum_mm << " + " << GI_Runon_from_BulkArea_Sum_mm << " + " << GI_Runon_from_GI_Sum_mm << " - " << GI_RunoffSurface_Sum_mm << " - " << GI_RunoffVaultDrain_Sum_mm << " - " << GI_Drainage_VadoseZone_Sum_mm << " - " << GI_Storage_Surface_TS_Last_mm << " - " << GI_PavementWater_TS_Last_mm << " - (" << GI_Storage_Soil_TS_last_mm << " - " << GI_Storage_Soil_TS_0th_mm << ") - " << GI_Storage_Vault_TS_Last_mm << " - " << GI_EvaporationSurface_Sum_mm << " - " << GI_EvaporationPavement_Sum_mm << " - " << GI_EvaporationSoil_Sum_mm << " - " << GI_EvaporationVault_Sum_mm << " + " << GI_GoundwaterMound_DynamicEquilibrium_mm << endl;
			outfile_WaterBudget_csv << " = " << Eq_GI_All_Layers_WaterBalance_mm << endl;
			outfile_WaterBudget_csv << endl;
		}
		//End if (Folder_Type_ID == Folder_GI_ID)

		//If Folder_Type_ID equals Folder_List.back(), last item in list, then write topographic index data for catchment
		if (Folder_Type_ID == Folder_List.back()) {
			//======================================================
			//Eq. 19 Runoff balance (mm) UnRouted - Routed (mm) relative to Folder area
			double total_Runoff_Impervious_mm, total_Runoff_Pervious_mm, total_Runoff_Water_mm, total_Runoff_Subsurface_mm, total_Runoff_mm, total_Runoff_Impervious_R_mm, total_Runoff_Pervious_R_mm, total_Runoff_Water_R_mm, total_Runoff_Subsurface_R_mm, total_Runoff_R_mm, CA_RunoffRouting_Balance_mm;
			//Catchment runoff divides variables of volume (m3) by catchment area CA_m2 (m2)
			total_Runoff_Impervious_mm = input->RepoDict["Runoff_Impervious_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_Pervious_mm = input->RepoDict["Runoff_Pervious_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_Water_mm = input->RepoDict["Runoff_Water_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_Subsurface_mm = input->RepoDict["Runoff_Subsurface_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			total_Runoff_mm = input->RepoDict["Runoff_Catchment_m3"] / CA_m2 * Ratio_mm_to_m;
			//Routed runoff depths (m) were generated in RoutingDiffusion::ScalingRunoff function
			total_Runoff_Impervious_R_mm = input->RepoDict["Runoff_Impervious_R_m"] * Ratio_mm_to_m;
			total_Runoff_Pervious_R_mm = input->RepoDict["Runoff_Pervious_R_m"] * Ratio_mm_to_m;
			total_Runoff_Water_R_mm = input->RepoDict["Runoff_Water_R_m"] * Ratio_mm_to_m;
			total_Runoff_Subsurface_R_mm = input->RepoDict["Runoff_Subsurface_R_m"] * Ratio_mm_to_m;
			total_Runoff_R_mm = input->RepoDict["Runoff_R_m"] * Ratio_mm_to_m;
			//Balance is simple input = (non-routed - routed) 
			CA_RunoffRouting_Balance_mm = (total_Runoff_Impervious_mm - total_Runoff_Impervious_R_mm) + (total_Runoff_Pervious_mm - total_Runoff_Pervious_R_mm) + (total_Runoff_Water_mm - total_Runoff_Water_R_mm) + (total_Runoff_Subsurface_mm - total_Runoff_Subsurface_R_mm) + (total_Runoff_mm - total_Runoff_R_mm);

			outfile_WaterBudget_csv << endl;
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_csv << "Total Catchment Runoff:" << endl;
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_csv << " CA_RunoffRouting_Balance_mm = (total_Runoff_Impervious_mm - total_Runoff_Impervious_R_mm) + (total_Runoff_Pervious_mm - total_Runoff_Pervious_R_mm) + (total_Runoff_Water_mm - total_Runoff_Water_R_mm) + (total_Runoff_Subsurface_mm - total_Runoff_Subsurface_R_mm) + (total_Runoff_mm - total_Runoff_R_mm)" << endl;
			outfile_WaterBudget_csv << " = (" << total_Runoff_Impervious_mm << " - " << total_Runoff_Impervious_R_mm << ") + (" << total_Runoff_Pervious_mm << " - " << total_Runoff_Pervious_R_mm << ") + (" << total_Runoff_Water_mm << " - " << total_Runoff_Water_R_mm << ") + (" << total_Runoff_Subsurface_mm << " - " << total_Runoff_Subsurface_R_mm << ") + (" << total_Runoff_mm << " - " << total_Runoff_R_mm << ")" << endl;
			outfile_WaterBudget_csv << " = " << CA_RunoffRouting_Balance_mm << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv << endl;
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
			outfile_WaterBudget_csv << "Topographic Index Calculation Approach (Exponential or Power Decay) and Parameters:" << endl;
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;

			//References for Topographic Index theory
			//Famiglietti, J.S., & Wood, E.F. (1994).Multiscale Modeling of Spatially Variable Water and Energy Balance Processes.Water Resources Research, 30(11), 3061 - 3078.
			//Hornberger, G. M., Beven, K. J., Cosby, B. J., & Sappington, D. E. (1985). Shenandoah Watershed Study: Calibration of a Topography-Based, Variable Contributing Area Hydrological Model to a Small Forested Catchment. Water Resources Research, 21(12), 1841-1850.
			//TMOD9502.FOR TOPMODEL DEMONSTRATION PROGRAM VERSION 95.02, By Keith Beven or Eq given as misc.S_mean[0] = - params.m * log(misc.qs0 / misc.qss) in parameter.init.c https://cran.r-project.org/web/packages/topmodel/index.html
			//Wang, J., Endreny, T. A., & Hassett, J. M. (2006). Power function decay of hydraulic conductivity for a TOPMODEL-based infiltration routine Hydrological Processes, 20(18), 3825-3834.
			//Wang, J., Endreny, T. A., & Hassett, J. M. (2005). Flexible Modeling Package for Topographically Based Watershed Hydrology. Journal of Hydrology, 314(1-4), 78-91. 
			//Note: Discharge_Subsurface_Max_mph = SZQ | Discharge_Subsurface_Initial_mph = Q0_m_p_dt | VadoseZone_Transmissivity_Max_m2ph = T0_m2_p_dt | ...
			//Note: StorageDeficit_VadoseZone_m = AveSMD | Parameter_m_KsatExpDecay = m_ScalingParam | Parameter_n_KsatPowerDecay = n_PowerDecayParam
			double Discharge_Subsurface_Max_mph, Discharge_Subsurface_Initial_mph, VadoseZone_Transmissivity_Max_m2ph, TopographicIndex_Avg, StorageDeficit_VadoseZone_m, Parameter_m_KsatExpDecay, Parameter_n_KsatPowerDecay;
			//StorageDeficit_VadoseZone_TI_m defined as vector<double> containing input->SimulationNumericalParams["TopographicIndexBins"]
			vector<double> StorageDeficit_VadoseZone_TI_m(input->SimulationNumericalParams["TopographicIndexBins"]);
			Discharge_Subsurface_Initial_mph = input->InputXml["Discharge_Subsurface_Initial_mph"];
			VadoseZone_Transmissivity_Max_m2ph = input->InputXml["VadoseZone_Transmissivity_Max_m2ph"];
			TopographicIndex_Avg = input->TI_Value_Average;
			Parameter_m_KsatExpDecay = input->InputXml["Parameter_m_KsatExpDecay"];

			//Infiltration type for decay function with hydraulic conductivity and computation of topographic index
			//Note: Consider refactor to remove this code and simply write out values computed from Inputs functions
			if (input->SimulationStringParams["Algorithm_SoilKsatDecay"] == "ExponentialDecay") {
				outfile_WaterBudget_csv << "HydraulicConductivityDecay_function,ExponentialDecay" << endl;
				//Discharge_Subsurface_Max_mph(m) is maximum potential subsurface flow rate, when groundwater table is at the ground surface
				//Eq for Discharge_Subsurface_Max_mph similar to Eq 31b for Q0 in Famiglietti and Wood (1994)
				//Eq for Discharge_Subsurface_Max_mph below Eq 3 of Wang et al. (2005), S_bar = -m * ln(R / T0) - m * Lambda, and Discharge_Subsurface_Max_mph becomes T0* exp(-Lambda), where R = Q0
				Discharge_Subsurface_Max_mph = VadoseZone_Transmissivity_Max_m2ph * exp(-1 * TopographicIndex_Avg);
				//Eq for StorageDeficit_VadoseZone_m derived from Eq 4 in Hornberger et al. (1985) Qb = Discharge_Subsurface_Max_mph * EXP(-S / SZM)
				//Eq for StorageDeficit_VadoseZone_m Eq given as SBAR = -SZM * ALOG(Q0 / Discharge_Subsurface_Max_mph) in TMOD9502.FOR or parameter.init.c
				//Eq for StorageDeficit_VadoseZone_m below Eq 3 of Wang et al. (2005), S_bar = -m * ln(R / T0) - m * Lambda
				//StorageDeficit_VadoseZone_m computation uses Natural log of(Discharge_Subsurface_Initial_mph / Discharge_Subsurface_Max_mph)
				//StorageDeficit_VadoseZone_m adjusted to PC_frac, which is fraction of catchment with soil moisture

				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				if (Discharge_Subsurface_Max_mph > 0 and Discharge_Subsurface_Initial_mph > 0) {
					StorageDeficit_VadoseZone_m = -1 * Parameter_m_KsatExpDecay * log(Discharge_Subsurface_Initial_mph / Discharge_Subsurface_Max_mph) * PC_frac;
				}
				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				else {
					StorageDeficit_VadoseZone_m = 0.0;
				}
			}
			if (input->SimulationStringParams["Algorithm_SoilKsatDecay"] == "PowerDecay") {
				outfile_WaterBudget_csv << "HydraulicConductivityDecay_function,PowerDecay" << endl;
				Parameter_n_KsatPowerDecay = input->InputXml["Parameter_n_KsatPowerDecay"];
				//Discharge_Subsurface_Max_mph(m) is maximum potential subsurface flow rate, when groundwater table is at the ground surface
				//Eq for Discharge_Subsurface_Max_mph is RHS of Eq for S_bar, StorageDeficit_VadoseZone_m, given in text below Eq 9 of Wang et al. (2006)
				Discharge_Subsurface_Max_mph = pow((Discharge_Subsurface_Initial_mph / VadoseZone_Transmissivity_Max_m2ph), (1 / Parameter_n_KsatPowerDecay)) * TopographicIndex_Avg;

				//Eq for StorageDeficit_VadoseZone_m is below Eq 9 of Wang et al. (2006) and below Eq 5 of Wang(2005)
				//StorageDeficit_VadoseZone_m (m) adjusted to PC_frac, which is fraction of catchment with soil moisture
				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				if (Discharge_Subsurface_Max_mph > 0 and Discharge_Subsurface_Initial_mph > 0) {
					StorageDeficit_VadoseZone_m = Parameter_m_KsatExpDecay * (1 - Discharge_Subsurface_Max_mph) * PC_frac;
				}
				//If ParamStringDict["Type"] != RainBarrel or RoofDisconnect and Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph <= 0 in HydroPlusConfig.xml, then model will abort in BuildDataOrganizer to avoid average SMD going to infinity
				// ... otherwise for Type = RainBarrel or RoofDisconnect it allows Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph as zero values in HydroPlusConfig.xml and sets StorageDeficit_VadoseZone_m = 0, which may represent Rain Barrel etc.
				else {
					//If Q0 or T0 are entered as zero values in HydroPlusConfig.xml, then set StorageDeficit_VadoseZone_m = 0
					StorageDeficit_VadoseZone_m = 0.0;
				}

			}
			outfile_WaterBudget_csv.precision(3);
			//Timestep for adjusting rate constants from HydroPlusConfig.xml
			outfile_WaterBudget_csv << "TimeStep_sec," << input->SimulationNumericalParams["TimeStep_sec"] << ",s" << endl;
			//Number of topographic index bins
			outfile_WaterBudget_csv << "TopographicIndexBins," << fixed << input->SimulationNumericalParams["TopographicIndexBins"] << ",#" << endl;
			//Average topographic index value
			outfile_WaterBudget_csv << "TopographicIndex_Avg," << fixed << input->TI_Value_Average << ",unitless" << endl;
			//Scaling parameter for adjusting soil moisture across topographic index
			outfile_WaterBudget_csv << "Parameter_m_KsatExpDecay," << scientific << input->InputXml["Parameter_m_KsatExpDecay"] << ",unitless" << endl;
			//Power decay parameter for adjusting conductivity across topographic index
			outfile_WaterBudget_csv << "Parameter_n_KsatPowerDecay," << fixed << input->InputXml["Parameter_n_KsatPowerDecay"] << ",unitless" << endl;
			//Transmissivity (m^2/dt) at catchment saturation
			outfile_WaterBudget_csv << "VadoseZone_Transmissivity_Max," << scientific << input->InputXml["VadoseZone_Transmissivity_Max_m2ph"] << ",m2/h" << endl;
			//Subsurface flow (m/dt) at start of simulation
			outfile_WaterBudget_csv << "Discharge_Subsurface_Initial," << scientific << input->InputXml["Discharge_Subsurface_Initial_mph"] << ",m/h" << endl;
			//Discharge_Subsurface_Max_mph (m/hr) is maximum discharge from catchment pervious area when soil storage full saturated
			outfile_WaterBudget_csv << "Discharge_Subsurface_Max," << scientific << Discharge_Subsurface_Max_mph << ",m/ts" << endl;
			//StorageDeficit_VadoseZone_m (m) is catchment average soil moisture deficit for water storage; multiplied by Ratio_mm_to_m to convert m to mm for output display
			outfile_WaterBudget_csv << "StorageDeficit_VadoseZone_TS_1st," << scientific << StorageDeficit_VadoseZone_m * Ratio_mm_to_m << ",mm" << endl;
			outfile_WaterBudget_csv << endl;

			outfile_WaterBudget_csv.precision(4);

			//If Model equals StatisticalHydro then proceed with TI histogram bins
			//Note Model SpatialTemperatureHydro does not use TI histogram bins
			if (input->SimulationStringParams["Model_Selection"] == "StatisticalHydro" || input->SimulationStringParams["Model_Selection"] == "SpatialBufferwGI") {
				//Loop through all topographic index bins, write out TI values; histogram of TI bins ordered from maximum and wettest to minimum and driest
				outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
				outfile_WaterBudget_csv << "Topographic Index,Histogram Bin ID (#),Value (unitless),Catchment Area (frac),Initial Vadose Zone Storage (mm),Values sorted from Wettest to Driest" << endl;
				outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
				for (int ia = 0; ia < input->Size_TI_Value_Vector; ia++) {
					StorageDeficit_VadoseZone_TI_m[ia] = StorageDeficit_VadoseZone_m + Parameter_m_KsatExpDecay * (TopographicIndex_Avg - input->TI_Value[ia]);
					outfile_WaterBudget_csv << "TopographicIndex," << ia << "," << fixed << input->TI_Value[ia] << "," << scientific << input->TI_Area_frac[ia] << "," << scientific << StorageDeficit_VadoseZone_TI_m[ia] << endl;
				}
				outfile_WaterBudget_csv << endl;
				outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;
			}
			//Groundwater_surficial_frac > 0.05 used to determine message about concern for excessive saturated area, using 0.05 as arbitrary threshold
			//Note: input->RepoVecDict["Groundwater_surficial_TS_last_frac"][0] is BulkArea value, with Folder_Type_ID = 0 for BulkArea
			if (input->RepoVecDict["Groundwater_surficial_TS_last_frac"][0] >= 0.05) {
				outfile_WaterBudget_csv << " Note: The simulation ended with " << setprecision(2) << input->RepoVecDict["Groundwater_surficial_TS_last_frac"][0] * 100 << "% of the catchment with groundwater at the soil surface. This is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values for each Topographic Index (TI) bin contains negative values and may erroneously over-represent saturated areas and its runoff." << endl;
			}
			//StorageDeficit_VadoseZone_TI_m[1] and input->TI_Area_frac[0] used to determine message about concern some saturated area, using 0 to 0.0499 as arbitrary range for accurate simulation
			else if (StorageDeficit_VadoseZone_TI_m[1] < 0 and input->TI_Area_frac[1] > 0) {
				outfile_WaterBudget_csv << " Note: The simulation ended with between 0 and 5% of the catchment with groundwater at the soil surface. This is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values for each Topographic Index (TI) bin contains negative values and may accurately represent saturated areas and its runoff." << endl;
			}
			//StorageDeficit_VadoseZone_TI_m[1] used to determine message about no saturated area, presuming some catchment area should be saturated (e.g., river channel area)
			else if (StorageDeficit_VadoseZone_TI_m[1] >= 0) {
				outfile_WaterBudget_csv << " Note: The simulation ended with 0% of the catchment with groundwater at the soil surface. This is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values for each Topographic Index (TI) bin with any fractional area contain no negative values and may erroneously under-represent saturated areas and its runoff." << endl;
			}
			//Potential that StorageDeficit_VadoseZone_TI_m[1] > 0 and StorageDeficit_VadoseZone_TI_m[0] < 0 to determine message about uncertain but likely small saturated area, presuming some catchment area should be saturated (e.g., river channel area)
			else {
				outfile_WaterBudget_csv << " Note: The simulation may have ended with little to none of the catchment with groundwater at the soil surface. This is tracked by parameter TS_Last_SurficialGroundwater_frac in the Model Variable list above." <<
					" The above list of Local Soil Moisture Deficit Values (SMD) for each Topographic Index (TI) bin with any fractional area contain no negative values and may erroneously under-represent saturated areas and its runoff." << endl;
			}
			//Discharge_Subsurface_Initial_mph or VadoseZone_Transmissivity_Max_m2ph
			outfile_WaterBudget_csv << endl;
			outfile_WaterBudget_csv << " Note: If the you wish to adjust the fraction of catchment with surficial groundwater and saturation excess runoff by adjusting StorageDeficit_VadoseZone_TI_m values, options include:" << endl;
			outfile_WaterBudget_csv << "      1. Adjust VadoseZone_Transmissivity_Max_m2ph; a larger VadoseZone_Transmissivity_Max_m2ph input reduces the number TI bins with negative StorageDeficit_VadoseZone_TI_m values - a smaller value increases the number TI bins with negative StorageDeficit_VadoseZone_TI_m values; VadoseZone_Transmissivity_Max_m2ph values can range order of magnitudes (e.g. 1E-3 to 1E3)." << endl;
			outfile_WaterBudget_csv << "      2. Adjust Discharge_Subsurface_Initial_mph; a smaller Discharge_Subsurface_Initial_mph input reduces the number TI bins with negative StorageDeficit_VadoseZone_TI_m values - a larger value increases the number TI bins with negative StorageDeficit_VadoseZone_TI_m values." << endl;
			outfile_WaterBudget_csv << "      3. Adjust TopographicIndexBins; reducing the number of TI bins between the maximum and average TI values reduces number of TI bins with negative StorageDeficit_VadoseZone_TI_m values - increasing has an opposite effect; the total fraction of area with negative or positive StorageDeficit_VadoseZone_TI_m may remain the same." << endl;
			outfile_WaterBudget_csv << "      4. Adjust DEM resolution; a larger DEM grid size may increase the minimum, average, and maximum TI values, and create a larger and deeper area of surficial groundwater; finer resolution DEM grid sizes could have the opposite effect. To maintain the same area and depth of surficial groundwater when coarsening the DEM it may require decreasing the value of the VadoseZone_Transmissivity_Max_m2ph and Discharge_Subsurface_Initial_mph parameters." << endl;
			outfile_WaterBudget_csv << "      5. Adjust Parameter_m_KsatExpDecay scaling parameter; a smaller Parameter_m_KsatExpDecay scaling parameter input reduces the local SMD value magnitude but not the sign or number of TI bins with negative or positive StorageDeficit_VadoseZone_TI_m values." << endl;
			outfile_WaterBudget_csv << endl;
			outfile_WaterBudget_csv << " --------------------------------------------------------------------------------------------------" << endl;

			outfile_WaterBudget_csv << " References:" << endl;
			outfile_WaterBudget_csv << " Famiglietti J.S.; & Wood E.F. (1994). Multiscale Modeling of Spatially Variable Water and Energy Balance Processes.Water Resources Research. 30(11): 3061 - 3078." << endl;
			outfile_WaterBudget_csv << " Hornberger G.M.; Beven K.J.; Cosby B.J.; & Sappington D.E. (1985). Shenandoah Watershed Study: Calibration of a Topography-Based Variable Contributing Area Hydrological Model to a Small Forested Catchment. Water Resources Research. 21(12): 1841-1850." << endl;
			outfile_WaterBudget_csv << " TMOD9502.FOR TOPMODEL DEMONSTRATION PROGRAM VERSION 95.02. By Keith Beven or Eq given as misc.S_mean[0] = - params.m * log(misc.qs0 / misc.qss) in parameter.init.c https://cran.r-project.org/web/packages/topmodel/index.html" << endl;
			outfile_WaterBudget_csv << " Wang J.; Endreny T.A.; & Hassett J.M. (2006). Power function decay of hydraulic conductivity for a TOPMODEL-based infiltration routine Hydrological Processes. 20(18): 3825-3834. " << endl;
			outfile_WaterBudget_csv << " Wang J.; Endreny T.A.; & Hassett J.M. (2005). Flexible Modeling Package for Topographically Based Watershed Hydrology. Journal of Hydrology. 314(1-4): 78-91. " << endl << endl;
			outfile_WaterBudget_csv << " Note Variable Alternative Names Between HydroPlus and Literature: " << endl;
			outfile_WaterBudget_csv << " Discharge_Subsurface_Max_mph = SZQ | Discharge_Subsurface_Initial_mph = Q0 | VadoseZone_Transmissivity_Max_m2ph = T0 | StorageDeficit_VadoseZone_m = AveSMD | Parameter_m_KsatExpDecay = m | Parameter_n_KsatPowerDecay = n" << endl;
			//outfile_WaterBudget_csv close
			outfile_WaterBudget_csv.close();
		}
	}
}

//WaterBalance_Summary::writeCaliQobs function writes the observed runoff to the output folder
//Note: This does not overwrite the observed runoff provided in the input folder
void WaterBalance_Summary::writeCaliQobs(Inputs *input){

	ofstream outfile_RunoffObserved(input->SimulationStringParams["OutputFolder_Path"] + "Runoff_Observed_for_Calibration.csv");

	//If outfile_RunoffPredicted is not available for opening then Abort
	if (!outfile_RunoffObserved.good()) {
		cout << "Warning: Could not create " << input->SimulationStringParams["OutputFolder_Path"] << "Runoff_Observed_for_Calibration.csv" << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Explanation: The HydroPlus model requires this file when comparing observed and predicted runoff." << endl;
		cout << "Correction: If you do not want this file written, set HydroPlusConfig.xml element Flag_CompareToObservedRunoff = 0." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}

	//Directory_Input_CLArg is the InputDirectory 
	string Directory_Input_CLArg = input->SimulationStringParams["InputDirectory"];

	//calibrate function creates Qi_Simulated_m object, sending pointers to Inputs and Directory, and the StartDate_YYYYMMDD and StopDate_YYYYMMDD to read time series
	//Note: Consider Refactor to replace _qobs with better naming, and explain calibrate is calling which function
	calibrate _qobs(input, Directory_Input_CLArg, input->SimulationNumericalParams["StartDate_YYYYMMDD"], input->SimulationNumericalParams["StopDate_YYYYMMDD"]);

	//Qi_Observed_m is set equal to input->Runoff_Observed_m (m) vector based on observed data
	vector<double>& Qi_Observed_m = input->Runoff_Observed_m;

	//outfile_RunoffObserved header set to YYYYMMDD,HH:MM:SS,Discharge(m)
	outfile_RunoffObserved << "YYYYMMDD,HH:MM:SS,Discharge(m)" << endl;

	//OutputDate_Day_Week_Month initialized as vector
	vector<int>OutputDate_Day_Week_Month;

	string timeStep_Calibration_str = input->SimulationStringParams["TimeStep_CompareToObservedRunoff_str"];

	//If TimeStep_CompareToObservedRunoff_str is Day then call WaterBalance_Summary::getDailyDate function to create OutputDate_Day_Week_Month
	if (timeStep_Calibration_str == "Day" || timeStep_Calibration_str == "day") {
		OutputDate_Day_Week_Month = WaterBalance_Summary::getDailyDate(input->SimulationDate_Output_GD);
	}

	//If TimeStep_CompareToObservedRunoff_str is Day then call WaterBalance_Summary::getWeeklyDate function to create OutputDate_Day_Week_Month
	if (timeStep_Calibration_str == "Week" || timeStep_Calibration_str == "week") {
		OutputDate_Day_Week_Month = WaterBalance_Summary::getWeeklyDate(input->SimulationDate_Output_GD);
	}

	//If TimeStep_CompareToObservedRunoff_str is Day then call WaterBalance_Summary::getMonthlyDate function to create OutputDate_Day_Week_Month
	if (timeStep_Calibration_str == "Month" || timeStep_Calibration_str == "month") {
		OutputDate_Day_Week_Month = WaterBalance_Summary::getMonthlyDate(input->SimulationDate_Output_GD);
	}

	//If SimulationStringParams TimeStep_CompareToObservedRunoff_str is Hour then use SimulationDate_Output_GD and SimulationTime_Output_HMS vectors
	if (timeStep_Calibration_str == "Hour" || timeStep_Calibration_str == "hour") {
		//For Loop through all Qi_Observed_m vector members, which is sized to calibration time step
		for (unsigned int timeStep = 0; timeStep < Qi_Observed_m.size(); timeStep++) {
			outfile_RunoffObserved << input->SimulationDate_Output_GD[timeStep]
				<< "," << input->SimulationTime_Output_HMS[timeStep]
				<< setiosflags(ios::fixed)
				<< setiosflags(ios::showpoint)
				<< setprecision(9)
				<< "," << Qi_Observed_m[timeStep] << endl;
		}
	}
	//Else If SimulationStringParams TimeStep_CompareToObservedRunoff_str is not Hour then use OutputDate_Day_Week_Month vectors
	else {
		//For Loop through all Qi_Observed_m vector members, which is sized to calibration time step
		//20231120 Li debugging use OutputDate_Day_Week_Month.size(). When observed timestep < TimeStep_CompareToObservedRunoff_str, writing incorrect  time series in outputs
		//e.g. When change test case ExponentialDecayTI from hour to day, writing wrong time series in Runoff_Observed_for_Calibration.csv
		for (unsigned int timeStep = 0; timeStep < OutputDate_Day_Week_Month.size(); timeStep++) {
			//outfile_RunoffPredicted is created
			outfile_RunoffObserved << OutputDate_Day_Week_Month[timeStep]
				<< "," << "00:00:00"
				<< setiosflags(ios::fixed)
				<< setiosflags(ios::showpoint)
				<< setprecision(9)
				<< "," << Qi_Observed_m[timeStep] << endl;
		}
	}

	//outfile_RunoffObserved closed
	outfile_RunoffObserved.close();
}

//WaterBalance_Summary::writeCaliQobs function used in calibration to write Runoff_Predicted_for_Calibration.csv
void WaterBalance_Summary::writeCaliTotalQQ(Inputs* input) {
	ofstream outfile_RunoffPredicted(input->SimulationStringParams["OutputFolder_Path"] + "Runoff_Predicted_for_Calibration.csv");

	//If outfile_RunoffPredicted is not available for opening then Abort
	if (!outfile_RunoffPredicted.good()) {
		cout << "Warning: Could not create " << input->SimulationStringParams["OutputFolder_Path"] << "Runoff_Predicted_for_Calibration.csv" << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Explanation: The HydroPlus model requires this file when comparing observed and predicted runoff." << endl;
		cout << "Correction: If you do not want this file written, set HydroPlusConfig.xml element Flag_CompareToObservedRunoff = 0." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}

	//vector<double> Qi_Simulated_m (m) initialized
	vector<double> Qi_Simulated_m;
	//If Flag_CompareToObservedRunoff equals 1, then send routed runoff Qi_Routed_m
	//Note: HydroPlusConfig.xmls element Flag_CompareToObservedRunoff can equal 0, 1 (routed runoff), or 2 (unrouted runoff)
	if (input->SimulationNumericalParams["Flag_CompareToObservedRunoff"] == 1) {
		//Qi_Simulated_m (m) is set equal to input->Qi_routed_m (m) vector to calibrate with routed runoff, generated by diffusion runoff parameters
		Qi_Simulated_m = input->Qi_Routed_m;
	}
	//Else If Flag_CompareToObservedRunoff equals 2, or equals 0, then send unrouted runoff Qi_Simulated_m
	//Note: HydroPlusConfig.xmls element Flag_CompareToObservedRunoff can equal 0, 1 (routed runoff), or 2 (unrouted runoff)
	else if (input->SimulationNumericalParams["Flag_CompareToObservedRunoff"] == 2 || input->SimulationNumericalParams["Flag_CompareToObservedRunoff"] == 0) {
		//Qi_Simulated_m (m) is set equal to input->Qi_m (m) vector to calibrate with unrouted runoff
		Qi_Simulated_m = input->Qi_m;
	}

	//Qi_Simulated_m created from function convertToTimeStep based on HydroPlusConfig.xml element TimeStep_CompareToObservedRunoff_str [Hour, Day, Week, Month
	//Note: Refactor needed to udpate subsequent functions to handle irregular duration of SimulationTimeStep_Duration_sec
	Qi_Simulated_m = convertToTimeStep(Qi_Simulated_m, input->SimulationStringParams["TimeStep_CompareToObservedRunoff_str"]);

	//outfile_RunoffPredicted header set to YYYYMMDD,HH:MM:SS,Discharge(m)
	outfile_RunoffPredicted << "YYYYMMDD,HH:MM:SS,Discharge(m)" << endl;

	//OutputDate_Day_Week_Month initialized as vector
	vector<int>OutputDate_Day_Week_Month;

	string timeStep_Calibration_str = input->SimulationStringParams["TimeStep_CompareToObservedRunoff_str"];

	//If TimeStep_CompareToObservedRunoff_str is Day then call WaterBalance_Summary::getDailyDate function to create OutputDate_Day_Week_Month
	if (timeStep_Calibration_str == "Day" || timeStep_Calibration_str == "day") {
		OutputDate_Day_Week_Month = WaterBalance_Summary::getDailyDate(input->SimulationDate_Output_GD);
	}

	//If TimeStep_CompareToObservedRunoff_str is Day then call WaterBalance_Summary::getWeeklyDate function to create OutputDate_Day_Week_Month
	if (timeStep_Calibration_str == "Week" || timeStep_Calibration_str == "week") {
		OutputDate_Day_Week_Month = WaterBalance_Summary::getWeeklyDate(input->SimulationDate_Output_GD);
	}

	//If TimeStep_CompareToObservedRunoff_str is Day then call WaterBalance_Summary::getMonthlyDate function to create OutputDate_Day_Week_Month
	if (timeStep_Calibration_str == "Month" || timeStep_Calibration_str == "month") {
		OutputDate_Day_Week_Month = WaterBalance_Summary::getMonthlyDate(input->SimulationDate_Output_GD);
	}

	//If SimulationStringParams TimeStep_CompareToObservedRunoff_str is Hour then use SimulationDate_Output_GD and SimulationTime_Output_HMS vectors
	if (timeStep_Calibration_str == "Hour" || timeStep_Calibration_str == "hour") {
		//For Loop through all Qi_Simulated_m vector members, which is sized to calibration time step
		for (unsigned int timeStep = 0; timeStep < Qi_Simulated_m.size(); timeStep++) {
			//outfile_RunoffPredicted is created
			outfile_RunoffPredicted << input->SimulationDate_Output_GD[timeStep]
				<< "," << input->SimulationTime_Output_HMS[timeStep]
				<< setiosflags(ios::fixed)
				<< setiosflags(ios::showpoint)
				<< setprecision(9)
				<< "," << Qi_Simulated_m[timeStep] << endl;
		}
	}
	//Else If SimulationStringParams TimeStep_CompareToObservedRunoff_str is not Hour then use OutputDate_Day_Week_Month vectors
	else {
		//For Loop through all Qi_Simulated_m vector members, which is sized to calibration time step
		for (unsigned int timeStep = 0; timeStep < Qi_Simulated_m.size(); timeStep++) {
			//outfile_RunoffPredicted is created
			outfile_RunoffPredicted << OutputDate_Day_Week_Month[timeStep]
				<< "," << "00:00:00"
				<< setiosflags(ios::fixed)
				<< setiosflags(ios::showpoint)
				<< setprecision(9)
				<< "," << Qi_Simulated_m[timeStep] << endl;
		}
	}
	//outfile_RunoffPredicted closed
	outfile_RunoffPredicted.close();
}

//Add new HydroPlusConfig.xml to allow observed to come as Hour time step when calibration is day or week or month
vector<double>WaterBalance_Summary::convertToTimeStep(const vector<double>& Qi_Simulated_m, const string timeStep_Calibration_str){
	//TimeSeries_Variable_value defined
	vector<double> Variable_TimeStep_original = Qi_Simulated_m;
	vector<double> Variable_TimeStep_resized = Qi_Simulated_m;

	//If timeStep_Calibration_str is Day or day then
	if (timeStep_Calibration_str == "Day" || timeStep_Calibration_str == "day") {
		//Variable_TimeStep_resized defined from function Convert_TimeSeries_Hour_to_Day_vector when sent Variable_TimeStep_original 
		Variable_TimeStep_resized = Convert_TimeSeries_Hour_to_Day_vector(Variable_TimeStep_original);
	}
	//If timeStep_Calibration_str is Week or week then
	if (timeStep_Calibration_str == "Week" || timeStep_Calibration_str == "week") {
		//Variable_TimeStep_resized defined from function Convert_TimeSeries_Hour_to_Week_vector when sent Variable_TimeStep_original 
		Variable_TimeStep_resized = Convert_TimeSeries_Hour_to_Week_vector(Variable_TimeStep_original);
	}
	//If timeStep_Calibration_str is Month or month then
	if (timeStep_Calibration_str == "Month" || timeStep_Calibration_str == "month") {
		//Variable_TimeStep_resized defined from function hourlyToMonth when sent Variable_TimeStep_original 
		Variable_TimeStep_resized = Convert_TimeSeries_Hour_to_Month_vector(Variable_TimeStep_original);
	}

	//Variable_TimeStep_resized returned
	return Variable_TimeStep_resized;
}

//WaterBalance_Summary::Convert_TimeSeries_Hour_to_Day_vector function for Runoff Simulated and Observed, sums not averages for each day
vector<double> WaterBalance_Summary::Convert_TimeSeries_Hour_to_Day_vector(const vector<double>& Variable_TimeSeries_vec) {
	//dailySize defined as Variable_TimeSeries_vec.size() / 24 to convert size from that based on hour to that based on day
	int dailySize = Variable_TimeSeries_vec.size() / 24;
	//Variable_Day_sum_vec vector defined to be length dailySize
	vector<double> Variable_Day_sum_vec(dailySize);
	//For loop with i while less than dailySize
	for (int i = 0; i<dailySize; i++) {
		//Variable_Day_sum_vec defined using Sum_TimeSeries_Hour_to_Day function to sum all hour values into a total for week
		Variable_Day_sum_vec[i] = Sum_TimeSeries_Hour_to_Day(Variable_TimeSeries_vec, i);
	}
	//return Variable_Day_sum_vec
	return Variable_Day_sum_vec;
}

//WaterBalance_Summary::Sum_TimeSeries_Hour_to_Day function sums the hour values into a total for the day 
double WaterBalance_Summary::Sum_TimeSeries_Hour_to_Day(const vector<double>& Variable_TimeSeries_vec, const int i) {
	//Variable_Day_sum initialized to 0
	double Variable_Day_sum = 0.0;
	//For loop with j while less than 24
	for (int j = 0; j<24; j++) {
		//Variable_Day_sum is increased by the input[i * 24 + j]
		Variable_Day_sum = Variable_Day_sum + Variable_TimeSeries_vec[i * 24 + j];
	}
	//Variable_Day_sum is returned
	return Variable_Day_sum;
}

//WaterBalance_Summary::Convert_TimeSeries_Hour_to_Week_vector function for Runoff Simulated and Observed, sums not averages for each week
vector<double> WaterBalance_Summary::Convert_TimeSeries_Hour_to_Week_vector(const vector<double>& Variable_TimeSeries_vec) {
	//weeklySize defined as Variable_TimeSeries_vec.size() / (24 * 7) to convert size from that based on hour to that based on week
	int weeklySize = Variable_TimeSeries_vec.size() / (24 * 7);
	//Variable_Week_sum_vec vector defined to be length weeklySize
	vector<double> Variable_Week_sum_vec(weeklySize);
	//For loop with i while less than weeklySize
	for (int i = 0; i<weeklySize; i++) {
		//Variable_Week_sum_vec defined using Sum_TimeSeries_Hour_to_Week function to sum all hour values into a total for week
		Variable_Week_sum_vec[i] = Sum_TimeSeries_Hour_to_Week(Variable_TimeSeries_vec, i);
	}
	//Variable_Week_sum_vec returned
	return Variable_Week_sum_vec;
}

//WaterBalance_Summary::Sum_TimeSeries_Hour_to_Week function sums the hour values into a total for the week
double WaterBalance_Summary::Sum_TimeSeries_Hour_to_Week(const vector<double>& Variable_TimeSeries_vec, const int i) {
	//Variable_Week_sum initialized to 0
	double Variable_Week_sum = 0.0;
	//For loop with j while less than (24 * 7)
	for (int j = 0; j < (24 * 7); j++) {
		//Variable_Week_sum is increased by the input[i * 24 * 7 + j]
		Variable_Week_sum = Variable_Week_sum + Variable_TimeSeries_vec[i * 24 * 7 + j];
	}
	//Variable_Week_sum returned
	return Variable_Week_sum;
}

//WaterBalance_Summary::Convert_TimeSeries_Hour_to_Month_vector function for Runoff Simulated and Observed, sums not averages for each month
vector<double> WaterBalance_Summary::Convert_TimeSeries_Hour_to_Month_vector(const vector<double>& Variable_TimeSeries_vec) {
	//monthlySize defined as Variable_TimeSeries_vec.size() / (24 * 30) to convert size from that based on hour to that based on month
	int monthlySize = Variable_TimeSeries_vec.size() / (24 * 30);
	//Variable_Month_sum_vec vector defined to be length monthlySize
	vector<double> Variable_Month_sum_vec(monthlySize);
	//For loop with i while less than monthlySize
	for (int i = 0; i<monthlySize; i++) {
		//Variable_Month_sum_vec defined using Sum_TimeSeries_Hour_to_Month function to sum all hour values into a total for month
		Variable_Month_sum_vec[i] = Sum_TimeSeries_Hour_to_Month(Variable_TimeSeries_vec, i);
	}
	//Variable_Month_sum_vec returned
	return Variable_Month_sum_vec;
}

//WaterBalance_Summary::Sum_TimeSeries_Hour_to_Month function sums the hour values into a total for the month
double WaterBalance_Summary::Sum_TimeSeries_Hour_to_Month(const vector<double>& Variable_TimeSeries_vec, const int i) {
	//Variable_Month_sum  initialized to 0
	double Variable_Month_sum = 0.0;
	//Note: Consider refactor to use Inputs::Month_Year_to_Days_in_Month to get actual number of days in month, not assuming 30 for all months
	//For loop with j while less than (24 * 30)
	for (int j = 0; j < (24 * 30); j++) {
		//Variable_Month_sum is increased by the input[i * 24 * 30 + j]
		Variable_Month_sum = Variable_Month_sum + Variable_TimeSeries_vec[i * 24 * 30 + j];
	}
	//Variable_Month_sum returned
	return Variable_Month_sum;
}

//Note: Needs refactor to accomodate use of irregular time step duration with SimulationTimeStep_Duration_sec and SimulationTimePeriod_timeSteps
vector<int> WaterBalance_Summary::getDailyDate(const vector<int> date)
{
	int size = date.size() / (24);
	vector<int> dailyDate(size);

	for (int i = 0; i<size; i++) {
		dailyDate[i] = date[i * 24];
	}
	return dailyDate;
}

//Note: Needs refactor to accomodate use of irregular time step duration with SimulationTimeStep_Duration_sec and SimulationTimePeriod_timeSteps
vector<int> WaterBalance_Summary::getWeeklyDate(const vector<int> date)
{
	int size = date.size() / (24 * 7);
	vector<int> weeklyDate(size);

	for (int i = 0; i<size; i++) {
		weeklyDate[i] = date[i * 24 * 7];
	}
	return weeklyDate;
}

//Note: Needs refactor to accomodate use of irregular time step duration with SimulationTimeStep_Duration_sec and SimulationTimePeriod_timeSteps
//Note: Needs refactor to obtain actual month duration using JulianDay_StartOfMonth_vec in Inputs::Date_YYYYMMDD_to_JulianDay
vector<int> WaterBalance_Summary::getMonthlyDate(const vector<int> date)
{
	int size = date.size() / (24 * 30);
	vector<int> monthlyDate(size);

	for (int i = 0; i<size; i++) {
		monthlyDate[i] = date[i * 24 * 30];
	}
	return monthlyDate;
}

//sumAcrossFolderTypes function will sum variables with indices for each FolderType
double WaterBalance_Summary::sumAcrossFolderTypes(const Inputs* input, const string& varName) {
	double Variable_sum = 0.0;

	//if (input->RepoVecDict.find(varName) == input->RepoVecDict.end(), then variable was not found
	if (input->RepoVecDict.find(varName) == input->RepoVecDict.end()) {
		//Alert user of issue
		cerr << "ERROR: sumAcrossFolderTypes could not find '" << varName << "' in RepoVecDict." << endl;
		cerr << "Aborting: Variable not initialized. Check that '" << varName << "' is defined before this call." << endl;
		abort();
	}
	//vector<double>& Variable_vector = input->RepoVecDict.at(varName)
	//Note: This obtains the variable vector within input->RepoVecDict 
	const vector<double>& Variable_vector = input->RepoVecDict.at(varName);
	//For Variable_value : Variable_vector
	for (double Variable_value : Variable_vector) {
		//Variable_sum = Variable_sum + Variable_value; increment value
		Variable_sum = Variable_sum + Variable_value;
	}
	//return Variable_sum
	return Variable_sum;
}