#include "FluxToImpervious.h"

#include <iostream>


void FluxToImpervious::calculate(Inputs *input, DataFolder *folder, int timeStep)
{
	//Initialize variable
	double Flux_to_ImperviousArea_Rain_SnowMelt_m = 0.0;
	
	//Note: Multiplication of variables by fractional area below is to adjust depths from one analysis area to entire folder area
	//Rationale: 1. variables arrive as depths for an analysis area (e.g., pervious area)
	//Rationale: 2. multiplying these depths by their analysis area gives a volume, which is conserved when transfered between different areas
	//Rationale: 3. multiplying these depths by their fractional area is equivalent to multiplying by their analysis area and dividing by the folder area, getting depth for folder area
	//Note: Multiplication is equivalent to multiplying by ratio of analysis area to folder area to convert from /analysis_area to /folder_area

	//Note: Division of variables by fractional area below is to adjust depths from entire folder area to one analysis area
	//Rationale: 1. variables arrive as depths for the entire folder area
	//Rationale: 2. multiplying these depths by the folder area gives a volume
	//Rationale: 3. dividing these depths by the analysis area is equivalent to multiplying by the folder area and dividing by the analysis area, getting depth for analysis area
	//Note: Division is equivalent to multiplying by ratio of folder area to analysis area to convert from /folder_area to /analysis_area

	//Flux_to_ImperviousArea_Rain_SnowMelt_m (m) is folder area depth and equals Precipitation_RainSnowMelt_underTreeCanopy_m (m) * TreeCanopyCover_overImpervious_frac (fraction)
	//Note: Precipitation_RainSnowMelt_underTreeCanopy_m (m) is cover area depth, multiplying by TreeCanopyCover_overImpervious_frac (fraction) converts to folder area depth
	Flux_to_ImperviousArea_Rain_SnowMelt_m = folder->VarDict["Precipitation_RainSnowMelt_underTreeCanopy_m"] * folder->ParamDict["TreeCanopyCover_overImpervious_frac"];

	//Flux_to_ImperviousArea_Rain_SnowMelt_m (m) is folder area depth and increased by Precipitation_RainSnowMelt_noCanopy_m (m) * ImperviousCover_noTreeCanopy_frac (fraction)
	//Note: Precipitation_RainSnowMelt_noCanopy_m (m) is cover area depth, multiplying by ImperviousCover_noTreeCanopy_frac (fraction) converts to folder area depth
	Flux_to_ImperviousArea_Rain_SnowMelt_m = Flux_to_ImperviousArea_Rain_SnowMelt_m + folder->VarDict["Precipitation_RainSnowMelt_noCanopy_m"] * folder->ParamDict["ImperviousCover_noTreeCanopy_frac"];
		
	//If ImperviousCover_frac 0, gather water from various catchment features
	if (folder->ParamDict["ImperviousCover_frac"] > 0) {
		//Flux_to_ImperviousArea_Rain_SnowMelt_m (m) becomes impervious area depth (m) when divided by ImperviousCover_frac (fraction) to convert from folder depth to cover type depth 
		//Note: For BulkArea with no GI, Flux_to_ImperviousArea_Rain_SnowMelt_m (m) will have the fraction DCIA sent to outlet 
		//Note: For BulkArea with GI, Flux_to_ImperviousArea_Rain_SnowMelt_m (m) may have a fraction of DCIA sent to GI
		//Note: For GI, Flux_to_ImperviousArea_Rain_SnowMelt_m (m) may have DCIA send flow to outlet, or retain flux for GI processing
		folder->VarDict["Flux_to_ImperviousArea_Rain_SnowMelt_m"] = Flux_to_ImperviousArea_Rain_SnowMelt_m / folder->ParamDict["ImperviousCover_frac"];
	}
	//No impervious area, so Flux_to_ImperviousArea_Rain_SnowMelt_m = 0
	else
	{
		folder->VarDict["Flux_to_ImperviousArea_Rain_SnowMelt_m"] = 0.0;
	}
}
