#include "FluxToPervious.h"
#include <iostream>


void FluxToPervious::calculate(Inputs *input, DataFolder *folder, int timeStep, std::vector<DataFolder*> &stafolders)
{
	//Note: Multiplication of variables by fractional area below is to adjust depths from one analysis area to entire folder area
	//Rationale: 1. variables arrive as depths for an analysis area (e.g., pervious area)
	//Rationale: 2. multiplying these depths by their analysis area gives a volume, which is conserved when transfered between different areas
	//Rationale: 3. multiplying these depths by their fractional area is equivalent to multiplying by their analysis area and dividing by the folder area, getting depth for folder area
	//Note: Multiplication is equivalent to multiplying by ratio of analysis area to folder area to convert from /analysis_area to /folder_area

	//Note: Division of variables by fractional area below is to adjust depths from entire folder area to one analysis area
	//Rationale: 1. variables arrive as depths for the entire folder area
	//Rationale: 2. multiplying these depths by the folder area gives a volume
	//Rationale: 3. dividing these depths by the analysis area is equivalent to multiplying by the folder area and dividing by the analysis area, getting depth for analysis area
	//Note: Division is equivalent to multiplying by ratio of folder area to analysis area to convert from /folder_area to /analysis_area

	// 1st: Summing 3 fractional area contributions, each adjusted to depth over folder area (e.g., bulk area) by multiplying by folder area fraction in analysis area
	//Flux_to_PerviousArea_Rain_SnowMelt_m (m) is folder area depth and equals Precipitation_RainSnowMelt_underTreeCanopy_m (m) * TreeCanopyCover_overPervious_frac (fraction)
	//Note: Precipitation_RainSnowMelt_underTreeCanopy_m (m) is cover area depth, multiplying by TreeCanopyCover_overPervious_frac (fraction) converts to folder area depth
	folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] = folder->VarDict["Precipitation_RainSnowMelt_underTreeCanopy_m"] * folder->ParamDict["TreeCanopyCover_overPervious_frac"];
	//Flux_to_PerviousArea_Rain_SnowMelt_m (m) is folder area depth and increased by Precipitation_RainSnowMelt_underSVegCanopy_m (m) * ShortVegCover_noTreeCanopy_frac
	//Note: Precipitation_RainSnowMelt_underSVegCanopy_m (m) is cover area depth, multiplying by ShortVegCover_noTreeCanopy_frac (fraction) equals folder area depth
	folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] = folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] + folder->VarDict["Precipitation_RainSnowMelt_underSVegCanopy_m"] * folder->ParamDict["ShortVegCover_noTreeCanopy_frac"];
	//Flux_to_PerviousArea_Rain_SnowMelt_m (m) is folder area depth and increased by Precipitation_RainSnowMelt_noCanopy_m (m) * SoilCover_noTreeCanopy_frac
	//Note: Precipitation_RainSnowMelt_noCanopy_m (m) is cover area depth, multiplying by (SoilCover_noTreeCanopy_frac + PermeablePavementCover_noTreeCanopy_frac) (fraction) equals folder area depth
	folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] = folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] + folder->VarDict["Precipitation_RainSnowMelt_noCanopy_m"] *	(folder->ParamDict["SoilCover_noTreeCanopy_frac"] + folder->ParamDict["PermeablePavementCover_noTreeCanopy_frac"]);

	//If PerviousCover_frac > 0 then divide by PerviousCover_frac
	//Mote: PerviousCover_frac = fraction of watershed with tree cover on pervious area + fraction of watershed with ShortVeg cover + bare soil area
	if (folder->ParamDict["PerviousCover_frac"] > 0) {
		// 2nd: Adjusting Flux_to_PerviousArea_Rain_SnowMelt_m becomes pervious area depth (m) with division by PerviousCover_frac to convert from folder depth to cover type depth 
		//Note: Division is equivalent to multiplying by ratio of folder area to analysis area to convert from /folder_area to /analysis_area
		//Flux_to_PerviousArea_Rain_SnowMelt_m (m) arrives as folder area depth by above multiplication w/ SoilCover_noTreeCanopy_frac, ShortVegCover_noTreeCanopy_frac, PermeablePavementCover_noTreeCanopy_frac, etc.
		//Flux_to_PerviousArea_Rain_SnowMelt_m (m) adjusted to pervious area depth by division with PerviousCover_frac
		folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] = folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] / folder->ParamDict["PerviousCover_frac"];

		// 3rd: Adjusting Flux_Impervious_to_Pervious_m to pervious area depth (m) with division by PerviousCover_frac
		//Note: Division is equivalent to multiplying by ratio of folder area to analysis area to convert from /folder_area to /analysis_area
		//Flux_Impervious_to_Pervious_m (m) arrives as folder area depth by earlier multiplication w/ ImperviousCover_DrainsToPervious_frac in ImperviousDepressionStorageCalc.cpp
		//Flux_Impervious_to_Pervious_m (m) remains folder depth, but added on RHS as pervious area depth by division with PerviousCover_frac
		//Flux_to_PerviousArea_Rain_SnowMelt_ImpRunon_m (m) represents pervious area depth in functors for infiltration
		folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_ImpRunon_m"] = folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_m"] +
			(folder->VarDict["Flux_Impervious_to_Pervious_m"] / folder->ParamDict["PerviousCover_frac"]);
	}
	//Else avoid division by zero
	else {
		folder->VarDict["Flux_to_PerviousArea_Rain_SnowMelt_ImpRunon_m"] = 0;
	}
	// Storing Flux_to_PerviousArea_Rain_SnowMelt_ImpRunon_m as VarDict["waterToPerArea"], with impervious area runon included
	if (folder->ParamStringDict["Type"] == "BulkArea") {

		//Flux_to_GI_PerviousArea_BARunon_m (m) defined as zero for BA
		folder->VarDict["Flux_to_GI_PerviousArea_BARunon_m"] = 0.0;
	}
	//If Type not BulkArea and PerviousCover_frac > 0 then enter and divide by PerviousCover_frac for transfer of water from BulkArea to GI infiltration
	//Note: Consider refactor to pass this water to GI infiltration from Inflow_StormwaterDevice.cpp
	if (folder->ParamStringDict["Type"] != "BulkArea" && folder->ParamDict["PerviousCover_frac"] > 0) {
		//PerviousCover_m2 (m2) is product of GI Area_m2 and PerviousCover_frac; Area_m2 is area of all GI units in DataFolder
		double PerviousCover_m2 = folder->ParamDict["Area_m2"] * folder->ParamDict["PerviousCover_frac"];

		//Flux_to_GI_PerviousArea_BARunon_m (m) goes to functors["calculateInfiltration"] 
		//Note: Flux_to_GI_PerviousArea_BARunon_m is sum of BulkArea transfers to GI, scaled from folder volume to pervious area depth
		//Note: All BulkArea transfers to GI presumed to go to GI pervious area, regardless of GI ImperviousCover_frac
		folder->VarDict["Flux_to_GI_PerviousArea_BARunon_m"] = (input->RepoDict["Runoff_Impervious_BAtoGI_m3"] +
			input->RepoDict["Runoff_SatExcess_BAtoGI_m3"] + input->RepoDict["Runoff_InfilExcess_BAtoGI_m3"]) /
			PerviousCover_m2;
	}
}
