#z_c_RescaleTI_Statistical_to_Spatial.py created by T. Endreny w ChatGPT to transition from HydroPlus StatisticalHydro to SpatialTemperatureHydro

import argparse
import os
import shutil
import xml.etree.ElementTree as ET
import pandas as pd
import numpy as np
from pathlib import Path
from decimal import Decimal, ROUND_HALF_UP

#Program expects 2 command line arguments:
# 1. old directory path of StatisticalHydro inputs with z_TI_ExponentialDecay.csv or z_TI_PowerDecay.csv, dem.asc, and HydroPlusConfig.xml
# 2. new directory for SpatialTemperatureHydro simulation with z_TIorganizer.asc and other inputs; program will create Inputs and Outputs sub-directories 

#Optional changes to output map values:
dem_m = 37.5 #100 #bol_it=37.5
z_FDorganizer_deg = 180
z_AspectGround_N_0_rad = 0
z_SlopeGround_rad = 0.001
AH_flux_Qtot_avg_Wpm2 = 0
AH_flux_Qcr_avg_Wpm2 = 0
AH_flux_Qncr_avg_Wpm2 = 0
#Note: blockgroup_flag 1=unique values for each pixel starting at blockgroup_starting_value; blockgroup_flag 0=contant values of blockgroup_starting_value
blockgroup_flag = 1
blockgroup_starting_value = 1

#NLCD_Class_list = [11,12,21,22,23,24,31,41,42,43,51,52,71,72,73,74,81,82,90,95]
#Note: 11=Open_Water; 12=Perennial_Ice_Snow; 
#Note: 21=Developed_Open; 22=Developed_LowIntensity; 23=Developed_MediumIntensity; 24=Developed_HighIntensity
#Note: 31=Barren_Land; 41=Forest_Deciduous; 42=Forest_Evergreen; 43=Forest_Mixed
#Note: 51=Dwarf_Scrub 52=Shrub_Scrub; 71=Grassland_Herbaceous; 72=Sedge_Herbaceous; 73=Lichens; 74=Moss
#Note: 81=Pasture_Hay; 82=Cultivated_Crops; 90=Wetlands_Woody; 95=Wetlands_Emergent_Herbaceous
NLCD_Class_list = [11,12,21,22,23,24,31,41,42,43,51,52,71,72,73,74,81,82,90,95]
#TreeCover_ImperviousCover_Increment = [100,100,50,50,50,50,100,100,100,100,100,100,100,100,100,100,100,100,100,100]
#Note: Used to determine increment between 0 and 100 percent when scaling tree cover and impervious cover
TreeCover_ImperviousCover_Increment = [100,100,50,50,50,50,100,100,100,100,100,100,100,100,100,100,100,100,100,100]

#ASCII_HEADER_KEYS = ["ncols", "nrows", "xllcorner", "yllcorner", "cellsize", "NODATA_value"]; used in header
ASCII_HEADER_KEYS = ["ncols", "nrows", "xllcorner", "yllcorner", "cellsize", "NODATA_value"]

#def read_ascii_header takes (filepath) and reads file header
def read_ascii_header(filepath):
    header = {}
    with open(filepath, 'r') as file:
        for _ in range(6):
            key, value = file.readline().strip().split()
            header[key] = float(value) if '.' in value else int(value)
    return header

#def write_ascii takes (filename, header, array, fmt='%.5f') and writes map, as float by default format
def write_ascii(filename, header, map_array, fmt='%.5f'):
    with open(filename, 'w') as f:
        for key in ASCII_HEADER_KEYS:
            f.write(f"{key} {int(header[key]) if isinstance(header[key], int) else header[key]}\n")
        np.savetxt(f, map_array, fmt=fmt)

#def create_diverse_map_with_nodata takes (data_1d, rows, cols, nodata_value=-9999); it reshapes the vector values into the map
def create_diverse_map_with_nodata(data_1d, TreeCover_ImperviousCover_Value_Count, rows, cols, nodata_value=-9999):
    print(f"rows={rows}, cols={cols}")
    map_array = np.full((rows, cols), nodata_value, dtype=float)
    # Index to track position in data_1d
    idx = 0  
    for row in range(rows):
        #tc_ic_combinations = TreeCover_ImperviousCover_Value_Count[row] ** 2; tc and ic combinations are TreeCover_ImperviousCover_Value_Count^2
        tc_ic_combinations = TreeCover_ImperviousCover_Value_Count[row] ** 2
        
        #error handler if idx + tc_ic_combinations > len(data_1d):
        if idx + tc_ic_combinations > len(data_1d):
            raise ValueError("Not enough values in data_1d to fill the requested pixels")
            
        #map_array[row, :tc_ic_combinations] = data_1d[idx:idx+tc_ic_combinations]
        map_array[row, :tc_ic_combinations] = data_1d[idx:idx+tc_ic_combinations]
        #idx += tc_ic_combinations
        idx += tc_ic_combinations   
    return map_array

#def create_flat_map_with_nodata takes (map_value, rows, cols, valid_pixel_count, nodata_value=-9999); it reshapes the constant value into the map
def create_flat_map_with_nodata(map_value, TreeCover_ImperviousCover_Value_Count,
                                rows, cols, valid_pixel_count, nodata_value=-9999):
    # force float nodata & float array so fractional values are preserved
    nodata = float(nodata_value)
    map_array = np.full((rows, cols), nodata, dtype=np.float64)  # CHANGED

    # build per-row counts; keep your **2 logic, but handle scalar vs array
    tcic = TreeCover_ImperviousCover_Value_Count
    if np.isscalar(tcic):
        counts = np.full(rows, int(tcic) ** 2, dtype=int)
    else:
        counts = np.asarray(tcic, dtype=int)
        if counts.size != rows:
            raise ValueError("TreeCover_ImperviousCover_Value_Count must have length == rows")
        counts = counts ** 2

    v = float(map_value)
    for row in range(rows):
        c = int(counts[row])
        if c <= 0:
            continue
        c = min(c, cols)               # guard against overflow
        map_array[row, :c] = v         # stays float

    return map_array


#def get_ti_bins takes (csv_path, Pixel_Count_w_data) to determine TI values in z_TIorganizer.asc file
#Note: Function converts TI values from z_TI_ExponentialDecay.csv or z_TI_PowerDecay.csv histogram into values within z_TIorganizer.asc map
#Note: Histogram files have columns of Area_in_TI_Value_Bin_frac and TI_Value_Histogram_Bin to guide creation of z_TIorganizer.asc map
#Step 1a: TI_area_per_bin_fraction takes Area_in_TI_Value_Bin_frac from histogram file
#Step 1b: TI_value_per_bin takes TI_Value_Histogram_Bin from histogram file
#Step 2. Initially set Scale_per_TI_bin is Pixel_Count_w_data / sum(TI_area_per_bin_fraction), then adjust as explained below
#Step 3. Compute Pixel_Count_per_TI_bin as product of TI_area_per_bin_fraction by Scale_per_TI_bin, rounded to integer
#Step 4. If sum of Pixel_Count_per_TI_bin does not equal Pixel_Count_w_data, then adjust Scale_per_TI_bin and return to step 3
#Step 5. Else write map of TI_organizer_values to z_TIorganizer.asc by assigning TI_value_per_bin to each Pixel_Count_per_TI_bin
#Step 6. Randomly distribute TI_organizer_values across 1D vector, and convert 1D vector to 2D map
def get_ti_bins(csv_path, Pixel_Count_w_data):
    
    df = pd.read_csv(csv_path, skiprows=2, header=None)
    TI_area_per_bin_fraction = df[0].values
    TI_value_per_bin = df[1].values
    Scale_per_TI_bin = Pixel_Count_w_data / sum(TI_area_per_bin_fraction)
    Pixel_Count_per_TI_bin = np.round(TI_area_per_bin_fraction * Scale_per_TI_bin).astype(int)

    # Adjust pixel count to total
    while Pixel_Count_per_TI_bin.sum() < Pixel_Count_w_data:
        Pixel_Count_per_TI_bin[np.argmax(TI_area_per_bin_fraction)] += 1
    while Pixel_Count_per_TI_bin.sum() > Pixel_Count_w_data:
        Pixel_Count_per_TI_bin[np.argmax(Pixel_Count_per_TI_bin)] -= 1

    #TI_organizer_values initialized
    TI_organizer_values = []
    #for pixel_count, TI_organizer_value in zip(Pixel_Count_per_TI_bin, TI_value_per_bin) write z_TIorganizer.asc map
    for pixel_count, TI_organizer_value in zip(Pixel_Count_per_TI_bin, TI_value_per_bin):
        #TI_organizer_values.extend([TI_organizer_value] * pixel_count) takes TI_organizer_value for pixel_count times
        TI_organizer_values.extend([TI_organizer_value] * pixel_count)
        
    #np.random.shuffle(TI_organizer_values); shuffle values randomly so they are not concentrated
    np.random.shuffle(TI_organizer_values)
    
    return np.array(TI_organizer_values)

#def adjust_config_xml takes (config_path, new_output_dir, model_selection) to udpate HydroPlusConfig.xml
def adjust_config_xml(config_path, new_output_dir, model_selection):
    
    #tree = ET.parse(config_path); tree defined using ET .parse function
    tree = ET.parse(config_path)
    root = tree.getroot()
    
    #for elem in root.iter(), find and update some HydroPlusConfig elements
    for elem in root.iter():
        if elem.tag == "CatchmentArea_m2":
            catchment_area = float(elem.text)
        elif elem.tag == "OutputFolder_Path":           
            elem.text = str(new_output_dir) + os.sep
        elif elem.tag == "Model_Selection":
            elem.text = model_selection
        elif elem.tag == "flag_Recompute_TopographicIndex":
            elem.text = "0"
        elif elem.tag == "Flag_DEM_RemovePitsFlats":
            elem.text = "0"
            
    #return tree, catchment_area; catchment_area used with rows and cols to size pixel area
    return tree, catchment_area

def fmt_from_dem_decimal(dem_m):
    d = Decimal(str(dem_m)).quantize(Decimal('0.01'), rounding=ROUND_HALF_UP)  # cap at 2 dp
    if d == d.quantize(Decimal('1')):
        return '%d', 0, int(d)
    elif d == d.quantize(Decimal('0.1')):
        return '%.1f', 1, float(d)
    else:
        return '%.2f', 2, float(d)

#def main will expect 2 command line arguments
def main():
    parser = argparse.ArgumentParser(description="Generate synthetic DEM input for HydroPlus")
    parser.add_argument("original_input_dir", help="Path to original input directory")
    parser.add_argument("new_project_dir", help="Path to new synthetic input directory")
    args = parser.parse_args()

    directory_old_input = args.original_input_dir
    # Base directory (e.g., C:\iTree\projects\CoolAir\bal_md\10_m\model_01)
    directory_new_project = Path(args.new_project_dir).resolve()

    # Create subdirectories
    directory_new_input = directory_new_project / "Inputs"
    directory_new_output = directory_new_project / "Outputs"

    # Ensure directories exist
    os.makedirs(directory_new_input, exist_ok=True)
    os.makedirs(directory_new_output, exist_ok=True)
   
    # Step 1: Compute how many unique values for each tc_ic_increment
    #TreeCover_ImperviousCover_Value_Count = 100 / TreeCover_ImperviousCover_Increment[] + 1; e.g., TreeCover_ImperviousCover_Combinations = 100 / 50 + 1 = 3 
    TreeCover_ImperviousCover_Value_Count = (100 // np.array(TreeCover_ImperviousCover_Increment)) + 1

    # Step 2: Compute combinations per row (tc_ic_number_of_unique_values ^ 2)
    #TreeCover_ImperviousCover_Combinations = TreeCover_ImperviousCover_Value_Count ^ 2; e.g., TreeCover_ImperviousCover_Combinations = 3^2 = 9
    TreeCover_ImperviousCover_Combinations = TreeCover_ImperviousCover_Value_Count ** 2

    # Step 3: Total data pixels
    #Pixel_Count_w_data = TreeCover_ImperviousCover_Combinations.sum(); these pixels will have data values
    Pixel_Count_w_data = TreeCover_ImperviousCover_Combinations.sum()

    # Step 4: Full grid size (rows × max number of combinations)
    #Pixel_Count_array = TreeCover_ImperviousCover_Combinations.max() * TreeCover_ImperviousCover_Combinations.size(); these pixels create uniform array
    Pixel_Count_array = TreeCover_ImperviousCover_Combinations.max() * TreeCover_ImperviousCover_Combinations.size

    # Step 5: NoData pixels
    Pixel_Count_no_data = Pixel_Count_array - Pixel_Count_w_data

    #rows = input("Please enter number of rows for new maps or accept default of 11: ") or 11
    #cols = input("Please enter number of cols for new maps or accept default of 11: ") or 11
    #Refactor: to define rows and cols with these new terms
    #rows = TreeCover_ImperviousCover_Combinations.size or len(NLCD_Class_list); based on length of vector of NLCD_Classes
    rows = TreeCover_ImperviousCover_Combinations.size
    #cols = TreeCover_ImperviousCover_Combinations.max(); based on maximum number of combinations, typically with developed NLCD Classes
    cols = TreeCover_ImperviousCover_Combinations.max()
    
    rows, cols = int(rows), int(cols)
    Pixel_Count_new_map = rows * cols
    No_Data_value = -9999

    print("\nAnalyzing original files, e.g., z_TI_ExponentialDecay.csv, file to create new inputs ...")
    # Read and adjust config
    config_path = os.path.join(directory_old_input, "HydroPlusConfig.xml")
    tree, catchment_area = adjust_config_xml(config_path, directory_new_output, "SpatialTemperatureHydro")
    cell_area = catchment_area / Pixel_Count_w_data
    cellsize = int(cell_area ** 0.5)

    # Read original header from dem.asc
    #header_path = os.path.join(directory_old_input, "dem.asc"); dem.asc is only required file with header
    header_path = os.path.join(directory_old_input, "dem.asc")
    header = read_ascii_header(header_path)
    #header.update({"ncols": cols, "nrows": rows, "cellsize": cellsize}); ncols, nrows, cellsize likely change for new maps
    header.update({"ncols": cols, "nrows": rows, "cellsize": cellsize})

    # Generate TI 
    #ti_csv_exp = os.path.join(directory_old_input, "z_TI_ExponentialDecay.csv") as possible file
    ti_csv_exp = os.path.join(directory_old_input, "z_TI_ExponentialDecay.csv")
    #ti_csv_power = os.path.join(directory_old_input, "z_TI_PowerDecay.csv") as possible file
    ti_csv_power = os.path.join(directory_old_input, "z_TI_PowerDecay.csv")

    #if os.path.isfile(ti_csv_exp), then use ti_csv_exp
    if os.path.isfile(ti_csv_exp):
        ti_csv = ti_csv_exp
    #elif os.path.isfile(ti_csv_power), then use ti_csv_power
    elif os.path.isfile(ti_csv_power):
        ti_csv = ti_csv_power
    
    #ti_grid is array obtained from get_ti_bins function, that determines the topographic index values for Pixel_Count_w_data
    ti_grid = get_ti_bins(ti_csv, Pixel_Count_w_data)

    #cte 2025
    flag_debug = True   
    if flag_debug: 
        # 0) all the same?
        print("all same:", bool((ti_grid == ti_grid[0]).all()))

        # 1) basic spread
        print("min/mean/max/std:", ti_grid.min(), ti_grid.mean(), ti_grid.max(), ti_grid.std())

        # 2) unique values + counts
        u, c = np.unique(ti_grid, return_counts=True)
        print("unique count:", len(u))
        print("top 10 bins by count:", list(zip(u, c))[:10])  # or print all

        # 3) quick histogram of bins (value -> count)
        counts = dict(zip(u, c))
        for v in sorted(counts):
            print(f"TI={v}: {counts[v]}")
    
    #ti_grid is reshaped from create_diverse_map_with_nodata(ti_grid, rows, cols, No_Data_value), that assigns 
    ti_grid = create_diverse_map_with_nodata(ti_grid, TreeCover_ImperviousCover_Value_Count, rows, cols, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "z_TIorganizer.asc"), header, ti_grid)

    #fmt and ndp defined by fmt_from_dem_decimal(dem_m)
    fmt, ndp, dem_m_rounded = fmt_from_dem_decimal(dem_m)
    #dem_array created with create_flat_map_with_nodata to use constant value in data pixels, rest is NoData
    dem_array = create_flat_map_with_nodata(dem_m_rounded, TreeCover_ImperviousCover_Value_Count,rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "dem.asc"), header, dem_array, fmt=fmt)
    
    #fd_array created with create_flat_map_with_nodata to use constant value in data pixels, rest is NoData
    fd_array = create_flat_map_with_nodata(z_FDorganizer_deg, TreeCover_ImperviousCover_Value_Count, rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "z_FDorganizer.asc"), header, fd_array, fmt='%d')

    # Create Aspect map (z_AspectGround_N_0_rad radians)
    aspect_array = create_flat_map_with_nodata(z_AspectGround_N_0_rad, TreeCover_ImperviousCover_Value_Count, rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "z_AspectGround_N_0_rad.asc"), header, aspect_array, fmt='%d')

    # Create Slope map (small slope like 0.001 rad)
    slope_array = create_flat_map_with_nodata(z_SlopeGround_rad, TreeCover_ImperviousCover_Value_Count, rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "z_SlopeGround_rad.asc"), header, slope_array)

    # Anthropogenic Heat (AH) flux maps (default = 0)
    ah_qtot_array = create_flat_map_with_nodata(AH_flux_Qtot_avg_Wpm2, TreeCover_ImperviousCover_Value_Count, rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "AH_flux_Qtot_avg_Wpm2.asc"), header, ah_qtot_array, fmt='%d')

    ah_qcr_array = create_flat_map_with_nodata(AH_flux_Qcr_avg_Wpm2, TreeCover_ImperviousCover_Value_Count, rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "AH_flux_Qcr_avg_Wpm2.asc"), header, ah_qcr_array, fmt='%d')

    ah_qncr_array = create_flat_map_with_nodata(AH_flux_Qncr_avg_Wpm2, TreeCover_ImperviousCover_Value_Count, rows, cols, Pixel_Count_w_data, No_Data_value)
    write_ascii(os.path.join(directory_new_input, "AH_flux_Qncr_avg_Wpm2.asc"), header, ah_qncr_array, fmt='%d')

    # Initialize empty arrays with NoData
    landcover_array = np.full((rows, cols), No_Data_value, dtype=int)
    impervious_array = np.full((rows, cols), No_Data_value, dtype=int)
    treecover_array = np.full((rows, cols), No_Data_value, dtype=int)
    blockgroup_array = np.full((rows, cols), No_Data_value, dtype=int)
    #global blockgroup_starting_value; allows variable to be modified
    global blockgroup_starting_value
    
    #for row in range(rows); loop through rows to get nlcd_value
    for row in range(rows):
        nlcd_value = NLCD_Class_list[row]
        #tc_ic_increment = TreeCover_ImperviousCover_Increment[row]; the rate tc or ic increases from 0 to 100, e.g, by 10, 25, or 100
        tc_ic_increment = TreeCover_ImperviousCover_Increment[row]
        #tc_ic_number_of_unique_values = 100 // tc_ic_increment + 1; how many tc or ic values needed to get from 0 to 100; add 1 account for 0
        tc_ic_number_of_unique_values = 100 // tc_ic_increment + 1

        #for col in range(tc_ic_number_of_unique_values ** 2); tc_ic_number_of_unique_values^2 is the number of tc and ic combinations
        # Create landcover.asc, treecover.asc and imperviouscover.asc with potentially varying patterns
        #... landcover.asc AS an example for NLCD Class 11; 11 11 11 11 -9999 -9999 -9999 -9999 -9999
        #... landcover.asc AS an example for NLCD Class 21; 21 21 21 21 21 21 21 21 21
        #... imperviouscover.asc AS an example for NLCD Class 11; 0 0 100 100 -9999 -9999 -9999 -9999 -9999
        #... imperviouscover.asc AS an example for NLCD Class 21; 0 0 0 50 50 50 100 100 100 
        #... treecover.asc AS an example for NLCD Class 11; 0 100 0 100 -9999 -9999 -9999 -9999 -9999
        #... treecover.asc AS an example for NLCD Class 21; 0 50 100 0 50 100 0 50 100
        for col in range(tc_ic_number_of_unique_values ** 2):
            landcover_array[row, col] = nlcd_value
            #impervious_array[row, col] = (col // tc_ic_number_of_unique_values) * tc_ic_increment  
            #Note: impervious_array repeats every tc_ic_number_of_unique_values; e.g, 0 0 0 50 50 50 100 100 100 
            impervious_array[row, col] = (col // tc_ic_number_of_unique_values) * tc_ic_increment  
            #treecover_array[row, col] = (col % tc_ic_number_of_unique_values) * tc_ic_increment
            #Note: treecover_array cycles every tc_ic_number_of_unique_values; e.g. 0 50 100 0 50 100  0 50 100
            treecover_array[row, col] = (col % tc_ic_number_of_unique_values) * tc_ic_increment
            if blockgroup_flag == 1:
                #blockgroup_array[row, col] = blockgroup_starting_value
                blockgroup_array[row, col] = blockgroup_starting_value
                blockgroup_starting_value += 1
            else: 
                blockgroup_array[row, col] = blockgroup_starting_value

    # Write to ASCII files
    write_ascii(os.path.join(directory_new_input, "landcover.asc"), header, landcover_array, fmt='%d')
    write_ascii(os.path.join(directory_new_input, "imperviouscover.asc"), header, impervious_array, fmt='%d')
    write_ascii(os.path.join(directory_new_input, "treecover.asc"), header, treecover_array, fmt='%d')
    write_ascii(os.path.join(directory_new_input, "blockgroup.asc"), header, blockgroup_array, fmt='%d')

    # Save modified config
    tree.write(os.path.join(directory_new_input, "HydroPlusConfig.xml"))

    # Copy Weather and Radiation files
    for fname in ["Weather.csv", "Radiation.csv"]:
        shutil.copy(os.path.join(directory_old_input, fname), os.path.join(directory_new_input, fname))

    print("... Process complete.")
    print("\nSynthetic z_TIorganizer.asc map created to approximate TI histogram of original dem.asc.")
    print("Landcover.asc held constant at 21, treecover.asc and imperviouscover.asc Scale_per_TI_bin from 0 to 100")
    print("HydroPlusConfig.xml and all maps can be adjusted as needed to achieve simulation goals.")
    print("\nThank you for using i-Tree tools to improve the world!")

if __name__ == "__main__":
    main()
