﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using NHibernate.Cfg;
using NHibernate.Dialect;
using NHibernate.Driver;
using System.Reflection;
using NHibernate;
//Located at: https://data.itreetools.org/sql
using LocationSpecies.Domain;

namespace WeatherPrep
{
    class CreateOutput
    {
        //These fields are never used
        //string locDB;
        //string nationID;
        //string priPartID;
        //string secPartID;
        //string terPartID;

        double LeafAreaIndexMax;
        double Evergreen_pct;
        VEGTYPE VegetationType;
        int startYr;
        int endYr;
        double Height_WeatherStationWindSensor_m;
        double Height_TreeCanopyTop_m;

        string MeteorologicalFile_string;
        string PrecipitationFile_string;
        string LeafAreaIndex_dataBase;
        string WeatherVariables_dBase;
        string model_type;

        // Variables sent by SimulationManager.cs: 
        //MeteorologicalDataInputFile_string, PrecipitationDataInputFile_string, LocationData_Object, LeafAreaIndex_dataBase, VegetationType_class, startYear, endYear, WeatherVariables_dataBase

        public CreateOutput(double LeafAreaIndex_max, double Evergreen_percent, string LeafAreaIndex_dataBase_string, int startYear, int endYear,
            VEGTYPE VegetationType_class, string MeteorologicalDataInputFile_string, string PrecipitationDataInputFile_string, string WeatherVariables_dataBase, string ModelType_string,
            double Height_WeatherStationWindSensor_t_m, double Height_TreeCanopyTop_t_m)
        {
            LeafAreaIndexMax = LeafAreaIndex_max;
            Evergreen_pct = Evergreen_percent;
            VegetationType = VegetationType_class;
            Height_WeatherStationWindSensor_m = Height_WeatherStationWindSensor_t_m;
            Height_TreeCanopyTop_m = Height_TreeCanopyTop_t_m;
            startYr = startYear;
            endYr = endYear;
            MeteorologicalFile_string = MeteorologicalDataInputFile_string;
            PrecipitationFile_string = PrecipitationDataInputFile_string;
            LeafAreaIndex_dataBase = LeafAreaIndex_dataBase_string;
            WeatherVariables_dBase = WeatherVariables_dataBase;
            model_type = ModelType_string;
        }
        //Helper determines if extra data are available to add for January 1 or December 31 given time zone clipped data 
        private static (bool jan1Missing, bool dec31Missing) ComputeEdgePaddingFlags(IReadOnlyList<DateTime> dtList)
        {
            bool Flag_EnoughDataAvailable = dtList != null && dtList.Count >= 48;
            bool Flag_PadDataForJan1 = Flag_EnoughDataAvailable && dtList[0].Month == 1 && dtList[0].Day == 2;
            bool Flag_PadDataForDec31 = Flag_EnoughDataAvailable && dtList[dtList.Count - 1].Month == 12 && dtList[dtList.Count - 1].Day == 30;
            return (Flag_PadDataForJan1, Flag_PadDataForDec31);
        }

        // Function now works with CSV files
        public void CreateOutputFiles(ISession NHibernate_Session, Location location)
        {
            try
            {
                LocationData LocationData_Object = new LocationData(NHibernate_Session);
                ForestData forData;
                LocationData_Object.ReadLocationData(location);
                forData = new ForestData(0, LeafAreaIndexMax, Evergreen_pct, 0);

                Console.WriteLine("\nQuerying server to determine leaf on and off dates and compute meteorological outputs.");

                //call LeafAreaIndex.ProcessLAI to compute terms for LAI database, where WeatherPrepConfig.xml has values for MaxLAI,
                //...  and PctEvGrnCov, startYr, endYr, and LeafAreaIndex_dataBase is file path and name to LAI.csv
                LeafAreaIndex.ProcessLAI(forData.MaxLAI, forData.PctEvGrnCov,
                                        LocationData_Object, startYr, endYr, LeafAreaIndex_dataBase);
                //Optional console output for debbugging
                //Console.WriteLine("Leaf area index timeseries DB constructed for use in evapotranspiration calculations.");

                //call ProcessSurfaceWeatherData to compute terms for weather database
                //Note: Variables sent to SurfaceWeather.cs ...
                //Note: ...  SurfaceWeatherDataFile, RainDataExcel, LocationData, LAIDatabase, vegType, StartYear, EndYear, WeatherDatabase, Height_WindSensor_m, Height_Tree_m
                SurfaceWeather.ProcessSurfaceWeatherData(MeteorologicalFile_string, PrecipitationFile_string, LocationData_Object, LeafAreaIndex_dataBase,
                                                        VegetationType, startYr, endYr, WeatherVariables_dBase,
                                                        Height_WeatherStationWindSensor_m, Height_TreeCanopyTop_m);
                //Optional console output for debbugging
                //Console.WriteLine("Meteorological timeseries DB constructed to generate output data files.\n");

                //assign names to output files, Weather.csv, Evaporation.csv, Radiation.csv
                string weatherFile = Path.GetDirectoryName(WeatherVariables_dBase) + "\\Weather.csv";
                string evFile = Path.GetDirectoryName(WeatherVariables_dBase) + "\\Evaporation.csv";
                string srFile = Path.GetDirectoryName(WeatherVariables_dBase) + "\\Radiation.csv";

                //call CreateWeatherDatCSV to create Weather.csv output
                CreateWeatherDatCSV(WeatherVariables_dBase, weatherFile);
                //Console output to manage expectations for upcoming program activity
                Console.WriteLine("Generating Weather, Evaporation, and Radiation data files ...");
                //Write output to user
                Console.WriteLine("The following output files were written:");
                Console.WriteLine(weatherFile);

                //call CreateEvapDatCSV to create Evaporation.csv output
                CreateEvapDatCSV(WeatherVariables_dBase, evFile);
                Console.WriteLine(evFile);

                //call CreateSolarRadDatCSV to create Radiation.csv output
                CreateSolarRadDatCSV(WeatherVariables_dBase, srFile);
                Console.WriteLine(srFile);

                //Note: LAIDatabase & WeatherDatabase are written to the default dirPath, defined above
                Console.WriteLine("Databases used to generate output, LAI.csv and Meteo.csv, were also provided.");

                //Write output to user
                Console.WriteLine("\nSimulation completed. Check above output files for results.");
                Console.WriteLine("Thank you for using i-Tree tools to improve the world!");
            }
            catch (Exception)
            {
                return;
            }
        }

        //CreateWeatherDatCSV function uses WeatherVariables_dBase from CSV file and weatherFile strings
        private void CreateWeatherDatCSV(string WeatherVariables_dBase, string weatherFile)
        {
            try
            {
                //Dimensionalize list variables to contain weather data
                List<DateTime> dtList = new List<DateTime>();
                List<double> Tair_F = new List<double>();
                List<double> Tdew_F = new List<double>();
                List<double> Radiation_Net_Wpm2 = new List<double>();
                List<double> WindSpeedms = new List<double>();
                List<double> Precipmh = new List<double>();
                List<double> Snowmh = new List<double>();
                List<double> WindDir_deg = new List<double>();
                List<double> AtmPressure_kPa = new List<double>();

                using (StreamReader WeatherVariables_StreamReader = new StreamReader(WeatherVariables_dBase))
                {
                    string Line_MeteorologicalVariables;
                    //read the header Line_MeteorologicalVariables
                    Line_MeteorologicalVariables = WeatherVariables_StreamReader.ReadLine();

                    //Loop through WeatherVariables_dBase and append weather data into new list variables
                    while ((Line_MeteorologicalVariables = WeatherVariables_StreamReader.ReadLine()) != null)
                    {
                        dtList.Add(DateTime.Parse(Line_MeteorologicalVariables.Split(',')[0]));
                        Tair_F.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[50]));
                        Tdew_F.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[53]));
                        Radiation_Net_Wpm2.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[41]));
                        WindSpeedms.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[57]));
                        Precipmh.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[43]));
                        Snowmh.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[47]));
                        WindDir_deg.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[55]));
                        AtmPressure_kPa.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[13]));
                    }
                }

                //Create sw as CSV file of weather data output using StreamWriter function
                using (StreamWriter sw = File.CreateText(weatherFile))
                {
                    //Write header to sw CSV file of weather data output
                    //Hydro model defined in WeatherPrepConfig.xml, for use w/ i-Tree Hydro input and contains 9 columns of variables
                    if (model_type == "Hydro")
                    {
                        sw.WriteLine("YYYYMMDD,HH:MM:SS,Temperature_Air_F(F),Temperature_DewPoint_F(F),Radiation_Net_Wpm2(W/m^2),Wind_Speed_mps(m/s),Pressure_Atmosphere_kPa(kPa),Precipitation_Rate_mph(m/h),Snow_DepthOnGround_m(m)");
                    }

                    //Energy model defined in WeatherPrepConfig.xml, for use w/ i-Tree Energy input and contains 10 columns of variables
                    if (model_type == "Energy")
                    {
                        sw.WriteLine("YYYYMMDD,HH:MM:SS,Temperature_Air_F(F),Temperature_DewPoint_F(F),Radiation_Net_Wpm2(W/m^2),Wind_Speed_mps(m/s),Wind_Direction_deg(deg),Pressure_Atmosphere_kPa(kPa),Precipitation_Rate_mph(m/h),Snow_DepthOnGround_m(m)");
                    }
                    //HydroEvap model defined in WeatherPrepConfig.xml, for use w/ i-Tree Hydro input and contains 8 columns of variables
                    if (model_type == "Hydro8")
                    {
                        sw.WriteLine("YYYYMMDD,HH:MM:SS,Temperature_Air_F(F),Temperature_DewPoint_F(F),Radiation_Net_Wpm2(W/m^2),Wind_Speed_mps(m/s),Precipitation_Rate_mph(m/h),Snow_DepthOnGround_m(m)");
                    }

                    //Flag_Date_January01_Missing is true if the first item in dtList[0] is not Month=1 and Day=1, yet January 2 exists
                    //Flag_Date_December31_Missing is true if the last item in dtList[dtList.Count-1] is not Month=12 and Day=31, December 30 exists
                    //Note: This multi-conditional ensures that if December 30 exists and December 31 doesn't safer to assume December 31 is missing
                    //Note: This multi-conditional ensures that if January 2 exists and January 1 doesn't safer to assume January 1 is missing
                    var (Flag_Date_January01_Missing, Flag_Date_December31_Missing) =
                        ComputeEdgePaddingFlags(dtList.Select(d => d.Date).ToList());

                    //If Flag_Date_January01_Missing is true then enter and insert missing date and data
                    if (Flag_Date_January01_Missing)
                    {
                        Console.WriteLine("Notice: Due to the time zone for your Place being west of UTC, ...");
                        Console.WriteLine("... January 1 data are incomplete and January 2 data were used.");
                        //Date_JanuaryFirst created as DateTime object, set equal to dtList[0].Date.AddDays(-1), where AddDays(-1) is subtracting one day
                        DateTime Date_JanuaryFirst = dtList[0].Date.AddDays(-1);
                        //temp_Dates_YYYYMMDD_HHMMSS created as DateTime object to contain a new list or vector of DateTime elements
                        List<DateTime> temp_Dates_YYYYMMDD_HHMMSS = new List<DateTime>();
                        //temp_Tair_F (F) created as List object to contain a new list or vector of double elements
                        List<double> temp_Tair_F = new List<double>();
                        List<double> temp_Tdew_F = new List<double>();
                        List<double> temp_NetRadWm2 = new List<double>();
                        List<double> temp_WindSpeedms = new List<double>();
                        List<double> temp_WindDir_deg = new List<double>();
                        List<double> temp_AtmPressure_kPa = new List<double>();
                        List<double> temp_Precipmh = new List<double>();
                        //temp_Snowmh (m/h) created as List object to contain a new list or vector of double elements
                        List<double> temp_Snowmh = new List<double>();

                        //For loop for hour, from 0 to 23, to create temporary vectors of the date and variables, 24 hours in length
                        for (int hour = 0; hour <= 23; hour++)
                        {
                            //temp_Dates_YYYYMMDD_HHMMSS uses Add to get dates from Date_JanuaryFirst vector created above, using AddHours function
                            temp_Dates_YYYYMMDD_HHMMSS.Add(Date_JanuaryFirst.AddHours(hour));
                            //temp_Tair_F uses Add to get Tair_F[hour] value, the existing vector value for the 24 hours on January 2
                            temp_Tair_F.Add(Tair_F[hour]);
                            temp_Tdew_F.Add(Tdew_F[hour]);
                            temp_NetRadWm2.Add(Radiation_Net_Wpm2[hour]);
                            temp_WindSpeedms.Add(WindSpeedms[hour]);
                            temp_WindDir_deg.Add(WindDir_deg[hour]);
                            temp_AtmPressure_kPa.Add(AtmPressure_kPa[hour]);
                            //temp_Precipmh (m/h) uses Add to get zero (0) value, ensuring precipitation is not overestimated for year
                            temp_Precipmh.Add(0.000);
                            //temp_Snowmh uses Add to get Snowmh[hour] value, the existing vector value for the 24 hours on January 2
                            temp_Snowmh.Add(Snowmh[hour]);
                        }

                        // Insert the temporary lists into dtList
                        //dtList uses InsertRange to add at the 0 location the vector temp_Dates_YYYYMMDD_HHMMSS, which contains 24 hours of data
                        dtList.InsertRange(0, temp_Dates_YYYYMMDD_HHMMSS);
                        //Tair_F uses InsertRange to add at the 0 location the vector temp_Tair_F, which contains 24 hours of data
                        Tair_F.InsertRange(0, temp_Tair_F);
                        Tdew_F.InsertRange(0, temp_Tdew_F);
                        Radiation_Net_Wpm2.InsertRange(0, temp_NetRadWm2);
                        WindSpeedms.InsertRange(0, temp_WindSpeedms);
                        WindDir_deg.InsertRange(0, temp_WindDir_deg);
                        AtmPressure_kPa.InsertRange(0, temp_AtmPressure_kPa);
                        //Precipmh (m/h) uses InsertRange to place temp_Precipmh=0, ensuring precipitation is not overestimated for year
                        Precipmh.InsertRange(0, temp_Precipmh);
                        //Snowmh uses InsertRange to add at the 0 location the vector temp_Snowmh (m/h), which contains 24 hours of data
                        Snowmh.InsertRange(0, temp_Snowmh);
                    }
                    // If Flag_Date_December31_Missing is true and dtList.Count >= 24, append missing date and data for December 31
                    if (Flag_Date_December31_Missing && dtList.Count >= 24)
                    {
                        // Date_December31 created as DateTime object, set equal to dtList[dtList.Count-1].Date.AddDays(1), where AddDays(1) adds one day
                        DateTime Date_December31 = dtList[dtList.Count - 1].Date.AddDays(1);
                        // Create temporary lists to hold the new data for December 31
                        List<DateTime> temp_Dates_YYYYMMDD_HHMMSS = new List<DateTime>();
                        List<double> temp_Tair_F = new List<double>();
                        List<double> temp_Tdew_F = new List<double>();
                        List<double> temp_NetRadWm2 = new List<double>();
                        List<double> temp_WindSpeedms = new List<double>();
                        List<double> temp_WindDir_deg = new List<double>();
                        List<double> temp_AtmPressure_kPa = new List<double>();
                        List<double> temp_Precipmh = new List<double>();
                        List<double> temp_Snowmh = new List<double>();

                        // For loop for hour, from 0 to 23, to create temporary vectors of the date and variables, 24 hours in length
                        for (int hour = 0; hour <= 23; hour++)
                        {
                            //temp_Dates_YYYYMMDD_HHMMSS uses Add to get dates from Date_December31 vector created above, using AddHours function
                            temp_Dates_YYYYMMDD_HHMMSS.Add(Date_December31.AddHours(hour));
                            int donor = Tair_F.Count - 24 + hour;
                            //temp_Tair_F uses Add to get Tair_F[hour] value, the existing vector value for the 24 hours on December 30
                            temp_Tair_F.Add(Tair_F[Tair_F.Count - 24 + hour]);
                            temp_Tdew_F.Add(Tdew_F[Tdew_F.Count - 24 + hour]);
                            temp_NetRadWm2.Add(Radiation_Net_Wpm2[Radiation_Net_Wpm2.Count - 24 + hour]);
                            temp_WindSpeedms.Add(WindSpeedms[WindSpeedms.Count - 24 + hour]);
                            temp_WindDir_deg.Add(WindDir_deg[WindDir_deg.Count - 24 + hour]);
                            temp_AtmPressure_kPa.Add(AtmPressure_kPa[AtmPressure_kPa.Count - 24 + hour]);
                            //temp_Precipmh (m/h) uses Add to get zero (0) value, ensuring precipitation is not overestimated for year
                            temp_Precipmh.Add(0.000);
                            // temp_Snowmh uses Add to get Snowmh[hour] value, the existing vector value for the 24 hours on December 30
                            temp_Snowmh.Add(Snowmh[Snowmh.Count - 24 + hour]);
                        }

                        // Append the temporary lists to dtList
                        // dtList uses AddRange to append the vector temp_Dates_YYYYMMDD_HHMMSS, which contains 24 hours of data
                        dtList.AddRange(temp_Dates_YYYYMMDD_HHMMSS);
                        // Tair_F uses AddRange to append the vector temp_Tair_F, which contains 24 hours of data
                        Tair_F.AddRange(temp_Tair_F);
                        Tdew_F.AddRange(temp_Tdew_F);
                        Radiation_Net_Wpm2.AddRange(temp_NetRadWm2);
                        WindSpeedms.AddRange(temp_WindSpeedms);
                        WindDir_deg.AddRange(temp_WindDir_deg);
                        AtmPressure_kPa.AddRange(temp_AtmPressure_kPa);
                        //Precipmh (m/h) uses AddRange to place temp_Precipmh=0, ensuring precipitation is not overestimated for year
                        Precipmh.AddRange(temp_Precipmh);
                        // Snowmh uses AddRange to append the vector temp_Snowmh (m/h), which contains 24 hours of data
                        Snowmh.AddRange(temp_Snowmh);
                    }

                    //Loop through index i from 0 to less than total items in variable list
                    for (int i = 0; i < dtList.Count; i++)
                        {
                            //HydroEvap model defined in WeatherPrepConfig.xml, for use w/ i-Tree Hydro w/ evaporation input and contains 9 columns of variables
                            if (model_type == "Hydro")
                            {
                                //Write data to sw CSV file, formatting to 8 digits precision
                                //Note: yyyyMMdd is format required by C# to write 8 digit date
                                sw.Write(dtList[i].Date.ToString("yyyyMMdd") + "," +
                                (dtList[i].Hour.ToString("00") + ":00:00") + "," +
                                First(Tair_F[i].ToString("0.00000000"), 11) + "," +
                                First(Tdew_F[i].ToString("0.00000000"), 11) + "," +
                                First(Radiation_Net_Wpm2[i].ToString("0.00000000"), 11) + "," +
                                First(WindSpeedms[i].ToString("0.00000000"), 11) + "," +
                                First(AtmPressure_kPa[i].ToString("0.00000000"), 11) + "," +
                                First(Precipmh[i].ToString("0.00000000"), 11) + "," +
                                First(Snowmh[i].ToString("0.00000000"), 11) + "\r\n");
                            }
                        //Energy model defined in WeatherPrepConfig.xml, for use w/ i-Tree Energy input and contains 10 columns of variables
                        if (model_type == "Energy")
                        {
                            //Write data to sw CSV file, formatting to 8 digits precision
                            //Note: yyyyMMdd is format required by C# to write 8 digit date
                            sw.Write(dtList[i].Date.ToString("yyyyMMdd") + "," +
                            (dtList[i].Hour.ToString("00") + ":00:00") + "," +
                            First(Tair_F[i].ToString("0.00000000"), 11) + "," +
                            First(Tdew_F[i].ToString("0.00000000"), 11) + "," +
                            First(Radiation_Net_Wpm2[i].ToString("0.00000000"), 11) + "," +
                            First(WindSpeedms[i].ToString("0.00000000"), 11) + "," +
                            First(WindDir_deg[i].ToString("0.00000000"), 11) + "," +
                            First(AtmPressure_kPa[i].ToString("0.00000000"), 11) + "," +
                            First(Precipmh[i].ToString("0.00000000"), 11) + "," +
                            First(Snowmh[i].ToString("0.00000000"), 11) + "\r\n");
                        }
                        //Hydro model defined in WeatherPrepConfig.xml, for use w/ i-Tree Hydro input and contains 8 columns of variables
                        if (model_type == "Hydro8")
                        {
                            //Write data to sw CSV file, formatting to 8 digits precision
                            //Note: yyyyMMdd is format required by C# to write 8 digit date
                            sw.Write(dtList[i].Date.ToString("yyyyMMdd") + "," +
                            (dtList[i].Hour.ToString("00") + ":00:00") + "," +
                            First(Tair_F[i].ToString("0.00000000"), 11) + "," +
                            First(Tdew_F[i].ToString("0.00000000"), 11) + "," +
                            First(Radiation_Net_Wpm2[i].ToString("0.00000000"), 11) + "," +
                            First(WindSpeedms[i].ToString("0.00000000"), 11) + "," +
                            First(Precipmh[i].ToString("0.00000000"), 11) + "," +
                            First(Snowmh[i].ToString("0.00000000"), 11) + "\r\n");
                        }
                    }
                    //Close sw CSV file
                    sw.Close();
                }
            }
            catch (Exception)
            {
                throw;
            }
        }
        private void CreateEvapDatCSV(string WeatherVariables_dBase, string evFile)
        {
            try
            {
                List<DateTime> dtList = new List<DateTime>();
                List<double> pET = new List<double>(); //pET.dat PtTrMh
                List<double> pE = new List<double>(); // pE.dat PeGrMh
                List<double> pETree = new List<double>(); //pEvTree -> pETree.dat PeTrMh
                List<double> PESnow = new List<double>();
                List<double> PeTreeSnow = new List<double>();

                using (StreamReader WeatherVariables_StreamReader = new StreamReader(WeatherVariables_dBase))
                {
                    string Line_MeteorologicalVariables;
                    //read the header Line_MeteorologicalVariables
                    WeatherVariables_StreamReader.ReadLine();

                    //Loop through WeatherVariables_dBase and append weather data into new list variables
                    while ((Line_MeteorologicalVariables = WeatherVariables_StreamReader.ReadLine()) != null)
                    {
                        dtList.Add(DateTime.Parse(Line_MeteorologicalVariables.Split(',')[0]));
                        pET.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[19]));
                        pE.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[16]));
                        pETree.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[15]));
                        PESnow.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[18]));
                        PeTreeSnow.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[17]));
                    }
                }
                using (StreamWriter sw = File.CreateText(evFile))
                {
                    //Write header line for Evaporation.csv
                    sw.WriteLine("YYYYMMDD,HH:MM:SS,PotentialEvapotranspiration_TreeCover_mph(m/h),PotentialEvaporation_WaterOnGround_mph(m/h),PotentialEvaporation_WaterOnTree_mph(m/h),PotentialSublimation_SnowOnGround_mph(m/h),PotentialSublimation_SnowOnTree_mph(m/h)");

                    //Flag_Date_January01_Missing is true if the first item in dtList[0] is not Month=1 and Day=1, yet January 2 exists
                    //Flag_Date_December31_Missing is true if the last item in dtList[dtList.Count-1] is not Month=12 and Day=31, December 30 exists
                    //Note: This multi-conditional ensures that if December 30 exists and December 31 doesn't safer to assume December 31 is missing
                    //Note: This multi-conditional ensures that if January 2 exists and January 1 doesn't safer to assume January 1 is missing
                    var (Flag_Date_January01_Missing, Flag_Date_December31_Missing) =
                        ComputeEdgePaddingFlags(dtList.Select(d => d.Date).ToList());

                    //If Flag_Date_January01_Missing is true then enter and insert missing date and data
                    if (Flag_Date_January01_Missing)
                    {
                        //Date_JanuaryFirst created as DateTime object, set equal to dtList[0].Date.AddDays(-1), where AddDays(-1) is subtracting one day
                        DateTime Date_JanuaryFirst = dtList[0].Date.AddDays(-1);
                        //temp_Dates_YYYYMMDD_HHMMSS created as DateTime object to contain a new list or vector of DateTime elements
                        List<DateTime> temp_Dates_YYYYMMDD_HHMMSS = new List<DateTime>();
                        //temp_pET (m) created as List object to contain a new list or vector of double elements
                        List<double> temp_pET = new List<double>();
                        List<double> temp_pE = new List<double>();
                        List<double> temp_pETree = new List<double>();
                        List<double> temp_pESnow = new List<double>();
                        List<double> temp_peTreeSnow = new List<double>();

                        //For loop for hour, from 0 to 23, to create temporary vectors of the date and variables, 24 hours in length
                        for (int hour = 0; hour <= 23; hour++)
                        {
                            //temp_Dates_YYYYMMDD_HHMMSS uses Add to get dates from Date_JanuaryFirst vector created above, using AddHours function
                            temp_Dates_YYYYMMDD_HHMMSS.Add(Date_JanuaryFirst.AddHours(hour));
                            //temp_pET uses Add to get pET[hour] value, the existing vector value for the 24 hours on January 2
                            temp_pET.Add(pET[hour]);
                            temp_pE.Add(pE[hour]);
                            temp_pETree.Add(pETree[hour]);
                            temp_pESnow.Add(PESnow[hour]);
                            temp_peTreeSnow.Add(PeTreeSnow[hour]);
                        }

                        // Insert the temporary lists into dtList
                        //dtList uses InsertRange to add at the 0 location the vector temp_Dates_YYYYMMDD_HHMMSS, which contains 24 hours of data
                        dtList.InsertRange(0, temp_Dates_YYYYMMDD_HHMMSS);
                        //pET uses InsertRange to add at the 0 location the vector temp_pET, which contains 24 hours of data
                        pET.InsertRange(0, temp_pET);
                        pE.InsertRange(0, temp_pE);
                        pETree.InsertRange(0, temp_pETree);
                        PESnow.InsertRange(0, temp_pESnow);
                        PeTreeSnow.InsertRange(0, temp_peTreeSnow);
                    }
                    // If Flag_Date_December31_Missing is true and dtList.Count >= 24, append missing date and data for December 31
                    if (Flag_Date_December31_Missing && dtList.Count >= 24)
                    {
                        // Date_December31 created as DateTime object, set equal to dtList[dtList.Count-1].Date.AddDays(1), where AddDays(1) adds one day
                        DateTime Date_December31 = dtList[dtList.Count - 1].Date.AddDays(1);
                        // Create temporary lists to hold the new data for December 31
                        List<DateTime> temp_Dates_YYYYMMDD_HHMMSS = new List<DateTime>();
                        List<double> temp_pET = new List<double>();
                        List<double> temp_pE = new List<double>();
                        List<double> temp_pETree = new List<double>();
                        List<double> temp_pESnow = new List<double>();
                        List<double> temp_peTreeSnow = new List<double>();
                        List<double> temp_Precipmh = new List<double>();
                        List<double> temp_Snowmh = new List<double>();

                        // For loop for hour, from 0 to 23, to create temporary vectors of the date and variables, 24 hours in length
                        for (int hour = 0; hour <= 23; hour++)
                        {
                            // temp_Dates_YYYYMMDD_HHMMSS uses Add to get dates from Date_December31 vector created above, using AddHours function
                            temp_Dates_YYYYMMDD_HHMMSS.Add(Date_December31.AddHours(hour));
                            // temp_pET uses Add to get pET[hour] value, the existing vector value for the 24 hours on December 30
                            temp_pET.Add(pET[pET.Count - 24 + hour]);
                            temp_pE.Add(pE[pE.Count - 24 + hour]);
                            temp_pETree.Add(pETree[pETree.Count - 24 + hour]);
                            temp_pESnow.Add(PESnow[PESnow.Count - 24 + hour]);
                            temp_peTreeSnow.Add(PeTreeSnow[PeTreeSnow.Count - 24 + hour]);
                        }

                        // Append the temporary lists to dtList
                        // dtList uses AddRange to append the vector temp_Dates_YYYYMMDD_HHMMSS, which contains 24 hours of data
                        dtList.AddRange(temp_Dates_YYYYMMDD_HHMMSS);
                        // pET uses AddRange to append the vector temp_pET, which contains 24 hours of data
                        pET.AddRange(temp_pET);
                        pE.AddRange(temp_pE);
                        pETree.AddRange(temp_pETree);
                        PESnow.AddRange(temp_pESnow);
                        PeTreeSnow.AddRange(temp_peTreeSnow);
                    }

                    for (int i = 0; i < dtList.Count; i++)
                    {
                        //Note: yyyyMMdd is format required by C# to write 8 digit date
                        sw.Write(dtList[i].Date.ToString("yyyyMMdd") + "," +
                            (dtList[i].Hour.ToString("00") + ":00:00") + "," +
                            First(pET[i].ToString("0.00000000"), 11) + "," +
                            First(pE[i].ToString("0.00000000"), 11) + "," +
                            First(pETree[i].ToString("0.00000000"), 11) + "," +
                            First(PESnow[i].ToString("0.00000000"), 11) + "," +
                            First(PeTreeSnow[i].ToString("0.00000000"), 11) + "\r\n");
                    }
                    sw.Close();
                }
            }
            catch (Exception)
            {
                throw;
            }
        }
        private void CreateSolarRadDatCSV(string WeatherVariables_dBase, string srFile)
        {
            try
            {
                List<DateTime> dtList = new List<DateTime>();
                List<double> DirSwWm2 = new List<double>();
                List<double> DiffSwWm2 = new List<double>();
                List<double> DownLwWm2 = new List<double>();
                List<double> UpLwWm2 = new List<double>();
                List<double> LatRad = new List<double>();
                List<double> HrAngleRad = new List<double>();
                List<double> DecAngleRad = new List<double>();

                using (StreamReader WeatherVariables_StreamReader = new StreamReader(WeatherVariables_dBase))
                {
                    string Line_MeteorologicalVariables;
                    //read the header Line_MeteorologicalVariables
                    WeatherVariables_StreamReader.ReadLine();

                    //Loop through WeatherVariables_dBase and append weather data into new list variables
                    while ((Line_MeteorologicalVariables = WeatherVariables_StreamReader.ReadLine()) != null)
                    {
                        dtList.Add(DateTime.Parse(Line_MeteorologicalVariables.Split(',')[0]));
                        DirSwWm2.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[36]));
                        DiffSwWm2.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[37]));
                        DownLwWm2.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[1]));
                        UpLwWm2.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[2]));
                        LatRad.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[3]));
                        HrAngleRad.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[4]));
                        DecAngleRad.Add(Double.Parse(Line_MeteorologicalVariables.Split(',')[5]));
                    }
                }
                using (StreamWriter sw = File.CreateText(srFile))
                {
                    sw.WriteLine("YYYYMMDD,HH:MM:SS,Radiation_Shortwave_Direct_Wpm2(W/m^2),Radiation_Shortwave_Diffuse_Wpm2(W/m^2),Radiation_Longwave_Downwelling_Wpm2(W/m^2),Radiation_Longwave_Upwelling_Wpm2(W/m^2),Latitude_rad(Radian),HourAngle_rad(Radian),DeclinationAngle_rad(Radian)");

                    //Flag_Date_January01_Missing is true if the first item in dtList[0] is not Month=1 and Day=1, yet January 2 exists
                    //Flag_Date_December31_Missing is true if the last item in dtList[dtList.Count-1] is not Month=12 and Day=31, December 30 exists
                    //Note: This multi-conditional ensures that if December 30 exists and December 31 doesn't safer to assume December 31 is missing
                    //Note: This multi-conditional ensures that if January 2 exists and January 1 doesn't safer to assume January 1 is missing
                    var (Flag_Date_January01_Missing, Flag_Date_December31_Missing) =
                        ComputeEdgePaddingFlags(dtList.Select(d => d.Date).ToList());

                    //If Flag_Date_January01_Missing is true then enter and insert missing date and data
                    if (Flag_Date_January01_Missing)
                    {
                        //Date_JanuaryFirst created as DateTime object, set equal to dtList[0].Date.AddDays(-1), where AddDays(-1) is subtracting one day
                        DateTime Date_JanuaryFirst = dtList[0].Date.AddDays(-1);
                        //temp_Dates_YYYYMMDD_HHMMSS created as DateTime object to contain a new list or vector of DateTime elements
                        List<DateTime> temp_Dates_YYYYMMDD_HHMMSS = new List<DateTime>();
                        //temp_DirSwWm2 (F) created as List object to contain a new list or vector of double elements
                        List<double> temp_DirSwWm2 = new List<double>();
                        List<double> temp_DiffSwWm2 = new List<double>();
                        List<double> temp_DownLwWm2 = new List<double>();
                        List<double> temp_UpLwWm2 = new List<double>();
                        List<double> temp_LatRad = new List<double>();
                        List<double> temp_HrAngleRad = new List<double>();
                        //temp_DecAngleRad (m/h) created as List object to contain a new list or vector of double elements
                        List<double> temp_DecAngleRad = new List<double>();

                        //For loop for hour, from 0 to 23, to create temporary vectors of the date and variables, 24 hours in length
                        for (int hour = 0; hour <= 23; hour++)
                        {
                            //temp_Dates_YYYYMMDD_HHMMSS uses Add to get dates from Date_JanuaryFirst vector created above, using AddHours function
                            temp_Dates_YYYYMMDD_HHMMSS.Add(Date_JanuaryFirst.AddHours(hour));
                            //temp_DirSwWm2 uses Add to get DirSwWm2[hour] value, the existing vector value for the 24 hours on January 2
                            temp_DirSwWm2.Add(DirSwWm2[hour]);
                            temp_DiffSwWm2.Add(DiffSwWm2[hour]);
                            temp_DownLwWm2.Add(DownLwWm2[hour]);
                            temp_UpLwWm2.Add(UpLwWm2[hour]);
                            temp_LatRad.Add(LatRad[hour]);
                            temp_HrAngleRad.Add(HrAngleRad[hour]);
                            //temp_DecAngleRad uses Add to get DecAngleRad[hour] value, the existing vector value for the 24 hours on January 2
                            temp_DecAngleRad.Add(DecAngleRad[hour]);
                        }

                        // Insert the temporary lists into dtList
                        //dtList uses InsertRange to add at the 0 location the vector temp_Dates_YYYYMMDD_HHMMSS, which contains 24 hours of data
                        dtList.InsertRange(0, temp_Dates_YYYYMMDD_HHMMSS);
                        //DirSwWm2 uses InsertRange to add at the 0 location the vector temp_DirSwWm2, which contains 24 hours of data
                        DirSwWm2.InsertRange(0, temp_DirSwWm2);
                        DiffSwWm2.InsertRange(0, temp_DiffSwWm2);
                        DownLwWm2.InsertRange(0, temp_DownLwWm2);
                        UpLwWm2.InsertRange(0, temp_UpLwWm2);
                        LatRad.InsertRange(0, temp_LatRad);
                        HrAngleRad.InsertRange(0, temp_HrAngleRad);
                        //DecAngleRad uses InsertRange to add at the 0 location the vector temp_DecAngleRad (m/h), which contains 24 hours of data
                        DecAngleRad.InsertRange(0, temp_DecAngleRad);
                    }
                    // If Flag_Date_December31_Missing is true and dtList.Count >= 24, append missing date and data for December 31
                    if (Flag_Date_December31_Missing && dtList.Count >= 24)
                    {
                        // Date_December31 created as DateTime object, set equal to dtList[dtList.Count-1].Date.AddDays(1), where AddDays(1) adds one day
                        DateTime Date_December31 = dtList[dtList.Count - 1].Date.AddDays(1);
                        // Create temporary lists to hold the new data for December 31
                        List<DateTime> temp_Dates_YYYYMMDD_HHMMSS = new List<DateTime>();
                        List<double> temp_DirSwWm2 = new List<double>();
                        List<double> temp_DiffSwWm2 = new List<double>();
                        List<double> temp_DownLwWm2 = new List<double>();
                        List<double> temp_UpLwWm2 = new List<double>();
                        List<double> temp_LatRad = new List<double>();
                        List<double> temp_HrAngleRad = new List<double>();
                        List<double> temp_DecAngleRad = new List<double>();

                        // For loop for hour, from 0 to 23, to create temporary vectors of the date and variables, 24 hours in length
                        for (int hour = 0; hour <= 23; hour++)
                        {
                            // temp_Dates_YYYYMMDD_HHMMSS uses Add to get dates from Date_December31 vector created above, using AddHours function
                            temp_Dates_YYYYMMDD_HHMMSS.Add(Date_December31.AddHours(hour));
                            // temp_DirSwWm2 uses Add to get DirSwWm2[hour] value, the existing vector value for the 24 hours on December 30
                            temp_DirSwWm2.Add(DirSwWm2[DirSwWm2.Count - 24 + hour]);
                            temp_DiffSwWm2.Add(DiffSwWm2[DiffSwWm2.Count - 24 + hour]);
                            temp_DownLwWm2.Add(DownLwWm2[DownLwWm2.Count - 24 + hour]);
                            temp_UpLwWm2.Add(UpLwWm2[UpLwWm2.Count - 24 + hour]);
                            temp_LatRad.Add(LatRad[LatRad.Count - 24 + hour]);
                            temp_HrAngleRad.Add(HrAngleRad[HrAngleRad.Count - 24 + hour]);
                            // temp_DecAngleRad uses Add to get DecAngleRad[hour] value, the existing vector value for the 24 hours on December 30
                            temp_DecAngleRad.Add(DecAngleRad[DecAngleRad.Count - 24 + hour]);
                        }

                        // Append the temporary lists to dtList
                        // dtList uses AddRange to append the vector temp_Dates_YYYYMMDD_HHMMSS, which contains 24 hours of data
                        dtList.AddRange(temp_Dates_YYYYMMDD_HHMMSS);
                        // DirSwWm2 uses AddRange to append the vector temp_DirSwWm2, which contains 24 hours of data
                        DirSwWm2.AddRange(temp_DirSwWm2);
                        DiffSwWm2.AddRange(temp_DiffSwWm2);
                        DownLwWm2.AddRange(temp_DownLwWm2);
                        UpLwWm2.AddRange(temp_UpLwWm2);
                        LatRad.AddRange(temp_LatRad);
                        HrAngleRad.AddRange(temp_HrAngleRad);
                        // DecAngleRad uses AddRange to append the vector temp_DecAngleRad (m/h), which contains 24 hours of data
                        DecAngleRad.AddRange(temp_DecAngleRad);
                    }

                    for (int i = 0; i < dtList.Count; i++)
                    {
                        //Note: yyyyMMdd is format required by C# to write 8 digit date
                        sw.Write(dtList[i].Date.ToString("yyyyMMdd") + "," +
                            (dtList[i].Hour.ToString("00") + ":00:00") + "," +
                             First(DirSwWm2[i].ToString("0.00000"), 7) + "," +
                             First(DiffSwWm2[i].ToString("0.00000"), 7) + "," +
                             First(DownLwWm2[i].ToString("0.00000"), 7) + "," +
                             First(UpLwWm2[i].ToString("0.00000"), 7) + "," +
                             First(LatRad[i].ToString("0.00000"), 7) + "," +
                             First(HrAngleRad[i].ToString("0.00000"), 7) + "," +
                             First(DecAngleRad[i].ToString("0.00000"), 7) + "\r\n");
                    }
                    sw.Close();
                }
            }
            catch (Exception)
            {
                throw;
            }
        }
        private String First(string p, int p_2)
        {
            return p.Substring(0, Math.Min(p.Length, p_2));
        }
    }
}

