﻿using NHibernate;
using System;
using System.Collections.Generic;
using LocationSpecies.Domain;

namespace WeatherPrep
{
    /// <summary>
    /// Location data for dry deposition calculations
    /// LocationData function called by CreateOutput.cs
    /// </summary>
    public class LocationData
    {
        #region "Class variables"
        // Variables for unused air pollution-related functions and old LocationSpecies.MDB connections
        //public string NationID;
        //public string Nation;
        //public string PriPartID;
        //public string PriPart;
        //public string SecPartID;
        //public string SecPart;
        //public string TerPartID;
        //public string TerPart;
        public double Latitude;
        public double Longitude;
        public int LeafOff_MidPoint_JD;
        public int LeafOn_MidPoint_JD;
        public double Elevation;
        public double GMTOffset;
        public double[] ozone;
        public int AlbedoStID;
        public int albedo;
        public double[] AlbedoVal;
        public double CoeffA;
        public double CoeffC;
        public double CoeffPHI;
        public double TerrainFactor;
        // Variables for unused air pollution-related functions and old LocationSpecies.MDB connections
        //public int[] MonStID;
        //public string[] COMonState;
        //public string[] COMonCounty;
        //public string[] COMonSite;
        //public string[] NO2MonState;
        //public string[] NO2MonCounty;
        //public string[] NO2MonSite;
        //public string[] O3MonState;
        //public string[] O3MonCounty;
        //public string[] O3MonSite;
        //public string[] PM10MonState;
        //public string[] PM10MonCounty;
        //public string[] PM10MonSite;
        //public double[] PM10MonLat;
        //public double[] PM10MonLong;
        //public string[] PM25MonState;
        //public string[] PM25MonCounty;
        //public string[] PM25MonSite;
        //public double[] PM25MonLat;
        //public double[] PM25MonLong;
        //public string[] SO2MonState;
        //public string[] SO2MonCounty;
        //public string[] SO2MonSite;
        //public double COBaseCost;
        //public int COBaseYear;
        //public double COBaseYearPPI;
        //public double NO2BaseCost;
        //public int NO2BaseYear;
        //public double NO2BaseYearPPI;
        //public double O3BaseCost;
        //public int O3BaseYear;
        //public double O3BaseYearPPI;
        //public double PM10BaseCost;
        //public int PM10BaseYear;
        //public double PM10BaseYearPPI;
        //public double PM25BaseCost;
        //public int PM25BaseYear;
        //public double PM25BaseYearPPI;
        //public double SO2BaseCost;
        //public int SO2BaseYear;
        //public double SO2BaseYearPPI;
        //public double ModelYearPPI;

        private ISession _s;
        #endregion
        /// <summary>
        /// Initialize arrays to store location related information.
        /// </summary>
        public LocationData(ISession session)
        {
            ozone = new double[12];
            AlbedoVal = new double[12];
            // Variables for unused air pollution-related functions and old LocationSpecies.MDB connections
            //MonStID = new int[0];
            //COMonState = new string[1];
            //COMonCounty = new string[1];
            //COMonSite = new string[1];
            //NO2MonState = new string[1];
            //NO2MonCounty = new string[1];
            //NO2MonSite = new string[1];
            //O3MonState = new string[1];
            //O3MonCounty = new string[1];
            //O3MonSite = new string[1];
            //PM10MonState = new string[1];
            //PM10MonCounty = new string[1];
            //PM10MonSite = new string[1];
            //PM10MonLat = new double[1];
            //PM10MonLong = new double[1];
            //PM25MonState = new string[1];
            //PM25MonCounty = new string[1];
            //PM25MonSite = new string[1];
            //PM25MonLat = new double[1];
            //PM25MonLong = new double[1];
            //SO2MonState = new string[1];
            //SO2MonCounty = new string[1];
            //SO2MonSite = new string[1];

            _s = session; 
        }

        //ReadLocationData function takes location of SQLite LocationSpecies Database, using NHibernate protocols
        public void ReadLocationData(Location location)
        {
            try
            {
                //NHibernate function .BeginTransaction with SQLite LocationSpecies Database
                using (ITransaction trans = _s.BeginTransaction())
                {
                    // Define Lat/Long and LeafOn/OffDOY using given location.
                    Latitude = location.Latitude;
                    Longitude = location.Longitude;
                    Elevation = location.Elevation;
                    LeafOff_MidPoint_JD = location.GrowthPeriod.LeafOffDays;
                    LeafOn_MidPoint_JD = location.GrowthPeriod.LeafOnDays;

                    //If LeafOff_MidPoint_JD and LeafOn_MidPoint_JD are both 0, then for North or South Hemisphere the LeafOff_MidPoint_JD = 365 and LeafOn_MidPoint_JD = 0
                    if (LeafOff_MidPoint_JD == 0 && LeafOn_MidPoint_JD == 0)
                    {
                        //LeafOff_MidPoint_JD set to 365
                        //Note: Will correct for leap year in function CreateLAI_AnnualDaily_NoFrostDays
                        LeafOff_MidPoint_JD = 365;
                    }

                    // Functions to walk-up location tree for given locaton if given location does not have needed data.
                    // Unique functions are provided because different parent locations may be needed to derive each data.
                    trans.Commit();
                }
                ReadTimezoneData(location);
                ReadOzoneData(location);
                ReadAlbedoData(location);
            }
            catch (Exception)
            {
                Console.Write("\n");
                Console.Write("Warning: Apparent error in WeatherPrepConfig.xml. ReadLocationData function cannot find that Place.\n");
                Console.Write("Suggestion: If the place does not exist in the i-Tree Location Database, then create it with Add to Database.\n");
                throw;
            }
        }

        void ReadTimezoneData(Location location)
        {
            using (ITransaction tx = _s.BeginTransaction())
            {
                var timeZone = location.TimeZone;
                while (location != null && timeZone == null)
                {
                    var lr = _s.QueryOver<LocationRelation>()
                        .Where(r => r.Location == location && r.Code != null)
                        .Cacheable()
                        .SingleOrDefault();

                    location = lr?.Parent;
                    timeZone = location?.TimeZone;
                }
                GMTOffset = timeZone.Offset;
                tx.Commit();
            }
        }

        void ReadOzoneData(Location location)
        {
            Console.WriteLine("Starting ozone data extraction...");

            IDictionary<Month, double> ozoneValues = null;

            using (ITransaction tx = _s.BeginTransaction())
            {
                Console.WriteLine("Transaction started for location: {0}", location?.Name);

                var stateLat = location.StateLat;
                while (location != null && stateLat == null)
                {
                    Console.WriteLine("Current location: {0}", location?.Name);

                    var lr = _s.QueryOver<LocationRelation>()
                        .Where(r => r.Location == location && r.Code != null)
                        .Cacheable()
                        .SingleOrDefault();

                    location = lr?.Parent;
                    stateLat = location?.StateLat;

                    Console.WriteLine("Moved to parent location: {0}, StateLat: {1}", location?.Name, stateLat);
                }

                ozoneValues = stateLat?.OzoneValues ?? new Dictionary<Month, double>();

                tx.Commit();
                Console.WriteLine("Transaction committed.");
            }

            Console.WriteLine("Ozone values retrieved: {0}", ozoneValues?.Count ?? 0);

            foreach (KeyValuePair<Month, double> kvp in ozoneValues)
            {
                ozone[((int)kvp.Key) - 1] = kvp.Value;
                Console.WriteLine("Ozone for Month {0}: {1}", kvp.Key, kvp.Value);
            }

            Console.WriteLine("Ozone data extraction completed.");
        }

        void ReadAlbedoData(Location location)
        {
            try
            {
                using (ITransaction trans = _s.BeginTransaction())
                {
                    AlbedoStID = location.AlbedoStation.Id;
                    CoeffA = location.AlbedoStation.CoefficientA;
                    CoeffC = location.AlbedoStation.CoefficientC;
                    CoeffPHI = location.AlbedoStation.CoefficientPhi;

                    // Print station ID and coefficients
                    Console.WriteLine($"Albedo Station ID: {AlbedoStID}");
                    Console.WriteLine($"Coefficient A: {CoeffA}");
                    Console.WriteLine($"Coefficient C: {CoeffC}");
                    Console.WriteLine($"Coefficient PHI: {CoeffPHI}");

                    foreach (KeyValuePair<Month, double> kvp in location.AlbedoStation.Albedo.Values)
                    {
                        AlbedoVal[((int)kvp.Key) - 1] = kvp.Value;

                        // Print month and corresponding albedo value
                        Console.WriteLine($"Month: {kvp.Key}, Albedo Value: {kvp.Value}");
                    }

                    trans.Commit();
                }
            }
            catch (Exception ex)
            {
                // Print error message for debugging
                Console.WriteLine($"An error occurred: {ex.Message}");
                throw;
            }
        }
    }
}


