﻿using NHibernate.Criterion;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Web.UI.WebControls.WebParts;


namespace WeatherPrep
{
    /// <summary>
    /// Several functions to read raw surface weather data and compute weather variables
    /// </summary>
    public enum VEGTYPE { TREE, SHRUB, GRASS, CORN };

    public class SurfaceWeather
    {
        #region class_variables
        const double TEMP_CK = 273.15;                      ///<value>Conversino factor for Celsius to Kelvin</value>
        const double XLIM = 0.000000001;
        const double ABS_PRS = 1013.25;                     ///<value>mBar</value>
        const double ABS_TEMP = 288.15;                     ///<value>Kelvin</value>
        const int SOL_CONST = 1367;
        const double ES_CONS = 0.6108;                      ///<value>saturated vapor pressure constant (kPa)</value>
        const double ATRN = 0.957;
        const double K1 = 0.1;
        const double BA = 0.84;
        const double PA_CONS = 3.486;                       ///<value>Density for moist air sensible heat flux constant</value>
        const double CHI_CONS = 2.501;                      ///<value>latent heaf for evaoration of water, adjusted by temperature (used in meteo and evap routines)</value>
        const double GAMMA_C = 0.0016286;                   ///<value>psychrometric constant, adjusted by vapor pressure (used in meteo and evap routines)</value>
        const double theta = 0.0000000567;                  ///<value>Stefan-Boltzmann constant (MJ m^-2 K^-4 day^-1)</value>
        const double ES = 0.97;                             ///<value>surface emissivity for long wave radiation</value>
        const double ES_RN = 0.98;                          ///<value>surface emissivity (M.C. Llasat, R.L. Snyder, Ag and Forest Meteorology, 1998)</value>
        const double CP = 1013;                             ///<value>specific heat of moist air (J/Kg C)</value>
        const int ZCONS = 2;                                ///<value>height of weather station (m)</value>
        const double rd = 0.00137;                          ///<value>roughness height for water (m)</value>
        const double rDS = 0.005;                           ///<value>roughness height for snow (m)</value>
        const double rDT = 0.95;                            ///<value>roughness height for tree (m)</value>
        const double rTreeHeight = 5;                       ///<value>tree height (m)</value>
        const double rGroundHeight_m = 0.1;                  ///<value>ground height (m)</value>
        const double rZom = 0.123;                          ///<value>roughness length coefficient controlling momentum transfer defined Eq 13.3 Chin (2021)</value>
        const double rZov = 0.0123;                         ///<value>roughness length coefficient controlling heat and vapor transfer defined Eq 13.4 and following text Chin (2021)</value>
        const double rZu = ZCONS;                           ///<value>wind measurement height (m) from constants</value>
        const double rZe = ZCONS;                           ///<value>vapor measurement height (m) from constants</value>
        const double rZuT = rTreeHeight + rZu;              ///<value>wind estimate height in tree (m)</value>
        const double rZuG = 0.1 + rd;                       ///<value>wind estimate height for ground (m)</value>
        const double rZuG_m = rGroundHeight_m + rZu;                       ///<value>wind estimate height for ground (m)</value>

        public const double LEAF_STORAGE_M = 0.0002;               ///<value>specific leaf storage of water = 0.2 mm = 0.0002 m</value>
        const double IMPERV_STORAGE_M = 0.0015;             ///<value>maximum impervious cover storage of water = 1.5 mm = 0.0015 m</value>
        const double PERV_STORAGE_M = 0.001;                ///<value>maximum pervious cover storage of water = 1 mm = 0.001 m</value>                                    

        public const double NODATA = -999;
        const int NOWBAN = 99999;
        public const string SFC_TABLE = "SurfaceWeather";
        const string HRMIX_TABLE = "HourlyMixHt";

        const string NEW_INTERNATIONAL = "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB WW WW WW ZZ ZZ ZZ W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01 PCP06 PCP24 PCPXX SD";
        const string NEW_US_CANADA = "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB MW MW MW MW AW AW AW AW W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01 PCP06 PCP24 PCPXX SD";
        const string OLD = "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB WW WW WW W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01 PCP06 PCP24 PCPXX SD";
        const string NARCCAP = "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB WW WW WW W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01      NETRAD PCPXX SD";
        // Field number for the old surface weather format
        const int USAF = 0;
        const int WBAN = 1;
        const int DATETIME = 2;
        const int DIR = 3;
        const int SPD = 4;
        const int GUS = 5;
        const int CLG = 6;
        const int SKC = 7;
        const int TEMP = 16;
        const int DEWP = 17;
        const int SLP = 18;
        const int ALT = 19;
        const int STP = 20;
        const int MAX = 21;
        const int MIN = 22;
        const int PCP01 = 23;
        const int NETRAD = 24;
        const int SD = 27;

        public DateTime TimeStamp;
        public int Jday;
        public int Year;
        public int Month;
        public int Day;
        public int Hour;
        public int Minute;
        public double Radiation_Longwave_Downwelling_Wpm2;
        public double Radiation_Longwave_Upwelling_Wpm2;
        public double LatRad;
        public double HourAngleRad;
        public double DecAngleRad;
        public double AirDens_kg_p_m3;
        public double airMass_relativeOptical_kg_p_m2;
        public double Altimeter;
        public double CeilingHeight_hundredths_ft;
        public double DewTempC;
        public double DewTempF;
        public double Radiation_Shortwave_Diffuse_Wpm2;
        public double Radiation_Shortwave_Direct_Wpm2;
        public double Radiation_Shortwave_Total_Wpm2;
        public double Radiation_Net_Wpm2;
        public double CloudCover_Opaque_tenths;
        public double PrsIn;
        public double PrsKpa;
        public double PrsMbar;
        public double PeGrMh;
        public double PeTrMh;
        public double PeSnGrMh;
        public double PeSnTrMh;
        public double PtTrMh;
        public double VegEvMh;                  ///<value>Evaporation from vegetations (m/h)</value>
        public double VegStMh;                  ///<value>Storage of precipitation by vegetation (m/h)</value>
        public double VegIntcptMh;              ///<value>Precipitation interception by vegetation (m/h)</value>
        public double UnderCanThrufallMh;       ///<value>Under canopy throughfall of precipitation (m/h)</value>
        public double UnderCanPervEvMh;         ///<value>Evaporation from under canopy pervious cover (m/h)</value>
        public double UnderCanPervStMh;         ///<value>Storage of precipitation by under canopy pervious cover (m/h)</value>                                       
        public double UnderCanPervInfilMh;      ///<value>Infiltration from under canopy pervious cover (m/h)</value>
        public double UnderCanImpervEvMh;       ///<value>Evaporation from under canopy impervious cover (m/h)</value>
        public double UnderCanImpervStMh;       ///<value>Storage of precipitation by under canopy impervious cover (m/h)</value>                                       
        public double UnderCanImpervRunoffMh;   ///<value>Runoff from under canopy impervious cover (m/h)</value>
        public double NoCanPervEvMh;            ///<value>Evaporation from no canopy pervious cover (m/h)</value>
        public double NoCanPervStMh;            ///<value>Storage of precipitation by no canopy pervious cover (m/h)</value>                                       
        public double NoCanPervInfilMh;         ///<value>Infiltration from no canopy pervious cover (m/h)</value>
        public double NoCanImpervEvMh;          ///<value>Evaporation from no canopy impervious cover (m/h)</value>
        public double NoCanImpervStMh;          ///<value>Storage of precipitation by no canopy impervious cover (m/h)</value>                                       
        public double NoCanImpervRunoffMh;      ///<value>Runoff from no canopy impervious cover (m/h)</value>
        public double PARuEm2s;
        public double PARWm2;
        public double RainInH;
        public double RainIn6H;
        public double RainMh;
        public double RelHum;
        public double SatVPrsKpa;
        public double SnowIn;
        public double SnowM;
        public double SolZenAgl_deg;
        public double TempC;
        public double TempF;
        public double TempK;
        public double CloudCover_Total_tenths;
        public double CloudCover_Translucent_tenths;
        public double VapPrsKpa;
        public double WdDir;
        public double WdSpdKnt;
        public double WdSpdMh;
        public double WdSpdMs;

        public enum WeatherDataFormat { OLD, NEWINTL, NEWUSCAN, NARCCAP };
        public static WeatherDataFormat weaFormat;
        #endregion
        //TrimToTimeSpan function: 
        private static int TrimToTimeSpan(List<SurfaceWeather> list, DateTime startTs, DateTime endTs)
        {
            if (list == null || list.Count == 0) return 0;
            int first = 0; while (first < list.Count && list[first].TimeStamp < startTs) first++;
            int last = list.Count - 1; while (last >= 0 && list[last].TimeStamp > endTs) last--;
            if (first >= list.Count || last < first) { list.Clear(); return 0; }
            if (last + 1 < list.Count) list.RemoveRange(last + 1, list.Count - (last + 1));
            if (first > 0) list.RemoveRange(0, first);
            return list.Count;
        }

        //ProcessSurfaceWeatherData receives variables from CreateOutput.cs
        //MeteorologicalDataInputFile_string, PrecipitationDataInputFile_string, LocationData_Object, LeafAreaIndex_dataBase, VegType, startYear, endYear, WeatherVariables_dataBase
        public static void ProcessSurfaceWeatherData(string MeteorologicalDataInputFile_string, string PrecipitationDataInputFile_string, LocationData LocationData_Object,
            string LeafAreaIndex_dataBase, VEGTYPE VegetationType_class, int startYear, int endYear, string WeatherVariables_dataBase,
            double Height_WeatherStationWindSensor_m, double Height_TreeCanopyTop_m)
        {
            List<SurfaceWeather> rawList = new List<SurfaceWeather>();
            List<SurfaceWeather> intList = new List<SurfaceWeather>();
            List<SurfaceWeather> finList = new List<SurfaceWeather>();
            List<LeafAreaIndex> LeafAreaIndex_list = new List<LeafAreaIndex>();
            int recCnt = 0;

            try
            {
                ReadSurfaceWeatherData(MeteorologicalDataInputFile_string, ref rawList, startYear);
                CheckPressure(rawList, startYear, endYear);  //after this point Altimeter is no longer needed
                AdjustTimeStamp(rawList);

                //firstRaw to lastRaw will check for actual length of record, and avoid assuming a full year
                int firstRaw = -1, lastRaw = -1;
                //For loop through rawList.Count will define actual record length, and avoid erroneously creating a full year
                for (int i = 0; i < rawList.Count; i++)
                {
                    //If conditional checking various variables we trust for presence in RAW data 
                    if (!double.IsNaN(rawList[i].TempF) || !double.IsNaN(rawList[i].DewTempF))
                    {
                        if (firstRaw == -1) firstRaw = i;
                        lastRaw = i;
                    }
                }

                //spanStart and spanEnd created as fallbacks if nothing found
                DateTime spanStart = (firstRaw == -1) ? new DateTime(startYear, 1, 1, 0, 0, 0)
                                                      : rawList[firstRaw].TimeStamp.Date;
                DateTime spanEnd = (lastRaw == -1) ? new DateTime(endYear, 12, 31, 23, 0, 0)
                                                      : rawList[lastRaw].TimeStamp.Date.AddHours(23);

                //CreateHourlyRecords function: intList created from rawList
                recCnt = CreateHourlyRecords(rawList, ref intList, startYear, endYear);
                //TrimToTimeSpan function: Trim calendar to raw observed span BEFORE any filling
                //Note: Avoid creating repeating values from last observation simply to fill a year of record
                recCnt = TrimToTimeSpan(intList, spanStart, spanEnd);

                FillExtrapolate(intList);
                FillInterpolate(intList);
                //ConvertData function: intList converted to proper hour and meteorological units
                ConvertData(intList, LocationData_Object.GMTOffset, recCnt);

                //ExtractValidData function: finList populated using data from intList
                recCnt = ExtractValidData(intList, recCnt, ref finList);
                //recCnt is finList.count
                recCnt = finList.Count;

                //If WeatherPrepConfig.xml has entry for PrecipitationDataCsv then update RainMh variable in finList 
                if (!string.IsNullOrEmpty(PrecipitationDataInputFile_string))
                {
                    //ReplaceRainData function sent the PrecipitationDataInputFile_string file name and finList, which was extracted from  
                    ReplaceRainData(PrecipitationDataInputFile_string, ref finList);
                }

                CalcAirDensity(finList, recCnt);
                CalcSolarZenithAngle(finList, LocationData_Object, recCnt);
                CalcSolarRadiation(finList, LocationData_Object, recCnt);
                LeafAreaIndex.ReadLAIPartialRecords(LeafAreaIndex_dataBase, ref LeafAreaIndex_list, finList[0].TimeStamp, finList[recCnt - 1].TimeStamp);
                CalcET(finList, LeafAreaIndex_list, recCnt, Height_WeatherStationWindSensor_m, Height_TreeCanopyTop_m);
                CalcPrecipInterceptByCanopy(finList, LeafAreaIndex_list, recCnt, VegetationType_class);
                CalcPrecipInterceptByUnderCanopyCover(finList, recCnt);
                CalcPrecipInterceptByNoCanopyCover(finList, recCnt);
                //SurfaceWeather.CreateSurfaceWeatherTable(WeatherVariables_dataBase);
                SurfaceWeather.WriteSurfaceWeatherRecords(WeatherVariables_dataBase, finList, recCnt);
            }
            catch (Exception)
            {
                throw;
            }
        }
        //ReplaceRainData function uses XML-config provided rainFile to replace raw NOAA precipitation record
        static void ReplaceRainData(string rainFile, ref List<SurfaceWeather> list)
        {
            List<SurfaceWeather> rList = new List<SurfaceWeather>();
            int i, j;
            //Call ReadPrecip function by sending Precip.csv file and rList to store the precip data from the csv file
            //Note: rList returned from ReadPrecip with the timestamp and precipitation values
            ReadPrecip(rainFile, ref rList);

            // Search for the rain records that has the same time stamp as that for the first record in NOAA weather file.
            for (j = 0; j < rList.Count; j++)
            {
                if (list[0].TimeStamp == rList[j].TimeStamp) break;
            }
            for (i = 0; i < list.Count; i++)
            {
                if (list[i].TimeStamp == rList[j].TimeStamp)
                {
                    list[i].RainMh = rList[j++].RainMh;
                }
            }
            Console.WriteLine("Precipitation timeseries from PrecipitationDataFile replaced that from SurfaceWeatherDataFile.");
        }
        //ReadPrecip is used to parse Date, Time and Precipitation data from the optional Precip.csv input file for replacing NOAA raw weather values
        //JK 20211029: Renamed from "ReadCsv" to be more precise, and to avoid confusion with the functions handling the new CSV files for LAI & weather inputs
        static void ReadPrecip(string sFile, ref List<SurfaceWeather> rList)
        {
            string line;
            SurfaceWeather data;
            Regex reg;
            Match m;

            // Updated regex to handle scientific notation in precipitation values
            reg = new Regex(@"^\s*(?<dt>\d{8}),\s*(?<hr>\d{1,2}):(?<mn>\d{2}):(?<sc>\d{2}),\s*(?<p>[0-9.]+(?:[eE][+-]?[0-9]+)?)\s*$");

            // sr created as StreamReader type of sFile passed to ReadPrecip
            using (StreamReader sr = new StreamReader(sFile))
            {
                try
                {
                    // Read the header line
                    sr.ReadLine();

                    while ((line = sr.ReadLine()) != null)
                    {
                        // Match the line using the updated regex
                        m = reg.Match(line);

                        // If the match is successful, continue processing
                        if (m.Success)
                        {
                            // Create a new SurfaceWeather object to hold the data
                            data = new SurfaceWeather();

                            // Parse date and time information
                            data.Year = int.Parse(m.Groups["dt"].ToString().Substring(0, 4));
                            data.Month = int.Parse(m.Groups["dt"].ToString().Substring(4, 2));
                            data.Day = int.Parse(m.Groups["dt"].ToString().Substring(6, 2));
                            data.Hour = int.Parse(m.Groups["hr"].ToString());
                            data.Minute = int.Parse(m.Groups["mn"].ToString());

                            // Create a timestamp from the parsed date and time
                            data.TimeStamp = DateTime.Parse($"{data.Month}/{data.Day}/{data.Year} {data.Hour}:{data.Minute}");

                            // Parse the precipitation value, handling both decimal and scientific notation
                            data.RainMh = double.Parse(m.Groups["p"].ToString());

                            // Add the parsed data to the result list
                            rList.Add(data);
                        }
                        else
                        {
                            // Handle the case where the line doesn't match the expected pattern
                            Console.WriteLine($"Warning: Line did not match the expected format: {line}");
                        }
                    }
                }
                catch (Exception ex)
                {
                    // Catch and re-throw any exceptions, or log them
                    Console.WriteLine("Error processing file: " + ex.Message);
                    throw;
                }
            }
        }

        public static WeatherDataFormat CheckWeatherDataFormat(string sFile)
        {
            string line;
            WeatherDataFormat rc = WeatherDataFormat.OLD;

            using (StreamReader sr = new StreamReader(sFile))
            {
                try
                {
                    //read the header line
                    line = sr.ReadLine();
                    //Console.WriteLine("\nMeteorological Input Data Header: " + line + "\n");

                    if (Regex.IsMatch(line, "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB WW WW WW ZZ ZZ ZZ W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01 PCP06 PCP24 PCPXX SD"))
                    {
                        rc = WeatherDataFormat.NEWINTL;
                    }
                    else if (Regex.IsMatch(line, "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB MW MW MW MW AW AW AW AW W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01 PCP06 PCP24 PCPXX SD"))
                    {
                        rc = WeatherDataFormat.NEWUSCAN;
                    }
                    else if (Regex.IsMatch(line, "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB WW WW WW W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01 PCP06 PCP24 PCPXX SD"))
                    {
                        rc = WeatherDataFormat.OLD;
                    }
                    else if (Regex.IsMatch(line, "  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB WW WW WW W TEMP DEWP    SLP   ALT    STP MAX MIN PCP01      NETRAD PCPXX SD"))
                    {
                        rc = WeatherDataFormat.NARCCAP;
                    }
                    else
                    {
                        //throw new InvalidOperationException("\nWarning: WeatherPrepConfig.xml seems to point to a faulty SurfaceWeatherDataFile.");
                        Console.WriteLine("\nWarning: The WeatherPrepConfig.xml file contains a faulty value for SurfaceWeatherDataFile.");
                        Console.WriteLine("Check: " + sFile);
                        Console.WriteLine("Confirm: The header should contain  USAF  WBAN YR--MODAHRMN DIR SPD GUS CLG SKC L M H  VSB ...");
                        Console.WriteLine("Suggestion: Files downloaded from NOAA often need to be processed by ishapp2.exe.");
                    }
                    return rc;
                }
                catch (Exception)
                {
                    throw;
                }
            }
        }
        //ReadSurfaceWeatherData function reads in the raw meteorological data, with USAF, WBAN, and other codes
        public static void ReadSurfaceWeatherData(string sFile, ref List<SurfaceWeather> WeatherVariables_list, int StartYear_input)
        {
            string line;
            SurfaceWeather data;
            Regex reg;

            weaFormat = CheckWeatherDataFormat(sFile);
            using (StreamReader sr = new StreamReader(sFile))
            {
                try
                {
                    switch (weaFormat)
                    {
                        case WeatherDataFormat.NEWINTL:
                            reg = new Regex(@"^(?<usaf>.{6}) (?<wban>.{5}) (?<dt>.{12}) (?<dir>.{3}) (?<spd>.{3}) .{3} (?<clg>.{3}) (?<skc>.{3}) .{30} (?<temp>.{4}) (?<dewp>.{4}) .{6} (?<alt>.{5}) (?<stp>.{6}) .{7} (?<pcp01>.{5}) (?<pcp06>.{5}) .{11} (?<sd>.{2})\s*$");
                            break;
                        case WeatherDataFormat.NEWUSCAN:
                            reg = new Regex(@"^(?<usaf>.{6}) (?<wban>.{5}) (?<dt>.{12}) (?<dir>.{3}) (?<spd>.{3}) .{3} (?<clg>.{3}) (?<skc>.{3}) .{36} (?<temp>.{4}) (?<dewp>.{4}) .{6} (?<alt>.{5}) (?<stp>.{6}) .{7} (?<pcp01>.{5}) (?<pcp06>.{5}) .{11} (?<sd>.{2})\s*$");
                            break;
                        case WeatherDataFormat.OLD:
                            reg = new Regex(@"^(?<usaf>.{6}) (?<wban>.{5}) (?<dt>.{12}) (?<dir>.{3}) (?<spd>.{3}) .{3} (?<clg>.{3}) (?<skc>.{3}) .{21} (?<temp>.{4}) (?<dewp>.{4}) .{6} (?<alt>.{5}) (?<stp>.{6}) .{7} (?<pcp01>.{5}) (?<pcp06>.{5}) .{11} (?<sd>.{2})\s*$");
                            break;
                        case WeatherDataFormat.NARCCAP:
                            reg = new Regex(@"^(?<usaf>.{6}) (?<wban>.{5}) (?<dt>.{12}) (?<dir>.{3}) (?<spd>.{3}) .{3} (?<clg>.{3}) (?<skc>.{3}) .{21} (?<temp>.{4}) (?<dewp>.{4}) .{6} (?<alt>.{5}) (?<stp>.{6}) .{7} (?<pcp>.{5}) (?<netrad>.{11}) .{5} (?<sd>.{2})\s*$");
                            break;
                        default:
                            reg = new Regex(@"^(?<usaf>.{6}) (?<wban>.{5}) (?<dt>.{12}) (?<dir>.{3}) (?<spd>.{3}) .{3} (?<clg>.{3}) (?<skc>.{3}) .{21} (?<temp>.{4}) (?<dewp>.{4}) .{6} (?<alt>.{5}) (?<stp>.{6}) .{7} (?<pcp01>.{5}) (?<pcp06>.{5}) .{11}  (?<sd>.{2})\s*$");
                            break;
                    }
                    Match m;
                    //read the header line
                    sr.ReadLine();

                    //CounterWhileLoop_int initialized to zero for error handling below, when CounterWhileLoop_int == 1
                    int CounterWhileLoop_int = 0;

                    while ((line = sr.ReadLine()) != null)
                    {
                        //CounterWhileLoop_int is advanced, and when CounterWhileLoop_int == 1 it will be used for error handling
                        CounterWhileLoop_int = CounterWhileLoop_int + 1;

                        line = Regex.Replace(line, "([0-9.]+)T", @"$1 ");   // remove trace (T) for PCP01
                        m = reg.Match(line);

                        data = new SurfaceWeather();

                        data.Year = (m.Groups["dt"].ToString().Substring(0, 4) == "****" || m.Groups["dt"].ToString().Substring(0, 4) == "    ") ? (int)NODATA : int.Parse(m.Groups["dt"].ToString().Substring(0, 4));

                        //If CounterWhileLoop_int equals 1 and the StartYear_input is not equal to data.Year then send error message
                        if (CounterWhileLoop_int == 1 & StartYear_input != data.Year)
                        {
                            //Notify user that the input instructions in the WeatherPrepConfigFile and weather data are problematic
                            Console.WriteLine("\nWarning: The WeatherPrepConfig.xml StartYear is " + StartYear_input + " and not the same as the input weather file year of " + data.Year + ".");
                            Console.WriteLine("Exiting: WeatherPrep cannot run when the desired year in WeatherPrepConfig.xml is different from the actual data.");
                            Console.WriteLine("Suggestion: Make sure the StartYear in WeatherPrepConfig.xml matches the year of your input data.");
                            //exit program when this error emerges
                            System.Environment.Exit(1);
                        }

                        data.Month = (m.Groups["dt"].ToString().Substring(4, 2) == "**" || m.Groups["dt"].ToString().Substring(4, 2) == "  ") ? (int)NODATA : int.Parse(m.Groups["dt"].ToString().Substring(4, 2));
                        data.Day = (m.Groups["dt"].ToString().Substring(6, 2) == "**" || m.Groups["dt"].ToString().Substring(6, 2) == "  ") ? (int)NODATA : int.Parse(m.Groups["dt"].ToString().Substring(6, 2));
                        data.Hour = (m.Groups["dt"].ToString().Substring(8, 2) == "**" || m.Groups["dt"].ToString().Substring(8, 2) == "  ") ? (int)NODATA : int.Parse(m.Groups["dt"].ToString().Substring(8, 2));
                        data.Minute = (m.Groups["dt"].ToString().Substring(10, 2) == "**" || m.Groups["dt"].ToString().Substring(10, 2) == "  ") ? (int)NODATA : int.Parse(m.Groups["dt"].ToString().Substring(10, 2));
                        data.TimeStamp = DateTime.Parse(data.Month + "/" +
                                                        data.Day + "/" +
                                                        data.Year + " " +
                                                        data.Hour + ":" +
                                                        data.Minute);
                        data.WdDir = (m.Groups["dir"].ToString() == "***" || m.Groups["dir"].ToString() == "   ") ? NODATA : double.Parse(m.Groups["dir"].ToString());
                        //read in wind spead (mph)
                        data.WdSpdMh = (m.Groups["spd"].ToString() == "***" || m.Groups["spd"].ToString() == "   ") ? NODATA : double.Parse(m.Groups["spd"].ToString());
                        //read in cloud ceiling (lowest opaque layer w/ 5/8 or greater coverage (hundreds ft); 722 = unlimited
                        data.CeilingHeight_hundredths_ft = (m.Groups["clg"].ToString() == "***" || m.Groups["clg"].ToString() == "   ") ? NODATA : double.Parse(m.Groups["clg"].ToString());
                        //read in sky cover (CLR-CLEAR, SCT-SCATTERED-1/8 TO 4/8, BKN-BROKEN-5/8 TO 7/8, OVC-OVERCAST, OBS-OBSCURED, POB-PARTIAL OBSCURATION
                        switch (m.Groups["skc"].ToString())
                        {
                            case "***":
                            case "   ":
                                data.CloudCover_Total_tenths = NODATA;
                                break;
                            case "CLR":
                                data.CloudCover_Total_tenths = 0;
                                break;
                            case "SCT":
                                //Note: CloudCover_Total_tenths = 3.125 is average of SKC = SCT = average(1/8, 2/8, 3/8,4/8) = 2.5/8 (not 3.75)
                                data.CloudCover_Total_tenths = 3.125;
                                break;
                            case "BKN":
                                data.CloudCover_Total_tenths = 7.5;
                                break;
                            case "OVC":
                                data.CloudCover_Total_tenths = 10;
                                break;
                            default:
                                data.CloudCover_Total_tenths = 10;
                                break;
                        }
                        //read in temperature (F)
                        data.TempF = (m.Groups["temp"].ToString() == "****" || m.Groups["temp"].ToString() == "    ") ? NODATA : double.Parse(m.Groups["temp"].ToString());
                        //read in dew point (F)
                        data.DewTempF = (m.Groups["dewp"].ToString() == "****" || m.Groups["dewp"].ToString() == "    ") ? NODATA : double.Parse(m.Groups["dewp"].ToString());
                        //read altimeter setting (inches and hundredths Hg)
                        data.Altimeter = (m.Groups["alt"].ToString() == "*****" || m.Groups["alt"].ToString() == "     ") ? NODATA : double.Parse(m.Groups["alt"].ToString());
                        //read in station atmospheric pressure (mbar)
                        data.PrsMbar = (m.Groups["stp"].ToString() == "******" || m.Groups["stp"].ToString() == "      ") ? NODATA : double.Parse(m.Groups["stp"].ToString());
                        //read in liquid precipitation for 1-hr (inches)
                        data.RainInH = (m.Groups["pcp01"].ToString() == "*****" || m.Groups["pcp01"].ToString() == "     ") ? 0 : double.Parse(m.Groups["pcp01"].ToString());
                        if (weaFormat == WeatherDataFormat.NARCCAP)
                        {
                            //read in net radiation (W/m2)
                            data.Radiation_Net_Wpm2 = (m.Groups["netrad"].ToString() == "***********" || m.Groups["netrad"].ToString() == "           ") ? NODATA : double.Parse(m.Groups["netrad"].ToString());
                        }
                        else
                        {
                            //read in liquid precipitation for 6-hr (inches)
                            data.RainIn6H = (m.Groups["pcp06"].ToString() == "*****" || m.Groups["pcp06"].ToString() == "     ") ? 0 : double.Parse(m.Groups["pcp06"].ToString());
                        }
                        //read in snow depth (inches)
                        data.SnowIn = (m.Groups["sd"].ToString() == "**" || m.Groups["sd"].ToString() == "  ") ? 0 : double.Parse(m.Groups["sd"].ToString());
                        WeatherVariables_list.Add(data);
                    }
                }
                catch (Exception)
                {
                    throw;
                }
            }
        }
        /// <summary>
        /// Check missing data count in pressure and altimeter setting. If the number of pressure data is not enough
        /// but altimeter data, convert altimeter setting to pressure.
        /// </summary>
        /// <remarks>Only weather stations with no more than 48 hours of consecutive missing data can be used for Eco analyses.
        /// Those with more than 48-hour consecutive missing data in both PrsMbar and Altimeter are excluded.
        /// Therefore, there could be a case in which PrsMbar is totally missing and Altimeter doesn't have 48-hour consecutive missing data.
        /// Here only this case is treated; if PrsMbar has more than 48-hour missing data and Altimeter doesn't,
        /// convert Altimeter to PrsMbar.</remarks>
        /// <param name="WeatherVariables_list">Surface weather data list</param>
        /// <param name="startYear">Start year for processing</param>
        /// <param name="endYear">End year for processing</param>
        public static void CheckPressure(List<SurfaceWeather> WeatherVariables_list, int startYear, int endYear)
        {
            if ((CheckPrsMbar(WeatherVariables_list, startYear, endYear) == false) &&
                (CheckAltimeter(WeatherVariables_list, startYear, endYear) == true))
            {
                Alt2Prs(WeatherVariables_list);
            }
        }
        /// <summary>
        /// Check if pressure variable is available or not.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        /// <param name="startYear">Start year for processing</param>
        /// <param name="endYear">End year for processing</param>
        /// <returns>True if pressure data is available, False if not</returns>
        static bool CheckPrsMbar(List<SurfaceWeather> list, int startYear, int endYear)
        {
            int i;
            DateTime lastDateTime;
            DateTime endDt;
            bool result = false;
            TimeSpan ts;

            //set true unless everything is NODATA
            for (i = 0; i < list.Count; i++)
            {
                if (list[i].PrsMbar != NODATA)
                {
                    result = true;
                    break;
                }
            }

            //check if the first measurement is more than 48 hrs apart from January 1 0:00
            lastDateTime = DateTime.Parse("1/1/" + startYear);
            ts = list[0].TimeStamp - lastDateTime;
            if (ts.TotalHours > 48)
            {
                result = false; ;
                return (result);
            }

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].PrsMbar != NODATA)
                {
                    lastDateTime = list[i].TimeStamp;
                }
                else
                {
                    ts = list[i].TimeStamp - lastDateTime;
                    if (ts.TotalHours > 48)
                    {
                        result = false;
                        return (result);
                    }
                }
            }

            //check if the last measurement is more than 48 hrs apart from 12/31/endYear 23:00
            endDt = DateTime.Parse("12/31/" + endYear + " 23:00");
            ts = endDt - lastDateTime;
            if (ts.TotalHours > 48)
            {
                result = false;
                return (result);
            }
            return (result);
        }
        /// <summary>
        /// Check if altimeter setting variable is available or not.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        /// <param name="startYear">Start year for processing</param>
        /// <param name="endYear">End year for processing</param>
        /// <returns>True if altimeter setting data is available, False if not</returns>
        static bool CheckAltimeter(List<SurfaceWeather> list, int startYear, int endYear)
        {
            int i;
            DateTime lastDateTime;
            DateTime endDt;
            bool result = false;
            TimeSpan ts;

            //set true unless everything is NODATA
            for (i = 0; i < list.Count; i++)
            {
                if (list[i].Altimeter != NODATA)
                {
                    result = true;
                    break;
                }
            }

            //check if the first measurement is more than 48 hrs apart from January 1 0:00
            lastDateTime = DateTime.Parse("1/1/" + startYear);
            ts = list[0].TimeStamp - lastDateTime;
            if (ts.TotalHours > 48)
            {
                result = false; ;
                return (result);
            }

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].Altimeter != NODATA)
                {
                    lastDateTime = list[i].TimeStamp;
                }
                else
                {
                    ts = list[i].TimeStamp - lastDateTime;
                    if (ts.TotalHours > 48)
                    {
                        result = false;
                        return (result);
                    }
                }
            }

            //check if the last measurement is more than 48 hrs apart from 12/31/endYear 23:00
            endDt = DateTime.Parse("12/31/" + endYear + " 23:00");
            ts = endDt - lastDateTime;
            if (ts.TotalHours > 48)
            {
                result = false;
                return (result);
            }
            return (result);
        }
        /// <summary>
        /// Convert altimeter setting values to pressure values.
        /// </summary>
        /// <param name="list">Surrace weather list</param>
        static void Alt2Prs(List<SurfaceWeather> list)
        {
            int i;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].Altimeter == NODATA)
                {
                    list[i].PrsMbar = NODATA;
                }
                else
                {
                    list[i].PrsMbar = list[i].Altimeter * 100 * 0.338639;
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of surface weather time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        public static void FillExtrapolate(List<SurfaceWeather> list)
        {
            ExtrapoloteCeiling(list);
            ExtrapolatePrsMbar(list);
            ExtrapolateToCldCov(list);
            ExtrapolateTempF(list);
            ExtrapolateDewTempF(list);
            ExtrapolateWdDir(list);
            ExtrapolateWdSpdMh(list);
            if (weaFormat == WeatherDataFormat.NARCCAP)
            {
                ExtrapolateNetRadWm2(list);
            }
        }
        /// <summary>
        /// Fill missing values at start or end of ceiling height time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapoloteCeiling(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].CeilingHeight_hundredths_ft == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].CeilingHeight_hundredths_ft != NODATA)
                    {
                        dblFill = list[i].CeilingHeight_hundredths_ft;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].CeilingHeight_hundredths_ft = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].CeilingHeight_hundredths_ft == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].CeilingHeight_hundredths_ft != NODATA)
                    {
                        dblFill = list[i].CeilingHeight_hundredths_ft;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].CeilingHeight_hundredths_ft = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of pressure time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolatePrsMbar(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].PrsMbar == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].PrsMbar != NODATA)
                    {
                        dblFill = list[i].PrsMbar;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].PrsMbar = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].PrsMbar == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].PrsMbar != NODATA)
                    {
                        dblFill = list[i].PrsMbar;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].PrsMbar = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of cloud cover time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolateToCldCov(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].CloudCover_Total_tenths == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].CloudCover_Total_tenths != NODATA)
                    {
                        dblFill = list[i].CloudCover_Total_tenths;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].CloudCover_Total_tenths = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].CloudCover_Total_tenths == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].CloudCover_Total_tenths != NODATA)
                    {
                        dblFill = list[i].CloudCover_Total_tenths;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].CloudCover_Total_tenths = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of temperature time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolateTempF(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].TempF == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].TempF != NODATA)
                    {
                        dblFill = list[i].TempF;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].TempF = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].TempF == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].TempF != NODATA)
                    {
                        dblFill = list[i].TempF;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].TempF = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of dew point temperature time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolateDewTempF(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].DewTempF == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].DewTempF != NODATA)
                    {
                        dblFill = list[i].DewTempF;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].DewTempF = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].DewTempF == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].DewTempF != NODATA)
                    {
                        dblFill = list[i].DewTempF;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].DewTempF = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of wind direction time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolateWdDir(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].WdDir == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].WdDir != NODATA)
                    {
                        dblFill = list[i].WdDir;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].WdDir = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].WdDir == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].WdDir != NODATA)
                    {
                        dblFill = list[i].WdDir;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].WdDir = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of wind speed time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolateWdSpdMh(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].WdSpdMh == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].WdSpdMh != NODATA)
                    {
                        dblFill = list[i].WdSpdMh;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].WdSpdMh = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].WdSpdMh == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].WdSpdMh != NODATA)
                    {
                        dblFill = list[i].WdSpdMh;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].WdSpdMh = dblFill;
                    }
                }
            }
        }
        /// <summary>
        /// Fill missing values at start or end of net radiation time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void ExtrapolateNetRadWm2(List<SurfaceWeather> list)
        {
            //This process fills in the missing values in the beginning of a list
            int i, j;
            double dblFill = 0;

            if (list[0].Radiation_Net_Wpm2 == NODATA)
            {
                //looks for the value that is not NODATA
                i = 1;
                while (true)
                {
                    if (list[i].Radiation_Net_Wpm2 != NODATA)
                    {
                        dblFill = list[i].Radiation_Net_Wpm2;
                        break;
                    }
                    i++;
                    if (i == list.Count)
                    {
                        break;
                    }
                }
                if (i != list.Count)
                {
                    for (j = i - 1; j >= 0; j--)
                    {
                        list[j].Radiation_Net_Wpm2 = dblFill;
                    }
                }
            }
            if (list[list.Count - 1].Radiation_Net_Wpm2 == NODATA)
            {
                //looks for the value that is not NODATA
                i = list.Count - 1;
                while (true)
                {
                    if (list[i].Radiation_Net_Wpm2 != NODATA)
                    {
                        dblFill = list[i].Radiation_Net_Wpm2;
                        break;
                    }
                    i--;
                    if (i == 0)
                    {
                        break;
                    }
                }
                if (i != 0)
                {
                    for (j = i + 1; j < list.Count; j++)
                    {
                        list[j].Radiation_Net_Wpm2 = dblFill;
                    }
                }
            }
        }

        /// <summary>
        /// Fill missing values in the middle of surface weather time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        public static void FillInterpolate(List<SurfaceWeather> list)
        {
            InterpolatePrsMbar(list);
            InterpolateTempF(list);
            InterpolateDewTempF(list);
            InterpolateWdDir(list);
            InterpolateWdSpdMh(list);
            InterpolateCeiling(list);
            InterpolateToCldCov(list);
            if (weaFormat == WeatherDataFormat.NARCCAP)
            {
                InterpolateNetRadWm2(list);
            }
        }
        /// <summary>
        /// Fill missing values in the middle of pressure time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolatePrsMbar(List<SurfaceWeather> list)
        {
            int i, j, k;
            double valDif, stpInc;
            int timeDif;
            int bfIdx;      //index for the record before the records with missing values
            int afIdx;      //index for the record after the records with missing values
            TimeSpan ts;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].PrsMbar == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;
                    while (true)
                    {
                        if (list[j].PrsMbar != NODATA)
                        {
                            afIdx = j;
                            break;
                        }
                        else
                        {
                            j++;
                        }
                    }

                    //taking the before to after differences of time and values
                    ts = list[afIdx].TimeStamp - list[bfIdx].TimeStamp;
                    timeDif = (int)ts.TotalMinutes;
                    valDif = list[afIdx].PrsMbar - list[bfIdx].PrsMbar;
                    //taking the increment value of the difference
                    stpInc = valDif / timeDif;
                    //looping through the NODATA value
                    for (k = bfIdx + 1; k < afIdx; k++)
                    {
                        ts = list[k].TimeStamp - list[bfIdx].TimeStamp;
                        list[k].PrsMbar = list[bfIdx].PrsMbar + stpInc * ts.TotalMinutes;
                    }
                    i = j;
                }
            }
        }
        /// <summary>
        /// Fill missing values in the middle of temperature time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateTempF(List<SurfaceWeather> list)
        {
            int i, j, k;
            double valDif, stpInc;
            int timeDif;
            int bfIdx;      //index for the record before the records with missing values
            int afIdx;      //index for the record after the records with missing values
            TimeSpan ts;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].TempF == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;
                    while (true)
                    {
                        if (list[j].TempF != NODATA)
                        {
                            afIdx = j;
                            break;
                        }
                        else
                        {
                            j++;
                        }
                    }

                    //taking the before to after differences of time and values
                    ts = list[afIdx].TimeStamp - list[bfIdx].TimeStamp;
                    timeDif = (int)ts.TotalMinutes;
                    valDif = list[afIdx].TempF - list[bfIdx].TempF;
                    //taking the increment value of the difference
                    stpInc = valDif / timeDif;
                    //looping through the NODATA value
                    for (k = bfIdx + 1; k < afIdx; k++)
                    {
                        ts = list[k].TimeStamp - list[bfIdx].TimeStamp;
                        list[k].TempF = list[bfIdx].TempF + stpInc * ts.TotalMinutes;
                    }
                    i = j;
                }
            }
        }
        /// <summary>
        /// Fill missing values in the middle of dew point temperature time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateDewTempF(List<SurfaceWeather> list)
        {
            int i, j, k;
            double valDif, stpInc;
            int timeDif;
            int bfIdx;      //index for the record before the records with missing values
            int afIdx;      //index for the record after the records with missing values
            TimeSpan ts;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].DewTempF == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;
                    while (true)
                    {
                        if (list[j].DewTempF != NODATA)
                        {
                            afIdx = j;
                            break;
                        }
                        else
                        {
                            j++;
                        }
                    }

                    //taking the before to after differences of time and values
                    ts = list[afIdx].TimeStamp - list[bfIdx].TimeStamp;
                    timeDif = (int)ts.TotalMinutes;
                    valDif = list[afIdx].DewTempF - list[bfIdx].DewTempF;
                    //taking the increment value of the difference
                    stpInc = valDif / timeDif;
                    //looping through the NODATA value
                    for (k = bfIdx + 1; k < afIdx; k++)
                    {
                        ts = list[k].TimeStamp - list[bfIdx].TimeStamp;
                        list[k].DewTempF = list[bfIdx].DewTempF + stpInc * ts.TotalMinutes;
                    }
                    i = j;
                }
            }
        }
        /// <summary>
        /// Fill missing values in the middle of wind direction time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateWdDir(List<SurfaceWeather> list)
        {
            int i, j, k;
            double valDif, stpInc;
            int timeDif;
            int bfIdx;      //index for the record before the records with missing values
            int afIdx;      //index for the record after the records with missing values
            TimeSpan ts;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].WdDir == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;
                    while (true)
                    {
                        if (list[j].WdDir != NODATA)
                        {
                            afIdx = j;
                            break;
                        }
                        else
                        {
                            j++;
                        }
                    }

                    //taking the before to after differences of time and values
                    ts = list[afIdx].TimeStamp - list[bfIdx].TimeStamp;
                    timeDif = (int)ts.TotalMinutes;
                    valDif = list[afIdx].WdDir - list[bfIdx].WdDir;
                    //taking the increment value of the difference
                    stpInc = valDif / timeDif;
                    //looping through the NODATA value
                    for (k = bfIdx + 1; k < afIdx; k++)
                    {
                        ts = list[k].TimeStamp - list[bfIdx].TimeStamp;
                        list[k].WdDir = list[bfIdx].WdDir + stpInc * ts.TotalMinutes;
                    }
                    i = j;
                }
            }
        }
        /// <summary>
        /// Fill missing values in the middle of wind speed time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateWdSpdMh(List<SurfaceWeather> list)
        {
            int i, j, k;
            double valDif, stpInc;
            int timeDif;
            int bfIdx;      //index for the record before the records with missing values
            int afIdx;      //index for the record after the records with missing values
            TimeSpan ts;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].WdSpdMh == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;
                    while (true)
                    {
                        if (list[j].WdSpdMh != NODATA)
                        {
                            afIdx = j;
                            break;
                        }
                        else
                        {
                            j++;
                        }
                    }

                    //taking the before to after differences of time and values
                    ts = list[afIdx].TimeStamp - list[bfIdx].TimeStamp;
                    timeDif = (int)ts.TotalMinutes;
                    valDif = list[afIdx].WdSpdMh - list[bfIdx].WdSpdMh;
                    //taking the increment value of the difference
                    stpInc = valDif / timeDif;
                    //looping through the NODATA value
                    for (k = bfIdx + 1; k < afIdx; k++)
                    {
                        ts = list[k].TimeStamp - list[bfIdx].TimeStamp;
                        list[k].WdSpdMh = list[bfIdx].WdSpdMh + stpInc * ts.TotalMinutes;
                    }
                    i = j;
                }
            }
        }
        /// <summary>
        /// Fill missing values in the middle of ceiling height time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateCeiling(List<SurfaceWeather> list)
        {
            int i, j, k;
            int bfIdx;      // index for the record before the records with missing values
            int afIdx;      // index for the record after the records with missing values
            bool allNODATA = true;  // Flag to check if all values are NODATA

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].CeilingHeight_hundredths_ft == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;

                    // Start looking for the next valid cloud cover value
                    while (true)
                    {
                        //cte 202409 Refactored to find if all values are NODATA, and break from while true 
                        //If j > list.Count then it has incremented beyond NODATA values with no real data
                        if (j >= list.Count)
                        {
                            break;  // Prevent infinite loop if all subsequent values are NODATA
                        }

                        // If a valid cloud cover is found, mark its index and unset the allNODATA flag
                        if (list[j].CeilingHeight_hundredths_ft != NODATA)
                        {
                            afIdx = j;
                            allNODATA = false;  // At least one valid value was found
                            break;
                        }
                        else
                        {
                            j++;  // Continue searching for a valid cloud cover value
                        }
                    }

                    //Refactored to enter only if some values are not NODATA
                    //If j < list.Count then it has real values 
                    if (j < list.Count)
                    {
                        afIdx = j;
                        // Loop through the NODATA values and fill them with the found valid cloud cover value
                        for (k = bfIdx + 1; k < afIdx; k++)
                        {
                            list[k].CeilingHeight_hundredths_ft = list[afIdx].CeilingHeight_hundredths_ft;
                        }

                        // Skip ahead in the list to the next element after the valid cloud cover value
                        i = j;
                    }
                }
                //Refactored to define allNODATA as false 
                else
                {
                    // If we find at least one valid value, set the flag to false
                    allNODATA = false;
                }
            }

            //Refactored to assign an arbitrary value of 100 if all CeilingHeight_hundredths_ft values are NODATA
            if (allNODATA)
            {
                Console.WriteLine("Warning: SurfaceWeatherDataFile only has NODATA values for CLG. Converting all values to 100.");
                Console.WriteLine("Suggestion: Replace SurfaceWeatherDataFile with one containing observations for CLG.");
                for (i = 0; i < list.Count; i++)
                {
                    list[i].CeilingHeight_hundredths_ft = 100;
                }
            }
        }

        /// <summary>
        /// Fill missing values in the middle of total cloud cover time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateToCldCov(List<SurfaceWeather> list)
        {
            int i, j, k;
            int bfIdx;      // index for the record before the records with missing values
            int afIdx;      // index for the record after the records with missing values
            bool allNODATA = true;  // Flag to check if all values are NODATA

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].CloudCover_Total_tenths == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;

                    // Start looking for the next valid cloud cover value
                    while (true)
                    {
                        //Refactored to find if all values are NODATA, and break from while true 
                        //If j > list.Count then it has incremented beyond NODATA values with no real data
                        if (j >= list.Count)
                        {
                            break;  // Prevent infinite loop if all subsequent values are NODATA
                        }

                        // If a valid cloud cover is found, mark its index and unset the allNODATA flag
                        if (list[j].CloudCover_Total_tenths != NODATA)
                        {
                            afIdx = j;
                            allNODATA = false;  // At least one valid value was found
                            break;
                        }
                        else
                        {
                            j++;  // Continue searching for a valid cloud cover value
                        }
                    }

                    //Refactored to enter only if some values are not NODATA
                    //If j < list.Count then it has real values 
                    if (j < list.Count)
                    {
                        afIdx = j;
                        // Loop through the NODATA values and fill them with the found valid cloud cover value
                        for (k = bfIdx + 1; k < afIdx; k++)
                        {
                            list[k].CloudCover_Total_tenths = list[afIdx].CloudCover_Total_tenths;
                        }

                        // Skip ahead in the list to the next element after the valid cloud cover value
                        i = j;
                    }
                }
                //cte 202409 Refactored to define allNODATA as false 
                else
                {
                    // If we find at least one valid value, set the flag to false
                    allNODATA = false;
                }
            }

            //Refactored to assign an arbitrary value of 3.75 if all CloudCover_Total_tenths values are NODATA
            if (allNODATA)
            {
                Console.WriteLine("Warning: SurfaceWeatherDataFile only has NODATA values for SKC. Converting all values to 3.75.");
                Console.WriteLine("Suggestion: Replace SurfaceWeatherDataFile with one containing observations for CLG.");
                for (i = 0; i < list.Count; i++)
                {
                    list[i].CloudCover_Total_tenths = 3.75;
                }
            }
        }
        /// Fill missing values in the middle of net radiation time series data if exists.
        /// </summary>
        /// <param name="list">Surface weather list</param>
        static void InterpolateNetRadWm2(List<SurfaceWeather> list)
        {
            int i, j, k;
            double valDif, stpInc;
            int timeDif;
            int bfIdx;      //index for the record before the records with missing values
            int afIdx;      //index for the record after the records with missing values
            TimeSpan ts;

            for (i = 0; i < list.Count; i++)
            {
                if (list[i].Radiation_Net_Wpm2 == NODATA)
                {
                    bfIdx = i - 1;
                    j = i + 1;
                    while (true)
                    {
                        if (list[j].Radiation_Net_Wpm2 != NODATA)
                        {
                            afIdx = j;
                            break;
                        }
                        else
                        {
                            j++;
                        }
                    }

                    //taking the before to after differences of time and values
                    ts = list[afIdx].TimeStamp - list[bfIdx].TimeStamp;
                    timeDif = (int)ts.TotalMinutes;
                    valDif = list[afIdx].Radiation_Net_Wpm2 - list[bfIdx].Radiation_Net_Wpm2;
                    //taking the increment value of the difference
                    stpInc = valDif / timeDif;
                    //looping through the NODATA value
                    for (k = bfIdx + 1; k < afIdx; k++)
                    {
                        ts = list[k].TimeStamp - list[bfIdx].TimeStamp;
                        list[k].Radiation_Net_Wpm2 = list[bfIdx].Radiation_Net_Wpm2 + stpInc * ts.TotalMinutes;
                    }
                    i = j;
                }
            }
        }
        /// <summary>
        /// Set timestamp to the nearest top-of-hour.
        /// </summary>
        /// <remarks> Below is a typical timestamp observed in ISH data. Data are typically recorded
        /// one per hour but the timestamp is not on the top-of-hour. During rain events, precipitation (PCP01)
        /// and other data tend to be recorded more often. This function adjusts the timestamp to the
        /// nearest next top-of-hour.
        /// 
        /// YR--MODAHRMN 
        /// 200501050054 -> 0100
        /// 200501050154 -> 0200
        /// 200501050254 -> 0300
        /// 200501050339 -> 0400
        /// 200501050354 -> 0400
        /// 200501050407 -> 0500
        /// 200501050454 -> 0500
        /// 200501050515 -> 0600
        /// 200501050536 -> 0600
        /// 200501050554 -> 0600
        /// 200501050600 -> 0600
        /// 200501050605 -> 0700
        /// 200501050623 -> 0700
        /// 200501050654 -> 0700
        /// 200501050700 -> 0700
        /// 
        /// Among those records with the same time stamp (e.g. 0600), variable values found
        /// the last will be used in the hourly records. For PCP01, the largest value will 
        /// be used.  </remarks>
        /// 
        /// <param name="rawList">Surface weather list</param>
        public static void AdjustTimeStamp(List<SurfaceWeather> rawList)
        {
            int i;

            for (i = 0; i < rawList.Count; i++)
            {
                if (rawList[i].TimeStamp.Minute != 0)
                {
                    rawList[i].TimeStamp = rawList[i].TimeStamp.AddMinutes(60 - rawList[i].TimeStamp.Minute);
                }
            }
        }
        /// <summary>
        /// With timestamps adjusted, rawList may have multiple records at the same hour.
        /// This method creates hourly time series data by extracting variable values
        /// found the last within the same timestamp records. For rain, the largest value
        /// within the same timestamp records will be extracted.
        /// </summary>
        /// <param name="rawList">List holding original surface weather data</param>
        /// <param name="hrList">List holding surface weather hourly data</param>
        /// <param name="startYear">Start year for processing</param>
        /// <param name="endYear">End year for processing</param>
        /// <returns>Record count in hrList list</returns>
        /// <summary>
        public static int CreateHourlyRecords(List<SurfaceWeather> rawList, ref List<SurfaceWeather> hrList,
                                                int startYear, int endYear)
        {
            DateTime startDt;
            int j;      //index for the raw data lists
            int i;      //index for the hourly data lists
            int recCnt = 0;
            SurfaceWeather data;

            for (i = startYear; i <= endYear; i++)
            {
                if (DateTime.IsLeapYear(i))
                {
                    recCnt += 24 * 366;
                }
                else
                {
                    recCnt += 24 * 365;
                }
            }
            startDt = DateTime.Parse("1/1/" + startYear + " 0:00");
            hrList = new List<SurfaceWeather>();

            // set the blank data with only timestamp throughout the year(s)
            //Loop through all records to set TimeStamp to incremental hours after start date
            //Loop through all records to set initial values of variables = NoData or = 0
            for (i = 0; i < recCnt; i++)
            {
                data = new SurfaceWeather();
                data.TimeStamp = startDt.AddHours(i);
                data.WdDir = data.WdSpdMh = data.CeilingHeight_hundredths_ft = data.CloudCover_Total_tenths =
                data.TempF = data.DewTempF = data.PrsMbar = data.Radiation_Net_Wpm2 = NODATA;
                data.RainInH = data.SnowIn = 0;
                hrList.Add(data);
            }
            try
            {
                j = 0;
                //Loop through all records to set certain variables to values != NoData if the raw data had non-zero values 
                //Loop through all records to set certain variables to values > 0 if the raw data had non-zero values 
                for (i = 0; i < recCnt; i++)
                {
                    //If the new and raw lsit TimeStamp are equal do the next comparison of values != NoData or >= 0
                    if (hrList[i].TimeStamp == rawList[j].TimeStamp)
                    {
                        do
                        {
                            if (rawList[j].RainInH >= hrList[i].RainInH)
                            {
                                hrList[i].RainInH = rawList[j].RainInH;
                            }
                            if (rawList[j].SnowIn >= hrList[i].SnowIn)
                            {
                                hrList[i].SnowIn = rawList[j].SnowIn;
                            }
                            if (rawList[j].WdDir != NODATA)
                            {
                                hrList[i].WdDir = rawList[j].WdDir;
                            }
                            if (rawList[j].WdSpdMh != NODATA)
                            {
                                hrList[i].WdSpdMh = rawList[j].WdSpdMh;
                            }
                            if (rawList[j].CeilingHeight_hundredths_ft != NODATA)
                            {
                                hrList[i].CeilingHeight_hundredths_ft = rawList[j].CeilingHeight_hundredths_ft;
                            }
                            if (rawList[j].CloudCover_Total_tenths != NODATA)
                            {
                                hrList[i].CloudCover_Total_tenths = rawList[j].CloudCover_Total_tenths;
                            }
                            if (rawList[j].TempF != NODATA)
                            {
                                hrList[i].TempF = rawList[j].TempF;
                            }
                            if (rawList[j].DewTempF != NODATA)
                            {
                                hrList[i].DewTempF = rawList[j].DewTempF;
                            }
                            if (rawList[j].PrsMbar != NODATA)
                            {
                                hrList[i].PrsMbar = rawList[j].PrsMbar;
                            }
                            if (weaFormat == WeatherDataFormat.NARCCAP)
                            {
                                if (rawList[j].Radiation_Net_Wpm2 != NODATA)
                                {
                                    hrList[i].Radiation_Net_Wpm2 = rawList[j].Radiation_Net_Wpm2;
                                }
                            }
                            j++;
                            if (j == rawList.Count)
                            {
                                return (recCnt);
                            }
                        } while (hrList[i].TimeStamp == rawList[j].TimeStamp);
                    }
                }
                return (recCnt);
            }
            catch (Exception)
            {
                return (recCnt);
                throw;
            }
        }
        /// <summary>
        /// Convert unit of surface weather to that required by i-Tree Eco.
        /// </summary>
        /// <param name="intList">List holding top of hour surface weather data</param>
        /// <param name="timeZone">Hour offset to GMT</param>
        /// <param name="recCnt">Record count in intList</param>
        public static void ConvertData(List<SurfaceWeather> intList, double timeZone, int recCnt)
        {
            DateTime dateTimeLST;
            int i;

            try
            {
                for (i = 0; i < recCnt; i++)
                {
                    dateTimeLST = intList[i].TimeStamp.AddHours((int)timeZone);    //GMT to LST

                    intList[i].TimeStamp = dateTimeLST;
                    intList[i].Year = dateTimeLST.Year;
                    intList[i].Month = dateTimeLST.Month;
                    intList[i].Day = dateTimeLST.Day;
                    intList[i].Hour = dateTimeLST.Hour;
                    intList[i].Jday = dateTimeLST.DayOfYear;
                    intList[i].PrsIn = intList[i].PrsMbar / 33.8639 * 100;       //mBar to hundredths of inches
                    intList[i].PrsKpa = intList[i].PrsMbar / 10;                 //mBar to kPa
                    intList[i].RainMh = intList[i].RainInH * 0.0254;             //inches/hour to meters/hour
                    intList[i].SnowM = intList[i].SnowIn * 0.0254;               //inches to meters
                    intList[i].TempC = (intList[i].TempF - 32) * 5 / 9;        //degrees Fahrenheit to degrees Celsius
                    intList[i].TempK = intList[i].TempC + TEMP_CK;               //degrees Celsius to Kelvin
                    intList[i].DewTempC = (intList[i].DewTempF - 32) * 5 / 9;  //degrees Fahrenheit to degrees Celsius
                    intList[i].WdSpdKnt = Math.Round(intList[i].WdSpdMh / 1.151);      //Miles/hour to knot
                    intList[i].WdSpdMs = intList[i].WdSpdMh / 2.237;             //Miles/hour to meters/sec
                }
            }
            catch (Exception)
            {
                throw;
            }
        }
        /// <summary>
         /// Extract data for the analysis year from intermediate top-of-hour data (intList),
         /// then clamp the tail to the last hour containing actual observations.
         /// </summary>
        public static int ExtractValidData(List<SurfaceWeather> intList, int recCnt, ref List<SurfaceWeather> finList)
        {
            if (intList == null || intList.Count == 0)
            {
                finList = new List<SurfaceWeather>();
                return 0;
            }

            int startIdx = 0;
            int endIdx = recCnt - 1;

            // 1) Honor the original intent: find a whole-day span (first 00:00 → last 23:00)
            if (intList[0].Hour != 0 || intList[recCnt - 1].Hour != 23)
            {
                // first 00:00
                for (int i = 0; i < recCnt; i++)
                {
                    if (intList[i].Hour == 0) { startIdx = i; break; }
                }
                // last 23:00
                for (int i = recCnt - 1; i >= 0; i--)
                {
                    if (intList[i].Hour == 23) { endIdx = i; break; }
                }
            }

            // 2) NEW: clamp the end to the last hour that contains any actual observation
            //    Replace SurfaceWeather.NODATA with your project’s sentinel if different.
            double NODATA = SurfaceWeather.NODATA;

            // Walk backward from endIdx to find the last hour with an observed signal.
            int lastDataIdx = startIdx;
            for (int i = endIdx; i >= startIdx; i--)
            {
                if (HasAnyObservation(intList[i], NODATA))
                {
                    lastDataIdx = i;
                    break;
                }
            }
            endIdx = Math.Max(lastDataIdx, startIdx);

            // 3) Materialize finList
            int newCnt = endIdx - startIdx + 1;
            finList = new List<SurfaceWeather>(capacity: newCnt);
            for (int i = startIdx; i <= endIdx; i++)
            {
                // copy item-by-item (structurally equivalent to your original loop)
                var src = intList[i];
                var dst = new SurfaceWeather
                {
                    TimeStamp = src.TimeStamp,
                    Jday = src.Jday,
                    Year = src.Year,
                    Month = src.Month,
                    Day = src.Day,
                    Hour = src.Hour,

                    CeilingHeight_hundredths_ft = src.CeilingHeight_hundredths_ft,
                    DewTempF = src.DewTempF,
                    RainInH = src.RainInH,
                    PrsMbar = src.PrsMbar,
                    SnowIn = src.SnowIn,
                    TempF = src.TempF,
                    CloudCover_Total_tenths = src.CloudCover_Total_tenths,
                    WdDir = src.WdDir,
                    WdSpdMh = src.WdSpdMh,

                    PrsIn = src.PrsIn,
                    PrsKpa = src.PrsKpa,
                    RainMh = src.RainMh,
                    SnowM = src.SnowM,
                    TempC = src.TempC,
                    TempK = src.TempK,
                    DewTempC = src.DewTempC,
                    WdSpdKnt = src.WdSpdKnt,
                    WdSpdMs = src.WdSpdMs
                };

                if (weaFormat == WeatherDataFormat.NARCCAP)
                {
                    dst.Radiation_Net_Wpm2 = src.Radiation_Net_Wpm2;
                }

                finList.Add(dst);
            }

            return newCnt;
        }

        /// <summary>
        /// Returns true if this hourly record contains any real observation signal.
        /// Adjust fields if your project uses different members.
        /// </summary>
        private static bool HasAnyObservation(SurfaceWeather dataFile, double NODATA)
        {
            // Treat precipitation/snow > 0 as signal.
            // Treat typical met drivers != NODATA as signal (values can be zero legitimately).
            return
                dataFile.TempF != NODATA ||
                dataFile.DewTempF != NODATA ||
                dataFile.PrsMbar != NODATA ||
                dataFile.Radiation_Net_Wpm2 != NODATA ||
                dataFile.WdSpdMh != NODATA ||
                dataFile.WdDir != NODATA ||
                dataFile.CloudCover_Total_tenths != NODATA ||
                dataFile.RainInH > 0.0 ||
                dataFile.SnowIn > 0.0;
        }

        /// <summary>
        /// Calculate air density.
        /// </summary>
        /// <param name="WeatherVariables_list">List holding hourly surface weather data</param>
        /// <param name="recCnt">Record count in WeatherVariables_list</param>
        public static void CalcAirDensity(List<SurfaceWeather> WeatherVariables_list, int recCnt)
        {
            int i;

            for (i = 0; i < recCnt; i++)
            {
                //DensityAir_kg_p_m3 (kg/m3) is density of air from Eq 4.2.4 from Shuttleworth (1993), w correction in conversion to C from K
                //Note: Given equation uses TempK, the 273.15 in the denominator is removed
                //Note: Eq 4.24 used incorrect denominator of 275 rather than 273.15 to convert from C to K; see Chin (2021) Eq 13.51
                //Note: Chin Eq 13.51 which unfortunately uses 3.45 in place of the correct 3.486.
                //Note: This tested well against values of air density from the EngineeringToolbox.com for air temperature from 0 to 50 C at standard atmospheric pressure
                WeatherVariables_list[i].AirDens_kg_p_m3 = 3.486 * (WeatherVariables_list[i].PrsKpa / WeatherVariables_list[i].TempK);
            }
        }
        /// <summary>
        /// Calculate solar zenith angle.
        /// </summary>
        /// <param name="WeatherVariables_list">List holding hourly surface weather data</param>
        /// <param name="LocationData_Object">Location data</param>
        /// <param name="recCnt">Record count in WeatherVariables_list</param>
        public static void CalcSolarZenithAngle(List<SurfaceWeather> WeatherVariables_list, LocationData LocationData_Object, int recCnt)
        {
            double dayAngle;  //day angle
            double solDecl; //solar declination
            double eqnTime;   //equation of time
            double stdMer;    //standard meridian
            double lat;       //local latitude in radians
            double trSolTime; //true solar time in hours
            double hourAngle; //hour angle
            double cosZ;      //cosine of the zenith angle
            double exAtmZenAgl_deg;  //exoatmospheric solar zenith angle
            double exAtmElvAgl_deg; //exoatmospheric solar elevation angle
            double refCorr_deg;   //refraction correction
            int i;

            for (i = 0; i < recCnt; i++)
            {
                //dayAngle (radians)
                dayAngle = Deg2Rad(360 * (WeatherVariables_list[i].Jday - 1) / 365.0);
                //eqnTime (decimal minutes), equation of time
                eqnTime = (0.000075 + 0.001868 * Math.Cos(dayAngle) - 0.032077 * Math.Sin(dayAngle)
                        - 0.014615 * Math.Cos(2 * dayAngle) - 0.040849 * Math.Sin(2 * dayAngle)) * 229.18;
                //solDecl (radians), solar declination angle
                solDecl = (0.006918 - 0.399912 * Math.Cos(dayAngle) + 0.070257 * Math.Sin(dayAngle)
                            - 0.006758 * Math.Cos(2 * dayAngle) + 0.000907 * Math.Sin(2 * dayAngle)
                            - 0.002697 * Math.Cos(3 * dayAngle) + 0.00148 * Math.Sin(3 * dayAngle));
                //WeatherVariables_list stores solDecl (radians)
                WeatherVariables_list[i].DecAngleRad = solDecl;
                //To handle both Northern/Southern hemisphere, accept negative latitude (for Southern)
                //lat (radians), latitude
                lat = Deg2Rad(LocationData_Object.Latitude);
                //WeatherVariables_list stores latitude
                WeatherVariables_list[i].LatRad = lat;
                //stdMer (degrees) takes 15 deg for each GMT offset hour
                stdMer = 15.0 * LocationData_Object.GMTOffset;
                //trSolTime (hour with decimal minutes), true solar time
                trSolTime = WeatherVariables_list[i].Hour + (4 * (Math.Abs(stdMer) - Math.Abs(LocationData_Object.Longitude)) + eqnTime) / 60.0;
                //hourAngle (degrees)
                hourAngle = 15.0 * trSolTime - 180;
                //hourAngle (radians)
                hourAngle = Deg2Rad(hourAngle);
                //WeatherVariables_list stores hourAngle
                WeatherVariables_list[i].HourAngleRad = hourAngle;
                //cosZ (radians) cosine Zenith angle
                cosZ = Math.Sin(solDecl) * Math.Sin(lat) + Math.Cos(solDecl) * Math.Cos(lat) * Math.Cos(hourAngle);
                if (cosZ > 1)
                {
                    cosZ = 1.0;
                }
                else if (cosZ < -1)
                {
                    cosZ = -1.0;
                }

                //calculate atmosphere correction for the solar zenith angle
                exAtmZenAgl_deg = Rad2Deg(Math.Acos(cosZ));    //in degrees
                exAtmElvAgl_deg = 90.0 - exAtmZenAgl_deg;           //in degrees

                refCorr_deg = 0.0;
                if (-1 <= exAtmElvAgl_deg && exAtmElvAgl_deg < 15)
                {
                    refCorr_deg = ABS_PRS / ABS_TEMP
                            * (0.1594 + 0.0196 * exAtmElvAgl_deg + 0.00002 * Math.Pow(exAtmElvAgl_deg, 2))
                            / (1 + 0.505 * exAtmElvAgl_deg + 0.0845 * Math.Pow(exAtmElvAgl_deg, 2));
                }
                else if (15 <= exAtmElvAgl_deg && exAtmElvAgl_deg < 90)
                {
                    refCorr_deg = 0.00452 * ABS_PRS / ABS_TEMP / Math.Tan(Deg2Rad(exAtmElvAgl_deg));
                }
                WeatherVariables_list[i].SolZenAgl_deg = exAtmZenAgl_deg - refCorr_deg;
            }
        }
        /// <summary>
        /// Convert unit of angle from degrees to radians.
        /// </summary>
        /// <param name="angle">Angle in degrees</param>
        /// <returns>Angle in radians</returns>
        static double Deg2Rad(double angle)
        {
            return Math.PI * angle / 180.0;
        }
        /// <summary>
        /// Convert unit of angle from radians to degrees.
        /// </summary>
        /// <param name="angle">Angle in radians</param>
        /// <returns>Angle in degrees</returns>
        static double Rad2Deg(double angle)
        {
            return 180.0 * angle / Math.PI;
        }
        /// <summary>
        /// Calculate solar radiation.
        /// </summary>
        /// <param name="WeatherVariables_list">List holding hourly surface weather data</param>
        /// <param name="LocationData_Object">Location data</param>
        /// <param name="recCnt">Record count in WeatherVariables_list</param>
        public static void CalcSolarRadiation(List<SurfaceWeather> WeatherVariables_list, LocationData LocationData_Object, int recCnt)
        {
            double ETR;       //ETR:   extraterrestrial radiation
            double airMass_relativeOpticalPressueCorrected_kg_p_m2; //M:     pressure-corrected relative optical air mass
            double opqn;      //OPQN:  adjusted opaque cloud cover for diffuse radiation calculation
            double opqd;      //OPQD:  adjusted opaque cloud cover for direct radiation calculation
            double Kn;        //Kn:    transmission value of extraterrestrial direct normal radiation
            double Tr;        //TR:    Rayleigh scattering transmittance
            double Toz;       //TO:    ozone absorption transmittance
            double Tum;       //TUM:   uniformly mixed gas transmittance
            double Ta;        //TA:    aerosol absorption and scattering transmittance
            double Kd;
            int i;

            for (i = 0; i < recCnt; i++)
            {
                CalcExtraterrestrialRadiation(WeatherVariables_list[i].Jday, WeatherVariables_list[i].SolZenAgl_deg, out ETR);

                CalcAirMass(WeatherVariables_list[i].SolZenAgl_deg, ref WeatherVariables_list[i].airMass_relativeOptical_kg_p_m2, WeatherVariables_list[i].PrsKpa, out airMass_relativeOpticalPressueCorrected_kg_p_m2);
                CalcCloudCover(WeatherVariables_list[i].airMass_relativeOptical_kg_p_m2, WeatherVariables_list[i].CeilingHeight_hundredths_ft, WeatherVariables_list[i].CloudCover_Total_tenths, ref WeatherVariables_list[i].CloudCover_Opaque_tenths, ref WeatherVariables_list[i].CloudCover_Translucent_tenths, out opqn, out opqd);

                //CalcAtmosphericTransmission_Kn calculates Eq 15 to obtain Kn
                CalcAtmosphericTransmission_Kn(WeatherVariables_list[i].Day, WeatherVariables_list[i].airMass_relativeOptical_kg_p_m2, WeatherVariables_list[i].TempC, WeatherVariables_list[i].DewTempC, WeatherVariables_list[i].PrsMbar, ref WeatherVariables_list[i].SatVPrsKpa, ref WeatherVariables_list[i].VapPrsKpa, ref WeatherVariables_list[i].RelHum, LocationData_Object.ozone[WeatherVariables_list[i].Month - 1], LocationData_Object.CoeffA, LocationData_Object.CoeffPHI, LocationData_Object.CoeffC, LocationData_Object.Elevation, airMass_relativeOpticalPressueCorrected_kg_p_m2, WeatherVariables_list[i].CloudCover_Opaque_tenths, WeatherVariables_list[i].CloudCover_Translucent_tenths, opqn, out Tr, out Toz, out Tum, out Ta, out Kn);

                //CalcAtmosphericTransmission_Kd calculates Eq 32 to obtain Kd
                CalcAtmosphericTransmission_Kd(WeatherVariables_list[i].airMass_relativeOptical_kg_p_m2, WeatherVariables_list[i].CloudCover_Opaque_tenths, WeatherVariables_list[i].CloudCover_Translucent_tenths, WeatherVariables_list[i].RainMh, LocationData_Object.AlbedoVal[WeatherVariables_list[i].Month - 1], Tr, Toz, Tum, Ta, opqd, Kn, out Kd);

                //Calculate Earth's Surface Direct Shortwave Radiation with Eq 14
                //Radiation_Shortwave_Direct_Wpm2 = Kn * ETR
                //Note: Radiation_Shortwave_Direct_Wpm2 from Eq 14 of Hirabayashi and Endreny (2025)
                WeatherVariables_list[i].Radiation_Shortwave_Direct_Wpm2 = Kn * ETR;
                //Calculate Earth's Surface Diffuse Shortwave Radiation
                //Radiation_Shortwave_Diffuse_Wpm2 = Kd * ETR
                //Note: Radiation_Shortwave_Diffuse_Wpm2 from Eq 31 of Hirabayashi and Endreny (2025)
                WeatherVariables_list[i].Radiation_Shortwave_Diffuse_Wpm2 = Kd * ETR;
                //Calculate Earth's Surface Total Shortwave Radiation
                //Radiation_Shortwave_Total_Wpm2 = (Kn + Kd) * ETR
                //Note: Radiation_Shortwave_Total_Wpm2 from combined Eq 14 and 31 of Hirabayashi and Endreny (2025)
                WeatherVariables_list[i].Radiation_Shortwave_Total_Wpm2 = (Kn + Kd) * ETR;
                WeatherVariables_list[i].PARWm2 = WeatherVariables_list[i].Radiation_Shortwave_Total_Wpm2 * 0.46;
                WeatherVariables_list[i].PARuEm2s = WeatherVariables_list[i].PARWm2 * 4.6;

                if (weaFormat != WeatherDataFormat.NARCCAP) {
                    //CalcNetRadWm2 calculates Radiation_Net_Wpm2
                    CalcNetRadWm2(WeatherVariables_list[i].TempK, WeatherVariables_list[i].DewTempC, WeatherVariables_list[i].CloudCover_Total_tenths,
                                WeatherVariables_list[i].Radiation_Shortwave_Total_Wpm2, ref WeatherVariables_list[i].Radiation_Longwave_Downwelling_Wpm2, ref WeatherVariables_list[i].Radiation_Longwave_Upwelling_Wpm2, ref WeatherVariables_list[i].Radiation_Net_Wpm2, LocationData_Object.AlbedoVal[WeatherVariables_list[i].Month - 1]);
                }
            }
        }
        /// <summary>
        /// Calculate 
        /// </summary>
        /// <param name="jDay"></param>
        /// <param name="zenAgl_deg"></param>
        /// <param name="ETR"></param>
        /// <param name="ETRN"></param>
        static void CalcExtraterrestrialRadiation(int jDay, double zenAgl_deg, out double ETR)
        {
            double dayAngle;  //psi:   day angle
            double eccent;    //eo:    eccentricity of the earth's orbit

            dayAngle = Deg2Rad(360 * (jDay - 1) / 365.0);                        //in radians
            eccent = 1.00011 + 0.034221 * Math.Cos(dayAngle) + 0.00128 * Math.Sin(dayAngle)
                    + 0.000719 * Math.Cos(2 * dayAngle) + 0.000077 * Math.Sin(2 * dayAngle);

            //ETR is extraterrestrial radiation on a curved earth, adjusted for zenAgl_deg, from Eq 11 and Eq 12 combined
            ETR = eccent * SOL_CONST * Math.Cos(Deg2Rad(zenAgl_deg));

            //If ETR < 0 then set to 0
            if (ETR < 0)
            {
                ETR = 0.0;
            }
        }
        static void CalcAirMass(double zenAgl_deg, ref double airMass_relativeOptical_kg_p_m2, double prsKpa, out double airMass_relativeOpticalPressueCorrected_kg_p_m2)
        {
            double solElvAgl_deg; //alpha: solar elevation angle
            double prsMbar = prsKpa * 10.0;

            solElvAgl_deg = 90.0 - zenAgl_deg;   //in degrees
            if (zenAgl_deg <= 90)
            {
                //airMass_relativeOptical_kg_p_m2 is 1 / (Math.Sin(Deg2Rad(solElvAgl_deg)) + 0.50572 * Math.Pow((solElvAgl_deg + 6.07995), -1.6364))
                //Note: airMass_relativeOptical_kg_p_m2 from Eq 18 of Hirabayashi and Endreny (2025) and Eq 3-1 of NREL (1995)
                //Note: airMass_relativeOptical_kg_p_m2 should range from 1 at solar noon, to about 30 or 40 at sunrise and sunset
                airMass_relativeOptical_kg_p_m2 = 1 / (Math.Sin(Deg2Rad(solElvAgl_deg)) + 0.50572 * Math.Pow((solElvAgl_deg + 6.07995), -1.6364));
            }
            else
            {
                airMass_relativeOptical_kg_p_m2 = 99.0;
            }
            if (airMass_relativeOptical_kg_p_m2 >= 30)
            {
                airMass_relativeOptical_kg_p_m2 = 30.0;
            }
            //airMass_relativeOpticalPressueCorrected_kg_p_m2 is airMass_relativeOptical_kg_p_m2 * (prsMbar) / 1013.0
            //Note: airMass_relativeOpticalPressueCorrected_kg_p_m2 from Eq 17 of Hirabayashi and Endreny (2025) and Table 3-4 of NREL (1995)
            airMass_relativeOpticalPressueCorrected_kg_p_m2 = airMass_relativeOptical_kg_p_m2 * (prsMbar) / 1013.0;
            //Warning: This seems to be an error, and prsKpa / 10 should be prsKpa * 10 to convert to prsMbar
            //cte old error: airMass_relativeOpticalPressueCorrected_kg_p_m2 = airMass_relativeOptical_kg_p_m2 * (prsKpa / 10.0) / 1013.0;
        }

        //CalcCloudCover function computes opqn and opqd
        static void CalcCloudCover(double airMass_relativeOptical_kg_p_m2, double CeilingHeight_hundredths_ft, double CloudCover_Total_tenths,
                            ref double CloudCover_Opaque_tenths, ref double CloudCover_Translucent_tenths, out double opqn, out double opqd)
        {
            double a1;
            double b1;
            double corrN;
            double corrD;

            //Algorithm is loosely based on Atkinson et al. (1992) and NREL (1995) NSRDB partitioning total cloud cover to opaque and translucent cover
            //Reference:
            //Atkinson, D. and R. F. Lee (1992). "Procedures for Substituting Values for Missing NWS Meteorological Data for Use in Regulatory Air Quality Models." Environmental Protection Agency (EPA).
            //USEPA(1997).Analysis of the Affect of ASOS - Derived Meteorological Data on Refined Modeling.Research Triangle Park, NC, US Environmental Protection Agency.
            /* 
            //Original SAS Code to assign opaque cloud cover based on ceiling height (CLG) and total cloud cover (SKC)
            //Presumed OPQCC is missing and OPQCC_T is CLG
            // IF OPQCC has a value then total cloud cover is OPQCC
            IF OPQCC GE .Z THEN OPQCC_T = OPQCC;
            // If OPQCC is missing and TOTCC is not missing
            IF OPQCC = . AND TOTCC NE . THEN OPQCC = TOTCC;
            ELSE IF OPQCC = . AND TOTCC =  . AND CEILING > 70 THEN OPQCC = 0;
            ELSE IF OPQCC = . AND TOTCC =  . AND CEILING LE 70 AND CEILING GT 60 THEN OPQCC = 7;
            ELSE IF OPQCC = . AND TOTCC =  . AND CEILING LE 60 AND CEILING GT 50 THEN OPQCC = 8;
            ELSE IF OPQCC = . AND TOTCC =  . AND CEILING LE 50 AND CEILING GT 30 THEN OPQCC = 9;
            ELSE IF OPQCC = . AND TOTCC =  . AND CEILING LE 30 THEN OPQCC = 10;
            ELSE IF OPQCC = . AND TOTCC =  . AND CEILING = .   THEN OPQCC = OPQCC_T;
            IF TOTCC  = . AND OPQCC NE . THEN TOTCC = OPQCC;
            */

            //if (CeilingHeight_hundredths_ft > 70 || CeilingHeight_hundredths_ft == 722) 
            //Note: If CeilingHeight_hundredths_ft == 722 then the CLG code 722 indicates unlimited, and set CloudCover_Opaque_tenths = 0
            if (CeilingHeight_hundredths_ft > 70 || CeilingHeight_hundredths_ft == 722) {
                CloudCover_Opaque_tenths = 0;
            }
            //else if (CeilingHeight_hundredths_ft <= 70 && CeilingHeight_hundredths_ft > 60) 
            else if (CeilingHeight_hundredths_ft <= 70 && CeilingHeight_hundredths_ft > 60) {
                CloudCover_Opaque_tenths = 7;
            }
            //else if (CeilingHeight_hundredths_ft <= 60 && CeilingHeight_hundredths_ft > 50)
            else if (CeilingHeight_hundredths_ft <= 60 && CeilingHeight_hundredths_ft > 50) {
                CloudCover_Opaque_tenths = 8;
            }
            //else if (CeilingHeight_hundredths_ft <= 50 && CeilingHeight_hundredths_ft > 30)
            else if (CeilingHeight_hundredths_ft <= 50 && CeilingHeight_hundredths_ft > 30) {
                CloudCover_Opaque_tenths = 9;
            }
            //else if (CeilingHeight_hundredths_ft <= 30 && CeilingHeight_hundredths_ft > 10)
            else if (CeilingHeight_hundredths_ft <= 30 && CeilingHeight_hundredths_ft > 10) {
                CloudCover_Opaque_tenths = 10;
            }
            //else if (CeilingHeight_hundredths_ft <= 30 && CeilingHeight_hundredths_ft > 10)
            else {
                CloudCover_Opaque_tenths = CloudCover_Total_tenths;
            }

            //CloudCover_Translucent_tenths = CloudCover_Total_tenths - CloudCover_Opaque_tenths
            //CloudCover_Translucent_tenths from Eq 44 of Hirabayashi and Endreny (2025) and Eq 3-24 of NREL (1995)
            CloudCover_Translucent_tenths = CloudCover_Total_tenths - CloudCover_Opaque_tenths;

            //if (CloudCover_Translucent_tenths < 0) then set to 0
            if (CloudCover_Translucent_tenths < 0) {
                CloudCover_Translucent_tenths = 0;
            }

            //If (1 <= CloudCover_Opaque_tenths && CloudCover_Opaque_tenths <= 9)
            //Note: From NREL (1995) Algorithm around Eq 3-16
            if (1 <= CloudCover_Opaque_tenths && CloudCover_Opaque_tenths <= 9) {
                //a1 = 4.955 * (1 - Math.Exp(-0.454 * airMass_relativeOptical_kg_p_m2)) - 3.4
                //Note: a1 from Eq 28 of Hirabayashi and Endreny (2025) and Eq 3-16 of NREL (1995)
                a1 = 4.955 * (1 - Math.Exp(-0.454 * airMass_relativeOptical_kg_p_m2)) - 3.4;
                //if (a1 <= 0)
                if (a1 <= 0) {
                    //b1 = -0.2 * a1
                    //Note: b1 from conditional Eq 29 of Hirabayashi and Endreny (2025) and Eq 3-17 of NREL (1995)
                    b1 = -0.2 * a1;
                }
                else {
                    //Note: b1 from conditional Eq 29 of Hirabayashi and Endreny (2025) and Eq 3-17 of NREL (1995)
                    b1 = 0.1 * a1;
                }
                //corrN = a1 * Math.Sin(Math.PI / 10 * CloudCover_Total_tenths) + b1 * Math.Sin(2 * Math.PI / 10 * CloudCover_Total_tenths)
                //Note: corrN from Eq 27 of Hirabayashi and Endreny (2025) and Eq 3-15 of NREL (1995)
                //Note: Original equation used sin(18 * OPQ) and sin(36 * OPQ) which is equivalent of sin(180 * OPQ/10) and sin(360 * OPQ/10) 
                //Note: ... where OPQ units tenths (1 to 10) and sin was operating on degrees not radians
                corrN = a1 * Math.Sin(Math.PI / 10 * CloudCover_Total_tenths) + b1 * Math.Sin(2 * Math.PI / 10 * CloudCover_Total_tenths);
                //corrD = a1 * Math.Sin(Math.PI / 10 * CloudCover_Total_tenths) * 0.5
                //Note: corrD from Eq 42 of Hirabayashi and Endreny (2025) and Eq 3-18 of NREL (1995)
                //Note: Original equation used sin(18 * OPQ) which is equivalent of sin(180 * OPQ/10) 
                //Note: ... where OPQ units tenths (1 to 10) and sin was operating on degrees not radians
                corrD = a1 * Math.Sin(Math.PI / 10 * CloudCover_Total_tenths) * 0.5;
            }
            //else Algorithm from below Eq 3-18 in NREL (1995)
            else
            {
                //corrN = 0.0 when OPQ = 0 or 10, algorithm from below Eq 3-18 in NREL (1995)
                corrN = 0.0;
                //corrD = 0.0 when OPQ = 0 or 10, algorithm from below Eq 3-18 in NREL (1995)
                corrD = 0.0;
            }
            //opqn = CloudCover_Opaque_tenths + corrN;
            //Note: opqn within Eq 26 of Hirabayashi and Endreny (2025) and Eq 3-20 of NREL (1995)
            opqn = CloudCover_Opaque_tenths + corrN;

            //opqd = CloudCover_Opaque_tenths + corrD
            //Note: opqd from Eq 41 of Hirabayashi and Endreny (2025) and below Eq 3-23 of NREL (1995)
            opqd = CloudCover_Opaque_tenths + corrD;
        }

        //CalcAtmosphericTransmission_Kn function computes direct normal transmittance for use in horizontal radiation
        static void CalcAtmosphericTransmission_Kn(int day, double airMass_relativeOptical_kg_p_m2, double tempC, double dewC, double AtmPres_Station_Mb, ref double SatVPrsKpa, ref double VapPrsKpa, ref double RelHum, double ozone, double a, double phi, double C, double ElevationStation_m, double airMass_relativeOpticalPressueCorrected_kg_p_m2, double CloudCover_Opaque_tenths, double CloudCover_Translucent_tenths, double opqn, out double Tr, out double Toz, out double Tum, out double Ta, out double Kn)
        {
            double Tw;        //Tw:    water vapor absorption transmittance
            double Ttrn;      //TTRN:  translucent cloud transmittance
            double Topq;      //TOPQ:  opaque cloud cover transmittance
            double Xo;        //XO:    total amount of ozone in cm in a slanted path
            double tau;       //tauA:  broadband aerosol optical depth
            double vPrsMb;    //vapor pressure in mBar
            double vPrs_StationPressureCorrected_Mb; //pressure adjusted vapor pressure in mBar
            double PrecipitableWater_mm;        //PrecipitableWater_mm:    total precipitable water
            double Xw;        //Xw:    precipitable water vapor in cm in a slant path
            double ATRN_directNormalTransmittanceIntercept = 1.0;
            double BTRN_directNormalTransmittanceSlope = 0.0;
            double M_PrecipitableWater_slope;
            double B_PrecipitableWater_intercept;

            //Tr is Raleigh scattering transmittance
            //Tr = Math.Exp((-0.0903 * Math.Pow(airMass_relativeOpticalPressueCorrected_kg_p_m2, 0.84)) * (1 + airMass_relativeOpticalPressueCorrected_kg_p_m2 - Math.Pow(airMass_relativeOpticalPressueCorrected_kg_p_m2, 1.01)))
            //Note: Tr from Eq 16 of Hirabayashi and Endreny (2025) and Eq 3-4 of NREL (1995) 
            Tr = Math.Exp((-0.0903 * Math.Pow(airMass_relativeOpticalPressueCorrected_kg_p_m2, 0.84)) * (1 + airMass_relativeOpticalPressueCorrected_kg_p_m2 - Math.Pow(airMass_relativeOpticalPressueCorrected_kg_p_m2, 1.01)));

            //Xo = ozone * airMass_relativeOptical_kg_p_m2
            //Note: Xo from Eq 20 of Hirabayashi and Endreny (2025) and Table (of symbols) 3-4 of NREL (1995) 
            Xo = ozone * airMass_relativeOptical_kg_p_m2;

            //Toz is ozone absorption transmittance
            //Toz = 1 - 0.1611 * Xo * Math.Pow((1 + 139.45 * Xo), -0.3035) - 0.002715 * Xo / (1 + 0.044 * Xo + 0.0003 * Math.Pow(Xo, 2))
            //Note: Toz from Eq 19 of Hirabayashi and Endreny (2025) and Eq 3-5 of NREL (1995) 
            Toz = 1 - 0.1611 * Xo * Math.Pow((1 + 139.45 * Xo), -0.3035) - 0.002715 * Xo / (1 + 0.044 * Xo + 0.0003 * Math.Pow(Xo, 2));

            //Tum is uniformly mixed gas absorption transmittance
            //Tum = Math.Exp(-0.0127 * Math.Pow(airMass_relativeOpticalPressueCorrected_kg_p_m2, 0.26))
            //Note: Tum from Eq 21 of Hirabayashi and Endreny (2025) and Eq 3-6 of NREL (1995) 
            Tum = Math.Exp(-0.0127 * Math.Pow(airMass_relativeOpticalPressueCorrected_kg_p_m2, 0.26));

            //tau = a * Math.Sin(360 * (double)day / 365 - phi) + C
            //Note: tau from Eq 25 of Hirabayashi and Endreny (2025) and Eq 6-5 of NREL (1995) and Eq 4 of Maxwell (1998)
            //Note: Algorithm and Coefficients a, phi, C from NREL (1995) Section 6.0, Daily Estimates of Aerosol Optical Depths
            tau = a * Math.Sin(360 * (double)day / 365 - phi) + C;

            //Ta is aerosol absorption and scattering transmittance
            //Ta = Math.Exp(-tau * airMass_relativeOptical_kg_p_m2)
            //Note: Ta from Eq 24 of Hirabayashi and Endreny (2025) and Eq 3-8 of NREL (1995)
            Ta = Math.Exp(-tau * airMass_relativeOptical_kg_p_m2);

            //SatVPrsKpa = ES_CONS * Math.Exp((17.27 * tempC) / (237.3 + tempC))
            //Note: SatVPrsKpa from Eq 66 of Hirabayashi and Endreny (2025)
            SatVPrsKpa = ES_CONS * Math.Exp((17.27 * tempC) / (237.3 + tempC));
            //VapPrsKpa = ES_CONS * Math.Exp((17.27 * dewC) / (237.3 + dewC))
            //Note: VapPrsKpa from Eq 67 of Hirabayashi and Endreny (2025)
            VapPrsKpa = ES_CONS * Math.Exp((17.27 * dewC) / (237.3 + dewC));
            //RelHum = VapPrsKpa / SatVPrsKpa
            //Note: Standard relation
            RelHum = VapPrsKpa / SatVPrsKpa;
            //vPrsMb = VapPrsKpa * 10.0
            //Note: Standard conversion
            vPrsMb = VapPrsKpa * 10.0;

            //vPrs_StationPressureCorrected_Mb = vPrsMb * AtmPres_Station_Mb / 1013.25
            //Note: vPrs_StationPressureCorrected_Mb from Eq (missing) of Hirabayashi and Endreny (2025) and Eq 5-8 of NREL (1995)
            //Note: Algorithm and Coefficients M and B from NREL (1995) Section 5.0, Hourly Estimates of Precipitable Water
            vPrs_StationPressureCorrected_Mb = vPrsMb * AtmPres_Station_Mb / 1013.25;

            //M_PrecipitableWater_slope = (0.0004 * ElevationStation_m + 1.1)
            //Note: M_PrecipitableWater_slope from Eq (missing) of Hirabayashi and Endreny (2025) and Eq 5-10 of NREL (1995) 
            M_PrecipitableWater_slope = (0.0004 * ElevationStation_m + 1.1);
            //B_PrecipitableWater_intercept is 1 
            //Warning: B_PrecipitableWater_intercept should be 1 or 2 based on humidity and Table 5-2 of NREL (1995)
            B_PrecipitableWater_intercept = 1;

            //PrecipitableWater_mm = vPrs_StationPressureCorrected_Mb * M_PrecipitableWater_slope + B_PrecipitableWater_intercept
            //Note: vPrs_StationPressureCorrected_Mb from Eq (missing) of Hirabayashi and Endreny (2025) and Eq 5-9 of NREL (1995) 
            //Warning: Coefficient B set to 1, but can be 1 or 2 as shown in Table 5-2 of NREL (1995)
            PrecipitableWater_mm = vPrs_StationPressureCorrected_Mb * M_PrecipitableWater_slope + B_PrecipitableWater_intercept;

            //PrecipitableWater_mm = PrecipitableWater_mm / 10.0
            //Note: PrecipitableWater_cm should have max value = 6 cm, stated below Eq 3-7 and as known in meteorology
            //Note: Standard conversion
            double PrecipitableWater_cm = PrecipitableWater_mm / 10.0;

            //Note: Xw = PrecipitableWater_cm from Eq (missing) of Hirabayashi and Endreny (2025) and Eq 5-9 and Eq 5-10 of NREL (1995)
            //Warning: PrecipitableWater_mm was erroneously multipled by airMass_relativeOptical_kg_p_m2; confused m air mass and slope symbols 
            Xw = PrecipitableWater_cm;

            //Tw is water vapor transmittance
            //Tw = 1 - 1.668 * Xw / (Math.Pow((1 + 54.6 * Xw), 0.637) + 4.042 * Xw)
            //Note: Tw from Eq 22 of Hirabayashi and Endreny (2025) and Eq 3-7 of NREL (1995) and Eq 3 of Maxwell (1998)
            //Note: Xw (cm) is the precipitable water vapor in a slant path through the atmosphere
            Tw = 1 - 1.668 * Xw / (Math.Pow((1 + 54.6 * Xw), 0.637) + 4.042 * Xw);

            //Translucent transmittance
            //Ttrn = 1, initialized to unlimitted; developed from 3.4.3 Translucent Cloud Algorithms of NREL (1995)
            //Note: Ttrn from Eq (missing) of Hirabayashi and Endreny (2025) and Eq 3-25 of NREL (1995)
            Ttrn = 1;
            //If CloudCover_Translucent_tenths >= 0.5 then enter
            if (CloudCover_Translucent_tenths > 0.5) {
                //BTRN_0 and Coeff_c initialized
                double BTRN_0 = 0.0;
                double Coeff_c = 0.0;
                //CloudCover_Translucent_tenths_int is CloudCover_Translucent_tenths rounded to nearest integer
                int CloudCover_Translucent_tenths_int = (int)Math.Round(CloudCover_Translucent_tenths);
                //GetTranslucentCloudCoefficients takes CloudCover_Translucent_tenths_int and returns BTRN_0 and Coeff_c
                //Note: GetTranslucentCloudCoefficients accesses values from Table 3-6 in NREL (1995)
                GetTranslucentCloudCoefficients(CloudCover_Translucent_tenths_int, out BTRN_0, out Coeff_c);

                //BTRN_directNormalTransmittanceSlope = BTRN_0 + Coeff_c * CloudCover_Opaque_tenths
                //Note: BTRN_directNormalTransmittanceSlope from Eq 3-28 in NREL (1995)
                BTRN_directNormalTransmittanceSlope = BTRN_0 + Coeff_c * CloudCover_Opaque_tenths;

                //if (CloudCover_Opaque_tenths + CloudCover_Translucent_tenths > 10)
                //Note: This condition should not happen, given 10 is the maximum sky cloud cover
                //Note: Conditioanl algorithm developed with DeepSeek to satisfy all values of OPQ and avoid using Table 3-7
                //Note: Eq 3-27 of NREL (1995) does not fit the data values in Table 3-7; it is valid only when m = 1.0 and OPQ = 0.
                if (CloudCover_Opaque_tenths + CloudCover_Translucent_tenths > 10) {
                    ATRN_directNormalTransmittanceIntercept = 0.0;
                }
                //else if (CloudCover_Opaque_tenths == 0 || CloudCover_Translucent_tenths_int == 1)
                //Note: This condition is follows both the top row and left column of Table 3-7 in NREL (1995)
                else if (CloudCover_Opaque_tenths == 0 || CloudCover_Translucent_tenths_int == 1) {
                    //ATRN_directNormalTransmittanceIntercept = 0.995 * Math.Pow(0.9958, CloudCover_Translucent_tenths - 1) * Math.Pow(0.9975, CloudCover_Opaque_tenths)
                    //Note: Equation was developed in DeepSeek to improve on retrieving values from Table 3-25 in NREL (1995)
                    ATRN_directNormalTransmittanceIntercept = 0.995 * Math.Pow(0.9958, CloudCover_Translucent_tenths - 1) * Math.Pow(0.9975, CloudCover_Opaque_tenths);
                }
                //else
                //Note: This condition is approximates all cells but top row and left column of Table 3-7 in NREL (1995)
                else {
                    //ATRN_directNormalTransmittanceIntercept = 0.995 * Math.Pow(0.993, CloudCover_Translucent_tenths - 1) * Math.Pow(0.994, CloudCover_Opaque_tenths)
                    //Note: Equation was developed in DeepSeek to improve on retrieving values from Table 3-25 in NREL (1995)
                    ATRN_directNormalTransmittanceIntercept = 0.995 * Math.Pow(0.993, CloudCover_Translucent_tenths - 1) * Math.Pow(0.994, CloudCover_Opaque_tenths);
                }
                //Ttrn = ATRN_directNormalTransmittanceIntercept - BTRN_directNormalTransmittanceSlope * airMass_relativeOptical_kg_p_m2
                //Note: Ttrn from Eq 30 of Hirabayashi and Endreny (2025) and Eq 3-25 of NREL (1995)
                Ttrn = ATRN_directNormalTransmittanceIntercept - BTRN_directNormalTransmittanceSlope * airMass_relativeOptical_kg_p_m2;
                //If (Ttrn < 0 ) then set to 0
                if (Ttrn < 0 ) {
                    Ttrn = 0;
                }
            }

            //Opaque cloud
            //Topq = (10 - opqn) / 10.0
            //Note: Topq from Eq 26 of Hirabayashi and Endreny (2025) and Eq 3-19 of NREL (1995)
            //Note: opqn combines opq + n from 
            Topq = (10 - opqn) / 10.0;

            //Kn is direct beam transmittance
            //Kn = 0.9751 * Tr * Toz * Tum * Tw * Ta * Topq * Ttrn
            //Note: Kn from Eq 15 of Hirabayashi and Endreny (2025) and Eq 5 of Maxwell (1998)
            Kn = 0.9751 * Tr * Toz * Tum * Tw * Ta * Topq * Ttrn;

            //Kt is effective global horizontal transmittance, Kt = Kn + Kd from Eq 2-4 of NREL (1995)
               
        }

        //GetTranslucentCloudCoefficients accesses values from Table 3-6 in NREL (1995)
        static void GetTranslucentCloudCoefficients(int CloudCover_Translucent_tenths, out double BTRN_0, out double Coeff_c)
        {
            BTRN_0 = 0.0;
            Coeff_c = 0.0;

            //switch contains values from Table 3-6 in NREL (1995)
            switch (CloudCover_Translucent_tenths)
            {
                case 1: BTRN_0 = 0.004; Coeff_c = 0.003; break;
                case 2: BTRN_0 = 0.006; Coeff_c = 0.004; break;
                case 3: BTRN_0 = 0.009; Coeff_c = 0.005; break;
                case 4: BTRN_0 = 0.012; Coeff_c = 0.006; break;
                case 5: BTRN_0 = 0.016; Coeff_c = 0.006; break;
                case 6: BTRN_0 = 0.020; Coeff_c = 0.006; break;
                case 7: BTRN_0 = 0.026; Coeff_c = 0.006; break;
                case 8: BTRN_0 = 0.032; Coeff_c = 0.006; break;
                case 9: BTRN_0 = 0.040; Coeff_c = 0.006; break;
                case 10: BTRN_0 = 0.050; Coeff_c = 0.006; break;
                default:
                    Console.WriteLine("Invalid translucent cloud cover value.");
                    break;
            }
        }

        //CalcAtmosphericTransmission_Kd function computes effective diffuse horizontal transmittance for use in horizontal radiation
        static void CalcAtmosphericTransmission_Kd(double airMass_relativeOptical_kg_p_m2, double CloudCover_Opaque_tenths, double Trans, double rain, double albedo, double
                        Tr, double Toz, double Tum, double Ta, double opqd, double Kn, out double Kd)
        {
            double Ksr;       //KSR:   diffuse radiation from Rayleigh scattering
            double Tas;       //TAS:   trensmittance of aerosol scattering
            double Taa;       //TAA:   transmittance of aerosol absorptance
            double Ksa;       //KSA:   diffuse radiation from aerosol scattering
            double fM;        //f(M):  empirical air mass function
            double Ksopq;     //KSOPQ:
            double Kstrn;     //KSTRN:
            double Ksgrf;     //KSGRF: diffuse radiation from ground reflectance
            double Ksgrf1;    //KSGRF1:
            double Ksgrf2;    //KSGRF2:

            double b2;
            double c2;
            double PSW_precipitableWaterSwitch;       //PSW:
            double Kd0;
            double Rcld;    //RCLD:  broadband (solar) cloud reflectance
            double Ratm;      //RATM:  broadband (solar) atmospheric reflectance

            //Taa = 1 - K1 * (1 - airMass_relativeOptical_kg_p_m2 + Math.Pow(airMass_relativeOptical_kg_p_m2, 1.06)) * (1 - Ta)
            //Note: Taa from Eq 36 of Hirabayashi and Endreny (2025) and Eq 3-9 of NREL (1995)
            Taa = 1 - K1 * (1 - airMass_relativeOptical_kg_p_m2 + Math.Pow(airMass_relativeOptical_kg_p_m2, 1.06)) * (1 - Ta);
            //Ksr = 0.5 * (1 - Tr) * Toz * Tum * Taa
            //Note: Ksr from Eq 35 of Hirabayashi and Endreny (2025) and Eq 3-11 of NREL (1995)
            Ksr = 0.5 * (1 - Tr) * Toz * Tum * Taa;
            //Ksa = BA * (1 - Ta) * Toz * Tum * Taa
            //Note: Ksa from Eq 37 of Hirabayashi and Endreny (2025) and Eq 3-12 of NREL (1995)
            Ksa = BA * (1 - Ta) * Toz * Tum * Taa;
            //fM = 0.38 + 0.925 * Math.Exp(-0.851 * airMass_relativeOptical_kg_p_m2)
            //Note: fM from Eq 34 of Hirabayashi and Endreny (2025) and Eq 3-14 of NREL (1995)
            fM = 0.38 + 0.925 * Math.Exp(-0.851 * airMass_relativeOptical_kg_p_m2);

            //    if CloudCover_Translucent_tenths > 0 {
            //        Kstrn = -0.00235 + 0.00689 * CloudCover_Translucent_tenths + 0.000203 * CloudCover_Translucent_tenths ^ 2
            //    }
            Kstrn = 0.0;

            Ksopq = 0.0;
            if (opqd >= 5)
            {
                //b2 = 0.0953 + 0.137 * opqd - 0.0409 * Math.Pow(opqd, 2) + 0.00579 * Math.Pow(opqd, 3) - 0.000328 * Math.Pow(opqd, 4)
                //Note: b2 from Eq 39 of Hirabayashi and Endreny (2025) and Eq 3-22 of NREL (1995)
                b2 = 0.0953 + 0.137 * opqd - 0.0409 * Math.Pow(opqd, 2) + 0.00579 * Math.Pow(opqd, 3) - 0.000328 * Math.Pow(opqd, 4);
                
                //c2 = -0.109 - 0.02 * opqd + 0.011 * Math.Pow(opqd, 2) - 0.00156 * Math.Pow(opqd, 3) + 0.000121 * Math.Pow(opqd, 4)
                //Note: c2 from Eq 40 of Hirabayashi and Endreny (2025) and Eq 3-23 of NREL (1995)
                c2 = -0.109 - 0.02 * opqd + 0.011 * Math.Pow(opqd, 2) - 0.00156 * Math.Pow(opqd, 3) + 0.000121 * Math.Pow(opqd, 4);
                //Ksopq = -0.06 + b2 * Ta + c2 * Math.Pow(Ta, 2)
                //Note: Ksopq from Eq 38 of Hirabayashi and Endreny (2025) and Eq 3-21 of NREL (1995)
                Ksopq = -0.06 + b2 * Ta + c2 * Math.Pow(Ta, 2);

                if (Ksopq <= 0)
                {
                    Ksopq = 0.0;
                }
            }

            //Note: PSW algorithm from Algorithm 3.5.2 The Precipitation Switch of NREL (1995) 
            //Note: PSW is used as a multiplier to modify the diffuse horizontal value
            if (CloudCover_Opaque_tenths >= 8) {
                if (rain != 0) {
                    PSW_precipitableWaterSwitch = 0.6;
                }
                else {
                    PSW_precipitableWaterSwitch = 1.0;
                }
            }
            else {
                PSW_precipitableWaterSwitch = 1.0;
            }

            //Kd0 = (fM * (Ksr + Ksa) + Ksopq + Kstrn) * PSW_precipitableWaterSwitch
            //Note: Kd0 from Eq 33 of Hirabayashi and Endreny (2025) and within Eq 6 of Maxwell (1998)
            //Note: Kd0 is computed to avoid an implicit function, separating out Ksgrf given Ksgrf is a function of Kd
            Kd0 = (fM * (Ksr + Ksa) + Ksopq + Kstrn) * PSW_precipitableWaterSwitch;

            //Rcld = 0.06 * CloudCover_Opaque_tenths + 0.02 * Trans
            //Note: Rcld from Eq 48 of Hirabayashi and Endreny (2025) and Eq 3-31 of NREL (1995)
            Rcld = 0.06 * CloudCover_Opaque_tenths + 0.02 * Trans;
            //Note: Rcld from Eq 46 of Hirabayashi and Endreny (2025) and Eq 3-33 of NREL (1995)
            Ksgrf1 = (Kn + Kd0) * Rcld * (albedo - 0.2);

            if (Ksgrf1 <= 0) {
                Ksgrf1 = 0.01;
            }
            //Tas = Ta / Taa
            //Note: Tas from Eq 50 of Hirabayashi and Endreny (2025) and Eq 3-10 of NREL (1995)
            Tas = Ta / Taa;
            //Ratm = (0.0685 + 0.16 * (1 - Tas)) * (10 - CloudCover_Opaque_tenths) / 10.0
            //Note: Ratm from Eq 49 of Hirabayashi and Endreny (2025) and Eq 3-32 of NREL (1995)
            Ratm = (0.0685 + 0.16 * (1 - Tas)) * (10 - CloudCover_Opaque_tenths) / 10.0;
            //Ksgrf2 = (Kn + Kd0) * Ratm * albedo
            //Note: Ksgrf2 from Eq 46 of Hirabayashi and Endreny (2025) and Eq 3-34 of NREL (1995)
            Ksgrf2 = (Kn + Kd0) * Ratm * albedo;

            //Ksgrf = Ksgrf1 + Ksgrf2
            //Note: Ksgrf from Eq 45 of Hirabayashi and Endreny (2025) and Eq 3-35 of NREL (1995)
            //Note: Ksgrf is not adjusted for PSW in Eq 6 of Maxwell (1998)
            Ksgrf = Ksgrf1 + Ksgrf2;

            //Kd is effective diffuse horizontal transmittance
            //Kd = Kd0 + Ksgrf
            //Note: Kd from Eq 32 of Hirabayashi and Endreny (2025) and within Eq 6 of Maxwell (1998)
            Kd = Kd0 + Ksgrf;
        }
        static void CalcNetRadWm2(double TempK, double dewC, double CloudCover_Total_tenths, double
                            Radiation_Shortwave_Total_Wpm2, ref double Radiation_Longwave_Downwelling_Wpm2, ref double Radiation_Longwave_Upwelling_Wpm2, ref double Radiation_Net_Wpm2, double albedo)
        {
            //Reference: Radiation_Net_Wpm2
            //Hirabayashi, S., & Endreny, T. (2016). Surface and Upper Weather Pre-processor for i-Tree Eco and Hydro. i-Tree Tools White Paper, Version 1.2. 

            double emissivity;
            double sfcLongRad;    //Long wave radiation from the surface (W/m^2)
            double cldLongRad;    //Long wave radiaton from cloud cover w/ Emissivity set to 1 (W/m^2)
            double skyLongRad;    //Long wave radiation from clear sky with Emissivity set to Ea*(1-cc) (W/m^2)
            double Radiation_Shortwave_Net_Wpm2;     //Total albedo corrected incoming short wave radiation (W/m^2)
            double Radiation_Longwave_Net_Wpm2;    //Total long wave radiation emissivity corrected (incoming sky - outgoing surface) (W/m^2)
            double emissivity_of_1 = 1.0;

            //Emissivity of a clear sky (fraction)
            //Note: emissivity from Eq 58 Hirabayashi and Endreny (2025)
            emissivity = 0.741 + (0.0062 * dewC);
            //Long wave radiation from the surface (W/m^2)
            //Note: sfcLongRad from Eq 59 Hirabayashi and Endreny (2025)
            sfcLongRad = theta * Math.Pow(TempK, 4);
            //Long wave radiaton downwelling from cloud cover (W/m2) w/ Emissivity set to 1 (W/m^2)
            //Note: cldLongRad from Eq 57 Hirabayashi and Endreny (2025)
            cldLongRad = emissivity_of_1 * CloudCover_Total_tenths / 10.0 * theta * Math.Pow(TempK, 4);
            //Long wave radiation from clear sky (W/m2) with Emissivity set to Ea*(1-cc) (W/m^2)
            //Note: skyLongRad from Eq 56 Hirabayashi and Endreny (2025)
            skyLongRad = emissivity * (1 - CloudCover_Total_tenths / 10) * theta * Math.Pow(TempK, 4);

            //Total albedo corrected incoming short wave radiation (W/m^2)
            //Note: Radiation_Shortwave_Net_Wpm2 from Eq 54 Hirabayashi and Endreny (2025)
            Radiation_Shortwave_Net_Wpm2 = (1 - albedo) * Radiation_Shortwave_Total_Wpm2;
            //Total long wave radiation emissivity corrected (incoming sky - outgoing surface) (W/m^2)
            //Note: Radiation_Longwave_Net_Wpm2 from Eq 55 Hirabayashi and Endreny (2025)
            Radiation_Longwave_Net_Wpm2 = ES * (skyLongRad + cldLongRad - sfcLongRad);
            //Total net radation (downwelling short wave and long wave, Radiation_Net_Wpm2) (W/m^2)
            //Note: Radiation_Net_Wpm2 from Eq 53 Hirabayashi and Endreny (2025)
            Radiation_Net_Wpm2 = Radiation_Shortwave_Net_Wpm2 + Radiation_Longwave_Net_Wpm2;

            //Radiation_Longwave_Downwelling_Wpm2 and Radiation_Longwave_Upwelling_Wpm2 sent back to function calling CalcNetRadWm2 for use in Radiation.csv
            //Note: part of Eq 55 Hirabayashi and Endreny (2025)
            Radiation_Longwave_Downwelling_Wpm2 = ES * (skyLongRad + cldLongRad);
            //Total longwave radiation upward (W/m2)
            //Note: part of Eq 55 Hirabayashi and Endreny (2025)
            Radiation_Longwave_Upwelling_Wpm2 = ES * sfcLongRad;

        }
        //Function CalcET using descriptive variable names with units
        public static void CalcET(List<SurfaceWeather> weatherData, List<LeafAreaIndex> LeafAreaIndex_list, int recCnt,
            double Height_WeatherStationWindSensor_m, double Height_TreeCanopyTop_m)
        {
            //References
            //Allen, R.G. (1998). Chapter 2 - FAO Penman-Monteith equation. https://www.fao.org/3/x0490e/x0490e06.htm
            //Chin, D. A. (2021). Water Resources Engineering, Fourth Edition. Hoboken, NJ: Pearson Education.
            //Fassnacht, S.R. 2004 Estimating..Snowpack Simulation, Hydrologic Processes 18:3481-3492 https://onlinelibrary.wiley.com/doi/10.1002/hyp.5806
            //Jensen, M. E., & Allen, R. G. (2015). Evaporation, Evapotranspiration, and Irrigation Water Requirements: Manual of Practice 70, Second Edition (2nd ed.). Reston, VA: American Society of Civil Engineers.
            //Light, P. (1941). Analysis of high rates of snow-melting. Eos, Transactions American Geophysical Union, 22(1), 195-205. doi:https://doi.org/10.1029/TR022i001p00195
            //Lundberg, A., Calder, I., & Harding, R. (1998). Evaporation of intercepted snow: measurement and modelling. Journal of Hydrology, 206(3-4), 151-163. 
            //McCutcheon, S. C., Martin, J. L., & Barnwell, T. O. J. (1993). Water Quality. In D. R. Maidment (Ed.), Handbook of Hydrology (pp. 11.11-11.73). New York: McGraw-Hill.
            //Shuttleworth, J. W. (1993). Evaporation. In D. R. Maidment (Ed.), Handbook of Hydrology (pp. 4.1-4.5.3). New York: McGraw-Hill.
            //Stull, R. B. (2000). Meteorology for Scientists and Engineers (2nd ed.). Pacific Grove, CA: Brooks/Cole.
            //UCAR (2011).Snowmelt Processes: International Edition. University Corporation for Atmospheric Research COMET MetEd Program. https://download.comet.ucar.edu/memory-stick/hydro/basic_int/snowmelt/index.htm

            int i;
            double VaporPressure_Actual_kPa;
            double VaporPressure_Saturated_kPa;
            double VaporPressure_Deficit_kPa;
            double VaporPressure_Deficit_Pa;
            double VaporPressure_Deficit_OverWater_Pa;
            double WindSpeed_Station_m_p_s;
            double WindSpeed_TreeCanopy_m_p_s;
            double DensityWater_kg_p_m3;
            double DensityAir_kg_p_m3;
            double Radiation_Net_Wpm2;
            double LeafAreaIndex_Active_Tree_m2_p_m2;
            double ResistanceSurface_Tree_and_Soil_s_p_m;
            double ResistanceSurface_Water_s_p_m;
            double ResistanceAerodynamic_TreeCanopy_s_p_m;
            double ResistanceAerodynamic_SurfaceWater_s_p_m;
            double ResistanceAerodynamic_SnowTreeCanopy_s_p_m;
            double ResistanceAerodynamic_SnowGround_s_p_m;
            double vonKarman_Constant = 0.41;
            double Ratio_sec_to_hr = 3600.0;
            double Ratio_Pa_to_kPa = 1000.0;
            double Ratio_J_to_MJ = 1000000.0;
            double Ratio_MJ_to_J = 1.0 / 1000000.0;
            //Roughness Lengths from Davenport-Wieringa Zo (m) to Classification and Landscape Table 4.1 in Stull (2000)
            //Note: Classificaiton smooth = Snow; Classificaiton Open = Airport; Note UCAR Community Land Model uses 0.024 
            double RoughnessLength_Snow_m = 0.005;
            double RoughnessLength_Airport_m = 0.03;
            //Roughness Lengths for open water from Shuttleworth (1993) page 4.15, above Eq 4.2.30, citing Thom and Oliver (1977)
            double RoughnessLength_Water_m = 0.00137;
            //ZeroPlaneDisplacementHeight_frac (fraction) from Eq 13.3 Chin (2021) adjusts cover height to get zero plane displacement height
            double ZeroPlaneDisplacementHeight_frac = 0.67;
            //Height_TreeWindSensor_m created to avoid negative logarithms below, ensuring tree height is below wind sensor height
            //Note: Error Handling: If users enter wind sensor height = 2 m, tree height = 5 m, then aerodynamic resistance equations fail
            //Note: ... Aerodynamic resistance equations take log of difference of sensor height and zero plane displacement height of object
            //Note: ... tree height * zero plane displacement fraction = 5 * 0.67 = 3.35; ln(2-3.35) = No Solution
            double Height_TreeWindSensor_m = Height_WeatherStationWindSensor_m + Height_TreeCanopyTop_m;
            double Height_TreeZeroPlaneDisplacement_m = Height_TreeCanopyTop_m * ZeroPlaneDisplacementHeight_frac;
            //Height_Ground_m is set to 0.1 m to represent surface irregularity
            //Note: If Height_Ground_m converted to roughness length with Zom scale factor = 0.123 then it has classification between open and smooth 
            double Height_Ground_m = 0.1;
            double Height_GroundWindSensor_m = Height_WeatherStationWindSensor_m + Height_Ground_m;
            double Height_GroundZeroPlaneDisplacement_m = Height_Ground_m * ZeroPlaneDisplacementHeight_frac;
            double LAI_plus_BAI_plus_GAI_Tree_to_SVeg_ratio;
            double LAI_plus_BAI_plus_GAI_Ground_to_Tree_ratio;
            //BarkAreaIndex_Tree_m2_p_m2 for trees is estimated at 1.7 from HydroPlus test cases
            double BarkAreaIndex_Tree_m2_p_m2 = 1.7;
            //LeafAreaIndex_SVeg_m2_p_m2 for shrubs is estimated at 2.3 from HydroPlus test cases
            double LeafAreaIndex_SVeg_m2_p_m2 = 2.2;
            //BarkAreaIndex_SVeg_m2_p_m2 for shrubs is estimated at 0.5 from HydroPlus test cases
            double BarkAreaIndex_SVeg_m2_p_m2 = 0.5;
            //GroundAreaIndex_m2_p_m2 for trees, shrubs, and ground is estimated at 1
            double GroundAreaIndex_m2_p_m2 = 1;

            //SpecificHeat_MoistAir_J_p_kg_p_C is specific heat of moist air at constant pressure, defined by Chin (2021) w Eq 13.37
            double SpecificHeat_MoistAir_J_p_kg_p_C = 1013;
            double LatentHeatOfVaporizaton_J_p_kg;
            double LatentHeatOfVaporizaton_MJ_p_kg;
            double PsychrometricConstant_Pa_p_C;
            double PsychrometricConstant_kPa_p_C;
            double VaporPressureGradient_Pa;
            double VaporPressureGradient_kPa;
            double Ratio_MolecularWeightVapor_to_MolecularWeightDryAir = 0.622;
            double SpecificHeat_MoistAir_MJ_p_kg_p_C;
            double GroundHeatFlux_W_p_m2;
            double ResistanceSurface_Vegetation_s_p_m;
            double BulkStomatalResistance_Adjustment_frac;
            double HeatStorage_Water_J_p_m2_p_s;
            double ResistanceAerodynamic_Snow_to_Rain_ratio;


            try
            {
                //For loop through recCnt, the length of the input data
                for (i = 0; i < recCnt; i++)
                {
                    //VaporPressure_Actual_kPa is actual vapor pressure (kPa)
                    VaporPressure_Actual_kPa = 0.6108 * Math.Exp((17.27 * weatherData[i].DewTempC) / (237.3 + weatherData[i].DewTempC));
                    VaporPressure_Saturated_kPa = 0.6108 * Math.Exp((17.27 * weatherData[i].TempC) / (237.3 + weatherData[i].TempC));
                    //VaporPressure_Deficit_kPa is vapor pressure deficit (kPa), using previously computed saturated vapor pressure SatVPrsKpa (kPa)
                    VaporPressure_Deficit_kPa = (VaporPressure_Saturated_kPa - VaporPressure_Actual_kPa);
                    //VaporPressure_Deficit_Pa (Pa) is vapor pressure deficit in fundamental units used in Penman-Monteith Eq 13.1 of Chin (2021) 
                    VaporPressure_Deficit_Pa = VaporPressure_Deficit_kPa * Ratio_Pa_to_kPa;
                    //VaporPressure_Deficit_OverWater_Pa (Pa) is vapor pressure deficit over water, initially equal to at station value
                    VaporPressure_Deficit_OverWater_Pa = VaporPressure_Deficit_Pa;
                    //Note: 202212 This option currently remains off until further evidence that it is needed.
                    //Note: Adjustment to approximate theory of Jensen and Allen (2015) Chapter 6, Evaporation from Water Surfaces:
                    //Note: ... "Evaporation can be low during summer when shortwave radiative energy is absorbed by the cold water bodies and
                    //Note: ... when vapor pressure gradients above the water are small due to high humidity of air caused by regional ET.
                    //Note: ... Evaporation is high during winter as dry air from nontranspiring frozen regions is coupled with saturation
                    //Note: ... vapor pressure at the water surface that may be much higher than the air."
                    //If TempC > freezing then reduce vapor pressure deficit due to high humidity of air by evaporation; otherwise no adjustment
                    /*
                    if (weatherData[i].TempC > 0) {
                        //VaporPressure_Deficit_Reduction_frac set to 0.75, but should be adjusted to fit evidence of altered deficit
                        double VaporPressure_Deficit_Reduction_frac = 0.75;
                        //VaporPressure_Deficit_Adjustment_frac (frac) formula derived here; should be adjusted to fit observations; 50 degrees used as asymptote
                        double VaporPressure_Deficit_Adjustment_frac = 1 - (Math.Pow(weatherData[i].TempC / 50, 2) * VaporPressure_Deficit_Reduction_frac);
                        //If VaporPressure_Deficit_Adjustment_frac goes below VaporPressure_Deficit_Reduction_frac, then set to that limit
                        if (VaporPressure_Deficit_Adjustment_frac < VaporPressure_Deficit_Reduction_frac) {
                            VaporPressure_Deficit_Adjustment_frac = VaporPressure_Deficit_Reduction_frac; }
                        //VaporPressure_Deficit_OverWater_Pa (Pa) reduced to VaporPressure_Deficit_Adjustment_frac its magnitude
                        VaporPressure_Deficit_OverWater_Pa = VaporPressure_Deficit_OverWater_Pa * VaporPressure_Deficit_Adjustment_frac;
                    }
                    */

                    //WindSpeed_Station_m_p_s is wind speed at station (m/s)
                    WindSpeed_Station_m_p_s = weatherData[i].WdSpdMs;
                    //If WindSpeed_Station_m_p_s < = zero, then set to 0.1 m/s to avoid division by zero
                    if (WindSpeed_Station_m_p_s <= 0)
                    {
                        WindSpeed_Station_m_p_s = 0.1;
                    }
                    //WindSpeed_TreeCanopy_m_p_s defined as wind speed at tree top (m/s) using Eq 4.14b of Stull (2000)
                    WindSpeed_TreeCanopy_m_p_s = WindSpeed_Station_m_p_s * (Math.Log(Height_TreeWindSensor_m / RoughnessLength_Airport_m) /
                        Math.Log(Height_WeatherStationWindSensor_m / RoughnessLength_Airport_m));

                    //DensityWater_kg_p_m3 is density of water (kg/m^3) from Equation in Figure 11.1.1 McCutcheon (1993)
                    DensityWater_kg_p_m3 = 1000 * (1 - ((weatherData[i].TempC + 288.9414) / (508929.2 * (weatherData[i].TempC + 68.12963))) *
                        Math.Pow((weatherData[i].TempC - 3.9863), 2));

                    //DensityAir_kg_p_m3 (kg/m3) is density of air from Eq 4.2.4 from Shuttleworth (1993), w correction in conversion to C from K
                    //Note: Eq 4.24 used incorrect denominator of 275 rather than 273.15 to convert from C to K; see Chin (2021) Eq 13.51
                    //Note: Chin Eq 13.51 which unfortunately uses 3.45 in place of the correct 3.486.
                    //Note: This tested well against values of air density from the EngineeringToolbox.com for air temperature from 0 to 50 C at standard atmospheric pressure
                    DensityAir_kg_p_m3 = 3.486 * (weatherData[i].PrsKpa / (273.15 + weatherData[i].TempC));

                    //LatentHeatOfVaporizaton_MJ_p_kg (MJ/kg) from Eq 13.36 Chin (2021) is latent heat of vaporization
                    LatentHeatOfVaporizaton_MJ_p_kg = 2.501 - 0.002361 * weatherData[i].TempC;
                    //LatentHeatOfVaporizaton_J_p_kg (J/kg) is converted from MJ to K
                    LatentHeatOfVaporizaton_J_p_kg = LatentHeatOfVaporizaton_MJ_p_kg * Ratio_J_to_MJ;

                    //PsychrometricConstant_kPa_p_C (kPa/C) from Eq 13.37 Chin (2021) is psychrometric constant
                    //Note: Eq 13.37 should use specific heat with units of MJ/kg/C, but incorrectly states units are kJ/kg/C
                    SpecificHeat_MoistAir_MJ_p_kg_p_C = SpecificHeat_MoistAir_J_p_kg_p_C * Ratio_MJ_to_J;
                    PsychrometricConstant_kPa_p_C = (SpecificHeat_MoistAir_MJ_p_kg_p_C * weatherData[i].PrsKpa) /
                        (Ratio_MolecularWeightVapor_to_MolecularWeightDryAir * LatentHeatOfVaporizaton_MJ_p_kg);
                    //PsychrometricConstant_Pa_p_C (Pa/C) is converted from kPa to Pa
                    PsychrometricConstant_Pa_p_C = PsychrometricConstant_kPa_p_C * Ratio_Pa_to_kPa;

                    //VaporPressureGradient_kPa (kPa) from Eq 13.43 Chin (2021) is gradient of saturation pressure vs temperature curve
                    VaporPressureGradient_kPa = 4098 * (0.6108 * Math.Exp((17.27 * weatherData[i].TempC) / (weatherData[i].TempC + 237.3))) /
                        Math.Pow((weatherData[i].TempC + 237.3), 2);
                    //VaporPressureGradient_Pa (Pa) is converted from kPa to Pa
                    VaporPressureGradient_Pa = VaporPressureGradient_kPa * Ratio_Pa_to_kPa;

                    //Radiation_Net_Wpm2 (W/m2) is energy from downwelling shortwave and longwave radiation (W/m^2)
                    Radiation_Net_Wpm2 = weatherData[i].Radiation_Net_Wpm2;

                    //GroundHeatFlux_W_p_m2 (W/m2) defined for two vegetation heights after Eq 13.32 by Chin (2021); also Jensen et. al., (2015) Chapter 5
                    //Mote: Jensen and Allen (2015) "Standardized Reference Evapotranspiration for Short Reference ETos: Reference ET calculated
                    //Note: ... for a short crop having height of 0.12 m (similar to grass), albedo of 0.23, surface resistance of 70 sm−1 for
                    //Note: ... 24-h calculation time steps, and 50 sm−1 for hourly or shorter periods during daytime and 200 sm−1 during"
                    //Note: Jensen and Allen (2015) "Standardized Reference Evapotranspiration for Tall Reference ETrs: Reference ET calculated
                    //Note: ... for a tall crop having height of 0.50m (similar to alfalfa), albedo of 0.23, surface resistance of 45 sm−1 for
                    //Note: ... 24-h calculation time steps, and 30 sm−1 for hourly or shorter periods during daytime and 200 sm−1 during nighttime
                    //Note: ... with daytime G=Rn =0.04 and nighttime G=Rn =0.2."
                    //Note: Short crop coefficients 0.1 during day, 0.5 during night. Tall crop coefficients 0.04 during day, 0.2 during night. 
                    //Note: Jensen et al. (2015) generally suggest using reference crop coefficients and then adjusting to actual conditions
                    //If Radiation_Net_Wpm2 (W/m2) positive, presume during day then GroundHeatFlux_W_p_m2 takes one form
                    if (Radiation_Net_Wpm2 > 0)
                    {
                        //GroundHeatFlux_W_p_m2 (W/m2) defined for tall crop after Jensen et. al., (2015) Chapter 5
                        GroundHeatFlux_W_p_m2 = 0.04 * Radiation_Net_Wpm2;
                    }
                    //Else If Radiation_Net_Wpm2 (W/m2) negative, presume during night then GroundHeatFlux_W_p_m2 takes another form
                    else
                    {
                        //GroundHeatFlux_W_p_m2 (W/m2) defined for tall crop after Jensen et. al., (2015) Chapter 5
                        GroundHeatFlux_W_p_m2 = 0.2 * Radiation_Net_Wpm2;
                    }

                    //LeafAreaIndex_Active_Tree_m2_p_m2 (m2/m2) Eq 13.11 from Chin (2021)
                    //Note: Chin (2021) cites Allen (1998), explaining crops typically have 50% of LAI active; trees may have less
                    LeafAreaIndex_Active_Tree_m2_p_m2 = LeafAreaIndex_list[i].Lai * 0.5;

                    //BulkStomatalResistance_s_p_m (s/m) = 200 based on Shuttleworth (1990) Eq 4.2.22 and interpretation of Allen (1998), Chin (2021) around Eq 13.9
                    //Note: ResistanceSurface_Tree_and_Soil_s_p_m derived from BulkStomatalResistance_s_p_m by division with LAI
                    //Note: Minimum values of ResistanceSurface_Tree_and_Soil_s_p_m ~ 100 s/m in Table 2 of Liang et al. (1994) VIC model and ...
                    //Note: ... Box 5 of Allen et al. (1998) Chapter 2 - FAO Penman-Monteith equation https://www.fao.org/3/x0490e/x0490e06.htm, ...
                    //Note: ... and minimum values of ResistanceSurface_Tree_and_Soil_s_p_m ~ 40 s/m are found in Chpt 11 of Jensen and Allen (2015)
                    ResistanceSurface_Vegetation_s_p_m = 200;

                    //BulkStomatalResistance_Adjustment_frac based on suggestion of Jensen and Allen (2015) Chapter 4 Energy Balance:
                    //Note: ... "In addition, a recommendation by Allen et al. (2006a) to use the same 50 sm−1 surface resistance for hourly
                    //Note: ... or shorter periods during daytime and 200 sm−1 during nighttime with the FAO-56 Penman-Monteith method has
                    //Note: ... made the FAO ETo and ETos references equivalent for hourly time steps and for 24-h periods."
                    //Note: Allen (personal communication) suggests larger stomatal resistance for vegetation in a landscape,
                    //Note: ... i.e., not irrigated reference crop, stating it evoled to survive by restricting water loss. 
                    //Note: Equation derived here and coefficient 0.9 was adjusted to fit expected trends in evapotranspiration 
                    BulkStomatalResistance_Adjustment_frac = Math.Pow((1 - VaporPressure_Deficit_kPa / weatherData[i].SatVPrsKpa), 0.9);
                    ResistanceSurface_Vegetation_s_p_m = ResistanceSurface_Vegetation_s_p_m / BulkStomatalResistance_Adjustment_frac;

                    //If LAI_Tree_m2_p_m2 < 1 then prohibit ResistanceSurface_Tree_and_Soil_s_p_m from going toward infinity with division by LAI < 1
                    if (LeafAreaIndex_list[i].Lai < 1)
                    {
                        ResistanceSurface_Tree_and_Soil_s_p_m = ResistanceSurface_Vegetation_s_p_m / 1.0;
                    }
                    //Else If LAI_Tree_m2_p_m2 >= 1 then ResistanceSurface_Tree_and_Soil_s_p_m divided by LeafAreaIndex_Active_Tree_m2_p_m2 
                    else
                    {
                        //ResistanceSurface_Tree_and_Soil_s_p_m (s/m) defined with Eq 13.9 in Chin (2021) and Eq 4.2.22 in Shuttleworth(1993)
                        ResistanceSurface_Tree_and_Soil_s_p_m = ResistanceSurface_Vegetation_s_p_m / LeafAreaIndex_Active_Tree_m2_p_m2;
                    }
                    //ResistanceSurface_Water_s_p_m defined as 0, which is the case for water surface, defined by Chin (2021) after Eq 13.12
                    ResistanceSurface_Water_s_p_m = 0;

                    //Note: These terms used in HydroPlus to compute evaporation for different cover types
                    //LAI_plus_BAI_plus_GAI_Tree_to_SVeg_ratio is ratio of (LAI + BAI + GAI) for tree to (LAI + BAI + GAI) for short veg
                    //Note: LAI = leaf area index, BAI = bark area index, GAI = ground area index, where GAI = 1 for bare and snow covered ground
                    LAI_plus_BAI_plus_GAI_Tree_to_SVeg_ratio = (LeafAreaIndex_list[i].Lai + BarkAreaIndex_Tree_m2_p_m2 + GroundAreaIndex_m2_p_m2) /
                        (LeafAreaIndex_SVeg_m2_p_m2 + BarkAreaIndex_SVeg_m2_p_m2 + GroundAreaIndex_m2_p_m2);
                    //LAI_plus_BAI_plus_GAI_Ground_to_Tree_ratio is ratio of (LAI + BAI + GAI) for ground to (LAI + BAI + GAI) for tree
                    //Note: (LAI + BAI + GAI) for ground = (0 + 0 + GAI) for ground, despite overhanging vegetation
                    LAI_plus_BAI_plus_GAI_Ground_to_Tree_ratio = (0 + 0 + GroundAreaIndex_m2_p_m2) /
                        (LeafAreaIndex_list[i].Lai + BarkAreaIndex_Tree_m2_p_m2 + GroundAreaIndex_m2_p_m2);

                    //ResistanceAerodynamic_TreeCanopy_s_p_m (s/m) from Eq 13.2 of Chin (2021) or Eq 4.2.25 of Shuttleworth (1993) 
                    //Note: Eq 4.2.25 should use the station windspeed, not a windspeed estimated at the height of any lower object
                    //Note: Height_WeatherStationWindSensor_m is used in place of Height_WeatherStationWindSensor_m to ensure real values from log
                    ResistanceAerodynamic_TreeCanopy_s_p_m = (Math.Log((Height_TreeWindSensor_m - Height_TreeZeroPlaneDisplacement_m) /
                        (rZom * Height_TreeCanopyTop_m)) * Math.Log((Height_TreeWindSensor_m - Height_TreeZeroPlaneDisplacement_m) /
                        (rZov * Height_TreeCanopyTop_m))) / (Math.Pow(vonKarman_Constant, 2) * WindSpeed_Station_m_p_s);

                    //ResistanceAerodynamic_SurfaceWater_s_p_m is Eq 13.8 Chin (2021) or Eq 4.2.29 from Shuttleworth (1993) for open water
                    //Note: RoughnessLength_Water_m = 1.37 mm from Chin (2021)
                    //Note: Height_WeatherStationWindSensor_m can be any height with accompanying WindSpeed_Station_m_p_s
                    ResistanceAerodynamic_SurfaceWater_s_p_m = (4.72 * Math.Pow((Math.Log(Height_WeatherStationWindSensor_m /
                        RoughnessLength_Water_m)), 2)) / (1 + 0.536 * WindSpeed_Station_m_p_s);

                    //ResistanceAerodynamic_Snow_to_Rain_ratio set to 10 according to Lundberg et al. (1998) Eq 6 and 7
                    //Note: ResistanceAerodynamic_Snow_to_Rain_ratio is ratio of aerodynamic resistance for snow to rain 
                    ResistanceAerodynamic_Snow_to_Rain_ratio = 10;

                    //ResistanceAerodynamic_SnowTreeCanopy_s_p_m contains resistance terms in Eq 4 of Fassnacht & Eq 3 of Light (1941)
                    ResistanceAerodynamic_SnowTreeCanopy_s_p_m = (Math.Log(Height_TreeWindSensor_m / RoughnessLength_Snow_m) *
                        Math.Log(Height_TreeWindSensor_m / RoughnessLength_Snow_m)) / (Math.Pow(vonKarman_Constant, 2) * WindSpeed_TreeCanopy_m_p_s);

                    //ResistanceAerodynamic_SnowGround_s_p_m contains resistance terms in Eq 4 of Fassnacht & Eq 3 of Light (1941)
                    //Note: Ground uses wind speed measured at station height
                    ResistanceAerodynamic_SnowGround_s_p_m = (Math.Log(Height_WeatherStationWindSensor_m / RoughnessLength_Snow_m) *
                        Math.Log(Height_WeatherStationWindSensor_m / RoughnessLength_Snow_m)) / (Math.Pow(vonKarman_Constant, 2) * WindSpeed_Station_m_p_s);

                    //PtTrMh (m/h), evapotranspiration from canopy and ground, from Eq Eq 13.1 of Chin (2021), Penman Monteith Evaporation method
                    //Note: Equation 13.1 of Chin (2021) requires all terms are in fundamental SI units (J, Pa, kg, s, m, C, etc.)
                    //Note: Issues with Eq 4.2.27 of Shuttleworth (1993) include not dividing by density of water, usng mixed units MJ/day, kPa/C, etc.
                    weatherData[i].PtTrMh = 1 / (LatentHeatOfVaporizaton_J_p_kg * DensityWater_kg_p_m3) *
                        ((VaporPressureGradient_Pa * (Radiation_Net_Wpm2 - GroundHeatFlux_W_p_m2)) +
                        (DensityAir_kg_p_m3 * SpecificHeat_MoistAir_J_p_kg_p_C * VaporPressure_Deficit_Pa / ResistanceAerodynamic_TreeCanopy_s_p_m)) /
                        (VaporPressureGradient_Pa + PsychrometricConstant_Pa_p_C *
                        (1 + ResistanceSurface_Tree_and_Soil_s_p_m / ResistanceAerodynamic_TreeCanopy_s_p_m));
                    //PtTrMh (m/h) converted from (m/s) with Ratio_sec_to_hr
                    weatherData[i].PtTrMh = weatherData[i].PtTrMh * Ratio_sec_to_hr;
                    //If negative pet to zero
                    if (weatherData[i].PtTrMh < 0) { weatherData[i].PtTrMh = 0.0; }

                    //Note: Adjustment to approximate theory of Jensen and Allen (2015) Chapter 6, Evaporation from Water Surfaces:
                    //Note: ... "An important distinction between Rn for a water body and Rn for vegetation or soil is that with soil and
                    //Note: ... vegetation, essentially all of the Rn quantity is captured at the “opaque” surface and is immediately available
                    //Note: ... solar radiation, Rs, for conversion to λE or H or conduction into the surface as G. 
                    //Note: ... With water, however, much of the penetrates to some depth in the water body, depending on the turbidity
                    //Note: ... of the water, where it is converted to Qt, heat storage." This approach is modified for smaller waters below.
                    //HeatStorage_Water_J_p_m2_p_s (J/m2/s) from Eq 6-14a and 6-14b of Jensen and Allen (2015) modified from large lakes
                    //Note: Large lakes used HeatStorage_Coefficient_a = 0.5 and HeatStorage_Coefficient_b = 0.8; reduced for smaller waters
                    //Note: Large lakes increased loss from long wave radiation at Julian Day = 180; not simulated for smaller waters
                    //Note: Modification includes Heat Storage transfering energy into Ground Heat Flux 
                    double HeatStorage_Coefficient_a = 0.25;
                    //HeatStorage_Coefficient_b reduced from large lake value of 0.8
                    double HeatStorage_Coefficient_b = 0.05;
                    //HeatStorage_Water_J_p_m2_p_s (J/m2/s) heat storage of water from Eq 6-14a and 6-14b of Jensen and Allen (2015) 
                    HeatStorage_Water_J_p_m2_p_s = HeatStorage_Coefficient_a * (weatherData[i].Radiation_Shortwave_Direct_Wpm2 + weatherData[i].Radiation_Shortwave_Diffuse_Wpm2) -
                        HeatStorage_Coefficient_b * weatherData[i].Radiation_Longwave_Upwelling_Wpm2;

                    //PeGrMh (m/h), evaporation from water surface on ground, from Eq Eq 13.1 of Chin (2021), Penman Monteith Evaporation method
                    //Note: Equation 13.1 of Chin (2021) requires all terms are in fundamental SI units (J, Pa, kg, s, m, C, etc.)
                    //Note: Issues with Eq 4.2.27 of Shuttleworth (1993) include not dividing by density of water, usng mixed units MJ/day, kPa/C, etc.
                    //Note: ResistanceSurface_Water_s_p_m set to zero. 
                    weatherData[i].PeGrMh = 1 / (LatentHeatOfVaporizaton_J_p_kg * DensityWater_kg_p_m3) *
                        ((VaporPressureGradient_Pa * (Radiation_Net_Wpm2 - HeatStorage_Water_J_p_m2_p_s)) +
                        (DensityAir_kg_p_m3 * SpecificHeat_MoistAir_J_p_kg_p_C * VaporPressure_Deficit_OverWater_Pa / ResistanceAerodynamic_SurfaceWater_s_p_m)) /
                        (VaporPressureGradient_Pa + PsychrometricConstant_Pa_p_C *
                        (1 + ResistanceSurface_Water_s_p_m / ResistanceAerodynamic_SurfaceWater_s_p_m));
                    //PeGrMh (m/h) converted from (m/s) with Ratio_sec_to_hr
                    weatherData[i].PeGrMh = weatherData[i].PeGrMh * Ratio_sec_to_hr;
                    //If negative pet to zero
                    if (weatherData[i].PeGrMh < 0) { weatherData[i].PeGrMh = 0.0; }

                    //PeTrMh (m/s), evaporation from canopy, set equal to PeGrMh  (m/s)
                    //Note: Variable name contains Mh but it was previously converted to m/s
                    weatherData[i].PeTrMh = weatherData[i].PeGrMh;

                    //PeSnGrMh (m/h), sublimation from ground, from Eq 2 of Lundberg et al. (1998), presented as Eq 13.1 of Chin (2021).
                    //Note: Eq 2 of Lundberg et al. (1998) has no rs/ra term in the denominator, which Eq 13.1 maintains, given surface resistance rs = 0
                    //Note: ResistanceAerodynamic_Snow_to_Rain_ratio term multiplied w to represent Lundberg et al. (1998) 10x factor from Eq 6 to 7
                    weatherData[i].PeSnGrMh = 1 / (LatentHeatOfVaporizaton_J_p_kg * DensityWater_kg_p_m3) *
                        ((VaporPressureGradient_Pa * (Radiation_Net_Wpm2 - HeatStorage_Water_J_p_m2_p_s)) +
                        (DensityAir_kg_p_m3 * SpecificHeat_MoistAir_J_p_kg_p_C * VaporPressure_Deficit_Pa /
                        (ResistanceAerodynamic_SnowGround_s_p_m * ResistanceAerodynamic_Snow_to_Rain_ratio))) /
                        (VaporPressureGradient_Pa + PsychrometricConstant_Pa_p_C *
                        (1 + ResistanceSurface_Water_s_p_m / ResistanceAerodynamic_SnowGround_s_p_m));
                    //PeSnGrMh (m/h) converted from (m/s) with Ratio_sec_to_hr
                    weatherData[i].PeSnGrMh = weatherData[i].PeSnGrMh * Ratio_sec_to_hr;
                    //If negative pet to zero
                    if (weatherData[i].PeSnGrMh < 0) { weatherData[i].PeSnGrMh = 0.0; }

                    //PeSnTrMh (m/h), sublimation from canopy, from Eq 2 of Lundberg et al. (1998), presented as Eq 13.1 of Chin (2021).
                    //Note: Eq 2 of Lundberg et al. (1998) has no rs/ra term in the denominator, which Eq 13.1 maintains, given surface resistance rs = 0
                    //Note: ResistanceAerodynamic_Snow_to_Rain_ratio term multiplied w to represent Lundberg et al. (1998) 10x factor from Eq 6 to 7
                    //Note: Theory: COMET UCAR Snowmelt Processes International Edition states: Snow on vegetation is exposed to ...
                    //Note: ... more wind and sun than snow on the ground and has a higher surface area to mass ratio, thereby ...
                    //Note: ... making it more prone to sublimation and/or melting.
                    weatherData[i].PeSnTrMh = 1 / (LatentHeatOfVaporizaton_J_p_kg * DensityWater_kg_p_m3) *
                        ((VaporPressureGradient_Pa * (Radiation_Net_Wpm2 - HeatStorage_Water_J_p_m2_p_s)) +
                        (DensityAir_kg_p_m3 * SpecificHeat_MoistAir_J_p_kg_p_C * VaporPressure_Deficit_Pa /
                        (ResistanceAerodynamic_SnowTreeCanopy_s_p_m * ResistanceAerodynamic_Snow_to_Rain_ratio))) /
                        (VaporPressureGradient_Pa + PsychrometricConstant_Pa_p_C *
                        (1 + ResistanceSurface_Water_s_p_m / ResistanceAerodynamic_SnowTreeCanopy_s_p_m));
                    //PeSnTrMh (m/h) converted from (m/s) with Ratio_sec_to_hr
                    weatherData[i].PeSnTrMh = weatherData[i].PeSnTrMh * Ratio_sec_to_hr;
                    //If negative pet to zero
                    if (weatherData[i].PeSnTrMh < 0) { weatherData[i].PeSnTrMh = 0.0; }
                }
            }
            catch (Exception)
            {
                throw;
            }
        }
        public static void CalcPrecipInterceptByCanopy(List<SurfaceWeather> wList, List<LeafAreaIndex> LeafAreaIndex_list, int recCnt, VEGTYPE vegType)
        {
            int i;
            double maxSt;       // maximum canopy storage of precipitation
            double canopyCover_frac;           // canopy cover fraction
            double potEvMh;     // potential evaporation for tree or ground (for vegType=Tree and Shrub, respectively)
            double canPrcp;     // precipitation fallen and contacted to canopy at the time step
            double freeThf;     // precipitation fallen through canopy without contact (free throughfall) at the time step
            double canDrip;     // drip of water from canopy in the 2nd stage
            double preSt;       // canopy storage of precipitation at the previous time step
            double preEv;       // evaporation from canopy at the previous time step

            for (i = 0; i < recCnt; i++)
            {
                // previous hour condition 
                preSt = (i == 0) ? 0 : wList[i - 1].VegStMh;
                preEv = (i == 0) ? 0 : wList[i - 1].VegEvMh;

                potEvMh = (vegType == VEGTYPE.TREE) ? wList[i].PeTrMh : wList[i].PeGrMh;
                canopyCover_frac = (vegType == VEGTYPE.TREE) ? 1 - Math.Exp(-0.7 * LeafAreaIndex_list[i].Lai) : 1 - Math.Exp(-0.3 * LeafAreaIndex_list[i].Lai);

                maxSt = LEAF_STORAGE_M * LeafAreaIndex_list[i].Lai;

                freeThf = wList[i].RainMh * (1 - canopyCover_frac);
                canPrcp = wList[i].RainMh - freeThf;

                // storage
                wList[i].VegStMh = preSt + canPrcp - preEv;
                if (wList[i].VegStMh > maxSt)
                {
                    wList[i].VegStMh = maxSt;
                }
                else if (wList[i].VegStMh < 0)
                {
                    wList[i].VegStMh = 0;
                }

                // evaporation
                if (maxSt == 0) // when evergreen % = 0, LAI = 0 in the winter and thus maxSt = 0
                {
                    wList[i].VegEvMh = 0;
                }
                else
                {
                    wList[i].VegEvMh = Math.Pow(wList[i].VegStMh / maxSt, 2.0 / 3.0) * potEvMh;
                    // cap evaporation with storage
                    if (wList[i].VegEvMh > wList[i].VegStMh)
                    {
                        wList[i].VegEvMh = wList[i].VegStMh;
                    }
                }

                // drip
                canDrip = 0;
                if (wList[i].VegStMh < maxSt)                     // 1st stage
                {
                    //                    wList[i].ThrufallMh = freeThf - wList[i].VegEvMh;   // should be this.
                    wList[i].UnderCanThrufallMh = freeThf;                       // but for compatibility with Hydro, use this
                }
                else
                {
                    if (preSt < maxSt)                          // 2nd stage for the first time
                    {
                        canDrip = canPrcp - (maxSt - preSt) - preEv;
                        if (canDrip < 0)
                        {
                            canDrip = 0;
                        }
                    }
                    else                                       // 2nd stage
                    {
                        canDrip = canPrcp - preEv;
                        if (canDrip < 0)
                        {
                            canDrip = 0;
                        }
                    }
                    wList[i].UnderCanThrufallMh = canDrip + freeThf;
                }

                // interception
                // While it's raining hourly interception is calculated from input - output of water
                // (i.e., Rain on canopy - Under canopy throughfall
                if (wList[i].RainInH > 0)
                {
                    wList[i].VegIntcptMh = wList[i].RainMh - wList[i].UnderCanThrufallMh;
                }
                else
                {
                    wList[i].VegIntcptMh = 0;
                }
            }
        }
        public static void CalcPrecipInterceptByUnderCanopyCover(List<SurfaceWeather> wList, int recCnt)
        {
            int i;
            double maxPervSt;       // maximum pervious cover storage of precipitation
            double maxImpervSt;     // maximum impervious cover storage of precipitation
            double potEvMh;         // potential evaporation for ground
            double prePervSt;       // pervious storage of precipitation at the previous time step
            double preImpervSt;     // impervious storage of precipitation at the previous time step
            double prePervEv;       // pervious evaporation at the previous time step
            double preImpervEv;     // impervious evaporation at the previous time step

            for (i = 0; i < recCnt; i++)
            {
                // previous hour condition 
                prePervSt = (i == 0) ? 0 : wList[i - 1].UnderCanPervStMh;
                preImpervSt = (i == 0) ? 0 : wList[i - 1].UnderCanImpervStMh;
                prePervEv = (i == 0) ? 0 : wList[i - 1].UnderCanPervEvMh;
                preImpervEv = (i == 0) ? 0 : wList[i - 1].UnderCanImpervEvMh;

                potEvMh = wList[i].PeGrMh;

                maxPervSt = PERV_STORAGE_M;
                maxImpervSt = IMPERV_STORAGE_M;

                // storage
                //  pervious cover
                wList[i].UnderCanPervStMh = prePervSt + wList[i].UnderCanThrufallMh - prePervEv;
                if (wList[i].UnderCanPervStMh > maxPervSt)
                {
                    wList[i].UnderCanPervStMh = maxPervSt;
                }
                else if (wList[i].UnderCanPervStMh < 0)
                {
                    wList[i].UnderCanPervStMh = 0;
                }
                //  impervious cover
                wList[i].UnderCanImpervStMh = preImpervSt + wList[i].UnderCanThrufallMh - preImpervEv;
                if (wList[i].UnderCanImpervStMh > maxImpervSt)
                {
                    wList[i].UnderCanImpervStMh = maxImpervSt;
                }
                else if (wList[i].UnderCanImpervStMh < 0)
                {
                    wList[i].UnderCanImpervStMh = 0;
                }

                // evaporation
                //  pervious cover
                wList[i].UnderCanPervEvMh = (wList[i].UnderCanPervStMh / maxPervSt) * potEvMh;
                //  cap evaporation with storage
                if (wList[i].UnderCanPervEvMh > wList[i].UnderCanPervStMh)
                {
                    wList[i].UnderCanPervEvMh = wList[i].UnderCanPervStMh;
                }
                //  impervious cover
                wList[i].UnderCanImpervEvMh = (wList[i].UnderCanImpervStMh / maxImpervSt) * potEvMh;
                //  cap evaporation with storage
                if (wList[i].UnderCanImpervEvMh > wList[i].UnderCanImpervStMh)
                {
                    wList[i].UnderCanImpervEvMh = wList[i].UnderCanImpervStMh;
                }

                // runoff
                wList[i].UnderCanImpervRunoffMh = 0;
                if (preImpervSt < maxImpervSt)
                {
                    wList[i].UnderCanImpervRunoffMh = wList[i].UnderCanThrufallMh - (maxImpervSt - preImpervSt) - wList[i].UnderCanImpervEvMh;
                }
                else if (preImpervSt == maxImpervSt)
                {
                    wList[i].UnderCanImpervRunoffMh = wList[i].UnderCanThrufallMh - wList[i].UnderCanImpervEvMh;
                }
                if (wList[i].UnderCanImpervRunoffMh < 0)
                {
                    wList[i].UnderCanImpervRunoffMh = 0;
                }

                // infiltration
                wList[i].UnderCanPervInfilMh = 0;
                if (prePervSt < maxPervSt)
                {
                    wList[i].UnderCanPervInfilMh = wList[i].UnderCanThrufallMh - (maxPervSt - prePervSt) - wList[i].UnderCanPervEvMh;
                }
                else if (prePervSt == maxPervSt)
                {
                    wList[i].UnderCanPervInfilMh = wList[i].UnderCanThrufallMh - wList[i].UnderCanPervEvMh;
                }
                if (wList[i].UnderCanPervInfilMh < 0)
                {
                    wList[i].UnderCanPervInfilMh = 0;
                }
            }
        }
        public static void CalcPrecipInterceptByNoCanopyCover(List<SurfaceWeather> wList, int recCnt)
        {
            int i;
            double maxPervSt;       // maximum pervious cover storage of precipitation
            double maxImpervSt;     // maximum impervious cover storage of precipitation
            double potEvMh;         // potential evaporation for ground
            double prePervSt;       // pervious storage of precipitation at the previous time step
            double preImpervSt;     // impervious storage of precipitation at the previous time step
            double prePervEv;       // pervious evaporation at the previous time step
            double preImpervEv;     // impervious evaporation at the previous time step

            for (i = 0; i < recCnt; i++)
            {
                // previous hour condition 
                prePervSt = (i == 0) ? 0 : wList[i - 1].NoCanPervStMh;
                preImpervSt = (i == 0) ? 0 : wList[i - 1].NoCanImpervStMh;
                prePervEv = (i == 0) ? 0 : wList[i - 1].NoCanPervEvMh;
                preImpervEv = (i == 0) ? 0 : wList[i - 1].NoCanImpervEvMh;

                potEvMh = wList[i].PeGrMh;

                maxPervSt = PERV_STORAGE_M;
                maxImpervSt = IMPERV_STORAGE_M;

                // storage
                //  pervious cover
                wList[i].NoCanPervStMh = prePervSt + wList[i].RainMh - prePervEv;
                if (wList[i].NoCanPervStMh > maxPervSt)
                {
                    wList[i].NoCanPervStMh = maxPervSt;
                }
                else if (wList[i].NoCanPervStMh < 0)
                {
                    wList[i].NoCanPervStMh = 0;
                }
                //  impervious cover
                wList[i].NoCanImpervStMh = preImpervSt + wList[i].RainMh - preImpervEv;
                if (wList[i].NoCanImpervStMh > maxImpervSt)
                {
                    wList[i].NoCanImpervStMh = maxImpervSt;
                }
                else if (wList[i].NoCanImpervStMh < 0)
                {
                    wList[i].NoCanImpervStMh = 0;
                }

                // evaporation
                //  pervious cover
                wList[i].NoCanPervEvMh = (wList[i].NoCanPervStMh / maxPervSt) * potEvMh;
                //  cap evaporation with storage
                if (wList[i].NoCanPervEvMh > wList[i].NoCanPervStMh)
                {
                    wList[i].NoCanPervEvMh = wList[i].NoCanPervStMh;
                }
                //  impervious cover
                wList[i].NoCanImpervEvMh = (wList[i].NoCanImpervStMh / maxImpervSt) * potEvMh;
                //  cap evaporation with storage
                if (wList[i].NoCanImpervEvMh > wList[i].NoCanImpervStMh)
                {
                    wList[i].NoCanImpervEvMh = wList[i].NoCanImpervStMh;
                }

                // runoff
                wList[i].NoCanImpervRunoffMh = 0;
                if (preImpervSt < maxImpervSt)
                {
                    wList[i].NoCanImpervRunoffMh = wList[i].RainMh - (maxImpervSt - preImpervSt) - wList[i].NoCanImpervEvMh;
                }
                else if (preImpervSt == maxImpervSt)
                {
                    wList[i].NoCanImpervRunoffMh = wList[i].RainMh - wList[i].NoCanImpervEvMh;
                }
                if (wList[i].NoCanImpervRunoffMh < 0)
                {
                    wList[i].NoCanImpervRunoffMh = 0;
                }

                // infiltration
                wList[i].NoCanPervInfilMh = 0;
                if (prePervSt < maxPervSt)
                {
                    wList[i].NoCanPervInfilMh = wList[i].RainMh - (maxPervSt - prePervSt) - wList[i].NoCanPervEvMh;
                }
                else if (prePervSt == maxPervSt)
                {
                    wList[i].NoCanPervInfilMh = wList[i].RainMh - wList[i].NoCanPervEvMh;
                }
                if (wList[i].NoCanPervInfilMh < 0)
                {
                    wList[i].NoCanPervInfilMh = 0;
                }
            }
        }
        public static void CopySurfaceWeatherData(string inFile, string outFile, bool blCopy)
        {
            StreamReader sr;
            StreamWriter sw;
            string lineOrg = "";
            string line = "";
            WeatherDataFormat weaFormat;

            using (sr = new StreamReader(inFile))
            {
                weaFormat = CheckWeatherDataFormat(inFile);
                try
                {
                    //First file to be copied: copy the entire input file to the output file.
                    if (blCopy)
                    {
                        using (sw = new StreamWriter(outFile, false))
                        {
                            while ((lineOrg = sr.ReadLine()) != null)
                            {
                                switch (weaFormat)
                                {
                                    case WeatherDataFormat.NEWINTL:
                                        line = lineOrg.Substring(0, 66) + lineOrg.Substring(75, 66);
                                        break;
                                    case WeatherDataFormat.NEWUSCAN:
                                        line = lineOrg.Substring(0, 66) + lineOrg.Substring(81, 66);
                                        break;
                                    case WeatherDataFormat.OLD:
                                        line = lineOrg;
                                        break;
                                    case WeatherDataFormat.NARCCAP:
                                        line = lineOrg;
                                        break;
                                    default:
                                        break;
                                }
                                sw.WriteLine(line);
                            }
                        }
                    }
                    //Append the file by excluding the first header line.
                    else
                    {
                        using (sw = new StreamWriter(outFile, true))
                        {
                            //Read the header line
                            lineOrg = sr.ReadLine();
                            while ((lineOrg = sr.ReadLine()) != null)
                            {
                                switch (weaFormat)
                                {
                                    case WeatherDataFormat.NEWINTL:
                                        line = lineOrg.Substring(0, 66) + lineOrg.Substring(75, 66);
                                        break;
                                    case WeatherDataFormat.NEWUSCAN:
                                        line = lineOrg.Substring(0, 66) + lineOrg.Substring(81, 66);
                                        break;
                                    case WeatherDataFormat.OLD:
                                        line = lineOrg;
                                        break;
                                    default:
                                        break;
                                }
                                sw.WriteLine(line);
                            }
                        }
                    }
                }
                catch (Exception)
                {
                    throw;
                }
            }
        }
        public static void WriteSurfaceWeatherRecords(string WeatherVariables_dataBase, List<SurfaceWeather> WeatherVariables_list, int recCnt)
        {
            //enoughForEdgeCheck checks for the intended 1-day timezone edge cases
            //Note: Ensure we actually have at least ~2 days of data
            bool enoughForEdgeCheck = WeatherVariables_list != null && WeatherVariables_list.Count >= 48;

            //Flag_Date_January01_Missing is true if the first item in WeatherVariables_list[0] is not Month=1 and Day=1, yet January 2 exists
            //Note: This multi-conditional ensures that if January 2 exists and January 1 doesn't safer to assume January 1 is missing
            bool Flag_Date_January01_Missing =
                enoughForEdgeCheck &&
                WeatherVariables_list[0].TimeStamp.Month == 1 &&
                WeatherVariables_list[0].TimeStamp.Day == 2;

            //Flag_Date_December31_Missing is true if the last item in WeatherVariables_list[WeatherVariables_list.Count-1] is not Month=12 and Day=31, December 30 exists
            //Note: This multi-conditional ensures that if December 30 exists and December 31 doesn't safer to assume December 31 is missing
            bool Flag_Date_December31_Missing =
                enoughForEdgeCheck &&
                WeatherVariables_list[WeatherVariables_list.Count - 1].TimeStamp.Month == 12 &&
                WeatherVariables_list[WeatherVariables_list.Count - 1].TimeStamp.Day == 30;

            try
            {
                string[] weaStrings = new string[WeatherVariables_list.Count + 1]; //String array to be written to Meteo.csv
                weaStrings[0] = "TimeStamp,Radiation_Longwave_Downwelling_Wpm2,Radiation_Longwave_Upwelling_Wpm2,LatRad,HourAngleRad,DecAngleRad,AirDens_kg_p_m3," +
                    "airMass_relativeOptical_kg_p_m2,CeilingHeight_hundredths_ft,CloudCover_Total_tenths,CloudCover_Opaque_tenths,CloudCover_Translucent_tenths,PrsIn,PrsKPa,PrsMBar,PeTrMh,PeGrMh," +
                    "PeSnTrMh,PeSnGrMh,PtTrMh,VegEvMh,VegStMh,VegIntcptMh,UnderCanThrufallMh," +
                    "UnderCanPervEvMh,UnderCanPervStMh,UnderCanPervInfilMh,UnderCanImpervEvMh," +
                    "UnderCanImpervStMh,UnderCanImpervRunoffMh,NoCanPervEvMh,NoCanPervStMh," +
                    "NoCanPervInfilMh,NoCanImpervEvMh,NoCanImpervStMh,NoCanImpervRunoffMh,Radiation_Shortwave_Direct_Wpm2," +
                    "Radiation_Shortwave_Diffuse_Wpm2,Radiation_Shortwave_Total_Wpm2,PARWm2,PARuEm2s,Radiation_Net_Wpm2,RainInH,RainMh,RelHum,SatVPrsKPa," +
                    "SnowIn,SnowM,SolZenAgl_deg,TempC,TempF,TempK,DewTempC,DewTempF,VapPrsKPa,WdDir,WdSpdMh," +
                    "WdSpdMs,WdSpdKnt"; //CSV header
                for (int i = 0; i < WeatherVariables_list.Count; i++)
                {
                    string[] rowString = new string[59]; //String array for new row in Meteo.csv
                    rowString[0] = Convert.ToString(WeatherVariables_list[i].TimeStamp);
                    rowString[1] = Convert.ToString(WeatherVariables_list[i].Radiation_Longwave_Downwelling_Wpm2);
                    rowString[2] = Convert.ToString(WeatherVariables_list[i].Radiation_Longwave_Upwelling_Wpm2);
                    rowString[3] = Convert.ToString(WeatherVariables_list[i].LatRad);
                    rowString[4] = Convert.ToString(WeatherVariables_list[i].HourAngleRad);
                    rowString[5] = Convert.ToString(WeatherVariables_list[i].DecAngleRad);
                    rowString[6] = Convert.ToString(WeatherVariables_list[i].AirDens_kg_p_m3);
                    rowString[7] = Convert.ToString(WeatherVariables_list[i].airMass_relativeOptical_kg_p_m2);
                    rowString[8] = Convert.ToString(WeatherVariables_list[i].CeilingHeight_hundredths_ft);
                    rowString[9] = Convert.ToString(WeatherVariables_list[i].CloudCover_Total_tenths);
                    rowString[10] = Convert.ToString(WeatherVariables_list[i].CloudCover_Opaque_tenths);
                    rowString[11] = Convert.ToString(WeatherVariables_list[i].CloudCover_Translucent_tenths);
                    rowString[12] = Convert.ToString(WeatherVariables_list[i].PrsIn);
                    rowString[13] = Convert.ToString(WeatherVariables_list[i].PrsKpa);
                    rowString[14] = Convert.ToString(WeatherVariables_list[i].PrsMbar);
                    rowString[15] = Convert.ToString(WeatherVariables_list[i].PeTrMh);
                    rowString[16] = Convert.ToString(WeatherVariables_list[i].PeGrMh);
                    rowString[17] = Convert.ToString(WeatherVariables_list[i].PeSnTrMh);
                    rowString[18] = Convert.ToString(WeatherVariables_list[i].PeSnGrMh);
                    rowString[19] = Convert.ToString(WeatherVariables_list[i].PtTrMh);
                    rowString[20] = Convert.ToString(WeatherVariables_list[i].VegEvMh);
                    rowString[21] = Convert.ToString(WeatherVariables_list[i].VegStMh);
                    rowString[22] = Convert.ToString(WeatherVariables_list[i].VegIntcptMh);
                    rowString[23] = Convert.ToString(WeatherVariables_list[i].UnderCanThrufallMh);
                    rowString[24] = Convert.ToString(WeatherVariables_list[i].UnderCanPervEvMh);
                    rowString[25] = Convert.ToString(WeatherVariables_list[i].UnderCanPervStMh);
                    rowString[26] = Convert.ToString(WeatherVariables_list[i].UnderCanPervInfilMh);
                    rowString[27] = Convert.ToString(WeatherVariables_list[i].UnderCanImpervEvMh);
                    rowString[28] = Convert.ToString(WeatherVariables_list[i].UnderCanImpervStMh);
                    rowString[29] = Convert.ToString(WeatherVariables_list[i].UnderCanImpervRunoffMh);
                    rowString[30] = Convert.ToString(WeatherVariables_list[i].NoCanPervEvMh);
                    rowString[31] = Convert.ToString(WeatherVariables_list[i].NoCanPervStMh);
                    rowString[32] = Convert.ToString(WeatherVariables_list[i].NoCanPervInfilMh);
                    rowString[33] = Convert.ToString(WeatherVariables_list[i].NoCanImpervEvMh);
                    rowString[34] = Convert.ToString(WeatherVariables_list[i].NoCanImpervStMh);
                    rowString[35] = Convert.ToString(WeatherVariables_list[i].NoCanImpervRunoffMh);
                    rowString[36] = Convert.ToString(WeatherVariables_list[i].Radiation_Shortwave_Direct_Wpm2);
                    rowString[37] = Convert.ToString(WeatherVariables_list[i].Radiation_Shortwave_Diffuse_Wpm2);
                    rowString[38] = Convert.ToString(WeatherVariables_list[i].Radiation_Shortwave_Total_Wpm2);
                    rowString[39] = Convert.ToString(WeatherVariables_list[i].PARWm2);
                    rowString[40] = Convert.ToString(WeatherVariables_list[i].PARuEm2s);
                    rowString[41] = Convert.ToString(WeatherVariables_list[i].Radiation_Net_Wpm2);
                    rowString[42] = Convert.ToString(WeatherVariables_list[i].RainInH);
                    rowString[43] = Convert.ToString(WeatherVariables_list[i].RainMh);
                    rowString[44] = Convert.ToString(WeatherVariables_list[i].RelHum);
                    rowString[45] = Convert.ToString(WeatherVariables_list[i].SatVPrsKpa);
                    rowString[46] = Convert.ToString(WeatherVariables_list[i].SnowIn);
                    rowString[47] = Convert.ToString(WeatherVariables_list[i].SnowM);
                    rowString[48] = Convert.ToString(WeatherVariables_list[i].SolZenAgl_deg);
                    rowString[49] = Convert.ToString(WeatherVariables_list[i].TempC);
                    rowString[50] = Convert.ToString(WeatherVariables_list[i].TempF);
                    rowString[51] = Convert.ToString(WeatherVariables_list[i].TempK);
                    rowString[52] = Convert.ToString(WeatherVariables_list[i].DewTempC);
                    rowString[53] = Convert.ToString(WeatherVariables_list[i].DewTempF);
                    rowString[54] = Convert.ToString(WeatherVariables_list[i].VapPrsKpa);
                    rowString[55] = Convert.ToString(WeatherVariables_list[i].WdDir);
                    rowString[56] = Convert.ToString(WeatherVariables_list[i].WdSpdMh);
                    rowString[57] = Convert.ToString(WeatherVariables_list[i].WdSpdMs);
                    rowString[58] = Convert.ToString(WeatherVariables_list[i].WdSpdKnt);
                    weaStrings[i + 1] = String.Join(",", rowString); //Join the segments of the new row, then add new row to weaStrings
                }
                string csv = String.Join("\r\n", weaStrings); //Join all CSV rows into one string, with comma + newline as separator
                File.WriteAllText(WeatherVariables_dataBase, csv);
            }
            catch (Exception)
            {
                throw;
            }
        }
        public static int ReadSurfaceAllRecords(string WeatherVariables_dataBase, ref List<SurfaceWeather> sfcData)
        {
            string line;
            SurfaceWeather data;

            using (StreamReader sr = new StreamReader(WeatherVariables_dataBase))
            {
                try
                {
                    //read the header line
                    sr.ReadLine();

                    while ((line = sr.ReadLine()) != null)
                    {
                        data = new SurfaceWeather();
                        data.TimeStamp = DateTime.Parse(line.Split(',')[0]);
                        data.Radiation_Longwave_Downwelling_Wpm2 = Double.Parse(line.Split(',')[1]);
                        data.Radiation_Longwave_Upwelling_Wpm2 = Double.Parse(line.Split(',')[2]);
                        data.LatRad = Double.Parse(line.Split(',')[3]);
                        data.HourAngleRad = Double.Parse(line.Split(',')[4]);
                        data.DecAngleRad = Double.Parse(line.Split(',')[5]);
                        data.AirDens_kg_p_m3 = Double.Parse(line.Split(',')[6]);
                        data.airMass_relativeOptical_kg_p_m2 = Double.Parse(line.Split(',')[7]);
                        data.CeilingHeight_hundredths_ft = Double.Parse(line.Split(',')[8]);
                        data.CloudCover_Total_tenths = Double.Parse(line.Split(',')[9]);
                        data.CloudCover_Opaque_tenths = Double.Parse(line.Split(',')[10]);
                        data.CloudCover_Translucent_tenths = Double.Parse(line.Split(',')[11]);
                        data.PrsIn = Double.Parse(line.Split(',')[12]);
                        data.PrsKpa = Double.Parse(line.Split(',')[13]);
                        data.PrsMbar = Double.Parse(line.Split(',')[14]);
                        data.PeTrMh = Double.Parse(line.Split(',')[15]);
                        data.PeGrMh = Double.Parse(line.Split(',')[16]);
                        data.PeSnTrMh = Double.Parse(line.Split(',')[17]);
                        data.PeSnGrMh = Double.Parse(line.Split(',')[18]);
                        data.PtTrMh = Double.Parse(line.Split(',')[19]);
                        data.VegEvMh = Double.Parse(line.Split(',')[20]);
                        data.VegStMh = Double.Parse(line.Split(',')[21]);
                        data.VegIntcptMh = Double.Parse(line.Split(',')[22]);
                        data.UnderCanThrufallMh = Double.Parse(line.Split(',')[23]);
                        data.UnderCanPervEvMh = Double.Parse(line.Split(',')[24]);
                        data.UnderCanPervStMh = Double.Parse(line.Split(',')[25]);
                        data.UnderCanPervInfilMh = Double.Parse(line.Split(',')[26]);
                        data.UnderCanImpervEvMh = Double.Parse(line.Split(',')[27]);
                        data.UnderCanImpervStMh = Double.Parse(line.Split(',')[28]);
                        data.UnderCanImpervRunoffMh = Double.Parse(line.Split(',')[29]);
                        data.NoCanPervEvMh = Double.Parse(line.Split(',')[30]);
                        data.NoCanPervStMh = Double.Parse(line.Split(',')[31]);
                        data.NoCanPervInfilMh = Double.Parse(line.Split(',')[32]);
                        data.NoCanImpervEvMh = Double.Parse(line.Split(',')[33]);
                        data.NoCanImpervStMh = Double.Parse(line.Split(',')[34]);
                        data.NoCanImpervRunoffMh = Double.Parse(line.Split(',')[35]);
                        data.Radiation_Shortwave_Direct_Wpm2 = Double.Parse(line.Split(',')[36]);
                        data.Radiation_Shortwave_Diffuse_Wpm2 = Double.Parse(line.Split(',')[37]);
                        data.Radiation_Shortwave_Total_Wpm2 = Double.Parse(line.Split(',')[38]);
                        data.PARWm2 = Double.Parse(line.Split(',')[39]);
                        data.PARuEm2s = Double.Parse(line.Split(',')[40]);
                        data.Radiation_Net_Wpm2 = Double.Parse(line.Split(',')[41]);
                        data.RainInH = Double.Parse(line.Split(',')[42]);
                        data.RainMh = Double.Parse(line.Split(',')[43]);
                        data.RelHum = Double.Parse(line.Split(',')[44]);
                        data.SatVPrsKpa = Double.Parse(line.Split(',')[45]);
                        data.SnowIn = Double.Parse(line.Split(',')[46]);
                        data.SnowM = Double.Parse(line.Split(',')[47]);
                        data.SolZenAgl_deg = Double.Parse(line.Split(',')[48]);
                        data.TempC = Double.Parse(line.Split(',')[49]);
                        data.TempF = Double.Parse(line.Split(',')[50]);
                        data.TempK = Double.Parse(line.Split(',')[51]);
                        data.DewTempC = Double.Parse(line.Split(',')[52]);
                        data.DewTempF = Double.Parse(line.Split(',')[53]);
                        data.VapPrsKpa = Double.Parse(line.Split(',')[54]);
                        data.WdDir = Double.Parse(line.Split(',')[55]);
                        data.WdSpdMh = Double.Parse(line.Split(',')[56]);
                        data.WdSpdMs = Double.Parse(line.Split(',')[57]);
                        data.WdSpdKnt = Double.Parse(line.Split(',')[58]);
                        sfcData.Add(data);
                    }
                    return (sfcData.Count);
                }
                catch (Exception)
                {
                    throw;
                }
            }
        }
    }
}
