#GUI_WeatherPrep.py created by Jay Heppler (Davey Tree) with assist from Li Zhang (SUNY ESF)
#Code provides a GUI to orchestrate three steps in i-Tree Tools HydroPlus weather processing:
#   1. Retrieving NOAA weather data by FTP to obtain a gun-zipped USAF-WBAN file
#   2. Converting NOAA weather data with ishapp2.exe from stage 1 to stage 2
#   3. Processing NOAA weather data to data formatted for HydroPlus tools


from PyQt5 import QtWidgets, QtCore, QtGui
import os
import sys
import gzip
import ftplib
from shutil import copy2
import subprocess
from lxml import etree as ET
import shutil
import pandas as pd
from datetime import datetime
from PyQt5.QtCore import QObject, pyqtSignal, QThread, Qt
from PyQt5.QtGui import QPixmap, QIcon, QPainter, QColor, QFont
from PyQt5.QtWidgets import QSplashScreen, QApplication, QTabWidget, QFileDialog
from subprocess import STARTUPINFO, STARTF_USESHOWWINDOW, SW_HIDE
import re
from lxml import etree as ET

<<<<<<< .mine
#Indexing to handle nested XML elements
_INDEX_RE = re.compile(r"\[\d+\]")
||||||| .r212
=======
#Indexing to handle nested XML elements
_INDEX_RE = re.compile(r"\[\d+\]")

>>>>>>> .r216
# Get the directory where the script is located
script_dir = os.path.dirname(os.path.abspath(sys.argv[0]))

# Go up one directory to access all other WeatherPrep tools
root_dir = os.path.dirname(script_dir)

# Set the current working directory to the script directory
os.chdir(root_dir)
#def to manage copying comments between XML files
def _indexless_xpath(tree, elem):
    return _INDEX_RE.sub("", tree.getpath(elem))

<<<<<<< .mine
def _element_keys(elem, tree):
    # Prefer stable id/name if present:
    for attr in ("id", "name"):
        if elem.get(attr) is not None:
            return (f"{elem.tag}|{attr}={elem.get(attr)}",)
    # Fallback: exact & relaxed absolute XPaths
    exact = tree.getpath(elem)
    relaxed = _indexless_xpath(tree, elem)
    return (exact,) if exact == relaxed else (exact, relaxed)

def _collect_adjacent_comments(parent, child):
    before, after = [], []
    kids = list(parent)
    try:
        i = kids.index(child)
    except ValueError:
        return before, after

    j = i - 1
    while j >= 0 and isinstance(kids[j], ET._Comment):
        txt = (kids[j].text or "").strip()
        if txt:
            before.append(txt)
        j -= 1
    before.reverse()

    j = i + 1
    while j < len(kids) and isinstance(kids[j], ET._Comment):
        txt = (kids[j].text or "").strip()
        if txt:
            after.append(txt)
        j += 1
    return before, after

def _build_comment_index(src_tree):
    idx = {}
    for e in src_tree.iter():
        if not isinstance(e.tag, str):   # skip comments/PIs
            continue
        p = e.getparent()
        if p is None:
            continue
        b, a = _collect_adjacent_comments(p, e)
        if not b and not a:
            continue
        for k in _element_keys(e, src_tree):
            if k not in idx:
                idx[k] = {"before": b, "after": a}
    return idx

def _indent_for(node):
    depth = 0
    p = node.getparent()
    while p is not None:
        depth += 1
        p = p.getparent()
    return "\n" + ("  " * depth)

def _ensure_comments(parent, child, before_comments, after_comments, style="inline_singleline"):
    # existing_before/after as you have now
    existing_before, existing_after = _collect_adjacent_comments(parent, child)

    # ---------- BEFORE (unchanged from your current code) ----------
    insert_pos = list(parent).index(child)
    leading_ws = child.tail if (child.tail is not None and child.tail.strip() == "") else _indent_for(child)
    for txt in before_comments:
        if txt in existing_before:
            continue
        c = ET.Comment(txt)
        c.tail = leading_ws
        parent.insert(insert_pos, c)
        insert_pos += 1
        existing_before.append(txt)

    # ---------- AFTER (new modes) ----------
    kids = list(parent)
    insert_pos = kids.index(child) + 1

    if style == "inline":
        # glue comment directly after </child>, then keep next element on its own line
        for txt in after_comments:
            if txt in existing_after:
                continue
            original_tail = child.tail or _indent_for(child)
            child.tail = ""     # no whitespace between </child> and <!-- ... -->
            c = ET.Comment(txt)
            c.tail = original_tail
            parent.insert(insert_pos, c)
            insert_pos += 1
            existing_after.append(txt)

    elif style == "inline_singleline":
        # </child> <!-- comment -->\n  <next ...>
        # ensure next element gets newline+indent
        next_line = child.tail if (child.tail and "\n" in child.tail) else _indent_for(child)
        for txt in after_comments:
            if txt in existing_after:
                continue
            # Put a space before the comment so it reads nicely on same line
            child.tail = " "    # single space before comment on the same line
            c = ET.Comment(txt)
            c.tail = next_line  # newline+indent for the next element
            parent.insert(insert_pos, c)
            insert_pos += 1
            existing_after.append(txt)

    else:  # "block"
        # </child>\n  <!-- comment -->\n  <next ...>
        if not child.tail or "\n" not in child.tail:
            child.tail = _indent_for(child)
        for txt in after_comments:
            if txt in existing_after:
                continue
            c = ET.Comment(txt)
            c.tail = child.tail
            parent.insert(insert_pos, c)
            insert_pos += 1
            existing_after.append(txt)

def _remove_all_comments_preserve_ws(tree):
    root = tree.getroot()
    to_remove = [n for n in root.iter() if isinstance(n, ET._Comment)]
    for c in to_remove:
        tail = c.tail or ""
        p = c.getparent()
        if p is None:
            continue
        kids = list(p)
        idx = kids.index(c)
        if idx > 0:
            prev = kids[idx - 1]
            prev.tail = (prev.tail or "") + tail
        else:
            p.text = (p.text or "") + tail
        p.remove(c)

def copy_comments_from_template(template_path, dst_tree, clear_existing=False):
    """
    Copy adjacent comments from template xml onto dst_tree.
    Matches by: id/name, exact xpath, relaxed xpath.
    """
    parser_src = ET.XMLParser(remove_blank_text=False, remove_comments=False)
    src_tree = ET.parse(template_path, parser_src)

    if clear_existing:
        _remove_all_comments_preserve_ws(dst_tree)

    idx = _build_comment_index(src_tree)

    count = 0
    for e in dst_tree.iter():
        if not isinstance(e.tag, str):
            continue
        p = e.getparent()
        if p is None:
            continue
        for k in _element_keys(e, dst_tree):
            if k in idx:
                b = idx[k]["before"]; a = idx[k]["after"]
                if b or a:
                    _ensure_comments(p, e, b, a)
                    count += len(b) + len(a)
                break
    return count

||||||| .r212
=======
#def to manage copying comments between XML files
def _indexless_xpath(tree, elem):
    return _INDEX_RE.sub("", tree.getpath(elem))

def _element_keys(elem, tree):
    # Prefer stable id/name if present:
    for attr in ("id", "name"):
        if elem.get(attr) is not None:
            return (f"{elem.tag}|{attr}={elem.get(attr)}",)
    # Fallback: exact & relaxed absolute XPaths
    exact = tree.getpath(elem)
    relaxed = _indexless_xpath(tree, elem)
    return (exact,) if exact == relaxed else (exact, relaxed)

def _collect_adjacent_comments(parent, child):
    before, after = [], []
    kids = list(parent)
    try:
        i = kids.index(child)
    except ValueError:
        return before, after

    j = i - 1
    while j >= 0 and isinstance(kids[j], ET._Comment):
        txt = (kids[j].text or "").strip()
        if txt:
            before.append(txt)
        j -= 1
    before.reverse()

    j = i + 1
    while j < len(kids) and isinstance(kids[j], ET._Comment):
        txt = (kids[j].text or "").strip()
        if txt:
            after.append(txt)
        j += 1
    return before, after

def _build_comment_index(src_tree):
    idx = {}
    for e in src_tree.iter():
        if not isinstance(e.tag, str):   # skip comments/PIs
            continue
        p = e.getparent()
        if p is None:
            continue
        b, a = _collect_adjacent_comments(p, e)
        if not b and not a:
            continue
        for k in _element_keys(e, src_tree):
            if k not in idx:
                idx[k] = {"before": b, "after": a}
    return idx

def _indent_for(node):
    depth = 0
    p = node.getparent()
    while p is not None:
        depth += 1
        p = p.getparent()
    return "\n" + ("  " * depth)

def _ensure_comments(parent, child, before_comments, after_comments, style="inline_singleline"):
    # existing_before/after as you have now
    existing_before, existing_after = _collect_adjacent_comments(parent, child)

    # ---------- BEFORE (unchanged from your current code) ----------
    insert_pos = list(parent).index(child)
    leading_ws = child.tail if (child.tail is not None and child.tail.strip() == "") else _indent_for(child)
    for txt in before_comments:
        if txt in existing_before:
            continue
        c = ET.Comment(txt)
        c.tail = leading_ws
        parent.insert(insert_pos, c)
        insert_pos += 1
        existing_before.append(txt)

    # ---------- AFTER (new modes) ----------
    kids = list(parent)
    insert_pos = kids.index(child) + 1

    if style == "inline":
        # glue comment directly after </child>, then keep next element on its own line
        for txt in after_comments:
            if txt in existing_after:
                continue
            original_tail = child.tail or _indent_for(child)
            child.tail = ""     # no whitespace between </child> and <!-- ... -->
            c = ET.Comment(txt)
            c.tail = original_tail
            parent.insert(insert_pos, c)
            insert_pos += 1
            existing_after.append(txt)

    elif style == "inline_singleline":
        # </child> <!-- comment -->\n  <next ...>
        # ensure next element gets newline+indent
        next_line = child.tail if (child.tail and "\n" in child.tail) else _indent_for(child)
        for txt in after_comments:
            if txt in existing_after:
                continue
            # Put a space before the comment so it reads nicely on same line
            child.tail = " "    # single space before comment on the same line
            c = ET.Comment(txt)
            c.tail = next_line  # newline+indent for the next element
            parent.insert(insert_pos, c)
            insert_pos += 1
            existing_after.append(txt)

    else:  # "block"
        # </child>\n  <!-- comment -->\n  <next ...>
        if not child.tail or "\n" not in child.tail:
            child.tail = _indent_for(child)
        for txt in after_comments:
            if txt in existing_after:
                continue
            c = ET.Comment(txt)
            c.tail = child.tail
            parent.insert(insert_pos, c)
            insert_pos += 1
            existing_after.append(txt)

def _remove_all_comments_preserve_ws(tree):
    root = tree.getroot()
    to_remove = [n for n in root.iter() if isinstance(n, ET._Comment)]
    for c in to_remove:
        tail = c.tail or ""
        p = c.getparent()
        if p is None:
            continue
        kids = list(p)
        idx = kids.index(c)
        if idx > 0:
            prev = kids[idx - 1]
            prev.tail = (prev.tail or "") + tail
        else:
            p.text = (p.text or "") + tail
        p.remove(c)

def copy_comments_from_template(template_path, dst_tree, clear_existing=False):
    """
    Copy adjacent comments from template xml onto dst_tree.
    Matches by: id/name, exact xpath, relaxed xpath.
    """
    parser_src = ET.XMLParser(remove_blank_text=False, remove_comments=False)
    src_tree = ET.parse(template_path, parser_src)

    if clear_existing:
        _remove_all_comments_preserve_ws(dst_tree)

    idx = _build_comment_index(src_tree)

    count = 0
    for e in dst_tree.iter():
        if not isinstance(e.tag, str):
            continue
        p = e.getparent()
        if p is None:
            continue
        for k in _element_keys(e, dst_tree):
            if k in idx:
                b = idx[k]["before"]; a = idx[k]["after"]
                if b or a:
                    _ensure_comments(p, e, b, a)
                    count += len(b) + len(a)
                break
    return count

>>>>>>> .r216
#~~~~~~~~~~~~~~~~~~ACCESSORIES TO MAIN APP~~~~~~~~~~~~~~
#LOADING SCREEN
class SplashScreen(QSplashScreen):
    def __init__(self):
        # Load the original logo image
        pix_map = os.path.join(script_dir, 'resources', 'iTreeLogo.png')
        logoPixmap = QPixmap(pix_map)
        #Create larger pixel map with margins to display text
        largerPixmap = QPixmap(logoPixmap.width() + 60, logoPixmap.height() + 90)  # Adding 60px margins
        largerPixmap.fill(QColor(Qt.white))  # Fill the pixmap with a white background

        # Draw the logo onto the center of the larger pixmap
        painter = QPainter(largerPixmap)
        painter.drawPixmap(30, 30, logoPixmap)  # Adjust these values if you need different margins
        painter.end()

        super().__init__(largerPixmap)
        self.setWindowIcon(QIcon(pix_map))  # Set the window icon to your logo

        # Customize the message font and color
        self.setFont(QFont('Arial', 12, QFont.Bold))  # Set the font and size for the message
        self.showMessage("i-Tree Research Suite - WeatherPrep\nLoading...",
                         int(QtCore.Qt.AlignBottom | QtCore.Qt.AlignCenter), QtCore.Qt.black)

#XML CONFIGURATION
class XMLConfigWindow(QtWidgets.QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.windSensorHeight_spinbox = None
        self.treeHeight_spinbox = None
        self.vegetation_type_dropdown = None
        self.evergreen_spinbox = None
        self.maxLAI_spinbox = None
        self.initUI()

    def open_file_dialog(self):
        options = QFileDialog.Options()
        fileName, _ = QFileDialog.getOpenFileName(self, "Select Precipitation File", "",
                                                  "All Files (*);;Text Files (*.txt)", options=options)
        if fileName:
            # Replace forward slashes with backslashes in the file path
            corrected_file_path = fileName.replace('/', '\\')
            self.precipitation_file_line_edit.setText(corrected_file_path)

    def initUI(self):
        self.setWindowTitle('Advanced XML Configuration')
        icon_path = os.path.join(script_dir, 'resources', 'iTree_transparent.ico')
        self.setWindowIcon(QIcon(icon_path))
        layout = QtWidgets.QGridLayout()

        # Toggle Switch for 'Hydro' and 'Energy'
        self.model_type = QtWidgets.QComboBox()
        self.model_type.addItems(['Hydro', 'Energy'])
        layout.addWidget(QtWidgets.QLabel('Model:'), 0, 0)  # Adding the label for the toggle
        layout.addWidget(self.model_type, 0, 1)

        # Maximum LAI
        layout.addWidget(QtWidgets.QLabel('Maximum LAI:'), 1, 0)  # Row 0, Column 0
        self.maxLAI_spinbox = QtWidgets.QDoubleSpinBox()
        self.maxLAI_spinbox.setRange(0, 10)
        self.maxLAI_spinbox.setDecimals(1)
        self.maxLAI_spinbox.setSingleStep(0.1)
        self.maxLAI_spinbox.setValue(5)
        layout.addWidget(self.maxLAI_spinbox, 1, 1)  # Row 0, Column 1

        # Evergreen (%)
        layout.addWidget(QtWidgets.QLabel('Evergreen (%):'), 2, 0)  # Row 1, Column 0
        self.evergreen_spinbox = QtWidgets.QSpinBox()
        self.evergreen_spinbox.setRange(0, 100)
        self.evergreen_spinbox.setValue(5)
        layout.addWidget(self.evergreen_spinbox, 2, 1)  # Row 1, Column 1

        # Vegetation Type
        layout.addWidget(QtWidgets.QLabel('Vegetation Type:'), 3, 0)  # Row 2, Column 0
        self.vegetation_type_dropdown = QtWidgets.QComboBox()
        self.vegetation_type_dropdown.addItems(['Tree', 'Shrub', 'Grass'])
        self.vegetation_type_dropdown.setCurrentIndex(self.vegetation_type_dropdown.findText('Tree'))
        layout.addWidget(self.vegetation_type_dropdown, 3, 1)  # Row 2, Column 1

        # Tree Height (m)
        layout.addWidget(QtWidgets.QLabel('Tree Height (m):'), 4, 0)  # Row 3, Column 0
        self.treeHeight_spinbox = QtWidgets.QSpinBox()
        self.treeHeight_spinbox.setRange(0, 50)
        self.treeHeight_spinbox.setValue(12)
        layout.addWidget(self.treeHeight_spinbox, 4, 1)  # Row 3, Column 1

        # Wind Sensor Height (m)
        layout.addWidget(QtWidgets.QLabel('Sensor Height (m):'), 5, 0)  # Row 4, Column 0
        self.windSensorHeight_spinbox = QtWidgets.QSpinBox()
        self.windSensorHeight_spinbox.setRange(0, 50)
        self.windSensorHeight_spinbox.setValue(10)
        layout.addWidget(self.windSensorHeight_spinbox, 5, 1)  # Row 4, Column 1

        # Precipitation File Selector
        layout.addWidget(QtWidgets.QLabel('Precipitation File:'), 6, 0)  # Adjusted for new row
        file_selector_layout = QtWidgets.QHBoxLayout()
        self.precipitation_file_line_edit = QtWidgets.QLineEdit()
        file_selector_button = QtWidgets.QPushButton('...')
        file_selector_button.clicked.connect(self.open_file_dialog)
        file_selector_layout.addWidget(self.precipitation_file_line_edit)
        file_selector_layout.addWidget(file_selector_button)
        layout.addLayout(file_selector_layout, 6, 1)  # Adjusted for new row

        self.setLayout(layout)


    def get_configuration_values(self):
        return {
            'model': self.model_type.currentText(),
            'max_lai': self.maxLAI_spinbox.value(),
            'evergreen_percent': self.evergreen_spinbox.value(),
            'vegetation_type': self.vegetation_type_dropdown.currentText(),
            'tree_height': self.treeHeight_spinbox.value(),
            'wind_sensor_height': self.windSensorHeight_spinbox.value(),
            'precip_file': self.precipitation_file_line_edit.text()
        }

#CUSTOM SPINBOX FOR SUB-HOURLY INTERVAL
class CustomSpinBox(QtWidgets.QSpinBox):
    def __init__(self, parent=None):
        super(CustomSpinBox, self).__init__(parent)
        self.setRange(1, 60)  # Setting a wider range initially
        self.valid_values = [1, 2, 3, 4, 5, 6, 10, 12, 15, 20, 30]
        self.setValue(2)

    def stepBy(self, steps):
        current_index = self.valid_values.index(self.value())
        new_index = max(0, min(current_index + steps, len(self.valid_values) - 1))
        self.setValue(self.valid_values[new_index])

class ResampleWindow(QtWidgets.QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Resample Timestep Options")
        self.main_layout = QtWidgets.QVBoxLayout(self)
        icon_path = os.path.join(script_dir, 'resources', 'iTree_transparent.ico')
        self.setWindowIcon(QtGui.QIcon(icon_path))

        # Info label below the header
        info_label = QtWidgets.QLabel("All unspecified dates will be preserved at a 1-hour timestep.")
        self.main_layout.addWidget(info_label)

        # Header Row
        header_layout = QtWidgets.QHBoxLayout()
        start_header = QtWidgets.QLabel("Start")
        start_header.setToolTip("Start YYYYMMDD for timestep")
        header_layout.addWidget(start_header)

        end_header = QtWidgets.QLabel("End")
        end_header.setToolTip("End YYYYMMDD for timestep")
        header_layout.addWidget(end_header)

        timestep_header = QtWidgets.QLabel("Timestep")
        timestep_header.setToolTip("Desired timestep in hours")
        header_layout.addWidget(timestep_header)

        header_layout.addWidget(QtWidgets.QLabel(""))
        self.main_layout.addLayout(header_layout)

        # Store the row widgets
        self.rows = []
        self.add_row()

    def add_row(self):
        row_layout = QtWidgets.QHBoxLayout()

        # Start Date Entry
        start_entry = QtWidgets.QLineEdit()
        start_entry.setPlaceholderText("YYYYMMDD")
        row_layout.addWidget(start_entry)

        # End Date Entry
        end_entry = QtWidgets.QLineEdit()
        end_entry.setPlaceholderText("YYYYMMDD")
        row_layout.addWidget(end_entry)

        # Timestep Entry (QLineEdit with validation)
        timestep_entry = QtWidgets.QLineEdit()
        timestep_entry.setPlaceholderText("1-24")
        timestep_entry.setValidator(TimestepValidator())  # Use the new validator
        row_layout.addWidget(timestep_entry)

        # Add Row Button
        add_button = QtWidgets.QPushButton("+")
        add_button.clicked.connect(self.add_row)
        row_layout.addWidget(add_button)

        # Add to main layout and rows list
        self.main_layout.addLayout(row_layout)
        self.rows.append((row_layout, start_entry, end_entry, timestep_entry, add_button))

        # Disable the previous add button
        if len(self.rows) > 1:
            self.rows[-2][4].setEnabled(False)

        start_entry.setFocus()  # Set focus to the new start_entry


# Custom QValidator for timestep input (1-24)
class TimestepValidator(QtGui.QValidator):
    def validate(self, input_str, pos):
        if not input_str:
            return QtGui.QValidator.Intermediate, input_str, pos  # Allow empty

        if input_str.isdigit():
            value = int(input_str)
            if 1 <= value <= 24:
                return QtGui.QValidator.Acceptable, input_str, pos
            else:
                return QtGui.QValidator.Invalid, input_str, pos
        else:
            return QtGui.QValidator.Invalid, input_str, pos


# ~~~~~~~~~~~~~WORKER CLASS FOR GUI MANAGEMENT ~~~~~~~~~~~~~~~~~~~
class Worker(QObject):
    update_console = pyqtSignal(str)
    finished = pyqtSignal()

    def __init__(self, params):
        super().__init__()
        self.params = params
        self.stop_requested = False

    # ~~~~~~~~~~~~~~~ WEATHER PROCESSING ~~~~~~~~~~~~~~~
    def download_noaa_data(self, usaf_wban, start_year, end_year):
        try:
            # FTP address and login
            ftp = ftplib.FTP('ftp.ncei.noaa.gov')
            ftp.login()
            # Navigate to present year
            for year in range(start_year, end_year + 1):
                ftp.cwd(f'/pub/data/noaa/{year}')
                file_name = f'{usaf_wban}-{year}.gz'
                # Open and write to local directory
                with open(file_name, 'wb') as f:
                    ftp.retrbinary(f'RETR {file_name}', f.write)
            self.flag_failure = False
            ftp.quit()
        except ftplib.error_perm as e:
            if "530" in str(e):
                self.update_console.emit(f"FTP Connection Error: {e}\nServer-side connection may be down. Please try again later.")
                self.flag_failure = True
            else:
                self.update_console.emit(f"FTP Connection Error: {e}")
                self.flag_failure = True

        except Exception as e:
            # Handle other exceptions
            self.update_console.emit(f"Error during download: {e}")
            self.flag_failure = True

    def unzip_gz_file(self, usaf_wban, start_year, end_year):
        for year in range(start_year, end_year + 1):
            file_name = f'{usaf_wban}-{year}.gz'
            # Unzip GZ to 'file' for input into ishapp
            with gzip.open(file_name, 'rb') as f_in:
                unzipped_file_name = file_name.replace('.gz', '')
                with open(unzipped_file_name, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)

    def run_ishapp2(self, ishapp_path, usaf_wban, start_year, end_year):
        for year in range(start_year, end_year + 1):
            input_file = f'{usaf_wban}-{year}'
            output_file = f'{usaf_wban}-{year}.txt'
            #cte print(f"ishapp_path = {ishapp_path}")\

            #Check if OS is Windows to hide ishapp window
            if sys.platform.startswith('win'):
                # Creating a STARTUPINFO object to modify the visibility of the subprocess
                startupinfo = STARTUPINFO()
                startupinfo.dwFlags |= STARTF_USESHOWWINDOW
                startupinfo.wShowWindow = SW_HIDE
                subprocess.run([ishapp_path, input_file, output_file],
                               stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL,
                               startupinfo=startupinfo)
            else:
                subprocess.run([ishapp_path, input_file, output_file],
                               stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
    def prepare_single_folder(self, wpc_path, output_folder, usaf_wban, start_year, end_year):
        start_year_str = str(start_year)
        end_year_str = str(end_year)
        folder_name = os.path.join(output_folder, f'{usaf_wban}', f'{start_year_str[2:4]}-{end_year_str[2:4]}')
        os.makedirs(folder_name, exist_ok=True)
        copy2(wpc_path, folder_name)

        # Check if start_year equals end_year, to determine if multiple years are being combined.
        if start_year != end_year:
            # Define the name for the combined file
            output_weather_txt = os.path.join(folder_name, f'{usaf_wban}-{start_year}-{end_year}.txt')
            with open(output_weather_txt, 'w') as output_multi:
                for year in range(start_year, end_year + 1):
                    yearly_weather = os.path.join(output_folder, f'{usaf_wban}-{year}.txt')
                    with open(yearly_weather, 'r') as year_file:
                        lines = year_file.readlines()

                        # If it's the first year, write all lines (including header)
                        if year == start_year:
                            output_multi.writelines(lines)
                        else:
                            # For subsequent years, skip the header (assuming the header is the first line)
                            output_multi.writelines(lines[1:])
            return folder_name

    def prepare_multi_folder(self, wpc_path, output_folder, usaf_wban, year):
        folder_name = os.path.join(output_folder + f'\\{usaf_wban}\\{year}')
        os.makedirs(folder_name, exist_ok=True)
        copy2(wpc_path, folder_name)
        copy2(os.path.join(output_folder + f'/{usaf_wban}-{year}.txt'), folder_name)
        return folder_name

    def update_multi_config_file(self, weatherprep_path, folder_name, usaf_wban, start_year, end_year,
                           country, state, county, place,
                           model, max_lai, evergreen_percent, vegetation_type, tree_height, wind_sensor_height, precip_file_path):
        # Construct the file path for the config file
        config_file = os.path.join(folder_name, 'WeatherPrepConfig.xml')
        print(f"Updating config file: {config_file}")

        # Parse the XML file with lxml to keep comments
        parser = ET.XMLParser(remove_blank_text=True, remove_comments=False)
        tree = ET.parse(config_file, parser)
        root = tree.getroot()
        # A function to safely update text in XML elements
        def update_text(element, text):
            if element is not None:
                element.text = str(text)
            else:
                print(f"Warning: Attempted to update a non-existent element for {text}")

        # Update year-related information
        update_text(root.find('.//StartYear'), start_year)
        update_text(root.find('.//EndYear'), end_year)

        # Update model and path information
        update_text(root.find('.//Model'), model)
        update_text(root.find('.//SurfaceWeatherDataFile'), os.path.join(folder_name, f'{usaf_wban}-{start_year}-{end_year}.txt'))

        # Update location information
        update_text(root.find('.//Nation'), country)
        update_text(root.find('.//State'), state)
        update_text(root.find('.//County'), county)
        update_text(root.find('.//Place'), place)

        # Update vegetation information
        update_text(root.find('.//MaximumLAI'), max_lai)
        update_text(root.find('.//EvergreenPercent'), evergreen_percent)
        update_text(root.find('.//VegetationType'), vegetation_type)
        update_text(root.find('.//Height_Tree_m'), tree_height)
        update_text(root.find('.//Height_WindSensor_m'), wind_sensor_height)

        # Update precipitation file path only if provided
        if precip_file_path is not None:
            update_text(root.find('.//PrecipitationDataCsv'), precip_file_path)
        else:
            print("No precipitation file path provided; existing path is preserved.")
            update_text(root.find('.//PrecipitationDataCsv'), '')
        # Copy comments from the template WeatherPrepConfig.xml (wpc_path) ---
        # CLEAR_EXISTING set to True to wipe any pre-existing comments in the target before copying.
        CLEAR_EXISTING = True
        wpc_path = os.path.join(script_dir, 'resources', 'WeatherPrepConfig.xml')
        # copy_comments_from_template function called: Use the actual template path you want; in this flow the template is 'wpc_path'.
        return_value = copy_comments_from_template(wpc_path, tree, clear_existing=CLEAR_EXISTING)

        #Copy comments from the template WeatherPrepConfig.xml (wpc_path) ---
        #CLEAR_EXISTING set to True to wipe any pre-existing comments in the target before copying.
        CLEAR_EXISTING = True
        wpc_path = os.path.join(script_dir, 'resources', 'WeatherPrepConfig.xml')
        #copy_comments_from_template function called: Use the actual template path you want; in this flow the template is 'wpc_path'.
        return_value = copy_comments_from_template(wpc_path, tree, clear_existing=CLEAR_EXISTING)
        
        # Save the updated XML file with pretty print
        tree.write(config_file, pretty_print=True, xml_declaration=True, encoding='UTF-8')
        print("Config file has been updated.")

        # Hide command prompt window on Windows OS
        if sys.platform.startswith('win'):
            startupinfo = STARTUPINFO()
            startupinfo.dwFlags |= STARTF_USESHOWWINDOW
            startupinfo.wShowWindow = SW_HIDE
            result = subprocess.run([weatherprep_path, folder_name], stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                    text=True, startupinfo=startupinfo)
        else:
            result = subprocess.run([weatherprep_path, folder_name], stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                    text=True)

        print(f"Subprocess finished with status: {result.returncode}")
        return result

    def update_config_file(self, weatherprep_path, folder_name, usaf_wban, year,
                           country, state, county, place,
                           model, max_lai, evergreen_percent, vegetation_type, tree_height, wind_sensor_height,
                           precip_file_path):
        # Construct the file path for the config file
        config_file = os.path.join(folder_name, 'WeatherPrepConfig.xml')
        print(f"Updating config file: {config_file}")

        # Parse the XML file with lxml to keep comments
        parser = ET.XMLParser(remove_blank_text=True, remove_comments=False)
        tree = ET.parse(config_file, parser)
        root = tree.getroot()

        # A function to safely update text in XML elements
        def update_text(element, text):
            if element is not None:
                element.text = str(text)
            else:
                print(f"Warning: Attempted to update a non-existent element for {text}")

        # Update year-related information
        update_text(root.find('.//StartYear'), year)
        update_text(root.find('.//EndYear'), year)

        # Update model and path information
        update_text(root.find('.//Model'), model)
        update_text(root.find('.//SurfaceWeatherDataFile'), os.path.join(folder_name, f'{usaf_wban}-{year}.txt'))

        # Update location information
        update_text(root.find('.//Nation'), country)
        update_text(root.find('.//State'), state)
        update_text(root.find('.//County'), county)
        update_text(root.find('.//Place'), place)

        # Update vegetation information
        update_text(root.find('.//MaximumLAI'), max_lai)
        update_text(root.find('.//EvergreenPercent'), evergreen_percent)
        update_text(root.find('.//VegetationType'), vegetation_type)
        update_text(root.find('.//Height_Tree_m'), tree_height)
        update_text(root.find('.//Height_WindSensor_m'), wind_sensor_height)

        # Update precipitation file path only if provided
        if precip_file_path is not None:
            update_text(root.find('.//PrecipitationDataCsv'), precip_file_path)
        else:
            print("No precipitation file path provided; existing path is preserved.")
            update_text(root.find('.//PrecipitationDataCsv'), '')

<<<<<<< .mine
        #Copy comments from the template WeatherPrepConfig.xml (wpc_path) ---
        #CLEAR_EXISTING set to True to wipe any pre-existing comments in the target before copying.
        CLEAR_EXISTING = True
        wpc_path = os.path.join(script_dir, 'resources', 'WeatherPrepConfig.xml')
        #copy_comments_from_template function called: Use the actual template path you want; in this flow the template is 'wpc_path'.
        return_value = copy_comments_from_template(wpc_path, tree, clear_existing=CLEAR_EXISTING)

||||||| .r212
=======
        #Copy comments from the template WeatherPrepConfig.xml (wpc_path) ---
        #CLEAR_EXISTING set to True to wipe any pre-existing comments in the target before copying.
        CLEAR_EXISTING = True
        wpc_path = os.path.join(script_dir, 'resources', 'WeatherPrepConfig.xml')
        #copy_comments_from_template function called: Use the actual template path you want; in this flow the template is 'wpc_path'.
        return_value = copy_comments_from_template(wpc_path, tree, clear_existing=CLEAR_EXISTING)
        
>>>>>>> .r216
        # Save the updated XML file with pretty print
        tree.write(config_file, pretty_print=True, xml_declaration=True, encoding='UTF-8')
        print("Config file has been updated.")
        # Hide command prompt window on Windows OS
        if sys.platform.startswith('win'):
            startupinfo = STARTUPINFO()
            startupinfo.dwFlags |= STARTF_USESHOWWINDOW
            startupinfo.wShowWindow = SW_HIDE
            result = subprocess.run([weatherprep_path, folder_name], stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                    text=True, startupinfo=startupinfo)
        else:
            result = subprocess.run([weatherprep_path, folder_name], stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                    text=True)

        print(f"Subprocess finished with status: {result.returncode}")
        return result


    def delete_downloaded_files(self, usaf_wban, year):
        # Delete downloaded and unzipped files
        os.remove(f'{usaf_wban}-{year}.gz')
        os.remove(f'{usaf_wban}-{year}.txt')
        os.remove(f'{usaf_wban}-{year}')

    # ~~~~~~~~~~~~~~ RESAMPLING ~~~~~~~~~~~~~~~

    def subhourly_processing(self, start_ymdh, end_ymdh, subhour_interval, folder_path):
        #Make new folder
        output_folder_path = os.path.join(folder_path, 'Subhourly')
        os.makedirs(output_folder_path, exist_ok=True)
        #Define file list
        meteo_list = ['Evaporation', 'Radiation', 'Weather']
        # Convert start_ymdh and end_ymdh to datetime objects
        start_date = pd.to_datetime(start_ymdh, format='%Y%m%d%H')
        end_date = pd.to_datetime(end_ymdh, format='%Y%m%d%H')

        for meteo in meteo_list:
            file_path = os.path.join(folder_path, f"{meteo}.csv")
            df_sh = pd.read_csv(file_path, parse_dates={'Datetime': ['YYYYMMDD', 'HH:MM:SS']})

            # Store the original header
            original_header = list(df_sh.columns)

            # Convert the 'Datetime' column to the desired format 'YYYYMMDD'
            df_sh['Datetime'] = pd.to_datetime(df_sh['Datetime'], format='%Y%m%d %H:%M:%S')

            # Filter data based on user-provided date range
            mask = (df_sh['Datetime'] >= start_date) & (
                        df_sh['Datetime'] <= (pd.to_datetime(end_date) + pd.Timedelta(days=1)))
            df_sh = df_sh[mask]

            # Resample data based on the specified interval
            df_sh.set_index('Datetime', inplace=True)
            interval_str = f'{subhour_interval}T'
            df_sh_resampled = df_sh.resample(interval_str).mean()

            # Use linear interpolation to fill data gaps
            df_sh_resampled.interpolate(method='linear', inplace=True)

            # Reset the index to restore the original 'Datetime' column
            df_sh_resampled.reset_index(inplace=True)

            # Create separate columns for 'YYYYMMDD' and 'HH:MM:SS'
            df_sh_resampled['YYYYMMDD'] = df_sh_resampled['Datetime'].dt.strftime('%Y%m%d')
            df_sh_resampled['HH:MM:SS'] = df_sh_resampled['Datetime'].dt.strftime('%H:%M:%S')

            # Drop the original 'Datetime' column
            df_sh_resampled.drop(columns=['Datetime'], inplace=True)

            # Restore the original header, excluding 'Datetime'
            df_sh_resampled = df_sh_resampled[
                ['YYYYMMDD', 'HH:MM:SS'] + [col for col in original_header if col != 'Datetime']]  # Reorder columns

            output_file_path = os.path.join(output_folder_path, f"{meteo}.csv")
            df_sh_resampled.to_csv(output_file_path, index=False, header=True, date_format='%Y%m%d %H:%M:%S')

    def _get_data_aggregation_method(self, var_name):
        """
        Helper method to pick the correct aggregation method
        for each variable name, matching your 'working' script logic.
        """
        # NOTE: Adjust these checks to your real variable naming:
        if "(yet to be defined)" in var_name:
            # Example: placeholders for precipitation (?)
            return 'sum'
        elif "(F)" in var_name or "(m/h)" in var_name or "(m/s)" in var_name \
             or "(kPa)" in var_name or "(W/m^2)" in var_name:
            return 'mean'
        elif "(m)" in var_name or "(Radian)" in var_name:
            return 'last'
        elif "(deg)" in var_name or "(yet to be defined)" in var_name:
            # Example: placeholders for wind direction
            return 'circular'
        else:
            # Default to mean if nothing else matches
            return 'mean'

    def _resample_file(
        self,
        input_file,
        output_file,
        date_ranges,
        time_ranges,
        time_steps
    ):
        """
        A near-copy of your working 'resample_timeStep_function',
        except as an internal method.
        We read from `input_file`, write to `output_file`,
        and use the same row-by-row aggregator logic.
        """
        # Open input and output
        with open(input_file, 'r') as infile, open(output_file, 'w') as outfile:
            # 1) Read header line
            header_values_vector = infile.readline().strip().split(',')
            # Write header to output
            outfile.write(','.join(header_values_vector) + '\n')

            # 2) Build aggregator dictionary
            data_and_aggregation_methods_dictionary = {}
            # Start from index=2 to skip date/time columns in the CSV
            for infile_variable in header_values_vector[2:]:
                method = self._get_data_aggregation_method(infile_variable)
                data_and_aggregation_methods_dictionary[infile_variable] = method

            # 3) Prepare aggregator structures
            variable_vectors = {
                data_var: [] for data_var in data_and_aggregation_methods_dictionary.keys()
            }
            aggregated_row   = {}
            start_datetime   = None
            time_duration_sec = 0
            target_time_step  = None
            first_row        = True

            # 4) Read each data row from the input
            for infile_row_values in infile:
                row_values_vector = infile_row_values.strip().split(',')
                # Skip malformed rows
                if len(row_values_vector) != len(header_values_vector):
                    print(f"[WARNING] Skipping malformed row: {infile_row_values.strip()}")
                    continue

                # Extract date/time
                date_str, time_str = row_values_vector[0], row_values_vector[1]
                current_datetime = datetime.strptime(
                    f"{date_str} {time_str}",
                    "%Y%m%d %H:%M:%S"
                )
                # Build a dict of variable_name -> float value (columns 2 onward)
                variable_values_vector = {
                    header_values_vector[i]: float(row_values_vector[i])
                    for i in range(2, len(row_values_vector))
                }

                # Determine if we are in any resampling range
                within_resampling = False

                # Check all intervals
                for (dr, tr, ts) in zip(date_ranges, time_ranges, time_steps):
                    start_date, end_date = dr  # e.g. (20110101, 20110301)
                    if start_date <= int(date_str) <= end_date:
                        # `tr` might be a single tuple or a list of tuples
                        if isinstance(tr, list):
                            # Possibly multiple time sub-ranges
                            for sub_time_range, sub_time_step in zip(tr, ts):
                                start_t, end_t = sub_time_range
                                if start_t <= time_str <= end_t:
                                    within_resampling = True
                                    target_time_step  = sub_time_step
                                    break
                        else:
                            # Single time range
                            start_t, end_t = tr
                            if start_t <= time_str <= end_t:
                                within_resampling = True
                                target_time_step  = ts
                        # If within resampling, stop checking the other intervals
                        if within_resampling:
                            break

                # 5) If first row, write it out directly
                if first_row:
                    outfile.write(infile_row_values)
                    first_row = False
                    start_datetime = current_datetime
                    aggregated_row['YYYYMMDD'] = date_str
                    aggregated_row['HH:MM:SS'] = time_str
                    continue

                # 6) If we are NOT within a resampling range, write data directly and continue
                if not within_resampling:
                    outfile.write(infile_row_values)
                    continue

                # 7) We are within a resampling block -> accumulate data
                time_duration_sec = (current_datetime - start_datetime).total_seconds()

                # Add the current row's variable values to aggregator
                for data_variable, _method in data_and_aggregation_methods_dictionary.items():
                    variable_vectors[data_variable].append(variable_values_vector[data_variable])

                # 8) Check if we have reached/exceeded the target time step
                if target_time_step is not None and time_duration_sec >= target_time_step:
                    # Compute aggregator for each variable
                    for data_variable, data_aggregation_method in data_and_aggregation_methods_dictionary.items():
                        if data_aggregation_method == 'sum':
                            aggregated_row[data_variable] = sum(variable_vectors[data_variable])
                        elif data_aggregation_method == 'mean':
                            arr = variable_vectors[data_variable]
                            aggregated_row[data_variable] = sum(arr) / len(arr)
                        elif data_aggregation_method == 'last':
                            arr = variable_vectors[data_variable]
                            aggregated_row[data_variable] = arr[-1]
                        elif data_aggregation_method == 'circular':
                            # Circular mean for angles
                            arr = variable_vectors[data_variable]
                            wind_dir_radians_vector = [math.radians(val) for val in arr]
                            x_component = sum(math.cos(rad) for rad in wind_dir_radians_vector)
                            y_component = sum(math.sin(rad) for rad in wind_dir_radians_vector)
                            wind_dir_mean_rad = math.atan2(y_component, x_component)
                            # Convert back to degrees, ensure [0, 360)
                            aggregated_row[data_variable] = (math.degrees(wind_dir_mean_rad) + 360) % 360

                    # Update date/time in aggregator row to current
                    aggregated_row['YYYYMMDD'] = date_str
                    aggregated_row['HH:MM:SS'] = time_str

                    # Write aggregated row to outfile
                    outfile.write(
                        f"{aggregated_row['YYYYMMDD']},{aggregated_row['HH:MM:SS']},"
                        + ",".join(
                            f"{aggregated_row.get(dv, 0):.8f}"
                            for dv in data_and_aggregation_methods_dictionary.keys()
                        )
                        + "\n"
                    )

                    # Reset accumulators
                    variable_vectors = {
                        data_var: [] for data_var in data_and_aggregation_methods_dictionary.keys()
                    }
                    start_datetime = current_datetime
                    time_duration_sec = 0

            # 9) Handle leftover partial interval data at end of file
            if time_duration_sec > 0:
                for data_variable, data_aggregation_method in data_and_aggregation_methods_dictionary.items():
                    if data_aggregation_method == 'sum':
                        aggregated_row[data_variable] = sum(variable_vectors[data_variable])
                    elif data_aggregation_method == 'mean':
                        arr = variable_vectors[data_variable]
                        aggregated_row[data_variable] = sum(arr) / len(arr) if arr else 0
                    elif data_aggregation_method == 'last':
                        arr = variable_vectors[data_variable]
                        aggregated_row[data_variable] = arr[-1] if arr else 0
                    elif data_aggregation_method == 'circular':
                        arr = variable_vectors[data_variable]
                        if len(arr) > 0:
                            wind_dir_radians_vector = [math.radians(val) for val in arr]
                            x_component = sum(math.cos(rad) for rad in wind_dir_radians_vector)
                            y_component = sum(math.sin(rad) for rad in wind_dir_radians_vector)
                            wind_dir_mean_rad = math.atan2(y_component, x_component)
                            aggregated_row[data_variable] = (math.degrees(wind_dir_mean_rad) + 360) % 360
                        else:
                            aggregated_row[data_variable] = 0

                # Write leftover aggregated row
                outfile.write(
                    f"{aggregated_row['YYYYMMDD']},{aggregated_row['HH:MM:SS']},"
                    + ",".join(
                        f"{aggregated_row.get(dv, 0):.8f}"
                        for dv in data_and_aggregation_methods_dictionary.keys()
                    )
                    + "\n"
                )
        # End with open...

    def resample_timesteps(self, output_folder_path, resample_intervals):
        """
        Recreate your working line-by-line resampling logic,
        but for files named 'Weather.csv' and 'Radiation.csv'
        in the output_folder_path. We rename the original files
        by appending '-hourly.csv' and then overwrite the original
        filenames with the newly resampled data.

        Parameters
        ----------
        self : class instance
            This is a method in your class.
        output_folder_path : str
            Directory containing 'Weather.csv' and 'Radiation.csv'.
        resample_intervals : list of tuples
            Each tuple is (start_date, end_date, hour_step).
            Example:
                [
                    ('20110101', '20110301', '12'),
                    ('20111001', '20111231', '12')
                ]
            which means 12-hour timesteps in Jan-Mar and Oct-Dec 2011.
        """

        # We must convert user-supplied intervals into the old script's
        # date_ranges, time_ranges, time_steps. The "time_range" in your
        # original code can default to "00:00:00" -> "23:59:59" for the entire day.
        date_ranges_list = []
        time_ranges_list = []
        time_steps_list  = []

        # Build the parallel lists:
        for (start_date_str, end_date_str, hour_step_str) in resample_intervals:
            # Convert strings to integer for date
            # e.g. '20110101' -> 20110101
            start_date_val = int(start_date_str)
            end_date_val   = int(end_date_str)
            # Convert the hour_step string to a float or int
            # e.g. '12' -> 12. Then convert hours -> seconds
            hour_step = float(hour_step_str) * 3600.0

            # For simplicity, cover the full day from 00:00:00 to 23:59:59
            time_range = ("00:00:00", "23:59:59")

            # Append to lists
            date_ranges_list.append((start_date_val, end_date_val))
            time_ranges_list.append(time_range)
            time_steps_list.append(hour_step)

        # Now process each of the two files: Weather.csv and Radiation.csv
        files_to_process = ["Weather.csv", "Radiation.csv"]
        for file_name in files_to_process:
            input_path = os.path.join(output_folder_path, file_name)

            if not os.path.isfile(input_path):
                print(f"[WARNING] Could not find {input_path}. Skipping.")
                continue

            # We want to rename the original to 'X-hourly.csv'
            renamed_file = os.path.join(
                output_folder_path,
                file_name.replace(".csv", "-hourly.csv")
            )
            try:
                os.rename(input_path, renamed_file)
                print(f"[INFO] Renamed {input_path} to {renamed_file}")
            except Exception as e:
                print(f"[ERROR] Could not rename file {input_path} -> {renamed_file}: {e}")
                # If rename fails, we might choose to skip or proceed using the existing file
                # We'll proceed, but aggregator input must be renamed_file if rename succeeded
                renamed_file = input_path  # fallback if rename fails

            # The newly resampled data will be saved to the original filename
            output_file = input_path

            print(f"[INFO] Resampling {renamed_file} -> {output_file}")
            self._resample_file(
                input_file=renamed_file,
                output_file=output_file,
                date_ranges=date_ranges_list,
                time_ranges=time_ranges_list,
                time_steps=time_steps_list
            )

        print("[INFO] Finished resampling all files.\n")

    # ~~~~~~~~~~~~~~TASK HANDLER FOR SAFE TERMINATION ~~~~~~~~~~~~~~~
    def request_stop(self):
        self.stop_requested = True

    # ~~~~~~~~~~~~~~~~~ TASK ~~~~~~~~~~~~~~~~~~
    def run_task(self):
        ishapp_path = os.path.join(root_dir, 'NOAA_data_tool', 'ishapp2.exe')
        weatherprep_path = os.path.join(root_dir, 'WeatherPrep', 'WeatherPrep', 'bin', 'Release', 'WeatherPrep.exe')
        wpc_path = os.path.join(script_dir, 'resources', 'WeatherPrepConfig.xml')
        # output folder
        output_folder = os.path.join(root_dir, 'Outputs')
        os.makedirs(output_folder, exist_ok=True)
        os.chdir(output_folder)
        # Location & Weather Station Parameters - Required
        usaf_wban = self.params['usaf_wban']
        start_year = int(self.params['start_year'])
        end_year = int(self.params['end_year'])
        country = self.params['country']
        state = self.params['state']
        county = self.params['county']
        place = self.params['place']
        #Advanced XML Parameters - Optional
        model = self.params['model']
        max_lai = self.params['max_lai']
        evergreen_percent = self.params['evergreen_percent']
        vegetation_type = self.params['vegetation_type']
        tree_height = self.params['tree_height']
        wind_sensor_height = self.params['wind_sensor_height']
        precip_file_path = self.params['precip_file']
        multiyear_flag = self.params['multiyear_flag']
        resample_flag = self.params['resample_flag']
        resample_intervals = self.params.get('resample_intervals', [])


        self.update_console.emit(f'Downloading weather data for {usaf_wban} from ftp.ncei.noaa.gov...')
        self.download_noaa_data(usaf_wban, start_year, end_year)
        
        if not self.flag_failure:
            self.update_console.emit(f'Unzipping .gz files...')
            self.unzip_gz_file(usaf_wban, start_year, end_year)

            self.update_console.emit(f'Running ishapp2.exe by NOAA for {usaf_wban}...')
            self.run_ishapp2(ishapp_path, usaf_wban, start_year, end_year)

            if multiyear_flag == True:
                self.update_console.emit(f'Preparing folder for {usaf_wban}...')
                ws_folder_path = self.prepare_single_folder(wpc_path, output_folder, usaf_wban, start_year, end_year)

                self.update_console.emit(f'Running WeatherPrep for {usaf_wban}...')
                result = self.update_multi_config_file(weatherprep_path, ws_folder_path, usaf_wban, start_year, end_year,
                                        country, state, county, place,
                                        model, max_lai, evergreen_percent, vegetation_type, tree_height, wind_sensor_height, precip_file_path
                                        )
                if "Simulation completed. Check above output files for results." in result.stdout:
                    print("Simulation completed successfully.")
                    flag_run_successful = True
                else:
                    print(
                        "WeatherPrep could not process your file. Please check your location for validity in https://database.itreetools.org/#/locations or select a different weather station.")
                    flag_run_successful = False

                for year in range(start_year, end_year + 1):
                    self.update_console.emit('Deleting intermediate files ...')
                    self.delete_downloaded_files(usaf_wban, year)

            else:
                for year in range(start_year, end_year + 1):
                    self.update_console.emit(f'Preparing folder for {usaf_wban}-{year}...')
                    ws_folder_path = self.prepare_multi_folder(wpc_path, output_folder, usaf_wban, year)
                    self.update_console.emit(f'Running WeatherPrep for {usaf_wban}-{year}...')
                    result = self.update_config_file(weatherprep_path, ws_folder_path, usaf_wban, year,
                                            country, state, county, place,
                                            model, max_lai, evergreen_percent, vegetation_type, tree_height,
                                            wind_sensor_height, precip_file_path
                                            )
                    if "Simulation completed. Check above output files for results." in result.stdout:
                        self.update_console.emit("Simulation completed successfully.")
                        flag_run_successful = True
                    else:
                        self.update_console.emit(
                            "WeatherPrep could not process your file.")
                        flag_run_successful = False
                    self.update_console.emit('Deleting intermediate files ...')
                    self.delete_downloaded_files(usaf_wban, year)

            # Check if the sub-hourly parameters exist
            if self.params['start_ymdh'] and self.params['end_ymdh'] is not None and flag_run_successful == True:
                start_ymdh = self.params['start_ymdh']
                end_ymdh = self.params['end_ymdh']
                subhour_interval = self.params['subhour_interval']
                self.update_console.emit('Creating sub-hourly weather files...')
                self.subhourly_processing(start_ymdh, end_ymdh, subhour_interval, ws_folder_path)
                output_folder_path = os.path.join(ws_folder_path, 'Subhourly')
                self.update_console.emit(f'\nWeather processing completed.')
                self.update_console.emit(f'Check output files for results at: {output_folder_path}.\n')
                self.update_console.emit(f'You can exit program or Reset Location for additional data processing.') 
                self.update_console.emit(f'Thank you for using i-Tree tools to improve the world!') 
                
            elif flag_run_successful == True:
                output_folder_path = os.path.join(output_folder, f'{usaf_wban}')
                if resample_flag:
                    self.resample_timesteps(ws_folder_path, resample_intervals)
                self.update_console.emit(f'\nWeather processing complete!')
                self.update_console.emit(f'Check output files for results at: {output_folder_path}.\n')
                self.update_console.emit(f'You can exit program or Reset Location for additional data processing.') 
                self.update_console.emit(f'Thank you for using i-Tree tools to improve the world!')


            elif flag_run_successful == False:
                self.update_console.emit(f'\nWeather processing could not complete. Please run WeatherPrep.exe from your command prompt or refer to the ReadMe.txt to troubleshoot.')
                self.update_console.emit(f'You can exit program or Reset Location for additional data processing.')
                self.update_console.emit(f'Thank you for using i-Tree tools to improve the world!')

            self.finished.emit()
        
        print("\n")
        print("Weather processing complete. To release command prompt window, exit the GUI.")
        print("Thank you for using i-Tree tools to improve the world!")

# ~~~~~~~~~~~~~~~~~~~~~ APP ~~~~~~~~~~~~~~~~~~~~~~~~
class WeatherProcessorApp(QtWidgets.QWidget):
    def __init__(self):
        super().__init__()
        # Mostly intialize empty params
        self.tab_widget = None
        self.console = None
        self.process_button = None
        self.data_table = None
        self.advanced_xml_cb = None
        self.place_dropdown = None
        self.county_dropdown = None
        self.reset_button = None
        self.country_dropdown = None
        self.worker = None
        self.xmlWindow = None
        self.state_dropdown = None

        csv_file_loc = os.path.join(script_dir, 'resources', 'itree-locations.csv')
        csv_file_isd = os.path.join(script_dir, 'resources', 'isd-history.csv')
        csv_file_inventory = os.path.join(script_dir, 'resources', 'isd-inventory.csv')
        self.df_loc = pd.read_csv(csv_file_loc)
        self.df_isd = pd.read_csv(csv_file_isd, dtype = {'USAF': str, 'WBAN': str})
        # Specify the data types for the columns in isd-inventory.csv
        dtype_inventory = {
            'USAF': str,
            'WBAN': str,
            'YEAR': int,
            'JAN': float,
            'FEB': float,
            'MAR': float,
            'APR': float,
            'MAY': float,
            'JUN': float,
            'JUL': float,
            'AUG': float,
            'SEP': float,
            'OCT': float,
            'NOV': float,
            'DEC': float
        }

        self.df_inv = pd.read_csv(csv_file_inventory, dtype=dtype_inventory)
        print("Data types in df_isd:")
        print(self.df_isd[['USAF', 'WBAN']].dtypes)

        print("Data types in df_inv:")
        print(self.df_inv[['USAF', 'WBAN']].dtypes)
        self.initUI()
        self.radio_button_group = QtWidgets.QButtonGroup()
        # Set up thread now in case user closes code before initializing worker
        self.thread = None
        self.xmlConfigWindow = XMLConfigWindow()

    # Advanced XML Set-up
    def toggleXMLWindow(self, state):
        #cte print("Toggle XML Window state:", state)  # Diagnostic print
        if state == QtCore.Qt.Checked:
            if not self.xmlWindow:  # Create the window if it does not exist
                self.xmlWindow = XMLConfigWindow()  # Removed self to not set parent
                self.xmlWindow.setGeometry(900, 300, 250, 200)  # Adjust position and size
            self.xmlWindow.show()  # Show the XML Configuration window
        else:
            if self.xmlWindow:
                self.xmlWindow.close()  # Close the XML Configuration window

    # ~~~~~~~~~~~~~~~~~~~ UI CODE ~~~~~~~~~~~~~~~~~~~
    def try_disconnect_signal(self, dropdown):
        # Disconnects index signals in reset_apps
        try:
            dropdown.currentIndexChanged.disconnect()
        except TypeError:
            pass

    def initialize_dropdowns(self):
        # Get unique nation names and sort them
        nations = sorted(self.df_loc['NationName'].unique())
        # Populate the country dropdown with sorted nation names
        nation_list = ['--Select--'] + nations
        self.country_dropdown.addItems(nation_list)
        self.country_dropdown.currentIndexChanged.connect(self.on_country_changed)
        self.country_dropdown.setEnabled(True)
        # Initially, disable other dropdowns
        self.state_dropdown.setEnabled(False)
        self.county_dropdown.setEnabled(False)
        self.place_dropdown.setEnabled(False)

    def reset_app(self):
        # Resetting dropdowns
        self.place_dropdown.clear()
        self.county_dropdown.clear()
        self.state_dropdown.clear()
        self.country_dropdown.clear()
        self.try_disconnect_signal(self.country_dropdown)
        self.try_disconnect_signal(self.state_dropdown)
        self.try_disconnect_signal(self.county_dropdown)
        self.try_disconnect_signal(self.place_dropdown)

        # Optionally, re-initialize the dropdowns or other parts of the app as needed
        self.initialize_dropdowns()

        # Clear any other data or fields as needed
        self.data_table.setRowCount(0)
        self.hourly_start_year_entry.clear()
        self.hourly_end_year_entry.clear()
        self.sub_hourly_end_entry.clear()
        self.sub_hourly_start_entry.clear()


        # Add any other reset logic here

    def create_centered_widget(self, widget):
        # Create a QWidget and a QHBoxLayout
        widget_holder = QtWidgets.QWidget()
        layout = QtWidgets.QHBoxLayout()

        # Add the widget (e.g., a radio button) to the layout
        layout.addWidget(widget)

        # Set alignment to center
        layout.setAlignment(QtCore.Qt.AlignCenter)

        # Set the layout margins to 0 for the widget to occupy the full space
        layout.setContentsMargins(0, 0, 0, 0)

        # Set the layout to the widget holder
        widget_holder.setLayout(layout)

        return widget_holder


    def update_console(self, text):
        self.console.append(text)

    # ~~~~~~~~~~~~~~~ LOCATION DROPDOWN ~~~~~~~~~~~~~~~~
    def calculate_distance(self, lat1, lon1, lat2, lon2):
        # Simplistic distance calculation, could be replaced with a more accurate method
        return ((lat1 - lat2) ** 2 + (lon1 - lon2) ** 2) ** 0.5

    def on_country_changed(self, index):
        if index != -1:  # Check if a selection is made
            country = self.country_dropdown.currentText()
            states = sorted(self.df_loc[self.df_loc['NationName'] == country]['PrimaryPN'].unique())
            # Order of operations - CLEAR dropdown, add items, attempt to disconnect signal, reconnect signal, enable signal
            self.state_dropdown.clear()
            state_list = ['--Select--'] + states
            self.state_dropdown.addItems(state_list)
            self.try_disconnect_signal(self.state_dropdown)
            self.state_dropdown.currentIndexChanged.connect(self.on_state_changed)
            self.state_dropdown.setEnabled(True)

    def on_state_changed(self, index):
        #cte print(f"State Changed Index: {index}")
        if index != -1:  # Check if a selection is made
            state = self.state_dropdown.currentText()
            counties = sorted(self.df_loc[(self.df_loc['NationName'] == self.country_dropdown.currentText()) &
                                      (self.df_loc['PrimaryPN'] == state)]['SecondaryPN'].unique())
            self.county_dropdown.clear()
            county_list = ['--Select--'] + counties
            self.county_dropdown.addItems(county_list)
            self.try_disconnect_signal(self.county_dropdown)
            self.county_dropdown.currentIndexChanged.connect(self.on_county_changed)
            self.county_dropdown.setEnabled(True)
            if len(counties) == 1:
                self.on_county_changed(0)

    def on_county_changed(self, index):
        if index != -1:  # Check if a selection is made
            county = self.county_dropdown.currentText()
            places = sorted(self.df_loc[(self.df_loc['NationName'] == self.country_dropdown.currentText()) &
                                    (self.df_loc['PrimaryPN'] == self.state_dropdown.currentText()) &
                                    (self.df_loc['SecondaryPN'] == county)]['TertiaryPN'].unique())
            self.place_dropdown.clear()
            place_list = ['--Select--'] + places
            self.place_dropdown.addItems(place_list)
            self.try_disconnect_signal(self.place_dropdown)
            self.place_dropdown.currentIndexChanged.connect(self.on_place_changed)
            self.place_dropdown.setEnabled(True)

    def on_place_changed(self, index):
        print("on_place_changed called with index:", index)
        if index != -1:
            # Extract selected location details
            place = self.place_dropdown.currentText()
            county = self.county_dropdown.currentText()
            state = self.state_dropdown.currentText()
            country = self.country_dropdown.currentText()
            print(f"Selected place: {place}, county: {county}, state: {state}, country: {country}")

            if place and county and state and country:
                selected_location = self.df_loc[
                    (self.df_loc['TertiaryPN'] == place) &
                    (self.df_loc['SecondaryPN'] == county) &
                    (self.df_loc['PrimaryPN'] == state) &
                    (self.df_loc['NationName'] == country)
                    ]
                print("Selected location DataFrame:")
                print(selected_location)

                if not selected_location.empty:
                    lat = selected_location.iloc[0]['Latitude']
                    lon = selected_location.iloc[0]['Longitude']
                    print(f"Latitude: {lat}, Longitude: {lon}")

                    # Filter isd_history_df by distance
                    isd_filtered_df = self.df_isd[
                        (self.df_isd['LAT'].between(lat - 0.5, lat + 0.5)) &
                        (self.df_isd['LON'].between(lon - 0.5, lon + 0.5))
                        ].copy()
                    print("Filtered isd_history DataFrame:")
                    print(isd_filtered_df.head())

                    # Calculate distances for each station
                    print("Calculating distances...")
                    isd_filtered_df['DISTANCE'] = isd_filtered_df.apply(
                        lambda row: self.calculate_distance(lat, lon, row.get('LAT', 0), row.get('LON', 0)),
                        axis=1
                    )
                    print("Distances calculated:")
                    print(isd_filtered_df[['USAF', 'WBAN', 'DISTANCE']].head())

                    # Sort by distance
                    isd_filtered_df = isd_filtered_df.sort_values(by='DISTANCE')
                    print("DataFrame sorted by DISTANCE:")
                    print(isd_filtered_df[['USAF', 'WBAN', 'DISTANCE']].head())



                    # Process isd_inventory.csv
                    years_of_interest = [2018, 2019, 2021, 2022, 2023]
                    print("Filtering inventory DataFrame for years:", years_of_interest)
                    df_inv_filtered = self.df_inv[self.df_inv['YEAR'].isin(years_of_interest)].copy()
                    print("Filtered inventory DataFrame:")
                    print(df_inv_filtered.head())



                    # Sum JAN-DEC for each station per year
                    month_columns = ['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN',
                                     'JUL', 'AUG', 'SEP', 'OCT', 'NOV', 'DEC']
                    print("Calculating YEAR_TOTAL for each station per year...")
                    df_inv_filtered['YEAR_TOTAL'] = df_inv_filtered[month_columns].sum(axis=1)
                    print("YEAR_TOTAL calculated:")
                    print(df_inv_filtered[['USAF', 'WBAN', 'YEAR', 'YEAR_TOTAL']].head())

                    # Group by USAF and WBAN, compute average YEAR_TOTAL
                    print("Grouping by USAF and WBAN to compute average YEAR_TOTAL...")
                    grouped = df_inv_filtered.groupby(['USAF', 'WBAN'])['YEAR_TOTAL'].mean().reset_index()
                    print("Grouped DataFrame with average YEAR_TOTAL:")
                    print(grouped.head())

                    print("Unique USAF codes in isd_filtered_df:")
                    print(isd_filtered_df['USAF'].unique())

                    print("Unique USAF codes in grouped:")
                    print(grouped['USAF'].unique())

                    # Check for intersection
                    common_usaf_codes = set(isd_filtered_df['USAF']).intersection(set(grouped['USAF']))
                    print("Common USAF codes between isd_filtered_df and grouped:")
                    print(common_usaf_codes)

                    # Determine QUALITY based on average
                    print("Determining QUALITY for each station...")

                    def determine_quality(avg):
                        if avg > 8040:
                            return 'Good'
                        elif avg > 4368:
                            return 'Fair'
                        else:
                            return 'Poor'

                    grouped['QUALITY'] = grouped['YEAR_TOTAL'].apply(determine_quality)
                    print("QUALITY determined:")
                    print(grouped[['USAF', 'WBAN', 'QUALITY']].head())

                    # Merge QUALITY into isd_filtered_df
                    print("Merging QUALITY into filtered isd DataFrame...")
                    # Merge QUALITY into isd_filtered_df
                    merged_df = pd.merge(
                        isd_filtered_df,
                        grouped[['USAF', 'WBAN', 'QUALITY']],
                        on=['USAF', 'WBAN'],
                        how='left'
                    )

                    # Fill NaN QUALITY values with 'Poor'
                    merged_df['QUALITY'] = merged_df['QUALITY'].fillna('Poor')

                    # Select required fields
                    required_fields = merged_df[['USAF', 'WBAN', 'STATION NAME', 'BEGIN', 'END', 'QUALITY']].copy()

                    print("Final required fields DataFrame:")
                    print(required_fields.head())

                    # Clear existing buttons and table content
                    print("Clearing existing radio buttons and table content...")
                    for button in self.radio_button_group.buttons():
                        self.radio_button_group.removeButton(button)

                    self.data_table.setRowCount(len(required_fields))
                    table_row = 0
                    first_button = None  # To keep track of the first radio button

                    print("Populating the data table...")
                    for _, row in required_fields.iterrows():
                        radio_button = QtWidgets.QRadioButton()
                        if table_row == 0:  # Auto-select the first radio button
                            radio_button.setChecked(True)
                            first_button = radio_button
                        self.radio_button_group.addButton(radio_button)

                        centered_radio_button = self.create_centered_widget(radio_button)
                        self.data_table.setCellWidget(table_row, 0, centered_radio_button)

                        for j, val in enumerate(row):
                            # Adjust 'END' column if necessary
                            if required_fields.columns[j] == 'END' and str(val).startswith('2024'):
                                val = 'Active'
                            self.data_table.setItem(table_row, j + 1, QtWidgets.QTableWidgetItem(str(val)))

                        table_row += 1

                    print("Resizing columns to contents...")
                    self.data_table.resizeColumnsToContents()
                    print("on_place_changed completed successfully.")

                else:
                    print("Selected location is empty after filtering.")
            else:
                print("One or more of place, county, state, or country is empty.")
        else:
            print("Index is -1, no action taken.")

    def get_selected_row_data(self):
        # Get the checked button from the button group
        checked_button = self.radio_button_group.checkedButton()

        if checked_button is not None:
            # Find the row index of the checked button
            for i in range(self.data_table.rowCount()):
                if self.data_table.cellWidget(i, 0).layout().itemAt(0).widget() == checked_button:
                    usaf = self.data_table.item(i, 1).text()  # Column 1 for USAF
                    wban = self.data_table.item(i, 2).text().zfill(5)
                    ws_begin = self.data_table.item(i, 4).text()
                    ws_end = self.data_table.item(i, 5).text()  # Column 2 for WBAN
                    return usaf, wban, ws_begin, ws_end

        return None, None  # Return None if no button is checked

    # ~~~~~~~~~~~~~~~ THREAD HANDLERS FOR MULTI-RUNS ~~~~~~~~~~~~~~~~
    def cleanup_thread(self):
        if self.thread:
            if self.thread.isRunning():
                self.worker.request_stop()
                self.thread.quit()
                self.thread.wait()

            # Disconnect all signals connected to the worker and thread here
            # to prevent "RuntimeError: wrapped C/C++ object of type QThread has been deleted"
            self.worker.finished.disconnect(self.cleanup_thread)
            # Add any other necessary disconnects here

            # Reset thread and worker to None after cleanup to prevent access to deleted objects
            self.thread = None
            self.worker = None

    def closeEvent(self, event):
        self.cleanup_thread()
        event.accept()

    def setup_worker_and_thread(self):
        self.worker.moveToThread(self.thread)

        self.worker.update_console.connect(self.update_console)
        self.worker.finished.connect(self.cleanup_thread)
        self.thread.started.connect(self.worker.run_task)

        self.thread.start()

    # ~~~~~~~~~~~~~~~~~~ MAIN SCRIPT ~~~~~~~~~~~~~~~~~~
    def process_data(self):
        usaf, wban, ws_start, ws_end = self.get_selected_row_data()
        start_year = self.hourly_start_year_entry.text()
        end_year = self.hourly_end_year_entry.text()
        place = self.place_dropdown.currentText()
        county = self.county_dropdown.currentText()
        state = self.state_dropdown.currentText()
        country = self.country_dropdown.currentText()

        # Default values for advanced configuration
        config_values = {
            'model': 'Hydro',
            'max_lai': 5,  # Default value for Maximum LAI
            'evergreen_percent': 5,  # Default value for Evergreen (%)
            'vegetation_type': 'Tree',  # Default value for Vegetation Type
            'tree_height': 12,  # Default value for Tree Height (m)
            'wind_sensor_height': 10,  # Default value for Wind Sensor Height (m)
            'precip_file': None
        }

        # OPTIONAL FIELDS initialized as None
        start_ymdh = None
        end_ymdh = None
        subhour_interval = None
        resample_intervals = []

        #True/False field for Combining Years
        if self.xmlWindow is not None:  # Check if the XMLConfigWindow is currently active
            # Update with values returned from get_configuration_Values
            config_values = self.xmlWindow.get_configuration_values()
            # Update from defaults
            config_values.update(config_values)

        # Identify which tab is active
        active_tab_index = self.tab_widget.currentIndex()
        active_tab_title = self.tab_widget.tabText(active_tab_index)

        # Process based on active tab
        if active_tab_title == "Hourly":
            start_year = self.hourly_start_year_entry.text()
            end_year = self.hourly_end_year_entry.text()
            #cte print(f"debug: hourly tab is active. startyear= {start_year}, endyear = {end_year}")
            multiple_years_flag = self.combine_years_checkbox.isChecked()
            resample_flag = self.resample_checkbox.isChecked()
            if self.resample_checkbox.isChecked() and self.resample_window:
                for row in self.resample_window.rows:
                    start_date = row[1].text()
                    end_date = row[2].text()
                    timestep = row[3].text()

                    # Validate start and end dates
                    if not ((len(start_date) == 8 or len(start_date) == 10) and (
                            len(end_date) == 8 or len(end_date) == 10)):
                        self.console.append(
                            "Error: Start and End dates must be 8 or 10 digits (YYYYMMDD or YYYYMMDDHH).")
                        return  # Stop processing

                    resample_intervals.append((start_date, end_date, timestep))

        elif active_tab_title == "Sub-Hourly":
            start_ymdh = self.sub_hourly_start_entry.text()
            end_ymdh = self.sub_hourly_end_entry.text()
            if not (len(start_ymdh) == 10 and len(end_ymdh) == 10):
                self.console.append(
                    "Error: 'Start YYYYMMDDHH' or 'End YYYYMMDDHH' format is incorrect. Please ensure the format is YYYYMMDDHH.")
                return  # Exit the method to prevent further execution
            # Extract just the year part for further processing
            subhour_interval = int(self.factor_spin_box.value())
            start_year = start_ymdh[:4]
            end_year = end_ymdh[:4]
            #Handler for rare case where user re-runs WeatherPrep with sub-hourly timestep when previously working with hourly, combined format.
            multiple_years = False


        # Check if all required fields are provided
        if not all([usaf, wban, start_year, end_year, place, county, state, country]):
            self.console.append("Error: Missing required information. Please ensure all fields are filled.")
            return  # Exit the process_data method

        try:
            start_year = int(start_year)
            end_year = int(end_year)
            ws_start = int(ws_start[:4])
            ws_end = int(ws_end[:4]) if ws_end != 'Active' else 2023
            #cte `print`(f"start_year = {start_year}. end_year = {end_year}. ws_start = {ws_start}. ws_end = {ws_end}")


        except ValueError:
            self.console.append("Error: Invalid year format. Please enter valid start and end years.")
            return

        # Check if end_year aligns with ws_end
        if ws_end != 2023 and end_year > ws_end:
            self.console.append(
                "Please confirm that your End Year is equal to or less than the weather station's end year.")
            return

        # Check if start_year aligns with ws_start
        if start_year < ws_start:
            self.console.append(
                "Please confirm that your Start Year is equal to or greater than the weather station's start year.")
            return
        # Gather all parameters
        params = {
            'usaf_wban': f'{usaf}-{wban}',
            'start_year': start_year,
            'end_year': end_year,
            'start_ymdh': start_ymdh,
            'end_ymdh': end_ymdh,
            'subhour_interval': subhour_interval,
            'place': place,
            'county': county,
            'state': state,
            'country': country,
            **config_values,
            'multiyear_flag': multiple_years_flag,
            'resample_flag': resample_flag,
            'resample_intervals': resample_intervals
        }
        # Check if there's an existing thread and clean up if necessary
        if self.thread is not None:
            self.cleanup_thread()

        # Create a QThread object
        self.thread = QThread()

        # Create a worker object with the params dictionary
        self.worker = Worker(params)

        # Setup worker and thread
        self.setup_worker_and_thread()

    def handle_resample_checkbox(self, state):
        """
        Handles the state change of the 'Resample Timestep' checkbox.

        Args:
            state (int): The state of the checkbox (Qt.Checked or Qt.Unchecked).
        """
        if state == QtCore.Qt.Checked:
            # Create and show the ResampleWindow
            self.resample_window = ResampleWindow()
            self.resample_window.setGeometry(900,550,400,100)
            self.resample_window.show()
        else:
            # Close the ResampleWindow if it exists
            if hasattr(self, 'resample_window') and self.resample_window:
                self.resample_window.close()
                self.resample_window = None  # Remove the reference to allow garbage collectio

    # ~~~~~~~~~~~~~~~~~~~~~ UI ~~~~~~~~~~~~~~~~~~~~~~~~~
    def initUI(self):
        # Set the title and size of the main window
        self.setWindowTitle('HydroPlus Weather Processor')
        icon_path = os.path.join(script_dir, 'resources', 'iTree_transparent.ico')
        self.setWindowIcon(QIcon(icon_path))
        self.setGeometry(300, 300, 600, 600)  # x, y, width, height

        # Create layout
        grid = QtWidgets.QGridLayout()
        self.setLayout(grid)

        # Country Dropdown (replacing Country Entry)
        grid.addWidget(QtWidgets.QLabel('Nation:'), 0, 0)
        self.country_dropdown = QtWidgets.QComboBox()
        grid.addWidget(self.country_dropdown, 0, 1, 1, 1)

        self.reset_button = QtWidgets.QPushButton('Reset Location')
        grid.addWidget(self.reset_button, 4, 0, 1, 4)
        self.reset_button.clicked.connect(self.reset_app)

        # State Dropdown (replacing State Entry)
        grid.addWidget(QtWidgets.QLabel('State:'), 1, 0, 1, 1)
        self.state_dropdown = QtWidgets.QComboBox()
        self.state_dropdown.setEnabled(False)  # Disabled initially
        grid.addWidget(self.state_dropdown, 1, 1, 1, 1)

        # County Dropdown (replacing County Entry)
        grid.addWidget(QtWidgets.QLabel('County:'), 2, 0, 1, 1)
        self.county_dropdown = QtWidgets.QComboBox()
        self.county_dropdown.setEnabled(False)  # Disabled initially
        grid.addWidget(self.county_dropdown, 2, 1, 1, 1)
        # Create a QLabel for instructions
        instructions = QtWidgets.QLabel("  Instructions:\n"
                                        "   1. Select nearest location to project area.\n"
                                        "   2. Select weather station.\n"
                                        "   3. Enter start and end year. \n"
                                        "   4. Click 'Process Data'")
        instructions.setWordWrap(True)
        font = QFont()
        font.setPointSize(10) 
        instructions.setFont(font)
        # Add the QLabel to the layout, spanning columns 2 and 3, and rows 2 through 4
        grid.addWidget(instructions, 0, 2, 3, 2)

        # Place Dropdown (replacing Place Entry)
        grid.addWidget(QtWidgets.QLabel('Place:'), 3, 0)
        self.place_dropdown = QtWidgets.QComboBox()
        self.place_dropdown.setEnabled(False)  # Disabled initially
        grid.addWidget(self.place_dropdown, 3, 1, 1, 1)

        # Advanced XML Configuration checkbox
        self.advanced_xml_cb = QtWidgets.QCheckBox('Advanced XML Configuration')
        grid.addWidget(self.advanced_xml_cb, 3, 3)  # Adjust grid positioning as needed
        self.advanced_xml_cb.stateChanged.connect(self.toggleXMLWindow)


        # Adding a QTableWidget to display the data
        self.data_table = QtWidgets.QTableWidget(self)
        self.data_table.setColumnCount(7)
        self.data_table.setHorizontalHeaderLabels(
            ['Select', 'USAF', 'WBAN', 'STATION_NA', 'BEGIN', 'END', 'QUALITY'])
        grid.addWidget(self.data_table, 5, 0, 1, 4) 

        # Initialize QTabWidget
        self.tab_widget = QtWidgets.QTabWidget()

        #~~~~~~~~~~~~~~~~~~~~~~~~~~
        ##TAB CREATION
        #~~~~~~~~~~~~~~~~~~~~~~~~~~
        hourly_tab = QtWidgets.QWidget()
        hourly_layout = QtWidgets.QGridLayout(hourly_tab)

        # Add widgets for 'Hourly' tab
        hourly_layout.addWidget(QtWidgets.QLabel('Start (YYYY):'), 0, 0)
        self.hourly_start_year_entry = QtWidgets.QLineEdit()
        hourly_layout.addWidget(self.hourly_start_year_entry, 0, 1)

        hourly_layout.addWidget(QtWidgets.QLabel('End (YYYY):'), 0, 2)
        self.hourly_end_year_entry = QtWidgets.QLineEdit()
        hourly_layout.addWidget(self.hourly_end_year_entry, 0, 3)

        self.combine_years_checkbox = QtWidgets.QCheckBox('Combine multiple years into one file')
        hourly_layout.addWidget(self.combine_years_checkbox, 1, 1)

        # Add the 'Resample Timestep' checkbox
        self.resample_checkbox = QtWidgets.QCheckBox('Resample Timestep')
        hourly_layout.addWidget(self.resample_checkbox, 1, 2)

        # Connect the stateChanged signal to a slot function
        self.resample_checkbox.stateChanged.connect(self.handle_resample_checkbox)

        self.tab_widget.addTab(hourly_tab, "Hourly")

        sub_hourly_tab = QtWidgets.QWidget()
        sub_hourly_layout = QtWidgets.QGridLayout(sub_hourly_tab)

        # Add widgets for 'Sub-Hourly' tab
        sub_hourly_layout.addWidget(QtWidgets.QLabel('Start (YYYYMMDDHH):'), 0, 0)
        self.sub_hourly_start_entry = QtWidgets.QLineEdit()
        sub_hourly_layout.addWidget(self.sub_hourly_start_entry, 0, 1)

        sub_hourly_layout.addWidget(QtWidgets.QLabel('End (YYYYMMDDHH):'), 0, 2)
        self.sub_hourly_end_entry = QtWidgets.QLineEdit()
        sub_hourly_layout.addWidget(self.sub_hourly_end_entry, 0, 3)

        # Adding the CustomSpinBox to your layout
        self.factor_spin_box = CustomSpinBox()
        sub_hourly_layout.addWidget(QtWidgets.QLabel('Interval:'), 1, 0)
        sub_hourly_layout.addWidget(self.factor_spin_box, 1, 1)
        sub_hourly_layout.addWidget(QtWidgets.QLabel('Minutes'), 1, 2)

        self.tab_widget.addTab(sub_hourly_tab, "Sub-Hourly")
        # Add the tab widget to the grid layout
        grid.addWidget(self.tab_widget, 6, 0, 1, 4)

        # Process Button
        self.process_button = QtWidgets.QPushButton('Process Data')
        self.process_button.clicked.connect(self.process_data)
        grid.addWidget(self.process_button, 7, 0, 1, 4)

        # Console
        self.console = QtWidgets.QTextEdit()  # QTextEdit is used for a scrollable text area
        self.console.setReadOnly(True)  # Make it read-only
        grid.addWidget(self.console, 8, 0, 2, 4)

        # ... [Include definitions for browse_folder and process_data methods] ...
        self.initialize_dropdowns()

if __name__ == '__main__':
    import sys

    app = QtWidgets.QApplication(sys.argv)
    splash = SplashScreen()
    splash.show()

    mainWindow = WeatherProcessorApp()
    splash.finish(mainWindow)
    mainWindow.show()
    sys.exit(app.exec_())
