#include "AggregateOutput.h"

//Pollutant string names for median and mean load for time step
string Name_Load_Catchment_Pollutant_Median_kg, Name_Load_Catchment_Pollutant_Mean_kg;
//Pollutant string names for sum of median and mean load for simulation
string Name_Load_Catchment_Pollutant_Median_Sum_kg, Name_Load_Catchment_Pollutant_Mean_Sum_kg;
//Pollutant string names for GI filtered median and mean load for time step 
string Name_Load_GI_Inflow_Pollutant_Mean_g, Name_Load_GI_Runoff_Surface_Pollutant_Mean_g, Name_Load_GI_Runoff_Vault_Pollutant_Mean_g, Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g;
string Name_Release_GI_Total_Pollutant_Mean_g;
//Pollutant string names for GI filtered sum of median and mean load for simulation
string Name_Load_GI_Inflow_Pollutant_Mean_Sum_kg, Name_Load_GI_Runoff_Surface_Pollutant_Mean_Sum_kg, Name_Load_GI_Runoff_Vault_Pollutant_Mean_Sum_kg, Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_Sum_kg;
string Name_Release_GI_Total_Pollutant_Mean_Sum_kg;

//Note: Folder variables arrive as their cover type depth (m) for folder area and leave as volumes (m3), unless otherwise noted
//Note: StorageDeficit_VadoseZone_m arrives as catchment depth, across all folders except those not connected to catchment vadose zone
//Note: Runoff variables arrive as BulkArea folder depths from RunoffSummation.cpp; 
//Note: GI variables arrive as GI folder depths from Flux_Manager_StormwaterDevice.cpp
//Note: Snow depths are liquid depths, also known as snow water equivalent

//AggregateOutput::SumCatchmentTotals function will sum and store all catchment variables as volumes (m3) used in the final budget analysis
//Note: For model = SpatialTemperatureHydro, DataDrawer_ID has maximum = maxRows * maxCols, and DataFolder_ID typically has maximum = 1
void AggregateOutput::SumCatchmentTotals(Inputs* input, CompactRagged* beC, int DataDrawer_ID, int DataFolder_ID, int timeStep)
{
	//Catchment_PerviousArea_AffectingStorageDeficit_m2 (m2) contains entire catchment pervious area draining to vadose zone
	//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 influences the catchment variable, StorageDeficit_VadoseZone_m
	double Catchment_PerviousArea_AffectingStorageDeficit_m2 = input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"];

	//If Counter_SimulatedUnit equals 1, then it is first map pixel simulated
	//Note: Reset all variable values for each time step when entering the first DataFolder of the first DataDrawer
	if (input->RepoDict["Counter_SimulatedUnit"] == 1) {
		//If timeStep > zero then record prior time step storage
		if (timeStep > 0) {
			//TS_Prior_Sum variables defined by the TS_Sum from the prior time step, before it is reset to zero
			input->RepoVecDict["StorageSnow_noCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_noCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_UnderTree_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_UnderSVeg_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_TreeCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageSnow_SVegCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageSnow_SVegCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageRain_TreeCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"];
			input->RepoVecDict["StorageRain_SVegCanopy_TS_Prior_Sum_m3"] = input->RepoVecDict["StorageRain_SVegCanopy_TS_Sum_m3"];
			input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"] = input->RepoVecDict["Storage_ImperviousPondedWater_TS_Sum_m3"];
			input->RepoVecDict["Storage_PerviousPondedWater_TS_Prior_Sum_m3"] = input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"];
		}
		//call Initialize function to prepare variables
		AggregateOutput::Initialize(input, timeStep);

		//StorageDeficit_VadoseZone_m (m) arrives from StorageDeficit_VadoseZone.cpp scaled to catchment pervious area draining to shared vadose zone
		//If timeStep equals zero then store variables used for start analysis, where timeStep starts at 0 in C++ 
		if (timeStep == 0) {
			//StorageDeficit_VadoseZone_TS_first_m3 (m3) is product of input->RepoDict["StorageDeficit_VadoseZone_m"] and Catchment_PerviousArea_AffectingStorageDeficit_m2
			//Note: StorageDeficit_VadoseZone_m (m) is not accumulated as a total or sum, but stored as a value per time step; deficits are not typically summed
			input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"] = input->RepoDict["StorageDeficit_VadoseZone_m"] * Catchment_PerviousArea_AffectingStorageDeficit_m2;
		}
		//If (timeStep+1) equals SimulationTimePeriod_timeSteps then store variables used for finish analysis
		else if ((timeStep + 1) == input->SimulationTimePeriod_timeSteps) {
			//StorageDeficit_VadoseZone_TS_last_m3 (m3) is product of input->RepoDict["StorageDeficit_VadoseZone_m"] and Catchment_PerviousArea_AffectingStorageDeficit_m2
			//Note: StorageDeficit_VadoseZone_m (m) is not accumulated as a total or sum, but stored as a value per time step; deficits are not typically summed
			input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"] = input->RepoDict["StorageDeficit_VadoseZone_m"] * Catchment_PerviousArea_AffectingStorageDeficit_m2;
		}
		//input->RepoDict["StorageDeficit_VadoseZone_Sum_m3"] (m3) is product of input->RepoDict["StorageDeficit_VadoseZone_m"] and Catchment_PerviousArea_AffectingStorageDeficit_m2
		//Note: StorageDeficit_VadoseZone_m (m) is not accumulated as a total or sum, but stored as a value per time step; deficits are not typically summed
		input->RepoDict["StorageDeficit_VadoseZone_Sum_m3"] = input->RepoDict["StorageDeficit_VadoseZone_m"] * Catchment_PerviousArea_AffectingStorageDeficit_m2;

	}
	int Folder_Type_ID;
	int Folder_BulkArea_ID = 0;
	int Folder_GI_ID = 1;
	double Folder_Area_m2;
	double Catchment_Pervious_Area_m2;
	double Ratio_FolderTypeArea_to_CatchmentTypeArea;
	double IC_m2;
	double IC_noTC_m2;
	double PC_m2;
	double TC_m2;
	double TC_overIC_m2;
	double TC_overPC_m2;
	double SVegC_noTC_m2;
	double SoilC_noTC_m2;
	double PermeablePavementC_noTC_m2;
	double WaterC_noTC_m2;
	double NoCanopy_m2;
	int Drawer_count = beC->n_drawers(beC);// old version: Folder_count = DataDrawers.size()?
	int Folder_count = beC->folders_in_drawer(beC, DataDrawer_ID);

	//If HydroPlusConfig.xml Type equals BulkArea then
	if (beC->by_key_str(DataDrawer_ID, DataFolder_ID, "Type") == "BulkArea") {
		//Folder_Type_ID equals zero, Folder_BulkArea_ID = 0
		Folder_Type_ID = Folder_BulkArea_ID;
	}
	//Else Type is GreenInfrastructure
	else {
		//Folder_Type_ID equals 1, Folder_GI_ID = 1
		Folder_Type_ID = Folder_GI_ID;
	}

	//Folder_Area_m2 (m2) set from HydroPlusConfig.xml element Area_m2, within each DataFolder, BulkArea and GI
	Folder_Area_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "Area_m2");
	//Catchment_Pervious_Area_m2 is pervious cover area of folder, which may be larger than the pervious area draining to the vadose zone
	//Note: For GI applications, Catchment_Pervious_Area_m2 is too small, and should be product of BulkArea And GI Area
	//Note: Folder_Area_m2 = Folder_Area_Bulk_m2 + Folder_Area_GI_m2, but it is not clear how to sum this within a DataDrawer with use of folders
	Catchment_Pervious_Area_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "PerviousCover_frac") * Folder_Area_m2;
	//Ratio_FolderTypeArea_to_CatchmentTypeArea (fraction) is ratio of Folder_Area_m2 (m2) to CatchmentArea_m2 (m2)
	//Note: GI applications had used ratio of Folder_Area_m2 (m2) to Catchment_Type_Area_m2 (m2)
	//Note: GI same w/ Ratio_FolderTypeArea_to_CatchmentTypeArea = Folder_Area_m2 / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
	Ratio_FolderTypeArea_to_CatchmentTypeArea = Folder_Area_m2 / input->SimulationNumericalParams["CatchmentArea_m2"];
	//PC_m2 is pervious cover area (m2) of catchment area
	PC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "PerviousCover_frac") * Folder_Area_m2;
	//IC_m2 is impervious cover area (m2) of catchment area
	IC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "ImperviousCover_frac")* Folder_Area_m2;
	//TC_m2 is tree cover area (m2) of catchment area
	TC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCover_frac") * Folder_Area_m2;
	//SVeg_noTC_m2 is short vegetation cover not under tree cover area (m2) of catchment area
	SVegC_noTC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "ShortVegCover_noTreeCanopy_frac") * Folder_Area_m2;
	//Soil_noTC_m2 is soil cover not under tree cover area (m2) of catchment area
	SoilC_noTC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "SoilCover_noTreeCanopy_frac") * Folder_Area_m2;
	//PermeablePavementC_noTC_m2 is permeable pavement cover not under tree cover area (m2) of catchment area
	PermeablePavementC_noTC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "PermeablePavementCover_noTreeCanopy_frac") * Folder_Area_m2;
	//Water_noTC_m2 is water cover not under tree cover area (m2) of catchment area
	WaterC_noTC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "WaterCover_noTreeCanopy_frac") * Folder_Area_m2;
	//TC_overPC_m2 is tree cover over pervious cover area (m2) of catchment area
	TC_overPC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCanopyCover_overPervious_frac") * Folder_Area_m2;
	//TC_overIC_m2 is tree cover over impervious cover area (m2) of catchment area
	TC_overIC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "TreeCanopyCover_overImpervious_frac") * Folder_Area_m2;
	//IC_noTC_m2 is impervious cover not under tree cover area (m2) of catchment area
	IC_noTC_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "ImperviousCover_noTreeCanopy_frac") * Folder_Area_m2;
	//NoCanopy_m2 is area (m2) without tree or short vegetation canopy cover
	//Note: Altneratively, use: NoCanopy_m2 = Folder_Area_m2 - TC_m2 - SVegC_noTC_m2;
	NoCanopy_m2 = SoilC_noTC_m2 + PermeablePavementC_noTC_m2 + IC_noTC_m2 + WaterC_noTC_m2;

	//If Folder_Type_ID and timeStep equals zero, Folder_BulkArea_ID then create catchment water balance parameters 
	if (Folder_Type_ID == Folder_BulkArea_ID and timeStep == 0) {
		input->RepoDict["Drainage_VadoseZone_Catchment_WB_m3"] = 0.0;
		input->RepoDict["Runoff_Subsurface_Catchment_WB_m3"] = 0.0;
	}

	//Store as catchment variables terms related to catchment StorageDeficit_VadoseZone_m balance 
	input->RepoDict["Runoff_Subsurface_Catchment_WB_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Subsurface_m") * Folder_Area_m2;
	//If Folder_Type_ID equals zero, Folder_BulkArea_ID then BulkArea
	if (Folder_Type_ID == Folder_BulkArea_ID) {
		input->RepoDict["Drainage_VadoseZone_Catchment_WB_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_VadoseZone_m") * PC_m2;
	}
	//Else If Folder_Type_ID equals one, Folder_GI_ID then GI
	else if (Folder_Type_ID == Folder_GI_ID) {
		input->RepoDict["Drainage_VadoseZone_Catchment_WB_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_VadoseZone_m3");
	}

	//If timeStep equals zero or (timeStep+1) equals SimulationTimePeriod_timeSteps then store variables used for start and finish analysis
	//Note: Given timeStep starts at 0 in c++, 1 must be added to ever equal SimulationTimePeriod_timeSteps
	if (timeStep == 0 || (timeStep + 1) == input->SimulationTimePeriod_timeSteps) {
		//TS_First_Last function stores values for 1st and last time step
		AggregateOutput::TS_First_Last(input, beC, DataDrawer_ID, DataFolder_ID, timeStep, Folder_Type_ID, Folder_Area_m2, PC_m2, IC_m2, TC_m2, SVegC_noTC_m2, SoilC_noTC_m2, PermeablePavementC_noTC_m2, WaterC_noTC_m2, IC_noTC_m2, NoCanopy_m2, Ratio_FolderTypeArea_to_CatchmentTypeArea, Catchment_Pervious_Area_m2);
	}
		
	//Note: Routine stores variables as RepoVecDict ending with "TS_Sum_m3"; these are volumes for the time step 
	input->RepoVecDict["InterceptionRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Interception_Rain_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["InterceptionSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Interception_Snow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["InterceptionRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Interception_Rain_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["InterceptionSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Interception_Snow_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["InterceptionRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Interception_RainSnow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["InterceptionRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Interception_RainSnow_SVegCanopy_m") * SVegC_noTC_m2;

	input->RepoVecDict["SublimationSnow_noCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_noCanopy_m") * NoCanopy_m2;
	input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_noCanopy_m") * IC_noTC_m2;
	input->RepoVecDict["SublimationSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_noCanopy_m") * SoilC_noTC_m2;
	input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_noCanopy_m") * PermeablePavementC_noTC_m2;
	input->RepoVecDict["SublimationSnow_WaterNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_noCanopy_m") * WaterC_noTC_m2;
	input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_underTreeCanopy_m") * TC_m2;
	input->RepoVecDict["SublimationSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_underSVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["SublimationSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["SublimationSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "SublimationSnow_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["EvaporationRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "EvaporationRain_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["EvaporationRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "EvaporationRain_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["EvaporationSublimation_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "EvaporationSublimation_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["EvaporationSublimation_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "EvaporationSublimation_SVegCanopy_m") * SVegC_noTC_m2;

	input->RepoVecDict["StorageSnow_noCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * NoCanopy_m2;
	input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * IC_noTC_m2;
	input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * SoilC_noTC_m2;
	input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * PermeablePavementC_noTC_m2;
	input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * WaterC_noTC_m2;
	input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_underTreeCanopy_m") * TC_m2;
	input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_underSVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["StorageSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Rain_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["StorageRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Rain_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["StorageRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_RainSnow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["StorageRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_RainSnow_SVegCanopy_m") * SVegC_noTC_m2;

	input->RepoVecDict["SnowMelt_noCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_noCanopy_m") * NoCanopy_m2;
	input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_noCanopy_m") * IC_noTC_m2;
	input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_noCanopy_m") * SoilC_noTC_m2;
	input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_noCanopy_m") * PermeablePavementC_noTC_m2;
	input->RepoVecDict["SnowMelt_WaterNoCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_noCanopy_m") * WaterC_noTC_m2;
	input->RepoVecDict["SnowMelt_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["SnowMelt_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_underTreeCanopy_m") * TC_m2;
	input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Melt_Snow_underSVegCanopy_m") * SVegC_noTC_m2;

	input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "ThroughFall_Rain_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "ThroughFall_Snow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "ThroughFall_Rain_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "ThroughFall_Snow_SVegCanopy_m") * SVegC_noTC_m2;
	input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "ThroughFall_RainSnow_TreeCanopy_m") * TC_m2;
	input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "ThroughFall_RainSnow_SVegCanopy_m") * SVegC_noTC_m2;

	input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_m") * IC_m2;
	input->RepoVecDict["Storage_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_ImperviousPondedWater_m") * IC_m2;
	input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_PerviousPondedWater_m") * PC_m2;
	input->RepoVecDict["Storage_WaterPondedWater_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_WaterPondedWater_m") * WaterC_noTC_m2;
	//Note: Division of variables by fractional area below is to adjust depths from entire folder area to one analysis area
	//Rationale: 1. variables arrive as depths for the entire folder area
	//Rationale: 2. multiplying these depths by the folder area gives a volume
	//Rationale: 3. dividing these depths by the analysis area is equivalent to multiplying by the folder area and dividing by the analysis area, getting depth for analysis area
	//Note: Division is equivalent to multiplying by ratio of folder area to analysis area to convert from /folder_area to /analysis_area
	//If PerviousCover_frac > 0 then divide by PerviousCover_frac
	if (beC->by_key(DataDrawer_ID, DataFolder_ID, "PerviousCover_frac") > 0) {
		//Flux_Impervious_to_Pervious_m (m) arrives as folder depth from ImperviousDepressionStorageCalc::calculate, converted to pervious cover depth by division
		input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] += (beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_Impervious_to_Pervious_m") /
			beC->by_key(DataDrawer_ID, DataFolder_ID, "PerviousCover_frac")) * PC_m2;
	}
	//Else avoid division by zero and add zero flow
	else { input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] += 0; }

	input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_m") * PC_m2;
	input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_m") * PC_m2;
	input->RepoVecDict["Flux_to_WaterArea_Rain_SnowMelt_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_to_WaterArea_Rain_SnowMelt_m") * PC_m2;
	input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Evaporation_ImperviousPondedWater_m") * IC_m2;
	input->RepoVecDict["Evaporation_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Evaporation_PerviousPondedWater_m") * PC_m2;
	input->RepoVecDict["Evaporation_WaterPondedWater_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Evaporation_WaterPondedWater_m") * WaterC_noTC_m2;

	input->RepoVecDict["Flux_to_Infiltration_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_to_Infiltration_m") * PC_m2;
	input->RepoVecDict["Drainage_macroPore_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_macroPore_m") * PC_m2;
	input->RepoVecDict["Infiltration_viaInfilEx_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Infiltration_viaInfilEx_m") * PC_m2;
	input->RepoVecDict["Infiltration_viaSatEx_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Infiltration_viaSatEx_m") * PC_m2;
	input->RepoVecDict["Infiltration_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Infiltration_m") * PC_m2;

	input->RepoVecDict["Infiltration_viaSatEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Infiltration_viaSatEx_PreRunoffSatExcess_m") * PC_m2;
	input->RepoVecDict["Infiltration_viaInfilEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Infiltration_viaInfilEx_PreRunoffSatExcess_m") * PC_m2;
	input->RepoVecDict["Drainage_macroPore_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_macroPore_PreRunoffSatExcess_m") * PC_m2;
	input->RepoVecDict["Drainage_SoilEvapZone_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_SoilEvapZone_PreRunoffSatExcess_m") * PC_m2;

	//Note: EvapoTranspiration_SoilEvapZone_m is weighted by PC_m2 and presumes any trees over impervious are evapotranspiring water from pervious
	input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "EvapoTranspiration_SoilEvapZone_m") * PC_m2;
	input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m") * PC_m2;
	//If Evapotranspiration_Depth_m > 0 then enter for division
	if (beC->by_key(DataDrawer_ID, DataFolder_ID, "Evapotranspiration_Depth_m") > 0) {

		//Note: Storage_SoilEvapZone_m3_p_m3 (m3/m3) defined for each folder, then summed for entire catchment
		//SoilMoisture_SoilEvapZone_Capillary_m3pm3 = (1 - (StorageDeficit_FieldCapacity_SoilEvapZone_m / StorageDeficit_FieldCapacity_SoilEvapZone_Max_m) * (Soil_FieldCapacity_m3pm3 - Soil_WiltingPoint_m3pm3) + Soil_WiltingPoint_m3pm3
		double SoilMoisture_SoilEvapZone_Capillary_m3pm3 = (1 - Inputs::SafeDivide(beC->by_key(DataDrawer_ID, DataFolder_ID, "StorageDeficit_FieldCapacity_SoilEvapZone_m"), beC->by_key(DataDrawer_ID, DataFolder_ID, "StorageDeficit_FieldCapacity_SoilEvapZone_Max_m"))) * (beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_FieldCapacity_m3pm3") - beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_WiltingPoint_m3pm3")) + input->InputXml["Soil_WiltingPoint_m3pm3"];

		//SoilMoisture_VadoseZone_Gravitational_m3pm3 = (Storage_VadoseZone_m / StorageDeficit_VadoseZone_m) * (Soil_SaturationPoint_m3pm3 - Soil_FieldCapacity_m3pm3)
		double SoilMoisture_VadoseZone_Gravitational_m3pm3 = Inputs::SafeDivide(beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_VadoseZone_m"), input->RepoDict["StorageDeficit_VadoseZone_m"]) * (beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_SaturationPoint_m3pm3") - beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_FieldCapacity_m3pm3"));

		//Storage_SoilEvapZone_m3_p_m3 = SoilMoisture_SoilEvapZone_Capillary_m3pm3 + SoilMoisture_VadoseZone_Gravitational_m3pm3
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m3_p_m3") = SoilMoisture_SoilEvapZone_Capillary_m3pm3 + SoilMoisture_VadoseZone_Gravitational_m3pm3;

		//Note: Prior computation of Capillary water had reversed engineered of the computation of Storage_SoilEvapZone_TS_0th_m term in BuildDataOrganizer.cpp; The product of Evapotranspiration_Depth_m (m) and Soil_FieldCapacity_m3pm3 gives the length of water retained at field capacity
		//beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m3_p_m3") = (beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m") - beC->by_key(MapPixel_ID, DataFolder_ID, "StorageDeficit_FieldCapacity_SoilEvapZone_Max_m") + beC->by_key(MapPixel_ID, DataFolder_ID, "Evapotranspiration_Depth_m") * beC->by_key(MapPixel_ID, DataFolder_ID, "Soil_FieldCapacity_m3pm3")) / beC->by_key(MapPixel_ID, DataFolder_ID, "Evapotranspiration_Depth_m");
	}
	//Else if Evapotranspiration_Depth_m = 0, then Storage_SoilEvapZone_m3_p_m3 = 0
	else { 
		//Storage_SoilEvapZone_m3_p_m3 (m3/m3) set to 0
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m3_p_m3") = 0;
	}
	//Storage_SoilEvapZone_TS_Sum_m3_p_m3 (m3/m3) is catchment average volumetric soil water, only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
	//Note: Storage_SoilEvapZone_m3_p_m3 is volumetric moisture, and division by Folder_count needed to weight the contribution of each folder
	input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3_p_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m3_p_m3") / Drawer_count;
	input->RepoVecDict["Drainage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_SoilEvapZone_m") * PC_m2;
	input->RepoVecDict["Storage_VadoseZone_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_VadoseZone_m") * PC_m2;
	input->RepoVecDict["Drainage_VadoseZone_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_VadoseZone_m") * PC_m2;

	//Depth_to_Groundwater_TS_m (m) is time series storage of Depth_to_Groundwater_Table_m (m), depth to groundwater table
	input->RepoDict["Depth_to_Groundwater_TS_m"] = input->InputXml["Depth_to_Groundwater_Table_m"];

	input->RepoVecDict["Groundwater_surficial_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Groundwater_surficial_m") * PC_m2;
	input->RepoVecDict["Flux_SatExGW_to_SoilFieldCapacity_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Flux_SatExGW_to_SoilFieldCapacity_m") * PC_m2;
	input->RepoVecDict["Groundwater_surficial_TS_Sum_frac"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Groundwater_surficial_frac") * Ratio_FolderTypeArea_to_CatchmentTypeArea;
	input->RepoVecDict["GI_Ratio_RunoffInfilExcess_to_RunoffPervious_TS_Sum_frac"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Ratio_RunoffInfilExcess_to_RunoffPervious") * Ratio_FolderTypeArea_to_CatchmentTypeArea;

	input->RepoVecDict["Runoff_InfilExcess_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_InfilExcess_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_SatExcess_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_SatExcess_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Pervious_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Impervious_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_Water_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Water_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Subsurface_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_Surface_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Surface_m") * Folder_Area_m2;
	input->RepoVecDict["Runoff_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_m") * Folder_Area_m2;

	//Accumulating catchment runoff volumes (m3), which will combine BulkArea and GI runoff
	input->RepoDict["Runoff_Pervious_Catchment_TS_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Pervious_m") * Folder_Area_m2;
	input->RepoDict["Runoff_Impervious_Catchment_TS_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Impervious_m") * Folder_Area_m2;
	input->RepoDict["Runoff_Water_Catchment_TS_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Water_m") * Folder_Area_m2;
	input->RepoDict["Runoff_Subsurface_Catchment_TS_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Subsurface_m") * Folder_Area_m2;
	input->RepoDict["Runoff_Surface_Catchment_TS_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Surface_m") * Folder_Area_m2;
	input->RepoDict["Runoff_Catchment_TS_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_m") * Folder_Area_m2;

	//Note: Transfers of water from BulkArea to GI; values are constant for each folder within a DataDrawer
	input->RepoVecDict["Runoff_Impervious_BAtoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_Impervious_BAtoGI_m3"];
	input->RepoVecDict["Runoff_Water_BAtoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_Water_BAtoGI_m3"];
	input->RepoVecDict["Runoff_SatExcess_BAtoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_SatExcess_BAtoGI_m3"];
	input->RepoVecDict["Runoff_InfilExcess_BAtoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_InfilExcess_BAtoGI_m3"];

	//Note: Transfers of water from upslope GI to downslope GI in series when Flag_Outflow_to_NextFolder = 1 from HydroPlusConfig.xml
	input->RepoVecDict["Runoff_Impervious_GItoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_Impervious_GItoGI_m3"];
	input->RepoVecDict["Runoff_Pervious_GItoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_Pervious_GItoGI_m3"];
	input->RepoVecDict["Runoff_Vault_GItoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_Vault_GItoGI_m3"];
	input->RepoVecDict["Runoff_Water_GItoGI_TS_Sum_m3"][Folder_Type_ID] += input->RepoDict["Runoff_Water_GItoGI_m3"];

	input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Irrigation_TreeInterceptionStorage_m") * TC_m2;
	input->RepoVecDict["Irrigation_SVegInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Irrigation_SVegInterceptionStorage_m") * SVegC_noTC_m2;
	input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Irrigation_ImperviousDepressionStorage_m") * IC_m2;
	input->RepoVecDict["Irrigation_PerviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Irrigation_PerviousDepressionStorage_m") * PC_m2;

	//If Folder_Type_ID equals one, Folder_GI_ID then GI Type folder and add to totals
	if (Folder_Type_ID == Folder_GI_ID) {
		//Note: GI folder variables arrive as GI folder area volumes, not GI device volumes; all variables have zero values for BulkArea folder
		input->RepoDict["GI_Precipitation_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Precipitation_m3");
		input->RepoDict["GI_Runon_from_BulkArea_Surface_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Inflow_from_BulkArea_m3");
		input->RepoDict["GI_Infiltration_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Infiltration_m3");
		input->RepoDict["GI_Runoff_Surface_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Runoff_Surface_m3");
		input->RepoDict["GI_Runoff_VaultDrain_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Runoff_Vault_m3");
		input->RepoDict["GI_Storage_Surface_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_SurfaceLayer_m3");
		input->RepoDict["GI_PavementWater_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_PavementLayer_m3");
		input->RepoDict["GI_Storage_Soil_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_SoilLayer_m3");
		input->RepoDict["GI_Storage_Vault_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_VaultLayer_m3");
		input->RepoDict["GI_Evaporation_Surface_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_EvaporationSurface_m3");
		input->RepoDict["GI_Evaporation_Pavement_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_EvaporationPavement_m3");
		input->RepoDict["GI_Evaporation_Soil_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_EvaporationSoil_m3");
		input->RepoDict["GI_Evaporation_Vault_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_EvaporationVault_m3");
		input->RepoDict["GI_Drainage_Percolation_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_Percolation_m3");
		input->RepoDict["GI_Drainage_PercolationPavement_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_PercolationPavement_m3");
		input->RepoDict["GI_Drainage_Exfiltration_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_Exfiltration_m3");
		input->RepoDict["GI_Drainage_VadoseZone_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_VadoseZone_m3");
		input->RepoDict["GI_Storage_GroundwaterMound_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GroundwaterMound_m3");
		input->RepoDict["GI_Storage_GI_Pavement_Max_Potential_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GI_Pavement_Max_Potential_m3");
		input->RepoDict["GI_Storage_GI_Pavement_Max_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GI_Pavement_Max_m3");
		input->RepoDict["GI_Storage_GI_Soil_Max_Potential_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GI_Soil_Max_Potential_m3");
		input->RepoDict["GI_Storage_GI_Soil_Max_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GI_Soil_Max_m3");
		input->RepoDict["GI_Storage_GI_Vault_Max_Potential_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GI_Vault_Max_Potential_m3");
		input->RepoDict["GI_Storage_GI_Vault_Max_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GI_Vault_Max_m3");

		//Define GI exfiltration to native soils based on presence of storage layer
		//If Vault_Thickness_m equals zero then GI_Drainage_Percolation_m connects to native soils
		if (beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Thickness_m") == 0.0) {
			//GI_DrainageVadoseZone_m (m) is updated with GI_Drainage_Percolation_m when there is only soil layer
			input->RepoDict["GI_Drainage_VadoseZone_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_Percolation_m") * Folder_Area_m2;
		}
		//Else If Vault_Thickness_m is not zero then GI_Drainage_Exfiltration_m connects to native soils
		else {
			//GI_DrainageVadoseZone_m (m) is updated with GI_Drainage_Exfiltration_m when there is a storage layer
			input->RepoDict["GI_Drainage_VadoseZone_TS_Sum_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Drainage_Exfiltration_m") * Folder_Area_m2;
		}
	}

	if (input->SimulationStringParams["Model_Selection"] != "SpatialTemperatureHydro") {
		//For Loop through pollutant names to create string with variable name and double with variable magnitude; length of For Loop is Pollutant_Name.size()
		//Note: Loop to store catchment totals for the time step. Mean/median need to be sum up for folders (bulk area + GI).
		for (int pollutant_ID = 0; pollutant_ID < input->Pollutant_Name.size(); pollutant_ID++) {
			//Create string to represent variable associated with Pollutant_Name
			Name_Load_Catchment_Pollutant_Median_kg = "Load_Catchment_" + input->Pollutant_Name[pollutant_ID] + "_Median_kg";
			Name_Load_Catchment_Pollutant_Mean_kg = "Load_Catchment_" + input->Pollutant_Name[pollutant_ID] + "_Mean_kg";
			//Load_Catchment_Pollutant_Median_kg (kg) will store catchment pollutant median load for time step
			input->RepoDict[Name_Load_Catchment_Pollutant_Median_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_Catchment_Pollutant_Median_kg);
			//Load_Catchment_Pollutant_Mean_kg (kg) will store catchment pollutant mean load for time step
			input->RepoDict[Name_Load_Catchment_Pollutant_Mean_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_Catchment_Pollutant_Mean_kg);

			//Create string to represent variable associated with Pollutant_Name
			Name_Load_Catchment_Pollutant_Median_Sum_kg = "Load_Catchment_" + input->Pollutant_Name[pollutant_ID] + "_Median_Sum_kg";
			Name_Load_Catchment_Pollutant_Mean_Sum_kg = "Load_Catchment_" + input->Pollutant_Name[pollutant_ID] + "_Mean_Sum_kg";
			//Name_Pollutant_Load_Median_Sum_kg (kg) will store catchment pollutant median load sum for all time steps
			input->RepoDict[Name_Load_Catchment_Pollutant_Median_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_Catchment_Pollutant_Median_kg);
			//Name_Pollutant_Load_Mean_Sum_kg (kg) will store catchment pollutant mean load sum for all time steps
			input->RepoDict[Name_Load_Catchment_Pollutant_Mean_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_Catchment_Pollutant_Mean_kg);

			//Create string to represent variable associated with Pollutant_Name
			Name_Load_GI_Inflow_Pollutant_Mean_g = "Load_GI_Inflow_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
			Name_Load_GI_Runoff_Surface_Pollutant_Mean_g = "Load_GI_Runoff_Surface_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
			Name_Load_GI_Runoff_Vault_Pollutant_Mean_g = "Load_GI_Runoff_Vault_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
			Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g = "Load_GI_Drainage_VadoseZone_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
			Name_Release_GI_Total_Pollutant_Mean_g = "Release_GI_Total_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";

			//Name_Load_GI_Inflow_Pollutant_Mean_g (g) will store catchment pollutant load for time step
			input->RepoDict[Name_Load_GI_Inflow_Pollutant_Mean_g] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Inflow_Pollutant_Mean_g);
			input->RepoDict[Name_Load_GI_Runoff_Surface_Pollutant_Mean_g] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Runoff_Surface_Pollutant_Mean_g);
			input->RepoDict[Name_Load_GI_Runoff_Vault_Pollutant_Mean_g] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Runoff_Vault_Pollutant_Mean_g);
			input->RepoDict[Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g);
			input->RepoDict[Name_Release_GI_Total_Pollutant_Mean_g] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Release_GI_Total_Pollutant_Mean_g);

			//Create string to represent variable associated with Pollutant_Name
			Name_Load_GI_Inflow_Pollutant_Mean_Sum_kg = "Load_GI_Inflow_" + input->Pollutant_Name[pollutant_ID] + "_Mean_Sum_kg";
			Name_Load_GI_Runoff_Surface_Pollutant_Mean_Sum_kg = "Load_GI_Runoff_Surface_" + input->Pollutant_Name[pollutant_ID] + "_Mean_Sum_kg";
			Name_Load_GI_Runoff_Vault_Pollutant_Mean_Sum_kg = "Load_GI_Runoff_Vault_" + input->Pollutant_Name[pollutant_ID] + "_Mean_Sum_kg";
			Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_Sum_kg = "Load_GI_Drainage_VadoseZone_" + input->Pollutant_Name[pollutant_ID] + "_Mean_Sum_kg";
			Name_Release_GI_Total_Pollutant_Mean_Sum_kg = "Release_GI_Total_" + input->Pollutant_Name[pollutant_ID] + "_Mean_Sum_kg";

			//Name_Load_GI_Inflow_Pollutant_Mean_Sum_kg (kg) will store catchment pollutant load sum (g) for all time steps multiplied by Ratio_kg_to_g 
			input->RepoDict[Name_Load_GI_Inflow_Pollutant_Mean_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Inflow_Pollutant_Mean_g) * Ratio_kg_to_g;
			input->RepoDict[Name_Load_GI_Runoff_Surface_Pollutant_Mean_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Runoff_Surface_Pollutant_Mean_g) * Ratio_kg_to_g;
			input->RepoDict[Name_Load_GI_Runoff_Vault_Pollutant_Mean_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Runoff_Vault_Pollutant_Mean_g) * Ratio_kg_to_g;
			input->RepoDict[Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g) * Ratio_kg_to_g;
			input->RepoDict[Name_Release_GI_Total_Pollutant_Mean_Sum_kg] += beC->by_key(DataDrawer_ID, DataFolder_ID, Name_Release_GI_Total_Pollutant_Mean_g) * Ratio_kg_to_g;
		}
	}

}

void AggregateOutput::SumTimeStepTotals(Inputs *input, int timeStep)
{
	double CA_m2;
	int Folder_BulkArea_ID = 0;
	int Folder_GI_ID = 1;
	vector<int> Folder_List;

	//Folder_List is defined to contain Folder_BulkArea_ID
	Folder_List = { Folder_BulkArea_ID };
	//If Flag_GI_Simulated equals 1 then add GI to Folder_List
	if (input->RepoDict["Flag_GI_Simulated"] == 1) {
		//Folder_List is redefined to contain Folder_BulkArea_ID and Folder_GI_ID
		Folder_List = { Folder_BulkArea_ID, Folder_GI_ID };
	}
	//For Loop with Folder_Type_ID, representing values of BulkArea and GI
	//for (int Folder_Type_ID = 0; Folder_Type_ID < (Folder_GI_ID + 1); ++Folder_Type_ID ) {
	//for (auto Folder_Type_ID = 0; Folder_Type_ID < Folder_List.size(); ++Folder_Type_ID) {
	for (auto Folder_Type_ID : Folder_List) {

		//CA_m2 (m2) is catchment area, defined the same for Model equals StatisticalHydro or SpatialTemperatureHydro
		CA_m2 = input->SimulationNumericalParams["CatchmentArea_m2"];

		//Routine to store variables named ... _Sum_m3, which contain simulation total time step value as volume (m3) for use in mass balance
		//Variables containing _Sum_m3, sum for simulation, accumulate variables containing _TS_Sum_m3, sum for time step
		input->RepoVecDict["InterceptionRain_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["InterceptionRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["InterceptionSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["InterceptionSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["InterceptionRain_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["InterceptionRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["InterceptionSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["InterceptionSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["InterceptionRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["InterceptionRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["InterceptionRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["InterceptionRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["SublimationSnow_noCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_noCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_SoilNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_WaterNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_WaterNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_UnderTree_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_UnderSVeg_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SublimationSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SublimationSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["EvaporationRain_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["EvaporationRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["EvaporationRain_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["EvaporationRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["EvaporationSublimation_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["EvaporationSublimation_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["EvaporationSublimation_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["EvaporationSublimation_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["StorageSnow_noCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_noCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_ImperviousNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_SoilNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_WaterNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_UnderTree_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_UnderSVeg_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageRain_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageRain_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["StorageRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["StorageRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["SnowMelt_noCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_noCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_ImperviousNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_SoilNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_WaterNoCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_WaterNoCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_UnderTree_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_UnderSVeg_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["SnowMelt_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["SnowMelt_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["ThroughFallRain_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["ThroughFallRain_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["ThroughFallSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["ThroughFallSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Storage_ImperviousPondedWater_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Storage_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Storage_PerviousPondedWater_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Storage_WaterPondedWater_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Storage_WaterPondedWater_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Flux_Impervious_to_Pervious_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Flux_to_WaterArea_Rain_SnowMelt_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_to_WaterArea_Rain_SnowMelt_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["Evaporation_ImperviousPondedWater_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Evaporation_PerviousPondedWater_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Evaporation_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Evaporation_WaterPondedWater_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Evaporation_WaterPondedWater_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["Flux_to_Infiltration_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_to_Infiltration_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Drainage_macroPore_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Drainage_macroPore_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Infiltration_viaInfilEx_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Infiltration_viaInfilEx_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Infiltration_viaSatEx_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Infiltration_viaSatEx_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Infiltration_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Infiltration_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["Infiltration_viaSatEx_PreSatExRunoff_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Infiltration_viaSatEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Infiltration_viaInfilEx_PreSatExRunoff_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Infiltration_viaInfilEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Drainage_macroPore_PreSatExRunoff_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Drainage_macroPore_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Drainage_SoilEvapZone_PreSatExRunoff_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Drainage_SoilEvapZone_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["EvapoTranspiration_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Storage_SoilEvapZone_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Drainage_SoilEvapZone_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Drainage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Drainage_VadoseZone_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Drainage_VadoseZone_TS_Sum_m3"][Folder_Type_ID];

		//Groundwater_surficial_Sum_m3 (m3) is increased to include Groundwater_surficial_TS_Sum_m3 (m3)
		input->RepoVecDict["Groundwater_surficial_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Groundwater_surficial_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Flux_SatExGW_to_SoilFieldCapacity_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Flux_SatExGW_to_SoilFieldCapacity_TS_Sum_m3"][Folder_Type_ID];

		//Runoff_Impervious_to_Outlet_m (m) is catchment area depth, transformed from pervious area depth in RunoffSummation.cpp
		input->RepoVecDict["Runoff_InfilExcess_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_InfilExcess_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_SatExcess_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_SatExcess_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Pervious_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Impervious_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Water_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Water_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Subsurface_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Surface_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Surface_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_TS_Sum_m3"][Folder_Type_ID];

		//Note: Transfers of water from BulkArea to GI; values are constant for each folder within a DataDrawer
		input->RepoVecDict["Runoff_Impervious_BAtoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Impervious_BAtoGI_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Water_BAtoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Water_BAtoGI_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_SatExcess_BAtoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_SatExcess_BAtoGI_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_InfilExcess_BAtoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_InfilExcess_BAtoGI_TS_Sum_m3"][Folder_Type_ID];

		//Note: Transfers of water from upslope GI to downslope GI in series when Flag_Outflow_to_NextFolder = 1 from HydroPlusConfig.xml
		input->RepoVecDict["Runoff_Impervious_GItoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Impervious_GItoGI_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Pervious_GItoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Pervious_GItoGI_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Vault_GItoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Vault_GItoGI_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Runoff_Water_GItoGI_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Runoff_Water_GItoGI_TS_Sum_m3"][Folder_Type_ID];

		input->RepoVecDict["Irrigation_TreeInterceptionStorage_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Irrigation_SVegInterceptionStorage_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Irrigation_SVegInterceptionStorage_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Irrigation_ImperviousDepressionStorage_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID];
		input->RepoVecDict["Irrigation_PerviousDepressionStorage_Sum_m3"][Folder_Type_ID] += input->RepoVecDict["Irrigation_PerviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID];

		//If Folder_Type_ID equals one, foFolder_GI_IDlder then GI Type and add to totals
		if (Folder_Type_ID == Folder_GI_ID) {
			input->RepoDict["GI_Precipitation_Sum_m3"] += input->RepoDict["GI_Precipitation_TS_Sum_m3"];
			input->RepoDict["GI_Runon_from_BulkArea_Surface_Sum_m3"] += input->RepoDict["GI_Runon_from_BulkArea_Surface_TS_Sum_m3"];
			input->RepoDict["GI_Infiltration_Sum_m3"] += input->RepoDict["GI_Infiltration_TS_Sum_m3"];
			input->RepoDict["GI_Runoff_Surface_Sum_m3"] += input->RepoDict["GI_Runoff_Surface_TS_Sum_m3"];
			input->RepoDict["GI_Runoff_VaultDrain_Sum_m3"] += input->RepoDict["GI_Runoff_VaultDrain_TS_Sum_m3"];
			input->RepoDict["GI_Storage_Surface_Sum_m3"] += input->RepoDict["GI_Storage_Surface_TS_Sum_m3"];
			input->RepoDict["GI_PavementWater_Sum_m3"] += input->RepoDict["GI_PavementWater_TS_Sum_m3"];
			input->RepoDict["GI_Storage_Soil_Sum_m3"] += input->RepoDict["GI_Storage_Soil_TS_Sum_m3"];
			input->RepoDict["GI_Storage_Vault_Sum_m3"] += input->RepoDict["GI_Storage_Vault_TS_Sum_m3"];
			input->RepoDict["GI_Evaporation_Surface_Sum_m3"] += input->RepoDict["GI_Evaporation_Surface_TS_Sum_m3"];
			input->RepoDict["GI_Evaporation_Pavement_Sum_m3"] += input->RepoDict["GI_Evaporation_Pavement_TS_Sum_m3"];
			input->RepoDict["GI_Evaporation_Soil_Sum_m3"] += input->RepoDict["GI_Evaporation_Soil_TS_Sum_m3"];
			input->RepoDict["GI_Evaporation_Vault_Sum_m3"] += input->RepoDict["GI_Evaporation_Vault_TS_Sum_m3"];
			input->RepoDict["GI_Drainage_Percolation_Sum_m3"] += input->RepoDict["GI_Drainage_Percolation_TS_Sum_m3"];
			input->RepoDict["GI_Drainage_PercolationPavement_Sum_m3"] += input->RepoDict["GI_Drainage_PercolationPavement_TS_Sum_m3"];
			input->RepoDict["GI_Drainage_Exfiltration_Sum_m3"] += input->RepoDict["GI_Drainage_Exfiltration_TS_Sum_m3"];
			input->RepoDict["GI_Drainage_VadoseZone_Sum_m3"] += input->RepoDict["GI_Drainage_VadoseZone_TS_Sum_m3"];
			input->RepoDict["GI_Storage_GroundwaterMound_Sum_m3"] += input->RepoDict["GI_Storage_GroundwaterMound_TS_Sum_m3"];
		}

		//If Folder_Type_ID equals Folder_List.size() - 1 then at last Folder_Type_ID and runoff vector can use push_back
		if (Folder_Type_ID == (Folder_List.size() - 1)) {
			//Summing catchment runoff volumes (m3), which will combine BulkArea and GI runoff
			//Note: This catchment summation must occur in last folder or it will overcount total runoff, adding Runoff_[LCType]_Catchment_TS_m3 multiple times
			input->RepoDict["Runoff_Pervious_Catchment_m3"] += input->RepoDict["Runoff_Pervious_Catchment_TS_m3"];
			input->RepoDict["Runoff_Impervious_Catchment_m3"] += input->RepoDict["Runoff_Impervious_Catchment_TS_m3"];
			input->RepoDict["Runoff_Water_Catchment_m3"] += input->RepoDict["Runoff_Water_Catchment_TS_m3"];
			input->RepoDict["Runoff_Subsurface_Catchment_m3"] += input->RepoDict["Runoff_Subsurface_Catchment_TS_m3"];
			input->RepoDict["Runoff_Surface_Catchment_m3"] += input->RepoDict["Runoff_Surface_Catchment_TS_m3"];
			input->RepoDict["Runoff_Catchment_m3"] += input->RepoDict["Runoff_Catchment_TS_m3"];

			//Store catchment runoff depth (m) as unrouted (e.g.,not time distributed) vector values for distribution in RoutingDiffusion.cpp
			//Note: These four input-> vector double variables were created in Inputs.h
			//PAQi_m (m) is Runoff_Pervious_m
			input->PAQi_m.push_back(input->RepoDict["Runoff_Pervious_Catchment_TS_m3"] / CA_m2);
			//DCIAQi_m (m) is Runoff_Impervious_m
			input->DCIAQi_m.push_back(input->RepoDict["Runoff_Impervious_Catchment_TS_m3"] / CA_m2);
			//WAQi_m (m) is Runoff_Water_m
			input->WAQi_m.push_back(input->RepoDict["Runoff_Water_Catchment_TS_m3"] / CA_m2);
			//SurfQi_m (m) is Runoff_Surface_m
			input->SurfQi_m.push_back(input->RepoDict["Runoff_Surface_Catchment_TS_m3"] / CA_m2);
			//SubQi_m (m) is Runoff_Subsurface_m
			input->SubQi_m.push_back(input->RepoDict["Runoff_Subsurface_Catchment_TS_m3"] / CA_m2);
			//Qi_m (m) is Runoff_m
			input->Qi_m.push_back(input->RepoDict["Runoff_Catchment_TS_m3"] / CA_m2);
		}
	}
}

//TS_First_Last function stores values for time step (TS) 1st and last for use in water balance
void AggregateOutput::TS_First_Last(Inputs* input, CompactRagged* beC, int DataDrawer_ID, int DataFolder_ID, int timeStep, int Folder_Type_ID, double Folder_Area_m2, double PC_m2, double IC_m2, double TC_m2, double SVegC_noTC_m2, double SoilC_noTC_m2, double PermeablePavementC_noTC_m2, double WaterC_noTC_m2, double IC_noTC_m2, double NoCanopy_m2, double Ratio_FolderTypeArea_to_CatchmentTypeArea, double Catchment_Pervious_Area_m2)
{
	if (timeStep == 0) {
		input->RepoVecDict["StorageSnow_TreeCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_TreeCanopy_m") * TC_m2;
		input->RepoVecDict["StorageRain_TreeCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Rain_TreeCanopy_m") * TC_m2;
		input->RepoVecDict["StorageSnow_SVegCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_SVegCanopy_m") * SVegC_noTC_m2;
		input->RepoVecDict["StorageRain_SVegCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Rain_SVegCanopy_m") * SVegC_noTC_m2;
		input->RepoVecDict["StorageSnow_noCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * NoCanopy_m2;
		input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * IC_noTC_m2;
		input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * SoilC_noTC_m2;
		input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * PermeablePavementC_noTC_m2;
		input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * WaterC_noTC_m2;
		input->RepoVecDict["StorageSnow_UnderTree_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_underTreeCanopy_m") * TC_m2;
		input->RepoVecDict["StorageSnow_UnderSVeg_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_underSVegCanopy_m") * SVegC_noTC_m2;
		input->RepoVecDict["Storage_ImperviousPondedWater_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_ImperviousPondedWater_m") * IC_m2;
		input->RepoVecDict["Storage_PerviousPondedWater_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_PerviousPondedWater_m") * PC_m2;
		input->RepoVecDict["Storage_WaterPondedWater_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_WaterPondedWater_m") * WaterC_noTC_m2;


		input->RepoVecDict["Storage_SoilEvapZone_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m") * PC_m2;
		//Storage_SoilEvapZone_TS_0th_m (m) computed using HydroPlusConfig.xml inputs to obtain exact initial condition, not end of time step condition
		input->RepoVecDict["Storage_SoilEvapZone_TS_0th_m"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_TS_0th_m") * PC_m2;
		input->RepoVecDict["Storage_VadoseZone_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_VadoseZone_m") * PC_m2;
		//Groundwater_surficial_TS_first_frac (fraction) is Groundwater_surficial_frac weighted by Ratio_FolderTypeArea_to_CatchmentTypeArea
		input->RepoVecDict["Groundwater_surficial_TS_first_frac"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Groundwater_surficial_frac") * Ratio_FolderTypeArea_to_CatchmentTypeArea;
		input->RepoVecDict["Groundwater_surficial_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Groundwater_surficial_m") * PC_m2;
		input->RepoVecDict["Runoff_Subsurface_TS_first_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Subsurface_m") * Folder_Area_m2;
		input->RepoDict["Runoff_Subsurface_TS_first_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Runoff_Subsurface_m") * Folder_Area_m2;
		input->RepoDict["GI_Storage_Soil_TS_first_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_SoilLayer_m3");
		input->RepoDict["GI_Storage_Soil_TS_0th_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_TS_0th_m3");
		input->RepoDict["GI_Storage_GroundwaterMound_TS_0th_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GroundwaterMound_m3");
	}
	//If (timeStep + 1) equals SimulationTimePeriod_timeSteps then store variables used for start and finish analysis
	//Note: Given timeStep starts at 0 in c++, 1 must be added to ever equal SimulationTimePeriod_timeSteps
	if ((timeStep + 1) == input->SimulationTimePeriod_timeSteps) {
		input->RepoVecDict["StorageSnow_TreeCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_TreeCanopy_m") * TC_m2;
		input->RepoVecDict["StorageRain_TreeCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Rain_TreeCanopy_m") * TC_m2;
		input->RepoVecDict["StorageSnow_SVegCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_SVegCanopy_m") * SVegC_noTC_m2;
		input->RepoVecDict["StorageRain_SVegCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Rain_SVegCanopy_m") * SVegC_noTC_m2;
		input->RepoVecDict["StorageSnow_noCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * NoCanopy_m2;
		input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * IC_noTC_m2;
		input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * SoilC_noTC_m2;
		input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * PermeablePavementC_noTC_m2;
		input->RepoVecDict["StorageSnow_WaterNoCanopy_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_noCanopy_m") * WaterC_noTC_m2;
		input->RepoVecDict["StorageSnow_UnderTree_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_underTreeCanopy_m") * TC_m2;
		input->RepoVecDict["StorageSnow_UnderSVeg_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_Snow_underSVegCanopy_m") * SVegC_noTC_m2;
		input->RepoVecDict["Storage_ImperviousPondedWater_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_ImperviousPondedWater_m") * IC_m2;
		input->RepoVecDict["Storage_PerviousPondedWater_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_PerviousPondedWater_m") * PC_m2;
		input->RepoVecDict["Storage_WaterPondedWater_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_WaterPondedWater_m") * WaterC_noTC_m2;

		input->RepoVecDict["Storage_SoilEvapZone_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_SoilEvapZone_m") * PC_m2;
		input->RepoVecDict["Storage_VadoseZone_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_VadoseZone_m") * PC_m2;

		//Groundwater_surficial_TS_last_frac (fraction) is Groundwater_surficial_frac weighted by Ratio_FolderTypeArea_to_CatchmentTypeArea
		input->RepoVecDict["Groundwater_surficial_TS_last_frac"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Groundwater_surficial_frac") * Ratio_FolderTypeArea_to_CatchmentTypeArea;
		input->RepoVecDict["Groundwater_surficial_TS_last_m3"][Folder_Type_ID] += beC->by_key(DataDrawer_ID, DataFolder_ID, "Groundwater_surficial_m") * PC_m2;
		input->RepoDict["GI_Storage_Surface_TS_Last_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_SurfaceLayer_m3");
		input->RepoDict["GI_PavementWater_TS_Last_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_PavementLayer_m3");
		input->RepoDict["GI_Storage_Soil_TS_last_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_SoilLayer_m3");
		input->RepoDict["GI_Storage_Vault_TS_Last_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_VaultLayer_m3");
		input->RepoDict["GI_Storage_GroundwaterMound_TS_Last_m3"] += beC->by_key(DataDrawer_ID, DataFolder_ID, "GI_Storage_GroundwaterMound_m3");
	}
}

//AggregateOutput::Initialize function called at each time step to prepare and reset variables
//Note: StorageDeficit_VadoseZone_TS_Sum_m3 is not tracked
void AggregateOutput::Initialize(Inputs* input, int timeStep)
{
	//nTypes = input->FolderType_List.size(); initialize nTypes 
	//Note: FolderType_List will either be 1 with BulkArea or 2 with BulkArea and GI
	size_t nTypes = input->FolderType_List.size();
	//initializeVector = [&](const string& variableVectorName) will initialize vector to 0.0
	auto initializeVector = [&](const string& variableVectorName) {
		//RepoVecDict[variableVectorName].assign(nTypes, 0.0); flexible for size of nTypes
		//Note: nTypes can grow beyond 2 items, BulkArea and GI
		input->RepoVecDict[variableVectorName].assign(nTypes, 0.0);
	};

	//For Loop through pollutant names to create string with variable name and set variables to zero; 
	//Note: length of For Loop is Pollutant_Name.size()
	for (int pollutant_ID = 0; pollutant_ID < input->Pollutant_Name.size(); pollutant_ID++) {
		//Create string to represent variable associated with Pollutant_Name
		Name_Load_Catchment_Pollutant_Median_kg = "Load_Catchment_" + input->Pollutant_Name[pollutant_ID] + "_Median_kg";
		Name_Load_Catchment_Pollutant_Mean_kg = "Load_Catchment_" + input->Pollutant_Name[pollutant_ID] + "_Mean_kg";
		//Reset to zero at start of each time step
		input->RepoDict[Name_Load_Catchment_Pollutant_Median_kg] = 0;
		input->RepoDict[Name_Load_Catchment_Pollutant_Mean_kg] = 0;

		//Create string to represent variable associated with Pollutant_Name
		Name_Load_GI_Inflow_Pollutant_Mean_g = "Load_GI_Inflow_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
		Name_Load_GI_Runoff_Surface_Pollutant_Mean_g = "Load_GI_Runoff_Surface_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
		Name_Load_GI_Runoff_Vault_Pollutant_Mean_g = "Load_GI_Runoff_Vault_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
		Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g = "Load_GI_Drainage_VadoseZone_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";
		Name_Release_GI_Total_Pollutant_Mean_g = "Release_GI_Total_" + input->Pollutant_Name[pollutant_ID] + "_Mean_g";

		//Reset to zero at start of each time step
		input->RepoDict[Name_Load_GI_Inflow_Pollutant_Mean_g] = 0;
		input->RepoDict[Name_Load_GI_Runoff_Surface_Pollutant_Mean_g] = 0;
		input->RepoDict[Name_Load_GI_Runoff_Vault_Pollutant_Mean_g] = 0;
		input->RepoDict[Name_Load_GI_Drainage_VadoseZone_Pollutant_Mean_g] = 0;
		input->RepoDict[Name_Release_GI_Total_Pollutant_Mean_g] = 0;
	}

	//Green Infrastructure variables storing time step sums
	input->RepoDict["GI_Precipitation_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Runon_from_BulkArea_Surface_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Runoff_Surface_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Infiltration_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Runoff_VaultDrain_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_Surface_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_PavementWater_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_Soil_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_Vault_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Evaporation_Surface_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Evaporation_Pavement_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Evaporation_Soil_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Evaporation_Vault_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Drainage_Percolation_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Drainage_PercolationPavement_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Drainage_Exfiltration_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Drainage_VadoseZone_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GroundwaterMound_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GI_Pavement_Max_Potential_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GI_Pavement_Max_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GI_Soil_Max_Potential_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GI_Soil_Max_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GI_Vault_Max_Potential_TS_Sum_m3"] = { 0 };
	input->RepoDict["GI_Storage_GI_Vault_Max_TS_Sum_m3"] = { 0 };

	//Variables will contain BulkArea and GI runoff combined, and populate input->Qi_m, DCIAQi_m, PAQi_m, WAQi_m, SubQi_m vectors
	input->RepoDict["Runoff_Pervious_Catchment_TS_m3"] = 0;
	input->RepoDict["Runoff_Impervious_Catchment_TS_m3"] = 0;
	input->RepoDict["Runoff_Water_Catchment_TS_m3"] = 0;
	input->RepoDict["Runoff_Subsurface_Catchment_TS_m3"] = 0;
	input->RepoDict["Runoff_Surface_Catchment_TS_m3"] = 0;
	input->RepoDict["Runoff_Catchment_TS_m3"] = 0;

	//Variables containing _TS_Sum_m3 will store sum for time step, initialized each time step
	//Note: Facilitate code development by keeping same variable sequence as _TS_Sum_m3 variables
	initializeVector("InterceptionRain_TreeCanopy_TS_Sum_m3");
	initializeVector("InterceptionSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("InterceptionRain_SVegCanopy_TS_Sum_m3");
	initializeVector("InterceptionSnow_SVegCanopy_TS_Sum_m3");
	initializeVector("InterceptionRainSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("InterceptionRainSnow_SVegCanopy_TS_Sum_m3");

	initializeVector("StorageSnow_noCanopy_TS_Sum_m3");
	initializeVector("StorageSnow_ImperviousNoCanopy_TS_Sum_m3");
	initializeVector("StorageSnow_SoilNoCanopy_TS_Sum_m3");
	initializeVector("StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3");
	initializeVector("StorageSnow_WaterNoCanopy_TS_Sum_m3");
	initializeVector("StorageSnow_UnderTree_TS_Sum_m3");
	initializeVector("StorageSnow_UnderSVeg_TS_Sum_m3");
	initializeVector("StorageSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("StorageSnow_SVegCanopy_TS_Sum_m3");
	initializeVector("StorageRain_TreeCanopy_TS_Sum_m3");
	initializeVector("StorageRain_SVegCanopy_TS_Sum_m3");
	initializeVector("StorageRainSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("StorageRainSnow_SVegCanopy_TS_Sum_m3");

	initializeVector("SublimationSnow_noCanopy_TS_Sum_m3");
	initializeVector("SublimationSnow_ImperviousNoCanopy_TS_Sum_m3");
	initializeVector("SublimationSnow_SoilNoCanopy_TS_Sum_m3");
	initializeVector("SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3");
	initializeVector("SublimationSnow_WaterNoCanopy_TS_Sum_m3");
	initializeVector("SublimationSnow_UnderTree_TS_Sum_m3");
	initializeVector("SublimationSnow_UnderSVeg_TS_Sum_m3");
	initializeVector("SublimationSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("SublimationSnow_SVegCanopy_TS_Sum_m3");
	initializeVector("EvaporationRain_TreeCanopy_TS_Sum_m3");
	initializeVector("EvaporationRain_SVegCanopy_TS_Sum_m3");
	initializeVector("EvaporationSublimation_TreeCanopy_TS_Sum_m3");
	initializeVector("EvaporationSublimation_SVegCanopy_TS_Sum_m3");

	initializeVector("SnowMelt_noCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_ImperviousNoCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_SoilNoCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_WaterNoCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_TreeCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_SVegCanopy_TS_Sum_m3");
	initializeVector("SnowMelt_UnderTree_TS_Sum_m3");
	initializeVector("SnowMelt_UnderSVeg_TS_Sum_m3");

	initializeVector("ThroughFallRain_TreeCanopy_TS_Sum_m3");
	initializeVector("ThroughFallRain_SVegCanopy_TS_Sum_m3");
	initializeVector("ThroughFallSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("ThroughFallSnow_SVegCanopy_TS_Sum_m3");
	initializeVector("ThroughFallRainSnow_TreeCanopy_TS_Sum_m3");
	initializeVector("ThroughFallRainSnow_SVegCanopy_TS_Sum_m3");

	initializeVector("Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3");
	initializeVector("Storage_ImperviousPondedWater_TS_Sum_m3");
	initializeVector("Storage_PerviousPondedWater_TS_Sum_m3");
	initializeVector("Storage_WaterPondedWater_TS_Sum_m3");
	initializeVector("Flux_Impervious_to_Pervious_TS_Sum_m3");
	initializeVector("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3");
	initializeVector("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3");
	initializeVector("Flux_to_WaterArea_Rain_SnowMelt_TS_Sum_m3");

	initializeVector("Evaporation_ImperviousPondedWater_TS_Sum_m3");
	initializeVector("Evaporation_PerviousPondedWater_TS_Sum_m3");
	initializeVector("Evaporation_WaterPondedWater_TS_Sum_m3");

	initializeVector("Flux_to_Infiltration_TS_Sum_m3");
	initializeVector("Drainage_macroPore_TS_Sum_m3");
	initializeVector("Infiltration_viaInfilEx_TS_Sum_m3");
	initializeVector("Infiltration_viaSatEx_TS_Sum_m3");
	initializeVector("Infiltration_TS_Sum_m3");

	initializeVector("Infiltration_viaSatEx_PreSatExRunoff_TS_Sum_m3");
	initializeVector("Infiltration_viaInfilEx_PreSatExRunoff_TS_Sum_m3");
	initializeVector("Drainage_macroPore_PreSatExRunoff_TS_Sum_m3");
	initializeVector("Drainage_SoilEvapZone_PreSatExRunoff_TS_Sum_m3");

	initializeVector("EvapoTranspiration_TS_Sum_m3");
	initializeVector("Storage_SoilEvapZone_TS_Sum_m3");
	initializeVector("Storage_SoilEvapZone_TS_Sum_m3_p_m3");
	initializeVector("Drainage_SoilEvapZone_TS_Sum_m3");
	initializeVector("Storage_VadoseZone_TS_Sum_m3");
	initializeVector("Drainage_VadoseZone_TS_Sum_m3");
	initializeVector("Groundwater_surficial_TS_Sum_m3");
	initializeVector("Flux_SatExGW_to_SoilFieldCapacity_TS_Sum_m3");
	initializeVector("Groundwater_surficial_TS_Sum_frac");
	initializeVector("GI_Ratio_RunoffInfilExcess_to_RunoffPervious_TS_Sum_frac");

	initializeVector("Runoff_InfilExcess_TS_Sum_m3");
	initializeVector("Runoff_SatExcess_TS_Sum_m3");
	initializeVector("Runoff_Pervious_TS_Sum_m3");
	initializeVector("Runoff_Impervious_TS_Sum_m3");
	initializeVector("Runoff_Water_TS_Sum_m3");
	initializeVector("Runoff_Subsurface_TS_Sum_m3");
	initializeVector("Runoff_Surface_TS_Sum_m3");
	initializeVector("Runoff_TS_Sum_m3");

	initializeVector("Runoff_Impervious_BAtoGI_TS_Sum_m3");
	initializeVector("Runoff_Water_BAtoGI_TS_Sum_m3");
	initializeVector("Runoff_SatExcess_BAtoGI_TS_Sum_m3");
	initializeVector("Runoff_InfilExcess_BAtoGI_TS_Sum_m3");

	initializeVector("Runoff_Impervious_GItoGI_TS_Sum_m3");
	initializeVector("Runoff_Pervious_GItoGI_TS_Sum_m3");
	initializeVector("Runoff_Vault_GItoGI_TS_Sum_m3");
	initializeVector("Runoff_Water_GItoGI_TS_Sum_m3");

	initializeVector("Irrigation_TreeInterceptionStorage_TS_Sum_m3");
	initializeVector("Irrigation_SVegInterceptionStorage_TS_Sum_m3");
	initializeVector("Irrigation_ImperviousDepressionStorage_TS_Sum_m3");
	initializeVector("Irrigation_PerviousDepressionStorage_TS_Sum_m3");

	//If timeStep equals 0 then initialize to zero variables that store simulation totals
	if (timeStep == 0) {

		//TS_Prior_Sum variables, not used in simulation Sum accounting
		initializeVector("StorageSnow_noCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_ImperviousNoCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_SoilNoCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_PermeablePavementNoCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_WaterNoCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_UnderTree_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_UnderSVeg_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_TreeCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageSnow_SVegCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageRain_TreeCanopy_TS_Prior_Sum_m3");
		initializeVector("StorageRain_SVegCanopy_TS_Prior_Sum_m3");
		initializeVector("Storage_ImperviousPondedWater_TS_Prior_Sum_m3");
		initializeVector("Storage_PerviousPondedWater_TS_Prior_Sum_m3");

		//Catchment variables related to StorageDeficit_VadoseZone_m
		input->RepoDict["Depth_to_Groundwater_TS_m"] = 0;
		input->RepoDict["Runoff_Subsurface_TS_first_m3"] = 0.0;

		//Variables storing Green Infrastructure for simulation sums
		input->RepoDict["GI_Precipitation_Sum_m3"] = { 0 };
		input->RepoDict["GI_Runon_from_BulkArea_Surface_Sum_m3"] = { 0 };
		input->RepoDict["GI_Runoff_Surface_Sum_m3"] = { 0 };
		input->RepoDict["GI_Infiltration_Sum_m3"] = { 0 };
		input->RepoDict["GI_Runoff_VaultDrain_Sum_m3"] = { 0 };
		input->RepoDict["GI_Storage_Surface_Sum_m3"] = { 0 };
		input->RepoDict["GI_PavementWater_Sum_m3"] = { 0 };
		input->RepoDict["GI_Storage_Soil_Sum_m3"] = { 0 };
		input->RepoDict["GI_Storage_Vault_Sum_m3"] = { 0 };
		input->RepoDict["GI_Evaporation_Surface_Sum_m3"] = { 0 };
		input->RepoDict["GI_Evaporation_Pavement_Sum_m3"] = { 0 };
		input->RepoDict["GI_Evaporation_Soil_Sum_m3"] = { 0 };
		input->RepoDict["GI_Evaporation_Vault_Sum_m3"] = { 0 };
		input->RepoDict["GI_Drainage_Percolation_Sum_m3"] = { 0 };
		input->RepoDict["GI_Drainage_PercolationPavement_Sum_m3"] = { 0 };
		input->RepoDict["GI_Drainage_Exfiltration_Sum_m3"] = { 0 };
		input->RepoDict["GI_Drainage_VadoseZone_Sum_m3"] = { 0 };
		input->RepoDict["GI_Storage_GroundwaterMound_Sum_m3"] = { 0 };

		input->RepoDict["StorageDeficit_VadoseZone_TS_first_m3"] = { 0 };
		input->RepoDict["GI_Storage_Soil_TS_first_m3"] = { 0 };
		input->RepoDict["GI_Storage_Soil_TS_0th_m3"] = { 0 };
		input->RepoDict["GI_Storage_GroundwaterMound_TS_0th_m3"] = { 0 };

		input->RepoDict["StorageDeficit_VadoseZone_TS_last_m3"] = { 0 };
		input->RepoDict["GI_Storage_Surface_TS_Last_m3"] = { 0 };
		input->RepoDict["GI_PavementWater_TS_Last_m3"] = { 0 };
		input->RepoDict["GI_Storage_Soil_TS_last_m3"] = { 0 };
		input->RepoDict["GI_Storage_Vault_TS_Last_m3"] = { 0 };
		input->RepoDict["GI_Storage_GroundwaterMound_TS_Last_m3"] = { 0 };

		input->RepoDict["StorageDeficit_VadoseZone_Sum_m3"] = { 0 };
		//Variables will contain BulkArea and GI runoff combined, and populate input->Qi_m, DCIAQi_m, PAQi_m, WAQi_m, SubQi_m vectors
		input->RepoDict["Runoff_Pervious_Catchment_m3"] = 0;
		input->RepoDict["Runoff_Impervious_Catchment_m3"] = 0;
		input->RepoDict["Runoff_Water_Catchment_m3"] = 0;
		input->RepoDict["Runoff_Subsurface_Catchment_m3"] = 0;
		input->RepoDict["Runoff_Surface_Catchment_m3"] = 0;
		input->RepoDict["Runoff_Catchment_m3"] = 0;

		//Variables storing 1st or last time step values
		initializeVector("StorageSnow_TreeCanopy_TS_first_m3");
		initializeVector("StorageRain_TreeCanopy_TS_first_m3");
		initializeVector("StorageSnow_SVegCanopy_TS_first_m3");
		initializeVector("StorageRain_SVegCanopy_TS_first_m3");
		initializeVector("StorageSnow_ImperviousNoCanopy_TS_first_m3");
		initializeVector("StorageSnow_SoilNoCanopy_TS_first_m3");
		initializeVector("StorageSnow_PermeablePavementNoCanopy_TS_first_m3");
		initializeVector("StorageSnow_WaterNoCanopy_TS_first_m3");
		initializeVector("StorageSnow_noCanopy_TS_first_m3");
		initializeVector("StorageSnow_UnderTree_TS_first_m3");
		initializeVector("StorageSnow_UnderSVeg_TS_first_m3");
		initializeVector("Storage_ImperviousPondedWater_TS_first_m3");
		initializeVector("Storage_PerviousPondedWater_TS_first_m3");
		initializeVector("Storage_WaterPondedWater_TS_first_m3");
		initializeVector("Storage_SoilEvapZone_TS_first_m3");
		initializeVector("Storage_SoilEvapZone_TS_0th_m");
		initializeVector("Storage_VadoseZone_TS_first_m3");
		initializeVector("Groundwater_surficial_TS_first_frac");
		initializeVector("Groundwater_surficial_TS_first_m3");
		initializeVector("Runoff_Subsurface_TS_first_m3");

		initializeVector("StorageSnow_TreeCanopy_TS_last_m3");
		initializeVector("StorageRain_TreeCanopy_TS_last_m3");
		initializeVector("StorageSnow_SVegCanopy_TS_last_m3");
		initializeVector("StorageRain_SVegCanopy_TS_last_m3");
		initializeVector("StorageSnow_ImperviousNoCanopy_TS_last_m3");
		initializeVector("StorageSnow_SoilNoCanopy_TS_last_m3");
		initializeVector("StorageSnow_PermeablePavementNoCanopy_TS_last_m3");
		initializeVector("StorageSnow_WaterNoCanopy_TS_last_m3");
		initializeVector("StorageSnow_noCanopy_TS_last_m3");
		initializeVector("StorageSnow_UnderTree_TS_last_m3");
		initializeVector("StorageSnow_UnderSVeg_TS_last_m3");
		initializeVector("Storage_ImperviousPondedWater_TS_last_m3");
		initializeVector("Storage_PerviousPondedWater_TS_last_m3");
		initializeVector("Storage_WaterPondedWater_TS_last_m3");
		initializeVector("Storage_SoilEvapZone_TS_last_m3");
		initializeVector("Storage_VadoseZone_TS_last_m3");
		initializeVector("Groundwater_surficial_TS_last_frac");
		initializeVector("Groundwater_surficial_TS_last_m3");

		//Variables containing _Sum_m3 will store sum for simulation, initialized first time step 
		//Note: Facilitate code development by keeping same variable sequence as _Sum_m3 variables
		initializeVector("InterceptionRain_TreeCanopy_Sum_m3");
		initializeVector("InterceptionSnow_TreeCanopy_Sum_m3");
		initializeVector("InterceptionRain_SVegCanopy_Sum_m3");
		initializeVector("InterceptionSnow_SVegCanopy_Sum_m3");
		initializeVector("InterceptionRainSnow_TreeCanopy_Sum_m3");
		initializeVector("InterceptionRainSnow_SVegCanopy_Sum_m3");

		initializeVector("SublimationSnow_noCanopy_Sum_m3");
		initializeVector("SublimationSnow_ImperviousNoCanopy_Sum_m3");
		initializeVector("SublimationSnow_SoilNoCanopy_Sum_m3");
		initializeVector("SublimationSnow_PermeablePavementNoCanopy_Sum_m3");
		initializeVector("SublimationSnow_WaterNoCanopy_Sum_m3");
		initializeVector("SublimationSnow_UnderTree_Sum_m3");
		initializeVector("SublimationSnow_UnderSVeg_Sum_m3");
		initializeVector("SublimationSnow_TreeCanopy_Sum_m3");
		initializeVector("SublimationSnow_SVegCanopy_Sum_m3");
		initializeVector("EvaporationRain_TreeCanopy_Sum_m3");
		initializeVector("EvaporationRain_SVegCanopy_Sum_m3");
		initializeVector("EvaporationSublimation_TreeCanopy_Sum_m3");
		initializeVector("EvaporationSublimation_SVegCanopy_Sum_m3");

		initializeVector("StorageSnow_noCanopy_Sum_m3");
		initializeVector("StorageSnow_ImperviousNoCanopy_Sum_m3");
		initializeVector("StorageSnow_SoilNoCanopy_Sum_m3");
		initializeVector("StorageSnow_PermeablePavementNoCanopy_Sum_m3");
		initializeVector("StorageSnow_WaterNoCanopy_Sum_m3");
		initializeVector("StorageSnow_UnderTree_Sum_m3");
		initializeVector("StorageSnow_UnderSVeg_Sum_m3");
		initializeVector("StorageSnow_TreeCanopy_Sum_m3");
		initializeVector("StorageSnow_SVegCanopy_Sum_m3");
		initializeVector("StorageRain_TreeCanopy_Sum_m3");
		initializeVector("StorageRain_SVegCanopy_Sum_m3");
		initializeVector("StorageRainSnow_TreeCanopy_Sum_m3");
		initializeVector("StorageRainSnow_SVegCanopy_Sum_m3");

		initializeVector("SnowMelt_noCanopy_Sum_m3");
		initializeVector("SnowMelt_ImperviousNoCanopy_Sum_m3");
		initializeVector("SnowMelt_SoilNoCanopy_Sum_m3");
		initializeVector("SnowMelt_PermeablePavementNoCanopy_Sum_m3");
		initializeVector("SnowMelt_WaterNoCanopy_Sum_m3");
		initializeVector("SnowMelt_UnderTree_Sum_m3");
		initializeVector("SnowMelt_UnderSVeg_Sum_m3");
		initializeVector("SnowMelt_TreeCanopy_Sum_m3");
		initializeVector("SnowMelt_SVegCanopy_Sum_m3");

		initializeVector("ThroughFallRain_TreeCanopy_Sum_m3");
		initializeVector("ThroughFallRain_SVegCanopy_Sum_m3");
		initializeVector("ThroughFallSnow_TreeCanopy_Sum_m3");
		initializeVector("ThroughFallSnow_SVegCanopy_Sum_m3");
		initializeVector("ThroughFallRainSnow_TreeCanopy_Sum_m3");
		initializeVector("ThroughFallRainSnow_SVegCanopy_Sum_m3");

		initializeVector("Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_Sum_m3");
		initializeVector("Storage_ImperviousPondedWater_Sum_m3");
		initializeVector("Storage_PerviousPondedWater_Sum_m3");
		initializeVector("Storage_WaterPondedWater_Sum_m3");
		initializeVector("Flux_Impervious_to_Pervious_Sum_m3");
		initializeVector("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Sum_m3");
		initializeVector("Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_Irrigation_ImpRunon_TS_Sum_m3");
		initializeVector("Flux_to_WaterArea_Rain_SnowMelt_Sum_m3");

		initializeVector("Evaporation_ImperviousPondedWater_Sum_m3");
		initializeVector("Evaporation_PerviousPondedWater_Sum_m3");
		initializeVector("Evaporation_WaterPondedWater_Sum_m3");

		initializeVector("Flux_to_Infiltration_Sum_m3");
		initializeVector("Drainage_macroPore_Sum_m3");
		initializeVector("Infiltration_viaInfilEx_Sum_m3");
		initializeVector("Infiltration_viaSatEx_Sum_m3");
		initializeVector("Infiltration_Sum_m3");

		initializeVector("Infiltration_viaSatEx_PreSatExRunoff_Sum_m3");
		initializeVector("Infiltration_viaInfilEx_PreSatExRunoff_Sum_m3");
		initializeVector("Drainage_macroPore_PreSatExRunoff_Sum_m3");
		initializeVector("Drainage_SoilEvapZone_PreSatExRunoff_Sum_m3");

		initializeVector("EvapoTranspiration_Sum_m3");
		initializeVector("Storage_SoilEvapZone_Sum_m3");
		initializeVector("Drainage_SoilEvapZone_Sum_m3");
		initializeVector("Storage_VadoseZone_Sum_m3");
		initializeVector("Drainage_VadoseZone_Sum_m3");

		initializeVector("Groundwater_surficial_Sum_m3");
		initializeVector("Flux_SatExGW_to_SoilFieldCapacity_Sum_m3");

		initializeVector("Runoff_InfilExcess_Sum_m3");
		initializeVector("Runoff_SatExcess_Sum_m3");
		initializeVector("Runoff_Pervious_Sum_m3");
		initializeVector("Runoff_Impervious_Sum_m3");
		initializeVector("Runoff_Water_Sum_m3");
		initializeVector("Runoff_Subsurface_Sum_m3");
		initializeVector("Runoff_Surface_Sum_m3");
		initializeVector("Runoff_Sum_m3");

		initializeVector("Runoff_Impervious_BAtoGI_Sum_m3");
		initializeVector("Runoff_Water_BAtoGI_Sum_m3");
		initializeVector("Runoff_SatExcess_BAtoGI_Sum_m3");
		initializeVector("Runoff_InfilExcess_BAtoGI_Sum_m3");

		initializeVector("Runoff_Impervious_GItoGI_Sum_m3");
		initializeVector("Runoff_Pervious_GItoGI_Sum_m3");
		initializeVector("Runoff_Vault_GItoGI_Sum_m3");
		initializeVector("Runoff_Water_GItoGI_Sum_m3");

		initializeVector("Irrigation_TreeInterceptionStorage_Sum_m3");
		initializeVector("Irrigation_SVegInterceptionStorage_Sum_m3");
		initializeVector("Irrigation_ImperviousDepressionStorage_Sum_m3");
		initializeVector("Irrigation_PerviousDepressionStorage_Sum_m3");
	}
}
