#include "Groundwater_StormwaterDevice.h"

//Note: Catchment average soil moisture deficit (StorageDeficit_VadoseZone_m, AveSMD) determines water storage in each Topographic Index bin
//Note: Saturation excess runoff is an outcome of AveSMD and local SMD values, based on variable source area and topographic index (TI) theory
//Note: The catchment Bulk Area uses local SMD values to determine saturation excess in each TI bin, StorageDeficit_VadoseZone_TI_m
//Note: The Green Infrastructure (GI) Area uses AveSMD via Depth_to_Groundwater_Table_m = StorageDeficit_VadoseZone_m / Porosity_Drainable_m3pm3
//Note: Green Infrastructure (GI) uses Depth_to_Groundwater_Table_m in Groundwater_StormwaterDevice.cpp to determine saturation by groundwater

//Saturation_by_WaterTable function computes upwelling of groundwater in GI for GI types where Flag_Exfiltration_to_Catchment == 1
void Groundwater_StormwaterDevice::Saturation_by_WaterTable(Inputs* input, CompactRagged* beC, int DataDrawer_ID, int DataFolder_ID, int timeStep)
{
	//Flag_SimulateGroundWaterMounding parameter enables temporary mounding of ground water
	bool Flag_SimulateGroundWaterMounding = 0;
	double Area_GI_Unit_Functional_m2 = 0.0;
	double Depth_to_PavementLayerBase_m = 0.0;
	double Depth_to_SoilLayerBase_m = 0.0;
	double Depth_to_VaultLayerBase_m = 0.0;
	double Thickness_Groundwater_PavementLayer_m = 0.0;
	double Thickness_Groundwater_SoilLayer_m = 0.0;
	double Thickness_Groundwater_VaultLayer_m = 0.0;
	double Storage_Groundwater_PavementLayer_m3 = 0.0;
	double Storage_Groundwater_SoilLayer_m3 = 0.0;
	double Storage_Groundwater_VaultLayer_m3 = 0.0;
	double Thickness_GroundwaterMound_m = 0.0;  
	double Depth_to_Groundwater_Table_m = input->InputXml["Depth_to_Groundwater_Table_m"];
	bool Flag_InitialGroundwaterMound = 0;
	double Depth_InitialGroundwaterMound_m = 0.0;

	//If timeStep is zero, then read in HydroPlusConfig.xml elements Flag_InitialGroundwaterMound and Depth_InitialGroundwaterMound_m
	if (timeStep == 0) {
		Flag_InitialGroundwaterMound = beC->by_key(DataDrawer_ID, DataFolder_ID, "Flag_InitialGroundwaterMound");
		Depth_InitialGroundwaterMound_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Depth_InitialGroundwaterMound_m");
	}

	//If Type is RainBarrel or RoofDisconnect or the vault has thickness > Constant_1E_negative10 (minimum used to avoid division by zero) set the functional area to GI footprint
	//Note: Vault is presumed to have area of Area_GI_Unit_m2, with water entering from (and evaporating) above through Area_GI_Unit_Pervious_m2
	if (beC->by_key_str(DataDrawer_ID, DataFolder_ID, "Type") == "RainBarrel" || beC->by_key_str(DataDrawer_ID, DataFolder_ID, "Type") == "RoofDisconnect" || beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Thickness_m") > Constant_1E_negative10) {
		//Area_GI_Unit_Functional_m2 (m2) is redefined to Area_GI_Unit_m2 (m2)
		Area_GI_Unit_Functional_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "Area_GI_Unit_m2");
	}
	//Else Type is not RainBarrel or RoofDisconnect or vault has thickness <= Constant_1E_negative10 then
	else {
		//Area_GI_Unit_Functional_m2 (m2) is defaulted to Area_GI_Unit_Pervious_m2 (m2)
		Area_GI_Unit_Functional_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "Area_GI_Unit_Pervious_m2");
	}

	//If Flag_InitialGroundwaterMound > 0, timeStep == 0 and Porosity_Drainable_m3pm3 and Area_GI_Unit_Functional_m2 greater than 0 then check to initialize groundwater mounding effect below GI device
	//Note: First timeStep can set an initial groundwater mounding height, which in subsequent timeSteps is updated by water balance
	if (Flag_InitialGroundwaterMound > 0 && timeStep == 0 && input->InputXml["Porosity_Drainable_m3pm3"] > 0 && Area_GI_Unit_Functional_m2 > 0) {
		//Depth_to_Groundwater_Table_m (m) is initialized to to Depth_InitialGroundwaterMound_m (m) based on HydroPlusConfig.xml Flag_InitialGroundwaterMound = 1
		Depth_to_Groundwater_Table_m = Depth_InitialGroundwaterMound_m;
		//Thickness_GroundwaterMound_m (m) is initialized to input->InputXml["Depth_to_Groundwater_Table_m"] - Depth_InitialGroundwaterMound_m 
		//Note: Equation derived from logic, presuming Depth_to_Groundwater_Table_m (m) is greater than Depth_InitialGroundwaterMound_m (m)
		Thickness_GroundwaterMound_m = input->InputXml["Depth_to_Groundwater_Table_m"] - Depth_InitialGroundwaterMound_m;
		//If Thickness_GroundwaterMound_m < 0 then notify user the HydroPlusConfig.xml Depth_InitialGroundwaterMound_m was overridden, and set Thickness_GroundwaterMound_m to 0
		if (Thickness_GroundwaterMound_m < 0) {
			cout << "Note: HydroPlusConfig.xml element Depth_InitialGroundwaterMound_m > Depth_to_Groundwater_Table_m, calculated using topographic index theory." << endl;
			cout << "Note: To maintain a physically realistic scenario this simulation used Depth_to_Groundwater_Table_m = " << input->InputXml["Depth_to_Groundwater_Table_m"] << "." << endl;
			//Depth_InitialGroundwaterMound_m (m) redefined to Depth_to_Groundwater_Table_m, whicg enforces a maximum depth, the closest possible to the user input value
			Depth_InitialGroundwaterMound_m = input->InputXml["Depth_to_Groundwater_Table_m"];
			//Thickness_GroundwaterMound_m (m) defined as zero, based on input->InputXml["Depth_to_Groundwater_Table_m"] - Depth_InitialGroundwaterMound_m
			Thickness_GroundwaterMound_m = 0;
		}
		//Storage_GroundwaterMound_m3 (m3) defined as product of Thickness_GroundwaterMound_m (m), Area_GI_Unit_Functional_m2 (m2), and Porosity_Drainable_m3pm3 (m3/m3)
		//Note: Porosity_Drainable_m3pm3 (m3/m3) is also used in StorageDeficit_VadoseZone::calculate function to determine Depth_to_Groundwater_Table_m (m)
		//Note: Use of Porosity_Drainable_m3pm3 presumes mounding is controlled by catchment native soils, rather than GI soils
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3") = Thickness_GroundwaterMound_m * Area_GI_Unit_Functional_m2 * input->InputXml["Porosity_Drainable_m3pm3"];
	}
	//Else If (input->InputXml["Porosity_Drainable_m3pm3"] > 0 && Area_GI_Unit_Functional_m2 > 0) {
	//Note: If Porosity_Drainable_m3pm3 and Area_GI_Unit_Functional_m2 greater than 0 then simulate groundwater mounding effect below GI device
	else if (input->InputXml["Porosity_Drainable_m3pm3"] > 0 && Area_GI_Unit_Functional_m2 > 0) {
		//Thickness_GroundwaterMound_m (m) is quotient of Storage_GroundwaterMound_m3 (m3) and Area_GI_Unit_Functional_m2 (m2), divided by Porosity_Drainable_m3pm3
		//Note: Area_GI_Unit_Pervious_m2 (m2) and hence Area_GI_Unit_Functional_m2 (m2) is always greater than zero due to conditional in Inflow_StormwaterDevice::ResetVariablesToZero function
		//Note: Porosity_Drainable_m3pm3 (m3/m3) is also used in StorageDeficit_VadoseZone::calculate function to determine Depth_to_Groundwater_Table_m (m)
		//Note: Use of Porosity_Drainable_m3pm3 presumes mounding is controlled by catchment native soils, rather than GI soils
		Thickness_GroundwaterMound_m = (beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3") / Area_GI_Unit_Functional_m2) / input->InputXml["Porosity_Drainable_m3pm3"];
		//Depth_to_Groundwater_Table_m (m) is reduced by Thickness_GroundwaterMound_m (m)
		Depth_to_Groundwater_Table_m = Depth_to_Groundwater_Table_m - Thickness_GroundwaterMound_m;
	}

	//Storage_GI_Pavement_Max_m3 (m3) initialized to Storage_GI_Pavement_Max_Potential_m3
	beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Pavement_Max_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Pavement_Max_Potential_m3");
	//Storage_GI_Soil_Max_m3 (m3) is initialized to Storage_GI_Soil_Max_Potential_m3
	beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_Max_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_Max_Potential_m3");
	//Storage_GI_Vault_Max_m3 (m3) is initialized to Storage_GI_Vault_Max_Potential_m3
	beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Vault_Max_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Vault_Max_Potential_m3");

	//Depth_to_PavementLayerBase_m (m) equals Pavement_Thickness_m (m) given layer sequence
	Depth_to_PavementLayerBase_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Pavement_Thickness_m");
	//Depth_to_SoilLayerBase_m (m) equals sum of Pavement_Thickness_m (m) and Soil_Thickness_m (m) given layer sequence
	Depth_to_SoilLayerBase_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Pavement_Thickness_m") + beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_Thickness_m");
	//Depth_to_VaultLayerBase_m (m) equals sum of Pavement_Thickness_m (m) Soil_Thickness_m (m), Vault_Thickness_m (m) given layer sequence
	Depth_to_VaultLayerBase_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Pavement_Thickness_m") + beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_Thickness_m") + beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Thickness_m");

	//If Depth_to_PavementLayerBase_m (m) > Depth_to_Groundwater_Table_m (m) then groundwater has entered the GI layer
	if (Depth_to_PavementLayerBase_m > Depth_to_Groundwater_Table_m) {
		//Thickness_Groundwater_PavementLayer_m (m) = Depth_to_PavementLayerBase_m (m) - Depth_to_Groundwater_Table_m (m)
		Thickness_Groundwater_PavementLayer_m = Depth_to_PavementLayerBase_m - Depth_to_Groundwater_Table_m;
		//If Thickness_Groundwater_PavementLayer_m > Pavement_Thickness_m then 
		if (Thickness_Groundwater_PavementLayer_m > beC->by_key(DataDrawer_ID, DataFolder_ID, "Pavement_Thickness_m")) {
			//Thickness_Groundwater_PavementLayer_m equals Pavement_Thickness_m then 
			Thickness_Groundwater_PavementLayer_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Pavement_Thickness_m");
		}
		//Storage_Groundwater_PavementLayer_m3 (m3) represents upwelling groundwater, not saturation from direct infiltration
		//Note: Thickness_Groundwater_PavementLayer_m multiplied by Pavement_Porosity_m3pm3 to obtain thickness of liquid water without substrate
		Storage_Groundwater_PavementLayer_m3 = Thickness_Groundwater_PavementLayer_m * beC->by_key(DataDrawer_ID, DataFolder_ID, "Pavement_Porosity_m3pm3") * Area_GI_Unit_Functional_m2;
		//Storage_GI_Pavement_Max_m3 (m3) for drainage is reduced by upwelling groundwater, Storage_Groundwater_PavementLayer_m3 (m3) 
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Pavement_Max_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Pavement_Max_m3") - Storage_Groundwater_PavementLayer_m3;
		//Storage_GI_Pavement_Max_m3 (m3) is maximum of Storage_GI_Pavement_Max_m3 (m3) and zero, preventing negative values
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Pavement_Max_m3") = MAX(beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Pavement_Max_m3"), 0);
	}
	//If Depth_to_SoilLayerBase_m (m) > Depth_to_Groundwater_Table_m (m) then then groundwater has entered the GI layer
	if (Depth_to_SoilLayerBase_m > Depth_to_Groundwater_Table_m) {
		//Thickness_Groundwater_SoilLayer_m (m) = Depth_to_SoilLayerBase_m (m) - Depth_to_Groundwater_Table_m (m)
		Thickness_Groundwater_SoilLayer_m = Depth_to_SoilLayerBase_m - Depth_to_Groundwater_Table_m;
		//If Thickness_Groundwater_SoilLayer_m > Soil_Thickness_m then 
		if (Thickness_Groundwater_SoilLayer_m > beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_Thickness_m")) {
			//Thickness_Groundwater_SoilLayer_m equals Soil_Thickness_m then 
			Thickness_Groundwater_SoilLayer_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Soil_Thickness_m");
		}
		//Storage_Groundwater_SoilLayer_m3 (m3) represents upwelling groundwater, not saturation from direct infiltration
		//Note: Thickness_Groundwater_SoilLayer_m multiplied by Porosity_ThetaSat_ThetaWp_m3pm3 to obtain thickness of liquid water without substrate
		Storage_Groundwater_SoilLayer_m3 = Thickness_Groundwater_SoilLayer_m * beC->by_key(DataDrawer_ID, DataFolder_ID, "Porosity_ThetaSat_ThetaWp_m3pm3") * Area_GI_Unit_Functional_m2;
		//Storage_GI_Soil_Max_m3 (m3) for drainage is reduced by upwelling groundwater, Storage_Groundwater_SoilLayer_m3 (m3) 
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_Max_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_Max_m3") - Storage_Groundwater_SoilLayer_m3;
		//Storage_GI_Soil_Max_m3 (m3) is maximum of Storage_GI_Soil_Max_m3 (m3) and zero, preventing negative values
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_Max_m3") = MAX(beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Soil_Max_m3"), 0);
	}
	//If Depth_to_VaultLayerBase_m (m) > Depth_to_Groundwater_Table_m (m) then then groundwater has entered the GI layer
	if (Depth_to_VaultLayerBase_m > Depth_to_Groundwater_Table_m) {
		//Thickness_Groundwater_VaultLayer_m (m) = Depth_to_VaultLayerBase_m (m) - Depth_to_Groundwater_Table_m (m)
		Thickness_Groundwater_VaultLayer_m = Depth_to_VaultLayerBase_m - Depth_to_Groundwater_Table_m;
		//If Thickness_Groundwater_VaultLayer_m > Vault_Thickness_m then 
		if (Thickness_Groundwater_VaultLayer_m > beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Thickness_m")) {
			//Thickness_Groundwater_VaultLayer_m equals Vault_Thickness_m then 
			Thickness_Groundwater_VaultLayer_m = beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Thickness_m");
		}
		//Storage_Groundwater_VaultLayer_m3 (m3) represents upwelling groundwater, not saturation from direct infiltration
		//Note: Thickness_Groundwater_VaultLayer_m multiplied by Vault_Porosity_m3pm3 to obtain thickness of liquid water without substrate
		Storage_Groundwater_VaultLayer_m3 = Thickness_Groundwater_VaultLayer_m * beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Porosity_m3pm3") * Area_GI_Unit_Functional_m2;
		//Storage_GI_Vault_Max_m3 (m3) for drainage is reduced by upwelling groundwater, Storage_Groundwater_VaultLayer_m3 (m3) 
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Vault_Max_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Vault_Max_m3") - Storage_Groundwater_VaultLayer_m3;
		//Storage_GI_Vault_Max_m3 (m3) is maximum of Storage_GI_Vault_Max_m3 (m3) and zero, preventing negative values
		beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Vault_Max_m3") = MAX(beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GI_Vault_Max_m3"), 0);
	}
}

//Groundwater_Mounding function modifies Drainage_VadoseZone_m3 with changes in saturated mounding beneath the GI device 
//Note: Rationale for simulation of groundwater mounding is based on Endreny and Collins (2009), showing how infiltration leads to mounding, whic can saturate GI devices
//Endreny, T., & Collins, V. (2009). Implications of bioretention basin spatial arrangements on stormwater recharge and groundwater mounding. Ecological Engineering, 35(5), 670-677. doi:10.1016/j.ecoleng.2008.10.017
void Groundwater_StormwaterDevice::Groundwater_Mounding(Inputs* input, CompactRagged* beC, int DataDrawer_ID, int DataFolder_ID, int timeStep)
{
	int Sides_GroundwaterMound = 4;
	double Area_GI_Unit_Functional_m2 = 0;
	double Thickness_GroundwaterMound_m = 0;
	double Length_GI_Unit_m = 0;
	double Area_GroundwaterMound_m2 = 0;
	double Length_GI_Unit_Centroid_to_Edge_m = 0;
	double Seepage_GroundwaterMound_m3 = 0;
	double Drainage_VadoseZone_m3 = beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_VadoseZone_m3");
	//HydraulicConductivity_Catchment_mpdt (m/dt) is product of Hydraulic Conductivity K0_mph (m/h) and Ratio_Hour_to_Second and SimulationTimeStep_Duration_sec[timeStep]
	double HydraulicConductivity_Catchment_mpdt = input->InputXml["Soil_Ksat_mph"] * Ratio_Hour_to_Second * input->SimulationTimeStep_Duration_sec[timeStep];


	//If Type is RainBarrel or RoofDisconnect or the vault has thickness > Constant_1E_negative10 (minimum used to avoid division by zero) set the functional area to GI footprint
	//Note: Vault is presumed to have area of Area_GI_Unit_m2, with water entering from (and evaporating) above through Area_GI_Unit_Pervious_m2
	if (beC->by_key_str(DataDrawer_ID, DataFolder_ID, "Type") == "RainBarrel" || beC->by_key_str(DataDrawer_ID, DataFolder_ID, "Type") == "RoofDisconnect" || beC->by_key(DataDrawer_ID, DataFolder_ID, "Vault_Thickness_m") > Constant_1E_negative10) {
		//Area_GI_Unit_Functional_m2 (m2) is redefined to Area_GI_Unit_m2 (m2)
		Area_GI_Unit_Functional_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "Area_GI_Unit_m2");
	}
	//Else Type is not RainBarrel or RoofDisconnect or vault has thickness <= Constant_1E_negative10 then
	else {
		//Area_GI_Unit_Functional_m2 (m2) is defaulted to Area_GI_Unit_Pervious_m2 (m2)
		Area_GI_Unit_Functional_m2 = beC->by_key(DataDrawer_ID, DataFolder_ID, "Area_GI_Unit_Pervious_m2");
	}

	//Thickness_GroundwaterMound_m (m) is quotient of Storage_GroundwaterMound_m3 (m3) and Area_GI_Unit_Pervious_m2 (m2), divided by Porosity_Drainable_m3pm3
	//Note: Area_GI_Unit_Pervious_m2 (m2) and hence Area_GI_Unit_Functional_m2 (m2) is always greater than zero due to conditional in Inflow_StormwaterDevice::ResetVariablesToZero function
	//Note: Porosity_Drainable_m3pm3 (m3/m3) is also used in StorageDeficit_VadoseZone::calculate function to determine Depth_to_Groundwater_Table_m (m)
	//Note: Use of Porosity_Drainable_m3pm3 presumes mounding is controlled by catchment native soils, rather than GI soils
	Thickness_GroundwaterMound_m = (beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3") / Area_GI_Unit_Functional_m2) / input->InputXml["Porosity_Drainable_m3pm3"];
	//Length_GI_Unit_m (m) is square root of Area_GI_Unit_m2 (m2)
	Length_GI_Unit_m = pow(beC->by_key(DataDrawer_ID, DataFolder_ID, "Area_GI_Unit_m2"), 0.5);
	//Length_GI_Unit_Centroid_to_Edge_m (m) is half of the Length_GI_Unit_m
	Length_GI_Unit_Centroid_to_Edge_m = Length_GI_Unit_m / 2;
	//Area_GroundwaterMound_m2 (m2) is (Length_GI_Unit_m * Thickness_GroundwaterMound_m) * Sides_GroundwaterMound, which is face area on all four sides 
	Area_GroundwaterMound_m2 = (Length_GI_Unit_m * Thickness_GroundwaterMound_m) * Sides_GroundwaterMound;
	//Seepage_GroundwaterMound_m3 (m3) is Darcy Equation, Area_GroundwaterMound_m2 * HydraulicConductivity_Catchment_mpdt * (Thickness_GroundwaterMound_m / Length_GI_Unit_Centroid_to_Edge_m)
	//Note: Eq 14.3 from Chin (2021) Water-Resources Engineering 4th Edition, Pearson ISBN 978-0-13-535770-5
	//Note: Thickness_GroundwaterMound_m (m) is equivalent to piezometric head, sum of pressure head and elevation head
	Seepage_GroundwaterMound_m3 = HydraulicConductivity_Catchment_mpdt * Area_GroundwaterMound_m2 * (Thickness_GroundwaterMound_m / Length_GI_Unit_Centroid_to_Edge_m);
	//Seepage_GroundwaterMound_m3 (m3) is maximum of Seepage_GroundwaterMound_m3 (m3) and Storage_GroundwaterMound_m3 (m3), available water
	Seepage_GroundwaterMound_m3 = MIN(Seepage_GroundwaterMound_m3, beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3"));

	//Storage_GroundwaterMound_m3 (m3) is determined as mass balance, adding inflow Drainage_VadoseZone_m3 (m3) removing outflow Seepage_GroundwaterMound_m3 (m3)
	beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3") = beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3") + Drainage_VadoseZone_m3 - Seepage_GroundwaterMound_m3;
	//Storage_GroundwaterMound_m3 (m3) is maximum of Storage_GroundwaterMound_m3 (m3) and zero, prohibiting negative values
	beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3") = MAX(beC->by_key(DataDrawer_ID, DataFolder_ID, "Storage_GroundwaterMound_m3"), 0);

	//Drainage_VadoseZone_m3 (m3) redefined as Seepage_GroundwaterMound_m3 (m3), becomeing integrated into catchment groundwater
	beC->by_key(DataDrawer_ID, DataFolder_ID, "Drainage_VadoseZone_m3") = Seepage_GroundwaterMound_m3;
}