#include "WaterBalance_TimeSeries.h"

//ofstream vectors created outside of function in order to only be created once, otherwise they will recreate and destroy the prior record
vector<ofstream> outfile_CanopyVariables_m, outfile_CanopyVariables_m3, outfile_PerviousCoverVariables_m, outfile_PerviousCoverVariables_m3,
outfile_ImperviousCoverVariables_m, outfile_ImperviousCoverVariables_m3, outfile_TreeCoverVariables_m, outfile_TreeCoverVariables_m3,
outfile_RunoffVariables_m, outfile_RunoffVariables_m3;

//Note: Water balance terms stored as input->RepoVecDict[] or input->RepoDict[] were created in AggregateOutput.cpp
//Note: Adjusting depth from folder area to catchment area was performed in AggregateOutput.cpp
//Note: Adjusting depth from cover type area (e.g., pervious or impervious) to catchment area will be performed in output writers
//Note: Snow depths are liquid depths, also known as snow water equivalent
//writeWeatherDriversAndPhenology writes time series values for tree and short vegetation leaf area index
void WaterBalance_TimeSeries::writeWeatherDriversAndPhenology(Inputs* input, int timeStep)
{
	string path;
	path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_WeatherDrivers_LeafArea.csv";
	//If timeStep equals zero
	if (timeStep == 0) {
		ofstream outfile_WeatherLAI(path);
		if (!outfile_WeatherLAI.good()) {
			cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
			cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
			cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
			//Call abort function, which ends the HydroPlus.exe simulation
			abort();
		}
		
		//If Radiation_Shortwave_Direct_Wpm2 vector size > 0 then data available for output
		//Note: Typically, Radiation_Shortwave_Direct_Wpm2 and related vectors are only availale for SpatialTemperatureModel
		if (input->Radiation_Shortwave_Direct_Wpm2.size() > 0) {
			outfile_WeatherLAI << "YYYYMMDD,Hr:Min:Sec,Temperature_Air_C,Temperature_DewPoint_C,Radiation_Net_W_p_m2,Wind_Speed_m_p_s,Pressure_Atmosphere_kPa,Precipitation_m,Rain_m,Snow_LiquidEquivalent_m,Radiation_Shortwave_Direct_W_p_m2,Radiation_Shortwave_Diffuse_W_p_m2,Radiation_Longwave_Downwelling_W_p_m2,Radiation_Longwave_Upwelling_W_p_m2,PotentialEvapotranspiration_TreeCover_m,PotentialEvaporation_WaterOnGround_m,PotentialEvaporation_WaterOnTree_m,PotentialSublimation_SnowOnGround_m,PotentialSublimation_SnowOnTree_m,LAI_BAI_Tree_m2_p_m2,LAI_BAI_SVeg_m2_p_m2" << endl;
		}
		else {
			outfile_WeatherLAI << "YYYYMMDD,Hr:Min:Sec,Temperature_Air_C,Temperature_DewPoint_C,Radiation_Net_W_p_m2,Wind_Speed_m_p_s,Pressure_Atmosphere_kPa,Precipitation_m,Rain_m,Snow_LiquidEquivalent_m,PotentialEvapotranspiration_TreeCover_m,PotentialEvaporation_WaterOnGround_m,PotentialEvaporation_WaterOnTree_m,PotentialSublimation_SnowOnGround_m,PotentialSublimation_SnowOnTree_m,LAI_BAI_Tree_m2_p_m2,LAI_BAI_SVeg_m2_p_m2" << endl;
		}
		outfile_WeatherLAI.close();
	}
	//ios::app allows for appending
	ofstream outfile_WeatherLAI(path, ios::app);
	if (!outfile_WeatherLAI.good()) {
		cout << "Warning: WaterBalance_Summary could not access path: " << path << endl;
		return;
	}
	//If Radiation_Shortwave_Direct_Wpm2 vector size > 0 then write Radiation.csv with Weather.csv and Evaporation.csv or equivalent terms
	//Note: Typically, Radiation_Shortwave_Direct_Wpm2 and related vectors are only availale for SpatialTemperatureModel
	if (input->Radiation_Shortwave_Direct_Wpm2.size() > 0) {
		//outfile_WeatherLAI stores SimulationDate_Output_GD (YYYYMMDD) SimulationTime_Output_HMS (HH:MM:SS) LAI_BAI_Tree_m2_p_m2, LAI_BAI_SVeg_m2_p_m2 values
		outfile_WeatherLAI << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			//Temperature_Air_C = input->Tair_C[timeStep]
			<< "," << input->Tair_C[timeStep]
			//Temperature_Dew_C = input->Tdew_C[timeStep]
			<< "," << input->Tdew_C[timeStep]
			<< "," << input->RadiationNet_Wpm2[timeStep]
			<< "," << input->WindSpd_mps[timeStep]
			<< "," << input->AtmPres_kPa[timeStep]
			<< "," << input->Precip_mpdt[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->Radiation_Shortwave_Direct_Wpm2[timeStep]
			<< "," << input->Radiation_Shortwave_Diffuse_Wpm2[timeStep]
			<< "," << input->Radiation_Longwave_Downwelling_Wpm2[timeStep]
			<< "," << input->Radiation_Longwave_Upwelling_Wpm2[timeStep]
			<< "," << input->PotentialEvapotranspiration_TreeCover_m[timeStep]
			<< "," << input->PotentialEvaporation_WaterOnGround_m[timeStep]
			<< "," << input->PotentialEvaporation_WaterOnTree_m[timeStep]
			<< "," << input->PotentialSublimation_SnowOnGround_m[timeStep]
			<< "," << input->PotentialSublimation_SnowOnTree_m[timeStep]
			<< "," << input->LAI_BAI_Tree_m2_p_m2[timeStep]
			<< "," << input->LAI_BAI_SVeg_m2_p_m2[timeStep]
			<< endl;
	}
	//Else Radiation_Shortwave_Direct_Wpm2 vector size is empty and only write Weather.csv and Evaporation.csv or equivalent terms
	else {
		//outfile_WeatherLAI stores SimulationDate_Output_GD (YYYYMMDD) SimulationTime_Output_HMS (HH:MM:SS) LAI_BAI_Tree_m2_p_m2, LAI_BAI_SVeg_m2_p_m2 values
		outfile_WeatherLAI << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			//Temperature_Air_C = input->Tair_C[timeStep]
			<< "," << input->Tair_C[timeStep]
			//Temperature_Dew_C = input->Tdew_C[timeStep]
			<< "," << input->Tdew_C[timeStep]
			<< "," << input->RadiationNet_Wpm2[timeStep]
			<< "," << input->WindSpd_mps[timeStep]
			<< "," << input->AtmPres_kPa[timeStep]
			<< "," << input->Precip_mpdt[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->PotentialEvapotranspiration_TreeCover_m[timeStep]
			<< "," << input->PotentialEvaporation_WaterOnGround_m[timeStep]
			<< "," << input->PotentialEvaporation_WaterOnTree_m[timeStep]
			<< "," << input->PotentialSublimation_SnowOnGround_m[timeStep]
			<< "," << input->PotentialSublimation_SnowOnTree_m[timeStep]
			<< "," << input->LAI_BAI_Tree_m2_p_m2[timeStep]
			<< "," << input->LAI_BAI_SVeg_m2_p_m2[timeStep]
			<< endl;
	}
	outfile_WeatherLAI.close();
}

//write_TimeSeries_CatchmentVariables function writes to several output files
void WaterBalance_TimeSeries::write_TimeSeries_CatchmentVariables(Inputs* input, int timeStep)
{
	int Folder_BulkArea_ID = 0;
	int Folder_GI_ID = 1;
	vector<int> Folder_List;
	//WriteOutputs* writeOutputs creates pointer for functions
	WriteOutputs* writeOutputs;

	//Folder_List is defined to contain Folder_BulkArea_ID
	Folder_List = { Folder_BulkArea_ID };
	//If Flag_GI_Simulated equals 1 then add GI to Folder_List
	if (input->RepoDict["Flag_GI_Simulated"] == 1) {
		//Folder_List is redefined to contain Folder_BulkArea_ID and Folder_GI_ID
		Folder_List = { Folder_BulkArea_ID, Folder_GI_ID };
	}

	//For Loop with Folder_Type_ID, representing values of BulkArea and GI
	//for (int Folder_Type_ID = 0; Folder_Type_ID < (Folder_GI_ID + 1); ++Folder_Type_ID ) {
	//for (auto Folder_Type_ID : Folder_List) {
	for (auto Folder_Type_ID = 0; Folder_Type_ID < Folder_List.size(); ++Folder_Type_ID) {

		//CA_GI_m2 is GI folder area initialized to zero
		double CA_GI_m2 = 0.0;
		//Name_Folder_Type_str initialized as string
		string Name_Folder_Type_str;
		//Length_Name_Folder_Type initialized as zero
		int Length_Name_Folder_Type = 0;

		//If Folder_Type_ID equals zero and Folder_List.size() < 2 then only BulkArea simulated, so call it CatchmentArea
		if (Folder_Type_ID == 0 && Folder_List.size() < 2) {
			//Name_Folder_Type_str set to Catchment
			Name_Folder_Type_str = "Catchment";
			//Length_Name_Folder_Type set to actual length
			Length_Name_Folder_Type = Name_Folder_Type_str.length();
			//CA_GI_m2 equal to 1 to avoid division by zero
			CA_GI_m2 = 1;
		}
		//If Folder_Type_ID equals zero and Folder_List.size() > 1 then more than BulkArea simulated, so call it BulkArea
		else if (Folder_Type_ID == 0 && Folder_List.size() > 1) {
			//Name_Folder_Type_str set to BulkArea
			Name_Folder_Type_str = "BulkArea";
			//Length_Name_Folder_Type set to actual length
			Length_Name_Folder_Type = Name_Folder_Type_str.length();
			//CA_GI_m2 equal to 1 to avoid division by zero
			CA_GI_m2 = 1;
		}
		//If Folder_Type_ID equals 1 then
		if (Folder_Type_ID == 1) {
			//Name_Folder_Type_str set to BulkArea
			Name_Folder_Type_str = "GreenInfrastructure";
			//Length_Name_Folder_Type set to actual length
			Length_Name_Folder_Type = Name_Folder_Type_str.length();
			//CA_GI_m2 equal to Catchment_Type_Area_m2 as function of Folder_Type_ID = 1
			CA_GI_m2 = input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
		}

		//If timeStep equals zero then open files
		if (timeStep == 0) {
			//Call WaterBalance_TimeSeries::openFiles_WaterBalanceTimeSeries(input, Folder_List.size(), Folder_Type_ID, Name_Folder_Type_str ....
			WaterBalance_TimeSeries::openFiles_WaterBalanceTimeSeries(input, Folder_List.size(), Folder_Type_ID, Name_Folder_Type_str, outfile_CanopyVariables_m, outfile_CanopyVariables_m3, outfile_PerviousCoverVariables_m, outfile_PerviousCoverVariables_m3, outfile_ImperviousCoverVariables_m, outfile_ImperviousCoverVariables_m3, outfile_TreeCoverVariables_m, outfile_TreeCoverVariables_m3, outfile_RunoffVariables_m, outfile_RunoffVariables_m3);
		}

		//CA_m2 is catchment area, CatchmentArea_m2 from HydroPlusConfig.xml
		double CA_m2 = input->SimulationNumericalParams["CatchmentArea_m2"];
		//CA_PC_frac is catchment pervious area fraction
		double CA_PC_frac = input->RepoDict["Catchment_TotalPerviousCover_frac"];
		//FA_m2 is folder area, Catchment_Type_Area_m2 as function of Folder_Type_ID
		double FA_m2 = input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
		//PC_m2 is pervious cover fraction of catchment area
		double PC_m2 = input->RepoVecDict["TotalPerviousCover_Type_m2"][Folder_Type_ID];
		//IC_m2 is impervious cover fraction of catchment area
		double IC_m2 = input->RepoVecDict["TotalImperviousCover_Type_m2"][Folder_Type_ID];
		//TC_m2 is tree cover fraction of catchment area
		double TC_m2 = input->RepoVecDict["TotalTreeCover_Type_m2"][Folder_Type_ID];
		//TC_overPC_m2 is tree cover over pervious cover fraction of catchment area
		double TC_overPC_m2 = input->RepoVecDict["TreeCanopyCover_overPervious_Type_m2"][Folder_Type_ID];
		//TC_overIC_m2 is tree cover over impervious cover fraction of catchment area
		double TC_overIC_m2 = input->RepoVecDict["TreeCanopyCover_overImpervious_Type_m2"][Folder_Type_ID];
		//IC_noTC_m2 is impervious cover not under tree cover fraction of catchment area
		double IC_noTC_m2 = input->RepoVecDict["ImperviousCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//PC_noTC_m2 is pervious area without tree cover 
		double PC_noTC_m2 = PC_m2 - TC_overPC_m2;
		//SVeg_noTC_m2 is short vegetation cover not under tree cover fraction of catchment area
		double SVegC_noTC_m2 = input->RepoVecDict["ShortVegCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//Soil_noTC_m2 is soil cover not under tree cover fraction of catchment area
		double SoilC_noTC_m2 = input->RepoVecDict["SoilCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//PermeablePavementC_noTC_m2 is permeable pavement cover not under tree cover fraction of catchment area
		double PermeablePavementC_noTC_m2 = input->RepoVecDict["PermeablePavementCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//Water_noTC_m2 is water cover not under tree cover fraction of catchment area
		double WaterC_noTC_m2 = input->RepoVecDict["WaterCover_noTreeCanopy_Type_m2"][Folder_Type_ID];
		//DCIA_m2 is directly connected impervious cover fraction of impervious area
		double DCIA_m2 = input->RepoVecDict["DCIA_Type_m2"][Folder_Type_ID];
		//InfilExSim_m2 is infiltration excess governed area fraction of catchment area
		double InfilExSim_m2 = input->RepoVecDict["InfiltExcessGovernedArea_Type_m2"][Folder_Type_ID];
		//Macropore_m2 is macropore fraction of catchment area
		double MacroPore_m2 = input->RepoVecDict["MacroPore_Type_m2"][Folder_Type_ID];
		//NoCanopy_m2 is area without tree or short vegetation canopy cover
		double NoCanopy_m2 = SoilC_noTC_m2 + PermeablePavementC_noTC_m2 + IC_noTC_m2 + WaterC_noTC_m2;
		//double NoCanopy_m2 = Folder_Area_m2 - TC_m2 - SVegC_noTC_m2;
		//EvapoTransCover_m2 = SVegC_noTC_m2 + SoilC_noTC_m2 + PermeablePavementC_noTC_m2 + TC_m2
		double EvapoTransCover_m2 = SVegC_noTC_m2 + SoilC_noTC_m2 + PermeablePavementC_noTC_m2 + TC_m2;

		double PC_frac = 0.0;
		double IC_frac = 0.0;
		double TC_frac = 0.0;
		double TC_overIC_frac = 0.0;
		double TC_overPC_frac = 0.0;
		double IC_noTC_frac = 0.0;
		double SVegC_noTC_frac = 0.0;
		double SoilC_noTC_frac = 0.0;
		double PermeablePavementC_noTC_frac = 0.0;
		double WaterC_noTC_frac = 0.0;
		double InfilExSim_frac = 0.0;
		double DCIA_frac = 0.0;
		double MacroPore_frac = 0.0;
		double PC_noTC_frac = 0.0;
		double Ratio_TC_overIC_to_TC = 0.0;
		double Ratio_TC_overPC_to_TC = 0.0;
		double Ratio_ICnoTC_to_noCanopy = 0.0;
		double Ratio_TC_to_EvapoTransCover_frac = 0.0;
		double Ratio_PC_to_EvapoTransCover_frac = 0.0;
		double Ratio_TCoIC_to_EvapoTransCover_frac = 0.0;
		double Ratio_CanopyCover_to_EvapoTransCover_frac = 0.0;

		//If Catchment_Type_Area_m2 > 0 then divide by Catchment_Type_Area_m2
		if (input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID] > 0) {
			PC_frac = input->RepoVecDict["TotalPerviousCover_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			IC_frac = input->RepoVecDict["TotalImperviousCover_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			TC_frac = input->RepoVecDict["TotalTreeCover_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			TC_overIC_frac = input->RepoVecDict["TreeCanopyCover_overImpervious_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			TC_overPC_frac = input->RepoVecDict["TreeCanopyCover_overPervious_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			IC_noTC_frac = input->RepoVecDict["ImperviousCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			SVegC_noTC_frac = input->RepoVecDict["ShortVegCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			SoilC_noTC_frac = input->RepoVecDict["SoilCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			WaterC_noTC_frac = input->RepoVecDict["WaterCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			InfilExSim_frac = input->RepoVecDict["InfiltExcessGovernedArea_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			MacroPore_frac = input->RepoVecDict["MacroPore_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
			DCIA_frac = input->RepoVecDict["DCIA_Type_m2"][Folder_Type_ID] / input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
		}
		//PC_noTC_frac (fraction) is pervious area without tree cover, PC_frac minus TC_overPC_frac
		PC_noTC_frac = PC_frac - TC_overPC_frac;
		//If TotalTreeCover_Type_m2 > 0 then divide by TotalTreeCover_Type_m2
		if (input->RepoVecDict["TotalTreeCover_Type_m2"][Folder_Type_ID] > 0) {
			Ratio_TC_overIC_to_TC = input->RepoVecDict["TreeCanopyCover_overImpervious_Type_m2"][Folder_Type_ID] / input->RepoVecDict["TotalTreeCover_Type_m2"][Folder_Type_ID];
			Ratio_TC_overPC_to_TC = input->RepoVecDict["TreeCanopyCover_overPervious_Type_m2"][Folder_Type_ID] / input->RepoVecDict["TotalTreeCover_Type_m2"][Folder_Type_ID];
		}
		//If EvapoTransCover_m2 > 0 then divide by EvapoTransCover_m2 
		if (EvapoTransCover_m2 > 0) {
			Ratio_TC_to_EvapoTransCover_frac = TC_m2 / EvapoTransCover_m2;
			Ratio_PC_to_EvapoTransCover_frac = PC_m2 / EvapoTransCover_m2;
			Ratio_TCoIC_to_EvapoTransCover_frac = TC_overIC_m2 / EvapoTransCover_m2;
			Ratio_CanopyCover_to_EvapoTransCover_frac = (TC_m2 + SVegC_noTC_m2) / EvapoTransCover_m2;
		}
		//If NoCanopy_m2 > 0 then divide by NoCanopy_m2
		if (NoCanopy_m2 > 0) {
			Ratio_ICnoTC_to_noCanopy = input->RepoVecDict["ImperviousCover_noTreeCanopy_Type_m2"][Folder_Type_ID] / NoCanopy_m2;
		}

		double Tree_WB_m = 0.0;
		double SVeg_WB_m = 0.0;
		double Tree_WB_m3 = 0.0;
		double SVeg_WB_m3 = 0.0;

		double PerArea_LiquidWater_WB_m = 0.0;
		double PerArea_FrozenWater_WB_m = 0.0;
		double PerArea_LiquidAndFrozenWater_WB_m = 0.0;

		double PerArea_LiquidWater_WB_m3 = 0.0;
		double PerArea_FrozenWater_WB_m3 = 0.0;
		double PerArea_LiquidAndFrozenWater_WB_m3 = 0.0;

		double ImpArea_LiquidWater_WB_m = 0.0;
		double ImpArea_FrozenWater_WB_m = 0.0;
		double ImpArea_comb_WB_m = 0.0;
		double ImpArea_Runoff_WB_m = 0.0;

		double ImpArea_LiquidWater_WB_m3 = 0.0;
		double ImpArea_FrozenWater_WB_m3 = 0.0;
		double ImpArea_comb_WB_m3 = 0.0;
		double ImpArea_Runoff_WB_m3 = 0.0;

		Tree_WB_m = (input->Rain_m[timeStep] + input->Snow_m[timeStep]) +
			(input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["StorageRain_TreeCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_TreeCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["EvaporationSublimation_TreeCanopy_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2);

		SVeg_WB_m = (input->Rain_m[timeStep] + input->Snow_m[timeStep]) +
			(input->RepoVecDict["Irrigation_SVegInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["StorageRain_SVegCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_SVegCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["EvaporationSublimation_SVegCanopy_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2);

		Tree_WB_m3 = (input->Rain_m[timeStep] + input->Snow_m[timeStep]) * TC_m2 +
			(input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["StorageRain_TreeCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_TreeCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["EvaporationSublimation_TreeCanopy_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["ThroughFallRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageRainSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]);

		SVeg_WB_m3 = (input->Rain_m[timeStep] + input->Snow_m[timeStep]) * SVegC_noTC_m2 +
			(input->RepoVecDict["Irrigation_SVegInterceptionStorage_TS_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["StorageRain_SVegCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_SVegCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["EvaporationSublimation_SVegCanopy_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["ThroughFallRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageRainSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]);

		PerArea_LiquidWater_WB_m = (input->Rain_m[timeStep] * SoilC_noTC_frac) +
			(input->RepoVecDict["Irrigation_PerviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) +
			(input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) +
			(input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] / FA_m2);

		PerArea_FrozenWater_WB_m = (input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["StorageSnow_UnderSVeg_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->Snow_m[timeStep] * SoilC_noTC_frac) +
			(input->Snow_m[timeStep] * PermeablePavementC_noTC_frac) +
			(input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) +
			(input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SublimationSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["SublimationSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2);

		PerArea_LiquidAndFrozenWater_WB_m = PerArea_LiquidWater_WB_m + PerArea_FrozenWater_WB_m;

		PerArea_LiquidWater_WB_m3 = (input->Rain_m[timeStep] * SoilC_noTC_m2) +
			(input->Rain_m[timeStep] * PermeablePavementC_noTC_m2) +
			input->RepoVecDict["Irrigation_PerviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] +
			(input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) +
			(input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) +
			(input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID]);

		PerArea_FrozenWater_WB_m3 = (input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Prior_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["StorageSnow_UnderSVeg_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->Snow_m[timeStep] * SoilC_noTC_m2) -
			(input->Snow_m[timeStep] * PermeablePavementC_noTC_m2) +
			(input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) +
			(input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SublimationSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC) -
			(input->RepoVecDict["SublimationSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]);

		PerArea_LiquidAndFrozenWater_WB_m3 = PerArea_LiquidWater_WB_m3 + PerArea_FrozenWater_WB_m3;

		ImpArea_LiquidWater_WB_m = (input->Rain_m[timeStep] * IC_noTC_frac) +
			input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2 +
			(input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC) +
			(input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC) -
			(input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] / FA_m2);

		//Note: snow depths are liquid equivalent; 
		//Note: ... precip, throughfall, and storage are frozen, sublimation is vapor, and the difference is snow melt
		ImpArea_FrozenWater_WB_m = (input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC) -
			(input->Snow_m[timeStep] * IC_noTC_frac) +
			(input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC) +
			(input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC) -
			(input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC) -
			(input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC);

		ImpArea_Runoff_WB_m = (input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] / FA_m2) +
			input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2 +
			(input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			(input->RepoVecDict["Storage_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID] / FA_m2) -
			input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 -
			input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2;

		ImpArea_LiquidWater_WB_m3 = (input->Rain_m[timeStep] * IC_noTC_frac) +
			input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] +
			(input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC) +
			(input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]) +
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC) +
			(input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID]);

		ImpArea_FrozenWater_WB_m3 = (input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Prior_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC) -
			(input->Snow_m[timeStep] * IC_noTC_frac) +
			(input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC) +
			(input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC) -
			(input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC) -
			(input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC);

		ImpArea_Runoff_WB_m3 = (input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID]) +
			input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] +
			(input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID]) -
			(input->RepoVecDict["Storage_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID]) -
			input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] -
			input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID];

		//If WB terms < Constant_1E_negative10 then set to zero
		if (Tree_WB_m < Constant_1E_negative10) { Tree_WB_m = 0; }
		if (SVeg_WB_m < Constant_1E_negative10) { SVeg_WB_m = 0; }
		if (Tree_WB_m3 < Constant_1E_negative10) { Tree_WB_m3 = 0; }
		if (SVeg_WB_m3 < Constant_1E_negative10) { SVeg_WB_m3 = 0; }
		if (PerArea_LiquidWater_WB_m < Constant_1E_negative10) { PerArea_LiquidWater_WB_m = 0; }
		if (PerArea_FrozenWater_WB_m < Constant_1E_negative10) { PerArea_FrozenWater_WB_m = 0; }
		if (PerArea_LiquidAndFrozenWater_WB_m < Constant_1E_negative10) { PerArea_LiquidAndFrozenWater_WB_m = 0; }
		if (PerArea_LiquidWater_WB_m3 < Constant_1E_negative10) { PerArea_LiquidWater_WB_m3 = 0; }
		if (PerArea_FrozenWater_WB_m3 < Constant_1E_negative10) { PerArea_FrozenWater_WB_m3 = 0; }
		if (PerArea_LiquidAndFrozenWater_WB_m3 < Constant_1E_negative10) { PerArea_LiquidAndFrozenWater_WB_m3 = 0; }
		if (ImpArea_LiquidWater_WB_m < Constant_1E_negative10) { ImpArea_LiquidWater_WB_m = 0; }
		if (ImpArea_FrozenWater_WB_m < Constant_1E_negative10) { ImpArea_FrozenWater_WB_m = 0; }
		if (ImpArea_Runoff_WB_m < Constant_1E_negative10) { ImpArea_Runoff_WB_m = 0; }
		if (ImpArea_LiquidWater_WB_m3 < Constant_1E_negative10) { ImpArea_LiquidWater_WB_m3 = 0; }
		if (ImpArea_FrozenWater_WB_m3 < Constant_1E_negative10) { ImpArea_FrozenWater_WB_m3 = 0; }
		if (ImpArea_Runoff_WB_m3 < Constant_1E_negative10) { ImpArea_Runoff_WB_m3 = 0; }

		//Write output
		//Note: Depths (m) are what is expected on cover area of interest, and not scaled to catchment area
		outfile_CanopyVariables_m[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["InterceptionRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["InterceptionSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["EvaporationRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << Tree_WB_m
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->RepoVecDict["Irrigation_SVegInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["InterceptionRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["InterceptionSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["EvaporationRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << SVeg_WB_m
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_CanopyCover_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_CanopyCover_to_EvapoTransCover_frac
			<< endl;

		outfile_CanopyVariables_m3[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep] * TC_m2
			<< "," << input->Snow_m[timeStep] * TC_m2
			<< "," << input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["InterceptionRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["InterceptionSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["EvaporationRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << Tree_WB_m3
			<< "," << input->Rain_m[timeStep] * SVegC_noTC_m2
			<< "," << input->Snow_m[timeStep] * SVegC_noTC_m2
			<< "," << input->RepoVecDict["Irrigation_SVegInterceptionStorage_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["InterceptionRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["InterceptionSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["EvaporationRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << SVeg_WB_m3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_CanopyCover_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] * Ratio_CanopyCover_to_EvapoTransCover_frac
			<< endl;

		//Note: Depths (m) are what is expected on cover area of interest, and not scaled to catchment area
		outfile_PerviousCoverVariables_m[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->RepoVecDict["Irrigation_PerviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << PerArea_LiquidWater_WB_m
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["SublimationSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << PerArea_FrozenWater_WB_m
			<< "," << PerArea_LiquidAndFrozenWater_WB_m
			<< "," << input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Evaporation_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Flux_to_Infiltration_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->SafeDivide(input->RepoVecDict["Infiltration_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Infiltration_viaInfilEx_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Infiltration_viaSatEx_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_macroPore_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			//Drainage_SoilEvapZone_TS_Sum_m (m) has header Drainage_SoilEvapZone_m, only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			//EvapoTranspiration_TS_Sum_m (m) has header EvapoTranspiration_SoilEvapZone_m, only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_PC_to_EvapoTransCover_frac
			<< "," << input->SafeDivide(input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_PC_to_EvapoTransCover_frac
			<< "," << input->SafeDivide(input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			//Storage_SoilEvapZone_TS_Sum_m3_p_m3 (m3/m3) has header Storage_SoilEvapZone_m3pm3, only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			<< "," << input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3_p_m3"][Folder_Type_ID]
			//Storage_VadoseZone_TS_Sum_m (m) has header Storage_VadoseZone_m, only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->SafeDivide(input->RepoVecDict["Storage_VadoseZone_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			//StorageDeficit_VadoseZone_TS_Sum_m (m) has header StorageDeficit_VadoseZone_m, is storage capacity for gravitational water across entire catchment vadose zone
			//Note: StorageDeficit_VadoseZone_TS_Sum_m (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 (m2) contains entire catchment pervious area draining to vadose zone
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 influences the catchment variable, StorageDeficit_VadoseZone_m
			<< "," << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_Sum_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"])
			<< "," << input->RepoVecDict["Groundwater_surficial_TS_Sum_frac"][Folder_Type_ID]
			//Drainage_VadoseZone_TS_Sum_m (m) has header Drainage_VadoseZone_m3, only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			//Note: Drainage_VadoseZone_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_VadoseZone_TS_Sum_m3"][Folder_Type_ID], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"])
			//Note: Runoff_Subsurface_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->SafeDivide(input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"])
			<< "," << input->SafeDivide(input->RepoVecDict["Runoff_SatExcess_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Runoff_InfilExcess_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Infiltration_viaSatEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Infiltration_viaInfilEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_macroPore_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_SoilEvapZone_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< endl;

		outfile_PerviousCoverVariables_m3[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep] * PC_m2
			<< "," << input->Snow_m[timeStep] * PC_m2
			<< "," << input->RepoVecDict["Irrigation_PerviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["ThroughFallRain_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SnowMelt_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SnowMelt_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["SnowMelt_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID]
			<< "," << PerArea_LiquidWater_WB_m3
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["ThroughFallSnow_SVegCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["StorageSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_SoilNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_PermeablePavementNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["SublimationSnow_UnderSVeg_TS_Sum_m3"][Folder_Type_ID]
			<< "," << PerArea_FrozenWater_WB_m3
			<< "," << PerArea_LiquidAndFrozenWater_WB_m3
			<< "," << input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Evaporation_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Flux_to_Infiltration_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Infiltration_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Infiltration_viaInfilEx_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Infiltration_viaSatEx_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Drainage_macroPore_TS_Sum_m3"][Folder_Type_ID]
			//Drainage_SoilEvapZone_TS_Sum_m3 (m3) has header Drainage_SoilEvapZone_m3, only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->RepoVecDict["Drainage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID]
			//EvapoTranspiration_TS_Sum_m3 (m3) has header EvapoTranspiration_PerviousArea_m3, only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_PC_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] * Ratio_PC_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID]
			//Storage_SoilEvapZone_TS_Sum_m3_p_m3 (m3/m3) has header Storage_SoilEvapZone_m3pm3, only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			<< "," << input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3_p_m3"][Folder_Type_ID]
			//Storage_VadoseZone_TS_Sum_m3 (m3) has header Storage_VadoseZone_m3, only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->RepoVecDict["Storage_VadoseZone_TS_Sum_m3"][Folder_Type_ID]
			//StorageDeficit_VadoseZone_TS_Sum_m3 (m3) has header StorageDeficit_VadoseZone_m3, is storage capacity for gravitational water across entire catchment vadose zone
			//Note: StorageDeficit_VadoseZone_TS_Sum_m (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 (m2) contains entire catchment pervious area draining to vadose zone
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 influences the catchment variable, StorageDeficit_VadoseZone_m
			<< "," << input->RepoDict["StorageDeficit_VadoseZone_TS_Sum_m3"]
			<< "," << input->RepoVecDict["Groundwater_surficial_TS_Sum_frac"][Folder_Type_ID]
			//Drainage_VadoseZone_TS_Sum_m3 (m3) has header Drainage_VadoseZone_m3, only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			//Note: Drainage_VadoseZone_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->RepoVecDict["Drainage_VadoseZone_TS_Sum_m3"][Folder_Type_ID]
			//Note: Runoff_Subsurface_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_SatExcess_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_InfilExcess_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Infiltration_viaSatEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Infiltration_viaInfilEx_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Drainage_macroPore_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Drainage_SoilEvapZone_PreSatExRunoff_TS_Sum_m3"][Folder_Type_ID] 
			<< endl;

		outfile_ImperviousCoverVariables_m[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << ImpArea_LiquidWater_WB_m
			<< "," << ImpArea_FrozenWater_WB_m
			<< "," << input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << ImpArea_Runoff_WB_m
			//EvapoTranspiration_TS_Sum_m (m) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_TCoIC_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * Ratio_TCoIC_to_EvapoTransCover_frac
			<< endl;

		outfile_ImperviousCoverVariables_m3[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep] * IC_m2
			<< "," << input->Snow_m[timeStep] * IC_m2
			<< "," << input->RepoVecDict["Irrigation_ImperviousDepressionStorage_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["SublimationSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["StorageSnow_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["Flux_to_ImperviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID]
			<< "," << ImpArea_LiquidWater_WB_m3
			<< "," << ImpArea_FrozenWater_WB_m3
			<< "," << input->RepoVecDict["SnowMelt_ImperviousNoCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overIC_to_TC
			<< "," << input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID]
			<< "," << ImpArea_Runoff_WB_m3
			//EvapoTranspiration_TS_Sum_m3 (m3) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_TCoIC_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] * Ratio_TCoIC_to_EvapoTransCover_frac
			<< endl;

		outfile_TreeCoverVariables_m[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			//Note: Terms not multiplied by Ratio_TC_overPC_to_TC or Ratio_TC_overIC_to_TC to give total tree cover value
			<< "," << input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["InterceptionRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["InterceptionSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["EvaporationRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << Tree_WB_m
			<< "," << input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			//Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3 (m) is water flux to tree as rain and snowmelt on pervious area
			<< "," << input->SafeDivide(input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			//Flux_Impervious_to_Pervious_TS_Sum_m (m) is water flux to tree as runoff from impervious area
			<< "," << input->SafeDivide(input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			//Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3 (m) is water flux to tree as rain and snowmelt on pervious area and runoff from impervious area
			<< "," << input->SafeDivide(input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"][Folder_Type_ID], IC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Evaporation_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID], PC_m2)
			<< "," << input->SafeDivide(input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID], IC_m2)
			//Consider Refactor to weight terms based on Ratio_TC_overPC_to_TC to represent tree with pervious access
			<< "," << input->SafeDivide(input->RepoVecDict["Flux_to_Infiltration_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_overPC_to_TC
			<< "," << input->SafeDivide(input->RepoVecDict["Infiltration_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_overPC_to_TC
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_macroPore_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_overPC_to_TC
			//Drainage_SoilEvapZone_TS_Sum_m (m) only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_overPC_to_TC
			//EvapoTranspiration_TS_Sum_m (m) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_TC_to_EvapoTransCover_frac
			<< "," << input->SafeDivide(input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_to_EvapoTransCover_frac
			//Storage_SoilEvapZone_TS_Sum_m (m) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			<< "," << input->SafeDivide(input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_overPC_to_TC
			//Storage_SoilEvapZone_TS_Sum_m3_p_m3 (m3/m3) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			<< "," << input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3_p_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//Storage_VadoseZone_TS_Sum_m (m) only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->SafeDivide(input->RepoVecDict["Storage_VadoseZone_TS_Sum_m3"][Folder_Type_ID], PC_m2) * Ratio_TC_overPC_to_TC
			//StorageDeficit_VadoseZone_TS_Sum_m (m) is storage capacity for gravitational water across entire catchment vadose zone
			//Note: StorageDeficit_VadoseZone_TS_Sum_m (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 (m2) contains entire catchment pervious area draining to vadose zone
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 influences the catchment variable, StorageDeficit_VadoseZone_m
			<< "," << input->SafeDivide(input->RepoDict["StorageDeficit_VadoseZone_Sum_m3"], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_TC_overPC_to_TC
			//Drainage_VadoseZone_TS_Sum_m (m) only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			//Note: Drainage_VadoseZone_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->SafeDivide(input->RepoVecDict["Drainage_VadoseZone_TS_Sum_m3"][Folder_Type_ID], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * Ratio_TC_overPC_to_TC
			//Note: Runoff_Subsurface_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->SafeDivide(input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * TC_overPC_frac
			<< "," << input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * TC_overPC_frac
			<< "," << input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * TC_overIC_frac
			<< "," << input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * TC_overPC_frac + input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * TC_overIC_frac
			<< "," << input->SafeDivide(input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID], input->RepoDict["Catchment_PerviousArea_AffectingStorageDeficit_m2"]) * TC_overPC_frac + input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * TC_overPC_frac + input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2 * TC_overIC_frac
			<< endl;

		outfile_TreeCoverVariables_m3[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep] * TC_m2
			<< "," << input->Snow_m[timeStep] * TC_m2
			<< "," << input->RepoVecDict["Irrigation_TreeInterceptionStorage_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["InterceptionRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["InterceptionSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["EvaporationRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallRain_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["ThroughFallSnow_TreeCanopy_TS_Sum_m3"][Folder_Type_ID]
			<< "," << Tree_WB_m3
			<< "," << input->RepoVecDict["SnowMelt_UnderTree_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["SublimationSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["StorageSnow_UnderTree_TS_Sum_m3"][Folder_Type_ID]
			//Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3 (m3) is water flux to tree as rain and snowmelt on pervious area
			<< "," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_TS_Sum_m3"][Folder_Type_ID]
			//Flux_Impervious_to_Pervious_TS_Sum_m3 (m3) is water flux to tree as runoff from impervious area
			<< "," << input->RepoVecDict["Flux_Impervious_to_Pervious_TS_Sum_m3"][Folder_Type_ID] 
			//Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3 (m3) is water flux to tree as rain and snowmelt on pervious area and runoff from impervious area
			<< "," << input->RepoVecDict["Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_ImpRunon_TS_Sum_m3"][Folder_Type_ID] 
			<< "," << input->RepoVecDict["Storage_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Storage_ImperviousPondedWater_TS_Prior_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Evaporation_PerviousPondedWater_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Evaporation_ImperviousPondedWater_TS_Sum_m3"][Folder_Type_ID]
			//Consider Refactor to weight terms based on Ratio_TC_overPC_to_TC to represent tree with pervious access
			<< "," << input->RepoVecDict["Flux_to_Infiltration_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["Infiltration_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			<< "," << input->RepoVecDict["Drainage_macroPore_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//Drainage_SoilEvapZone_TS_Sum_m3 (m3) only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->RepoVecDict["Drainage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//EvapoTranspiration_TS_Sum_m3 (m3) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			//EvapoTranspiration_TS_Sum_m3 is scaled by Ratio_TC_to_EvapoTransCover_frac
			<< "," << input->RepoVecDict["EvapoTranspiration_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_to_EvapoTransCover_frac 
			//Storage_SoilEvapZone_TS_Sum_m3 (m3) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			<< "," << input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//Storage_SoilEvapZone_TS_Sum_m3_p_m3 (m3/m3) only between Soil_FieldCapacity_m3pm3 and Soil_WiltingPoint_m3pm3
			<< "," << input->RepoVecDict["Storage_SoilEvapZone_TS_Sum_m3_p_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//Storage_VadoseZone_TS_Sum_m3 (m3) only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			<< "," << input->RepoVecDict["Storage_VadoseZone_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//StorageDeficit_VadoseZone_TS_Sum_m3 (m3) is storage capacity for gravitational water across entire catchment vadose zone
			//Note: StorageDeficit_VadoseZone_TS_Sum_m (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 (m2) contains entire catchment pervious area draining to vadose zone
			//Note: Catchment_PerviousArea_AffectingStorageDeficit_m2 influences the catchment variable, StorageDeficit_VadoseZone_m
			<< "," << input->RepoDict["StorageDeficit_VadoseZone_TS_Sum_m3"] * Ratio_TC_overPC_to_TC
			//Drainage_VadoseZone_TS_Sum_m3 (m3) only between Soil_SaturationPoint_m3pm3 and Soil_FieldCapacity_m3pm3
			//Note: Drainage_VadoseZone_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->RepoVecDict["Drainage_VadoseZone_TS_Sum_m3"][Folder_Type_ID] * Ratio_TC_overPC_to_TC
			//Note: Runoff_Subsurface_TS_Sum_m3 (m3) converted to catchment depth (m) by division with Catchment_PerviousArea_AffectingStorageDeficit_m2
			<< "," << input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID] * TC_overPC_frac
			<< "," << input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] * TC_overPC_frac
			<< "," << input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] * TC_overIC_frac
			<< "," << input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] * TC_overPC_frac + input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] * TC_overIC_frac
			<< "," << input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID]* TC_overPC_frac + input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] * TC_overPC_frac + input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] * TC_overIC_frac
			<< endl;

		outfile_RunoffVariables_m[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			//Folder unrouted runoff
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep]
			<< "," << input->Snow_m[timeStep]
			<< "," << input->RepoVecDict["Runoff_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_Surface_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_Water_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_SatExcess_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< "," << input->RepoVecDict["Runoff_InfilExcess_TS_Sum_m3"][Folder_Type_ID] / FA_m2
			<< endl;

		outfile_RunoffVariables_m3[Folder_Type_ID] << input->SimulationDate_Output_GD[timeStep]
			<< "," << input->SimulationTime_Output_HMS[timeStep]
			<< "," << input->Rain_m[timeStep] * FA_m2
			<< "," << input->Snow_m[timeStep] * FA_m2
			//Folder unrouted runoff
			<< "," << input->RepoVecDict["Runoff_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_Surface_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_Subsurface_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_Impervious_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_Water_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_Pervious_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_SatExcess_TS_Sum_m3"][Folder_Type_ID]
			<< "," << input->RepoVecDict["Runoff_InfilExcess_TS_Sum_m3"][Folder_Type_ID]
			<< endl;

		//If timeStep equals SimulationTimePeriod_timeSteps then close files
		if ((timeStep + 1) == input->SimulationTimePeriod_timeSteps) {
			outfile_CanopyVariables_m[Folder_Type_ID].close();
			outfile_CanopyVariables_m3[Folder_Type_ID].close();
			outfile_PerviousCoverVariables_m[Folder_Type_ID].close();
			outfile_PerviousCoverVariables_m3[Folder_Type_ID].close();
			outfile_ImperviousCoverVariables_m[Folder_Type_ID].close();
			outfile_ImperviousCoverVariables_m3[Folder_Type_ID].close();
			outfile_TreeCoverVariables_m[Folder_Type_ID].close();
			outfile_TreeCoverVariables_m3[Folder_Type_ID].close();
			outfile_RunoffVariables_m[Folder_Type_ID].close();
			outfile_RunoffVariables_m3[Folder_Type_ID].close();
		}
	}
}

//writeWaterBalanceHeader_GIUnit function writes header for all GI unit types
void WaterBalance_TimeSeries::writeWaterBalanceHeader_GIUnit(Inputs* input, int timeStep)
{
	string file_path;
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_Budget_Water_GI.csv";
	ofstream outfile_Budget_Water_GI(file_path);
	if (!outfile_Budget_Water_GI.good()) { return; }
	outfile_Budget_Water_GI.precision(6);
	outfile_Budget_Water_GI.setf(ios::showpoint);

	outfile_Budget_Water_GI << "YYYYMMDD"
		<< "," << "HH:MM:SS"
		<< "," << "GI_Precipitation_m"
		<< "," << "GI_Inflow_m"
		<< "," << "GI_Runoff_Surface_m"
		<< "," << "GI_Runoff_Vault_m"
		<< "," << "GI_Infiltration_m"
		<< "," << "GI_Drainage_Percolation_m"
		<< "," << "GI_Drainage_Exfiltration_m"
		<< "," << "GI_Storage_Surface_m"
		<< "," << "GI_PavementWater_m"
		<< "," << "GI_Storage_Soil_m"
		<< "," << "GI_Storage_Vault_m"
		<< "," << "GI_EvaporationSurface_m"
		<< "," << "GI_EvaporationPavement_m"
		<< "," << "GI_EvaporationSoil_m"
		<< "," << "GI_EvaporationVault_m"
		<< "," << "GI_WaterTableDepth_m"
		<< "," << "GI_Storage_GroundwaterMound_m"
		<< "," << "GI_Pavement_Max_Potential_m"
		<< "," << "GI_Pavement_Max_m"
		<< "," << "GI_Soil_Max_Potential_m"
		<< "," << "GI_Soil_Max_m"
		<< "," << "GI_Vault_Max_Potential_m"
		<< "," << "GI_Vault_Max_m"
		<< endl;
}
//writeWaterBalanceTS_GIUnit function writes water balance time series for all GI unit types
void WaterBalance_TimeSeries::writeWaterBalanceTS_GIUnit(Inputs* input, int timeStep)
{
	int Folder_GI_ID = 1;
	int Folder_Type_ID = Folder_GI_ID;
	double FA_m2 = input->RepoVecDict["Catchment_Type_Area_m2"][Folder_Type_ID];
	string file_path;
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_Budget_Water_GI.csv";

	ofstream outfile_Budget_Water_GI(file_path, ios::app);
	if (!outfile_Budget_Water_GI.good()) { return; }
	outfile_Budget_Water_GI.precision(6);
	outfile_Budget_Water_GI.setf(ios::showpoint);

	outfile_Budget_Water_GI << input->SimulationDate_Output_GD[timeStep]
		<< "," << input->SimulationTime_Output_HMS[timeStep]
		<< "," << input->RepoDict["GI_Precipitation_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Runon_from_BulkArea_Surface_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Runoff_Surface_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Runoff_VaultDrain_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Infiltration_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Drainage_Percolation_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Drainage_Exfiltration_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_Surface_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_PavementWater_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_Soil_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_Vault_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Evaporation_Surface_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Evaporation_Pavement_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Evaporation_Soil_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Evaporation_Vault_TS_Sum_m3"] / FA_m2
		//Note: Depth_to_Groundwater_TS_m is not a volume, and not divided by FA_m2
		<< "," << input->RepoDict["Depth_to_Groundwater_TS_m"]
		<< "," << input->RepoDict["GI_Storage_GroundwaterMound_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_GI_Pavement_Max_Potential_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_GI_Pavement_Max_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_GI_Soil_Max_Potential_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_GI_Soil_Max_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_GI_Vault_Max_Potential_TS_Sum_m3"] / FA_m2
		<< "," << input->RepoDict["GI_Storage_GI_Vault_Max_TS_Sum_m3"] / FA_m2
		<< endl;

	//If timeStep equals SimulationTimePeriod_timeSteps then close files
	if ((timeStep + 1) == input->SimulationTimePeriod_timeSteps) {
		outfile_Budget_Water_GI.close();
	}
}

//writeWaterBalanceTS_GIUnit function writes water balance time series for all GI unit types in append mode, using .emplace_back
//Note: Prior function needs to create a clean version of the files
void WaterBalance_TimeSeries::openFiles_WaterBalanceTimeSeries(Inputs* input, int Folder_List_size, int Folder_Type_ID, string Name_Folder_Type_str, 
	vector<ofstream>& outfile_CanopyVariables_m, vector<ofstream>& outfile_CanopyVariables_m3, vector<ofstream>& outfile_PerviousCoverVariables_m, 
	vector<ofstream>& outfile_PerviousCoverVariables_m3, vector<ofstream>& outfile_ImperviousCoverVariables_m, 
	vector<ofstream>& outfile_ImperviousCoverVariables_m3, vector<ofstream>& outfile_TreeCoverVariables_m,
	vector<ofstream>& outfile_TreeCoverVariables_m3, vector<ofstream>& outfile_RunoffVariables_m, vector<ofstream>& outfile_RunoffVariables_m3)
{
	string file_path;
	//Flag_Clear_and_Append initialized to 1
	bool Flag_Clear_and_Append = 1;
	//if (input->SimulationStringParams["Model_Selection"] == "SpatialBufferwGI") then
	if (input->SimulationStringParams["Model_Selection"] == "SpatialBufferwGI") {
		//Flag_Clear_and_Append = 0; to ensure that files are not cleared first
		Flag_Clear_and_Append = 0;
	}

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_CanopyVariables_" + Name_Folder_Type_str + "_m.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_CanopyVariables_m.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_CanopyVariables_m.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_CanopyVariables_m.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_CanopyVariables_m[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_CanopyVariables_m[Folder_Type_ID].precision(6);
	outfile_CanopyVariables_m[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_CanopyVariables_" + Name_Folder_Type_str + "_m3.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_CanopyVariables_m3.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_CanopyVariables_m3.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_CanopyVariables_m3.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_CanopyVariables_m3[Folder_Type_ID].good()) {
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_CanopyVariables_m3[Folder_Type_ID].precision(6);
	outfile_CanopyVariables_m3[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_PerviousVariables_" + Name_Folder_Type_str + "_m.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_PerviousCoverVariables_m.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_PerviousCoverVariables_m.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_PerviousCoverVariables_m.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_PerviousCoverVariables_m[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_PerviousCoverVariables_m[Folder_Type_ID].precision(6);
	outfile_PerviousCoverVariables_m[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_PerviousVariables_" + Name_Folder_Type_str + "_m3.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_PerviousCoverVariables_m3.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_PerviousCoverVariables_m3.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_PerviousCoverVariables_m3.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_PerviousCoverVariables_m3[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_PerviousCoverVariables_m3[Folder_Type_ID].precision(6);
	outfile_PerviousCoverVariables_m3[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_ImperviousVariables_" + Name_Folder_Type_str + "_m.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_ImperviousCoverVariables_m.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_ImperviousCoverVariables_m.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_ImperviousCoverVariables_m.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_ImperviousCoverVariables_m[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_ImperviousCoverVariables_m[Folder_Type_ID].precision(6);
	outfile_ImperviousCoverVariables_m[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_ImperviousVariables_" + Name_Folder_Type_str + "_m3.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_ImperviousCoverVariables_m3.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_ImperviousCoverVariables_m3.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_ImperviousCoverVariables_m3.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_ImperviousCoverVariables_m3[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_ImperviousCoverVariables_m3[Folder_Type_ID].precision(6);
	outfile_ImperviousCoverVariables_m3[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_TreeCoverVariables_" + Name_Folder_Type_str + "_m.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_TreeCoverVariables_m.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_TreeCoverVariables_m.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_TreeCoverVariables_m.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_TreeCoverVariables_m[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_TreeCoverVariables_m[Folder_Type_ID].precision(6);
	outfile_TreeCoverVariables_m[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_TreeCoverVariables_" + Name_Folder_Type_str + "_m3.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_TreeCoverVariables_m3.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_TreeCoverVariables_m3.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_TreeCoverVariables_m3.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_TreeCoverVariables_m3[Folder_Type_ID].good()) {
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_TreeCoverVariables_m3[Folder_Type_ID].precision(6);
	outfile_TreeCoverVariables_m3[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_Runoff_" + Name_Folder_Type_str + "_m.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_RunoffVariables_m.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_RunoffVariables_m.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_RunoffVariables_m.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_RunoffVariables_m[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_RunoffVariables_m[Folder_Type_ID].precision(6);
	outfile_RunoffVariables_m[Folder_Type_ID].setf(ios::showpoint);

	//file_path contains directory and file name, with Name_Folder_Type_str variable
	file_path = input->SimulationStringParams["OutputFolder_Path"] + "TimeSeries_Runoff_" + Name_Folder_Type_str + "_m3.csv";
	//reserve function creates Folder_list_size speces in ofstream vector
	outfile_RunoffVariables_m3.reserve(Folder_List_size);

	//If Flag_Clear_and_Append > 0 then clear file
	if (Flag_Clear_and_Append > 0) {
		//emplace_back(ofstream{ file_path, ios::out | ios::trunc }) will truncate and clear to write; used when called once during simulation
		outfile_RunoffVariables_m3.emplace_back(ofstream{ file_path, ios::out | ios::trunc });
	}
	//Else Flag_Clear_and_Append is 0 then only append file
	else {
		//emplace_back(ofstream{ file_path, ios_base::app }) will append without clearing; used when called multiple times with SpatialBufferwGI
		outfile_RunoffVariables_m3.emplace_back(ofstream{ file_path, ios_base::app });
	}

	if (!outfile_RunoffVariables_m3[Folder_Type_ID].good()) { 
		cout << "Warning: Output folder does not exist or an output file in that folder is open and cannot be overwritten." << endl;
		cout << "Aborting: This warning triggers the HydroPlus simulation to abort." << endl;
		cout << "Correction: Confirm the HydroPlusConfig.xml parameter OutputFolder_Path and files in that folder are closed." << endl;
		//Call abort function, which ends the HydroPlus.exe simulation
		abort();
	}
	outfile_RunoffVariables_m3[Folder_Type_ID].precision(6);
	outfile_RunoffVariables_m3[Folder_Type_ID].setf(ios::showpoint);

		//ofstream vector receives header 
		outfile_CanopyVariables_m[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_TreeCanopy_m"
			<< "," << "Precipitation_Snow_WaterEquivalent_TreeCanopy_m"
			<< "," << "Irrigation_TreeInterceptionStorage_m"
			<< "," << "Interception_TreeCanopy_Rain_m"
			<< "," << "Interception_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "Evaporation_TreeCanopy_Rain_m"
			<< "," << "Sublimation_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "Storage_TreeCanopy_Rain_m"
			<< "," << "Storage_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "ThroughFall_TreeCanopy_Rain_m"
			<< "," << "ThroughFall_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "WaterBalance_TreeCanopy_m"
			<< "," << "Precipitation_Rain_ShortVegCanopy_m"
			<< "," << "Precipitation_Snow_WaterEquivalent_ShortVegCanopy_m"
			<< "," << "Irrigation_ShortVegInterceptionStorage_m" 
			<< "," << "Interception_ShortVegCanopy_Rain_m"
			<< "," << "Interception_SW_Snow_WaterEquivalent_m"
			<< "," << "Evaporation_ShortVegCanopy_Rain_m"
			<< "," << "Sublimation_ShortVegCanopy_Snow_WaterEquivalent_m"
			<< "," << "Storage_ShortVegCanopy_Rain_m"
			<< "," << "Storage_ShortVegCanopy_Snow_WaterEquivalent_m"
			<< "," << "ThroughFall_ShortVegCanopy_Rain_m"
			<< "," << "ThroughFall_ShortVegCanopy_Snow_WaterEquivalent_m"
			<< "," << "WaterBalance_ShortVegCanopy_m"
			<< "," << "EvapoTranspiration_m"
			<< endl;

		outfile_CanopyVariables_m3[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_TreeCanopy_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_TreeCanopy_m3"
			<< "," << "Irrigation_TreeInterceptionStorage_m3"
			<< "," << "Interception_TreeCanopy_Rain_m3"
			<< "," << "Interception_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "Evaporation_TreeCanopy_Rain_m3"
			<< "," << "Sublimation_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "Storage_TreeCanopy_Rain_m3"
			<< "," << "Storage_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "ThroughFall_TreeCanopy_Rain_m3"
			<< "," << "ThroughFall_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "WaterBalance_TreeCanopy_m3"
			<< "," << "Precipitation_Rain_ShortVegCanopy_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_ShortVegCanopy_m3"
			<< "," << "Irrigation_ShortVegInterceptionStorage_m3"
			<< "," << "Interception_ShortVegCanopy_Rain_m3"
			<< "," << "Interception_SW_Snow_WaterEquivalent_m3"
			<< "," << "Evaporation_ShortVegCanopy_Rain_m3"
			<< "," << "Sublimation_ShortVegCanopy_Snow_WaterEquivalent_m3"
			<< "," << "Storage_ShortVegCanopy_Rain_m3"
			<< "," << "Storage_ShortVegCanopy_Snow_WaterEquivalent_m3"
			<< "," << "ThroughFall_ShortVegCanopy_Rain_m3"
			<< "," << "ThroughFall_ShortVegCanopy_Snow_WaterEquivalent_m3"
			<< "," << "WaterBalance_ShortVegCanopy_m3"
			<< "," << "EvapoTranspiration_m3"
			<< endl;

		outfile_PerviousCoverVariables_m[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_PerviousArea_m"
			<< "," << "Precipitation_Snow_WaterEquivalent_PerviousArea_m"
			<< "," << "Irrigation_PerviousDepressionStorage_m"
			<< "," << "ThroughFall_TreeCanopy_PerviousArea_m"
			<< "," << "ThroughFall_ShortVegCanopy_m"
			<< "," << "SnowMelt_SoilNoCanopy_m"
			<< "," << "SnowMelt_PermeablePavementNoCanopy_m" 
			<< "," << "SnowMelt_TreeCanopy_PerviousArea_m"
			<< "," << "SnowMelt_ShortVegCanopy_m"
			<< "," << "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_m"
			<< "," << "WaterBalance_PerviousArea_Rain_SnowMelt_Irrigation_m"
			<< "," << "ThroughFall_TreeCanopy_PerviousArea_Snow_WaterEquivalent_m"
			<< "," << "ThroughFall_ShortVegCanopy_Snow_WaterEquivalent_m"
			<< "," << "Storage_Snow_WaterEquivalent_SoilNoCanopy_m"
			<< "," << "Storage_Snow_WaterEquivalent_PermeablePavementNoCanopy_m"
			<< "," << "Storage_Snow_WaterEquivalent_TreeCanopy_PerviousArea_m"
			<< "," << "Storage_Snow_WaterEquivalent_ShortVegCanopy_m"
			<< "," << "Sublimation_Snow_WaterEquivalent_TreeCanopy_PerviousArea_m"
			<< "," << "Sublimation_Snow_WaterEquivalent_SoilNoCanopy_m"
			<< "," << "Sublimation_Snow_WaterEquivalent_PermeablePavementNoCanopy_m"
			<< "," << "Sublimation_Snow_WaterEquivalent_ShortVegCanopy_m"
			<< "," << "WaterBalance_PerviousArea_Snow_WaterEquivalent_m"
			<< "," << "WaterBalance_PerviousArea_Precipitation_m"
			<< "," << "Flux_ImperviousAreaRunon_to_PerviousArea_m"
			<< "," << "Flux_to_PerviousArea_m"
			<< "," << "Storage_Surface_m"
			<< "," << "Evaporation_Surface_m"
			<< "," << "Flux_to_Infiltration_m"
			<< "," << "Infiltration_m"
			<< "," << "Infiltration_viaInfilEx_m"
			<< "," << "Infiltration_viaSatEx_m"
			<< "," << "Drainage_macroPore_m"
			<< "," << "Drainage_SoilEvapZone_m"
			<< "," << "EvapoTranspiration_m"
			<< "," << "Storage_SoilEvapZone_m"
			<< "," << "Storage_SoilEvapZone_m3pm3"
			<< "," << "Storage_VadoseZone_m"
			<< "," << "StorageDeficit_VadoseZone_m"
			<< "," << "Groundwater_surficial_frac"
			<< "," << "Drainage_VadoseZone_m"
			<< "," << "Runoff_SatExcess_m"
			<< "," << "Runoff_InfilExcess_m"
			<< "," << "Runoff_Subsurface_m"
			<< "," << "Infiltration_viaSatEx_PreSatExRunoff_m"
			<< "," << "Infiltration_viaInfilEx_PreSatExRunoff_m"
			<< "," << "Drainage_macroPore_PreSatExRunoff_m"
			<< "," << "Drainage_SoilEvapZone_PreSatExRunoff_m"
			<< endl;

		outfile_PerviousCoverVariables_m3[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_PerviousArea_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_PerviousArea_m3"
			<< "," << "Irrigation_PerviousDepressionStorage_m3"
			<< "," << "ThroughFall_TreeCanopy_PerviousArea_m3"
			<< "," << "ThroughFall_ShortVegCanopy_m3"
			<< "," << "SnowMelt_SoilNoCanopy_m3"
			<< "," << "SnowMelt_PermeablePavementNoCanopy_m3"
			<< "," << "SnowMelt_TreeCanopy_PerviousArea_m3"
			<< "," << "SnowMelt_ShortVegCanopy_m3"
			<< "," << "Flux_to_PerviousArea_Rain_SnowMelt_Irrigation_m3"
			<< "," << "WaterBalance_PerviousArea_Rain_SnowMelt_Irrigation_m3"
			<< "," << "ThroughFall_TreeCanopy_PerviousArea_Snow_WaterEquivalent_m3"
			<< "," << "ThroughFall_ShortVegCanopy_Snow_WaterEquivalent_m3"
			<< "," << "Storage_Snow_WaterEquivalent_SoilNoCanopy_m3"
			<< "," << "Storage_Snow_WaterEquivalent_PermeablePavementNoCanopy_m3"
			<< "," << "Storage_Snow_WaterEquivalent_TreeCanopy_PerviousArea_m3"
			<< "," << "Storage_Snow_WaterEquivalent_ShortVegCanopy_m3"
			<< "," << "Sublimation_Snow_WaterEquivalent_TreeCanopy_PerviousArea_m3"
			<< "," << "Sublimation_Snow_WaterEquivalent_SoilNoCanopy_m3"
			<< "," << "Sublimation_Snow_WaterEquivalent_PermeablePavementNoCanopy_m3"
			<< "," << "Sublimation_Snow_WaterEquivalent_ShortVegCanopy_m3"
			<< "," << "WaterBalance_PerviousArea_Snow_WaterEquivalent_m3"
			<< "," << "WaterBalance_PerviousArea_Precipitation_m3"
			<< "," << "Flux_ImperviousAreaRunon_to_PerviousArea_m3"
			<< "," << "Flux_to_PerviousArea_m3"
			<< "," << "Storage_Surface_m3"
			<< "," << "Evaporation_Surface_m3"
			<< "," << "Flux_to_Infiltration_m3"
			<< "," << "Infiltration_m3"
			<< "," << "Infiltration_viaInfilEx_m3"
			<< "," << "Infiltration_viaSatEx_m3"
			<< "," << "Drainage_macroPore_m3"
			<< "," << "Drainage_SoilEvapZone_m3"
			<< "," << "EvapoTranspiration_PerviousArea_m3"
			<< "," << "Storage_SoilEvapZone_m3"
			<< "," << "Storage_SoilEvapZone_m3pm3"
			<< "," << "Storage_VadoseZone_m3"
			<< "," << "StorageDeficit_VadoseZone_m3"
			<< "," << "Groundwater_surficial_frac"
			<< "," << "Drainage_VadoseZone_m3"
			<< "," << "Runoff_SatExcess_m3"
			<< "," << "Runoff_InfilExcess_m3"
			<< "," << "Runoff_Subsurface_m3"
			<< "," << "Infiltration_viaSatEx_PreSatExRunoff_m3"
			<< "," << "Infiltration_viaInfilEx_PreSatExRunoff_m3"
			<< "," << "Drainage_macroPore_PreSatExRunoff_m3"
			<< "," << "Drainage_SoilEvapZone_PreSatExRunoff_m3"
			<< endl;

		outfile_ImperviousCoverVariables_m[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_over_ImperviousArea_m"
			<< "," << "Precipitation_Snow_WaterEquivalent_over_ImperviousArea_m" 
			<< "," << "Irrigation_ImperviousDepressionStorage_m"
			<< "," << "ThroughFall_Precipitation_Rain_TreeCanopy_ImperviousArea_m"
			<< "," << "ThroughFall_Snow_TreeCanopy_ImperviousArea_m"
			<< "," << "Sublimation_noTreeCanopy_ImperviousArea_m"
			<< "," << "Sublimation_TreeCanopy_ImperviousArea_m"
			<< "," << "Storage_Snow_noTreeCanopy_ImperviousArea_m"
			<< "," << "Storage_Snow_TreeCanopy_ImperviousArea_m"
			<< "," << "Flux_to_ImperviousArea_m"
			<< "," << "WaterBalance_ImperviousArea_m"
			<< "," << "SnowWaterBalance_ImperviousArea_m"
			<< "," << "SnowMelt_noTreeCanopy_ImperviousArea_m"
			<< "," << "SnowMelt_TreeCanopy_ImperviousArea_m"
			<< "," << "Storage_Surface_ImperviousArea_m"
			<< "," << "Evaporation_Surface_ImperviousArea_m"
			<< "," << "Runoff_fromImperviousArea_to_PerviousArea_m"
			<< "," << "Runoff_to_Outlet_m"
			<< "," << "WaterBalance_ImperviousArea_Runoff_m" 
			<< "," << "EvapoTranspiration_m"
			<< endl;

		outfile_ImperviousCoverVariables_m3[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_over_ImperviousArea_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_over_ImperviousArea_m3" 
			<< "," << "Irrigation_ImperviousDepressionStorage_m3"
			<< "," << "ThroughFall_Precipitation_Rain_TreeCanopy_ImperviousArea_m3"
			<< "," << "ThroughFall_Snow_TreeCanopy_ImperviousArea_m3"
			<< "," << "Sublimation_noTreeCanopy_ImperviousArea_m3"
			<< "," << "Sublimation_TreeCanopy_ImperviousArea_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_on_noTreeCanopy_ImperviousArea_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_on_TreeCanopy_ImperviousArea_m3"
			<< "," << "Flux_to_ImperviousArea_m3"
			<< "," << "WaterBalance_ImperviousArea_m3"
			<< "," << "SnowWaterBalance_ImperviousArea_m"
			<< "," << "SnowMelt_noTreeCanopy_ImperviousArea_m3"
			<< "," << "SnowMelt_TreeCanopy_ImperviousArea_m3"
			<< "," << "Storage_Surface_ImperviousArea_m3"
			<< "," << "Evaporation_Surface_ImperviousArea_m3"
			<< "," << "Runoff_fromImperviousArea_to_PerviousArea_m3"
			<< "," << "Runoff_to_Outlet_m3"
			<< "," << "WaterBalance_ImperviousArea_m3"
			<< "," << "EvapoTranspiration_m3"
			<< endl;

		outfile_TreeCoverVariables_m[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_TreeCover_m"
			<< "," << "Precipitation_Snow_WaterEquivalent_TreeCover_m"
			<< "," << "Irrigation_TreeInterceptionStorage_m"
			<< "," << "Interception_TreeCanopy_Rain_m"
			<< "," << "Interception_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "Evaporation_TreeCanopy_Rain_m"
			<< "," << "Sublimation_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "Storage_TreeCanopy_Rain_m"
			<< "," << "Storage_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "WaterBalance_TreeCanopy_m"
			<< "," << "ThroughFall_TreeCanopy_Rain_and_SnowMelt_m"
			<< "," << "ThroughFall_TreeCanopy_Snow_WaterEquivalent_m"
			<< "," << "SnowMelt_UnderTree_m"
			<< "," << "Sublimation_UnderTree_m"
			<< "," << "Storage_Snow_WaterEquivalent_UnderTree_m"
			<< "," << "Flux_to_TreeCover_PerviousArea_Rain_SnowMelt_Irrigation_m"
			<< "," << "Flux_ImperviousAreaRunon_to_TreeCover_PerviousArea_m"
			<< "," << "Flux_to_TreeCover_PerviousArea_m"
			<< "," << "Storage_Surface_TreeCover_PerviousArea_m"
			<< "," << "Storage_Surface_TreeCover_ImperviousArea_m"
			<< "," << "Evaporation_Surface_PerviousArea_m"
			<< "," << "Evaporation_Surface_ImperviousArea_m"
			<< "," << "Flux_to_Infiltration_TreeCover_PerviousArea_m"
			<< "," << "Infiltration_TreeCover_PerviousArea_m"
			<< "," << "Drainage_macroPore_TreeCover_PerviousArea_m"
			<< "," << "Drainage_SoilEvapZone_TreeCover_PerviousArea_m"
			<< "," << "EvapoTranspiration_TreeCover_m"
			<< "," << "Storage_SoilEvapZone_TreeCover_PerviousArea_m"
			<< "," << "Storage_SoilEvapZone_TreeCover_PerviousArea_m3pm3"
			<< "," << "Storage_VadoseZone_TreeCover_PerviousArea_m"
			<< "," << "StorageDeficit_VadoseZone_TreeCover_PerviousArea_m"
			<< "," << "Drainage_VadoseZone_TreeCover_PerviousArea_m"
			<< "," << "Runoff_Subsurface_TreeCover_PerviousArea_m"
			<< "," << "Runoff_Surface_TreeCover_PerviousArea_m"
			<< "," << "Runoff_Surface_TreeCover_ImperviousArea_m"
			<< "," << "Runoff_Surface_TreeCover_m"
			<< "," << "Runoff_TreeCover_m"
			<< endl;

		outfile_TreeCoverVariables_m3[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_TreeCover_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_TreeCover_m3"
			<< "," << "Irrigation_TreeInterceptionStorage_m3"
			<< "," << "Interception_TreeCanopy_Rain_m3"
			<< "," << "Interception_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "Evaporation_TreeCanopy_Rain_m3"
			<< "," << "Sublimation_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "Storage_TreeCanopy_Rain_m3"
			<< "," << "Storage_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "ThroughFall_TreeCanopy_Rain_and_SnowMelt_m3"
			<< "," << "ThroughFall_TreeCanopy_Snow_WaterEquivalent_m3"
			<< "," << "WaterBalance_TreeCanopy_m3"
			<< "," << "SnowMelt_UnderTree_m3"
			<< "," << "Sublimation_UnderTree_m3"
			<< "," << "Storage_Snow_WaterEquivalent_UnderTree_m3"
			<< "," << "Flux_to_TreeCover_PerviousArea_Rain_SnowMelt_Irrigation_m3"
			<< "," << "Flux_ImperviousAreaRunon_to_TreeCover_PerviousArea_m3"
			<< "," << "Flux_to_TreeCover_PerviousArea_m3"
			<< "," << "Storage_Surface_TreeCover_PerviousArea_m3"
			<< "," << "Storage_Surface_TreeCover_ImperviousArea_m3"
			<< "," << "Evaporation_Surface_PerviousArea_m3"
			<< "," << "Evaporation_Surface_ImperviousArea_m3"
			<< "," << "Flux_to_Infiltration_TreeCover_PerviousArea_m3"
			<< "," << "Infiltration_TreeCover_PerviousArea_m3"
			<< "," << "Drainage_macroPore_TreeCover_PerviousArea_m3"
			<< "," << "Drainage_SoilEvapZone_TreeCover_PerviousArea_m3"
			<< "," << "EvapoTranspiration_TreeCover_m3"
			<< "," << "Storage_SoilEvapZone_TreeCover_PerviousArea_m3"
			<< "," << "Storage_SoilEvapZone_TreeCover_PerviousArea_m3pm3"
			<< "," << "Storage_VadoseZone_TreeCover_PerviousArea_m3"
			<< "," << "StorageDeficit_VadoseZone_TreeCover_PerviousArea_m3"
			<< "," << "Drainage_VadoseZone_TreeCover_PerviousArea_m3"
			<< "," << "Runoff_Subsurface_TreeCover_PerviousArea_m3"
			<< "," << "Runoff_Surface_TreeCover_PerviousArea_m3"
			<< "," << "Runoff_Surface_TreeCover_ImperviousArea_m3"
			<< "," << "Runoff_Surface_TreeCover_m3"
			<< "," << "Runoff_TreeCover_m3"
			<< endl;

		outfile_RunoffVariables_m[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_m"
			<< "," << "Precipitation_Snow_WaterEquivalent_m"
			<< "," << "Runoff_All_m"
			<< "," << "Runoff_Surface_m"
			<< "," << "Runoff_Subsurface_m"
			<< "," << "Runoff_Impervious_m"
			<< "," << "Runoff_Water_m"
			<< "," << "Runoff_Pervious_m"
			<< "," << "Runoff_Pervious_SatExcess_m"
			<< "," << "Runoff_Pervious_InfilExcess_m"
			<< endl;

		outfile_RunoffVariables_m3[Folder_Type_ID] << "YYYYMMDD"
			<< "," << "HH:MM:SS"
			<< "," << "Precipitation_Rain_m3"
			<< "," << "Precipitation_Snow_WaterEquivalent_m3"
		<< "," << "Runoff_All_m3"
		<< "," << "Runoff_Surface_m3"
		<< "," << "Runoff_Subsurface_m3"
		<< "," << "Runoff_Impervious_m3"
		<< "," << "Runoff_Water_m3"
		<< "," << "Runoff_Pervious_m3"
		<< "," << "Runoff_Pervious_SatExcess_m3"
		<< "," << "Runoff_Pervious_InfilExcess_m3"
		<< endl;
}
